/*
 *  Project:		nscache
 *  File:		stringbuf.c
 *  started:		28.08.2004
 *  last modified:	02.03.2007
 *  Author:		Harald Foerster
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include <string.h>

#include "stringbuf.h"



StringBuf* stringbuf_new(StringBuf* buffer, size_t size)
{
	StringBuf* p = buffer;

	if(p)
	{
		/* memory for structure not allocated */
		p->ptr = NULL;
	}

	else
	{
		p = malloc(sizeof(StringBuf));

		if(p == NULL)
		{
			return p;
		}

		/* memory for structure has to be freed */
		p->ptr = p;
	}

	if(size == 0)
	{
		size = STRINGBUF_INIT_SIZE - 1;
	}

	/* add byte for terminating '\0' */

	size++;

	p->str = malloc(size);

	if(p->str == NULL)
	{
		p = stringbuf_delete(p);
	} 

	else
	{
		p->max		= size;
		p->str[0]	= '\0';
		p->len		= 0;
		p->sz		= 1;
	}

	return p;

} /* StringBuf* stringbuf_new(StringBuf*, size_t) */



StringBuf* stringbuf_init(StringBuf* buffer, const char* string, size_t size)
{
	StringBuf* p = stringbuf_new(buffer, size);

	if(p && stringbuf_copy(p, string, size) == NULL)
	{
		p = stringbuf_delete(p);
	}

	return p;


} /* StringBuf* stringbuf_init(StringBuf*, const char*, size_t) */



StringBuf* stringbuf_from_string(StringBuf* buffer, char* string, size_t length)
{
	StringBuf* p;

	if(string == NULL)
	{
		return stringbuf_new(buffer, length);
	}

	p = buffer;

	if(p)
	{
		/* memory for structure not allocated */
		p->ptr = NULL;
	}

	else
	{
		p = malloc(sizeof(StringBuf));

		if(p == NULL)
		{
			return p;
		}

		/* memory for structure has to be freed */
		p->ptr = p;
	}

	p->str = string;

	if(length == 0)
	{
		length = strlen(string);
	}

	/* add byte for terminating '\0' */

	p->len	= length;
	p->max	= length + 1;
	p->sz	= length + 1;

	return p;

} /* StringBuf* stringbuf_from_string(StringBuf*, char*, size_t) */



StringBuf* stringbuf_delete(StringBuf* buffer)
{
	if(buffer)
	{
		if(buffer->str)
		{
			free(buffer->str);
			buffer->str = NULL;
		}

		if(buffer->ptr)
		{
			free(buffer->ptr);
		}
	}

	return NULL;

} /* StringBuf* stringbuf_delete(StringBuf*) */



char* stringbuf_realloc(StringBuf* buffer, size_t size)
{
	char* string = buffer->str;

	if(size > buffer->max)
	{
		string = realloc(buffer->str, buffer->max + size);

		if(string)
		{
			buffer->str	= string;
			buffer->max	+= size;
		}
	}

	return string;

} /* char* stringbuf_realloc(StringBuf*, size_t) */



char* stringbuf_move(StringBuf* buffer, size_t position, int length)
{
	/* move 'buffer->str' from 'position' to 'position + length'*/

	int dst, src, len;

	len = buffer->sz + length;

	if(len <= 0 || position >= buffer->sz)
	{
		return NULL;
	}

	/* move from 'position' 'length' bytes up or down */

	if(length < 0)
	{
		dst = position;
		src = position - length; /* -(-) */
	}

	else
	{
		if(len > buffer->max && stringbuf_realloc(buffer, len) == NULL)
		{
			return NULL;
		}

		dst = position + length;
		src = position;
	}

	/* bytes to move (including trailing zero) */

	length = buffer->sz - src;

	/* the new length */

	buffer->sz = len;

	/* string length without trailing zero */

	buffer->len = len - 1;

	/* move the string */

	return memmove(&(buffer->str[dst]), &(buffer->str[src]), length);

} /* char* stringbuf_move(StringBuf* buffer, size_t position, int length) */



char* stringbuf_copy(StringBuf* buffer, const char* string, size_t length)
{
	/* copy the string to buffer start */

	buffer->sz 	= 1;
	buffer->len	= 0;
	buffer->str[0]	= '\0';

	return stringbuf_append(buffer, string, length);

} /* char* stringbuf_copy(StringBuf*, const char*, size_t) */



char* stringbuf_append(StringBuf* buffer, const char* string, size_t length)
{
	/* put string at the end */

	if(string && string[0] != '\0')
	{
		/* buffer length with trailing zero */

		if(length == 0)
		{
			length = strlen(string);
		}

		length += buffer->sz;

		if(length > buffer->max && stringbuf_realloc(buffer, length) == NULL)
		{
			return NULL;
		}

		/* first append the string (copy with trailing zero) */

		memcpy(&(buffer->str[buffer->len]), string, length - buffer->len);

		/* and then set the new length */

		buffer->sz = length;

		/* string length without trailing zero */

		buffer->len = length - 1;
	}

	return buffer->str;

} /* char* stringbuf_append(StringBuf*, const char*, size_t) */



char* stringbuf_insert(StringBuf* buffer, const char* string, size_t length, size_t position)
{
	/* insert 'string' at 'position' */

	if(string != NULL && string[0] != '\0')
	{
		/* buffer length with trailing zero */

		if(length == 0)
		{
			length = strlen(string);
		}

		/* move the string from 'position' 'length' bytes up */

		if(stringbuf_move(buffer, position, length) == NULL)
		{
			/* out of memory */

			return NULL;
		}

		/* copy 'string' to position (without trailing zero) */

		memmove(&(buffer->str[position]), string, length);
	}

	return &(buffer->str[position]);

} /* char* stringbuf_insert(StringBuf*, const char*, size_t, size_t) */



char* stringbuf_replace(StringBuf* buffer, const char* s1, size_t l1, const char* s2, size_t l2)
{
	/* replace first 's1' with 's2' */

	if(s1 != NULL && s1[0] != '\0')
	{
		/* find string */

		char* s = strstr(buffer->str, s1);

		if(s != NULL)
		{
			int differ;

			if(l1 == 0)
			{
				l1 = strlen(s1);
			}

			differ = -l1;

			if(s2)
			{
				if(l2 == 0)
				{
					l2 = strlen(s2);
				}

				differ += l2;
			}

			if(differ != 0)
			{
				/* move the string from 'position' 'len' bytes up or down */

				if(stringbuf_move(buffer, s - buffer->str, differ) == NULL)
				{
					return NULL;
				}
			}

			/* copy 'string' to position (without trailing zero) */

			if(s2)
			{
				memmove(s, s2, l2);
			}

			return s;

		} /* if(s != NULL) */

	} /* if(s1 != NULL && s1[0] != '\0') */

	return NIL;

} /* char* stringbuf_replace(StringBuf*, const char*, size_t, const char*, size_t) */



char* stringbuf_replace_last(StringBuf* buffer, const char* s1, size_t l1, const char* s2, size_t l2)
{
	/* replace last 's1' with 's2' */

	if(s1 != NULL && s1[0] != '\0')
	{
		char* c;
		char* s;

		if(l1 == 0)
		{
			l1 = strlen(s1);
		}

		s = NULL;
		c = buffer->str;

		/* find last occurence of string */

		while( (c = strstr(c, s1)) != NULL)
		{
			/* remember address */

			s = c;

			/* skip found string and continue */

			c += l1;
		}

		if(s != NULL)
		{
			int differ = -l1;

			if(s2)
			{
				if(l2 == 0)
				{
					l2 = strlen(s2);
				}

				differ += l2;
			}

			if(differ != 0)
			{
				/* move the string from 'position' 'len' bytes up or down */

				if(stringbuf_move(buffer, s - buffer->str, differ) == NULL)
				{
					return NULL;
				}
			}

			/* copy 'string' to position (without trailing zero) */

			if(s2)
			{
				memmove(s, s2, l2);
			}

			return s;

		} /* if(s != NULL) */

	} /* if(s1 != NULL && s1[0] != '\0') */

	return NIL;

} /* char* stringbuf_replace_last(StringBuf*, const char*, size_t, const char*, size_t) */



char* stringbuf_truncate(StringBuf* buffer, size_t length)
{
	/* shorten the string to 'length' */

	if(buffer->len > length)
	{
		/* string length without trailing zero */

		buffer->len  = length;

		/* buffer size with trailing zero */

		buffer->sz   = length + 1;

		/* truncate string to 'length' */

		buffer->str[length] = '\0';
	}

	return buffer->str;

} /* char* stringbuf_truncate(StringBuf*, size_t) */



char* stringbuf_character_append(StringBuf* buffer, int ch)
{
	/* put character at the end */

	char* p = stringbuf_character_replace(buffer, ch, buffer->len);

	if(p)
	{
		/* increment the length and terminate */

		buffer->len = buffer->sz;
		buffer->sz++;

		p[1] = '\0';
	}

	return p;

} /* char* stringbuf_character_append(StringBuf*, int) */



char* stringbuf_character_replace(StringBuf* buffer, int ch, size_t position)
{
	/* replace character with 'ch' at string 'position' */

	char* p;

	if(position > buffer->len)
	{
		if(position > buffer->max &&
				stringbuf_realloc(buffer, position + 1) == NULL)
		{
			return NULL; 
		}

		buffer->sz  = position + 1;
		buffer->len = position;
	}

	p = &(buffer->str[position]);

	*p = ch;

	return p;

} /* char* stringbuf_character_replace(StringBuf*, int, size_t) */



char* stringbuf_character_insert(StringBuf* buffer, int ch, size_t position)
{
	/* insert character 'ch' at 'position' */

	char string[2];

	string[0] = ch;
	string[1] = '\0';

	return stringbuf_insert(buffer, string, 1, position);

} /* char* stringbuf_character_insert(StringBuf*, int, size_t) */

/* EOF */
