/*
 *  Project:		nscache
 *  File:		stringbuf.h
 *  started:		28.08.2004
 *  last modified:	20.12.2005
 *  Author:		Harald Foerster
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#ifndef STRINGBUF_H
#define STRINGBUF_H

#include <stdlib.h>


enum
{
	STRINGBUF_INIT_SIZE	= 512
};


/* Not In List */
#define NIL	((void*) -1)

typedef struct
{
	char*		str;	/* string buffer */
	size_t		len;	/* string length */
	size_t		sz;	/* including '\0' (len+1) */
	size_t		max;	/* max buffer size */
	void*		ptr;	/* if memory for this structure was malloc'd */

} StringBuf;


/* allocate memory for the string and the structure (if 'buffer' is NULL) */
StringBuf* stringbuf_new(StringBuf* buffer, size_t size);

/* call 'stringbuf_new' and 'stringbuf_copy' */
StringBuf* stringbuf_init(StringBuf* buffer, const char* string, size_t size);

/* make a stringbuf from string ('string' must be malloc'ed memory) */
StringBuf* stringbuf_from_string(StringBuf* buffer, char* string, size_t length);

/* free allocated memory and returns always NULL */
StringBuf* stringbuf_delete(StringBuf* buffer);

/* enlarge string memory */
char* stringbuf_realloc(StringBuf* buffer, size_t size);

/* move 'buffer->str' from 'position' to 'position + length' (length < 0 moves down) */
char* stringbuf_move(StringBuf* buffer, size_t position, int length);

/* copy 'string' at start of string memory */
char* stringbuf_copy(StringBuf* buffer, const char* string, size_t length);

/* append 'string' at end */
char* stringbuf_append(StringBuf* buffer, const char* string, size_t length);

/* insert 'string' at 'position' */
char* stringbuf_insert(StringBuf* buffer, const char* string, size_t length, size_t position);

/* replace first occurence of 's1' with 's2' and return start of 's1' (NIL not found, NULL no memory) */
char* stringbuf_replace(StringBuf* buffer, const char* s1, size_t l1, const char* s2, size_t l2);

/* replace last occurence of 's1' with 's2' and return start of 's1' (NIL not found, NULL no memory) */
char* stringbuf_replace_last(StringBuf* buffer, const char* s1, size_t l1, const char* s2, size_t l2);

/* shorten the string to 'length' */
char* stringbuf_truncate(StringBuf* buffer, size_t length);

/* append the character at end */
char* stringbuf_character_append(StringBuf* buffer, int ch);

/* replace character with 'ch' at string 'position' */
char* stringbuf_character_replace(StringBuf* buffer, int ch, size_t position);

/* insert character 'ch' at string 'position' */
char* stringbuf_character_insert(StringBuf* buffer, int ch, size_t position);


/* insert 'string' at beginning of buffer */
#define stringbuf_prepend(buffer, string, length)	\
		stringbuf_insert(buffer, string, length, 0)

/* remove first occurence of 'string' and return start of 's1' (or NULL if none found ) */
#define stringbuf_remove(buffer, string, length)	\
		stringbuf_replace(buffer, string, length, NULL, 0)

/* remove first occurence of 'string' and return start of 's1' (or NULL if none found ) */
#define stringbuf_remove_last(buffer, string, length)	\
		stringbuf_replace_last(buffer, string, length, NULL, 0)


#endif /* STRINGBUF_H */
