/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		modified 2005,2006,2007 by Harald Foerster		   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include "nscache.h"
#include "file.h"

#include <ctype.h>
#include <string.h>

#ifdef HAVE_GLOB
#include <glob.h>
#endif

/*
#include <glib.h>
#include "stringbuf.h"
*/

#ifndef PRE_INCREMENT
#ifdef __powerpc__
#define PRE_INCREMENT		1
#endif
#endif

#define kAsciiLineFeed		0x0a
#define kAsciiCarriageReturn	0x0d

int tl_is_ascii_str(const char *str)
{
	unsigned int		c;
	const unsigned char	*p;

#ifdef PRE_INCREMENT
	p = str - 1;
	while ((c = *(++p)) != '\0')
#else
	p = str;
	while ((c = *(p++)) != '\0')
#endif
	{
		if (c > 0x7f)
		{
			return FALSE;
		}
	}

	return TRUE;

} /* int tl_is_ascii_str(const char*) */

int tl_is_alnum_str(const char *beg, const char *str, int slen)
{
	if ( (( str == beg || isalnum(str[-1]) == FALSE ) &&
					isalnum(str[slen]) == FALSE) )
	{
		return TRUE;
	}

	return FALSE;

} /* int tl_is_alnum_str(const char*, const char*, int) */

int tl_count_lines(const char *p, const char *end)
{
	int count = 1;

#ifdef PRE_INCREMENT
	p--;
	end--;
#endif

	while (end > p)
	{
		unsigned int c;

#ifdef PRE_INCREMENT
		c = *(++p);
#else
		c = *(p++);
#endif
		if (c == kAsciiLineFeed)
		{
			/* Unix */
			count++;
		}

		else if (c == kAsciiCarriageReturn)
		{
			/* Mac ... */

			if (end > p)
			{
#ifdef PRE_INCREMENT
				c = p[1];
#else
				c = p[0];
#endif
				if (c == kAsciiLineFeed)
				{
					/* ... or DOS */
					p++;
				}
			}

			count++;
		}
	}

	return count;

} /* int tl_count_lines(const char*, const char*) */

char *tl_strndup_to_lower(const char *str, int slen)
{
	char *ptr;

	if (slen < 0)
	{
		slen = strlen(str);
	}

	ptr = g_malloc(slen + 1);
	ptr[slen] = '\0';

	while (--slen >= 0)
	{
		unsigned int c = str[slen];

		ptr[slen] = tolower(c);
	}

	return ptr;

} /* char *tl_strndup_to_lower(const char*, int) */

char *tl_replace_non_ascii_chars_in_str(char *str)
{
	unsigned int	c;
	unsigned char	*p;

#ifdef PRE_INCREMENT
	p = str - 1;
	while ((c = *(++p)) != '\0')
	{
		if (c > 0x7f)
		{
			p[0] = '?';
		}
	}
#else
	p = str;
	while ((c = *(p++)) != '\0')
	{
		if (c > 0x7f)
		{
			p[-1] = '?';
		}
	}
#endif
	return str;

} /* char *tl_replace_non_ascii_chars_in_str(char*) */

char *tl_get_linestart(const char *beg, const char *p)
{
	while (p > beg)
	{
		unsigned int c = *(--p);

		if (c < 0x80 && isprint(c) == FALSE)
		{
			return (char *) p + 1;
		}
	}

	return (char *) beg;

} /* char *tl_get_linestart(const char*, const char*) */

char *tl_get_linestop(const char *p, const char *end)
{
	unsigned int c;

#ifdef PRE_INCREMENT
	p--;
	end--;
	while (end > p)
	{
		c = *(++p);

		if (c < 0x80 && isprint(c) == FALSE)
		{
			return (char *) p;
		}
	}

	return (char *) end + 1;
#else
	while (end > p)
	{
		c = *(p++);

		if (c < 0x80 && isprint(c) == FALSE)
		{
			return (char *) p - 1;
		}
	}

	return (char *) end;
#endif

} /* char *tl_get_linestop(const char*, const char*) */

static char *tl_mem_find_init(const char *mem, int mlen, const char *str, int slen)
{
	if (mem == NULL || str == NULL ||
				mlen == 0 || slen == 0)
	{
		return NULL;
	}

	if (mlen < 0)
	{
		mlen = strlen(mem);
	}

	if (slen < 0)
	{
		slen = strlen(str);
	}

	mlen -= slen;

	if (mlen < 0)
	{
		return NULL;
	}

	return (gchar *) &mem[mlen];

} /* static char *tl_mem_find_init(const char*, int, const char*, int) */

char *tl_mem_find_str(const char *mem, int mlen, const char *str, int slen)
{
	unsigned int	firstc, c;
	const char	*stop;

	stop = tl_mem_find_init(mem, mlen, str, slen);

	if (stop == NULL)
	{
		return NULL;
	}

	firstc = *str;

#ifdef PRE_INCREMENT

	mem--;

	while (c = *(++mem), stop >= mem)
#else
	stop++;

	while (c = *(mem++), stop >= mem)
#endif
	{
		if (c == firstc)
		{
			const unsigned char *s1, *s2;
#ifdef PRE_INCREMENT
			s1 = &mem[slen];
#else
			s1 = &mem[slen - 1];
#endif
			s2 = &str[slen];
			
			do
			{
				c = *(--s2);

				if (str == (const char *) s2)
				{
#ifdef PRE_INCREMENT
					return (char *) mem;
#else
					return (char *) mem - 1;
#endif
				}

			}
			while (c == *(--s1));

		} /* if (c == firstc) */

	} /* while (c = *(mem++), stop >= mem) */

	return NULL;

} /* char *tl_mem_find_str(const char*, int, const char*, int) */

char *tl_mem_find_str_igncase(const char *mem, int mlen, const char *str, int slen)
{
	unsigned int	firstc, c;
	const char	*stop;

	stop = tl_mem_find_init(mem, mlen, str, slen);

	if (stop == NULL)
	{
		return NULL;
	}

	firstc = *str;

#ifdef PRE_INCREMENT

	mem--;

	while (c = tolower(*(++mem)), stop >= mem)
#else
	stop++;

	while (c = tolower(*(mem++)), stop >= mem)
#endif
	{
		if (c == firstc)
		{
			const unsigned char *s1, *s2;
#ifdef PRE_INCREMENT
			s1 = &mem[slen];
#else
			s1 = &mem[slen - 1];
#endif
			s2 = &str[slen];
			
			do
			{
				c = *(--s2);

				if (str == (const char *) s2)
				{
#ifdef PRE_INCREMENT
					return (char *) mem;
#else
					return (char *) mem - 1;
#endif
				}

			}
			while (c == tolower(*(--s1)));

		} /* if (c == firstc) */

	} /* while (c = *(mem++), stop >= mem) */

	return NULL;

} /* char *tl_mem_find_str_igncase(const char*, int, const char*, int) */

char *tl_get_mime_value_str(char *mem, int mem_len,
					const char *mime_id, int mid_len)
{
	unsigned char *beg, *stop;

	if (mem == NULL)
	{
		return NULL;
	}

	stop = &mem[mem_len];
	mem++;

	while ((beg =
		tl_mem_find_str_igncase(mem, (char *) stop - (char *) mem,
							mime_id, mid_len)) != NULL)
	{
		if (iscntrl(beg[-1]) || isspace(beg[-1]))
		{
			beg += mid_len;

			while ( stop > beg && (isprint(*beg) == FALSE || isspace(*beg)) )
			{
				beg++;
			}

			if (isalpha(*beg))
			{
				unsigned char *end = beg;

				while (stop > end && isprint(*end))
				{
					if (end[1] == ';')
					{
						break;
					}

					end++;
				}

				while (isspace(*end))
				{
					end--;
				}

				end[1] = '\0';

				return beg;

			} /* if (isalpha(*beg)) */

		} /* if (iscntrl(beg[-1]) || isspace(beg[-1])) */

		mem = &beg[1];

	} /* while ((beg = tl_mem_find_str_igncase(..)) != NULL) */

	return NULL;

} /* char *tl_get_mime_value_str(char*, int, const char*, int) */

char *tl_get_url_start(const char *urlstr)
{
	/*
		If we've got something like this
		"id=123456b&uri=http://www.foo.bar/index.php"
		then we'll skip the first part of the URL and use
		"http://www.foo.bar/index.php"
	*/

	char *beg = NULL;

	if (urlstr && (beg = strstr(urlstr, "://")))
	{
		while (beg > urlstr && (isalnum(beg[-1]) || beg[-1] == '-'))
		{
			beg--;
		}
	}

	return beg;

} /* char *tl_get_url_start(const char*) */

char *tl_get_url_filename(const char *urlstr)
{
	/*
		If we've got something like this
		"http://www.foo.bar/load.html?http://www.foo.bar/index.html"
		then the file name will set to
		"load.html?http://www.foo.bar/index.html"
	*/

	if (urlstr)
	{
		int   chr;
		char *beg, *end;

		end = (char *) &urlstr[strcspn(urlstr, "?;")];

		 chr = *end;
		*end = '\0';

		beg = strrchr(urlstr, '/');

		*end = chr;

		if (beg && *(++beg) != '\0')
		{
			return beg;
		}
	}

	return NULL;

} /* char *tl_get_url_filename(const char*) */

int tl_str_escape_chars(char *buf, int len, const char *str, const char *esc_chars)
{
	int	chr;
	char	*beg, *end;

	if (len <= 0)
	{
		return -1;
	}

	if (esc_chars == NULL)
	{
		esc_chars = "\0";
	}

	chr = '\0';
	beg = buf;

	if (str)
	{
		end = &buf[len - 1];

		while ((chr = *(str++)) != '\0' && buf < end)
		{
			if (strchr(esc_chars, chr))
			{
				*(buf++) = '\\';

				if (buf >= end)
				{
					break;
				}
			}

			*(buf++) = chr;
		}
	}

	*buf = '\0';

	return (chr == '\0') ? buf - beg : -1;

} /* int tl_str_escape_chars(char*, int, const char*, const char*) */

char *tl_get_1qstr(char *str)
{
	static char *ptr = NULL;

	if (str == NULL)
	{
		str = ptr;
	}

	if (str)
	{
		unsigned int chr;

		char *beg = NULL;

		while ( (chr = *(str++)) != '\0' )
		{
			if (chr =='\\')
			{
				memmove(&str[-1], str, strlen(str) + 1);
			}

			else if (chr =='\"')
			{
				if (beg)
				{
					str[-1] = '\0';
					ptr = str;
					return beg;
				}

				beg = str;
			}
		}
	}

	ptr = NULL;

	return NULL;

} /* char *tl_get_1qstr(char*) */

#ifdef HAVE_GLOB

CompletitonData *tl_fname_completion(const gchar *fname)
{
	static const char esc_chars[] = "\\*?[]";

	int		len;
	size_t		idx, path_len, pattern_len;
	char		*beg, *str;
	CompletitonData	*retv;
	char		buf[4096];

	if (fname == NULL)
	{
		return NULL;
	}

	str = strrchr(fname, '/');

	pattern_len = strlen(str == NULL ? fname : &str[1]);

	idx = 0;
	path_len = 0;

	if (*fname == '~' && fname[1] == '/')
	{
		const char *home = g_get_home_dir();

		if (home)
		{
			len = tl_str_escape_chars(buf,
					sizeof(buf) - 1, home, esc_chars);

			if (len < 0)
			{
				return NULL;
			}

			idx = 2;
			buf[len] = '/';
			path_len = len + 1;
		}
	}

	len = tl_str_escape_chars(&buf[path_len],
			sizeof(buf) - 1 - path_len, &fname[idx], esc_chars);
	if (len < 0)
	{
		return NULL;
	}

	path_len += len;
	buf[path_len] = '*';
	buf[path_len + 1] = '\0';

	retv = g_malloc(sizeof(CompletitonData));
	retv->globdata = g_malloc(sizeof(glob_t));

	retv->fnamev = NULL;
	retv->append = NULL;

	if (glob(buf, (GLOB_ERR | GLOB_MARK), NULL, retv->globdata) ||
				((glob_t *) retv->globdata)->gl_pathc == 0)
	{
		tl_completion_free(retv);
		return NULL;
	}

	/* copy pointers to filenames, skip the path */

	beg = ((glob_t *) retv->globdata)->gl_pathv[0];
	str = strrchr(beg, '/');

	if (str == NULL)
	{
		/* filename */
		path_len = 0;
	}

	else
	{
		if (str[1] == '\0')
		{
			/* directory */
			while (*(--str) != '/')
			{
				if (str <= beg)
				{
					break;
				}
			}
		}

		if (*(str) == '/')
		{
			/* skip slash */
			str++;
		}

		path_len = str - beg;
	}

	retv->fnamec = ((glob_t *) retv->globdata)->gl_pathc;
	retv->fnamev = g_malloc((retv->fnamec + 1) * sizeof(char*));

	idx = 0;

	do
	{
		str = ((glob_t *) retv->globdata)->gl_pathv[idx];
		retv->fnamev[idx] = &str[path_len];

	}
	while (++idx < retv->fnamec);

	retv->fnamev[idx] = NULL;

	/*
		if single match, 'append' points
		to start of new characters
	*/

	retv->append = retv->fnamev[0] + pattern_len;

	if (retv->fnamec > 1)
	{
		/*
			else 'append' points to a copy of the
			first matching characters from start
		*/

		int pos;

		pos = pattern_len;
		beg = retv->fnamev[0];

		for (;;)
		{
			int	chr;
			char	**array;

			chr   = beg[pos];
			array = &retv->fnamev[1];

			while ((str = *(array++)) != NULL)
			{
				if (str[pos] != chr)
				{
					goto __done;
				}
			}

			pos++;
		}
__done:
		retv->append = g_strndup(retv->append, pos - pattern_len);
	}

	return retv;

} /* CompletitonData *tl_fname_completion(const gchar*) */

void tl_completion_free(CompletitonData *ptr)
{
	if (ptr)
	{
		g_free(ptr->fnamev);

		if (((glob_t *) ptr->globdata)->gl_pathc > 1)
		{
			g_free(ptr->append);
		}

		globfree(ptr->globdata);
		g_free(ptr->globdata);

		g_free(ptr);
	}

} /* void tl_completion_free(CompletitonData*) */

#endif /* HAVE_GLOB */

#ifdef CONFIG_BERKELEY_DB

char *tl_strnrsep(char *str, int len, int *sep)
{
	int c = *sep;

	if (c && len > 0)
	{
		int  n;
		char *s;

		if (c < 0)
		{
			static const char sep_str[] = "/\\:";

#ifdef PRE_INCREMENT
			const char *p = sep_str - 1;
			while (s = &str[len], (c = *(++p)) != '\0')
#else
			const char *p = sep_str;
			while (s = &str[len], (c = *(p++)) != '\0')
#endif
			{
				while (s > str)
				{
					if (*(--s) == c)
					{
						break;
					}
				}

				if (s > str)
				{
					break;
				}
			}

			*sep = c;
		}

		n = 2;
		s = &str[len];

		while (s > str)
		{
			if (*(--s) == c)
			{
				*s = '/';

				if (--n <= 0)
				{
					return &s[1];
				}
			}
		}

	} /* if (c && len > 0) */

	return str;

} /* char *tl_strnrsep(char*, int, int*) */

#endif /* CONFIG_BERKELEY_DB */

#if GLIB_MAJOR_VERSION == 1

char *tl_strchr_nth(const char *str, int chr, int n)
{
	str--;

	while ((str = strchr(++str, chr)) != NULL)
	{
		if (--n == 0)
		{
			break;
		}
	}

	return (char *) str;

} /* char *tl_strchr_nth(const char*, int, int) */

#endif /* GLIB_MAJOR_VERSION == 1 */

/* EOF */
