/***************************************************************************/
/* 		This code is part of Nscache - viewer of Netscape(tm)	   */
/*		browsers disk cache					   */
/*		Copyright (c) 1999,2000 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		created and modified 2005 ... 2008 by Harald Foerster	   */
/*		(harald_foerster@users.sourceforge.net)			   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "file.h"
#include "stringbuf.h"
#include "viewer.h"


pid_t parent_pid;
pid_t parent_uid;


/* temporary file name */
const char *viewer_temp_dir = "/tmp";

/* parameter is URL or file */
const char uof_param_str[][3] = {"%u", "%f"};


/* error types */

#define FORK_ERROR	-1
#define EXEC_ERROR	 1


/* pid and status of last viewer */

#define VIEWER_DONE	-1
#define VIEWER_NOERR	 0

volatile pid_t	viewer_pid;
volatile int	viewer_status;


/* allocated for each started viewer examining a temporary file */

typedef struct
{
	pid_t	child;
	char*	temp_file;

} viewer_data_t;

static GSList* children	= NULL;


static void unlink_temporary_file(gpointer slist_data, gpointer user_data)
{
	/* Remove temporary file */

	viewer_data_t* viewer_ptr = slist_data;
	unlink(viewer_ptr->temp_file);

} /* static void unlink_temporary_file(gpointer, gpointer) */

void viewer_atexit(void)
{
	/* Remove all temporary files */

	g_slist_foreach(children, unlink_temporary_file, NULL);

} /* void viewer_atexit(void) */

static int search_child_func(gconstpointer slist_data, gconstpointer user_data)
{
	/* Compare viewer pid with user_data and return zero if equal */

	const viewer_data_t* viewer_ptr = slist_data;

	pid_t pid = *(pid_t *) user_data;

	return viewer_ptr->child - pid;

} /* static int search_child_func(gconstpointer, gconstpointer) */

void viewer_sigchld(int signum)
{
	/* Remove temporary file and free memory  */

	int	status;
	pid_t	pid;

	while( (pid = waitpid(-1, &status, WNOHANG)) > 0 )
	{
		GSList*	slist;

		if(viewer_pid == pid)
		{
			/*
				The viewer couldn't either be started
				or finished immediatly with an error.
			*/

			viewer_status = WIFEXITED(status) == FALSE ?
						-1 : WEXITSTATUS(status);
			viewer_pid = VIEWER_DONE;
		}

		slist = g_slist_find_custom(children, &pid,
					(GCompareFunc) search_child_func);
		if(slist)
		{
			viewer_data_t* viewer_ptr = slist->data;

			unlink(viewer_ptr->temp_file);
			free(viewer_ptr->temp_file);
			free(viewer_ptr);
			children = g_slist_remove(children, viewer_ptr);

		} /* if(slist) */

	} /* while((pid = waitpid(-1, &status, WNOHANG) > 0) */

} /* void viewer_sigchld(int) */

static int start_viewer(const char *application, char *url_or_file,
						int uof_len, int viewer_type)
{
	pid_t		child_pid;
	sigset_t	sig_block, sig_unblock;
	int		argc, error, pipe_fd[2];
	unsigned int	chr, end, exec_len;
	char*		str;
	char*		uof_arg;
	const char*	uof_repl;
	char*		exec_args;
	char*		argv[100];

	exec_len 	= strlen(application) + 1;
	uof_repl	= uof_param_str[viewer_type];
	str		= (char *) application;

	while ((str = strstr(++str, uof_repl)) != NULL)
	{
		exec_len += uof_len;
	}

	exec_args = malloc(exec_len);

	if(exec_args == NULL)
	{
		return -1;
	}

	/* break 'application' in pieces */

	uof_arg = NULL;

	argc = 0;
	str  = exec_args;
	end  = ' ';
	chr  = *(application++);

	while(chr != '\0')
	{
		argv[argc] = str;
		argc++;

		if(argc >= (sizeof(argv) / sizeof(char*)) - 1)
		{
			/* this shouldn't happen */
			exit(ENOMEM);
		}

		do
		{
			if(chr == end)
			{
				break;
			}

			if(chr == '\\')
			{
				chr = *(application++);

				if(chr == '\0')
				{
					break;
				}
			}

			else if(chr == '%' && argc > 1)
			{
				if(*application == uof_repl[1] &&
						! isalnum(application[1]))
				{
					str = strcpy(str, url_or_file) + uof_len;
					uof_arg = url_or_file;
					application++;
					continue;
				}
			}

			*(str++) = chr;
		}
		while( (chr = *(application++)) != '\0');

		*(str++) = '\0';
		end	 = ' ';

		while(chr == end) 
		{
			chr = *(application++);
		}

		if(chr == '\'')
		{
			end = chr;
			chr = *(application++);
		}
	}

	if(uof_arg == NULL)
	{
		argv[argc] = url_or_file;
		argc++;
	}

	argv[argc] = NULL;

	/* block SIGCHLD (in fact all signals) */

	sigfillset(&sig_block);
	sigprocmask(SIG_BLOCK, &sig_block, &sig_unblock);

	if(pipe(pipe_fd) == 0)
	{
		switch(child_pid = fork())
		{
			case 0:
			{
				/* tell parent I am ready */

				close(pipe_fd[0]);
				close(pipe_fd[1]);

				execvp(argv[0], argv);
				_exit(errno == 0 ? ENOENT : errno);

			} /* case  0 (child) */

			case -1:
			{
				close(pipe_fd[0]);
				close(pipe_fd[1]);
				break;
			}

			default:
			{
				char pipe_buf[1];

				viewer_pid = child_pid;

				/* wait for child */

				close(pipe_fd[1]);
				read(pipe_fd[0], pipe_buf, sizeof(pipe_buf));
				close(pipe_fd[0]);

				if(viewer_type == VIEWER_FILE)
				{
					viewer_data_t* viewer_ptr;

					viewer_ptr = malloc(sizeof(viewer_data_t));

					if(viewer_ptr == NULL)
					{
						break;
					}

					viewer_ptr->child     = child_pid;
					viewer_ptr->temp_file = url_or_file;

					children = g_slist_append(children, viewer_ptr);
				}

				free(exec_args);

				/*
					call 'viewer_sigchld' directly
					because signals are blocked
				*/

				viewer_status = VIEWER_NOERR;
				viewer_sigchld(SIGCHLD);

				/* unblock SIGCHLD */
				sigprocmask(SIG_SETMASK, &sig_unblock, NULL);

				errno = viewer_status;

				return errno ? EXEC_ERROR : 0;

			} /* default (parent) */

		} /* switch( fork() )) */

	} /* if (pipe(pipe_fd) == 0) */

	viewer_pid = VIEWER_DONE;
	error = errno;

	free(exec_args);

	/* unblock SIGCHLD */
	sigprocmask(SIG_SETMASK, &sig_unblock, NULL);

	errno = error;

	return FORK_ERROR;

} /* static int start_viewer(const char*, char*, int, int) */

int viewer_start(const char* appassign, const char* url, const nscache_record_t* rec)
{
	static const char time_fmt_str[] = "%d%m%H%M%S";

	int		error, length;
	char*		slash;
	time_t		tm;
	char		buf[256];
	StringBuf	temp;

	if(appassign == NULL)
	{
		errno = EINVAL;
		return -1;
	}

	if(url)
	{
		return start_viewer(appassign, (char *) url, strlen(url), VIEWER_URL);
	}

	if(rec == NULL)
	{
		errno = EINVAL;
		return -1;
	}

	/* uid, pid and length*/
	length = sprintf(buf, "%X%X%X",
			parent_uid, parent_pid, rec->content_length);

	/* append last access */
	length  += strftime(&buf[length], sizeof(buf) - length,
			time_fmt_str, localtime(&(rec->last_accessed)));

	/* append date and time */
	tm	 = time(NULL);
	length  += strftime(&buf[length], sizeof(buf) - length,
					time_fmt_str, localtime(&tm));

	/* now try to create an unique file name */
	stringbuf_init(&temp, viewer_temp_dir, 0);
	stringbuf_append(&temp, "/mozcache-", sizeof("/mozcache-") - 1);
	stringbuf_append(&temp, buf, length);

	/* search last slash */
	slash = strrchr(rec->urlstr, '/');

	/* search following extension (e.g. '.html') */
	if(slash)
	{
		char*	ext;
		size_t	len;

		ext = strrchr(slash, '.');

		if(ext)
		{
			len = strlen(ext);

			if(temp.len + len >= NAME_MAX)
			{
				ext = NULL;
			}
		}

		else if( (len = slash[1]) == '\0')
		{
			ext = ".html";
			len = 5;
		}

		/* append extension */
		stringbuf_append(&temp, ext, len);
	}

	if(temp.str == NULL)
	{
		return -1;
	}

	error = file_copy(rec, temp.str, FILE_PERM_TEMP);

	if(error)
	{
		stringbuf_delete(&temp);
	}

	else
	{
		char* p = realloc(temp.str, temp.sz);

		if(p == NULL)
		{
			goto _viewer_quit;
		}

		temp.str = p;
		error = start_viewer(appassign, temp.str, temp.len, VIEWER_FILE);

		if(error == FORK_ERROR && temp.str)
		{
_viewer_quit:
			unlink(temp.str);
			stringbuf_delete(&temp);
		}
	}

	return error;

} /* int viewer_start(const char*, const char*, const nscache_record_t*) */

/* EOF */
