/***************************************************************************
 *   Copyright (C) 2002-2005 by Serghei Amelian                            *
 *   serghei.amelian@gmail.com                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _QFRTIFFIO_H_
#define _QFRTIFFIO_H_

#include <tiffio.h>

/** <p><b>libtiff</b> is a library for reading and writing data files encoded with the Tag Image File
  format, Revision 6.0 (or revision 5.0 or revision 4.0).   This  file  format  is  suitable
  for archiving multi-color and monochromatic image data.<br><br>
  The  library supports several compression algorithms, as indicated by the Compression field,
  including: no compression (1), CCITT 1D Huffman compression (2),  CCITT  Group  3  Facsimile
  compression  (3),  CCITT  Group  4 Facsimile compression (4), Lempel-Ziv & Welch compression
  (5), baseline JPEG compression (7), word-aligned 1D Huffman compression (32771),  and
  PackBits  compression (32773).  In addition, several nonstandard compression algorithms are
  supported: the 4-bit compression algorithm used by the ThunderScan program (32809)
  (decompression only), NeXT's 2-bit compression algorithm (32766) (decompression only), an
  experimental LZ-style algorithm known as Deflate (32946), and  an  experimental  CIE  LogLuv
  compression scheme designed for images with high dynamic range (32845 for LogL and 32845 for
  LogLuv). Directory information may be in either little- or big-endian  byte  order-byte
  swapping  is automatically  done  by  the  library.  Data bit ordering may be either Most
  Significant Bit (MSB) to Least Significant Bit (LSB) or LSB to MSB.  Finally, the library does
  not support files in which the BitsPerSample, Compression, MinSampleValue, or MaxSampleValue
  fields are defined differently on a per-sample basis (in Rev. 6.0 the Compression tag is not
  defined on a per-sample basis, so this is immaterial).<br><br>
  The  library makes extensive use of C typedefs to promote portability.  Two sets of typedefs
  are used, one for communication with clients of the library and one for internal data
  structures  and  parsing  of the TIFF format.  The following typedefs are exposed to users
  either through function definitions or through parameters passed through the varargs
  interfaces.</p><br><br>
  <table border=0>
  <tr><td>typedef unsigned short uint16;</td><td>16-bit unsigned integer</tr>
  <tr><td>typedef unsigned \<thing\> uint32;</td><td>32-bit unsigned integer</tr>
  <tr><td>typedef unsigned int ttag_t;</td><td>directory tag</tr>
  <tr><td>typedef uint16 tdir_t;</td><td>directory index</tr>
  <tr><td>typedef uint16 tsample_t;</td><td>sample number</tr>
  <tr><td>typedef uint32 tstrip_t;</td><td>strip number</tr>
  <tr><td>typedef uint32 ttile_t;</td><td>tile number</tr>
  <tr><td>typedef int32 tsize_t;</td><td>i/o size in bytes</tr>
  <tr><td>typedef void* tdata_t;</td><td>image data ref</tr>
  <tr><td>typedef void* thandle_t;</td><td>client data handle</tr>
  <tr><td>typedef int32 toff_t;</td><td>file offset</tr>
  </table>
  <p>Note that tstrip_t, ttile_t, and tsize_t are constrained to be no more than 32-bit quantities
  by 32-bit fields they are stored in in the TIFF image.  Likewise tsample_t is limited
  by the 16-bit field used to store the SamplesPerPixel tag.  tdir_t  constrains  the  maximum
  number  of  IFDs  that  may  appear  in an image and may be an arbitrary size (w/o penalty).
  ttag_t must be either int, unsigned int, pointer, or  double  because  the  library  uses  a
  varargs  interface and ANSI C restricts the type of the parameter before an ellipsis to be a
  promoted type.  toff_t is defined as int32 because TIFF file offsets are  (unsigned)  32-bit
  quantities.   A  signed  value is used because some interfaces return -1 on error.  Finally,
  note that user-specified data references are passed as opaque handles and only cast  at  the
  lowest layers where their type is presumed.</p> */
class QfrTiffIO {
public:
  QfrTiffIO();
  ~QfrTiffIO();
  //int CheckpointDirectory();
  int CheckTile(uint32 x, uint32 y, uint32 z, tsample_t s = 0);
  TIFF *ClientOpen(const char *name, const char *mode, thandle_t clientdata, TIFFReadWriteProc readproc, TIFFReadWriteProc writeproc, TIFFSeekProc seekproc, TIFFCloseProc closeproc, TIFFSizeProc sizeproc, TIFFMapFileProc mapproc, TIFFUnmapFileProc unmapproc);
  void Close();
  tstrip_t ComputeStrip(uint32 row, tsample_t sample);
  ttile_t ComputeTile(uint32 x, uint32 y, uint32 z, tsample_t s);
  tdir_t CurrentDirectory();
  uint32 CurrentRow();
  tstrip_t CurrentStrip();
  ttile_t CurrentTile();
  TIFF *FdOpen(int fd, const char *name, const char *mode);
  const char *FileName();
  int Fileno();
  int Flush();
  int FlushData();
  int GetField(ttag_t tag, ...);
  int GetFieldDefaulted(ttag_t tag, ...);
  int GetMode();
  int IsTiled();
  int IsByteSwapped();
  uint16 NumberOfDirectories();
  tstrip_t NumberOfStrips();
  ttile_t NumberOfTiles();
  TIFF *Open(const char *name, const char *mode);
  TIFF *Open(TIFF *tif);
  void PrintDirectory(FILE *fd, long flags);
  int ReadBufferSetup(tdata_t bp, tsize_t size);
  int ReadDirectory();
  tsize_t ReadEncodedStrip(tstrip_t strip, tdata_t buf, tsize_t size);
  tsize_t ReadEncodedTile(ttile_t tile, tdata_t buf, tsize_t size);
  tsize_t ReadRawStrip(tstrip_t strip, tdata_t buf, tsize_t size);
  tsize_t ReadRawTile(ttile_t tile, tdata_t buf, tsize_t size);
  int ReadRGBAImage(uint32 rwidth, uint32 rheight, uint32 *raster, int stop = 0);
  //int ReadRGBAImageOriented(uint32 rwidth, uint32 rheight, uint32 *raster, int orientation);
  int ReadScanline(tdata_t buf, uint32 row, tsample_t sample = 0);
  tsize_t ReadTile(tdata_t buf, uint32 x, uint32 y, uint32 z, tsample_t s);
  int RGBAImageBegin(TIFFRGBAImage *img, int stop, char emsg[1024]);
  void RGBAImageEnd(TIFFRGBAImage *img);
  int RGBAImageGet(TIFFRGBAImage *img, uint32 *raster, uint32 w, uint32 h);
  int RGBAImageOK(char emsg[1024]);
  tsize_t ScanlineSize();
  int SetDirectory(tdir_t dirn);
  int SetSubDirectory(uint32 diroff);
  int SetField(ttag_t tag, ...);
  tsize_t StripSize();
  //tsize_t RawStripSize(tstrip_t strip);
  tsize_t TileRowSize();
  tsize_t TileSize();
  int VGetField(ttag_t tag, va_list ap);
  int VGetFieldDefaulted(ttag_t tag, va_list ap);
  int VSetField(ttag_t tag, va_list ap);
  tsize_t VStripSize(uint32 nrows);
  int WriteDirectory();
  tsize_t WriteEncodedStrip(tstrip_t strip, tdata_t data, tsize_t cc);
  tsize_t WriteEncodedTile(ttile_t tile, tdata_t data, tsize_t cc);
  tsize_t WriteRawStrip(tstrip_t strip, tdata_t data, tsize_t cc);
  tsize_t WriteRawTile(ttile_t tile, tdata_t data, tsize_t cc);
  int WriteScanline(tdata_t buf, uint32 row, tsample_t sample);
  tsize_t WriteTile(tdata_t buf, uint32 x, uint32 y, uint32 z, tsample_t s);
protected:
  TIFF *tif;
};

/** Constructor */
inline QfrTiffIO::QfrTiffIO()
: tif(0)
{
}

/** Destructor */
inline QfrTiffIO::~QfrTiffIO()
{
  //Close();
}

/** Similar to TIFFWriteDirectory(), writes the directory out
 but leaves all data structures in memory so that it can be
 written again.  This will make a partially written TIFF file
 readable before it is successfully completed/closed. */
//inline int QfrTiffIO::CheckpointDirectory()
//{
//  return ::TIFFCheckpointDirectory(tif);
//}

/** Check an (x,y,z,s) coordinate against the image bounds. */
inline int QfrTiffIO::CheckTile(uint32 x, uint32 y, uint32 z, tsample_t s)
{
  return TIFFCheckTile(tif, x, y, z, s);
}

/** open a file for reading or writing */
inline TIFF *QfrTiffIO::ClientOpen(const char *name, const char *mode, thandle_t clientdata, TIFFReadWriteProc readproc, TIFFReadWriteProc writeproc, TIFFSeekProc seekproc, TIFFCloseProc closeproc, TIFFSizeProc sizeproc, TIFFMapFileProc mapproc, TIFFUnmapFileProc unmapproc)
{
  return tif = ::TIFFClientOpen(name, mode, clientdata, readproc, writeproc, seekproc, closeproc, sizeproc, mapproc, unmapproc);
}

/** close an open file */
inline void QfrTiffIO::Close()
{
  if(tif)
    ::TIFFClose(tif);
}

/** return strip containing y,sample */
inline tstrip_t QfrTiffIO::ComputeStrip(uint32 row, tsample_t sample)
{
  return ::TIFFComputeStrip(tif, row, sample);
}

/** return tile containing x,y,z,sample */
inline ttile_t QfrTiffIO::ComputeTile(uint32 x, uint32 y, uint32 z, tsample_t s)
{
  return ::TIFFComputeTile(tif, x, y, z, s);
}

/** return index of current directory */
inline tdir_t QfrTiffIO::CurrentDirectory()
{
  return ::TIFFCurrentDirectory(tif);
}

/** return index of current scanlineComputeStrip */
inline uint32 QfrTiffIO::CurrentRow()
{
  return ::TIFFCurrentRow(tif);
}

/** return index of current strip */
inline tstrip_t QfrTiffIO::CurrentStrip()
{
  return ::TIFFCurrentStrip(tif);
}

/** return index of current tile */
inline ttile_t QfrTiffIO::CurrentTile()
{
  return ::TIFFCurrentTile(tif);
}

/** open a file for reading or writing */
inline TIFF *QfrTiffIO::FdOpen(int fd, const char *name, const char *mode)
{
  return tif = ::TIFFFdOpen(fd, name, mode);
}

/** return name of open file */
inline const char *QfrTiffIO::FileName()
{
  return ::TIFFFileName(tif);
}

/** return open file descriptor */
inline int QfrTiffIO::Fileno()
{
  return ::TIFFFileno(tif);
}

/** flush all pending writes */
inline int QfrTiffIO::Flush()
{
  return ::TIFFFlush(tif);
}

/** flush pending data writes */
inline int QfrTiffIO::FlushData()
{
  return ::TIFFFlushData(tif);
}

/** return open file mode */
inline int QfrTiffIO::GetMode()
{
  return ::TIFFGetMode(tif);
}

/** return true if image data is tiled */
inline int QfrTiffIO::IsTiled()
{
  return ::TIFFIsTiled(tif);
}

/** return true if image data is byte-swapped */
inline int QfrTiffIO::IsByteSwapped()
{
  return ::TIFFIsByteSwapped(tif);
}

inline uint16 QfrTiffIO::NumberOfDirectories()
{
  return ::TIFFNumberOfDirectories(tif);
}

/** return number of strips in an image */
inline tstrip_t QfrTiffIO::NumberOfStrips()
{
  return ::TIFFNumberOfStrips(tif);
}

/** return number of tiles in an image */
inline ttile_t QfrTiffIO::NumberOfTiles()
{
  return ::TIFFNumberOfTiles(tif);
}

/** open a file for reading or writing */
inline TIFF *QfrTiffIO::Open(const char *name, const char *mode)
{
  return tif = ::TIFFOpen(name, mode);
}

/** Set handler from file opened before */
inline TIFF *QfrTiffIO::Open(TIFF *tif)
{
  return this->tif = tif;
}

/** print description of the current directory */
inline void QfrTiffIO::PrintDirectory(FILE *fd, long flags)
{
  ::TIFFPrintDirectory(tif, fd, flags);
}

/** Setup the raw data buffer in preparation for
 reading a strip of raw data.  If the buffer
 is specified as zero, then a buffer of appropriate
 size is allocated by the library.  Otherwise,
 the client must guarantee that the buffer is
 large enough to hold any individual strip of
 raw data. */
inline int QfrTiffIO::ReadBufferSetup(tdata_t bp, tsize_t size)
{
  return ::TIFFReadBufferSetup(tif, bp, size);
}

/** Read the next TIFF directory from a file
 and convert it to the internal format.
 We read directories sequentially. */
inline int QfrTiffIO::ReadDirectory()
{
  return ::TIFFReadDirectory(tif);
}

/** Read a strip of data and decompress the specified
 amount into the user-supplied buffer. */
inline tsize_t QfrTiffIO::ReadEncodedStrip(tstrip_t strip, tdata_t buf, tsize_t size)
{
  return ::TIFFReadEncodedStrip(tif, strip, buf, size);
}

/** Read a tile of data and decompress the specified
 amount into the user-supplied buffer. */
inline tsize_t QfrTiffIO::ReadEncodedTile(ttile_t tile, tdata_t buf, tsize_t size)
{
  return ::TIFFReadEncodedTile(tif, tile, buf, size);
}

/** Read a strip of data from the file. */
inline tsize_t QfrTiffIO::ReadRawStrip(tstrip_t strip, tdata_t buf, tsize_t size)
{
  return ::TIFFReadRawStrip(tif, strip, buf, size);
}

/** Read a tile of data from the file. */
inline tsize_t QfrTiffIO::ReadRawTile(ttile_t tile, tdata_t buf, tsize_t size)
{
  return ::TIFFReadRawTile(tif, tile, buf, size);
}

/** Read the specified image into an ABGR-format raster. Use bottom left
 origin for raster by default. */
inline int QfrTiffIO::ReadRGBAImage(uint32 rwidth, uint32 rheight, uint32* raster, int stop)
{
  return ::TIFFReadRGBAImage(tif, rwidth, rheight, raster, stop);
}

//inline int QfrTiffIO::ReadRGBAImageOriented(uint32 rwidth, uint32 rheight, uint32 *raster, int orientation)
//{
//  return ::TIFFReadRGBAImageOriented(tif, rwidth, rheight, raster, orientation);
//}

/** read and decode a row of data */
inline int QfrTiffIO::ReadScanline(tdata_t buf, uint32 row, tsample_t sample)
{
  return ::TIFFReadScanline(tif, buf, row, sample);
}

/** Read and decompress a tile of data.  The
 tile is selected by the (x,y,z,s) coordinates. */
inline tsize_t QfrTiffIO::ReadTile(tdata_t buf, uint32 x, uint32 y, uint32 z, tsample_t s)
{
  return ::TIFFReadTile(tif, buf, x, y, z, s);
}

/** Setup decoder state for TIFFRGBAImageGet */
inline int QfrTiffIO::RGBAImageBegin(TIFFRGBAImage *img, int stop, char emsg[1024])
{
  return ::TIFFRGBAImageBegin(img, tif, stop, emsg);
}

/** Release TIFFRGBAImage decoder state */
inline void QfrTiffIO::RGBAImageEnd(TIFFRGBAImage *img)
{
  return ::TIFFRGBAImageEnd(img);
}

/** Read and decode an image */
inline int QfrTiffIO::RGBAImageGet(TIFFRGBAImage *img, uint32 *raster, uint32 w, uint32 h)
{
  return ::TIFFRGBAImageGet(img, raster, w, h);
}

/** Check the image to see if TIFFReadRGBAImage can deal with it.
 1/0 is returned according to whether or not the image can
 be handled.  If 0 is returned, emsg contains the reason
 why it is being rejected. */
inline int QfrTiffIO::RGBAImageOK(char emsg[1024])
{
  return ::TIFFRGBAImageOK(tif, emsg);
}

/** Return the number of bytes to read/write in a call to
 one of the scanline-oriented i/o routines.  Note that
 this number may be 1/samples-per-pixel if data is
 stored as separate planes. */
inline tsize_t QfrTiffIO::ScanlineSize()
{
  return ::TIFFScanlineSize(tif);
}

/** Set the n-th directory as the current directory.
 NB: Directories are numbered starting at 0. */
inline int QfrTiffIO::SetDirectory(tdir_t dirn)
{
  return ::TIFFSetDirectory(tif, dirn);
}

/** Set the current directory to be the directory
 located at the specified file offset.  This interface
 is used mainly to access directories linked with
 the SubIFD tag (e.g. thumbnail images). */
inline int QfrTiffIO::SetSubDirectory(uint32 diroff)
{
  return ::TIFFSetSubDirectory(tif, diroff);
}

/** Compute the # bytes in a (row-aligned) strip.<br><br>
 Note that if RowsPerStrip is larger than the
 recorded ImageLength, then the strip size is
 truncated to reflect the actual space required
 to hold the strip. */
inline tsize_t QfrTiffIO::StripSize()
{
  return ::TIFFStripSize(tif);
}

/** Compute the # bytes in a raw strip. */
//inline tsize_t QfrTiffIO::RawStripSize(tstrip_t strip)
//{
//  return ::TIFFRawStripSize(tif, strip);
//}

/** Compute the # bytes in each row of a tile. */
inline tsize_t QfrTiffIO::TileRowSize()
{
  return ::TIFFTileRowSize(tif);
}

/** Compute the # bytes in a row-aligned tile. */
inline tsize_t QfrTiffIO::TileSize()
{
  return ::TIFFTileSize(tif);
}

/** Like TIFFGetField, but taking a varargs
 parameter list.  This routine is useful
 for building higher-level interfaces on
 top of the library. */
inline int QfrTiffIO::VGetField(ttag_t tag, va_list ap)
{
  return ::TIFFVGetField(tif, tag, ap);
}

/** Like TIFFGetField, but return any default
 value if the tag is not present in the directory.<br><br>
 NB:  We use the value in the directory, rather than
 explicit values so that defaults exist only one
 place in the library -- in TIFFDefaultDirectory. */
inline int QfrTiffIO::VGetFieldDefaulted(ttag_t tag, va_list ap)
{
  return ::TIFFVGetFieldDefaulted(tif, tag, ap);
}

/** Like TIFFSetField, but taking a varargs
 parameter list.  This routine is useful
 for building higher-level interfaces on
 top of the library. */
inline int QfrTiffIO::VSetField(ttag_t tag, va_list ap)
{
  return ::TIFFVSetField(tif, tag, ap);
}

/** Compute the # bytes in a variable height,
 row-aligned strip. */
inline tsize_t QfrTiffIO::VStripSize(uint32 nrows)
{
  return ::TIFFVStripSize(tif, nrows);
}

/** write the current directory */
inline int QfrTiffIO::WriteDirectory()
{
  return ::TIFFWriteDirectory(tif);
}

/** Encode the supplied data and write it to the
 specified strip. <br><br>
 NB: Image length must be setup before writing. */
inline tsize_t QfrTiffIO::WriteEncodedStrip(tstrip_t strip, tdata_t data, tsize_t cc)
{
  return ::TIFFWriteEncodedStrip(tif, strip, data, cc);
}

/**  Encode the supplied data and write it to the
 specified tile.  There must be space for the
 data.  The function clamps individual writes
 to a tile to the tile size, but does not (and
 can not) check that multiple writes to the same
 tile do not write more than tile size data.<br><br>
 NB: Image length must be setup before writing; this
 interface does not support automatically growing
 the image on each write (as TIFFWriteScanline does). */
inline tsize_t QfrTiffIO::WriteEncodedTile(ttile_t tile, tdata_t data, tsize_t cc)
{
  return ::TIFFWriteEncodedTile(tif, tile, data, cc);
}

/** Write the supplied data to the specified strip.<br><br>
NB: Image length must be setup before writing.*/
inline tsize_t QfrTiffIO::WriteRawStrip(tstrip_t strip, tdata_t data, tsize_t cc)
{
  return ::TIFFWriteRawStrip(tif, strip, data, cc);
}

/** Write the supplied data to the specified strip.
 There must be space for the data; we don't check
 if strips overlap!<br><br>
 NB: Image length must be setup before writing; this
 interface does not support automatically growing
  the image on each write (as TIFFWriteScanline does). */
inline tsize_t QfrTiffIO::WriteRawTile(ttile_t tile, tdata_t data, tsize_t cc)
{
  return ::TIFFWriteRawTile(tif, tile, data, cc);
}

/** write a scanline of data */
inline int QfrTiffIO::WriteScanline(tdata_t buf, uint32 row, tsample_t sample)
{
  return ::TIFFWriteScanline(tif, buf, row, sample);
}

/** Write and compress a tile of data.  The
 tile is selected by the (x,y,z,s) coordinates. */
inline tsize_t QfrTiffIO::WriteTile(tdata_t buf, uint32 x, uint32 y, uint32 z, tsample_t s)
{
  return ::TIFFWriteTile(tif, buf, x, y, z, s);
}

#endif
