/***************************************************************************
 *   Copyright (C) 2002-2005 by Serghei Amelian                            *
 *   serghei.amelian@gmail.com                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qbitmap.h>
#include <qpainter.h>
#include <qpopupmenu.h>

#include "qfrviewer.h"


#define hand1_width 16
#define hand1_height 16
#define hand1_x 7
#define hand1_y 7
static unsigned char hand1_bits[] = {
   0x80, 0x01, 0x58, 0x0e, 0x64, 0x12, 0x64, 0x52, 0x48, 0xb2, 0x48, 0x92,
   0x16, 0x90, 0x19, 0x80, 0x11, 0x40, 0x02, 0x40, 0x04, 0x40, 0x04, 0x20,
   0x08, 0x20, 0x10, 0x10, 0x20, 0x10, 0x20, 0x10
};


#define hand1_mask_width 16
#define hand1_mask_height 16
static unsigned char hand1_mask_bits[] = {
   0x80, 0x01, 0xd8, 0x0f, 0xfc, 0x1f, 0xfc, 0x5f, 0xf8, 0xff, 0xf8, 0xff,
   0xf6, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0x7f, 0xfc, 0x7f, 0xfc, 0x3f,
   0xf8, 0x3f, 0xf0, 0x1f, 0xe0, 0x1f, 0xe0, 0x1f
};


#define hand2_width 16
#define hand2_height 16
#define hand2_x 7
#define hand2_y 7
static unsigned char hand2_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0x0d, 0x48, 0x32,
   0x08, 0x50, 0x18, 0x40, 0x14, 0x40, 0x04, 0x40, 0x04, 0x40, 0x04, 0x20,
   0x08, 0x20, 0x10, 0x10, 0x20, 0x10, 0x20, 0x10
};


#define hand2_mask_width 16
#define hand2_mask_height 16
static unsigned char hand2_mask_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0x0d, 0xf8, 0x3f,
   0xf8, 0x7f, 0xf8, 0x7f, 0xfc, 0x7f, 0xfc, 0x7f, 0xfc, 0x7f, 0xfc, 0x3f,
   0xf8, 0x3f, 0xf0, 0x1f, 0xe0, 0x1f, 0xe0, 0x1f
};


/***************************************************************************
 *                                                                         *
 *   QfrCanvas                                                             *
 *                                                                         *
 ***************************************************************************/


QfrCanvas::QfrCanvas(QWidget *parent)
: QWidget(parent, "QfrCanvas", /*WNoAutoErase|*/WStaticContents), m_popup(NULL)
{
  setEraseColor(white);

  // create hand cursors
  QSize hand1_size(hand1_width, hand1_height);
  QBitmap hand1 = QBitmap(hand1_size, hand1_bits, true);
  QBitmap hand1_mask = QBitmap(hand1_size, hand1_mask_bits, true);
  m_hand1 = QCursor(hand1, hand1_mask, hand1_x, hand1_y);

  QSize hand2_size(hand2_width, hand2_height);
  QBitmap hand2 = QBitmap(hand2_size, hand2_bits, true);
  QBitmap hand2_mask = QBitmap(hand2_size, hand2_mask_bits, true);
  m_hand2 = QCursor(hand2, hand2_mask, hand2_x, hand2_y);

  setCursor(m_hand1);
}


void QfrCanvas::paintEvent(QPaintEvent *e)
{
  int x = e->rect().x();
  int y = e->rect().y();
  int w = e->rect().width();
  int h = e->rect().height();

  QPainter p(this);
  p.drawPixmap(x, y, m_pixmap, x, y, w, h);
}


void QfrCanvas::mousePressEvent(QMouseEvent *e)
{
  if(e->button() == LeftButton)
    {
      m_dragGrabPos = e->globalPos();
      m_readyForDrag = true;
      setCursor(m_hand2);
    }
  else
    QWidget::mousePressEvent(e);
}


void QfrCanvas::mouseReleaseEvent(QMouseEvent *e)
{
  if(e->button() == LeftButton)
    {
      m_readyForDrag = false;
      setCursor(m_hand1);
    }
  else
    QWidget::mousePressEvent(e);
}


void QfrCanvas::mouseMoveEvent(QMouseEvent *e)
{
  if(m_readyForDrag)
    {
      QPoint delta = m_dragGrabPos - e->globalPos();
      QScrollView *sw = (QScrollView*)(parent()->parent());
      sw->scrollBy(delta.x(), delta.y());
      m_dragGrabPos = e->globalPos();
    }
  else
    QWidget::mouseMoveEvent(e);
}


void QfrCanvas::contextMenuEvent(QContextMenuEvent *e)
{
  if(m_popup)
    m_popup->exec(e->globalPos());
}


/***************************************************************************
 *                                                                         *
 *   QfrViewer                                                             *
 *                                                                         *
 ***************************************************************************/


QfrViewer::QfrViewer(QWidget *parent)
: QScrollView(parent, "QfrViewer"), m_centerCanvas(false)
{
  viewport()->setEraseColor(eraseColor());
  m_canvas = new QfrCanvas(viewport());
  addChild(m_canvas);
  connect(&m_timer, SIGNAL(timeout()), this, SLOT(delayedResizeEvent()));
}


QfrViewer::~QfrViewer()
{
}


void QfrViewer::setPixmap(const QPixmap &pixmap)
{
  m_canvas->m_pixmap = pixmap;
  m_canvas->setFixedSize(pixmap.width(), pixmap.height());

  if(pixmap.isNull())
    hideCanvas();
  else
    showCanvas();

  m_canvas->update();

  update();
  centerCanvas();
}


void QfrViewer::drawContents(QPainter *p, int, int, int, int)
{

  if(!m_canvas->isShown())
    return;

  int x = childX(m_canvas) - 1;
  int y = childY(m_canvas) - 1;
  int w = m_canvas->width() + 2;
  int h = m_canvas->height() + 2;

  // draw borders
  p->setPen(QColor(0x00, 0x00, 0x00));
  p->drawLine(x, y + h - 1, x + w, y + h - 1);
  p->drawLine(x + w - 1, y, x + w - 1, y + h - 1);
  //p->setPen(QColor(0x77, 0x77, 0x77));
  p->setPen(QColor(eraseColor().red() / 2, eraseColor().green() / 2, eraseColor().blue() / 2));
  p->drawLine(x, y, x + w, y);
  p->drawLine(x, y, x, y + h);
  p->drawLine(x, y + h, x + w, y + h);
  p->drawLine(x + w, y, x + w, y + h);
  //p->setPen(QColor(0xd1, 0xd1, 0xd1));
  p->setPen(QColor(eraseColor().red() - 0x20, eraseColor().green() - 0x20, eraseColor().blue() - 0x20));
  p->drawLine(x, y + h + 1, x + w, y + h + 1);
  p->drawLine(x + w + 1, y, x + w + 1, y + h + 1);

  // clean corners
  p->setPen(viewport()->eraseColor());
  p->drawLine(x, y + h, x + 1, y + h);
  p->drawLine(x + w, y, x + w, y + 1);
  p->drawLine(x, y + h + 1, x + 2, y + h + 1);
  p->drawLine(x + w + 1, y, x + w + 1, y + 2);
  p->drawPoint(x + w + 1, y + h + 1);
  p->setPen(QColor(eraseColor().red() - 0x20, eraseColor().green() - 0x20, eraseColor().blue() - 0x20));
  p->drawPoint(x + 2, y + h);
  p->drawPoint(x + w, y + 2);
}


void QfrViewer::resizeEvent(QResizeEvent *e)
{
  m_timer.start(250, true);
  QScrollView::resizeEvent(e);
}


void QfrViewer::centerCanvas()
{
  int newX = 0;
  int newY = 0;

  QSize newViewportSize = viewportSize(m_canvas->width(), m_canvas->height());

  if(newViewportSize.width() > m_canvas->width())
    newX = (newViewportSize.width() - m_canvas->width()) / 2;
  if(newViewportSize.height() > m_canvas->height())
    newY = (newViewportSize.height() - m_canvas->height()) / 2;

  m_centerCanvas = true;
  moveChild(m_canvas, newX, newY);
  m_centerCanvas = false;
  viewport()->repaint(true);
}


void QfrViewer::delayedResizeEvent()
{
  emit resized();
  if(!m_centerCanvas)
    centerCanvas();
}


