#include <qprocess.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qstringlist.h>

#include "alarmsv.h"
#include "config.h"
#include "note.h"

extern Config *defCfg;

// ----------------- ALARM PROCESS --------------------
AlarmProcess::AlarmProcess(QString fileName) {
	QString cmd = defCfg->alarmCmd.stripWhiteSpace();
	bool launchCmd = false, gotPerc = false;
	QString parsedCmd;
	QStringList args;
	QString noteName = Note::loadNameFromFile(&fileName);
	
	if(!cmd.isEmpty()) {
		for(uint i = 0; i < cmd.length(); i++) {
			if(gotPerc) {
				switch(cmd.at(i).latin1()) {
					case 'n':
						parsedCmd += noteName;
						break;
					case 'f':
						parsedCmd += fileName;
						break;
					default:
						parsedCmd += cmd.at(i);
				}
				gotPerc = false;
			}
			else {
				if(cmd.at(i) == '%')
					gotPerc = true;
				else
					parsedCmd += cmd.at(i);
			}
		}
		
		args = QStringList::split(QChar(' '), parsedCmd.stripWhiteSpace());
		setArguments(args);
		connect(this, SIGNAL(processExited()), SLOT(exitHandler()));
		launchCmd = true;
	}
	
	if(defCfg->popupNote) {
		Note *n = Note::Open(&fileName);
		n->flash();
	}

	if(launchCmd) {
		if(!start())
			delete this;
	}
	else
		delete this;
}

AlarmProcess::~AlarmProcess() {
	defCfg->save();
}

void AlarmProcess::exitHandler() {
	delete this;
}

	
// ------------------ ALARM SERVER --------------------	
AlarmSv::AlarmSv() : QTimer() {
	update(true);
	connect(this, SIGNAL(timeout()), SLOT(handler()));
	start(1000);
}
	
void AlarmSv::update(bool initialCheck) {
	AlarmTimeStamp i, current;
	alarmList.clear();
	
	QDir browser(defCfg->prgPath, QString("note.*"), QDir::Name, QDir::Files);
	
	for (unsigned int j = 0; j < browser.count(); j++) {
		i = Note::loadTimeStampFromFile(&browser[j]);
		if(i.enabled) {
			alarmList.insert(std::make_pair(browser[j], i));
			if(initialCheck)
				if(histCheck(&current, i))
					new AlarmProcess(browser[j]);
		}
	}
}

void AlarmSv::handler() {
	QDate d(QDate::currentDate());
	QTime t(QTime::currentTime());
	
	QString dts = d.toString("yyyyddMM") + t.toString("hhmm");
	
	if(dts != dateTimeStamp && !alarmList.empty()) {
		std::multimap<QString, AlarmTimeStamp >::iterator it;
		for(it = alarmList.begin(); it != alarmList.end(); it++) {
			if((*it).second.year == 1970 || (*it).second.year == d.year()) {  // Year is correct
				if((*it).second.month == 0 || (*it).second.month == d.month()) { // Month -||-
					if((*it).second.day == -7 || (*it).second.day == d.day() ||  // Day #1
						((*it).second.day < 1 && (*it).second.day > -7 &&        // Day #2a
						  QDate::shortDayName((*it).second.day + 7) == d.shortDayName(d.dayOfWeek()))) {  // Day #2b
						if((*it).second.hour == -1 || (*it).second.hour == t.hour()) { // Hour
							if((*it).second.minute == -1 || (*it).second.minute == t.minute()) { // Minute is correct
								new AlarmProcess((*it).first);
							}
					 	}
					}
				}
			}
		}
		
		dateTimeStamp = dts;
	}
}

bool AlarmSv::histCheck(AlarmTimeStamp *c, AlarmTimeStamp i) {
	if(i.year == 1970) {
		for (int j = defCfg->dateTimeTag.year; j <= c->year; j++) {
			i.year = j;
			if (checkMonth(c, i))
				return true;
		}
	}
	else
		return checkMonth(c, i);
	
	return false;
}

bool AlarmSv::checkMonth(AlarmTimeStamp *c, AlarmTimeStamp i) {
	if(i.month == 0) {
		int start, end;
		if(i.year > defCfg->dateTimeTag.year)
			start = 1;
		else
			start = defCfg->dateTimeTag.month;
		
		if(i.year < c->year)
			end = 12;
		else
			end = c->month;
		
		for(int j = start; j <= end; j++) {
			i.month = j;
			if(checkDay(c, i))
				return true;
		}
	}
	else
		return checkDay(c, i);
	
	return false;
}

bool AlarmSv::checkDay(AlarmTimeStamp *c, AlarmTimeStamp i) {
	if(i.day < 1) {
		int start, end;
		
		if(i.month > defCfg->dateTimeTag.month)
			start = 1;
		else
			start = defCfg->dateTimeTag.day;
		
		if(i.month < c->month)
			end = QDate(i.year, i.month, 1).daysInMonth();
		else
			end = c->day;
		
		if(i.day > -7) {
			int diff = i.day + 7 - QDate(i.year, i.month, start).dayOfWeek();
			if(diff < 0)
				diff += 7;
			
			start += diff;
			for(int j = start; j <= end; j += 7) {
				i.day = j;
				if(checkHour(c, i))
					return true;
			}
		}
		else
			for(int j = start; j <= end; j++) {
				i.day = j;
				if(checkHour(c, i))
					return true;
			}
	}
	else
		return checkHour(c, i);
	
	return false;
}

bool AlarmSv::checkHour(AlarmTimeStamp *c, AlarmTimeStamp i) {
	if(i.hour == -1) {
		int start, end;
		if(i.day > defCfg->dateTimeTag.day)
			start = 0;
		else
			start = defCfg->dateTimeTag.hour;
		
		if(i.day < c->day)
			end = 23;
		else
			end = c->hour;
		
		for(int j = start; j <= end; j++) {
			i.hour = j;
			if(checkMinute(c, i))
				return true;
		}
	}
	else
		return checkMinute(c, i);
	
	return false;
}

bool AlarmSv::checkMinute(AlarmTimeStamp *c, AlarmTimeStamp i) {
	if(i.minute == -1) {
		int start, end;
		if(i.hour > defCfg->dateTimeTag.hour)
			start = 0;
		else
			start = defCfg->dateTimeTag.minute;
		
		if(i.hour < c->hour)
			end = 59;
		else
			end = c->minute;
		
		for(int j = start; j <= end; j++) {
			i.minute = j;
			if(checkDate(c, i))
				return true;
		}
	}
	else
		return checkDate(c, i);
	
	return false;
}

bool AlarmSv::checkDate(AlarmTimeStamp *c, AlarmTimeStamp i) {
	QDateTime i_dt = QDateTime(QDate(i.year, i.month, i.day), QTime(i.hour, i.minute));
	int diff1 = i_dt.secsTo(QDateTime(QDate(c->year, c->month, c->day), QTime(c->hour, c->minute)));
	int diff2 = i_dt.secsTo(QDateTime(QDate(defCfg->dateTimeTag.year, defCfg->dateTimeTag.month, defCfg->dateTimeTag.day), 
		QTime(defCfg->dateTimeTag.hour, defCfg->dateTimeTag.minute)));
	
	return ((diff1 >= 0) && (diff2 <= 0));
}
