#include <qtextstream.h>
#include <qfile.h>
#include <qmessagebox.h>

#include "config.h"
#include "dialogfonts.h"
#include "x11routines.h"

Config::Config() :
	QWidget(0, "Preferences", WType_Dialog | WStyle_DialogBorder | WShowModal),
	pSnotes(NULL) {
		setDefaults(this);
		setCaption("Sticky Notes - Global preferences");
}

void Config::show() {
	setupWidgets();
	
	setupData(this);

	adjustSize();
	
	QWidget::show();
	X11Rt::setClassName(this, QString("Config"), QString("Snotes"));
}

Config::~Config() {
	if(pSnotes)
		if(iconSavePos) {
			iconPosX = pSnotes->x();
			iconPosY = pSnotes->y();
		}

	save();
}	

void Config::hide() {
	QWidget::hide();

	save();
	
	delete layout;
	delete buttonOK;
	delete buttonApply;
	delete buttonDefaults;
	delete buttonCancel;
	delete tab1;
	delete tab2;
	delete tab3;
}

void Config::setupWidgets() {
		// Main layout
		layout = new QGridLayout(this, 2, 4, 5, 10);
		
		// Tab bar
		tabBar = new QTabWidget(this);
		layout->addMultiCellWidget(tabBar, 0, 0, 0, 3);
		
		// Set up tab 1: Icon
		tab1 = new tabIcon(tabBar);
		tabBar->addTab(tab1, "&Icon");
		
		// Set up tab 2: Notes
		tab2 = new tabNotes(tabBar);
		tabBar->addTab(tab2, "&Notes");
		tab2->hide();

		// Set up tab 3: Commands
		tab3 = new tabCommands(tabBar);
		tabBar->addTab(tab3, "Commands");
		tab3->hide();
		
		tabBar->resize(tabBar->sizeHint());
		
		// Set up OK, Apply, Defaults and Cancel
		buttonOK = new QPushButton(QString("&OK"), this);
		buttonApply = new QPushButton(QString("&Apply"), this);
		buttonDefaults = new QPushButton(QString("&Defaults"), this);
		buttonCancel = new QPushButton(QString("&Cancel"), this);
		buttonOK->resize(buttonOK->sizeHint());
		buttonApply->resize(buttonApply->sizeHint());
		buttonDefaults->resize(buttonDefaults->sizeHint());
		buttonCancel->resize(buttonCancel->sizeHint());
		layout->addWidget(buttonOK, 1, 0);
		layout->addWidget(buttonApply, 1, 1);
		layout->addWidget(buttonDefaults, 1, 2);
		layout->addWidget(buttonCancel, 1, 3);
		
		// Set up fonts
		buttonOK->setFont(font_normtext);
		buttonApply->setFont(font_normtext);
		buttonDefaults->setFont(font_normtext);
		buttonCancel->setFont(font_normtext);
		
		// Connect signals
		connect(buttonOK, SIGNAL(clicked()), SLOT(clickOK()));
		connect(buttonApply, SIGNAL(clicked()), SLOT(clickApply()));
		connect(buttonDefaults, SIGNAL(clicked()), SLOT(clickDefaults()));
		connect(buttonCancel, SIGNAL(clicked()), SLOT(clickCancel()));
		connect(tabBar, SIGNAL(currentChanged(QWidget *)), SLOT(polish()));
}

void Config::clickApply() {
	tab1->saveData();
	tab2->saveData();
	tab3->saveData();
	pSnotes->updateConfig();
}

void Config::setupData(Config *c) {
	tab1->loadData(c);
	tab2->loadData(c);
	tab3->loadData(c);
}

void Config::clickOK() {
	clickApply();
	hide();
}

void Config::clickDefaults() {
	Config *c = new Config();
	setDefaults(c);
	setupData(c);
	delete c;
}

void Config::clickCancel() {
	hide();
}

void Config::setDefaults(Config *obj) {
	obj->fontMenu = QFont("Arial", 12, 50, false);
	obj->fontBody = QFont("Arial", 10, 50, false);
	obj->fontTitle = QFont("Arial", 10, 75, false);
	obj->fontHeading = QFont("Arial", 12, 75, false);
	obj->noteWidth = 200;
	obj->noteHeight = 200;
	obj->iconPosX = -1;
	obj->iconPosY = -1;
	obj->colorBodyBg = QColor(255, 246, 0);
	obj->colorBodyFg = QColor(0, 0, 0);
	obj->colorTitleBg = QColor(222, 219, 0);
	obj->colorTitleFg = QColor(0, 0, 0);
	obj->iconSavePos = true;
	obj->iconSkipTaskbar= true;
	obj->iconSticky = true;
	obj->iconLayer = 1; // above
	obj->noteAutoIndent = true;
	obj->noteSkipTaskbar = true;
	obj->noteSticky = true;
	obj->noteLayer = 1; // above
	obj->noteTabSize = 4;
	obj->cmdPipe.clear();
	obj->cmdRun.clear();
	obj->cmdRun << "lp" << "mail" << "wc";
	obj->alarmCmd = "";
	obj->popupNote = true;
}

void Config::keyPressEvent (QKeyEvent *e) {
	if (e->ascii() == 27)
		buttonCancel->animateClick();
	else if (e->ascii() == 13)
		buttonOK->animateClick();
	else
		e->ignore();
}

bool Config::load() {
	QString iLine, value, valueString, key, section = "- none -";
	QStringList valueList;
	int i, lineNr = 0;
	bool parsed;

	Flags secFlag = FLAG_UNKNOWN;

	QFile file(prgPath + "config");
	
	if (!file.open(IO_ReadOnly)) {
		qWarning("Could not open file: %s.", file.name().ascii());
		return true;  // Not really any fault - just defaulting...
	}

	while (file.readLine(iLine, 1024) != -1) {
		lineNr++;
		parsed = false;
		iLine = iLine.stripWhiteSpace();
		if(!iLine.isEmpty()) {
			// Comments
			if(iLine.at(0) == '#')
				parsed = true;		
			// Flags
			else if(iLine.at(0) == '[' && iLine.at(iLine.length() - 1) == ']') {
				parsed = true;
				if(iLine.lower() == "[fonts]") {
					secFlag = FLAG_FONT;
					section = iLine;
					}
				else if (iLine.lower() == "[colors]") {
					secFlag = FLAG_COL;
					section = iLine;
					}
				else if (iLine.lower() == "[geometry]") {
					secFlag = FLAG_GEO;
					section = iLine;
					}
				else if (iLine.lower() == "[options]") {
					secFlag = FLAG_OPT;
					section = iLine;
					}
				else if (iLine.lower() == "[wmhints]") {
					secFlag = FLAG_WMHINTS;
					section = iLine;
					}
				else if (iLine.lower() == "[alarm]") {
					secFlag = FLAG_ALARM;
					section = iLine;
					}
				else if (iLine.lower() == "[commands]") {
					secFlag = FLAG_CMDS;
					section = iLine;
					cmdPipe.clear();  // Clear old commands
					cmdRun.clear();  // Clear old commands
					}
				else {
					qWarning("Unknown section: %s", iLine.ascii());
					secFlag = FLAG_UNKNOWN;
				}
			}

			// Key - vals
			else if ((i = iLine.find('=')) != -1) {
				key = iLine.left(i).stripWhiteSpace().lower();
				valueString = iLine.right(iLine.length() - i - 1).stripWhiteSpace();
				valueList = QStringList::split(QChar(' '), valueString);
				if(!(key.isEmpty() || valueString.isEmpty())) {
					switch(secFlag) {
						case FLAG_FONT:
							if (key == "menu")
								parsed = parseConfig(&valueString, &fontMenu);
							else if(key == "body")
								parsed = parseConfig(&valueString, &fontBody);
							else if(key == "title")
								parsed = parseConfig(&valueString, &fontTitle);
							else if(key == "heading")
								parsed = parseConfig(&valueString, &fontHeading);
							else
								parsed = false;

							break;

						case FLAG_COL:
							if (key == "titlefg")
								parsed = parseConfig(&valueString, &valueList, &colorTitleFg);
							else if (key == "titlebg")
								parsed = parseConfig(&valueString, &valueList, &colorTitleBg);
							else if(key == "bodyfg")
								parsed = parseConfig(&valueString, &valueList, &colorBodyFg);
							else if(key == "bodybg")
								parsed = parseConfig(&valueString, &valueList, &colorBodyBg);
							else
								parsed = false;

							break;

						case FLAG_GEO:
							if (key == "notewidth")
								parsed = parseConfig(&valueString, &noteWidth);
							else if(key == "noteheight")
								parsed = parseConfig(&valueString, &noteHeight);
							else if(key == "iconposx")
								parsed = parseConfig(&valueString, &iconPosX);
							else if(key == "iconposy")
								parsed = parseConfig(&valueString, &iconPosY);
							else
								parsed = false;

							break;
							
						case FLAG_OPT:
							if (key == "iconsavepos")
								parsed = parseConfig(&valueString, &iconSavePos);
							else if(key == "noteautoindent")
								parsed = parseConfig(&valueString, &noteAutoIndent);
							else if(key == "notetabsize")
								parsed = parseConfig(&valueString, &noteTabSize);
							else
								parsed = false;

							break;
							
						case FLAG_WMHINTS:
							if(key == "iconskiptaskbar")
								parsed = parseConfig(&valueString, &iconSkipTaskbar);
							else if(key == "iconsticky")
								parsed = parseConfig(&valueString, &iconSticky);
							else if(key == "noteskiptaskbar")
								parsed = parseConfig(&valueString, &noteSkipTaskbar);
							else if(key == "notesticky")
								parsed = parseConfig(&valueString, &noteSticky);
							else if(key == "iconlayer") {
								parsed = parseConfig(&valueString, &iconLayer);
								iconLayer = iconLayer & 3;
								if (iconLayer == 3)
									iconLayer = 0;
							}
							else if(key == "notelayer") {
								parsed = parseConfig(&valueString, &noteLayer);
								noteLayer = noteLayer & 3;
								if (noteLayer == 3)
									noteLayer = 0;
							}
							else
								parsed = false;

							break;
							
						case FLAG_ALARM:
							if (key == "alarmcmd") {
								parsed = true;
								alarmCmd = valueString;
							}
							else if (key == "popupnote")
								parsed = parseConfig(&valueString, &popupNote);
							else if (key == "datetimetag")
								parsed = parseConfig(&valueString, &dateTimeTag);
							else
								parsed = false;

							break;

						case FLAG_CMDS:
							if(key == "commandrun") {
								cmdRun << valueString;
								parsed = true;
							}
							else if(key == "commandpipe") {
								cmdPipe << valueString;
								parsed = true;
							}
							else
								parsed = false;
							
							break;
						case FLAG_UNKNOWN:
							parsed = true;
					}
				}
			}
			
			if (!parsed)
				qWarning("Could not parse \"%s\" in section %s line %d.", iLine.ascii(), section.ascii(), lineNr);
		}
	}
	
	file.close();

	cmdRun.sort();
	cmdPipe.sort();
	
	return true;
}

bool Config::save(){
	QFile file(prgPath + "config");
	
	dateTimeTag.update();  // Update our tag til current date/time
	
	if (file.open(IO_WriteOnly)) {
		QTextStream stream( &file );
		
		//fonts
		stream << "[Fonts]\n";
		stream << "Heading" << " = " << fontHeading.toString() << "\n";
		stream << "Menu" << " = " << fontMenu.toString() << "\n";
		stream << "Title" << " = " << fontTitle.toString() << "\n";
		stream << "Body" << " = " << fontBody.toString() << "\n";
		
		stream << "\n";
		
		
		//colors
		stream << "[Colors]\n";
		stream << "TitleBg" << " = " << colorTitleBg.name() << "\n";
		stream << "TitleFg" << " = " << colorTitleFg.name() << "\n";
		stream << "BodyBg" << " = " << colorBodyBg.name() << "\n";
		stream << "BodyFg" << " = " << colorBodyFg.name() << "\n";
		
		stream << "\n";
		

		//geometry
		stream << "[Geometry]\n";
		stream << "IconPosX" << " = " << iconPosX << "\n";
		stream << "IconPosY" << " = " << iconPosY << "\n";
		stream << "NoteWidth" << " = " << noteWidth << "\n";
		stream << "NoteHeight" << " = " << noteHeight << "\n";

		
		stream << "\n";
		
		//options
		stream << "[Options]\n";
		stream << "IconSavePos" << " = " << (iconSavePos ? 1 : 0) << "\n";
		stream << "NoteAutoIndent" << " = " << (noteAutoIndent ? 1 : 0) << "\n";
		stream << "NoteTabSize" << " = " << noteTabSize << "\n";
		

		stream << "\n";


		//WmHints
		stream << "[WmHints]\n";
		stream << "IconLayer" << " = " << iconLayer << "\n";
		stream << "IconSkipTaskbar" << " = " << (iconSkipTaskbar ? 1 : 0) << "\n";
		stream << "IconSticky" << " = " << (iconSticky ? 1 : 0) << "\n";
		stream << "NoteLayer" << " = " << noteLayer << "\n";
		stream << "NoteSkipTaskbar" << " = " << (noteSkipTaskbar ? 1 : 0) << "\n";
		stream << "NoteSticky" << " = " << (noteSticky ? 1 : 0) << "\n";


		stream << "\n";


		//Alarm
		stream << "[Alarm]\n";
		if(!alarmCmd.isEmpty())
			stream << "AlarmCmd" << " = " << alarmCmd << "\n";
		stream << "PopupNote" << " = " << (popupNote ? 1 : 0) << "\n";
		stream << "DateTimeTag = " << dateTimeTag.year << ":" <<
									  dateTimeTag.month << ":" <<
									  dateTimeTag.day << ":" <<
									  dateTimeTag.hour << ":" <<
									  dateTimeTag.minute << "\n";

		stream << "\n";

		// Commands
		stream << "[Commands]\n";
		for(QStringList::iterator it = cmdRun.begin(); it != cmdRun.end(); it++)
			stream << "CommandRun" << " = " << *it << "\n";
		for(QStringList::iterator it = cmdPipe.begin(); it != cmdPipe.end(); it++)
			stream << "CommandPipe" << " = " << *it << "\n";

		file.close();
	}
	else {
		qWarning("Could not write config file: %s.", file.name().ascii());
		return false;
	}
	
	return true;
}

bool Config::parseConfig(QString * inp, bool * out) {
	*out = (*inp == "true" || *inp == "1");
	
	return true;

}

bool Config::parseConfig(QString * inp, QStringList * list, QColor * out) {
	int i;
	bool parsed, ok1, ok2, ok3;
	
	if ((i = list->count()) == 1) {
		if (parsed = (inp->at(0) == '#' && inp->length() == 7))
			out->setRgb((inp->mid(1,2)).toUInt(&ok1, 16),
						(inp->mid(3,2)).toUInt(&ok2, 16),
						(inp->mid(5,2)).toUInt(&ok3, 16));
	}
	else if(parsed = (i == 3))
		out->setRgb((*list->at(0)).toUInt(&ok1),
					(*list->at(1)).toUInt(&ok2),
					(*list->at(2)).toUInt(&ok3));
	
	return parsed && ok1 && ok2 && ok3;
}

bool Config::parseConfig(QString *inp, QFont *out) {
	return out->fromString(*inp);
}

bool Config::parseConfig(QString *inp, int *out) {
	*out = inp->toInt();
	
	return true;
}

bool Config::parseConfig(QString *inp, QWidget *out) {
	QStringList i = QStringList::split(QChar(':'), *inp);
	if(i.count() == 4) {
		out->move((*i.at(0)).toInt(), (*i.at(1)).toInt());
		out->resize((*i.at(2)).toInt(), (*i.at(3)).toInt());
		return true;
	}
	
	return false;
}

bool Config::parseConfig(QString *inp, AlarmTimeStamp *out) {
	QStringList i = QStringList::split(QChar(':'), *inp);
	if(i.count() == 5) {
		out->year = (*i.at(0)).toInt();
		out->month = (*i.at(1)).toInt();
		out->day = (*i.at(2)).toInt();
		out->hour = (*i.at(3)).toInt();
		out->minute = (*i.at(4)).toInt();
		
		return true;
	}
	
	return false;
}

void Config::setpSnotes(Snotes *p) {
	pSnotes = p;
	reparent(p, getWFlags(), pos(), isShown());
}

void Config::polish() {
	QWidget::polish();
	adjustSize();
}
