#include <qfile.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qclipboard.h>
#include <qpainter.h>
#include <qbrush.h>
#include <qrect.h>
#include <qapplication.h>
#include <qcursor.h>

#include "note.h"
#include "pipe.h"
#include "x11routines.h"
#include "alarmcl.h"
#include "alarmsv.h"

Note *Note::first = NULL;
Note *Note::last = NULL;
bool Note::appIsClosing = false;

extern AlarmSv *alarmServer;
extern Config *defCfg;
extern const char *xpm_main_icon[];

const QString CONTENT_TAG = QString("--- BEGIN ---");
const int borderWidth = 2;

Note::Note(QString *filename, QString *newName, bool dataFromClipboard, QString content) : 
	QWidget(defCfg->pSnotes, *filename, WDestructiveClose | WGroupLeader | WType_TopLevel | WStyle_Customize | WStyle_NoBorder),
	flashCounter(0),
	myConfig(new NoteConfig (this, filename)),
	body(new TextEdit(this)),
	mouseOffset(QPoint()),
	windowOffset(QRect()),
	flagNoSave(false), 
	flagMouseDrag(Lifted) {

	setIcon(QPixmap(xpm_main_icon));
	X11Rt::setClassName(this, QString("Note"), QString("Snotes"));
	
	if(newName) {  // Create new
		myConfig->name = *newName;
		if (dataFromClipboard) { // Copy data from clipboard
			QString cb = QApplication::clipboard()->text(QClipboard::Selection);  // First try mouse selection
			
			if (cb.isNull())
				cb = QApplication::clipboard()->text(QClipboard::Clipboard);   // Then try global clipboard
			
			if (!cb.isNull())
				body->setText(cb);
		}
		else if(!content.isEmpty())
			body->setText(content);
		
		setGeometry (x(), y(), defCfg->noteWidth, defCfg->noteHeight);
		
		saveData();
	}
	else  // open existing
		loadData();
	
	menu = new NoteMenu(this);

	widgetLayout = new QGridLayout(this, 2, 2, borderWidth, 0);
	title = new QLabel(myConfig->name, this);
	closeButton = new buttonClose(this);

	setFocusPolicy(QWidget::StrongFocus);
	setFocusProxy(body);
	setMouseTracking(true);

	widgetLayout->addWidget(title, 0, 0, AlignCenter);
	widgetLayout->addWidget(closeButton, 0, 1, AlignHCenter | AlignTop);
	widgetLayout->addMultiCellWidget(body, 1, 1, 0, 1);
	widgetLayout->setColSpacing(1, 12);
	widgetLayout->setRowSpacing(0, 10);
	widgetLayout->setColStretch(0, 1);

	updateConfig();
	
	setCaption(myConfig->name);
	show();

	X11Rt::setWmFlags(this, myConfig->layer, myConfig->sticky, myConfig->skipTaskbar);
	
	connect(&flashTimer, SIGNAL(timeout()), SLOT(flash()));
}

Note::~Note() {
	if (!flagNoSave)  // Just deleted the note file, and we do not
		saveData();   // want it back...

	delete menu;
	delete title;
	delete closeButton;
	delete body;
	delete widgetLayout;
	delete myConfig;

	if(first == this)
		first = next;
	
	if(last == this)
		last = prev;
	
	if(prev)
		prev->next = next;
		
	if (next)
		next->prev = prev;
}

void Note::updateConfig() {
	title->setFont(myConfig->fontTitle);
	body->setFont(myConfig->fontBody);
	body->setTabStopWidth(myConfig->tabSize * myConfig->fontBody.pointSize());
	
	menu->updateFonts();

	body->autoIndent = myConfig->autoIndent;

	if(isVisible())
		X11Rt::setWmFlags(this, myConfig->layer, myConfig->sticky, myConfig->skipTaskbar);

	updateColors();
}

void Note::updateColors() {
	QColor bgCol(body->hasFocus() ? 
		myConfig->colorTitleBg : myConfig->colorBodyBg);
	
	//X11Rt::setBackgroundPixmap(body, myConfig->colorBodyBg);
	body->setPaper(myConfig->colorBodyBg);
	body->setPaletteForegroundColor(myConfig->colorBodyFg);
	setPaletteBackgroundColor (bgCol);
	title->setBackgroundColor(bgCol);
	title->setPaletteForegroundColor(myConfig->colorTitleFg);
	closeButton->updateColors(&myConfig->colorTitleFg, &bgCol);

	update();
}

void Note::updateAll() {
	Note *p = first;

	while (p) {
		p->updateFromDefaults();
		p = p->next;
	}
}

void Note::updateFromDefaults() {
	// Fonts
	if (myConfig->flagDefFonts) {  // flagDefFonts
		myConfig->fontTitle = defCfg->fontTitle;
		myConfig->fontBody = defCfg->fontBody;
	}
	
	// Colors
	if (myConfig->flagDefCols) {  // flagDefCols
		myConfig->colorTitleFg = defCfg->colorTitleFg;
		myConfig->colorTitleBg = defCfg->colorTitleBg;
		myConfig->colorBodyFg= defCfg->colorBodyFg;
		myConfig->colorBodyBg= defCfg->colorBodyBg;
	}
	
	// Extra options
	if (myConfig->flagDefOpt) {  // flagDefOpt
		myConfig->tabSize = defCfg->noteTabSize;
		myConfig->autoIndent = defCfg->noteAutoIndent;
	}
	
	// window manager hints
	if(myConfig->flagDefWmHints) {
		myConfig->skipTaskbar = defCfg->noteSkipTaskbar;
		myConfig->sticky = defCfg->noteSticky;
		myConfig->layer = defCfg->noteLayer;
	}
	
	
	updateConfig();
}

Note * Note::Open(QString *filename, QString *newName, bool dataFromClipboard, QString content) {
	Note *pNote;

	if(newName)  //Create new
		pNote = new Note(filename, newName, dataFromClipboard, content);
	else {  // Open existing
		if (Note::findNote(filename, &pNote)) {
			pNote->setActiveWindow();
			pNote->raise();
			return pNote;
		}
		
		pNote = new Note(filename);
	}


	if (first) {
		// Set up pointers
		last->next = pNote;
		pNote->next = NULL;
		pNote->prev = last;
		last = pNote;
		
	}
	else {
		// Set up pointers
		first = last = pNote;
		last->prev = last->next = NULL;
	}
	
	return pNote;
}

void Note::New(bool dataFromClipboard) {
	int n = 0;
	QString name, filename;
	
	while(QFile::exists (defCfg->prgPath + "note." + QString::number(n)))
		n++;
	
	filename = "note." + QString::number(n);
    name = QDateTime::currentDateTime().toString(QString("yyyy.MM.dd hh:mm"));

	Note::Open(&filename, &name, dataFromClipboard);
	
	defCfg->pSnotes->updateNotesList();
}

void Note::New(QString name, const QString &text) {
	int n = 0;
	QString filename;
	
	while(QFile::exists (defCfg->prgPath + "note." + QString::number(n)))
		n++;
	
	filename = "note." + QString::number(n);
    name = QDateTime::currentDateTime().toString(QString("yyyy.MM.dd hh:mm")) + 
		QString(" [") + name + QString("]");

	Note::Open(&filename, &name, false, text);
	
	defCfg->pSnotes->updateNotesList();
}

void Note::loadData() {
	QString iLine, value, valueString, key;
	QStringList valueList;
	int i, lineNr = 0;
	bool parsed;

	
	QFile file(defCfg->prgPath + myConfig->filename);
	
	if (!file.open(IO_ReadOnly)) {
		qWarning("Could not open file: %s.", file.name().ascii());
		return;  // Not really any fault - just defaulting...
	}
	
	// Read config
	while (file.readLine(iLine, 1024) != -1) {
		iLine = iLine.stripWhiteSpace();
		lineNr++;
		parsed = false;
		
		if(!iLine.isEmpty()) {
			if (iLine == CONTENT_TAG)  // Start of note content
				break;
			else if(iLine.at(0) == '#')  // Comments
				parsed = true;		
			else if ((i = iLine.find('=')) != -1) {
				key = iLine.left(i).stripWhiteSpace().lower();
				valueString = iLine.right(iLine.length() - i - 1).stripWhiteSpace();
				valueList = QStringList::split(QChar(' '), valueString);
				if(!(key.isEmpty() || valueString.isEmpty())) {
					if (key == "name") {
						myConfig->name = valueString;
						parsed = true;
					}
					else if(key == "openasdefault")
							parsed = true;
					else if (key == "geometry")
						parsed = Config::parseConfig(&valueString, this);
					else if (key == "fonttitle") {
						parsed = Config::parseConfig(&valueString, &myConfig->fontTitle);
						myConfig->flagDefFonts = false;
					}
					else if (key == "fontbody") {
						parsed = Config::parseConfig(&valueString, &myConfig->fontBody);
						myConfig->flagDefFonts = false;
					}						
					else if (key == "colortitlefg") {
						parsed = Config::parseConfig(&valueString, &valueList, &myConfig->colorTitleFg);
						myConfig->flagDefCols = false;
					}
					else if (key == "colortitlebg") {
						parsed = Config::parseConfig(&valueString, &valueList, &myConfig->colorTitleBg);
						myConfig->flagDefCols = false;
					}
					else if (key == "colorbodyfg") {
						parsed = Config::parseConfig(&valueString, &valueList, &myConfig->colorBodyFg);
						myConfig->flagDefCols = false;
					}
					else if (key == "colorbodybg") {
						parsed = Config::parseConfig(&valueString, &valueList, &myConfig->colorBodyBg);
						myConfig->flagDefCols = false;
					}
					else if (key == "tabsize") {
						parsed = Config::parseConfig(&valueString, &myConfig->tabSize);
						myConfig->flagDefOpt = false;
					}
					else if (key == "autoindent") {
						parsed = Config::parseConfig(&valueString, &myConfig->autoIndent);
						myConfig->flagDefOpt = false;
					}
					else if (key == "skiptaskbar") {
						parsed = Config::parseConfig(&valueString, &myConfig->skipTaskbar);
						myConfig->flagDefWmHints = false;
					}
					else if (key == "sticky") {
						parsed = Config::parseConfig(&valueString, &myConfig->sticky);
						myConfig->flagDefWmHints = false;
					}
					else if (key == "layer") {
						parsed = Config::parseConfig(&valueString, &myConfig->layer);
						myConfig->layer = myConfig->layer & 3;
						if (myConfig->layer == 3)
							myConfig->layer = 0;
						myConfig->flagDefWmHints = false;
					}
					else if (key == "alarmenable")
						parsed = Config::parseConfig(&valueString, &myConfig->alarmData.enabled);
					else if (key == "alarmdate")
						parsed = Config::parseConfig(&valueString, &myConfig->alarmData);
				}
			}
			
			if (!parsed)
				qWarning("Could not parse \"%s\", line %d.", iLine.ascii(), lineNr);
		}
	}

	// Read content of file
	while (file.readLine(iLine, 1024) != -1)
		body->append(iLine);

	file.close();
	
	// Scroll to top of note
	body->setCursorPosition(0,0);
}

void Note::saveData() {
	QFile file(defCfg->prgPath + myConfig->filename);
	
	if (file.open(IO_WriteOnly)) {
		QTextStream stream( &file );
		
		// Save name
		stream << "Name = " << myConfig->name << "\n";
		
		// Write wether to open on startup
		// If app is closing af we are here (=visible) we should open next time
		// the app is started
		stream << "OpenAsDefault = " << (appIsClosing ? 1 : 0) << "\n";
		
		// Write alarm data
		stream << "AlarmEnable = " << (myConfig->alarmData.enabled ? 1 : 0) << "\n";
		stream << "AlarmDate = " << myConfig->alarmData.year << ":" <<
									myConfig->alarmData.month << ":" <<
									myConfig->alarmData.day << ":" <<
									myConfig->alarmData.hour << ":" <<
									myConfig->alarmData.minute << "\n";

		// Save geometry
		stream << "Geometry = " << pos().x() << ":" <<
			pos().y() << ":" <<
			frameGeometry().width() << ":" <<
			frameGeometry().height() << "\n";
		
		// Save fonts
		if (!myConfig->flagDefFonts)
			stream << "FontTitle = " << myConfig->fontTitle.toString() << "\n" <<
				"FontBody = " << myConfig->fontBody.toString() << "\n";
		
		// Save colors
		if (!myConfig->flagDefCols)
			stream << "ColorTitleFg = " << myConfig->colorTitleFg.name() << "\n" <<
				"ColorTitleBg = " << myConfig->colorTitleBg.name() << "\n" <<
				"ColorBodyFg = " << myConfig->colorBodyFg.name() << "\n" <<
				"ColorBodyBg = " << myConfig->colorBodyBg.name() << "\n";


		// Save options
		if (!myConfig->flagDefOpt)
			stream << "TabSize = " << myConfig->tabSize << "\n" <<
				"AutoIndent = " << (myConfig->autoIndent ? 1 : 0) << "\n";

		// Save window manager hints		
		if(!myConfig->flagDefWmHints)
			stream << "SkipTaskbar = " << (myConfig->skipTaskbar ? 1 : 0) << "\n" <<
				"Sticky = " << (myConfig->sticky ? 1 : 0) << "\n" <<
				"Layer = " << myConfig->layer << "\n";


		// Save content
		stream << CONTENT_TAG << "\n";
		stream << body->text();  // Note - no newline

		file.close();
	}
	else
		qWarning("Could not write notefile file: %s.", myConfig->filename.ascii());
}

QString Note::loadNameFromFile(const QString *filename) {
	QString iLine, value, valueString, key, retVal = *filename;
	int i;

	QFile file(defCfg->prgPath + (*filename));
	
	if (!file.open(IO_ReadOnly)) {
		qWarning("Could not open file: %s.", file.name().ascii());
		return "Could not open file!";  // Not really any fault - just defaulting...
	}
	
	while (file.readLine(iLine, 1024) != -1) {
		iLine = iLine.stripWhiteSpace();
		if(!iLine.isEmpty()) {
			if (iLine == "--- BEGIN ---")
				break;
			else if ((i = iLine.find('=')) != -1) {
				key = iLine.left(i).stripWhiteSpace().lower();
				valueString = iLine.right(iLine.length() - i - 1).stripWhiteSpace();
				
				if(!(key.isEmpty() || valueString.isEmpty())) {
					if (key == "name") {
						retVal = valueString;
						break;
					}
				}
			}
		}
	}

	file.close();

	return retVal;
}

AlarmTimeStamp Note::loadTimeStampFromFile(const QString *filename) {
	QString iLine, value, valueString, key;
	AlarmTimeStamp retVal;
	int i;
	bool gotEnabled = false, gotDate = false;

	QFile file(defCfg->prgPath + (*filename));
	
	if (!file.open(IO_ReadOnly)) {
		qWarning("Could not open file: %s.", file.name().ascii());
		return retVal;  // Not really any fault - just defaulting...
	}
	
	while (file.readLine(iLine, 1024) != -1) {
		iLine = iLine.stripWhiteSpace();
		if(!iLine.isEmpty()) {
			if (iLine == "--- BEGIN ---")
				break;
			else if ((i = iLine.find('=')) != -1) {
				key = iLine.left(i).stripWhiteSpace().lower();
				valueString = iLine.right(iLine.length() - i - 1).stripWhiteSpace();
				
				if(!(key.isEmpty() || valueString.isEmpty())) {
					if (key == "alarmenable") {
						Config::parseConfig(&valueString, &retVal.enabled);
						gotEnabled = true;
					}
					else if (key == "alarmdate") {
						Config::parseConfig(&valueString, &retVal);
						gotDate = true;
					}
					
					if(gotDate && gotEnabled)
						break;
				}
			}
		}
	}

	file.close();
	
	return retVal;
}

void Note::resizeEvent (QResizeEvent *) {
	updateGeometry();
}

void Note::closeAll() {
	appIsClosing = true;
	
	Note * pNote = first;
	Note * npNote;

	while (pNote) {
		npNote = pNote->next;
		pNote->QWidget::close();
		pNote = npNote;
	}
}

void Note::openNotes() {
	QDir browser(defCfg->prgPath, QString("note.*"), QDir::Name, QDir::Files);
	
	for (unsigned int j = 0; j < browser.count(); j++)
		if(openOnStart(&browser[j]))
			Note::Open(&browser[j]);
}

bool Note::openOnStart(QString *filename) {
	QString iLine, value, valueString, key;
	int i;
	bool retVal = false;

	
	QFile file(defCfg->prgPath + (*filename));
	
	if (!file.open(IO_ReadOnly)) {
		qWarning("Could not open file: %s.", file.name().ascii());
		return false;  // Not really any fault - just defaulting...
	}
	
	while (file.readLine(iLine, 1024) != -1) {
		iLine = iLine.stripWhiteSpace();
		if(!iLine.isEmpty()) {
			if (iLine == "--- BEGIN ---")
				break;
			else if ((i = iLine.find('=')) != -1) {
				key = iLine.left(i).stripWhiteSpace().lower();
				valueString = iLine.right(iLine.length() - i - 1).stripWhiteSpace();
				
				if(!(key.isEmpty() || valueString.isEmpty())) {
					if (key == "openasdefault") {
						Config::parseConfig(&valueString, &retVal);
						break;
					}
				}
			}
		}
	}

	file.close();

	return retVal;
}

bool Note::findNote(const QString *filename, Note **ret) {
	*ret = first;

	while (*ret) {
		if ((*ret)->myConfig->filename == *filename)
			return true;

		*ret = (*ret)->next;
	}
	
	return false;
}

QString Note::getName() {
	return myConfig->name;
}

bool Note::getAutoIndent() {
	return myConfig->autoIndent;
}

mouseDragFlags Note::getMouseMode(QMouseEvent *e) {
	/* This function assumes that mouse is inside widget
	** It is only called from mouse-events (below) */
	
	int cornerSize = borderWidth;
	
	if (width() < cornerSize * 2)
		cornerSize = width() / 2;
	
	if((e->x() < cornerSize) && (e->y() < cornerSize))  // Top left
		return ResizeTL;
	else if((e->x() >= (width() - cornerSize)) && (e->y() < cornerSize))  // Top right
		return ResizeTR;
	else if((e->x() < cornerSize) && (e->y() >= (height() - cornerSize)))  // Bottom left
		return ResizeBL;
	else if((e->x() >= (width() - cornerSize)) && (e->y() >= (height() - cornerSize)))  // Bottom right
		return ResizeBR;
	else if(e->x() < cornerSize) // Left
		return ResizeL;
	else if(e->x() >= (width() - cornerSize)) // Right
		return ResizeR;
	else if(e->y() < cornerSize) // Top
		return ResizeT;
	else if(e->y() >= (height() - cornerSize)) // Bottom
		return ResizeB;
	else if(e->y() < (title->height() + borderWidth)) // Move
		return Move;
	else
		return Lifted;
}

void Note::flash() {
	if(!flashTimer.isActive())
		flashTimer.start(300);
	
	if(++flashCounter <= flashNum) {
		QColor bgCol(flashCounter % 2 == 0 ? 
			myConfig->colorTitleBg : myConfig->colorBodyBg);

		setPaletteBackgroundColor (bgCol);
		title->setBackgroundColor(bgCol);
		closeButton->updateColors(&myConfig->colorTitleFg, &bgCol);
	}
	else {
		flashTimer.stop();
		flashCounter = 0;
		updateColors();
	}
}
	

// ------------------------- Events -------------------

void Note::mousePressEvent(QMouseEvent *e) {
	if (e->button() == RightButton)
			menu->popup(e->globalPos());
	else if(e->button() == LeftButton) {
		flagMouseDrag = getMouseMode(e);
		mouseOffset = e->globalPos();
		windowOffset = frameGeometry();
	}
	else
		e->ignore();
}


void Note::mouseMoveEvent(QMouseEvent *e) {
	switch(flagMouseDrag) {
		case Lifted:
			switch(getMouseMode(e)) {
				case Lifted:
				case Move:
					setCursor(QCursor(ArrowCursor));
					break;
				case ResizeTL:
				case ResizeBR:
					setCursor(QCursor(SizeFDiagCursor));
					break;
				case ResizeTR:
				case ResizeBL:
					setCursor(QCursor(SizeBDiagCursor));
					break;
				case ResizeT:
				case ResizeB:
					setCursor(QCursor(SizeVerCursor));
					break;
				case ResizeL:
				case ResizeR:
					setCursor(QCursor(SizeHorCursor));
					break;
				}
			break;
		case Move:
			move(windowOffset.topLeft() - mouseOffset + e->globalPos());
			setCursor(QCursor(SizeAllCursor));
			break;
		case ResizeTL:
			resize(windowOffset.width() - e->globalX() + mouseOffset.x(),
				windowOffset.height() - e->globalY() + mouseOffset.y());
			move(windowOffset.right() - frameGeometry().width() + 1,
				windowOffset.bottom() - frameGeometry().height() + 1);
			break;
		case ResizeTR:
			resize(windowOffset.width() + e->globalX() - mouseOffset.x(),
				windowOffset.height() - e->globalY() + mouseOffset.y());
			move(windowOffset.left(),
				windowOffset.bottom() - frameGeometry().height() + 1);
			break;
		case ResizeBL:
			resize(windowOffset.width() - e->globalX() + mouseOffset.x(),
				windowOffset.height() + e->globalY() - mouseOffset.y());
			move(windowOffset.right() - frameGeometry().width() + 1,
				windowOffset.top());
			break;
		case ResizeBR:
			resize(windowOffset.width() + e->globalX() - mouseOffset.x(),
				windowOffset.height() + e->globalY() - mouseOffset.y());
			break;
		case ResizeT:
			resize(windowOffset.width(), 
				windowOffset.height() - e->globalY() + mouseOffset.y());
			move(windowOffset.left(), 
				windowOffset.bottom() - frameGeometry().height() + 1);
			break;
		case ResizeL:
			resize(windowOffset.width() - e->globalX() + mouseOffset.x(),
				windowOffset.height());
			move(windowOffset.right() - frameGeometry().width() + 1,
				windowOffset.top());
			break;
		case ResizeR:
			resize(windowOffset.width() + e->globalX() - mouseOffset.x(),
				windowOffset.height());
			move(windowOffset.left(), windowOffset.bottom() - height() + 1);
			break;
		case ResizeB:
			resize(windowOffset.width(), 
				windowOffset.height() + e->globalY() - mouseOffset.y());
			move(windowOffset.right() - width() + 1, windowOffset.top());
			break;
	}
}

void Note::mouseReleaseEvent(QMouseEvent *e) {
	if(e->button() == LeftButton)
		flagMouseDrag = Lifted;
		setCursor(QCursor(ArrowCursor));
}

void Note::mouseDoubleClickEvent (QMouseEvent * e) {
	if (e->button() == LeftButton) 
		rename();
	else
		e->ignore();
}

// -------------------------- SLOTS -----------------------------
void Note::insertDate() {
	body->insert(QDateTime::currentDateTime().toString(QString("yyyy.MM.dd hh:mm")));
}

void Note::pipe() {
	if(body->hasSelectedText())
		new Pipe(this, body->text(), body->selectedText());
	else 
		new Pipe(this, body->text());
}

void Note::showPrefs() {
	myConfig->show();
}

void Note::showAlarmCl() {
	new AlarmCl(this, myConfig);
}

void Note::rename() {
	bool ok;
	
	QString newName = QInputDialog::getText (QString("New name"), QString("Please enter a new name:"),
		QLineEdit::Normal, myConfig->name, &ok, this);
	
	if (ok) {
		newName = newName.stripWhiteSpace();
		if(newName.isEmpty())
			newName = myConfig->filename;
		
		myConfig->name = newName;
		title->setText(newName);
		setCaption(newName);
		saveData();
		defCfg->pSnotes->updateNotesList();
		menu->updateTitle(&newName);
	}
}

void Note::Delete() {
	int retVal = QMessageBox::warning(this, "Delete note?",
		"Are you sure you want to delete \"" + myConfig->name + "\"?\n"
		"This cannot be undone!",
		"Yes", "No", NULL ,
		1, 1);
	
	if (retVal == 0) {
		flagNoSave = true;  // Dont save data from destructor
		close();
		
		if (!QFile::remove (defCfg->prgPath + myConfig->filename))
			qWarning("An error occured, trying to delete file %s", myConfig->filename.ascii());
		
		defCfg->pSnotes->updateNotesList();
		alarmServer->update();
	}
}
