#include <qfont.h>
#include <qcolordialog.h>
#include <qfontdialog.h>

#include "noteconfig.h"
#include "dialogfonts.h"
#include "x11routines.h"

extern Config *defCfg;

NoteConfig::NoteConfig(Note *p, const QString *filename) :
	QWidget(p, "Configuration",	WType_Dialog | WStyle_DialogBorder | WShowModal),
	parent(p),
	name(*filename),
	filename(*filename),
	tabSize(defCfg->noteTabSize),
	fontTitle(defCfg->fontTitle), fontBody(defCfg->fontBody),
	colorBodyBg(defCfg->colorBodyBg), colorBodyFg(defCfg->colorBodyFg),
	colorTitleBg(defCfg->colorTitleBg), colorTitleFg(defCfg->colorTitleFg),
	autoIndent(defCfg->noteAutoIndent), skipTaskbar(defCfg->noteSkipTaskbar),
	sticky(defCfg->noteSticky), layer(defCfg->noteLayer),
	flagDefFonts(true), flagDefCols(true), flagDefOpt(true), flagDefWmHints(true) {
	X11Rt::setClassName(this, QString("NoteConfig"), QString("Snotes"));
}

void NoteConfig::show() {
	setupWidgets();
	
	// Fonts
	fontTitleViewer->setFont(fontTitle);
	fontTitleViewer->setText(fontTitle.family() + QString(", ") + QString::number(fontTitle.pointSize()));
	fontBodyViewer->setFont(fontBody);
	fontBodyViewer->setText(fontBody.family() + QString(", ") + QString::number(fontBody.pointSize()));
	
	// Colors
	colorSetTitleFg->setPaletteBackgroundColor(colorTitleFg);
	colorSetTitleBg->setPaletteBackgroundColor(colorTitleBg);
	colorSetBodyFg->setPaletteBackgroundColor(colorBodyFg);
	colorSetBodyBg->setPaletteBackgroundColor(colorBodyBg);
	
	// Extra options
	optionTabSize->setValue(tabSize);
	optionAutoIndent->setChecked(autoIndent);
	
	// window manager hints
	optionSkipTaskbar->setChecked(skipTaskbar);
	optionSticky->setChecked(sticky);
	switch(layer) {
		case 1:
			optionLayerAbove->setChecked(true);
			break;
		
		case 2:
			optionLayerBelow->setChecked(true);
			break;
		default:
			optionLayerNormal->setChecked(true);
	}
	
	// Enable defaults
	groupFonts->setChecked(!flagDefFonts);
	groupColors->setChecked(!flagDefCols);
	groupOptions->setChecked(!flagDefOpt);
	groupWmHints->setChecked(!flagDefWmHints);
	
	// Final things
	setCaption(QString("Configuration for note ") + name);
	
	resize(sizeHint());
	QWidget::show();
}

void NoteConfig::hide() {
	QWidget::hide();
	
	delete layout;
	delete groupFonts;
	delete groupColors;
	delete groupOptions;
	delete groupWmHints;
	delete buttonOK;
	delete buttonApply;
	delete buttonCancel;
}

void NoteConfig::setupWidgets() {
		// Main layout
		layout = new QGridLayout(this, 4, 4, 3, 6);
		layout->setColStretch(0, 1);
		layout->setColStretch(1, 1);
		layout->setColStretch(2, 4);
		layout->setColStretch(3, 1);
	
		// Group boxes
		groupFonts = new QGroupBox(QString("Fonts:"), this);
		groupColors = new QGroupBox(QString("Colors:"), this);
		groupOptions = new QGroupBox(QString("Other options:"), this);
		groupWmHints = new QGroupBox(QString("Window manager hints:"), this);
		groupFonts->setCheckable(true);
		groupColors->setCheckable(true);
		groupOptions->setCheckable(true);
		groupWmHints->setCheckable(true);
		groupFonts->setFont(font_groupHeading);
		groupColors->setFont(font_groupHeading);
		groupOptions->setFont(font_groupHeading);
		groupWmHints->setFont(font_groupHeading);
		layout->addMultiCellWidget(groupFonts, 0, 0, 0, 3);
		layout->addMultiCellWidget(groupColors, 1, 1, 0, 3);
		layout->addMultiCellWidget(groupOptions, 2, 2, 0, 1);
		layout->addMultiCellWidget(groupWmHints, 2, 2, 2, 3);
				
		// Font section setup
		layoutFonts = new QGridLayout (groupFonts, 3, 5, 5, 10);
		fontLabel1 = new QLabel("Title bar", groupFonts);
		fontLabel2 = new QLabel("Text area", groupFonts);
		fontLabel1->setFont(font_heading);
		fontLabel2->setFont(font_heading);
		fontTitleViewer = new QLabel(groupFonts);
		fontBodyViewer = new QLabel(groupFonts);
		fontSetTitle = new QPushButton("Select...", groupFonts);
		fontSetBody = new QPushButton("Select...", groupFonts);
		fontSetTitle->setFont(font_normtext);
		fontSetBody->setFont(font_normtext);
		layoutFonts->addMultiCellWidget(fontLabel1, 1, 1, 0, 1, AlignHCenter | AlignBottom);
		layoutFonts->addMultiCellWidget(fontLabel2, 1, 1, 3, 4, AlignHCenter | AlignBottom);
		layoutFonts->addWidget(fontSetTitle, 2, 0, AlignVCenter | AlignRight);
		layoutFonts->addWidget(fontTitleViewer, 2, 1, AlignVCenter | AlignLeft);
		layoutFonts->addWidget(fontSetBody, 2, 3, AlignVCenter | AlignRight);
		layoutFonts->addWidget(fontBodyViewer, 2, 4, AlignVCenter | AlignLeft);
		layoutFonts->setRowSpacing(0, 15);
		layoutFonts->setColSpacing(2, 15);
		layoutFonts->setColStretch(0, 1);
		layoutFonts->setColStretch(1, 1);
		layoutFonts->setColStretch(2, 5);
		layoutFonts->setColStretch(3, 1);
		layoutFonts->setColStretch(4, 1);
		layoutFonts->setRowStretch(0, 3);
		layoutFonts->setRowStretch(1, 1);
		layoutFonts->setRowStretch(2, 1);
		
		// Colors section setup
		layoutColors = new QGridLayout (groupColors, 4, 5, 5, 10);
		colorLabel1 = new QLabel("Title bar", groupColors);
		colorLabel2 = new QLabel("Text area", groupColors);
		colorLabel1->setFont(font_heading);
		colorLabel2->setFont(font_heading);
		colorLabel3 = new QLabel("Foreground:", groupColors);
		colorLabel4 = new QLabel("Background:", groupColors);
		colorLabel5 = new QLabel("Foreground:", groupColors);
		colorLabel6 = new QLabel("Background:", groupColors);
		colorLabel3->setFont(font_normtext);
		colorLabel4->setFont(font_normtext);
		colorLabel5->setFont(font_normtext);
		colorLabel6->setFont(font_normtext);
		colorSetTitleFg	= new QPushButton(groupColors);
		colorSetTitleBg = new QPushButton(groupColors);
		colorSetBodyFg = new QPushButton(groupColors);
		colorSetBodyBg = new QPushButton(groupColors);
		layoutColors->addMultiCellWidget(colorLabel1, 1, 1, 0, 1, AlignHCenter | AlignBottom);
		layoutColors->addMultiCellWidget(colorLabel2, 1, 1, 3, 4, AlignHCenter | AlignBottom);
		layoutColors->addWidget(colorLabel3, 2, 0, AlignVCenter | AlignRight);
		layoutColors->addWidget(colorLabel4, 3, 0, AlignVCenter | AlignRight);
		layoutColors->addWidget(colorLabel5, 2, 3, AlignVCenter | AlignRight);
		layoutColors->addWidget(colorLabel6, 3, 3, AlignVCenter | AlignRight);
		layoutColors->addWidget(colorSetTitleFg, 2, 1, AlignVCenter | AlignLeft);
		layoutColors->addWidget(colorSetTitleBg, 3, 1, AlignVCenter | AlignLeft);
		layoutColors->addWidget(colorSetBodyFg, 2, 4, AlignVCenter | AlignLeft);
		layoutColors->addWidget(colorSetBodyBg, 3, 4, AlignVCenter | AlignLeft);
		layoutColors->setRowSpacing(0, 15);
		layoutColors->setColSpacing(2, 15);
		layoutColors->setColStretch(0, 1);
		layoutColors->setColStretch(1, 1);
		layoutColors->setColStretch(2, 5);
		layoutColors->setColStretch(3, 1);
		layoutColors->setColStretch(4, 1);
		layoutColors->setRowStretch(0, 4);
		layoutColors->setRowStretch(1, 1);
		layoutColors->setRowStretch(2, 1);
		layoutColors->setRowStretch(3, 1);
		
		// Options section
		layoutOptions = new QGridLayout(groupOptions, 4, 2, 5, 10);
		optionLabel = new QLabel("Tab size:", groupOptions);
		optionTabSize = new QSpinBox(1, 256, 1, groupOptions);
		optionAutoIndent = new QCheckBox("Auto indent", groupOptions);
		optionLabel->setFont(font_normtext);
		optionTabSize->setFont(font_normtext);
		optionAutoIndent->setFont(font_normtext);
		optionTabSize->setMaximumWidth(50);
		layoutOptions->addWidget(optionLabel, 1, 0, AlignVCenter | AlignRight);
		layoutOptions->addWidget(optionTabSize, 1, 1, AlignVCenter | AlignLeft);
		layoutOptions->addMultiCellWidget(optionAutoIndent, 2, 2, 0, 1, AlignCenter);
		layoutOptions->setRowSpacing(0, 15);
		layoutOptions->setRowStretch(3, 1);
		layoutOptions->setColStretch(0, 1);
		layoutOptions->setColStretch(1, 1);
		layoutOptions->setColStretch(2, 1);
		
		// Window manager hints
		layoutWmHints = new QGridLayout(groupWmHints, 3, 2, 5, 10);
		optionSkipTaskbar = new QCheckBox("Skip taskbar", groupWmHints);
		optionSticky = new QCheckBox("Sticky", groupWmHints);
		groupLayer = new QButtonGroup(3, Vertical, "Layer:", groupWmHints);
		optionLayerAbove = new QRadioButton ("Above", groupLayer);
		optionLayerNormal = new QRadioButton("Normal", groupLayer);
		optionLayerBelow = new QRadioButton ("Below", groupLayer);
		groupLayer->setFont(font_groupHeading);
		optionLayerAbove->setFont(font_normtext);
		optionLayerNormal->setFont(font_normtext);
		optionLayerBelow->setFont(font_normtext);
		optionSkipTaskbar->setFont(font_normtext);
		optionSticky->setFont(font_normtext);
		layoutWmHints->setRowSpacing(0, 15);
		layoutWmHints->addMultiCellWidget(groupLayer, 1, 2, 0, 0);
		layoutWmHints->addWidget(optionSkipTaskbar, 1, 1);
		layoutWmHints->addWidget(optionSticky, 2, 1);

		// Ok, Cancel and Apply buttons
		buttonOK = new QPushButton(QString("&Ok"), this);
		buttonApply = new QPushButton(QString("&Apply"), this);
		buttonCancel = new QPushButton(QString("&Cancel"), this);
		buttonOK->setFont(font_normtext);
		buttonApply->setFont(font_normtext);
		buttonCancel->setFont(font_normtext);
		layout->addWidget(buttonOK, 3, 0);
		layout->addWidget(buttonApply, 3, 1);
		layout->addWidget(buttonCancel, 3, 3);
		
		// Connect signals
		connect(buttonOK, SIGNAL(clicked()), SLOT(clickOK()));
		connect(buttonApply, SIGNAL(clicked()), SLOT(clickApply()));
		connect(buttonCancel, SIGNAL(clicked()), SLOT(clickCancel()));
		connect(fontSetTitle, SIGNAL(clicked()), SLOT(showFontTitle()));
		connect(fontSetBody, SIGNAL(clicked()), SLOT(showFontBody()));
		connect(colorSetTitleFg, SIGNAL(clicked()), SLOT(showColorTitleFg()));
		connect(colorSetTitleBg, SIGNAL(clicked()), SLOT(showColorTitleBg()));
		connect(colorSetBodyFg, SIGNAL(clicked()), SLOT(showColorBodyFg()));
		connect(colorSetBodyBg, SIGNAL(clicked()), SLOT(showColorBodyBg()));
		connect(groupFonts, SIGNAL(toggled(bool)), SLOT(toggleDefFonts(bool)));
		connect(groupColors, SIGNAL(toggled(bool)), SLOT(toggleDefColors(bool)));
		connect(groupOptions, SIGNAL(toggled(bool)), SLOT(toggleDefOptions(bool)));
		connect(groupWmHints, SIGNAL(toggled(bool)), SLOT(toggleDefWmHints(bool)));
		
		// Set tab order
		setTabOrder(groupFonts, fontSetTitle);
		setTabOrder(fontSetTitle, fontSetBody);
		setTabOrder(fontSetBody, groupColors);
		setTabOrder(groupColors, colorSetTitleFg);
		setTabOrder(colorSetTitleFg, colorSetTitleBg);
		setTabOrder(colorSetTitleBg, colorSetBodyFg);
		setTabOrder(colorSetBodyFg, colorSetBodyBg);
		setTabOrder(colorSetBodyBg, groupOptions);
		setTabOrder(groupOptions, optionTabSize);
		setTabOrder(optionTabSize, optionAutoIndent);
		setTabOrder(optionAutoIndent, groupLayer);
		setTabOrder(groupLayer, optionSkipTaskbar);
		setTabOrder(optionSkipTaskbar, optionSticky);
}

void NoteConfig::keyPressEvent (QKeyEvent *e) {
	if (e->ascii() == 27)
		buttonCancel->animateClick();
	else if (e->ascii() == 13)
		buttonOK->animateClick();
	else
		e->ignore();
}

// -----------------------------  SLOTS  --------------------------------
void NoteConfig::clickOK() {
	clickApply();
	hide();
}

void NoteConfig::clickApply(){
	// Fonts
	if (flagDefFonts = !(groupFonts->isChecked())) {  // flagDefFonts
		fontTitle = defCfg->fontTitle;
		fontBody = defCfg->fontBody;
	}
	else {
		fontTitle = fontTitleViewer->font();
		fontBody = fontBodyViewer->font();
	}
	
	// Colors
	if (flagDefCols = !(groupColors->isChecked())) {  // flagDefCols
		colorTitleFg = defCfg->colorTitleFg;
		colorTitleBg = defCfg->colorTitleBg;
		colorBodyFg= defCfg->colorBodyFg;
		colorBodyBg= defCfg->colorBodyBg;
	}
	else {
		colorTitleFg = colorSetTitleFg->paletteBackgroundColor();
		colorTitleBg = colorSetTitleBg->paletteBackgroundColor();
		colorBodyFg= colorSetBodyFg->paletteBackgroundColor();
		colorBodyBg= colorSetBodyBg->paletteBackgroundColor();
	}

	
	// Extra options
	if (flagDefOpt = !(groupOptions->isChecked())) {  // flagDefOpt
		tabSize = defCfg->noteTabSize;
		autoIndent = defCfg->noteAutoIndent;
	}
	else {
		tabSize = optionTabSize->value();
		autoIndent = optionAutoIndent->isChecked();
	}

	// Window manager hints
	if (flagDefWmHints = !(groupWmHints->isChecked())) {  // flagDefWmHints
		skipTaskbar = defCfg->noteSkipTaskbar;
		sticky = defCfg->noteSticky;
		layer = defCfg->noteLayer;
	}
	else {
		skipTaskbar = optionSkipTaskbar->isChecked();
		sticky = optionSticky->isChecked();
		if (optionLayerAbove->isChecked())
			layer = 1;
		else if(optionLayerBelow->isChecked())
			layer = 2;
		else
			layer = 0;
	}
	
	parent->updateConfig();
}

void NoteConfig::clickCancel() {
	hide();
}

void NoteConfig::showFontTitle() {
	bool OK;  // Nescesary for QFontDialog - but not used.
	fontTitleViewer->setFont(QFontDialog::getFont(&OK,
		fontTitleViewer->font(), this));
	fontTitleViewer->setText(fontTitleViewer->font().family() + 
		QString(", ") +
		QString::number(fontTitleViewer->font().pointSize()));
}

void NoteConfig::showFontBody() {
	bool OK;  // Nescesary for QFontDialog - but not used.
	fontBodyViewer->setFont(QFontDialog::getFont(&OK,
		fontBodyViewer->font(), this));
	fontBodyViewer->setText(fontBodyViewer->font().family() + 
		QString(", ") +
		QString::number(fontBodyViewer->font().pointSize()));
}

void NoteConfig::showColorTitleFg() {
	QColor nColor(QColorDialog::getColor(colorSetTitleFg->paletteBackgroundColor(), this));
	if (nColor.isValid())
		colorSetTitleFg->setPaletteBackgroundColor(nColor);	
}

void NoteConfig::showColorTitleBg() {
	QColor nColor(QColorDialog::getColor(colorSetTitleBg->paletteBackgroundColor(), this));
	if (nColor.isValid())
		colorSetTitleBg->setPaletteBackgroundColor(nColor);	
}

void NoteConfig::showColorBodyFg() {
	QColor nColor(QColorDialog::getColor(colorSetBodyFg->paletteBackgroundColor(), this));
	if (nColor.isValid())
		colorSetBodyFg->setPaletteBackgroundColor(nColor);
}

void NoteConfig::showColorBodyBg() {
	QColor nColor(QColorDialog::getColor(colorSetBodyBg->paletteBackgroundColor(), this));
	if (nColor.isValid())
		colorSetBodyBg->setPaletteBackgroundColor(nColor);	
}

void NoteConfig::toggleDefFonts(bool i) {
	if(i) {
		fontTitleViewer->setFont(fontTitle);
		fontTitleViewer->setText(fontTitle.family() +
			QString(", ") +
			QString::number(fontTitle.pointSize()));
		fontBodyViewer->setFont(fontBody);
		fontBodyViewer->setText(fontBody.family() + 
		QString(", ") +	
		QString::number(fontBody.pointSize()));
	}
	else {
		fontTitleViewer->setFont(defCfg->fontTitle);
		fontTitleViewer->setText(defCfg->fontTitle.family() +
			QString(", ") +
			QString::number(defCfg->fontTitle.pointSize()));
		fontBodyViewer->setFont(defCfg->fontBody);
		fontBodyViewer->setText(defCfg->fontBody.family() + 
			QString(", ") +
			QString::number(defCfg->fontBody.pointSize()));
	}
}

void NoteConfig::toggleDefColors(bool i) {
	if(i) {
		colorSetTitleFg->setPaletteBackgroundColor(colorTitleFg);
		colorSetTitleBg->setPaletteBackgroundColor(colorTitleBg);
		colorSetBodyFg->setPaletteBackgroundColor(colorBodyFg);
		colorSetBodyBg->setPaletteBackgroundColor(colorBodyBg);		
	}
	else {
		colorSetTitleFg->setPaletteBackgroundColor(defCfg->colorTitleFg);
		colorSetTitleBg->setPaletteBackgroundColor(defCfg->colorTitleBg);
		colorSetBodyFg->setPaletteBackgroundColor(defCfg->colorBodyFg);
		colorSetBodyBg->setPaletteBackgroundColor(defCfg->colorBodyBg);
	}
}

void NoteConfig::toggleDefOptions(bool i) {
	if(i) {
		optionTabSize->setValue(tabSize);
		optionAutoIndent->setChecked(autoIndent);
	}
	else {
		optionTabSize->setValue(defCfg->noteTabSize);
		optionAutoIndent->setChecked(defCfg->noteAutoIndent);
	}
}

void NoteConfig::toggleDefWmHints(bool i) {
	if(i) {
		optionSkipTaskbar->setChecked(skipTaskbar);
		optionSticky->setChecked(sticky);
		switch(layer) {
			case 1:
				optionLayerAbove->setChecked(true);
				break;
			
			case 2:
				optionLayerBelow->setChecked(true);
				break;
			default:
				optionLayerNormal->setChecked(true);
		}
	}
	else {
		optionSkipTaskbar->setChecked(defCfg->noteSkipTaskbar);
		optionSticky->setChecked(defCfg->noteSticky);
		switch(defCfg->noteLayer) {
			case 1:
				optionLayerAbove->setChecked(true);
				break;
			
			case 2:
				optionLayerBelow->setChecked(true);
				break;
			default:
				optionLayerNormal->setChecked(true);
		}
	}
}
