#include <qinputdialog.h>
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qcstring.h>

#include "pipe.h"
#include "dialogfonts.h"
#include "x11routines.h"

extern Config *defCfg;

Pipe::Pipe(Note *p, QString bT, QString bST) :
	QWidget(p, "Pipe Command",	WType_Dialog | WStyle_DialogBorder | WShowModal), 
	parent(p),
	ok(false),
	bufText(new QString(bT)), 
	bufSelText(new QString(bST)) {
		setCaption("Pipe command...");
		X11Rt::setClassName(this, QString("PipeCmd"), QString("Snotes"));
		
		// Layout
		QGridLayout layout(this, 4, 2, 5, 15);
		
		// Widgets
		buttonOK = new QPushButton("OK", this);
		buttonCancel = new QPushButton("Cancel", this);
		listBox = new QComboBox(true, this);
		checkPipe = new QCheckBox("stdout to new note.", this);
		checkSelText = new QCheckBox("Use selected text", this);
		heading = new QLabel("Select or type in command:", this);
		
		// Fonts
		heading->setFont(font_heading);
		listBox->setFont(font_normtext);
		checkPipe->setFont(font_normtext);
		checkSelText->setFont(font_normtext);
		buttonCancel->setFont(font_normtext);
		buttonOK->setFont(font_normtext);
		
		// Config of widgets
		QStringList all(defCfg->cmdRun);
		for (QStringList::iterator it = defCfg->cmdPipe.begin(); it != defCfg->cmdPipe.end(); it++)
			all << (*it + QString(" [Pipe]"));
		all.sort();
		listBox->insertStringList(all);
		checkSelText->setEnabled(!bufSelText->isEmpty());
		checkSelText->setChecked(!bufSelText->isEmpty());
		
		// Connect signals
		connect(buttonOK, SIGNAL(clicked()), SLOT(clickOK()));
		connect(buttonCancel, SIGNAL(clicked()), SLOT(clickCancel()));
		connect(listBox, SIGNAL(activated(const QString&)), SLOT(textChanged(const QString&)));
		
		// Put widgets on layout
		layout.addMultiCellWidget(heading, 0, 0, 0, 1, AlignCenter);
		layout.addWidget(listBox, 1, 0);
		layout.addWidget(checkPipe, 1, 1);
		layout.addMultiCellWidget(checkSelText, 2, 2, 0, 1);
		layout.addWidget(buttonOK, 3, 0);
		layout.addWidget(buttonCancel, 3, 1, AlignRight);
		
		show();
}

Pipe::~Pipe() {
	if (ok) {
		delete process;
		delete stdOut;
		delete stdErr;
	}
	
	delete buttonOK;
	delete buttonCancel;
	delete listBox;
	delete checkPipe;
	delete checkSelText;
	delete bufText;
	delete bufSelText;
}

void Pipe::clickOK() {
	hide();
	
	if (!listBox->currentText().isEmpty()) {
		QString stdIn;
		
		ok = true;
		stdErr = new QString();
		stdOut = new QString();

		QStringList args = QStringList::split(QChar(' '), listBox->currentText().stripWhiteSpace());
		
		process = new QProcess(args);
		QObject::connect(process, SIGNAL(readyReadStdout()), this, SLOT(readStdOut()));
		QObject::connect(process, SIGNAL(readyReadStderr()), this, SLOT(readStdErr()));
		QObject::connect(process, SIGNAL(processExited()), this, SLOT(cmdFinished()));
		
		
		if(checkSelText->isChecked())
			stdIn = *bufSelText;
		else
			stdIn = *bufText;
		
		if(stdIn.at(stdIn.length() - 1) != '\n')
			stdIn += '\n';
		
		if(!process->launch(stdIn))
			QMessageBox::about(parent, QString("Command failure"), 
				QString("Could not launch command: ") + listBox->currentText() + QString("."));
	}
	else
		delete this;
}

void Pipe::clickCancel() {
	hide();
	delete this;
}

void Pipe::textChanged(const QString &t) {
	if (t.right(7) == " [Pipe]") {
		listBox->setEditText(t.left(t.length() - 7));
		checkPipe->setChecked(true);
	}
	else
		checkPipe->setChecked(false);
}

void Pipe::keyPressEvent (QKeyEvent *e) {
	if (e->ascii() == 27)
		buttonCancel->animateClick();
	else if (e->ascii() == 13)
		buttonOK->animateClick();
	else
		e->ignore();
}

void Pipe::readStdOut() {
	if (checkPipe->isChecked())
		*stdOut += process->readStdout();
	else
		*stdErr += process->readStdout();
}

void Pipe::readStdErr() {
	*stdErr += process->readStderr();
}

void Pipe::cmdFinished() {
	if (checkPipe->isChecked() && !stdOut->isEmpty())
		parent->New(listBox->currentText(), *stdOut);
	
	if(!checkPipe->isChecked() || !stdErr->isEmpty() || process->exitStatus() != 0) {
		QString mesg = QString("Command exited with exit status ") +
			QString::number(process->exitStatus());
	
		if (!stdErr->isEmpty())
			mesg +=QString(":\nOutput:\n") + *stdErr;
	
		QMessageBox::about(parent, QString("Command finished"), mesg);
	}
	
	delete this;
}
