/**************************************************************************
   ukopp - disk to disk backup and restore program

   Free software licensed under GNU General Public License v.2
   source URL:  kornelix.squarespace.com
***************************************************************************/

#include <dirent.h>
#include <fcntl.h>
#include "zfuncs.h"

#define ukopp_title "ukopp v.3.3.1"                                        //  version v.3.3.1
#define ukopp_license "GNU General Public License v.2"

//  parameters and limits

#define normfont "monospace"
#define boldfont "monospace bold"
#define vrcc  512*1024                                                     //  verify read I/O size
#define maxnx 100                                                          //  max include/exclude in job file
#define maxfs 200000                                                       //  max disk files
#define modtimetolr 1.0                                                    //  tolerance for "equal" file mod times
#define nano 0.000000001                                                   //  nanosecond
#define mega 1000000.0                                                     //  million
#define VSEP1 " ("                                                         //  file version appendage format:
#define VSEP2 ")"                                                          //     ...filename (nnn)

//  special control files in backup directory

#define BD_UKOPPDIRK  "/ukopp-data"                                        //  directory for special files
#define BD_POOPFILE   "/ukopp-data/poopfile"                               //  file owner & permissions file
#define BD_JOBFILE    "/ukopp-data/jobfile"                                //  backup job file
#define BD_DATETIME   "/ukopp-data/datetime"                               //  backup date-time file

//  GTK GUI widgets

GtkWidget      *mWin, *mVbox, *mScroll, *mLog;                             //  main window
GtkWidget      *fc_dialogbox, *fc_widget;                                  //  file-chooser dialog
GtkWidget      *editwidget;                                                //  edit box in file selection dialogs
PangoFontDescription    *monofont;                                         //  fixed-width font

//  file scope variables

int      killFlag;                                                         //  tell function to quit
int      pauseFlag;                                                        //  tell function to pause/resume
int      menuLock;                                                         //  menu lock flag
int      threadcount;                                                      //  count of running threads
int      Fdialog;                                                          //  dialog in progress
int      clrun;                                                            //  flag, command line 'run' command
char     subprocName[20];                                                  //  name of created subprocess

char     TFbakfiles[100];                                                  //  /home/user/.ukopp/xxx temp. files
char     TFjobfile[100], TFpoopfile[100];
char     TFdatetime[100], TFformatscript[100];

//  disk devices and mount points

char     diskdev[100][40];                                                 //  /dev/xxx
char     diskdesc[100][60];                                                //  device description
char     diskmp[100][60];                                                  //  mount point, /media/xxxx
int      Ndisk, maxdisk = 99;                                              //  max. disks / partitions

int      devMounted = 0;                                                   //  backup device mounted status
char     mountdev[40];                                                     //  current mount data
char     mountdirk[200];

//  backup job data

char     BJfilespec[maxfcc];                                               //  backup job file
int      BJnx;                                                             //  filespec count, 0...maxnx
char    *BJrec[maxnx];                                                     //    job record
int      BJrtype[maxnx];                                                   //    record type: retain, include ...
char    *BJfspec[maxnx];                                                   //    filespec (wild) in job record
int      BJfiles[maxnx];                                                   //    count of matching disk files
double   BJbytes[maxnx];                                                   //    matching files byte count
char     BJdev[40] = "";                                                   //  backup device (maybe)
char     BJdirk[200] = "";                                                 //  backup target directory
int      BJdcc;                                                            //  target directory cc
int      BJflush;                                                          //  flush method: 1=sync, 2=remount
int      BJvmode;                                                          //  verify: 0/1/2/3 = no/incr/full/comp
int      BJval;                                                            //  backup job valid flag

//  disk files included in backup job

struct dfrec {                                                             //  disk file record
   char    *file;                                                          //    file: /directory.../filename
   double   size;                                                          //    byte count
   double   mtime;                                                         //    mod time
   int      err;                                                           //    fstat() status
   int      jindx;                                                         //    index to job data BJrec[] etc.
   int      finc;                                                          //    flag, included in disk file >> backup
   char     disp;                                                          //    status: new mod unch
};

int      Dnf;                                                              //  actual file count < maxfs
double   Dbytes;                                                           //  disk files, total bytes
dfrec    Drec[maxfs];                                                      //  disk file data records

//  backup files (backup copies on backup medium)

struct   bfrec {                                                           //  backup file record
   char    *file;                                                          //    file: /directory.../filename
   double   size;                                                          //    byte count
   double   mtime;                                                         //    mod time
   int      err;                                                           //    fstat() status
   int      retND;                                                         //    retention days and versions
   int      retNV;                                                         //    (from backup job)
   int      lover, hiver;                                                  //    lover-hiver are previous versions
   int      expver;                                                        //    lover-expver are expired versions
   int      finc;                                                          //    flag, included in backup >> disk file
   char     disp;                                                          //    status: del mod unch
};

int      Bnf;                                                              //  actual file count < maxfs
double   Bbytes;                                                           //  backup files, total bytes
bfrec    Brec[maxfs];                                                      //  backup file data records
                                                                           //  backup file statistics:
int      Cfiles;                                                           //    curr. version file count
double   Cbytes;                                                           //       and total bytes
int      Vfiles;                                                           //    prior version file count
double   Vbytes;                                                           //       and total bytes
int      Pfiles;                                                           //    expired prior versions
double   Pbytes;                                                           //       and total bytes

//  disk::backup comparison data

int      nnew, ndel, nmod, nunc;                                           //  new, del, mod, unch file counts
int      Mfiles;                                                           //  new + mod + del file count
double   Mbytes;                                                           //  new + mod files, total bytes

//  restore job data

char     RJfrom[300];                                                      //  restore copy-from: /directory/.../
char     RJto[300];                                                        //  restore copy-to: /directory/.../
int      RJnx;                                                             //  filespec count, 0...maxnx
char    *RJrec[maxnx];                                                     //    include/exclude /directory.../*
int      RJrtype[maxnx];                                                   //    record type: include/exclude
char    *RJfspec[maxnx];                                                   //    filespec of include/exclude
int      RJval;                                                            //  restore job valid flag

//  restore file data 

struct   rfrec {                                                           //  restore file record
   char     *file;                                                         //  restore filespec: /directory.../file
   int      finc;                                                          //  flag, file restore was done
};

rfrec    Rrec[maxfs];                                                      //  restore file data records
int      Rnf;                                                              //  actual file count < maxfs

//  ukopp functions    

int initfunc(void *data);                                                  //  GTK init function
void buttonfunc(GtkWidget *, cchar *menu);                                 //  process toolbar button event
void menufunc(GtkWidget *, cchar *menu);                                   //  process menu select event
void * menu_thread_func(void *);                                           //  run menu function in a thread

int quit_ukopp(cchar *);                                                   //  exit application
int clearScreen(cchar *);                                                  //  clear logging window
int signalFunc(cchar *);                                                   //  kill/pause/resume curr. function
int checkKillPause();                                                      //  test flags: killFlag and pauseFlag

int BDpoop();                                                              //  get all devices and mount points
int chooseTarget(cchar *);                                                 //  choose device and mount point

int BJfileOpen(cchar *);                                                   //  job file open dialog
int BJfileSave(cchar *);                                                   //  job file save dialog
int BJload(cchar *fspec);                                                  //  backup job data <<< file
int BJstore(cchar *fspec);                                                 //  backup job data >>> file
int BJlist(cchar *);                                                       //  backup job >>> log window

int BJedit(cchar *);                                                       //  backup job edit dialog
int BJedit_compl(zdialog *zd, int zstat);                                  //  dialog completion function

int Backup(cchar *);                                                       //  backup function
int Synch(cchar *);                                                        //  synchronize function
int Verify(cchar *);                                                       //  verify functions
int Report(cchar *);                                                       //  report functions

int RJedit(cchar *);                                                       //  restore job edit dialog
int RJedit_compl(zdialog *zd, int zstat);                                  //  dialog completion function
int RJlist(cchar *);                                                       //  list backup files to be restored
int Restore(cchar *);                                                      //  file restore function

int Format(cchar *);                                                       //  format disk function
int helpFunc(cchar *);                                                     //  help function

cchar * parseJobrec(cchar *rec, int &typ, char *&fspec);                   //  parse/check job record
cchar * parseTarget(cchar *text);                                          //  parse backup target record
int mount(cchar *);                                                        //  mount target device
int unmount(cchar *);                                                      //  unmount target device
int flushcache();                                                          //  flush I/O memory buffers to device
int saveScreen(cchar *);                                                   //  save logging window to file
int writeDT();                                                             //  write date-time to temp file
int synch_poop(const char *mode);                                          //  synch owner and permissions data  v.26

int fc_dialog(cchar *dirk);                                                //  file chooser dialog
int fc_response(GtkDialog *, int, void *);                                 //  fc_dialog response

int dGetFiles();                                                           //  generate backup file list from job data
int bGetFiles();                                                           //  get backup file list
int rGetFiles();                                                           //  generate file list from restore job
int setFileDisps();                                                        //  set file disps: new del mod unch
int SortFileList(char *recs, int RL, int NR, char sort);                   //  sort file list in memory
int filecomp(cchar *file1, cchar *file2);                                  //  compare files, directories before files

int BJreset();                                                             //  reset backup job file data
int RJreset();                                                             //  reset restore job data
int dFilesReset();                                                         //  reset disk file data and free memory
int bFilesReset();                                                         //  reset backup file data, free memory
int rFilesReset();                                                         //  reset restore file data and free memory

cchar * copyFile(cchar *file1, cchar *file2, int mpf);                     //  copy backup file << >> disk file
cchar * deleteFile(cchar *file);                                           //  delete backup file
cchar * checkFile(cchar *file, int compf, double &bcc);                    //  validate file and return length
cchar * renameFile(bfrec &);                                               //  rename backup file: assign next version
cchar * purgeFile(bfrec &);                                                //  purge backup file: delete expired vers.

int setFileVersion(char *file, int vers);                                  //  set/change filespec version in memory
int do_shell(cchar *pname, cchar *command);                                //  do shell command and echo outputs 

//  ukopp menu table

struct menuent {
   char     menu1[20], menu2[40];                                          //  top-menu, sub-menu
   int      lock;                                                          //  lock funcs: no run parallel
   int      thread;                                                        //  run in thread
   int      (*mfunc)(cchar *);                                             //  processing function
};

#define nmenu  37
struct menuent menus[nmenu] = {
//  top-menu    sub-menu                lock  thread  menu-function
{  "button",   "edit job",                0,    0,    BJedit         },
{  "button",   "target",                  0,    0,    chooseTarget   },
{  "button",   "clear",                   0,    0,    clearScreen    },
{  "button",   "run job",                 1,    1,    Backup         },
{  "button",   "mount",                   1,    0,    mount          },
{  "button",   "unmount",                 1,    0,    unmount        },
{  "button",   "pause",                   0,    0,    signalFunc     },
{  "button",   "resume",                  0,    0,    signalFunc     },
{  "button",   "kill job",                0,    0,    signalFunc     },
{  "button",   "quit",                    0,    0,    quit_ukopp     },
{  "File",     "open job",                1,    0,    BJfileOpen     },
{  "File",     "edit job",                1,    0,    BJedit         },
{  "File",     "list job",                0,    0,    BJlist         },
{  "File",     "save job",                0,    0,    BJfileSave     },
{  "File",     "run job",                 1,    1,    Backup         },
{  "File",     "quit",                    0,    0,    quit_ukopp     },
{  "Backup",   "backup",                  1,    1,    Backup         },
{  "Backup",   "synchronize",             1,    1,    Synch          },
{  "Verify",   "incremental",             1,    1,    Verify         },
{  "Verify",   "full",                    1,    1,    Verify         },
{  "Verify",   "compare",                 1,    1,    Verify         },
{  "Report",   "get disk files",          1,    1,    Report         },
{  "Report",   "diffs summary",           1,    1,    Report         },
{  "Report",   "diffs by directory",      1,    1,    Report         },
{  "Report",   "diffs by file",           1,    1,    Report         },
{  "Report",   "version summary",         1,    1,    Report         },
{  "Report",   "expired versions",        1,    1,    Report         },
{  "Report",   "list disk files",         1,    1,    Report         },
{  "Report",   "list backup files",       1,    1,    Report         },
{  "Report",   "find files",              1,    1,    Report         },
{  "Report",   "save screen",             0,    0,    saveScreen     },
{  "Restore",  "setup restore job",       1,    0,    RJedit         },
{  "Restore",  "list restore files",      1,    1,    RJlist         },
{  "Restore",  "restore files",           1,    1,    Restore        },
{  "Format",   "format device",           1,    0,    Format         },
{  "Help",     "about",                   0,    0,    helpFunc       },
{  "Help",     "contents",                0,    0,    helpFunc       }  };


//  ukopp main program

int main(int argc, char *argv[])
{
   GtkWidget   *mbar, *tbar;
   GtkWidget   *mFile, *mBackup, *mVerify, *mReport, *mRestore;
   GtkWidget   *mFormat, *mHelp;
   int         ii;
   
   gtk_init(&argc, &argv);                                                 //  GTK command line options
   if (! g_thread_supported())                                             //  suddenly required for new gtk
         g_thread_init(null);                                              //  initz. GTK for threads
   gdk_threads_init();
   zlockInit();

   initz_appfiles("ukopp",null);                                           //  setup app directories

   clrun = 0;                                                              //  no command line run command
   *BJfilespec = 0;                                                        //  no backup job file

   for (ii = 1; ii < argc; ii++)                                           //  process command line
   {
      if (strEqu(argv[ii],"-job") && argc > ii+1)                          //  -job jobfile  (load only)
            strcpy(BJfilespec,argv[++ii]);
      else if (strEqu(argv[ii],"-run") && argc > ii+1)                     //  -run jobfile  (load and run)
          { strcpy(BJfilespec,argv[++ii]); clrun++; }
      else  strcpy(BJfilespec,argv[ii]);                                   //  assume a job file and load it
   }
   
   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mWin),ukopp_title);
   gtk_window_set_position(GTK_WINDOW(mWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mWin),800,500);
   
   mVbox = gtk_vbox_new(0,0);                                              //  vertical packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);                           //  add to main window

   mScroll = gtk_scrolled_window_new(0,0);                                 //  scrolled window
   gtk_box_pack_end(GTK_BOX(mVbox),mScroll,1,1,0);                         //  add to main window mVbox
   
   mLog = gtk_text_view_new();                                             //  text edit window
   gtk_container_add(GTK_CONTAINER(mScroll),mLog);                         //  add to scrolled window

   monofont = pango_font_description_from_string("Monospace");             //  set fixed pitch font
   gtk_widget_modify_font(mLog,monofont);

   mbar = create_menubar(mVbox,16);                                        //  create menu bar

   mFile = add_menubar_item(mbar,"File",menufunc);                         //  add menu bar items
      add_submenu_item(mFile,"open job",null,menufunc);
      add_submenu_item(mFile,"edit job",null,menufunc);
      add_submenu_item(mFile,"list job",null,menufunc);
      add_submenu_item(mFile,"save job",null,menufunc);
      add_submenu_item(mFile,"run job",null,menufunc);
      add_submenu_item(mFile,"quit",null,menufunc);
   mBackup = add_menubar_item(mbar,"Backup",menufunc);
      add_submenu_item(mBackup,"backup",null,menufunc);
      add_submenu_item(mBackup,"synchronize",null,menufunc);
   mVerify = add_menubar_item(mbar,"Verify",menufunc);
      add_submenu_item(mVerify,"incremental",null,menufunc);
      add_submenu_item(mVerify,"full",null,menufunc);
      add_submenu_item(mVerify,"compare",null,menufunc);
   mReport = add_menubar_item(mbar,"Report",menufunc);
      add_submenu_item(mReport,"get disk files",null,menufunc);
      add_submenu_item(mReport,"diffs summary",null,menufunc);
      add_submenu_item(mReport,"diffs by directory",null,menufunc);
      add_submenu_item(mReport,"diffs by file",null,menufunc);
      add_submenu_item(mReport,"version summary",null,menufunc);
      add_submenu_item(mReport,"expired versions",null,menufunc);
      add_submenu_item(mReport,"list disk files",null,menufunc);
      add_submenu_item(mReport,"list backup files",null,menufunc);
      add_submenu_item(mReport,"find files",null,menufunc);
      add_submenu_item(mReport,"save screen",null,menufunc);
   mRestore = add_menubar_item(mbar,"Restore",menufunc);
      add_submenu_item(mRestore,"setup restore job",null,menufunc);
      add_submenu_item(mRestore,"list restore files",null,menufunc);
      add_submenu_item(mRestore,"restore files",null,menufunc);
   mFormat = add_menubar_item(mbar,"Format",menufunc);
      add_submenu_item(mFormat,"format device",null,menufunc);
   mHelp = add_menubar_item(mbar,"Help",menufunc);
      add_submenu_item(mHelp,"about",null,menufunc);
      add_submenu_item(mHelp,"contents",null,menufunc);

   tbar = create_toolbar(mVbox);                                           //  create toolbar and buttons

   add_toolbar_button(tbar,"target","select backup device or directory","usbstick.png",buttonfunc);
   add_toolbar_button(tbar,"mount","mount target device","mount.png",buttonfunc);
   add_toolbar_button(tbar,"unmount","unmount target device","unmount.png",buttonfunc);
   add_toolbar_button(tbar,"edit job","edit backup job","edit.png",buttonfunc);
   add_toolbar_button(tbar,"run job","run backup job","run.png",buttonfunc);
   add_toolbar_button(tbar,"pause","pause running job","gtk-media-pause",buttonfunc); 
   add_toolbar_button(tbar,"resume","resume running job","gtk-media-play",buttonfunc); 
   add_toolbar_button(tbar,"kill job","kill running job","gtk-stop",buttonfunc); 
   add_toolbar_button(tbar,"clear","clear screen","gtk-clear",buttonfunc);
   add_toolbar_button(tbar,"quit","quit ukopp","gtk-quit",buttonfunc); 

   gtk_widget_show_all(mWin);                                              //  show all widgets

   G_SIGNAL(mWin,"destroy",quit_ukopp,0)                                   //  connect window destroy event
   
   gtk_init_add((GtkFunction) initfunc,0);                                 //  setup initial call from gtk_main()

   gdk_threads_enter();
   gtk_main();                                                             //  process window events
   gdk_threads_leave();

   return 0;
}


//  initial function called from gtk_main() at startup

int initfunc(void *data)
{
   int         ii;
   const char  *home, *appdirk;
   time_t      datetime;

   menufunc(null,"Help");                                                  //  show version and license
   menufunc(null,"about");

   appdirk = get_zuserdir();
   sprintf(TFbakfiles,"%s/bakfiles",appdirk);                              //  make temp file names
   sprintf(TFpoopfile,"%s/poopfile",appdirk);
   sprintf(TFjobfile,"%s/jobfile",appdirk);
   sprintf(TFdatetime,"%s/datetime",appdirk);
   sprintf(TFformatscript,"%s/formatscript.sh",appdirk);

   datetime = time(0);
   printf("\n""ukopp errlog %s \n",ctime(&datetime));

   menuLock = threadcount = Fdialog = 0;                                   //  initialize controls
   killFlag = pauseFlag = 0;
   strcpy(subprocName,"");
   
   BJnx = 6;                                                               //  default backup job data
   for (ii = 0; ii < BJnx; ii++) {
      BJrec[ii] = zmalloc(60);
      BJfspec[ii] = zmalloc(60);
   }
   home = getenv("HOME");                                                  //  get "/home/username"
   if (! home) home = "/root";
   strcpy(BJrec[0],"# ukopp default backup job");                          //  comment
   strcpy(BJrec[1],"retain 0 0  *");                                       //  retain 0 0 * 
   sprintf(BJrec[2],"include %s/*",home);                                  //  include /home/username/*
   sprintf(BJrec[3],"exclude %s/.Trash/*",home);                           //  exclude /home/username/.Trash/*
   sprintf(BJrec[4],"exclude %s/.thumbnails/*",home);                      //  exclude /home/username/.thumbnails/*
   sprintf(BJrec[5],"verify incremental");                                 //  verify incremental
   BJrtype[0] = 0;
   BJrtype[1] = 1;
   BJrtype[2] = 2;
   BJrtype[3] = 3;
   BJrtype[4] = 3;
   BJrtype[5] = 4;
   BJfspec[0] = null;
   strcpy(BJfspec[1],"*");                                                 //  bugfix   v.24
   sprintf(BJfspec[2],"%s/*",home);                                        //  /home/username/*
   sprintf(BJfspec[3],"%s/.Trash/*",home);                                 //  /home/username/.Trash/*
   sprintf(BJfspec[4],"%s/.thumbnails/*",home);                            //  /home/username/.thumbnails/*
   BJfspec[5] = null;
   strcpy(BJdev,"");                                                       //  backup target device (maybe)
   strcpy(BJdirk,"/unknown");                                              //  backup target directory, cc
   BJdcc = strlen(BJdirk);
   BJflush = 1;                                                            //  flush method = sync command
   BJvmode = 0;                                                            //  no verify
   BJval = 0;                                                              //  not validated
   
   strcpy(RJfrom,"/home/");                                                //  file restore copy-from location
   strcpy(RJto,"/home/");                                                  //  file restore copy-to location
   RJnx = 0;                                                               //  no. restore include/exclude recs
   RJval = 0;                                                              //  not validated
   
   BDpoop();                                                               //  find devices and mount points
   
   if (*BJfilespec) BJload(BJfilespec);                                    //  load command line job file
   else snprintf(BJfilespec,maxfcc,"%s/ukopp.job",get_zuserdir());

   if (clrun) {
      menufunc(null,"File");                                               //  run command line job file
      menufunc(null,"run job");
   }

   return 0;
}


//  process toolbar button events (simulate menu selection)

void buttonfunc(GtkWidget *, cchar *button)
{
   char     button2[20], *pp;
   
   strncpy0(button2,button,19);
   pp = strchr(button2,'\n');                                              //  replace \n with blank
   if (pp) *pp = ' ';

   menufunc(0,"button");
   menufunc(0,button2);
   return;
}


//  process menu selection event

void menufunc(GtkWidget *, cchar *menu)
{
   static int     ii;
   static char    menu1[20] = "", menu2[40] = "";
   char           command[100];
   pthread_t      tid;
   
   for (ii = 0; ii < nmenu; ii++) 
         if (strEqu(menu,menus[ii].menu1)) break;                          //  mark top-menu selection
   if (ii < nmenu) { strcpy(menu1,menu); return;  }
   
   for (ii = 0; ii < nmenu; ii++) 
         if (strEqu(menu1,menus[ii].menu1) && 
             strEqu(menu,menus[ii].menu2)) break;                          //  mark sub-menu selection
   if (ii < nmenu) strcpy(menu2,menu);

   else {                                                                  //  no match to menus
      wprintf(mLog," *** bad command: %s \n",menu);
      return;
   }

   if (menuLock && menus[ii].lock) {                                       //  no lock funcs can run parallel
      zmessageACK("wait for current function to complete");
      return;
   }

   if (! menuLock) {   
      killFlag = pauseFlag = 0;                                            //  reset controls
      *subprocName = 0;
   }

   snprintf(command,99,"\n""command: %s > %s \n",menu1,menu2);
   wprintx(mLog,0,command,boldfont);                                    
   
   if (menus[ii].thread) {
      ++threadcount;                                                       //  start thread to run menu function
      if (menus[ii].lock) ++menuLock;
      pthread_create(&tid,0,menu_thread_func,(void *) &ii);                //  &ii for 64-bit compat.
   }
   
   else  {
      if (menus[ii].lock) ++menuLock;
      menus[ii].mfunc(menu2);                                              //  or call menu function directly
      if (menus[ii].lock) --menuLock;
   }

   return;
}


//  thread shell function - run menu function in a thread

void * menu_thread_func(void *mii)
{
   int   ii = *((int *) mii);

   menus[ii].mfunc(menus[ii].menu2);                                       //  call menu function
   if (menus[ii].lock) --menuLock;
   --threadcount;
   return 0;
}


//  quit ukopp

int quit_ukopp(cchar *menu)
{
   if (devMounted && BJflush == 2) unmount("");                            //  v.3.2
   gtk_main_quit();                                                        //  tell gtk_main() to quit
   return 0;
}


//  clear logging window

int clearScreen(cchar *menu)
{
   wclear(mLog);
   return 0;
}


//  kill/pause/resume current function - called from menu function

int signalFunc(cchar *menu)
{
   if (strEqu(menu,"kill job"))
   {
      if (! menuLock) {
         wprintf(mLog,"\n""ready \n");
         return 0;
      }
      
      if (killFlag) {
         if (*subprocName) {
            wprintf(mLog," *** kill again: %s \n",subprocName);
            signalProc(subprocName,"kill");
         }
         else wprintf(mLog," *** waiting for function to quit \n");
         return 0;
      }

      wprintf(mLog," *** KILL current function \n");
      pauseFlag = 0;
      killFlag = 1;

      if (*subprocName) {
         signalProc(subprocName,"resume");
         signalProc(subprocName,"kill");
      }

      return 0;
   }

   if (strEqu(menu,"pause")) {
      pauseFlag = 1;
      if (*subprocName) signalProc(subprocName,"pause");
      return 0;
   }

   if (strEqu(menu,"resume")) {
      pauseFlag = 0;
      if (*subprocName) signalProc(subprocName,"resume");
      return 0;
   }
   
   else zappcrash("signalFunc: %s",menu);
   return 0;
}


//  check kill and pause flags
//  called periodically from long-running functions

int checkKillPause()
{
   while (pauseFlag)                                                       //  idle loop while paused
   {
      zsleep(0.1);
      zmainloop();                                                         //  process menus
   }

   if (killFlag) return 1;                                                 //  return true = stop now
   return 0;                                                               //  return false = continue
}


//  find all disk devices and mount points via Linux utilities

int BDpoop()                                                               //  v.3.3  new udevinfo format
{
   int      ii, jj, contx = 0, pii, pjj;
   int      diskf, filsysf, usbf, Nth, Nmounted;
   char     *buff, diskdev1[40], diskdesc1[60], work[100];
   cchar    *pp1, *pp2;
   
   Ndisk = diskf = filsysf = usbf = 0;

   while ((buff = command_output(contx,"udevinfo -e")))                    //  next device info record
   {
      if (strnEqu(buff,"P: ",3)) {                                         //  start new device
         if (diskf && filsysf) {                                           //  if last device = formatted disk
            strncpy0(diskdev[Ndisk],diskdev1,39);                          //    save /dev/devid
            strncpy0(diskdesc[Ndisk],diskdesc1,59);                        //    save description
            if (usbf) strcat(diskdesc[Ndisk]," (USB)");                    //    note if USB device
            strcpy(diskmp[Ndisk],"(not mounted)");                         //    mount point TBD
            Ndisk++;
            if (Ndisk == maxdisk) {
               wprintf(mLog," *** exceeded %d devices \n",maxdisk);
               break;
            }
         }

         diskf = filsysf = usbf = 0;                                       //  clear new device flags
      }

      if (strnEqu(buff,"N: ",3)) {
         strcpy(diskdev1,"/dev/");
         strncat(diskdev1,buff+3,14);                                      //  save /dev/devid
      }
         
      if (strnEqu(buff,"E: ",3)) {
         pp1 = strstr(buff,"ID_TYPE=disk");
         if (pp1) diskf = 1;                                               //  device is a disk
         pp1 = strstr(buff,"ID_FS_TYPE=");
         if (pp1) filsysf = 1;                                             //  device has a file system
         pp1 = strstr(buff,"ID_BUS=usb");
         if (pp1) usbf = 1;                                                //  device is a USB device
         pp1 = strstr(buff,"ID_MODEL=");
         if (pp1) strncpy0(diskdesc1,pp1+9,59);                            //  save description
      }
   }

   if (! Ndisk) {
      wprintf(mLog," no devices found \n");
      return 0;
   }

   contx = Nmounted = 0;

   while ((buff = command_output(contx,"cat /proc/mounts")))               //  get mounted disk info   v.3.2
   {
      if (strnNeq(buff,"/dev/",5)) continue;                               //  not a /dev/xxx record

      Nth = 1;
      pp1 = strField(buff,' ',Nth++);                                      //  parse /dev/xxx /media/xxx
      pp2 = strField(buff,' ',Nth++);

      for (ii = 0; ii < Ndisk; ii++)                                       //  look for matching device
      {
         if (strNeq(pp1,diskdev[ii])) continue;
         strncpy0(diskmp[ii],pp2,59);                                      //  copy its mount point
         strTrim(diskmp[ii]);
         Nmounted++;
         break;
      }
   }

   #define swap(name,ii,jj) {             \
         strcpy(work,name[ii]);           \
         strcpy(name[ii],name[jj]);       \
         strcpy(name[jj],work); }

   for (ii = 0; ii < Ndisk; ii++)                                          //  sort USB and mounted devices
   for (jj = ii + 1; jj < Ndisk; jj++)                                     //    to the top of the list
   {
      pii = pjj = 0;
      if (strstr(diskdesc[ii],"(USB)")) pii += 2;
      if (! strEqu(diskmp[ii],"(not mounted)")) pii += 1;
      if (strstr(diskdesc[jj],"(USB)")) pjj += 2;
      if (! strEqu(diskmp[jj],"(not mounted)")) pjj += 1;
      if (pjj > pii) {
         swap(diskdev,jj,ii);
         swap(diskmp,jj,ii);
         swap(diskdesc,jj,ii);
      }
   }

   return Nmounted;
}


//  choose backup device or enter a target directory
//  update backup job target device and directory

int chooseTarget(cchar *)                                                  //  overhauled   v.3.2
{
   int            ii, zstat;
   char           text[200];
   zdialog        *zd;
   cchar          *instruct = "select target device or directory";
   cchar          *errmess;
   
   BDpoop();                                                               //  refresh available devices
   
   zd = zdialog_new("choose backup target",mWin,"OK","cancel",null);
   zdialog_add_widget(zd,"vbox","vb1","dialog","","space=10");
   zdialog_add_widget(zd,"label","lab1","vb1",instruct);                   //    select backup device ...
   zdialog_add_widget(zd,"comboE","entbak","vb1",BJdirk);                  //   [__________________________][v]
   zdialog_add_widget(zd,"hbox","hb1","vb1","space=10");
   zdialog_add_widget(zd,"label","lab2","hb1","flush method");             //   flush method  (o) sync  (o) remount
   zdialog_add_widget(zd,"radio","sync","hb1","sync","space=10");
   zdialog_add_widget(zd,"radio","remount","hb1","remount");

   for (ii = 0; ii < Ndisk; ii++)
   {                                                                       //  load combo box with device poop
      strcpy(text,diskdev[ii]);                                            //    /dev/xxx /media/xxx description
      strncatv(text,199," ",diskmp[ii],"   ",diskdesc[ii],null);
      zdialog_cb_app(zd,"entbak",text);
   }

   if (BJflush == 2) {                                                     //  set flush method    v.3.2
      zdialog_stuff(zd,"sync",0);
      zdialog_stuff(zd,"remount",1);
   }
   else {                                                                  //  sync or undefined >> sync
      zdialog_stuff(zd,"sync",1);
      zdialog_stuff(zd,"remount",0);
   }

   zstat = zdialog_run(zd);                                                //  run dialog - blocking
   if (zstat != 1) {
      zdialog_free(zd);
      return 0;
   }

   zdialog_fetch(zd,"entbak",text,199);                                    //  get device or target directory

   zdialog_fetch(zd,"remount",ii);                                         //  get flush method
   if (ii == 1) BJflush = 2;
   else BJflush = 1;

   zdialog_free(zd);                                                       //  kill dialog

   errmess = parseTarget(text);                                            //  parse selected device, directory
   wprintf(mLog," new target: %s %s \n",BJdev,BJdirk);
   if (errmess) wprintf(mLog," *** %s \n",errmess);
   
   return 0;
}


//  job file open dialog - get backup job data from a file
//  return 1 if OK, else 0

int BJfileOpen(cchar *menu)
{
   char        *file;
   int         nstat = 0;
   
   ++Fdialog;

   file = zgetfile("open backup job",BJfilespec,"open","hidden");          //  get file from user
   if (file) {
      strncpy0(BJfilespec,file,maxfcc-2);
      zfree(file);
      nstat = BJload(BJfilespec);                                          //  load job file, set BJval
   }

   --Fdialog;
   return nstat;
}


//  job file save dialog - save backup job data to a file
//  return 1 if OK, else 0

int BJfileSave(cchar *menu)
{
   char        *file;
   int         nstat = 0;
   
   ++Fdialog;
   
   file = zgetfile("save backup job",BJfilespec,"save","hidden");
   if (file) {
      strncpy0(BJfilespec,file,maxfcc-2);
      zfree(file);
      nstat = BJstore(BJfilespec);
   }
   
   --Fdialog;
   return nstat;
}


//  backup job data <<< jobfile
//  return 1 if loaded, else return 0

int BJload(cchar *jobfile)
{
   FILE           *fid;
   char           buff[1000];
   char           *pp;

   BJreset();                                                              //  reset all job data

   snprintf(buff,999,"\n""loading job file: %s \n",jobfile);
   wprintx(mLog,0,buff,boldfont);

   fid = fopen(jobfile,"r");                                               //  open job file
   if (! fid) {
      wprintf(mLog," *** cannot open job file: %s \n",jobfile);
      return 0;
   }

   for (BJnx = 0; BJnx < maxnx; BJnx++)
   {
      pp = fgets_trim(buff,999,fid,1);                                     //  read next job record
      if (! pp) break;                                                     //  EOF
      BJrec[BJnx] = strdupz(buff);                                         //  >> next job memory record
   }
   
   fclose(fid);                                                            //  close file

   BJlist("");                                                             //  list job data and verify
   return 1;
}


//  backup job data >>> jobfile
//  return 1 if OK, else 0

int BJstore(cchar *fspec)
{
   FILE     *fid;

   fid = fopen(fspec,"w");                                                 //  open file
   if (! fid) { 
      wprintf(mLog," *** cannot open job file: %s \n",fspec); 
      return 0; 
   }
   
   for (int ii = 0; ii < BJnx; ii++)                                       //  write all job recs
      fprintf(fid,"%s \n",BJrec[ii]);

   fclose(fid);
   return 1;
}


//  list backup job data to log window with diagnostics

int BJlist(cchar *menu)
{
   int         rtype, nerrs = 0;
   const char  *errmess;
   char        *fspec;

   wprintf(mLog,"\n backup job file: %s \n",BJfilespec);                   //  job file      v.21

   for (int ii = 0; ii < BJnx; ii++)                                       //  job recs
   {
      errmess = parseJobrec(BJrec[ii],rtype,fspec);                        //  validate (may update rec.)
      wprintf(mLog," %s \n",BJrec[ii]);                                    //  output rec.
      if (errmess) wprintf(mLog," *** %s \n",errmess);
      if (errmess) nerrs++;

      BJrtype[ii] = rtype;
      if (fspec) BJfspec[ii] = fspec;                                      //  include/exclude filespec
      else BJfspec[ii] = 0;
      BJbytes[ii] = BJfiles[ii] = 0;
   }

   if (BJnx == maxnx) {
      wprintf(mLog," *** max job records exceeded \n");
      nerrs++;
   }

   if (nerrs) BJval = 0;
   else BJval = 1;
   
   return BJval;
}

      
//  edit dialog for backup job data

int BJedit(cchar *menu)
{
   zdialog     *zd;
   const char  *inexmess = "Include / Exclude Files";

   ++Fdialog;
   
   zd = zdialog_new("edit backup job",mWin,"browse","done","cancel","clear",null);
   
   zdialog_add_widget(zd,"hsep","sep2","dialog");                          //  edit box for job recs
   zdialog_add_widget(zd,"label","labinex","dialog",inexmess);
   zdialog_add_widget(zd,"frame","frminex","dialog",0,"expand");
   zdialog_add_widget(zd,"scrwin","scrwinex","frminex");
   zdialog_add_widget(zd,"edit","edinex","scrwinex");

   editwidget = zdialog_widget(zd,"edinex");
   wclear(editwidget);                                                     //  stuff job recs into dialog
   for (int ii = 0; ii < BJnx; ii++)
      wprintf(editwidget,"%s \n",BJrec[ii]);

   zdialog_resize(zd,500,400);
   zdialog_run(zd,null,BJedit_compl);                                      //  run dialog
   return 0;
}


//  edit dialog completion function

int BJedit_compl(zdialog *zd, int zstat)
{
   int      ftf = 1;
   char     *pp;

   if (zstat == 1) {                                                       //  browse, do file-chooser dialog
      fc_dialog("/home");
      return 0;
   }

   if (zstat == 4) {
      wclear(editwidget);                                                  //  clear include/exclude recs
      return 0;
   }

   if (zstat != 2) {                                                       //  cancel or kill
      zdialog_free(zd);
      --Fdialog;
      return 0;
   }

   BJreset();                                                              //  done, reset job data
   
   for (BJnx = 0; BJnx < maxnx; BJnx++)                                    //  get job data from dialog widgets
   {
      pp = wscanf(editwidget,ftf);                                         //  next record from edit widget
      if (! pp) break;
      strTrim(pp);
      BJrec[BJnx] = strdupz(pp);                                           //  >> next job memory record
   }

   BJlist("");                                                             //  list job and validate

   zdialog_free(zd);                                                       //  destroy dialog
   --Fdialog;
   return 0;
}


//  thread function
//  Copy new and modified disk files to backup location.
//  Delete backup files exceeding age and version limits.

int Backup(cchar *menu)
{
   char        work[100];
   int         vmode = 0, terr = 0, ii;
   int         upvers = 0, deleted = 0;
   char        disp, *dfile = 0;
   cchar       *errmess = null;
   double      bsecs, bbytes, bspeed;
   timeval     time0;

   if (! BJlist("")) return 0;                                             //  list and validate job data
   if (! mount("")) return 0;                                              //  validate and mount target  v.3.2

   Report("diffs summary");                                                //  refresh all file data, report diffs

   snprintf(work,99,"\n""begin backup \n");
   wprintx(mLog,0,work,boldfont);
   wprintf(mLog," files: %d  bytes: %.1f MB \n",Mfiles,Mbytes/mega);       //  files and bytes to copy

   if (Mfiles == 0) {
      wprintf(mLog," *** nothing to back-up \n");
      return 0;
   }

   ii = zmessageYN("backup target: %s %s \n"
                   "%d files (%.1f MB) will be copied \n"
                   "continue?",BJdev,BJdirk,Mfiles,Mbytes/mega);           //  confirm backup target   v.23
   if (! ii) return 0;

   wprintf(mLog," using backup directory: %s %s \n",BJdev,BJdirk);

   if (strEqu(menu,"backup")) vmode = 0;                                   //  backup command, no auto verify
   if (strEqu(menu,"run job")) vmode = BJvmode;


   wprintf(mLog," assign new version numbers to modified backup files \n"
                " and purge expired versions from backup location \n\n");
   
   for (ii = 0; ii < Bnf; ii++) {                                          //  scan files at backup location
      disp = Brec[ii].disp;
      dfile = Brec[ii].file;

      wprintf(mLog,-2," %s \n",dfile);
      errmess = null;

      if (disp == 'm' || disp == 'd') {                                    //  modified or deleted,
         errmess = renameFile(Brec[ii]);                                   //  rename to next version number
         Brec[ii].err = -1;                                                //  mark file gone
         if (disp =='m') upvers++;                                         //  update counts
         if (disp =='d') deleted++;
      }

      if (! errmess && Brec[ii].expver) 
         errmess = purgeFile(Brec[ii]);                                    //  purge expired file versions

      if (errmess) {
         wprintf(mLog,-1," *** %s \n",errmess);                            //  log error
         wprintf(mLog,"\n");
         terr++;
         if (terr > 100) goto backup_fail;
      }

      if (checkKillPause()) goto backup_fail;                              //  killed by user
   }

   wprintf(mLog," %d backup files were assigned new versions \n",upvers);
   wprintf(mLog," %d backup files were deleted \n",deleted);
   wprintf(mLog," %d expired versions (%.1f MB) were purged \n\n",Pfiles,Pbytes/mega);
   Pbytes = Pfiles = 0;

   start_timer(&time0);                                                    //  start timer
   bbytes = Mbytes;

   BJstore(TFjobfile);                                                     //  copy job file to temp file
   writeDT();                                                              //  create date-time temp file

   wprintf(mLog,-2," %s \n",BD_JOBFILE);
   errmess = copyFile(TFjobfile,BD_JOBFILE,2);                             //  copy job file to backup location
   if (errmess) goto backup_fail;

   wprintf(mLog,-2," %s \n",BD_DATETIME);
   errmess = copyFile(TFdatetime,BD_DATETIME,2);                           //  copy date-time file
   if (errmess) goto backup_fail;

   wprintf(mLog," copying new and modified files from disk to backup location \n\n");

   for (ii = 0; ii < Dnf; ii++) {                                          //  scan all disk files
      disp = Drec[ii].disp;
      dfile = Drec[ii].file;
      Drec[ii].finc = 0;                                                   //  not included yet

      if (disp == 'n' || disp == 'm')                                      //  new or modified file
      {
         wprintf(mLog,-2," %s \n",dfile);
         errmess = copyFile(dfile,dfile,2);                                //  copy disk file to backup
         if (errmess) {
            Drec[ii].err = 1;
            wprintf(mLog,-1," *** %s \n",errmess);                         //  log error
            wprintf(mLog,"\n");
            terr++;
            if (terr > 100) goto backup_fail;
         }
         else Drec[ii].finc = 1;                                           //  set included file flag
      }

      if (checkKillPause()) goto backup_fail;                              //  killed by user
   }

   if (terr) wprintf(mLog," *** %d files were not copied \n",terr);
   
   synch_poop("backup");                                                   //  synch owner and permissions data  v.26

   flushcache();                                                           //  flush I/O memory buffers to device

   bsecs = get_timer(&time0);                                               //  output perf. statistics
   wprintf(mLog," backup time: %.1f secs \n",bsecs);
   bspeed = bbytes/mega/bsecs;
   wprintf(mLog," backup speed: %.2f MB/sec \n",bspeed);
   wprintf(mLog," backup complete \n");

   sleep(2);

   if (vmode == 1) Verify("incr");                                         //  do verify if required
   else if (vmode == 2) Verify("full");
   else if (vmode == 3) Verify("comp");
   
   if (BJflush == 2) unmount("");                                          //  leave unmounted  v.3.2

   wprintf(mLog," ready \n");
   return 0;

backup_fail:
   if (terr > 100) wprintf(mLog," too many errors, giving up \n");
   else if (errmess) wprintf(mLog," %s \n",errmess);
   wprintx(mLog,0," *** BACKUP FAILED \n",boldfont);
   bFilesReset();
   return 0;
}


//  thread function, synchronize disk and backup files                     //  v.25
//  bi-directional copy of new and newer files

int Synch(cchar *menu)
{
   int         ii, dii, bii, comp;
   char        disp, *dfile = 0;
   time_t      btime, dtime;
   cchar       *errmess = null;

   if (! BJlist("")) return 0;                                             //  list and validate job data
   if (! mount("")) return 0;                                              //  validate and mount target  v.3.2

   ii = zmessageYN("backup target: %s %s \n continue?",BJdev,BJdirk);      //  confirm backup target 
   if (! ii) return 0;
   wprintf(mLog," using backup directory: %s %s \n",BJdev,BJdirk);
   
   dGetFiles();                                                            //  get disk files of backup job
   if (bGetFiles() < 0) goto synch_exit;                                   //  get files in backup location
   setFileDisps();                                                         //  compare and set dispositions

   wprintf(mLog,"\n begin synchronize \n");

   BJstore(TFjobfile);                                                     //  copy job file to temp file
   writeDT();                                                              //  create date-time temp file

   wprintf(mLog,-2," %s \n",BD_JOBFILE);
   errmess = copyFile(TFjobfile,BD_JOBFILE,2);                             //  copy job file to backup location
   if (errmess) goto synch_exit;

   wprintf(mLog,-2," %s \n",BD_DATETIME);
   errmess = copyFile(TFdatetime,BD_DATETIME,2);                           //  copy date-time file
   if (errmess) goto synch_exit;

   for (ii = 0; ii < Dnf; ii++)                                            //  copy new disk files >> backup loc.
   {
      disp = Drec[ii].disp;
      dfile = Drec[ii].file;
      if (disp != 'n') continue;
      wprintf(mLog," disk >> backup: %s \n",dfile);
      errmess = copyFile(dfile,dfile,2);
      if (errmess) wprintf(mLog," *** %s \n",errmess);
      else Drec[ii].finc = 1;
      if (checkKillPause()) goto synch_exit;
   }

   for (ii = 0; ii < Bnf; ii++)                                            //  copy new backup files >> disk
   {                                                                       //  (aka "deleted" disk files)
      disp = Brec[ii].disp;
      dfile = Brec[ii].file;
      if (disp != 'd') continue;
      wprintf(mLog," backup >> disk: %s \n",dfile);
      errmess = copyFile(dfile,dfile,1);
      if (errmess) wprintf(mLog," *** %s \n",errmess);
      else Brec[ii].finc = 1;
      if (checkKillPause()) goto synch_exit;
   }

   dii = bii = 0;

   while ((dii < Dnf) || (bii < Bnf))                                      //  scan disk and backup files parallel
   {
      if ((dii < Dnf) && (bii == Bnf)) comp = -1;
      else if ((dii == Dnf) && (bii < Bnf)) comp = +1;
      else comp = strcmp(Drec[dii].file, Brec[bii].file);

      if (comp < 0) { dii++; continue; }                                   //  next disk file
      if (comp > 0) { bii++; continue; }                                   //  next backup file

      disp = Drec[dii].disp;
      dfile = Drec[dii].file;

      if (disp == 'm')                                                     //  screen for modified status
      {
         btime = int(Brec[bii].mtime);
         dtime = int(Drec[dii].mtime);

         if (btime > dtime) {                                              //  copy newer backup file >> disk
            wprintf(mLog," backup >> disk: %s \n",dfile);
            errmess = copyFile(dfile,dfile,1);
            if (errmess) wprintf(mLog," *** %s \n",errmess);
            else Brec[bii].finc = 1;
         }

         else {                                                            //  copy newer disk file >> backup
            wprintf(mLog," disk >> backup: %s \n",dfile);
            errmess = copyFile(dfile,dfile,2);
            if (errmess) wprintf(mLog," *** %s \n",errmess);
            else Drec[dii].finc = 1;
         }
      }

      dii++;                                                               //  next disk and backup files
      bii++;

      if (checkKillPause()) break;
   }
   
   errmess = null;
   
synch_exit:

   if (errmess) wprintf(mLog," *** %s \n",errmess);

   synch_poop("synch");                                                    //  synch owner and permissions data  v.26

   flushcache();                                                           //  flush I/O memory buffers to device

   Verify("incremental");                                                  //  verify all files copied

   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, verify integrity of backup files

int Verify(cchar *menu)
{
   int            ii, vers, comp, vfiles;
   int            dfiles1 = 0, dfiles2 = 0;
   int            verrs = 0, cerrs = 0;
   char           filespec[maxfcc];
   cchar          *errmess = null;
   double         secs, dcc1, vbytes, vspeed;
   double         mtime, diff;
   timeval        time0;
   struct stat64  filestat;

   vfiles = verrs = cerrs = 0;
   vbytes = 0.0;
   if (! mount("")) return 0;                                              //  validate and mount target  v.3.2

   start_timer(&time0);

   if (strnEqu(menu,"incremental",4))                                      //  verify new/modified files only
   {
      wprintx(mLog,0,"\n""Verify files copied in prior backup or synch \n",boldfont);

      for (ii = 0; ii < Dnf; ii++)                                         //  scan disk file list
      {
         if (! Drec[ii].finc) continue;                                    //  file included in last backup
         strncpy0(filespec,Drec[ii].file,maxfcc-1);
         wprintf(mLog,"  %s \n",filespec);                                 //  output filespec

         errmess = checkFile(filespec,1,dcc1);                             //  compare disk/backup files, get length
         if (errmess) {
            wprintf(mLog,"  *** %s \n\n",errmess);                         //  log and count errors
            if (strstr(errmess,"compare")) cerrs++;                        //  backup - disk compare failure
            else  verrs++;
         }

         vfiles++;                                                         //  count files and bytes
         vbytes += dcc1;
         if (checkKillPause()) goto verify_exit;                           //  killed by user
      }

      for (ii = 0; ii < Bnf; ii++)                                         //  scan backup file list     v.25
      {
         if (! Brec[ii].finc) continue;                                    //  file included in last backup
         strncpy0(filespec,Brec[ii].file,maxfcc-1);
         wprintf(mLog,"  %s \n",filespec);                                 //  output filespec

         errmess = checkFile(filespec,1,dcc1);                             //  compare disk/backup files, get length
         if (errmess) {
            wprintf(mLog,"  *** %s \n\n",errmess);                         //  log and count errors
            if (strstr(errmess,"compare")) cerrs++;                        //  backup - disk compare failure
            else  verrs++;
         }

         vfiles++;                                                         //  count files and bytes
         vbytes += dcc1;
         if (checkKillPause()) goto verify_exit;                           //  killed by user
      }
   }

   if (strEqu(menu,"full"))                                                //  verify all files are readable
   {
      wprintx(mLog,0,"\n""Read and verify ALL backup files \n\n",boldfont);
      
      bGetFiles();                                                         //  get all files at backup location
      wprintf(mLog," %d backup files \n",Bnf);
      if (! Bnf) goto verify_exit;

      for (ii = 0; ii < Bnf; ii++)                                         //  scan backup file list
      {
         strncpy0(filespec,Brec[ii].file,maxfcc-10);                       //  /directory.../filename
         
         if (Brec[ii].err == 0) 
         {                                                                 //  check current file
            wprintf(mLog,-2," %s \n",filespec);
            errmess = checkFile(filespec,0,dcc1);                          //  verify file, get length
            if (errmess) {
               wprintf(mLog,-1," *** %s \n",errmess);                      //  log and count error
               wprintf(mLog,"\n");
               verrs++;
            }
            vfiles++;                                                      //  count files and bytes
            vbytes += dcc1;
         }

         if (Brec[ii].lover)
         for (vers = Brec[ii].lover; vers <= Brec[ii].hiver; vers++)       //  check previous versions
         {
            setFileVersion(filespec,vers);                                 //  append version if > 0
            wprintf(mLog,-2," %s \n",filespec);
            errmess = checkFile(filespec,0,dcc1);                          //  verify file, get length
            if (errmess) {
               wprintf(mLog,-1," *** %s \n",errmess);                      //  log and count error
               wprintf(mLog,"\n");
               verrs++;
            }
            vfiles++;                                                      //  count files and bytes
            vbytes += dcc1;
         }

         if (checkKillPause()) goto verify_exit;                           //  killed by user
      }
   }
   
   if (strnEqu(menu,"compare",4))                                          //  compare backup files to disk files
   {
      wprintx(mLog,0,"\n Read and verify ALL backup files. \n",boldfont);
      wprintf(mLog," Compare to correspending disk files (if present). \n\n");

      bGetFiles();                                                         //  get all files at backup location
      wprintf(mLog," %d backup files \n",Bnf);
      if (! Bnf) goto verify_exit;

      for (ii = 0; ii < Bnf; ii++)                                         //  scan backup file list
      {
         strncpy0(filespec,Brec[ii].file,maxfcc-10);                       //  /directory.../filename
         
         if (Brec[ii].err == 0) 
         {                                                                 //  check current file
            comp = 0;
            if (lstat64(filespec,&filestat) == 0) {                        //  corresponding disk file exists   v.3.0
               mtime = filestat.st_mtime + filestat.st_mtim.tv_nsec * nano;
               diff = fabs(mtime - Brec[ii].mtime);                        //  compare disk and backup file mod times
               if (diff < modtimetolr) comp = 1;                           //  equal within file system resolution
               dfiles1++;                                                  //  count matching disk names
               dfiles2 += comp;                                            //  count matching mod times
            }

            wprintf(mLog,-2," %s \n",filespec);
            errmess = checkFile(filespec,comp,dcc1);                       //  verify, get length, (opt) compare disk
            if (errmess) {
               wprintf(mLog,-1," *** %s \n",errmess);                      //  log and count error
               wprintf(mLog,"\n");
               if (strstr(errmess,"compare")) cerrs++;                     //  backup - disk compare failure
               else  verrs++;
            }
            vfiles++;                                                      //  count files and bytes
            vbytes += dcc1;
         }

         if (Brec[ii].lover)
         for (vers = Brec[ii].lover; vers <= Brec[ii].hiver; vers++)       //  check previous versions
         {
            setFileVersion(filespec,vers);                                 //  append version if > 0
            wprintf(mLog,-2," %s \n",filespec);
            errmess = checkFile(filespec,0,dcc1);                          //  verify file, get length
            if (errmess) {
               wprintf(mLog,-1," *** %s \n",errmess);                      //  log and count error
               wprintf(mLog,"\n");
               verrs++;
            }
            vfiles++;                                                      //  count files and bytes
            vbytes += dcc1;
         }

         if (checkKillPause()) goto verify_exit;                           //  killed by user
      }
   }

   wprintf(mLog," backup files: %d  bytes: %.0f \n",vfiles,vbytes);
   wprintf(mLog," backup file read errors: %d \n",verrs);

   if (strnEqu(menu,"incremental",4)) 
         wprintf(mLog," compare failures: %d \n",cerrs);

   if (strnEqu(menu,"compare",4)) {
      wprintf(mLog," matching disk names: %d  mod times: %d \n",dfiles1,dfiles2);
      wprintf(mLog," compare failures: %d \n",cerrs);
   }

   secs = get_timer(&time0);
   wprintf(mLog," verify time: %.1f secs \n",secs);
   vspeed = vbytes/mega/secs;
   wprintf(mLog," verify speed: %.2f MB/sec \n",vspeed);

   if (verrs + cerrs) wprintx(mLog,0," *** THERE WERE ERRORS *** \n",boldfont);
   else wprintf(mLog," NO ERRORS \n");

verify_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, various kinds of reports 

int Report(cchar *menu)
{
   cchar          *fspec1;
   char           fspec2[200], bfile[maxfcc];
   char           *pslash, *pdirk, ppdirk[maxfcc];
   int            ii, kfiles, knew, kdel, kmod;
   int            dii, bii, comp;
   double         nbytes, mb1, mb2;
   int            vers, lover, hiver, expver;
   int            age, loage, hiage, err;
   struct tm      tmdt;
   time_t         btime, dtime;
   char           bmod[20], dmod[20];
   const char     *copy;
   struct stat64  filestat; 

   //  get all disk files in backup job
   //  report file and byte counts per include and exclude record
   
   if (strEqu(menu, "get disk files"))
   {
      dGetFiles();                                                         //  get all files on disk

      wprintx(mLog,0,"\n""  files    Mbytes   include/exclude filespec \n",boldfont);

      for (ii = 0; ii < BJnx; ii++)                                        //  formatted report
         if (BJfspec[ii]) {
            if (BJfiles[ii]) wprintf(mLog," %6d %9.1f   %s \n",BJfiles[ii],BJbytes[ii]/mega,BJrec[ii]);
            else wprintf(mLog,"                    %s \n",BJrec[ii]);
         }
      wprintf(mLog," %6d %9.1f   TOTALS \n", Dnf, Dbytes/mega);
      goto ready_exit;
   }
   
   //  report disk / backup differences: new, modified, and deleted files

   if (strEqu(menu, "diffs summary"))
   {
      dGetFiles();
      if (bGetFiles() < 0) goto ready_exit;
      setFileDisps();

      wprintf(mLog,"\n disk files: %d  backup files: %d \n",Dnf,Bnf);
      wprintf(mLog,"\n Differences between files on disk and backup files: \n");
      wprintf(mLog," %6d  disk files not found on backup (new files) \n",nnew);
      wprintf(mLog," %6d  files with different data (modified files) \n",nmod);
      wprintf(mLog," %6d  backup files not found on disk (deleted files) \n",ndel);
      wprintf(mLog," %6d  files with identical data (unchanged files) \n",nunc);
      wprintf(mLog," Total differences: %d files  %.3f MB \n\n",Mfiles,Mbytes/mega);
      goto ready_exit;
   }

   //  report disk / backup differences per directory level

   if (strEqu(menu, "diffs by directory"))
   {
      dGetFiles();
      if (bGetFiles() < 0) goto ready_exit;
      setFileDisps();

      SortFileList((char *) Drec, sizeof(dfrec), Dnf, 'D');                //  re-sort, directories first
      SortFileList((char *) Brec, sizeof(bfrec), Bnf, 'D');

      wprintf(mLog,"\n differences by directory \n");

      wprintf(mLog,"   new   mod   del   MBytes  directory \n");
      
      nbytes = kfiles = knew = kmod = kdel = 0;
      dii = bii = 0;

      while ((dii < Dnf) || (bii < Bnf))                                   //  scan disk and backup files parallel
      {
         if ((dii < Dnf) && (bii == Bnf)) comp = -1;
         else if ((dii == Dnf) && (bii < Bnf)) comp = +1;
         else comp = filecomp(Drec[dii].file, Brec[bii].file);
         
         if (comp > 0) pdirk = Brec[bii].file;                             //  get disk or backup file
         else pdirk = Drec[dii].file;

         pslash = strrchr(pdirk,'/');                                      //  isolate directory
         if (pslash) *pslash = 0;
         if (strNeq(pdirk,ppdirk)) {                                       //  if directory changed, output
            if (kfiles > 0) wprintf(mLog," %5d %5d %5d %8.1f  %s \n",      //    totals from prior directory
                                    knew,kmod,kdel,nbytes/mega,ppdirk);
            nbytes = kfiles = knew = kmod = kdel = 0;                      //  reset totals
            strcpy(ppdirk,pdirk);                                          //  start new directory
         }
         if (pslash) *pslash = '/';

         if (comp < 0) {                                                   //  unmatched disk file: new
            knew++;                                                        //  count new files
            kfiles++;
            nbytes += Drec[dii].size;
            dii++;
         }

         else if (comp > 0) {                                              //  unmatched backup file
            if (Brec[bii].disp == 'd') {
               kdel++;                                                     //  count deleted files
               kfiles++;
            }
            bii++;
         }

         else if (comp == 0) {                                             //  file present on disk and backup
            if (Drec[dii].disp == 'm') kmod++;                             //  count modified files
            if (Drec[dii].disp == 'n') knew++;                             //  count new files (backup disp is 'v')
            if (Drec[dii].disp != 'u') {
               kfiles++;                                                   //  count unless unchanged
               nbytes += Drec[dii].size;
            }
            dii++;
            bii++;
         }
      }

      if (kfiles > 0) wprintf(mLog," %5d %5d %5d %8.1f  %s \n",            //  totals from last directory
                              knew,kmod,kdel,nbytes/mega,ppdirk);

      SortFileList((char *) Drec, sizeof(dfrec), Dnf, 'A');                //  restore straight ascii sort
      SortFileList((char *) Brec, sizeof(bfrec), Bnf, 'A');
      goto ready_exit;
   }

   //  report disk / backup differences by file

   if (strEqu(menu, "diffs by file"))
   {
      dGetFiles();
      if (bGetFiles() < 0) goto ready_exit;
      setFileDisps();

      wprintf(mLog,"\n Detailed list of disk:backup differences: \n");

      wprintf(mLog,"\n %d disk files not found on backup \n",nnew);

      for (ii = 0; ii < Dnf; ii++) 
      {
         if (Drec[ii].disp != 'n') continue;
         wprintf(mLog,"  %s \n",Drec[ii].file);
         if (checkKillPause()) break;
      }

      wprintf(mLog,"\n %d backup files not found on disk \n",ndel);

      for (ii = 0; ii < Bnf; ii++) 
      {
         if (Brec[ii].disp != 'd') continue;
         wprintf(mLog,"  %s \n",Brec[ii].file);
         if (checkKillPause()) break;
      }

      wprintf(mLog,"\n %d files with different data \n",nmod);
      wprintf(mLog,"  backup mod date   copy  disk mod date     filespec \n");

      dii = bii = 0;

      while ((dii < Dnf) || (bii < Bnf))                                   //  scan disk and backup files parallel
      {                                                                    //  revised       v.25
         if ((dii < Dnf) && (bii == Bnf)) comp = -1;
         else if ((dii == Dnf) && (bii < Bnf)) comp = +1;
         else comp = strcmp(Drec[dii].file, Brec[bii].file);

         if (comp < 0) { dii++; continue; }                                //  next disk file
         if (comp > 0) { bii++; continue; }                                //  next backup file
         
         if (Drec[dii].disp == 'm')                                        //  screen for modified status
         {
            btime = int(Brec[bii].mtime);                                  //  mod time on backup
            dtime = int(Drec[dii].mtime);                                  //  mod time on disk

            copy = "<<<<";                                                 //  copy direction, disk to backup
            if (btime > dtime) copy = "!!!!";                              //  flag if backup to disk

            tmdt = *localtime(&btime);
            snprintf(bmod,19,"%4d.%02d.%02d-%02d:%02d",tmdt.tm_year+1900,
                     tmdt.tm_mon+1,tmdt.tm_mday,tmdt.tm_hour,tmdt.tm_min);

            tmdt = *localtime(&dtime);
            snprintf(dmod,19,"%4d.%02d.%02d-%02d:%02d",tmdt.tm_year+1900,
                     tmdt.tm_mon+1,tmdt.tm_mday,tmdt.tm_hour,tmdt.tm_min);

            wprintf(mLog,"  %s  %s  %s  %s \n",bmod,copy,dmod,Drec[dii].file);
         }

         dii++;                                                            //  next disk and backup files
         bii++;

         if (checkKillPause()) break;
      }

      goto ready_exit;
   }

   //  report versions and expired versions per file
   
   if (strEqu(menu, "version summary"))
   {
      Report("diffs summary");
      if (Bnf < 1) goto ready_exit;

      wprintf(mLog,"\n  lover hiver expver  loage hiage  tot-MB exp-MB  filespec \n");

      for (ii = 0; ii < Bnf; ii++) 
      {
         lover = Brec[ii].lover;
         hiver = Brec[ii].hiver;
         expver = Brec[ii].expver;
         if (! lover) continue;

         strcpy(bfile,BJdirk);
         strcat(bfile,Brec[ii].file);
         loage = hiage = 0;
         mb1 = mb2 = 0.0;

         for (vers = lover; vers <= hiver; vers++)                         //  loop each file version
         {
            setFileVersion(bfile,vers);
            err = lstat64(bfile,&filestat);                                //  check file exists on backup   v.3.0
            if (err) continue;
            
            age = int((time(0)-filestat.st_mtime)/24.0/3600.0 + 0.5);      //  file age in days
            if (! loage) loage = hiage = age;                              //  bugfix     v.21
            if (age < loage) loage = age;
            if (age > hiage) hiage = age;

            mb1 += filestat.st_size;                                       //  accumulate total bytes
            if (vers <= expver) mb2 += filestat.st_size;                   //  and total expired bytes
         }

         mb1 = mb1 / mega;
         mb2 = mb2 / mega;         
         wprintf(mLog," %5d %5d %5d   %5d %5d   %6.2f %6.2f  %s \n",
                 lover,hiver,expver,loage,hiage,mb1,mb2,Brec[ii].file);

         if (checkKillPause()) break;
      }

      goto ready_exit;
   }

   //  report expired file versions (will be purged)                       //  v.22
   
   if (strEqu(menu, "expired versions"))
   {
      Report("diffs summary");
      if (Bnf < 1) goto ready_exit;

      wprintf(mLog,"\n  expired files (purge from backup location) \n");
      wprintf(mLog,"\n  vers   age    MB   filespec \n");

      for (ii = 0; ii < Bnf; ii++) 
      {
         lover = Brec[ii].lover;
         expver = Brec[ii].expver;
         if (! expver) continue;

         strcpy(bfile,BJdirk);
         strcat(bfile,Brec[ii].file);
         mb1 = mb2 = 0.0;

         for (vers = lover; vers <= expver; vers++)                        //  loop each expired version
         {
            setFileVersion(bfile,vers);
            err = lstat64(bfile,&filestat);                                //  check file exists on backup   v.3.0
            if (err) continue;
            age = int((time(0)-filestat.st_mtime)/24.0/3600.0 + 0.5);      //  age in days, size in MB
            mb1 = filestat.st_size / mega;
            wprintf(mLog," %5d %5d %6.2f  %s \n",vers,age,mb1,Brec[ii].file);
         }

         if (checkKillPause()) break;
      }

      goto ready_exit;
   }

   //  list all files in backup job set

   if (strEqu(menu, "list disk files"))
   {
      wprintf(mLog," List all files in backup file set: \n");

      dGetFiles();
      wprintf(mLog,"   %d files found \n",Dnf);

      for (ii = 0; ii < Dnf; ii++)
      {
         wprintf(mLog," %s \n",Drec[ii].file);
         if (checkKillPause()) break;
      }

      goto ready_exit;
   }
   
   //  list all files on backup

   if (strEqu(menu, "list backup files"))
   {
      wprintf(mLog," List all backup files: \n");
      if (bGetFiles() < 0) goto ready_exit;

      for (ii = 0; ii < Bnf; ii++)
      {
         if (Brec[ii].hiver) wprintf(mLog," %s (vers %d-%d) \n", 
                     Brec[ii].file, Brec[ii].lover, Brec[ii].hiver);
         else  wprintf(mLog," %s \n",Brec[ii].file);
         if (checkKillPause()) break;
      }

      goto ready_exit;
   }
   
   //  search disk and backup file list for match with wild search pattern

   if (strEqu(menu, "find files"))
   {
      wprintf(mLog," Find files matching wildcard pattern \n");

      dGetFiles();
      bGetFiles();
      if (!(Dnf + Bnf)) goto ready_exit;

      fspec1 = dialogText("enter (wildcard) filespec:","/dir*/file* ");
      if (! fspec1) goto ready_exit;
      while (*fspec1 == ' ') fspec1++;
      if (! *fspec1) goto ready_exit;
      strncpy0(fspec2,fspec1,199);
      strTrim(fspec2);

      wprintf(mLog,"\n matching disk files: \n");

      for (ii = 0; ii < Dnf; ii++)
      {
         if (MatchWild(fspec2,Drec[ii].file) == 0) 
               wprintf(mLog," %s \n",Drec[ii].file);
         if (checkKillPause()) break;
      }

      wprintf(mLog,"\n matching backup files: \n");

      for (ii = 0; ii < Bnf; ii++)
      {
         if (MatchWild(fspec2,Brec[ii].file) == 0) {
            if (Brec[ii].hiver) wprintf(mLog," %s (vers %d-%d) \n", 
                        Brec[ii].file, Brec[ii].lover, Brec[ii].hiver);
            else  wprintf(mLog," %s \n",Brec[ii].file);
         }
         if (checkKillPause()) break;
      }

      goto ready_exit;
   }

ready_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  file restore dialog - specify backup files to be restored

int RJedit(cchar *menu)
{
   zdialog        *zd;
   
   wprintf(mLog,"\n Restore files from backup \n");   

   if (bGetFiles() < 0) return 0;                                          //  get files in backup location
   wprintf(mLog,"   %d backup files found \n",Bnf);
   if (! Bnf) return 0;

   ++Fdialog;
   
   zd = zdialog_new("copy files from backup",mWin,"browse","done","cancel",null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labfrom","vb1","copy-from backup");      //  copy-from backup    [_____________]
   zdialog_add_widget(zd,"label","labto","vb1","copy-to disk");            //  copy-to disk        [_____________]
   zdialog_add_widget(zd,"entry","entfrom","vb2",RJfrom);
   zdialog_add_widget(zd,"entry","entto","vb2",RJto);
   zdialog_add_widget(zd,"hsep","hsep1","dialog");
   zdialog_add_widget(zd,"label","labf","dialog","files to restore");      //  files to restore
   zdialog_add_widget(zd,"frame","framef","dialog",0,"expand");            //  scrolling edit window
   zdialog_add_widget(zd,"scrwin","scrf","framef");
   zdialog_add_widget(zd,"edit","editf","scrf");

   editwidget = zdialog_widget(zd,"editf");

   if (RJnx)
      for (int ii = 0; ii < RJnx; ii++)                                    //  get restore include/exclude recs,
         wprintf(editwidget,"%s""\n",RJrec[ii]);                           //   pack into file selection edit box

   zdialog_resize(zd,400,400);

   zdialog_run(zd,0,RJedit_compl);                                         //  run dialog with response function
   return 0;
}


//  restore dialog completion function
//  get restore job data from dialog widgets and validate

int RJedit_compl(zdialog *zd, int zstat)
{
   DIR         *pdirk;
   char        *pp, *fspec, rdirk[300];
   int         ftf = 1, cc, rtype, nerrs = 0;
   const char  *errmess;

   if (zstat != 1 && zstat != 2) goto end_dialog;                          //  cancel or destroy
   
   if (zstat == 1) 
   {                                                                       //  "browse" button, file-chooser dialog
      zdialog_fetch(zd,"entfrom",RJfrom,299);                              //  copy-from location /dirk/xxx/.../
      strTrim(RJfrom);
      strcpy(rdirk,BJdirk);                                                //  start at /media/xxx/dirk/xxx/
      strncat(rdirk,RJfrom,299);
      fc_dialog(rdirk);
      return 0;                                                            //  dialog continues
   }

   RJreset();                                                              //  edit done, reset job data

   zdialog_fetch(zd,"entfrom",RJfrom,299);                                 //  copy-from location /dirk/xxx/.../
   strTrim(RJfrom);

   strcpy(rdirk,BJdirk);                                                   //  validate copy-from location
   strncat(rdirk,RJfrom,299);                                              //  /media/xxx/dirk/...
   pdirk = opendir(rdirk);
   if (! pdirk) {
      wprintf(mLog," *** invalid copy-from location \n");
      nerrs++;
   }
   else closedir(pdirk);

   cc = strlen(RJfrom);                                                    //  insure '/' at end
   if (RJfrom[cc-1] != '/') strcat(RJfrom,"/");

   zdialog_fetch(zd,"entto",RJto,299);                                     //  copy-to location  /dirk/yyy/.../
   strTrim(RJto);

   pdirk = opendir(RJto);                                                  //  validate copy-to location
   if (! pdirk) {
      wprintf(mLog," *** invalid copy-to location \n");
      nerrs++;
   }
   else closedir(pdirk);

   cc = strlen(RJto);                                                      //  insure '/' at end
   if (RJto[cc-1] != '/') strcat(RJto,"/");

   for (RJnx = 0; RJnx < maxnx; RJnx++)                                    //  include/exclude recs from edit box
   {
      pp = wscanf(editwidget,ftf);                                         //  next record from edit widget
      if (! pp) break;

      strTrim(pp);                                                         //  remove trailing blanks
      wprintf(mLog," %s \n",pp);
      
      RJrec[RJnx] = strdupz(pp);                                           //  save job record
      RJrtype[RJnx] = 0;
      RJfspec[RJnx] = null;                                                //  bug fix
      
      errmess = parseJobrec(pp,rtype,fspec);                               //  parse and validate job rec
      if (errmess) {
         wprintf(mLog," *** %s \n",errmess);
         nerrs++;
         continue;
      }
      
      if (rtype == 0) continue;                                            //  comment

      if (rtype != 2 && rtype != 3) {                                      //  allow only include, exclude
         wprintf(mLog," *** invalid include/exclude record \n");
         nerrs++;
         continue;
      }
      
      RJrtype[RJnx] = rtype;
      RJfspec[RJnx] = fspec;
   }

   if (RJnx == maxnx) {
      wprintf(mLog," *** max job records exceeded \n");
      nerrs++;
   }

   if (nerrs == 0) RJval = 1;
   if (RJval) rGetFiles();                                                 //  get files to restore

end_dialog:
   zdialog_destroy(zd);                                                    //  destroy dialog
   --Fdialog;
   return 0;
}


//  thread function, list and validate backup files to be restored

int RJlist(cchar *menu)
{
   int       cc1, cc2, errs = 0;
   char     *file1, file2[maxfcc];
   
   if (! RJval) wprintf(mLog," *** restore job has errors \n");
   if (! Rnf) goto ready_exit;

   wprintf(mLog,"\n copy %d files from backup: %s \n",Rnf, RJfrom);
   wprintf(mLog,"    to directory: %s \n",RJto);
   wprintf(mLog,"\n resulting files will be the following: \n");
   
   cc1 = strlen(RJfrom);                                                   //  from: /dirk/xxx/.../
   cc2 = strlen(RJto);                                                     //    to: /dirk/yyy/.../

   for (int ii = 0; ii < Rnf; ii++)
   {
      if (checkKillPause()) break;

      file1 = Rrec[ii].file;

      if (! strnEqu(file1,RJfrom,cc1)) {
         wprintf(mLog," *** not within copy-from: %s \n",file1);
         errs++;
         continue;
      }
      
      strcpy(file2,RJto);
      strcpy(file2+cc2,file1+cc1);
      wprintf(mLog," %s \n",file2);
   }

   if (errs) {
      wprintf(mLog," *** %d errors \n",errs);
      RJval = 0;
   }

ready_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, restore files based on data from restore dialog

int Restore(cchar *menu)
{
   int         ii, nn, ccf;
   char        dfile[maxfcc];
   cchar       *errmess;

   if (! RJval || ! Rnf) {
      wprintf(mLog," *** restore job has errors \n");
      goto ready_exit;
   }

   nn = zmessageYN("Restore %d files from: %s%s \n     to: %s \n"
                   "Proceed with file restore ?",Rnf,BJdirk,RJfrom,RJto);
   if (! nn) goto ready_exit;
   
   snprintf(dfile,maxfcc-2,"\n""begin restore of %d files to: %s \n",Rnf,RJto);
   wprintx(mLog,0,dfile,boldfont);

   ccf = strlen(RJfrom);                                                   //  from: /media/xxx/filespec

   for (ii = 0; ii < Rnf; ii++)
   {
      if (checkKillPause()) goto ready_exit;
      strcpy(dfile,RJto);                                                  //  to: /destination/filespec
      strcat(dfile,Rrec[ii].file + ccf);
      wprintf(mLog," %s \n",dfile);
      errmess = copyFile(Rrec[ii].file,dfile,1);
      if (errmess) wprintf(mLog," *** %s \n",errmess);
      else Rrec[ii].finc = 1;
   }

   synch_poop("restore");                                                  //  synch owner and permissions data  v.26

ready_exit:   
   wprintf(mLog," ready \n");
   return 0;
}


//  format disk backup device with vfat or ext2 file system
//  uses existing partitions only - no changes to partition table          //  v.3.3.1

int Format(cchar *menu)
{
   int         ii, jj, zstat, yn, contx = 0;
   char        text[200], device[20], filesys[20], label[20], *crec;
   zdialog     *zd;
   FILE        *fid;
   
   wprintf(mLog,"\n Format a backup device \n");

   zd = zdialog_new("format backup device",mWin,"start","cancel",null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1","","homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1","","homog|expand");            //   backup device   [________][v]
   zdialog_add_widget(zd,"label","labdev","vb1"," backup device");         //   device label    [________]
   zdialog_add_widget(zd,"comboE","entdev","vb2");                         //   file system     [________][v]
   zdialog_add_widget(zd,"label","lablab","vb1","    device label");
   zdialog_add_widget(zd,"entry","entlab","vb2","ukopp");
   zdialog_add_widget(zd,"label","labfs","vb1","      file system");
   zdialog_add_widget(zd,"comboE","entfs","vb2","ext2");

   unmount("");                                                            //  unmount mounted device
   BDpoop();                                                               //  refresh available devices

   for (ii = 0; ii < Ndisk; ii++)                                          //  load combo box with device
   {
      strcpy(text,diskdev[ii]);                                            //  /dev/xxxx  description
      strncatv(text,199,"  ",diskdesc[ii],null);
      zdialog_cb_app(zd,"entdev",text);
   }
   
   zdialog_cb_app(zd,"entfs","ext2");                                      //  load combo box with file systems
   zdialog_cb_app(zd,"entfs","vfat");

   zdialog_resize(zd,300,0);
   zstat = zdialog_run(zd);                                                //  run dialog - modal
   zdialog_destroy(zd);
   if (zstat != 1) return 0;

   zdialog_fetch(zd,"entdev",device,19);                                   //  get chosen device and file system
   zdialog_fetch(zd,"entfs",filesys,19);
   zdialog_fetch(zd,"entlab",label,19);
   
   for (ii = 1; device[ii] > ' '; ii++);                                   //  strip off device description
   if (ii > 19) ii = 19;
   device[ii] = 0;

   yn = zmessageYN("device: %s  label: %s  file sys: %s \n"
                   "WARNING: all data will be lost! \n"
                   "Proceed with formatting?",device,label,filesys);
   if (! yn) goto ready_exit;

   wprintf(mLog," formatting %s with file system %s \n",device,filesys);
   
   fid = fopen(TFformatscript,"w");
   if (! fid) { 
      wprintf(mLog," *** cannot create format script file \n"); 
      goto ready_exit;
   }
   
   fprintf(fid,"umount %s \n",device);                                     //  unmount /dev/xxxx
   fprintf(fid,"sleep 2 \n");
   if (*filesys == 'v')
      fprintf(fid,"mkfs -t vfat -F 32 -n %s %s \n",label,device);          //  make vfat file system
   if (*filesys == 'e') 
      fprintf(fid,"mkfs -t ext2 -L %s %s \n",label,device);                //  or ext2 file system
   fprintf(fid,"exit 0 \n");
   fclose(fid);
   chmod(TFformatscript,0744);

   while ((crec = command_output(contx,TFformatscript)))                   //  v.3.3.1
   {
      zsleep(0.1);                                                         //  throttle a little
      for (ii = jj = 0; crec[jj]; jj++)
      {                                                                    //  get rid of weird characters
         if (crec[jj] < ' ') continue;                                     //    in mkfs output
         crec[ii] = crec[jj];
         ii++;
      }
      crec[ii] = 0;
      wprintf(mLog," format: %s \n",crec);                                 //  print command output
   }

ready_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  thread function to display help/about or help/contents

int helpFunc(cchar *menu)
{
   if (strEqu(menu,"about")) {
      wprintf(mLog," %s \n",ukopp_title);
      wprintf(mLog," free software: %s \n",ukopp_license);
   }

   if (strEqu(menu,"contents")) showz_helpfile();
   return 0;
}


//  parse and validate a backup or restore job record
//  filespec* means a /path.../filename with wildcards                     //  v.3.2
//
//  # comment
//  retain  ND  NV  filespec*
//  include  filespec*
//  exclude  filespec*
//  verify  [ none | incr | full | comp ]
//  target  [ /dev/xxx | /directory/ ] [ sync | remount ]                  //  opt. flush method    v.3.2

cchar * parseJobrec(const char *jobrec, int &rtype, char *&fspec)
{
   int         dd, vv, nn, Nth = 1, err = 0;
   const char  *pp1, *pp2, *errmess;
   
   rtype = -1;      
   fspec = null;
   
   pp1 = strField(jobrec,' ',Nth++);
   if (! pp1 || *pp1 == 0 || *pp1 == '#') {                                //  comment record
      rtype = 0;
      return 0;
   }

   if (strEqu(pp1,"retain")) rtype = 1;
   if (strEqu(pp1,"include")) rtype = 2;
   if (strEqu(pp1,"exclude")) rtype = 3;
   if (strEqu(pp1,"verify")) rtype = 4;
   if (strEqu(pp1,"target")) rtype = 5;
   if (rtype < 1) return "unrecognized record type";

   if (rtype == 1) {                                                       //  retain  ND  NV  filespec*
      pp1 = strField(jobrec,' ',Nth++);
      if (! pp1) return "no retention days";
      err = convSI(pp1,dd,0,9999);
      if (err) return "retention days not 0-9999";
      pp1 = strField(jobrec,' ',Nth++);
      if (! pp1) return "no retention versions";
      err = convSI(pp1,vv,0,9999);
      if (err) return "retention versions not 0-9999";
      pp1 = strField(jobrec,' ',Nth++);
      if (! pp1) fspec = strdupz("*");                                     //  no filespec >> all files
      else  fspec = strdupz(pp1);
      pp1 = strchr(fspec,'*');
      if (! pp1) return "retention filespec needs wildcard";
      return 0;
   }
   
   if (rtype == 2 || rtype == 3) {                                         //  [ include | exclude ] /path/spec*
      pp1 = strField(jobrec,' ',Nth++);
      if (! pp1) return "missing filespec";
      fspec = strdupz(pp1);
      if (fspec[0] != '/') return "filespec missing /topdir/";
      pp1 = strchr(fspec+1,'/');
      if (!pp1) return "filespec missing /topdir/";
      pp2 = strchr(fspec,'*');
      if (pp2 && pp2 < pp1) return "wildcards in /topdir/ not allowed";
      pp2 = strchr(fspec,'?');
      if (pp2 && pp2 < pp1) return "wildcards in /topdir/ not allowed";
      return 0;
   }

   if (rtype == 4) {                                                       //  verify  [ none | incr | full | comp ]
      pp1 = strField(jobrec,' ',Nth++);
      if (! pp1) return "missing verify type";
      nn = -1;
      if (strEqu(pp1,"none")) nn = 0;
      if (strnEqu(pp1,"incr",4)) nn = 1;
      if (strEqu(pp1,"full")) nn = 2;
      if (strnEqu(pp1,"comp",4)) nn = 3;
      if (nn < 0) return "verify not none, incr, full, or comp";
      BJvmode = nn;                                                        //  verify mode
      return 0;
   }

   if (rtype == 5) {                                                       //  target device & directory
      if (*BJdev && *BJdirk) return 0;                                     //  already set         v.3.2
      errmess = parseTarget(jobrec);
      return errmess;
   }

   return "unrecognized record type";
}


//  parse a target record and set target device and directory accordingly
//  update backup job target record to match device and/or directory
//  format: [ target ] [ /dev/xxx ] [ /directory ] [ sync | remount ]

cchar * parseTarget(const char *text)
{
   int            ii, err, cc, Nth = 1;
   int            warnuser = 0;
   char           ch;
   const char     *pp1;
   struct stat    dstat;

   bFilesReset();                                                          //  no files at backup location
   *BJdev = *BJdirk = BJdcc = 0;                                           //  no target device, directory

   pp1 = strField(text,' ',Nth++);
   if (pp1 && strEqu(pp1,"target")) pp1 = strField(text,' ',Nth++);

   if (! pp1) return "no backup target specified";
   if (strnEqu(pp1,"/dev/",5)) {
      strncpy0(BJdev,pp1,39);                                              //  have /dev/xxxx
      pp1 = strField(text,' ',Nth++);
   }

   if (pp1 && *pp1 == '/') {
      strncpy0(BJdirk,pp1,199);                                            //  have /directory
      BJdcc = strlen(BJdirk);
      pp1 = strField(text,' ',Nth++);
   }

   if (pp1) {
      if (strEqu(pp1,"sync")) BJflush = 1;                                 //  look for opt. flush method
      if (strEqu(pp1,"remount")) BJflush = 2;
   }
   
   if (! *BJdev && ! *BJdirk) return "no backup target specified";
   
   BDpoop();                                                               //  refresh device data
   
   if (*BJdev && ! *BJdirk) {                                              //  get mount point for device
      for (ii = 0; ii < Ndisk; ii++)
         if (strEqu(BJdev,diskdev[ii])) break;
      if (ii < Ndisk && *diskmp[ii] == '/') strcpy(BJdirk,diskmp[ii]);
   }
   
   if (! *BJdev && *BJdirk) {                                              //  get device for mount point
      for (ii = 0; ii < Ndisk; ii++)
         if (strEqu(BJdirk,diskmp[ii])) break;
      if (ii < Ndisk) strcpy(BJdev,diskdev[ii]);
   }
   
   if (! *BJdirk) strcpy(BJdirk,"/media/disk");
   BJdcc = strlen(BJdirk);                                                 //  keep target directory cc
   
   err = stat(BJdirk,&dstat);
   if (! err && S_ISDIR(dstat.st_mode)) {                                  //  directory already exists
      if (! *BJdev) goto already_mounted;                                  //  no device, accept directory
      for (ii = 0; ii < Ndisk; ii++)
         if (strEqu(BJdev,diskdev[ii])) break;                             //  look for device
      if (ii == Ndisk || *diskmp[ii] != '/') warnuser = 1;                 //  device apparently not mounted
      cc = strlen(diskmp[ii]);
      if (! strnEqu(diskmp[ii],BJdirk,cc)) warnuser = 1;                   //  directory not on this device
      ch = BJdirk[cc];
      if (ch && ch != '/') warnuser = 1;
      if (warnuser) 
            return "Target directory not on target device (?) \n"
                   "Please erase device or revise directory";
   already_mounted:
      wprintf(mLog," target directory is valid \n");
      devMounted = 1;                                                      //  device mounted, directory OK
      strcpy(mountdev,BJdev);                                              //  save mount poop
      strcpy(mountdirk,BJdirk);
      return 0;
   }

   if (! *BJdev) return "could not find target device";
   return 0;
}


//  Mount target device. Return 1 if success, else 0.
//  menu caller: menu arg is present
//  internal caller: menu arg is ""

int mount(cchar *menu)                                                     //  v.3.2
{
   int            err;
   char           work[200];

   if (devMounted) {
      if (strEqu(BJdev,mountdev) && strEqu(BJdirk,mountdirk)) {            //  already mounted, no changes
         if (*menu) wprintf(mLog," *** already mounted \n");
         return 1;
      }
      unmount("");                                                         //  target has changed
      devMounted = 0;
   }
   
   if (! *BJdev || ! *BJdirk) {
      wprintf(mLog," *** missing mount data: %s %s \n",BJdev,BJdirk);
      return 0;
   }

   snprintf(work,199,"mkdir -p %s",BJdirk);                                //  create mount point = target directory
   do_shell("mkdir",work);                                                 //  (OK if already there)

   snprintf(work,199,"mount -noatime %s %s",BJdev,BJdirk);                 //  mount device at target directory
   err = do_shell("mount",work);
   if (err) {
      wprintf(mLog," *** unable to mount target device \n");
      return 0;
   }

   wprintf(mLog," mounted OK \n");

   bFilesReset();                                                          //  no files at backup location   

   devMounted = 1;
   strcpy(mountdev,BJdev);                                                 //  save mount poop
   strcpy(mountdirk,BJdirk);

   return 1;
}


//  unmount target device

int unmount(cchar *menu)                                                   //  v.3.2
{
   int         ii;
   char        work[200];
   
   bFilesReset();                                                          //  no files at backup location
   
   if (! devMounted) {
      if (*menu) wprintf(mLog," *** not mounted \n");
      return 0;
   }

   devMounted = 0;
   
   snprintf(work,199,"umount %s",mountdirk);                               //  unmount backup device
   do_shell("umount",work);

   if (BJflush == 2) {                                                     //  if flush method = remount,
      snprintf(work,199,"rmdir %s",mountdirk);                             //    remove directory
      do_shell("rmdir",work);
   }
   
   BDpoop();                                                               //  refresh device data

   for (ii = 0; ii < Ndisk; ii++)
      if (strEqu(mountdev,diskdev[ii])) break;                             //  if mounted, unmount
   if (ii < Ndisk && *diskmp[ii] == '/') {
      snprintf(work,199,"umount %s",diskmp[ii]);
      do_shell("umount",work);
   }
   
   return 0;
}


//  flush I/O buffers in memory to physical device, between backup and verify

int flushcache()
{
   wprintf(mLog,"\n flushing file cache to backup device \n");             //  flush memory cache to device
   wprintf(mLog," (this may need some time ...) \n");

   if (BJflush == 1) do_shell("sync","sync");

   if (BJflush == 2) {
      unmount("");
      mount("");
   }
   
   return 0;
}


//  save logging window as text file

int saveScreen(cchar *menu)
{
   wfilesave(mLog);
   return 0;
}


//  backup helper function
//  write date and time to temp file

int writeDT()
{
   time_t      dt1;
   char        *dt2;
   FILE        *fid;
   int         cc;
   
   time(&dt1);
   dt2 = ctime(&dt1);                                                      //  get string date-time
   cc = strlen(dt2);
   if (cc && (dt2[cc-1] == '\n')) dt2[cc-1] = 0;                           //  save without trailing \n

   fid = fopen(TFdatetime,"w");
   if (! fid) zappcrash("cannot open scratch file %s",TFdatetime);

   fprintf(fid,"%s \n",dt2);
   fclose(fid);
   return 0;
}


//  synchronize owner and permissions data using poopfile at backup location        v.26
//   - for files copied backup >> disk, set owner and permissions from poopfile
//   - refresh poopfile data from disk files
//  mode is "backup" "restore" or "synch"

int synch_poop(const char *mode)
{
   int            ii, err, nn, uid, gid, perms;
   int            cc, ccf, cct;
   char           file[maxfcc], file2[maxfcc];
   char           dirk[maxfcc], pdirk[maxfcc];
   char           *pp, poopfile[100];
   cchar          *errmess = null;
   FILE           *fid;
   struct stat64  dstat;

   if (strEqu(mode,"synch"))                                               //  set poop for updated disk files
   {
      strcpy(poopfile,BJdirk);
      strcat(poopfile,BD_POOPFILE);
      fid = fopen(poopfile,"r");                                           //  open poopfile
      if (! fid) {
         wprintf(mLog," *** no owner/permissions file: %s \n",poopfile);
         return 0;
      }

      ii = 0;

      while (true)                                                         //  read poopfile records
      {
         nn = fscanf(fid,"%d:%d %o %[^\n]",&uid,&gid,&perms,file);         //  uid, gid, perms, file or directory
         if (nn == EOF) break;
         if (nn != 4) continue;
         
         cc = strlen(file);

         while (ii < Bnf)                                                  //  match poopfile file or directory
         {                                                                 //    to backup files copied to disk
            nn = strncmp(Brec[ii].file,file,cc);                           //  (logic assumes ascii sort)
            if (nn >= 0) break;
            ii++;
         }

         if (ii == Bnf) break;                                             //  EOL
         if (nn > 0) continue;                                             //  file not in backup file list
         if (Brec[ii].finc == 0) continue;                                 //  file not copied to disk

         wprintf(mLog," set owner/perms %d:%d %04o %s \n",uid,gid,perms,file);
         err = chown(file,uid,gid);
         if (err) wprintf(mLog," *** error: %s \n",strerror(errno));
         err = chmod(file,perms);
         if (err) wprintf(mLog," *** error: %s \n",strerror(errno));
      }
      
      fclose(fid);
   }

   if (strEqu(mode,"restore"))                                             //  set poop for restored disk files
   {
      strcpy(poopfile,BJdirk);
      strcat(poopfile,BD_POOPFILE);
      fid = fopen(poopfile,"r");
      if (! fid) {
         wprintf(mLog," *** no owner/permissions file: %s \n",poopfile);
         return 0;
      }

      ccf = strlen(RJfrom);
      cct = strlen(RJto);
      ii = 0;

      while (true)
      {
         nn = fscanf(fid,"%d:%d %o %[^\n]",&uid,&gid,&perms,file);
         if (nn == EOF) break;
         if (nn != 4) continue;
         
         cc = strlen(file);
         if (cc <= ccf) continue;

         while (ii < Rnf)
         {
            nn = strncmp(Rrec[ii].file,file,cc);
            if (nn >= 0) break;
            ii++;
         }

         if (ii == Rnf) break;
         if (nn > 0) continue;
         if (Rrec[ii].finc == 0) continue;
         
         strcpy(file2,RJto);                                               //  offset restore 'from' and 'to' paths
         strcpy(file2 + cct, file + ccf);

         wprintf(mLog," set owner/perms %d:%d %04o %s \n",uid,gid,perms,file2);
         err = chown(file2,uid,gid);
         if (err) wprintf(mLog," *** error: %s \n",strerror(errno));
         err = chmod(file2,perms);
         if (err) wprintf(mLog," *** error: %s \n",strerror(errno));
      }
      
      fclose(fid);
   }

   if (strEqu(mode,"backup") || strEqu(mode,"synch"))                      //  make new poop file from disk files
   {
      fid = fopen(TFpoopfile,"w");
      if (! fid) zappcrash("cannot open temp file %s",TFpoopfile);

      *pdirk = 0;                                                          //  no prior directory
      
      for (ii = 0; ii < Dnf; ii++)
      {
         strcpy(dirk,Drec[ii].file);                                       //  next file on disk
         pp = dirk;

         while (true)                                                      //  set directory owner & permissions
         {
            pp = strchr(pp+1,'/');                                         //  next (last) directory level
            if (! pp) break;
            cc = pp - dirk + 1;                                            //  cc incl. '/'
            if (strncmp(dirk,pdirk,cc) == 0) continue;                     //  matches prior, skip

            *pp = 0;                                                       //  terminate this directory level

            err = lstat64(dirk,&dstat);                                    //  get owner and permissions   v.3.0
            if (err) {
               wprintf(mLog," *** error: %s  file: %s \n",strerror(errno),dirk);
               break;
            }

            dstat.st_mode = dstat.st_mode & 0777;

            fprintf(fid,"%4d:%4d %3o %s/\n",                               //  output uid:gid perms directory/
                    dstat.st_uid, dstat.st_gid, dstat.st_mode, dirk);
            
            *pp = '/';                                                     //  restore '/'
         }
         
         strcpy(pdirk,dirk);                                               //  prior = this directory
         
         strcpy(file,Drec[ii].file);                                       //  disk file, again

         err = lstat64(file,&dstat);                                       //  get owner and permissions    v.3.0
         if (err) {
            wprintf(mLog," *** error: %s  file: %s \n",strerror(errno),file);
            continue;
         }

         dstat.st_mode = dstat.st_mode & 0777;

         fprintf(fid,"%4d:%4d %3o %s\n",                                   //  output uid:gid perms file
                 dstat.st_uid, dstat.st_gid, dstat.st_mode, file);
      }

      fclose(fid);

      errmess = copyFile(TFpoopfile,BD_POOPFILE,2);                        //  copy file owner/permissions file
      if (errmess) wprintf(mLog," *** poopfile error: %s \n",errmess);
   }

   return 0;
}


//  construct file-chooser dialog box 
//  note: Fdialog++ unnecessary: this dialog called from other dialogs

int fc_dialog(cchar *dirk)
{
   fc_dialogbox = gtk_dialog_new_with_buttons("choose files", 
                  GTK_WINDOW(mWin), GTK_DIALOG_MODAL, "hidden",100, 
                  "include",101, "exclude",102, "done",103, null);

   gtk_window_set_default_size(GTK_WINDOW(fc_dialogbox),600,500);
   G_SIGNAL(fc_dialogbox,"response",fc_response,0)

   fc_widget = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_OPEN);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fc_dialogbox)->vbox),fc_widget);

   gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fc_widget),dirk);
   gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(fc_widget),1);
   gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(fc_widget),0);

   gtk_widget_show_all(fc_dialogbox);
   return 0;
}


//  file-chooser dialog handler (file selection, done, cancel, kill)

int fc_response(GtkDialog *dwin, int arg, void *data)
{
   GtkTextBuffer  *textBuff;
   GSList         *flist = 0;
   char           *file1, *file2, *ppf;
   int             ii, err, hide;
   struct stat64   filestat;
   
   if (arg == 103 || arg == -4)                                            //  done or kill
   {
      gtk_widget_destroy(GTK_WIDGET(dwin));
      return 0;
   }
   
   if (arg == 100)                                                         //  toggle hidden files
   {
      hide = gtk_file_chooser_get_show_hidden(GTK_FILE_CHOOSER(fc_widget));
      hide = 1 - hide;
      gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(fc_widget),hide);
   }
   
   if (arg == 101 || arg == 102)                                           //  include, exclude
   {
      flist = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(fc_widget));

      for (ii = 0; ; ii++)                                                 //  process selected files
      {
         file1 = (char *) g_slist_nth_data(flist,ii);
         if (! file1) break;

         file2 = strdupz(file1,2);                                         //  extra space for wildcard
         g_free(file1);

         err = lstat64(file2,&filestat);
         if (err) wprintf(mLog," *** error: %s  file: %s \n",strerror(errno),file2);

         if (S_ISDIR(filestat.st_mode)) strcat(file2,"/*");                //  if directory, append wildcard
                                                                           //  (discard symlinks removed)   v.3.0
         ppf = file2;
         if (strnEqu(ppf,BJdirk,BJdcc)) ppf += BJdcc;                      //  omit backup mount point

         textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(editwidget));   //  copy selected file back into
         if (arg == 101) wprintf(editwidget,"include %s""\n",ppf);         //    job edit dialog, edit widget
         if (arg == 102) wprintf(editwidget,"exclude %s""\n",ppf);

         zfree(file2);
      }
   }

   gtk_file_chooser_unselect_all(GTK_FILE_CHOOSER(fc_widget));
   g_slist_free(flist);
   return 0;
}


//  get all disk files specified by include/exclude records
//  save in Drec[] array

int dGetFiles()
{
   const char     *fsp, *psep2;
   char           *fspec, *psep1;
   int            ftf, wstat, err, dups;
   int            rtype, ii, jj, st, nfiles;
   int            fcc, vers;
   double         nbytes;
   struct stat64  filestat;

   dFilesReset();
   wprintx(mLog,0,"\n""generating backup file set \n",boldfont);
   
   for (ii = 0; ii < BJnx; ii++)                                           //  process include/exclude recs
   {
      BJfiles[ii] = 0;                                                     //  initz. include/exclude rec stats
      BJbytes[ii] = 0.0;

      rtype = BJrtype[ii];
      fspec = BJfspec[ii];      

      if (rtype == 2)                                                      //  include filespec
      {
         ftf = 1;

         while (1)
         {
            fsp = SearchWild(fspec,ftf);                                   //  find matching files
            if (! fsp) break;

            Drec[Dnf].file = strdupz(fsp);

            err = lstat64(fsp,&filestat);                                  //  check accessibility
            if (! err) {
               Drec[Dnf].err = 0;
               if (! S_ISREG(filestat.st_mode) &&                          //  reg. files + symlinks only  v.3.0
                   ! S_ISLNK(filestat.st_mode)) continue;
            }
            else Drec[Dnf].err = errno;                                    //  save file error status
/**/
            fcc = strlen(fsp);
            psep1 = strstr(fsp+fcc-10,VSEP1);                              //  look for file version   v.3.2
            if (psep1) {
               vers = 0;
               st = convSI(psep1+2,vers,&psep2);                           //  if format not valid, take
               if (st < 2) vers = 1;                                       //    as non-versioned file
               if (strNeq(psep2,VSEP2)) vers = 0;
               if (*(psep2+1)) vers = 0;                                   //  VSEP2 must be at end 
               if (vers) {
                  wprintf(mLog," *** omit versioned file: %s \n",fsp);
                  continue;
               }
            }
/**/
            Drec[Dnf].jindx = ii;                                          //  save pointer to include rec
            Drec[Dnf].size = filestat.st_size;                             //  save file size
            Drec[Dnf].mtime = filestat.st_mtime                            //  save last mod time
                            + filestat.st_mtim.tv_nsec * nano;             //    (nanosec resolution)
            if (err) Drec[Dnf].size = Drec[Dnf].mtime = 0;                 //  inaccessible file
            Drec[Dnf].finc = 0;                                            //  not copied yet

            BJfiles[ii]++;                                                 //  count included files and bytes
            BJbytes[ii] += Drec[Dnf].size;

            if (++Dnf == maxfs) {
               wprintf(mLog," *** max files exceeded \n");
               break;
            }
         }
      }
 
      if (rtype == 3)                                                      //  exclude filespec
      {
         for (jj = 0; jj < Dnf; jj++)                                      //  check all included files (SO FAR)
         {
            if (! Drec[jj].file) continue;
            wstat = MatchWild(fspec,Drec[jj].file);
            if (wstat != 0) continue;
            BJfiles[ii]--;                                                 //  un-count excluded file and bytes
            BJbytes[ii] -= Drec[jj].size;
            zfree(Drec[jj].file);                                          //  clear file data entry
            Drec[jj].file = 0;
            Drec[jj].err = 0;
         }
      }
   }                                                                       //  end of include/exclude recs

   for (ii = 0; ii < Dnf; ii++)                                            //  list and remove error files
   {                                                                       //  (after excluded files removed)
      if (Drec[ii].err) {
         wprintf(mLog," *** %s  omit: %s \n",strerror(Drec[ii].err),Drec[ii].file);
         jj = Drec[ii].jindx;
         BJfiles[jj]--;                                                    //  un-count file and bytes
         BJbytes[jj] -= Drec[ii].size;
         zfree(Drec[ii].file);
         Drec[ii].file = 0;
      }
   }

   ii = jj = 0;                                                            //  repack file arrays after deletions
   while (ii < Dnf)
   {
      if (Drec[ii].file == 0) ii++;
      else {
         if (ii > jj) {
            if (Drec[jj].file) zfree(Drec[jj].file);
            Drec[jj] = Drec[ii];
            Drec[ii].file = 0;
         }
         ii++;
         jj++;
      }
   }

   Dnf = jj;                                                               //  final file count in backup set
   
   Dbytes = 0.0;
   for (ii = 0; ii < Dnf; ii++) Dbytes += Drec[ii].size;                   //  compute total bytes from files

   nfiles = 0;
   nbytes = 0.0;

   for (ii = 0; ii < BJnx; ii++)                                           //  compute total files and bytes
   {                                                                       //    from include/exclude recs
      nfiles += BJfiles[ii];
      nbytes += BJbytes[ii];
   }
   
   wprintf(mLog," disk files: %d  %.1f MB \n",nfiles,nbytes/mega);
   
   if ((nfiles != Dnf) || (Dbytes != nbytes)) {                            //  must match
      wprintf(mLog," *** bug: nfiles: %d  Dnf: %d \n",nfiles,Dnf);
      wprintf(mLog,"          nbytes: %.0f  Dbytes: %.0f \n",nbytes,Dbytes);
      goto errret;
   }

   SortFileList((char *) Drec,sizeof(dfrec),Dnf,'A');                      //  sort Drec[Dnf] by Drec[].file
   
   for (ii = dups = 0; ii < Dnf-1; ii++)                                   //  look for duplicate files
      if (strEqu(Drec[ii].file,Drec[ii+1].file)) {
         wprintf(mLog," *** duplicate file: %s \n",Drec[ii].file);
         dups++;
      }

   if (dups) goto errret;
   return 0;

errret:
   BJval = 0;
   dFilesReset();
   return 0;
}


//  get existing files at backup location, save in Brec[] array 
//  return -1 if error, else count of backup files
//
//  Linux sort command: 
//    '.' sorts before ' ' (0x2E < 0x20) which is crazy. 
//    Workaround implemented.

int bGetFiles()
{
   int            gcc, fcc, err, vers, jj;
   int            bb, bbp, rtype, noret = 0, expver;
   double         bfage;
   char           command[300], *pp, *psep1;
   char           bfile[maxfcc], *bfile2;
   const char     *psep2, *delim;
   FILE           *fid;
   struct stat64  filestat;

   bFilesReset();                                                          //  reset backup file list
   if (! mount("")) return 0;                                              //  validate and mount target  v.3.2

   wprintx(mLog,0,"\n""find all files at backup location \n",boldfont);

   sprintf(command,"find %s -type f -or -type l >%s",BJdirk,TFbakfiles);   //  backup filespecs to temp file   v.3.0
   err = do_shell("find",command);
   if (err) return -1; 

   //  read filespecs into memory and use memory sort instead of linux sort utility
   //  (apparently cannot do a straight ascii sort, even with LC_ALL=C)

   gcc = strlen(BD_UKOPPDIRK);                                             //  directory for ukopp special files

   fid = fopen(TFbakfiles,"r");                                            //  read file list
   if (! fid) zappcrash("cannot open scratch file %s",TFbakfiles);

   for (bb = 0; bb < maxfs; )                                              //  loop all files at backup location
   {
      pp = fgets_trim(bfile,maxfcc-1,fid);                                 //  next file
      if (! pp) break;                                                     //  eof
      
      bfile2 = bfile + BJdcc;                                              //  remove backup mount point
      if (strnEqu(bfile2,BD_UKOPPDIRK,gcc)) continue;

      fcc = strlen(bfile2);
      if (fcc > maxfcc-BJdcc-10) {                                         //  cannot handle files near limit
         wprintf(mLog," *** filespec too big, omit: %s...",bfile2);
         continue;
      }

      err = lstat64(bfile,&filestat);                                      //  check accessibility
      if (err) {
         wprintf(mLog," *** %s, omit: %s",strerror(errno),bfile2);
         continue;
      }
      else  if (! S_ISREG(filestat.st_mode) &&                             //  reg. files and symlinks only    v.3.0
                ! S_ISLNK(filestat.st_mode)) continue;

      //  build memory record for file data

      Brec[bb].file = strdupz(bfile2);                                     //  filespec
      Brec[bb].err = 0;
      Brec[bb].size = filestat.st_size;                                    //  file size
      Brec[bb].mtime = filestat.st_mtime                                   //  last mod time
                      + filestat.st_mtim.tv_nsec * nano;
      Brec[bb].lover = Brec[bb].hiver = Brec[bb].expver = 0;               //  set no versions yet
      Brec[bb].finc = 0;                                                   //  no backup yet
      bb++;
   }

   fclose (fid);

   Bnf = bb;
   wprintf(mLog," %6d backup files \n",Bnf);

   if (Bnf == maxfs) {
      wprintf(mLog," *** max files exceeded \n");
      bFilesReset();
      return -1;
   }

   SortFileList((char *) Brec,sizeof(bfrec),Bnf,'A');                      //  sort Brec[Bnf] by Brec[].file

   for (bb = 0, bbp = -1; bb < Bnf; bb++)                                  //  loop all files      revised  v.28
   {
      bfile2 = Brec[bb].file;
      fcc = strlen(bfile2);

      vers = 0;
      psep1 = strstr(bfile2+fcc-10,VSEP1);                                 //  look for file version
      if (psep1) {
         err = convSI(psep1+2,vers,1,9999,&psep2);                         //  if format not valid,
         if (err > 1) vers = 0;                                            //    assume a current file (vers 0)
         if (strNeq(psep2,VSEP2)) vers = 0;
         if (*(psep2+1)) vers = 0;                                         //  VSEP2 must be at end   v.3.2
         if (vers) *psep1 = 0;                                             //  remove version from file name
      }
      
      if (! vers)                                                          //  a current file, not prior version
      {
         bbp++;                                                            //  add new file record
         Brec[bbp] = Brec[bb];                                             //  copy all data
      }

      if (vers)                                                            //  a prior version, 1-9999
      {
         if (bbp > -1 && strEqu(Brec[bbp].file,bfile2)) {                  //  look back for match with prior file
            if (Brec[bbp].lover == 0) Brec[bbp].lover = vers;              //  set first version found
            if (vers < Brec[bbp].lover) Brec[bbp].lover = vers;            //  (10) sorts before (9)
            if (vers > Brec[bbp].hiver) Brec[bbp].hiver = vers;            //  track lowest and highest vers. found
            zfree(bfile2);                                                 //  free duplicate filespec
         }
         else  {                                                           //  version present, but no curr. file
            bbp++;                                                         //  add new file record
            Brec[bbp] = Brec[bb];                                          //  copy all data
            Brec[bbp].err = -1;                                            //  mark file (vers 0) not present
            Brec[bbp].size = Brec[bbp].mtime = 0;
            Brec[bbp].lover = Brec[bbp].hiver = vers;                      //  track prior versions present
         }
      }
   }

   Bnf = bbp + 1;

   for (bb = 0; bb < Bnf; bb++)
   {
      strcpy(bfile,BJdirk);                                                //  loop all files at backup location
      strcat(bfile,Brec[bb].file);
      bfile2 = bfile + BJdcc;

      if (BJnx > 0) {
         for (jj = 0; jj < BJnx; jj++) {                                   //  find matching backup include rec.
            rtype = BJrtype[jj];
            if (rtype != 2) continue;
            if (MatchWild(BJfspec[jj],bfile2) == 0) break;
         }
         if (jj == BJnx) {                                                 //  this file not in backup set
            Brec[bb].retND = Brec[bb].retNV = 0;                           //  no retention specs
            noret++;
         }
         else {
            for (jj--; jj >= 0; jj--) {                                    //  find prior matching retain rec.  v.22
               rtype = BJrtype[jj];
               if (rtype != 1) continue;
               if (MatchWild(BJfspec[jj],bfile2) == 0) break;
            }
            if (jj < 0) Brec[bb].retND = Brec[bb].retNV = 0;               //  none, no retention
            else {
               convSI(BJrec[jj]+6, Brec[bb].retND, &delim);
               convSI(delim+1, Brec[bb].retNV);
            }
         }
      }
      
      if (Brec[bb].err == 0) {
         Cfiles++;                                                         //  count curr. version files
         Cbytes += Brec[bb].size;                                          //    and total bytes
      }
      
      if (Brec[bb].lover == 0) continue;                                   //  no versions present

      expver = Brec[bb].hiver - Brec[bb].retNV;                            //  max expired version based on retNV
      if (expver < 0) expver = 0;

      for (vers = Brec[bb].lover; vers <= Brec[bb].hiver; vers++)          //  loop each file version
      {
         setFileVersion(bfile,vers);
         err = lstat64(bfile,&filestat);                                   //  check file exists on backup  v.3.0
         if (err) {
            wprintf(mLog," *** version %d missing: %s \n",vers,bfile2);
            continue;
         }

         Vfiles++;                                                         //  total files and bytes for versions
         Vbytes += filestat.st_size;

         bfage = (time(null) - filestat.st_mtime) / 24.0 / 3600.0;         //  file age in days
         if (bfage >= Brec[bb].retND && vers <= expver) {                  //  file expired for both NV and ND
            Brec[bb].expver = vers;                                        //  set highest version expired
            Pfiles++;                                                      //  total expired files and bytes
            Pbytes += filestat.st_size;                                    //  (to be purged)
         }
      }
   }

   wprintf(mLog," %6d files not in backup set (unknown retention) \n",noret);
   wprintf(mLog," %6d (%.3f MB) curr. file versions \n",Cfiles,Cbytes/mega);
   wprintf(mLog," %6d (%.3f MB) prior file versions \n",Vfiles,Vbytes/mega);
   wprintf(mLog," %6d (%.3f MB) expired prior versions \n",Pfiles,Pbytes/mega);

   return Bnf;
}


//  get all restore files specified by include/exclude records
//  save in Rrec[] array

int rGetFiles()
{
   int         ii, jj, cc, rtype, wstat, ninc, nexc;
   char       *fspec;

   if (! RJval) return 0;
   rFilesReset();                                                          //  clear restore files
   if (bGetFiles() < 1) return 0;                                          //  get backup files

   wprintf(mLog,"\n generating restore file set \n");
   
   for (ii = 0; ii < RJnx; ii++)                                           //  process include/exclude recs
   {
      wprintf(mLog," %s \n",RJrec[ii]);

      rtype = RJrtype[ii];
      fspec = RJfspec[ii];

      if (rtype == 2)                                                      //  include filespec
      {
         ninc = 0;                                                         //  count of included files

         for (jj = 0; jj < Bnf; jj++)                                      //  screen all files in backup loc.
         {
            wstat = MatchWild(fspec,Brec[jj].file);
            if (wstat != 0) continue;
            if (Brec[jj].err) continue;
            Rrec[Rnf].file = strdupz(Brec[jj].file);                       //  add matching files
            Rrec[Rnf].finc = 0;
            Rnf++; ninc++;
            if (Rnf == maxfs) {
               wprintf(mLog," *** max files exceeded \n");
               break;
            }
         }
            
         wprintf(mLog,"  %d files added \n",ninc);
      }

      if (rtype == 3)                                                      //  exclude filespec
      {
         nexc = 0;

         for (jj = 0; jj < Rnf; jj++)                                      //  check all included files (SO FAR)
         {
            if (! Rrec[jj].file) continue;

            wstat = MatchWild(fspec,Rrec[jj].file);
            if (wstat != 0) continue;
            zfree(Rrec[jj].file);                                          //  remove matching files
            Rrec[jj].file = 0;
            nexc++;
         }

         wprintf(mLog,"  %d files removed \n",nexc);
      }
   }

   ii = jj = 0;                                                            //  repack after deletions
   while (ii < Rnf)
   {
      if (Rrec[ii].file == 0) ii++;
      else
      {
         if (ii > jj) 
         {
            if (Rrec[jj].file) zfree(Rrec[jj].file);
            Rrec[jj].file = Rrec[ii].file;
            Rrec[ii].file = 0;
         }
         ii++;
         jj++;
      }
   }

   Rnf = jj;
   wprintf(mLog," total file count: %d \n",Rnf);

   cc = strlen(RJfrom);                                                    //  copy from: /dirk/.../

   for (ii = 0; ii < Rnf; ii++)                                            //  get selected backup files to restore
   {
      if (! strnEqu(Rrec[ii].file,RJfrom,cc)) {
         wprintf(mLog," *** not within copy-from; %s \n",Rrec[ii].file);
         RJval = 0;                                                        //  mark restore job invalid
         continue;
      }
   }

   SortFileList((char *) Rrec,sizeof(rfrec),Rnf,'A');                      //  sort Rrec[Rnf] by Rrec[].file
   return 0;
}


//  helper function for backups and reports
//
//  compare disk and backup files, set disp in Drec[] and Brec[] arrays:
//       n  new         on disk, not on backup
//       d  deleted     on backup, not on disk
//       m  modified    on both, but not equal
//       u  unchanged   on both, and equal
//       v  versions    on backup, only prev. versions present

int setFileDisps()
{
   int            dii, bii, comp;
   char           disp;
   double         diff;
   
   dii = bii = 0;
   nnew = nmod = nunc = ndel = comp = 0;
   Mbytes = 0.0;                                                           //  total bytes, new and modified files
   
   while ((dii < Dnf) || (bii < Bnf))                                      //  scan disk and backup files parallel
   {
      if ((dii < Dnf) && (bii == Bnf)) comp = -1;
      else if ((dii == Dnf) && (bii < Bnf)) comp = +1;
      else comp = strcmp(Drec[dii].file, Brec[bii].file);
      
      if (comp < 0) {                                                      //  unmatched disk file
         Drec[dii].disp = 'n';                                             //  new
         nnew++;                                                           //  count new files
         Mbytes += Drec[dii].size;                                         //  accumulate Mbytes
         dii++;
      }

      else if (comp > 0) {                                                 //  unmatched backup file
         if (Brec[bii].err == 0) {                                         //  if current version is present,
            Brec[bii].disp = 'd';                                          //    file was deleted from disk
            ndel++;                                                        //  count deleted files
         }
         else Brec[bii].disp = 'v';                                        //  only old versions on backup
         bii++;
      }

      else if (comp == 0) {                                                //  file present on disk and backup
         if (Brec[bii].err == 0) {
            diff = Drec[dii].mtime - Brec[bii].mtime;                      //  check if equal mod times
            if (fabs(diff) > modtimetolr) disp = 'm';                      //  (do not check file sizes   v.25)
            else disp = 'u';                                               //  yes, assume unchanged
            Drec[dii].disp = Brec[bii].disp = disp;
            if (disp == 'u') nunc++;                                       //  count unchanged files
            if (disp == 'm') nmod++;                                       //  count modified files
            if (disp == 'm') Mbytes += Drec[dii].size;                     //    and accumulate Mbytes
         }
         else {
            Brec[bii].disp = 'v';                                          //  only old versions on backup
            Drec[dii].disp = 'n';                                          //  disk file is logically new
            nnew++;                                                        //  count new files
            Mbytes += Drec[dii].size;                                      //  accumulate Mbytes
         }
         dii++;
         bii++;
      }
   }
   
   Mfiles = nnew + nmod + ndel;
   return 0;
}


//  Sort file list in memory (disk files, backup files, restore files).
//  Sort ascii sequence, or sort subdirectories in a directory before files.

int SortFileList(char *recs, int RL, int NR, char sort)
{
   HeapSortUcomp fcompA, fcompD;                                           //  filespec compare funcs
   if (sort == 'A') HeapSort(recs,RL,NR,fcompA);                           //  ascii compare
   if (sort == 'D') HeapSort(recs,RL,NR,fcompD);                           //  special compare (directories first)
   return 0;
}

int fcompA(cchar *rec1, cchar *rec2)                                       //  ascii comparison
{                                                                          //  current file (no version) sorts first
   dfrec  *r1 = (dfrec *) rec1;
   dfrec  *r2 = (dfrec *) rec2;
   return strcmp(r1->file,r2->file);
}

int fcompD(cchar *rec1, cchar *rec2)                                       //  special compare filenames
{                                                                          //  subdirectories in a directory compare
   dfrec  *r1 = (dfrec *) rec1;                                            //    less than files in the directory
   dfrec  *r2 = (dfrec *) rec2;
   return filecomp(r1->file,r2->file);
}

int filecomp(cchar *file1, cchar *file2)                                   //  special compare filenames
{                                                                          //  subdirectories compare before files
   cchar       *pp1, *pp10, *pp2, *pp20;
   cchar       slash = '/';
   int         cc1, cc2, comp;
   
   pp1 = file1;                                                            //  first directory level or file
   pp2 = file2;

   while (true)
   {
      pp10 = strchr(pp1,slash);                                            //  find next slash
      pp20 = strchr(pp2,slash);
      
      if (pp10 && pp20) {                                                  //  both are directories
         cc1 = pp10 - pp1;
         cc2 = pp20 - pp2;
         if (cc1 < cc2) comp = strncmp(pp1,pp2,cc1);                       //  compare the directories
         else comp = strncmp(pp1,pp2,cc2);
         if (comp) return comp;
         else if (cc1 != cc2) return (cc1 - cc2);
         pp1 = pp10 + 1;                                                   //  equal, check next level
         pp2 = pp20 + 1;
         continue;
      }
      
      if (pp10 && ! pp20) return -1;                                       //  only one is a directory,
      if (pp20 && ! pp10) return 1;                                        //    the directory is first
      
      comp = strcmp(pp1,pp2);                                              //  both are files, compare
      return comp;
   }
}


//  reset all backup job data and free allocated memory

int BJreset()
{
   for (int ii = 0; ii < BJnx; ii++) 
   {
      if (BJrec[ii]) zfree(BJrec[ii]);
      if (BJfspec[ii]) zfree(BJfspec[ii]);
   }

   BJnx = BJval = 0;
   BJvmode = 0;
   dFilesReset();                                                          //  reset dependent disk file data
   return 0;
}


//  reset all restore job data and free allocated memory

int RJreset()
{
   for (int ii = 0; ii < RJnx; ii++)
   {
      if (RJrec[ii]) zfree(RJrec[ii]);
      if (RJfspec[ii]) zfree(RJfspec[ii]);
   }

   RJval = RJnx = 0;
   rFilesReset();                                                          //  reset dependent disk file data
   return 0;
}


//  reset all file data and free allocated memory

int dFilesReset()
{                                                                          //  disk files data
   for (int ii = 0; ii < Dnf; ii++) 
   {
      zfree(Drec[ii].file);
      Drec[ii].file = 0;
   }

   Dnf = 0;
   Dbytes = Mbytes = 0.0;
   return 0;
}

int bFilesReset()
{                                                                          //  backup files data
   for (int ii = 0; ii < Bnf; ii++) 
   {
      zfree(Brec[ii].file);
      Brec[ii].file = 0;
   }

   Bbytes = Bnf = 0;
   Cbytes = Cfiles = 0;
   Mbytes = Mfiles = 0;
   Vbytes = Vfiles = 0;
   Pbytes = Pfiles = 0;
   return 0;
}

int rFilesReset()
{                                                                          //  restore files data
   for (int ii = 0; ii < Rnf; ii++) 
   {
      zfree(Rrec[ii].file);
      Rrec[ii].file = 0;
   }

   Rnf = 0;
   return 0;
}


//  Helper function to copy a file between disk and backup location.
//  Owner and permissions are transferred for copied files and directories,
//  but this will do nothing in case target is VFAT (Microsoft) file system.

cchar * copyFile(cchar *sfile, cchar *dfile, int mpf)
{
   char              file1[maxfcc], file2[maxfcc];
   int               fid1, fid2, err, rcc, dlevs, ignore;
   char              *pp1, *pp2, buff[vrcc];
   cchar             *errmess;
   struct stat64     fstat1, fstat2;
   struct timeval    ftimes[2];

   *file1 = *file2 = 0;   
   if (mpf == 1) strcpy(file1,BJdirk);                                     //  prepend mount point if req.
   strcat(file1,sfile);
   if (mpf == 2) strcpy(file2,BJdirk);
   strcat(file2,dfile);
   
   pp2 = file2;
   dlevs = 0;

   while (true) {                                                          //  v.25
      pp2 = strchr(pp2+1,'/');                                             //  create missing directory levels
      if (! pp2) break;                                                    //  (check and create from top down)
      *pp2 = 0;
      err = stat64(file2,&fstat2);
      if (err) {
         err = mkdir(file2,0731);
         if (err) return strerror(errno);
         dlevs++;
      }
      *pp2 = '/';
   }

   while (dlevs) {                                                         //  v.25
      pp1 = strrchr(file1,'/');                                            //  for created output directories, 
      if (! pp1) break;                                                    //   copy owner and permissions from
      pp2 = strrchr(file2,'/');                                            //    corresponding input directory
      if (! pp2) break;                                                    //     (measured from bottom up)
      *pp1 = *pp2 = 0;                                                     //  (possibly top levels not set)
      err = stat64(file1,&fstat1);
      if (err) return strerror(errno);
      chmod(file2,fstat1.st_mode);
      ignore = chown(file2,fstat1.st_uid,fstat1.st_gid);
      dlevs--;
   }

   *file1 = *file2 = 0;   
   if (mpf == 1) strcpy(file1,BJdirk);                                     //  refresh filespecs
   strcat(file1,sfile);
   if (mpf == 2) strcpy(file2,BJdirk);
   strcat(file2,dfile);

   err = lstat64(file1,&fstat1);                                           //  get input file attributes  v.3.0
   if (err) return strerror(errno);
   
   if (S_ISLNK(fstat1.st_mode)) {                                          //  input file is symlink
      rcc = readlink(file1,buff,maxfcc);
      if (rcc < 0 || rcc > maxfcc-2) return strerror(errno);
      buff[rcc] = 0;
      err = symlink(buff,file2);                                           //  create output symlink
      if (err) return strerror(errno);
      ftimes[0].tv_sec = fstat1.st_atime;                                  //  get input file access time  v.3.0
      ftimes[0].tv_usec = fstat1.st_atim.tv_nsec / 1000;                   //    in microsecs.
      ftimes[1].tv_sec = fstat1.st_mtime;
      ftimes[1].tv_usec = fstat1.st_mtim.tv_nsec / 1000;
      lutimes(file2,ftimes);                                               //  set output file access time
      return 0;
   }

   fid1 = open(file1,O_RDONLY+O_NOATIME+O_LARGEFILE);                      //  open input file
   if (fid1 == -1) return strerror(errno);

   fid2 = open(file2,O_WRONLY+O_CREAT+O_TRUNC+O_LARGEFILE,0700);           //  open output file
   if (fid2 == -1) {
      errmess = strerror(errno);
      close(fid1);
      return errmess;
   }
   
   while (true)
   {
      rcc = read(fid1,buff,vrcc);                                          //  read huge blocks
      if (rcc == 0) break;
      if (rcc == -1) {
         errmess = strerror(errno);
         close(fid1);
         close(fid2);
         return errmess;
      }

      rcc = write(fid2,buff,rcc);                                          //  write blocks
      if (rcc == -1) {
         errmess = strerror(errno);
         close(fid1);
         close(fid2);
         return errmess;
      }
   }

   close(fid1);                                                            //  close files
   err = close(fid2);
   if (err) return strerror(errno);                                        //  output file I/O error

   err = lstat64(file1,&fstat1);                                           //  get input file attributes  v.3.0
   if (err) return strerror(errno);

   chmod(file2,fstat1.st_mode);                                            //  copy owner and permissions
   ignore = chown(file2,fstat1.st_uid,fstat1.st_gid);                      //    from input to output file

   ftimes[0].tv_sec = fstat1.st_atime;                                     //  get input file access time
   ftimes[0].tv_usec = fstat1.st_atim.tv_nsec / 1000;                      //    in microsecs.
   ftimes[1].tv_sec = fstat1.st_mtime;
   ftimes[1].tv_usec = fstat1.st_mtim.tv_nsec / 1000;
   utimes(file2,ftimes);                                                   //  set output file access time

   return 0;
}


//  helper function to delete a file from backup location.
//  delete parent directories if they are now empty.

cchar * deleteFile(cchar *file)
{
   int      err;
   char     dfile[maxfcc], *pp;

   strcpy(dfile,BJdirk);                                                   //  delete file
   strcat(dfile,file);   
   err = remove(dfile);
   if (err) return strerror(errno);
   
   while ((pp = strrchr(dfile,'/')))                                       //  delete empty directory
   {
      *pp = 0;
      err = rmdir(dfile);
      if (! err) continue;                                                 //  and parents ...
      if (errno == ENOTEMPTY) return 0;
      return strerror(errno);
   }
   
   return 0;
}


//  Verify helper function
//  Verify that file on backup medium is readable, return its length.
//  Optionally compare backup file to disk file, byte for byte.
//  returns error message or null if OK.

cchar * checkFile(cchar *dfile, int compf, double &tcc)
{
   int            vfid = 0, dfid = 0;
   int            err, vcc, dcc, cmperr = 0;
   char           vfile[maxfcc], *vbuff = 0, *dbuff = 0;
   cchar          *errmess = 0;
   double         dtime, vtime;
   struct stat64  filestat;
   static int     open_flags = O_RDONLY+O_NOATIME+O_LARGEFILE;
   
   tcc = 0.0;

   strcpy(vfile,BJdirk);                                                   //  prepend mount point
   strcat(vfile,dfile);

   lstat64(vfile,&filestat);                                               //  if symlink, check readable   v.3.1
   if (S_ISLNK(filestat.st_mode)) {
      vbuff = (char *) malloc(maxfcc);
      vcc = readlink(vfile,vbuff,maxfcc);
      if (vcc == -1) errmess = strerror(errno);
      goto cleanup;
   }

   if (compf) goto comparefiles;
   
   vfid = open(vfile,open_flags);
   if (vfid == -1) return strerror(errno);

   err = posix_memalign((void**) &vbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");

   while (1)
   {
      vcc = read(vfid,vbuff,vrcc);
      if (vcc == 0) break;
      if (vcc == -1) { errmess = strerror(errno); break; }
      tcc += vcc;                                                          //  accumulate length      
      if (checkKillPause()) break;
   }
   goto cleanup;

comparefiles:

   vfid = open(vfile,open_flags);
   if (vfid == -1) return strerror(errno);

   dfid = open(dfile,open_flags);
   if (dfid == -1) { errmess = strerror(errno); goto cleanup; }

   err = posix_memalign((void**) &vbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");
   err = posix_memalign((void**) &dbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");

   while (1)
   {
      vcc = read(vfid,vbuff,vrcc);                                         //  read two files
      if (vcc == -1) { errmess = strerror(errno); goto cleanup; }

      dcc = read(dfid,dbuff,vrcc);
      if (dcc == -1) { errmess = strerror(errno); goto cleanup; }

      if (vcc != dcc) cmperr++;                                            //  compare buffers 
      if (memcmp(vbuff,dbuff,vcc)) cmperr++;

      tcc += vcc;                                                          //  accumulate length
      if (vcc == 0) break;
      if (dcc == 0) break;
      if (checkKillPause()) break;
   }

   if (vcc != dcc) cmperr++;

   if (cmperr) {                                                           //  compare error
      lstat64(dfile,&filestat);                                            //                                  v.3.0
      dtime = filestat.st_mtime + filestat.st_mtim.tv_nsec * nano;         //  file modified since snapshot?
      lstat64(vfile,&filestat);                                            //                                  v.3.0
      vtime = filestat.st_mtime + filestat.st_mtim.tv_nsec * nano;
      if (fabs(dtime-vtime) < modtimetolr) errmess = "compare error";      //  no, a real compare error
   }

cleanup:
   if (vfid) close(vfid);
   if (dfid) close(dfid);
   if (vbuff) free(vbuff);
   if (dbuff) free(dbuff);
   return errmess;
}


//  modify filespec to have a specified version
//  0 = no version = current version, +N = previous version
//  returns cc of resulting filespec
//  warning: filespec must have space for version numbers

int setFileVersion(char *filespec, int vers)
{
   int         fcc, overs, err;
   char        *psep1;
   const char  *psep2;

   fcc = strlen(filespec);
   psep1 = strstr(filespec+fcc-10,VSEP1);                                  //  look for file version   v.3.2
   if (psep1) {
      overs = 0;
      err = convSI(psep1+2,overs,&psep2);                                  //  if format not valid, take
      if (err < 2) overs = 1;                                              //    as non-versioned file
      if (strNeq(psep2,VSEP2)) overs = 0;
      if (*(psep2+1)) overs = 0;                                           //  VSEP2 must be at end 
      if (overs) *psep1 = 0;
      fcc = psep1 - filespec;
   }

   if (vers == 0) return fcc;
   
   if (! psep1) psep1 = filespec + fcc;
   strcpy(psep1,VSEP1);
   sprintf(psep1+2,"%d",vers);
   strcat(psep1+2,VSEP2);
   
   return fcc + strlen(psep1);
}


//  rename a backup file to assign the next version number
//  update the passed backup file data record, bakrec
//  returns error message or null if all OK

const char * renameFile(bfrec &bakrec)
{
   char     *fspec1, *fspec2;
   int      vers, fcc, err;
   
   fcc = strlen(bakrec.file);
   fspec1 = zmalloc(fcc + BJdcc + 1);
   fspec2 = zmalloc(fcc + BJdcc + 10);

   strcpy(fspec1,BJdirk);
   strcpy(fspec1 + BJdcc, bakrec.file);
   strcpy(fspec2,fspec1);

   vers = bakrec.hiver + 1;
   fcc = setFileVersion(fspec2,vers);
   
   err = rename(fspec1,fspec2);
   if (err) {
      zfree(fspec1);
      zfree(fspec2);
      return strerror(errno);
   }

   zfree(fspec1);
   zfree(fspec2);
   bakrec.hiver = vers;
   return null;
}


//  purge expired file versions in backup location
//  update backup file data record bakrec
//  returns error message or null if all OK

const char * purgeFile(bfrec &bakrec)
{
   int         vers;
   char        *fspec;
   const char  *mess = null;

   fspec = strdupz(bakrec.file,10);

   for (vers = bakrec.lover; vers <= bakrec.expver; vers++)
   {
      setFileVersion(fspec,vers);
      mess = deleteFile(fspec);                                            //  prepends mount point
      if (mess) break;
      bakrec.lover = vers + 1;
   }
   
   if (mess) printf("delete backup file: %s \n  error: %s \n",fspec,mess);
   
   zfree(fspec);
   bakrec.expver = 0;
   return mess;
}


//  do shell command (subprocess) and echo outputs to log window
//  returns command status: 0 = OK, +N = error

int do_shell(cchar *pname, cchar *command)
{
   char     buff[500], *crec;
   int      err, contx = 0;

   snprintf(buff,499,"\n""shell: %s \n",command);
   wprintx(mLog,0,buff,boldfont);
   printf(" %s \n",command);

   while ((crec = command_output(contx,command)))                          //  bug fix: remove colon   v.3.2
   {
      wprintf(mLog," %s: %s \n",pname,crec);
      zsleep(0.1);                                                         //  throttle output a little
   }
   
   err = command_status(contx);
   if (err) wprintf(mLog," %s status: %s \n", pname, strerror(err));
   if (err == 32) {
      wprintf(mLog," (already mounted?) \n");                              //  Linux broken pipe BS
      err = 0;
   }
   return err;
}





