/***************************************************************************
 *   Copyright (C) 2005 by Oleg Martemjanov                                *
 *   demogorgorn@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "indextab.h"
#include "variouswidgets.h"
#include "htmlstyles.h"

#include <qdir.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qtextcodec.h>

#include <qlayout.h>
#include <qwidgetstack.h>
#include <qlabel.h>
#include <qframe.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qregexp.h>
#include <qprocess.h>
#include <qdatetime.h>

#include <klineedit.h>
#include <kpushbutton.h>
#include <klocale.h>
#include <ktabwidget.h>
#include <ktextedit.h>
#include <kurlrequester.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kmessagebox.h>

IndexTab::IndexTab(QWidget *p, const char *name) :
	QWidget(p, name), parent(p)
{
	config = KGlobal::config();
	
	QVBoxLayout *inLay = new QVBoxLayout(this);
	inLay->setAlignment(Qt::AlignTop);
	
	setupPages();
	connectSlots();
	
	loadSettings();

	KTabWidget *tab = new KTabWidget(this);
	tab->addTab(page_1,SmallIcon("folder"),i18n("Directory"));
	tab->addTab(page_2,SmallIcon("exec"),i18n("File Types"));
	tab->addTab(page_3,SmallIcon("html"),i18n("HTML Document"));
	tab->addTab(page_4,SmallIcon("enhanced_browsing"),i18n("Result Page"));
	tab->addTab(page_5,SmallIcon("launch"),i18n("Index"));
	
	inLay->add(tab);
}

void IndexTab::setupPages()
{
	setupPage1();
	setupPage2();
	setupPage3();
	setupPage4();
	setupPage5();
}

void IndexTab::setupPage1()
{
	// Directory to index tab
	page_1 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_1, 11, 6 );
    
	QGridLayout *topLay = new QGridLayout( 1, 1, 2, "Layout1");
	topLay->setAlignment(Qt::AlignTop);

	QLabel *m_pathLbl = new QLabel(i18n("Directory to index :"),page_1);
	topLay->addWidget(m_pathLbl,0,0);
	
	m_dirPathReq = new KDirRequester(page_1);
	topLay->addWidget(m_dirPathReq,1,0);
	
	m_writeNamesCb = new QCheckBox(i18n("Write directories\'s names"),page_1) ;
	m_writeNamesCb->setChecked(TRUE);
	topLay->addWidget(m_writeNamesCb,2,0);
	
	m_dntEnterCb = new QCheckBox(i18n("Do not enter \"_files\" directory"),page_1);
	m_dntEnterCb->setChecked(TRUE);
	topLay->addWidget(m_dntEnterCb,3,0);
	
	m_recursiveCb = new QCheckBox(i18n("Recursive"),page_1);
	m_recursiveCb->setChecked(TRUE);
	topLay->addWidget(m_recursiveCb,4,0);
	
	m_sortDirsCb = new QCheckBox(i18n("Sort files in directories"),page_1);
	m_sortDirsCb->setChecked(FALSE);
	topLay->addWidget(m_sortDirsCb,5,0);

	/// @todo do later a custom widget - MyLine
	QFrame *m_line1 = new QFrame(page_1);
	m_line1->setFrameShape( QFrame::HLine );
	m_line1->setFrameShadow( QFrame::Sunken );
	m_line1->setFrameShape( QFrame::HLine );
	topLay->addWidget(m_line1,6,0);
   
	pageLayout->addLayout( topLay );
    
	QToolTip::add( m_writeNamesCb, i18n( "Writes folder\'s names in the result page." ) );
	QToolTip::add( m_dntEnterCb, i18n( "Do not index directories with \"_files\" suffix." ) );
	QToolTip::add( m_recursiveCb, i18n( "Index dirs recursively" ) );
}

void IndexTab::setupPage2()
{
	// File types tab
	page_2 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_2, 11, 1 );

	m_htmlDocsRb = new QRadioButton(i18n("HTML documents"),page_2);
	m_htmlDocsRb->setChecked(TRUE);
	
	m_otherDocsRb = new QRadioButton(i18n("Other types"),page_2);
	
	m_otherDocsEdit = new KTextEdit(page_2);
	m_otherDocsEdit->setText("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
	m_otherDocsEdit->setMaximumHeight(200);
	m_otherDocsEdit->setEnabled(FALSE);
	
	
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_htmlDocsRb );
	pageLayout->addWidget( m_otherDocsRb );
	pageLayout->addWidget( m_otherDocsEdit );
	pageLayout->addItem( spacer );
    
	QToolTip::add( m_htmlDocsRb, i18n( "Program will index common html files." ) );
	QToolTip::add( m_otherDocsRb, i18n( "You can define file types to index yourself." ) );
}

void IndexTab::setupPage3()
{
	// HTML document tab
	page_3 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_3, 11, 1 );

	QLabel *m_titleLbl = new QLabel(i18n("Index HTML document"),page_3);
	QLabel *m_titleFLbl = new QLabel(i18n("from"),page_3);
	
	m_titleFEdit = new KLineEdit("<TITLE>",page_3);
	
	QLabel *m_titleTLbl = new QLabel(i18n("to"),page_3);
	
	m_titleTEdit = new KLineEdit("</TITLE>",page_3);
	
	QHBoxLayout *horLay = new QHBoxLayout();
        horLay->addWidget( m_titleFLbl );
        horLay->addWidget( m_titleFEdit );
	horLay->addWidget( m_titleTLbl );
	horLay->addWidget( m_titleTEdit );
	
	m_removeILLCharsCb = new QCheckBox(i18n("Remove illegal charachters from title"),page_3);
	m_removeILLCharsCb->setChecked(TRUE);
	
	m_removeIncTagsCb = new QCheckBox(i18n("Remove included tags"),page_3);
	m_removeIncTagsCb->setChecked(TRUE);
	
	m_useDetectEncCb = new QCheckBox(i18n("Use automatic detection of encoding (using Enca library)"),page_3);
	
	QFrame *m_line6 = new QFrame(page_3);
	m_line6->setFrameShape( QFrame::HLine );
	m_line6->setFrameShadow( QFrame::Sunken );
	m_line6->setFrameShape( QFrame::HLine );
	
	
	m_indexContainCb = new QCheckBox(i18n("Index only pages that contain in title (You can write QRegExp rule if you want) "),page_3);
	
	m_indexContainEdit = new KLineEdit(page_3);
	m_indexContainEdit->setEnabled(FALSE);
	
	QHBoxLayout *horLay1 = new QHBoxLayout();
        horLay1->addWidget( m_indexContainCb );
        horLay1->addWidget( m_indexContainEdit );
	
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_titleLbl );
	pageLayout->addLayout( horLay );
	pageLayout->addWidget( m_removeILLCharsCb );
	pageLayout->addWidget( m_removeIncTagsCb );
	pageLayout->addWidget( m_useDetectEncCb );
	pageLayout->addWidget( m_line6 );
	pageLayout->addLayout( horLay1 );
	pageLayout->addItem( spacer );

//     
	QToolTip::add( m_removeILLCharsCb, i18n( "Remove unwanted charachters from title." ) );
	QToolTip::add( m_removeIncTagsCb, i18n( "Remove tags from the title." ) );
	QToolTip::add( m_useDetectEncCb, i18n( "If you have non-english pages please check this box." ) );
}

void IndexTab::setupPage4()
{
	// Result page tab
	page_4 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_4, 11, 4 );

	QLabel *m_resultPageLbl = new QLabel(i18n("Filename of the result page:"),page_4);
	
	m_resultPageReq = new KURLRequester(page_4);
	m_resultPageReq->setCaption(i18n("Filename of the result page"));
	m_resultPageReq->setFilter("*.html|HTML pages (*.html)");
	
	QFrame *m_line2 = new QFrame(page_4);
	m_line2->setFrameShape( QFrame::HLine );
	m_line2->setFrameShadow( QFrame::Sunken );
	m_line2->setFrameShape( QFrame::HLine );
	
	m_dotableCb = new QCheckBox(i18n("Write table instead simple layout"),page_4);
	
	QLabel *m_delimeterLbl = new QLabel(i18n("Delimeter beetween links (Simple Layout):"),page_4);
	
	m_brDelimeterRb = new QRadioButton(i18n("Break <br>"),page_4);
	m_brDelimeterRb->setChecked(TRUE);
	m_pDelimeterRb = new QRadioButton(i18n("Paragraph <p>"),page_4);
	
	QFrame *m_line3 = new QFrame(page_4);
	m_line3->setFrameShape( QFrame::HLine );
	m_line3->setFrameShadow( QFrame::Sunken );
	m_line3->setFrameShape( QFrame::HLine );
	
	m_useStylesheetCb = new QCheckBox(i18n("Use stylesheet for the result page"),page_4);
	
	m_stylesheetReq = new KURLRequester(page_4);
	m_stylesheetReq->setCaption(i18n("Choose stylesheet to use in page"));
	m_stylesheetReq->setFilter("*.css|StyleSheets (*.css)");
	
	QFrame *m_line4 = new QFrame(page_4);
	m_line4->setFrameShape( QFrame::HLine );
	m_line4->setFrameShadow( QFrame::Sunken );
	m_line4->setFrameShape( QFrame::HLine );
	
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_resultPageLbl );
	pageLayout->addWidget( m_resultPageReq );
	pageLayout->addWidget( m_line2 );
	pageLayout->addWidget( m_dotableCb );
	pageLayout->addWidget( m_delimeterLbl );
	pageLayout->addWidget( m_brDelimeterRb );
	pageLayout->addWidget( m_pDelimeterRb );
	pageLayout->addWidget( m_line3 );
	pageLayout->addWidget( m_useStylesheetCb );
	pageLayout->addWidget( m_stylesheetReq );
	pageLayout->addWidget( m_line4 );
	pageLayout->addItem( spacer );

	/// @todo write tooltips for Result tab widgets     
	QToolTip::add( m_useStylesheetCb, i18n( "If you want to use stylesheet just check this option"
		" and choose the file.\n"
		"If you leave this edit empty (and checkbox checked) program \n"
		"will use nice built-in stylesheet!" ) );
// 	QToolTip::add( m_removeIncTagsCb, i18n( "Remove tags from the title." ) );
// 	QToolTip::add( m_useDetectEncCb, i18n( "If you have non-english pages please check this box." ) );
}

void IndexTab::setupPage5()
{
	// File types tab
	page_5 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_5, 11, 1 );

	m_showPageCb = new QCheckBox(i18n("Show result page after finishing"),page_5);
	m_closeAppCb = new QCheckBox(i18n("Close application after proceed"),page_5);
	
	QFrame *m_line5 = new QFrame(page_5);
	m_line5->setFrameShape( QFrame::HLine );
	m_line5->setFrameShadow( QFrame::Sunken );
	m_line5->setFrameShape( QFrame::HLine ); 
	
	//m_logList = new KListView(page_5);
	QSpacerItem* spacer = new QSpacerItem( 20, 5, QSizePolicy::Minimum, QSizePolicy::Expanding );
	
	
	m_indexBtn = new KPushButton(i18n("Index"),page_5);
   
	pageLayout->addWidget( m_showPageCb );
	pageLayout->addWidget( m_closeAppCb );
	pageLayout->addWidget( m_line5 );
	//pageLayout->addWidget( m_allDocsTagsRb );
	//pageLayout->addWidget( m_logList );
	pageLayout->addItem( spacer );
	pageLayout->addWidget( m_indexBtn );
    
// 	QToolTip::add( m_htmlDocsRb, i18n( "Program will index common html files." ) );
// 	QToolTip::add( m_otherDocsRb, i18n( "You can define fyle types to index yourself." ) );
// 	QToolTip::add( m_allDocsTagsRb, i18n( "Program will index all files that contain tags." ) );
}

// void IndexTab::setupPage6()
// {
// 
// 	page_6 = new QWidget( parent );
// 	
// 	QVBoxLayout *pageLayout = new QVBoxLayout( page_6, 11, 1 );
// 	
// 	QLabel *m_renameLbl = new QLabel(i18n("Directory to rename files in"),page_6);
// 	
// 	m_renameDirReq = new KDirRequester(page_6);
// 	
// 	QFrame *m_line7 = new QFrame(page_6);
// 	m_line7->setFrameShape( QFrame::HLine );
// 	m_line7->setFrameShadow( QFrame::Sunken );
// 	m_line7->setFrameShape( QFrame::HLine );
// 	
// 	m_removeCharsCb = new QCheckBox(i18n("Remove illegal charachters from file name"),page_6);
// 	
// 	m_openFmCb = new QCheckBox(i18n("Open directory after finishing"),page_6);
// 	
// 	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
// 	
// 	m_renameBtn = new KPushButton(i18n("Rename"),page_6);
// 	
// 	pageLayout->addWidget( m_renameLbl );
// 	pageLayout->addWidget( m_renameDirReq );
// 	pageLayout->addWidget( m_line7 );
// 	pageLayout->addWidget( m_removeCharsCb );
// 	pageLayout->addWidget( m_openFmCb );
// 	pageLayout->addItem( spacer );
// 	pageLayout->addWidget( m_renameBtn );
// }

void IndexTab::connectSlots()
{
	connect( m_htmlDocsRb, SIGNAL(clicked()), this, SLOT(slotHtmlTypes()) );
	connect( m_otherDocsRb, SIGNAL(clicked()), this, SLOT(slotOtherTypes()) );
	
	connect( m_useDetectEncCb, SIGNAL(clicked()), this, SLOT(slotUseDetection()) );
	connect( m_indexContainCb, SIGNAL(clicked()), this, SLOT(slotIndexWordClicked()) );
	
	connect( m_dotableCb, SIGNAL(clicked()), this, SLOT(slotWriteTableClicked()) );
	connect( m_brDelimeterRb, SIGNAL(clicked()), this, SLOT(slotDelimBrClicked()) );
	connect( m_pDelimeterRb, SIGNAL(clicked()), this, SLOT(slotDelimPClicked()) );
	
	connect( m_indexBtn, SIGNAL(clicked()), this, SLOT(indexDir()) );
}

void IndexTab::indexDir()
{
	if (directoryName().isEmpty()) 
	{
		KMessageBox::information( this, i18n(
		 "<b>You have not chosen the directory to index!</b><br>"
		 "Please choose some path to index on the \"Directory\" tab. "
		), i18n("Directory variable is empty"), "WPI Dir Warning" );
		return;
	}
	
	if (resultFile().isEmpty()) 
	{
		KMessageBox::information( this, i18n(
		 "<b>Result file is not saved (chosen)!</b><br>"
		 "Please choose name for the result file on the \"Result Page\" tab. "
		), i18n("Result File variable is empty"), "WPI ResFile Warning" );
		return;
	}
	
	startIndexing( directoryName() );
	
	savePage();
	
	if (showResult())
		slotOpenPage();
		
	if (closeApp())
		close();
}

void IndexTab::startIndexing(const QString &dirName)
{
	QDir dir(dirName);
	
	// dir entry ini
	DirEntry m_diren(dirName);
	
	QStringList::Iterator it;
	QStringList files = dir.entryList(fileTypes(), QDir::Files);
	it = files.begin();
	
	// starting indexing files
	// and filling the m_diren
	while (it != files.end()) 
	{
		QString m_filename = dirName + "/" + *it;
		QString m_title = getTitle(m_filename);
		
		if (indexByWord())
		{
			if ( m_title.contains(QRegExp( indexWord() )) )
			{
				m_diren.appendEntry( FileEntry(m_filename, m_title) );
			}
		}
		else
		{
			m_diren.appendEntry( FileEntry(m_filename, m_title) );
		}
		
		++it;
	}
	
	// adding new dir entry
	m_dirList.append( m_diren );
	
	
	// recurSive
	if (recurSive())
	{
		QStringList dirs = dir.entryList(QDir::Dirs);
		it = dirs.begin();
	
		QString tmp_empty;
		while (it != dirs.end()) 
		{
			if (*it != "." && *it != "..") 
			{
				tmp_empty = *it;
				
				if (dntEnterFilesDir())
				{
					if (!tmp_empty.contains("_files",false)) startIndexing(dirName + "/" + *it);
					else return;
				}
				else 
				{
					startIndexing(dirName + "/" + *it);
				}
		
			}
			++it;
		}
	}
}

void IndexTab::savePage()
{
	QFile index_file( resultFile() );
	if (!index_file.open(IO_WriteOnly))
	{
		KMessageBox::information( this, i18n(
		 "<b>An error occured while creating Result file!</b><br>"
		 "Please check permissions of the directory you want to create file in."
		), i18n("Error"), "WPI Warning" );
		return;
	}
	
	// i'm using an utf-8 because indexed pages may contain titles 
	// in different languages, e.g russian, english, deutsch, italiano,
	// polish, and etc...
	QTextCodec *codec = QTextCodec::codecForName("UTF-8");
	QTextStream stream(&index_file);
	stream.setCodec( codec );
	
	// and now we're start writing page
	
	// header
	stream << htmlHeader << endl;
	// stylesheet
	if ( (m_useStylesheetCb->isChecked()) && (!styleSheet().isEmpty()) )
	{
		QFile stf( styleSheet() );
		if ( stf.open( IO_ReadOnly ) ) 
		{
			QTextStream streamt( &stf );
			QString text = streamt.read();
			stream << "<style type=\"text/css\">\n" << text << "</style>\n\n";
		}
		stf.close();
	}
	else 
	if ( (m_useStylesheetCb->isChecked()) && (styleSheet().isEmpty()) )
	{
		stream << previewStyle << endl;
	}
	
	stream << "</head>\n<body>" << endl;
	// end header
	
	if (useTable()) 
		writeTable(stream);
	else
		writeSimpleLayout(stream);
		
	stream << htmlEnd << endl;
		
	index_file.close();
	
	
	
}

void IndexTab::writeTable(QTextStream &m_stream)
{
	// use table layout but don't write folders names
	if ( dntWriteFolderNames() )
	{
		m_stream << tableBig << endl;
		
		for (int i = 0; i < (int)m_dirList.size(); ++i)
		{
			if (sortingDirs()) m_dirList[i].sortList();
			
			for (int e = 0; e < m_dirList[i].getListSize() ; e++)
			{
				
				m_stream << tdBig << endl;
				m_stream << m_dirList[i].getReadyLink(e) << endl;
				m_stream << tdEnd << endl;
				
			}
		}
		
		m_stream << tableEnd << endl;
	
	}
	else
	{
		for (int i = 0; i < (int)m_dirList.size(); ++i)
		{
			if (!m_dirList[i].directoryIsEmpty())
			{
				m_stream << "<b>Path: </b>" + m_dirList[i].directoryName() << endl;
				m_stream << tableBig << endl;
			}
			
			if (sortingDirs()) m_dirList[i].sortList();
			
			for (int e = 0; e < m_dirList[i].getListSize() ; e++)
			{
				m_stream << tdBig << endl;
				m_stream << m_dirList[i].getReadyLink(e) << endl;
				m_stream << tdEnd << endl;
			}
			
			if (!m_dirList[i].directoryIsEmpty())
			{
				m_stream << tableEnd << endl;
				m_stream << "<br>" << endl;
			}
		}
	}

}

void IndexTab::writeSimpleLayout(QTextStream &m_stream)
{
	// dnt write folder names
	if ( dntWriteFolderNames() )
	{
		for (int i = 0; i < (int)m_dirList.size(); ++i)
		{
			if (sortingDirs()) m_dirList[i].sortList();
			
			for (int e = 0; e < m_dirList[i].getListSize() ; e++)
			{
				m_stream << m_dirList[i].getReadyLink(e) << endl;
				m_stream << deliMeter() << endl;
			}
		}
	}
	else
	{
		for (int i = 0; i < (int)m_dirList.size(); ++i)
		{
			if (!m_dirList[i].directoryIsEmpty())
			{
				m_stream << m_dirList[i].directoryName() << endl;
				m_stream << deliMeter() << endl;
			}
			
			if (sortingDirs()) m_dirList[i].sortList();
		
			for (int e = 0; e < m_dirList[i].getListSize() ; e++)
			{
				m_stream << m_dirList[i].getReadyLink(e) << endl;
				m_stream << deliMeter() << endl;
			}
		}
	}
}

QString IndexTab::getTitle(const QString& filename) 
{
	QString title;
	QString encoding = getEncoding(filename);
	QFile f(filename);
	if (f.open(IO_ReadOnly)) 
	{
		QTextCodec *codec = QTextCodec::codecForName(encoding);
		QTextStream stream(&f);
		stream.setCodec(codec);
		QString text;
		text = stream.read();

		int start = text.find( titleFrom() ,0,false) + 7;
		int end = text.find( titleTo() ,0,false);

		if ((start < 8)||(end < 0)) 
		{
			title = i18n("[Untitled]");
		} 
		else 
		{
			title = text.mid(start,end - start);
		}
		
// 		another check
		if (title.isEmpty()) title = "[Empty tag]";
	}
	f.close();

	// later modify
	if (m_removeILLCharsCb->isChecked())
	{
		title.remove(QRegExp("&[^;]*;"));
		title.remove("\n");
		title.remove("\t");
	}
	
	if (m_removeIncTagsCb->isChecked())
		title.remove(QRegExp("<[^>]*>"));
	
	return title;
}

bool IndexTab::checkDepends()
{
	if ( system( "enca --version >/dev/null" ) )
	{
		KMessageBox::information( this, i18n(
		 "<b>Enca Library was not found in your system!</b><br>"
		 "Please install it if you want to use detection of "
		 "encoding and so you should index non-english pages. "
		), i18n("Enca Library is missing"), "WPI Enca Warning" );
		return FALSE;
	}
	else return TRUE;
}

void IndexTab::slotHtmlTypes()
{
	if (m_htmlDocsRb->isChecked())
	{
		m_otherDocsEdit->setEnabled(FALSE);
		m_otherDocsRb->setChecked(FALSE);
		
	}
}

void IndexTab::slotOtherTypes()
{
	if (m_otherDocsRb->isChecked())
	{
		m_otherDocsEdit->setEnabled(TRUE);
		m_htmlDocsRb->setChecked(FALSE);
	
	}
}

void IndexTab::slotUseDetection()
{
	if (!checkDepends())
		m_useDetectEncCb->setChecked(FALSE);
}

void IndexTab::slotDelimBrClicked()
{
	if (m_brDelimeterRb->isChecked())
		m_pDelimeterRb->setChecked(FALSE);
}

void IndexTab::slotDelimPClicked()
{
	if (m_pDelimeterRb->isChecked())
		m_brDelimeterRb->setChecked(FALSE);
}

void IndexTab::slotWriteTableClicked()
{
	if (m_dotableCb->isChecked())
	{
		m_brDelimeterRb->setEnabled(FALSE);
		m_pDelimeterRb->setEnabled(FALSE);
	}
	else
	{
		m_brDelimeterRb->setEnabled(TRUE);
		m_pDelimeterRb->setEnabled(TRUE);
	}
}

void IndexTab::slotIndexWordClicked()
{
	if (m_indexContainCb->isChecked())
		m_indexContainEdit->setEnabled(TRUE);
	else
		m_indexContainEdit->setEnabled(FALSE);
}

void IndexTab::slotOpenPage()
{
	QProcess *Br;
	Br = new QProcess(this);
	Br->addArgument( browserApp() );
	Br->addArgument( resultFile() );
	Br->start();
	delete Br;
}

// V_A_L_U_E_S

QString IndexTab::directoryName() const
{
	QString tmp_dir = m_dirPathReq->dirEdit->text();
	if (!tmp_dir.isEmpty())
		return tmp_dir;
	else 
		return QString::null;
}

QString IndexTab::resultFile() const
{
	QString tmp_name = m_resultPageReq->url();
	if (!tmp_name.isEmpty())
		return tmp_name;
	else
		return QString::null;
}

QString IndexTab::fileTypes() const
{
	if (m_htmlDocsRb->isChecked())
	{
		return QString("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
	} else
	if (m_otherDocsRb->isChecked())
	{
		QString tmp_types = m_otherDocsEdit->text();
		tmp_types.replace("\n"," ");
		return tmp_types;
	} 
	else
	return QString("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
}

QString IndexTab::homeDir() const
{
	QString home_path = QDir::homeDirPath();
	
	return home_path;
}

QString IndexTab::styleSheet() const
{
	QString style = m_stylesheetReq->url();
	
	if (!style.isEmpty())
		return style;
	else
		return QString::null;
}

QString IndexTab::deliMeter() const
{
	if (m_brDelimeterRb->isChecked())
		return QString("<br>");
	else 
	if (m_pDelimeterRb->isChecked())
		return QString("<p/>");
	else 
		return QString("<br>");
}

bool IndexTab::useTable() const
{
	if (m_dotableCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

bool IndexTab::recurSive() const
{
	if (m_recursiveCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

bool IndexTab::dntEnterFilesDir() const
{
	if (m_dntEnterCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

QString IndexTab::titleFrom() const
{
	QString titleF = m_titleFEdit->text();
	if (!titleF.isEmpty())
		return titleF;
	else
		return QString("<TITLE>");
}

QString IndexTab::titleTo() const
{
	QString titleT = m_titleTEdit->text();
	if (!titleT.isEmpty())
		return titleT;
	else
		return QString("</TITLE>");
}

bool IndexTab::indexByWord() const
{
	if (m_indexContainCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

bool IndexTab::dntWriteFolderNames() const
{
	if ( !m_writeNamesCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

QString IndexTab::indexWord() const
{
	QString index_word = m_indexContainEdit->text();
	
	if (!index_word.isEmpty())
		return index_word;
	else
		return QString::null;
}

bool IndexTab::sortingDirs() const
{
	if (m_sortDirsCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

bool IndexTab::showResult() const
{
	if (m_showPageCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

bool IndexTab::closeApp() const
{
	if (m_closeAppCb->isChecked())
		return TRUE;
	else
		return FALSE;
}

QString IndexTab::browserApp() const
{
	config->setGroup("Programs");
	QString browser = config->readEntry( "Browser","konqueror");
	return browser;
}

//

void IndexTab::loadSettings()
{
	config->setGroup("IndexTab");
	m_writeNamesCb->setChecked( config->readBoolEntry("WriteFolderNames", TRUE));
	m_dntEnterCb->setChecked( config->readBoolEntry("DoNotEnterFilDir", TRUE));
	m_recursiveCb->setChecked( config->readBoolEntry("Recursive", TRUE));
	m_sortDirsCb->setChecked( config->readBoolEntry("SortFilesInDir", TRUE));
	
	if ( (config->readEntry("Docs","html")) == "html")
	{
		m_htmlDocsRb->setChecked(TRUE);
		slotHtmlTypes();
	}
	else
	{
		m_otherDocsRb->setChecked(TRUE);
		slotOtherTypes();
	}
	
	m_removeILLCharsCb->setChecked( config->readBoolEntry("RemoveIllChars", TRUE));
	m_removeIncTagsCb->setChecked( config->readBoolEntry("RemoveIncludedTags", TRUE));
	m_useDetectEncCb->setChecked( config->readBoolEntry("UseEnca", TRUE));
	
	if (config->readBoolEntry("WriteTable", TRUE))
	{
		m_dotableCb->setChecked(TRUE);
		slotWriteTableClicked();
	}
	
	if ( (config->readEntry("Delimeter","br") == "br"))
	{
		m_brDelimeterRb->setChecked(TRUE);
		slotDelimBrClicked();
	} 
	else
	{
		m_pDelimeterRb->setChecked(TRUE);
		slotDelimPClicked();
	}
	
	config->readBoolEntry("CloseApp",FALSE);
	
	config->setGroup("Programs");
	config->readBoolEntry("ExecBrowser",FALSE);
	
	
}

void IndexTab::saveSettings()
{
	config->setGroup("IndexTab");
	config->writeEntry("WriteFolderNames",m_writeNamesCb->isChecked());
	config->writeEntry("DoNotEnterFilDir",m_dntEnterCb->isChecked());
	config->writeEntry("Recursive",m_recursiveCb->isChecked());
	config->writeEntry("SortFilesInDir",m_sortDirsCb->isChecked());
	
	if (m_htmlDocsRb->isChecked())
		config->writeEntry("Docs","html");
	else
		config->writeEntry("Docs","other");
	
	config->writeEntry("RemoveIllChars",m_removeILLCharsCb->isChecked());
	config->writeEntry("RemoveIncludedTags",m_removeIncTagsCb->isChecked());
	config->writeEntry("UseEnca",m_useDetectEncCb->isChecked());
	
	config->writeEntry("WriteTable",m_dotableCb->isChecked());
	
	if (m_brDelimeterRb->isChecked())
	{
		config->writeEntry("Delimeter","br");
	}
	else
	{
		config->writeEntry("Delimeter","p");
	}
	
	config->writeEntry("CloseApp",m_closeAppCb->isChecked());
	
	config->setGroup("Programs");
	config->writeEntry("ExecBrowser",m_showPageCb->isChecked());
	
	config->sync();
	
}































