/***************************************************************************
 *   Copyright (C) 2005 by Oleg Martemjanov                                *
 *   demogorgorn@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "renametab.h"
#include "variouswidgets.h"
#include "utils.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qprocess.h>
#include <qtextcodec.h>
#include <qtextstream.h>
#include <qregexp.h>

#include <kpushbutton.h>
#include <klocale.h>
#include <kiconloader.h>
#include <ktabwidget.h>
#include <ktextedit.h>
#include <kmessagebox.h>
#include <klineedit.h>

 
RenameTab::RenameTab(QWidget *p, const char *name) :
	QWidget(p,name), parent( p )
{
	config = KGlobal::config();

	QVBoxLayout *inLay = new QVBoxLayout(this);
	inLay->setAlignment(Qt::AlignTop);
	
	setupPage1();
	setupPage2();
	setupPage3();
	setupPage4();
	connectSlots();
	
	loadSettings();
	
	KTabWidget *tab = new KTabWidget(this);
	tab->addTab(page_1,SmallIcon("folder"),i18n("Directory"));
	tab->addTab(page_2,SmallIcon("exec"),i18n("File Types"));
	tab->addTab(page_3,SmallIcon("html"),i18n("Filename"));
	tab->addTab(page_4,SmallIcon("enhanced_browsing"),i18n("Rename"));
	
	inLay->add(tab);
}

void RenameTab::connectSlots()
{
	connect( m_htmlDocsRb, SIGNAL(clicked()), this, SLOT(slotHtmlTypes()) );
	connect( m_otherDocsRb, SIGNAL(clicked()), this, SLOT(slotOtherTypes()) );
	
	connect( m_renameBtn, SIGNAL(clicked()), this, SLOT(renameDir()) );
	
}

void RenameTab::setupPage1()
{
	page_1 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_1, 11, 1 );
	
	QLabel *m_label1 = new QLabel(i18n("Directory to rename files in:"),page_1);
	
	m_renameDirReq = new KDirRequester(page_1);
	
	m_recursiveCb = new QCheckBox(i18n("Recursive"),page_1);
	m_recursiveCb->setChecked(TRUE);
	
	m_dntEnterFilesDirCb = new QCheckBox(i18n("Do not enter \"_files\" directory"),page_1);
	m_dntEnterFilesDirCb->setChecked(TRUE);
	
	QFrame *m_line1 = new QFrame(page_1);
	m_line1->setFrameShape( QFrame::HLine );
	m_line1->setFrameShadow( QFrame::Sunken );
	m_line1->setFrameShape( QFrame::HLine );
	
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_label1 );
	pageLayout->addWidget( m_renameDirReq );
	pageLayout->addWidget( m_recursiveCb );
	pageLayout->addWidget( m_dntEnterFilesDirCb );
	pageLayout->addWidget( m_line1 );
	pageLayout->addItem( spacer );
    
	//QToolTip::add( m_htmlDocsRb, i18n( "Program will index common html files." ) );
	//QToolTip::add( m_otherDocsRb, i18n( "You can define file types to index yourself." ) );
}

void RenameTab::setupPage2()
{
	// File types tab
	page_2 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_2, 11, 1 );

	m_htmlDocsRb = new QRadioButton(i18n("HTML documents"),page_2);
	m_htmlDocsRb->setChecked(TRUE);
	
	m_otherDocsRb = new QRadioButton(i18n("Other types"),page_2);
	
	m_otherDocsEdit = new KTextEdit(page_2);
	m_otherDocsEdit->setText("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
	m_otherDocsEdit->setMaximumHeight(200);
	m_otherDocsEdit->setEnabled(FALSE);
	
	
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_htmlDocsRb );
	pageLayout->addWidget( m_otherDocsRb );
	pageLayout->addWidget( m_otherDocsEdit );
	pageLayout->addItem( spacer );
    
	QToolTip::add( m_htmlDocsRb, i18n( "Program will index common html files." ) );
	QToolTip::add( m_otherDocsRb, i18n( "You can define file types to index yourself." ) );
}

void RenameTab::setupPage3()
{
	// HTML document tab
	page_3 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_3, 11, 1 );

	m_removeILLCharsCb = new QCheckBox(i18n("Remove illegal charachters from filename"),page_3);
	m_removeILLCharsCb->setChecked(TRUE);
	
	m_removeIncTagsCb = new QCheckBox(i18n("Remove included tags"),page_3);
	m_removeIncTagsCb->setChecked(TRUE);
	
	m_replaceSpacesCb = new QCheckBox(i18n("Replace \" \" (spaces) with \"_\""),page_3);
	m_replaceSpacesCb->setChecked(TRUE);
	
	QFrame *m_line2 = new QFrame(page_3);
	m_line2->setFrameShape( QFrame::HLine );
	m_line2->setFrameShadow( QFrame::Sunken );
	m_line2->setFrameShape( QFrame::HLine );
	

	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_removeILLCharsCb );
	pageLayout->addWidget( m_removeIncTagsCb );
	pageLayout->addWidget( m_replaceSpacesCb );
	pageLayout->addWidget( m_line2 );
	pageLayout->addItem( spacer );

//     
	QToolTip::add( m_removeILLCharsCb, i18n( "Remove unwanted charachters from title." ) );
	QToolTip::add( m_removeIncTagsCb, i18n( "Remove tags from the title." ) );
}

void RenameTab::setupPage4()
{
	// rename tab
	page_4 = new QWidget( parent );
	
	QVBoxLayout *pageLayout = new QVBoxLayout( page_4, 11, 1 );

	m_showDirCb = new QCheckBox(i18n("Open directory after finishing"),page_4);
	
	m_closeAppCb = new QCheckBox(i18n("Close application after renaming"),page_4);
	
	m_renameBtn = new KPushButton(i18n("Rename"),page_4);
	
	QFrame *m_line3 = new QFrame(page_4);
	m_line3->setFrameShape( QFrame::HLine );
	m_line3->setFrameShadow( QFrame::Sunken );
	m_line3->setFrameShape( QFrame::HLine );
	

	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
   
	pageLayout->addWidget( m_showDirCb );
	pageLayout->addWidget( m_closeAppCb );
	pageLayout->addWidget( m_line3 );
	pageLayout->addItem( spacer );
	pageLayout->addWidget( m_renameBtn );

//     

}

// MAIN

void RenameTab::renameDir()
{
	if (directoryName().isEmpty()) 
	{
		KMessageBox::information( this, i18n(
		 "<b>You have not chosen the directory to rename files in!</b><br>"
		 "Please choose some path to rename on the \"Directory\" tab. "
		), i18n("Directory variable is empty"), "WPI Ren Dir Warning" );
		return;
	}
	
	startRenaming( directoryName() );
	
	if ( showDir() )
		slotOpenDir();
	
}

QString RenameTab::getTitle(const QString& filename) 
{
	QString title;
	QString encoding = getEncoding(filename);
	QFile f(filename);
	if (f.open(IO_ReadOnly)) 
	{
		QTextCodec *codec = QTextCodec::codecForName(encoding);
		QTextStream stream(&f);
		stream.setCodec(codec);
		QString text;
		text = stream.read();

		int start = text.find( "<TITLE>" ,0,false) + 7;
		int end = text.find( "</TITLE>" ,0,false);

		if ((start < 8)||(end < 0)) 
		{
			title = i18n("[Untitled]");
		} 
		else 
		{
			title = text.mid(start,end - start);
		}
		
// 		another check
		if (title.isEmpty()) title = "[Empty tag]";
	}
	f.close();

	// later modify
	if ( removeIncludedTags() )
	{
		title.remove(QRegExp("<[^>]*>"));
	}
	
	if ( removeIllegalChars() )
	{
		title.remove("'");
		title.remove(":");
		title.remove("/");
		title.remove("\\");
		title.remove("|");
		title.remove("?");
		title.remove("!");
		title.remove(",");
		title.remove("$");
		title.remove("#");
		title.remove("%");
		title.remove("~");
		title.remove("@");
		title.remove("^");
		title.remove("*");
		title.remove("\"");
		title.remove(">");
		title.remove("<");
		title.remove(QRegExp("&[^;]*;"));
		title.remove("&");
		title.remove("\n");
		title.remove("\t");
	}
	
	if ( replaceSpaces() )
	{
		title.replace( " ", "_" );
	}
	
	return title;
}

// Rename function was taken from my previous program KHTMLOrganizer and
// is un-modified, there are some changes but they are unimportant.
// I'll reorganize the renaming functions in the next release!
void RenameTab::startRenaming(const QString &dirName)
{
	QDir dir(dirName);
	QStringList::Iterator it;

	QStringList files = dir.entryList( fileTypes(), QDir::Files );

	it = files.begin();

	QString futurename;
	int i;
	i=1;
	while (it != files.end()) {
		futurename = getTitle(dirName + "/" + *it);
		
		QFile fn(dirName + "/" + futurename + ".html");
		if (fn.exists()) {
			dir.rename(dirName + "/" + *it, futurename + "_" + QString::number(i) + ".html");
			i++;
		} else
		dir.rename(dirName + "/" + *it, futurename + ".html" );
		
		++it;
	}

	QStringList dirs = dir.entryList(QDir::Dirs);
	it = dirs.begin();

	QString tmp_empty;
	while (it != dirs.end()) {
		if (*it != "." && *it != "..") {
			tmp_empty = *it;

			if ( dontEnterFilesDir() )
			{
				if (!tmp_empty.contains("_files",false)) startRenaming(dirName + "/" + *it);
				else return;
			}
			else 
			{
				startRenaming(dirName + "/" + *it);
			}
			
		}
	++it;
	}
}

// SLOTS
void RenameTab::slotHtmlTypes()
{
	if (m_htmlDocsRb->isChecked())
	{
		m_otherDocsEdit->setEnabled(FALSE);
		m_otherDocsRb->setChecked(FALSE);
		
	}
}

void RenameTab::slotOtherTypes()
{
	if (m_otherDocsRb->isChecked())
	{
		m_otherDocsEdit->setEnabled(TRUE);
		m_htmlDocsRb->setChecked(FALSE);
	
	}
}

void RenameTab::slotOpenDir()
{
	QProcess *Br;
	Br = new QProcess(this);
	Br->addArgument( fmApp() );
	Br->addArgument( directoryName() );
	Br->start();
	delete Br;
}


// VALUES
bool RenameTab::recursive() const
{
	if ( m_recursiveCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}


bool RenameTab::dontEnterFilesDir() const
{
	if ( m_dntEnterFilesDirCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

QString RenameTab::fileTypes() const
{
	if (m_htmlDocsRb->isChecked())
	{
		return QString("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
	} else
	if (m_otherDocsRb->isChecked())
	{
		QString tmp_types = m_otherDocsEdit->text();
		tmp_types.replace("\n"," ");
		return tmp_types;
	} 
	else
	return QString("*.html *.htm *.asp *.jhtml *.jsp *.shtml *.shtm *.shtml.html");
}

QString RenameTab::directoryName() const
{
	QString tmp_dir = m_renameDirReq->dirEdit->text();
	if (!tmp_dir.isEmpty())
		return tmp_dir;
	else 
		return QString::null;
}

bool RenameTab::removeIllegalChars() const
{
	if ( m_removeILLCharsCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

bool RenameTab::removeIncludedTags() const
{
	if ( m_removeIncTagsCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

bool RenameTab::replaceSpaces() const
{
	if ( m_replaceSpacesCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

QString RenameTab::fmApp() const
{
	config->setGroup("Programs");
	QString fm = config->readEntry( "FM","konqueror");
	return fm;
}

bool RenameTab::showDir() const
{
	if ( m_showDirCb->isChecked() )
		return TRUE;
	else
		return FALSE;
}

// // // // // // // // // // // // // // // 

void RenameTab::loadSettings()
{
	config->setGroup("RenameTab");
	m_recursiveCb->setChecked( config->readBoolEntry("Recursive", TRUE));
	m_dntEnterFilesDirCb->setChecked( config->readBoolEntry("DoNotEnterFilesDir", TRUE));
	
	if ( (config->readEntry("Docs","html")) == "html")
	{
		m_htmlDocsRb->setChecked(TRUE);
		slotHtmlTypes();
	}
	else
	{
		m_otherDocsRb->setChecked(TRUE);
		slotOtherTypes();
	}
	
	m_removeILLCharsCb->setChecked( config->readBoolEntry("RemoveIllegalChars", TRUE));
	m_removeIncTagsCb->setChecked( config->readBoolEntry("RemoveIncludedTags", TRUE));
	m_replaceSpacesCb->setChecked( config->readBoolEntry("ReplaceSpaces", TRUE));

}

void RenameTab::saveSettings()
{
	config->setGroup("RenameTab");
	config->writeEntry("Recursive",m_recursiveCb->isChecked());
	config->writeEntry("DoNotEnterFilesDir",m_dntEnterFilesDirCb->isChecked());
	
	if (m_htmlDocsRb->isChecked())
		config->writeEntry("Docs","html");
	else
		config->writeEntry("Docs","other");
	
	config->writeEntry("RemoveIllegalChars",m_removeILLCharsCb->isChecked());
	config->writeEntry("RemoveIncludedTags",m_removeIncTagsCb->isChecked());
	config->writeEntry("ReplaceSpaces",m_replaceSpacesCb->isChecked());
	
	
	
	config->sync();
}






















