/*
 * Copyright (C) 2004, 2005, 2006, 2007 Moritz Orbach <zufall@apfelboymchen.homeunix.net>
 *
 * Zufall is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zufall is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * $Id: cfg.c,v 1.6 2007/12/19 10:39:46 mori Exp $
 *
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <getopt.h>

#include "config.h"

#include "list.h"
#include "status.h"
#include "cfg.h"
#include "memman.h"

#ifdef FENCE
#include "efence.h"
#endif

t_config cfg;

static struct option long_options[] =
{
	{ "help",       no_argument,       NULL, 'h' },
	{ "version",    no_argument,       NULL, 'v' },
	{ "delay",      required_argument, NULL, 'd' },
	{ "maxscale",   required_argument, NULL, 's' },
	{ "minborder",  required_argument, NULL, 'b' },
	{ "minsize",    required_argument, NULL, 'm' },
	{ "command",    required_argument, NULL, 'c' },
	{ "no-set",     no_argument,       NULL, 'n' },
	{ "list",       required_argument, NULL, 'l' },
	{ "passes",     required_argument, NULL, 'p' },
	{ "sequential", no_argument,       NULL, 'S' },
	{ NULL, 0, NULL, 0 }
};


void set_defaults()
{
	/* delay 15 min */
	cfg.delay = 900;
	cfg.maxscale = 150;
	cfg.minborder = 100;
	cfg.minsize = 0;
	cfg.setbg = 1;
	cfg.command = NULL;
	cfg.passes = -1;
	cfg.sequential = 0;
}

/* ------ */

/* unsused */
#ifdef I_DECICDED_THIS_IS_WORTHWILE
void cfg_file()
{
	FILE *fp;
	const char *rcfile = "/home/mori/.zufallrc\0";
	char line[82];
	int length = 0;

	char option[80];
	char argument[80];

	fp = fopen(rcfile, "r");
	if (fp == NULL)
		return;

	while (fgets (line + length, 82 - length, fp))
		printf("line: %s", line);
	sscanf(line, "%s = %s\n", option, argument);
	printf("option: %s\nargument: %s\n", option, argument);

	fclose(fp);
	return;
}
#endif
/* ------ */

void version(void)
{
	printf("%s\n", PACKAGE_STRING);
	printf("Written by %s\n", AUTHORS);
	printf("Send bugreports to <%s>\n", PACKAGE_BUGREPORT);
	exit(EXIT_SUCCESS);
}
 
void usage(int status)
{
	printf("Usage: %s /path/to/imagedir-or-file [/path/to/imagedir-or-file]\n", program_name);
	/* -pedantic: warning: string length `585' is greater than the length `509' ISO C89 compilers are required to support */
	printf(
			"  -d, --delay=DELAY          number of seconds an image is shown (default 900)\n"
			"  -s  --maxscale=PERCENT     maximum PERCENT the image will be scaled\n"
			"                             (default 150)\n"
			"  -b  --minborder=PIXEL      ignore --maxscale if the remaining border would be\n"
			"                             smaller than --minborder (default 100)\n"
	      );
	printf(
			"  -m  --minsize=KILOBYTE     only show images bigger than KILOBYTE.\n"
			"                             A size of 0 shows all images (default 0)\n"
			"\n"
			"  -l  --list=FILE            load list of images from file\n"
                        "                             - format is one directory or file per line\n"
                        "                             - can also be a collection from gqview\n"
                        "                             - if FILE is \"-\", the list will be read from\n"
                        "                               stdin\n"
	      );

	printf(
			"  -S  --sequential           don't randomize the imagelist (useful with -l)\n"
			"  -c  --command=COMMAND      execute COMMAND on every picture change\n"
			"                             Within COMMAND\n"
			"                               %%c gets replaced with the background-color\n"
			"                               %%i gets replaced with the image-filename\n"
                        "                               %%w gets replaced with the scaled width\n"
                        "                               %%h gets replaced with the scaled height\n"
	      );
	printf(
                        "                               %%%% gets replaced with %%\n"
			"  -n  --no-set               don't set the background\n"
			"                             (but execute command if given)\n"
			"  -p  --passes=PASSES        exit after given number of images\n"
			"\n"
			"      --help                 display this help and exit\n"
			"      --version              output version information and exit\n"
			"\n"
			"SIGNALS:\n"
			"  HUP                        change background\n"
			"  USR1                       info\n"
			"  TERM                       exit\n"
	      );
	exit(status);
}


/*
 * string to int
 * returns
 *   0 if string is no number
 *   1 success
 */
int s2int(char *str, int *number)
{
	char *endptr;

	errno = 0;	/* To distinguish success/failure after call */
	*number = strtol(str, &endptr, 10);
	if (errno || *endptr != '\0') {
		if (errno) {
			perror(PACKAGE_NAME);
		}
		return 0;
	}
	return 1;
}

/*
 * interpret commandline
 * - malloc's memory for *collections
 */
int cfg_cmdln(int *argc, char **argv, int *diroffset, char ****collections, int *collectionsc)
{
	char c;
	char ***temp = NULL;

	set_defaults();

	*collectionsc = 0;
	program_name = argv[0];

	if (*argc == 1) {
		fprintf(stderr, "%s: please specify a directory or image\n", program_name);
		exit(EXIT_FAILURE);
	}

	while ((c = getopt_long (*argc, argv, "hvSd:s:b:m:nc:l:p:", long_options, NULL)) != -1)
		switch(c)
		{
			case 'h': usage(EXIT_SUCCESS);
				  break;
			case 'v': version();
				  break;

			case 'd': if (s2int(optarg, &cfg.delay)) {
					  if (cfg.delay < 1) {
						  fprintf(stderr, "Delay too small. 1 is minimum.\n");
						  exit(EXIT_FAILURE);
					  }
				  }
				  else {
					  fprintf(stderr, "invalid delay \"%s\"\n", optarg);
					  exit(EXIT_FAILURE);
				  }
				  break;
			case 's': if (s2int(optarg, &cfg.maxscale)) {
					  if (cfg.maxscale < 1) {
						  fprintf(stderr, "size too small. 1 is minimum.\n");
						  exit(EXIT_FAILURE);
					  }
				  } else {
					  fprintf(stderr, "invalid size \"%s\"\n", optarg);
					  exit(EXIT_FAILURE);
				  }
				  break;
			case 'p': if (s2int(optarg, &cfg.passes)) {
					  if (cfg.passes < 1) {
						  fprintf(stderr, "Number of passes too small. 1 is minimum\n");
						  exit(EXIT_FAILURE);
					  }
				  } else {
					  fprintf(stderr, "invalid number of passes \"%s\"\n", optarg);
					  exit(EXIT_FAILURE);
				  }
				  break;
			case 'm': if (s2int(optarg, &cfg.minsize)) {
					  if (cfg.minsize < 0) {
						  fprintf(stderr, "Minimum filesize is too small. 0 is minimum\n");
						  exit(EXIT_FAILURE);
					  } else {
						  /* convert to byte */
						  cfg.minsize*=1024;
					  }
				  } else {
					  fprintf(stderr, "invalid minimum filesize \"%s\"\n", optarg);
					  exit(EXIT_FAILURE);
				  }
				  break;
			case 'b': cfg.minborder = atoi(optarg);
			case 'n': cfg.setbg = 0;
				  break;
			case 'c': cfg.command = optarg;
				  break;
			case 'l': temp = extendmem(temp, (*collectionsc + 1) * sizeof(*temp), sizeof(*temp));
				  /* i know what i'm doing... i think. */
				  temp[*collectionsc] = (char **)optarg;
				  *collectionsc += 1;
				  break;
			case 'S': cfg.sequential = 1;
				  break;
			case '?': /* errormessage comes from getopt_long */
				  exit(EXIT_FAILURE);
		}

	if (cfg.setbg == 0 && cfg.command == NULL)
		/* the user's wish is my order */
		fprintf(stderr, "oh, well. Have fun ;)\n");

	*diroffset = optind;
	*collections = temp;
	return 1;
}

