/*
 * Copyright (C) 2004, 2005, 2006, 2007 Moritz Orbach <zufall@apfelboymchen.homeunix.net>
 *
 * Zufall is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zufall is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Parts of this file are ripped from Esetroot.c written by Nat Friedman 
 * <ndf@mit.edu> with modifications by Gerald Britton <gbritton@mit.edu> and 
 * Michael Jennings <mej@eterm.org>
 *
 * $Id: changeimage.c,v 1.13 2007/12/27 17:34:24 mori Exp $
 *
 */

/*
 * this code isn't so nice, but it gets better with every release
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xos.h>
#include <Imlib2.h>

#include "config.h"

#include "cfg.h"
#include "command.h"

#ifdef FENCE
#include "efence.h"
#endif

Pixmap CurrentPixmap = None;
Atom prop_root = None;
Atom prop_esetroot = None;


Pixmap get_pixmap(Display *Xdisplay, Window Xroot, Atom atom)
{
	Atom type = None;
	int format;
	unsigned long length, after;
	unsigned char *data_root;
	int status;

	status = XGetWindowProperty(Xdisplay, Xroot, atom, 0L, 1L, False, AnyPropertyType, &type, &format, &length, &after, &data_root);
	if (status == Success && type != XA_PIXMAP) {
		return None;
	}
	/* pointer to pointer to pixmap */
	return *((Pixmap *)data_root);
}

/*
 *
 * returns true if the root is set permanent
 * prop_setroot and data_esetroot get set.
 *
 * hm.
 * http://mail.gnome.org/archives/wm-spec-list/2002-January/msg00006.html
 *
 * The server should be grabbed.
 */
int root_is_set_permanent(Display *Xdisplay, Window Xroot, Atom *prop_esetroot, unsigned char **data_esetroot)
{
	Atom prop_root = None;
	Atom type = None;
	int format;
	unsigned long length, after;
	unsigned char *data_root;

	/* printf("is root set permanent?\n"); */
	/* First check to see if the properties already exist, and if they are equal */
	prop_root = XInternAtom(Xdisplay, "_XROOTPMAP_ID", True);
	*prop_esetroot = XInternAtom(Xdisplay, "ESETROOT_PMAP_ID", True);

	if (prop_root != None && *prop_esetroot != None) {
		/* both atoms exist */
		XGetWindowProperty(Xdisplay, Xroot, prop_root, 0L, 1L, False, AnyPropertyType, &type, &format, &length, &after, &data_root);
		if (type == XA_PIXMAP) {
			/* there's a pixmap on the root-window */
			XGetWindowProperty(Xdisplay, Xroot, *prop_esetroot, 0L, 1L, False, AnyPropertyType, &type, &format, &length, &after, data_esetroot);
			if (data_root && *data_esetroot && type == XA_PIXMAP && *((Pixmap *) data_root) == *((Pixmap *)*data_esetroot)) {
				/*
				 * both atoms exist &&
				 * _XROOTPMAP_ID points to a pixmap &&
				 * ESETROOT_PMAP_ID points to a pixmap, and points to the *same* pixmap
				 * 
				 * -> the rootpixmap was set to permanent and can be killed
				 */
				/* printf("\tJUP\n"); */
				return 1;
			}
		}
	}
	/* printf("\tnoe\n"); */
	*data_esetroot = NULL;
	return 0;
}

/*
 * frees the pixmap if the _XROOTPMAP_ID was touched by another program
 */
void property_changed(Display *Xdisplay, Window Xroot)
{
	XEvent e;
	char *name;

	while (XCheckMaskEvent(Xdisplay, PropertyChangeMask, &e)) {
		if (CurrentPixmap == None) {
			/* no sense in looking at the changes if there is no pixmap to delete */
			continue;
		}
		/* dispatch events */
		name = XGetAtomName(Xdisplay, e.xproperty.atom);
		/* printf("geaendertes atom: %s (%d)\n", name, e.xproperty.atom); DEBUG */
		if (strcmp(name, "_XROOTPMAP_ID") == 0) {
			/* somebody else set a new background. Free own. */
			/* printf("atom-freeing 0x%6X\n\n", CurrentPixmap); */
			XFreePixmap(Xdisplay, CurrentPixmap);
			CurrentPixmap = None;
		}
		XFree(name);
	}


}

/*
 * returns
 *   - the ESETROOT_PMAP_ID-atom if its clients' resources were killed
 *   - None if there was no previous pixmap to kill
 *     (either none set or set by a still-running client)
 */

/*
 * sets the pixmap so that zufall can exit (ESETROOT_PMAP_ID to _XROOTPMAP_ID)
 *
 * helpful commands for debugging:
 * Xnest -ac -geometry 320x200 :1 &
 * DISPLAY=:1
 * export _Xdebug=1
 * xlsatoms -name _XROOTPMAP_ID -name ESETROOT_PMAP_ID 2>/dev/null
 * xprop -root
 * xprop -root -spy -root
 * xprop -root -remove ESETROOT_PMAP_ID
 *
 * while ./zufall /public/icons/tp.xpm           && test ! -f /tmp/fin;do xprop -root; sleep 0.25 ;done;echo $? > /tmp/fin
 * while ./zufall /public/icons/fvwm.xpm         && test ! -f /tmp/fin;do xprop -root; sleep 0.25 ;done;echo $? > /tmp/fin
 * while ./zufall /public/icons/zoom_img_out.png && test ! -f /tmp/fin;do xprop -root; sleep 0.25 ;done;echo $? > /tmp/fin
 * Esetroot /public/bilder/yosemite\ -\ akw.jpg
 *
 * hm. even Esetroot fails this test.
 *
 * server must be grabbed before calling this function
 *
 * http://www.eterm.org/docs/view.php?doc=ref#trans
 * http://foo-projects.org/pipermail/xfce4-dev/2003-January/001699.html
 * 
 * In summary:
 * _XROOTPMAP_ID is the pixmap to be displayed. ESETROOT_PMAP_ID is the
 * pixmap to be destroyed, if any. A program that quits after setting the
 * backdrop will set ESETROOT_PMAP_ID, otherwise it won't, so both types of
 * program are supported.
 *
 * ESETROOT_PMAP_ID is not deleted
 *
 * server must be grabbed before calling this function
 * TODO: check sparen weil noch gegrabbed?
 */
int set_current_permanent(Display *Xdisplay, Window Xroot)
{
	Atom prop_root, prop_esetroot;

	if (CurrentPixmap == None) {
		/* no image set */
		return 0;
	}
	
	prop_root = XInternAtom(Xdisplay, "_XROOTPMAP_ID", True);
	if (prop_root == None) {
		fprintf(stderr, "%s: Could not get background\n", PACKAGE_NAME);
		return 0;
	}

	/* printf("\e[1;30mmein pixmap        0x%6X\e[0m\n", (unsigned int)CurrentPixmap); */
	if (get_pixmap(Xdisplay, Xroot, prop_root) != CurrentPixmap) {
		/*
		 * zufall's pixmap is not shown anymore! in the meanwhile 
		 * another program set the background
		 * and should have killed the pixmap?
		 * just clean up the own pixmap
		 */
		/* printf("\e[1;35mnicht mehr mein background!\e[0m\n"); */
		XFreePixmap(Xdisplay, CurrentPixmap);
		return 1;
	}

	/* still own pixmap -- set permanent */
	/* printf("\t\tstill mine\n"); */

	/* get / create */
	prop_esetroot = XInternAtom(Xdisplay, "ESETROOT_PMAP_ID", False);
	if (prop_esetroot == None) {
		/* huh? */
		fprintf(stderr, "%s: Could not create property\n", PACKAGE_NAME);
		return 0;
	}

	/* printf("permanent: %d\n", prop_esetroot); */

	/*
	 * Setting ESETROOT_PMAP_ID to _XROOTPMAP_ID tells the next program 
	 * that it may kill the pixmap.
	 * From here on, if retain is set, it could be that zufall is killed by 
	 * another process cleaning up the "old" pixmap
	 */
	XChangeProperty(Xdisplay, Xroot, prop_esetroot, XA_PIXMAP, 32, PropModeReplace, (unsigned char *) &CurrentPixmap, 1);
	/* leave pixmap in X-server after exit. */
	XSetCloseDownMode(Xdisplay, RetainPermanent);

	return 1;
}

/* the server is not grabbed on exit */
void permanent_on_exit(Display *Xdisplay, Window Xroot)
{
	/* printf("permanent_on_exit\n"); */
	XGrabServer(Xdisplay);
	set_current_permanent(Xdisplay, Xroot);
	XUngrabServer(Xdisplay);
	XFlush(Xdisplay); /* neccessary? */
}

/*
 * sets the pixmap for permanent running of zufall, killing the old pixmap for 
 * a temporary client
 * - zufall permanent
 * - pixmap temporary (will not stay in memory if zufall is killed)
 * The server should be grabbed.
 */
void set_pixmap_temporary(Display *Xdisplay, Window Xroot, Pixmap p)
{
	Atom prop_root, prop_esetroot;
	unsigned char *pixmap_esetroot; /* TODO. Pixmap *pixmap_esetroot; */

	prop_root = XInternAtom(Xdisplay, "_XROOTPMAP_ID", True);

	/* only create property if it doesn't already exists */
	if (prop_root == None) {
		prop_root = XInternAtom(Xdisplay, "_XROOTPMAP_ID", False);
		/* if the property is still not there something is wrong */
		if (prop_root == None) {
			fprintf(stderr, "%s: creation of pixmap property failed.\n", PACKAGE_NAME);
			exit(EXIT_FAILURE);
		}
	}

	if (root_is_set_permanent(Xdisplay, Xroot, &prop_esetroot, &pixmap_esetroot)) {
		/*
		 * in the meantime another program set the background and exited
		 * this is the last chance to kill the previous pixmap before 
		 * the reference is overwritten
		 */
		/* printf("delete prop_esetroot! %d\n", prop_esetroot); */
		/* printf("KILLING 0x%6X\n\n", *(Pixmap *)pixmap_esetroot); */
		XKillClient(Xdisplay, *((Pixmap *)pixmap_esetroot));
		/* delete ESETROOT_PMAP_ID, should not be set for a still-running client */
		XDeleteProperty(Xdisplay, Xroot, prop_esetroot);
	}

	/* XDeleteProperty(Xdisplay, Xroot, XInternAtom(dpy, OPACITY, False));  */
	/* printf("\tprop_esetroot jetzt %d\n", XInternAtom(Xdisplay, "ESETROOT_PMAP_ID", True)); */
	/* set prop_root to the new pixmap */
	XChangeProperty(Xdisplay, Xroot, prop_root, XA_PIXMAP, 32, PropModeReplace, (unsigned char *) &p, 1);
	/* printf("%d\tset new: %d (0x%6X)\n", getpid(), prop_root, p); */

}

/* end of hairy X stuff */


/* convert dezimal color-value to hex and append to array */
void append_color(int color_dez, char color[7])
{
	/* 2 hex + \0 */
	char temp[3];

	snprintf(temp, 3, "%.2x\n", color_dez);
	strncat(color, temp, 6);
}

unsigned int rnd(unsigned int max)
{
	return rand() /(int)(((unsigned)RAND_MAX + 1) / max);
}

/* unused */
void randomcolor(int maxval, char color[7])
{
	int rgb;
	unsigned int r;

	color[0] = '#';
	/* ohne weiss strncat nicht, wo der string aufhoert */
	color[1] = '\0';

	for (rgb = 0; rgb < 3; rgb++)
	{
		r = rnd(maxval);

		append_color(r, color);
		printf("color: %d\n", r);
	}
	printf("fertig: %s\n", color);
}

/* alle 4 ecken angucken, durchschnitt der farben bilden */
/* oder: 2 ecken angucken, nur wenn nicht gleich durchschnitt */
/*
 * lieber doch nur 2 ecken. Sonst gibt's komische farben wenn in einer ecke 
 * z.B. ein dummes logo ist. Auch ausreichend.
 */
void lookupcolor(int w, int h, char color[7])
{
	Imlib_Color color_return;
	int rgb[3][3], border, i, temp;

	color[0] = '#';
	color[1] = '\0';

	/* links oben */
	imlib_image_query_pixel(0, 0, &color_return);
	rgb[0][0] = color_return.red;
	rgb[0][1] = color_return.green;
	rgb[0][2] = color_return.blue;

	/* links unten */
	imlib_image_query_pixel(0, h-1, &color_return);
	rgb[1][0] = color_return.red;
	rgb[1][1] = color_return.green;
	rgb[1][2] = color_return.blue;

	/* durchschnitt der ersten beiden ecken */
	for (i = 0; i < 3; i++) {
		temp = 0;
		for (border = 0; border < 2; border++)
			temp += rgb[border][i];

		temp = temp / 2;

		append_color(temp, color);
	}
}

/*
 * builds the new pixmap and sets it
 *
 * A program that wants to set the background should:
 *  - Create a persistant pixmap by calling to the server with:
 *        dpy = XOpenDisplay(display_name);
 *        XSetCloseDownDisplay (dpy, RetainPixmap);
 *        pmap = XCreatePixmap (dpy, DefaultRootWindow (dpy), width, height,
 *                              DefaultDepthOfScreen (DefaultScreenOfDisplay (dpy)));
 *        XCloseDisplay (dpy)
 *
 * - Grab the server
 * - See if ESETROOT_PMAP_ID is set
 * - If so, call XKillClient(xdisplay, *(Pixmap*)data_esetroot)
 * - Change ESETROOT_PMAP_ID and _XROOTPMAP_ID to point to the new pixmap
 * - Set the pixmap as the background for the root window
 * - Ungrab the server
 *
 * http://mail.gnome.org/archives/wm-spec-list/2002-January/msg00006.html
 * http://www.eterm.org/docs/view.php?doc=ref
 *
 *
 * http://foo-projects.org/pipermail/xfce4-dev/2003-January/001703.html
 * > Or do you mean I should look for ESETROOT_PMAP_ID and destroy it when
 * > xfdesktop starts?
 * Yes, I think you should XKillClient the owner of  ESETROOT_PMAP_ID and
 * remove the property. 
 */
int setroot(Display *Xdisplay, Window Xroot, Screen *scr, int screen, char *color,
		int w, int h, int retain)
{
	GC gc;
	XGCValues gcv;
	XColor xcolor;
	int x, y;

	Pixmap NewPixmap = None;
	Pixmap temp_pmap = None;
	Pixmap m         = None;

	NewPixmap      = XCreatePixmap(Xdisplay, Xroot, scr->width, scr->height, DefaultDepth(Xdisplay, screen));
	gcv.foreground = gcv.background = BlackPixel(Xdisplay, screen);

	if (color && XParseColor(Xdisplay, DefaultColormap(Xdisplay, screen), color, &xcolor)
			&& XAllocColor(Xdisplay, DefaultColormap(Xdisplay, screen), &xcolor)) {
		gcv.foreground = gcv.background = xcolor.pixel;
	}
	/* Create a Graphics Context TODO: fehler abfangen */
	gc = XCreateGC(Xdisplay, NewPixmap, (GCForeground | GCBackground), &gcv);
	XFillRectangle(Xdisplay, NewPixmap, gc, 0, 0, scr->width, scr->height);

	x = (scr->width - w) >> 1;
	y = (scr->height - h) >> 1;
	imlib_context_set_drawable(Xroot);

	imlib_render_pixmaps_for_whole_image_at_size(&temp_pmap, &m, w, h);
	if (temp_pmap != None) {
		XSetTile(Xdisplay, gc, temp_pmap);
		/* free pixmap (temp_pmap) and mask (m) used by imlib */
		imlib_free_pixmap_and_mask(temp_pmap);

		XSetTSOrigin(Xdisplay, gc, x, y);
		XSetFillStyle(Xdisplay, gc, FillTiled);
		XFillRectangle(Xdisplay, NewPixmap, gc, x, y, w, h);

		/* comment from control-center (applier.c):
		 * Set the root pixmap, and properties pointing to it. We
		 * do this atomically with XGrabServer to make sure that
		 * we won't leak the pixmap if somebody else it setting
		 * it at the same time. (This assumes that they follow the
		 * same conventions we do)
		 *
		 * http://mail.gnome.org/archives/wm-spec-list/2002-January/msg00014.html:
		 * I've take a look at the Esetroot source code (0.9.1) and it
		 * does not grab/ungrab the server. This cause a program (e.g.,
		 * rxvt) which use parental relativity for pseudo-transparency
		 * and which monitor _XROOTPMAP_ID for background update to fail
		 * to update its background (as it receives the _XROOTPMAP_ID
		 * property before the background change).
		 */
		/* ----- GRAB ------ */
		XGrabServer(Xdisplay);

		set_pixmap_temporary(Xdisplay, Xroot, NewPixmap);

		/*
		 * kill own previous pixmap
		 * this is independant from other programs setting the 
		 * background. zufall's pixmap only gets killed by other 
		 * programs if not running anymore.
		 */
		if (CurrentPixmap != None) {
			/* printf("freeing 0x%6X\n\n", CurrentPixmap); */
			XFreePixmap(Xdisplay, CurrentPixmap);
		}
		CurrentPixmap = NewPixmap;

		/* set the background pixmap of the window to the specified pixmap */
		XSetWindowBackgroundPixmap(Xdisplay, Xroot, NewPixmap);
		/* "clear" the window with the background */
		XClearWindow(Xdisplay, Xroot);

		/* send the request to the server. This is neccessary to update the root-window
		 * True to discard the own changes of the property. Otherwise 
		 * xsleep will have changes pending on a SIGUSR1 later!
		 */
		XSync(Xdisplay, True);
		XUngrabServer(Xdisplay);
		/* ----- BARG ------ */

	}

	XFreeGC(Xdisplay, gc);

	return 1;
}

int changeimage(Display *Xdisplay, Window Xroot, Screen *scr, int screen,
		const char *image_path, int retain)
{
	double maxscale = (double)cfg.maxscale / 100;
	int minborder = cfg.minborder;
	Imlib_Image imlib_image;
	int w, h;
	char color[8];

	double x_ratio, y_ratio;
	int *border, *full;

	/* load image */
	imlib_image = imlib_load_image_immediately(image_path);
	if (imlib_image == NULL) {
		return 0;
	}

	imlib_context_set_image(imlib_image);

	w = imlib_image_get_width();
	h = imlib_image_get_height();

	/* border-color */
	lookupcolor(w, h, color);

	x_ratio = ((double)scr->width) / ((double)w);
	y_ratio = ((double)scr->height) / ((double)h);
	/* nicht zerren, kleinere ratio nehmen sonst laeuft's ueber */
	if (x_ratio > y_ratio) {
		/* hochkant */
		x_ratio = y_ratio;
		/* rand an dem frei bleibt */
		border = &h;
		full = &scr->height;
	} else {
		/* quer */
		/* rand an dem frei bleibt */
		border = &w;
		full = &scr->width;
	}

	/* don't make bigger than maxscale */
	if ((int) (*border * x_ratio) > (int) *border * maxscale)
		x_ratio = maxscale;

	/* ignore maxscale if the resulting border would be smaller minborder */
	if (*full - *border * x_ratio < minborder)
		x_ratio = ((double) *full) / ((double) *border);

	w = (int) (w * x_ratio);
	h = (int) (h * x_ratio);


	if (cfg.setbg) {
		setroot(Xdisplay, Xroot, scr, screen, color, w, h, retain);
	}
	
	if (cfg.command != NULL) {
		command(cfg.command, color, image_path, w, h);
	}
	imlib_free_image();
	return 1;

}
