/*
 * Copyright (C) 2004, 2005, 2007 Moritz Orbach <zufall@apfelboymchen.homeunix.net>
 *
 * Zufall is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zufall is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * $Id: command.c,v 1.3 2007/12/19 10:39:46 mori Exp $
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "memman.h"

/*
 * returns position of c in array s,
 * or -1 if not found
 * or -1 if s is the last character
 * */
unsigned int findformat(const char *s, char c)
{
	int i = 0;

	/* count up to end or c */
	while (s[i] != '\0' && s[i] != c)
		i++;

	/* return i if c is not last char (or \0) */
	if (s[i] != '\0' && s[i+1] != '\0')
		return i;
	else
		return -1;
}

/*
 * cats number to the end of the string
 * size of the string is expanded if needed
 * returns new size of string via size
 */
void itoacat(char **dest, int src, size_t *size)
{
	int new;
	int ret;

	/* return size of new string, without \0 */
	new = snprintf(*dest, 0, "%d", src);
	if (new == 0)
		return;

	/* ... without \0 ...*/
	new++;

	*size += new;
	*dest = extendmem(*dest, *size, new);

	ret = snprintf(*dest + strlen(*dest), *size, "%d", src); 	/* return doesn't include \0 */

	if (ret != new - 1)
		/* this really shoudn't happen */
		fprintf(stderr, "%s: %d != %d, a sad day for mankind...\n", PACKAGE_NAME, ret, new);
}

/*
 * appends src to dest, expanding size if necessary
 * curlen is the current number of byte allocated for the string
 */
void xstrcat (char **pdest, const char *src, size_t *curlen)
{
	int new;

	/* doesn't work directly! */
	char *dest;

	dest = *pdest;

	/* return curlen of new string, without \0 */
	new = strlen(src);
	if (new < 1)
		return;

	/* append space for \0 */
	new++;

	*curlen += new;
	dest = extendmem(dest, *curlen, new);

	strncpy(dest + strlen(dest), src, new);
	/* strncpy: maximum "curlen" chars! */
	*pdest = dest;
}                                            

/*
 * appends count character from src to dest
 * curlen is the current number of byte allocated for the string
 * src must be null-terminated!
 * dest must be null-terminated!
 * dest will be null-terminated
 */
void xstrncat (char **pdest, const char *src, size_t count, size_t *curlen)
{
	int new = 0;
	char *dest;

	if (count < 1)
		return;

	/* doesn't work directly! */
	dest = *pdest;

	/* don't copy more than there is */
	if (count + 1 > strlen(src)) {
		fprintf(stderr, "you encounterd a meaningless, user-unfriendly errormessage\n");
		return;
	}

	if (strlen(dest) + count + 1 > *curlen)  {
		new = (strlen(dest) + count + 1) - *curlen;
		/* string has to be expanded */
		*curlen += new;
		dest = extendmem(dest, *curlen, count);
	}

	/* strlen could segfault otherwise! */
	dest[*curlen - 1] = '\0';
	strncpy(dest + strlen(dest), src, count);
	*pdest = dest;

	/* hammer bin ich doof! */
}                                            

int command(const char *command, const char *color, const char *image, int w, int h)
{
	char *execute;
	int i;
	int ret;
	size_t len = 1;

	execute = getmem(len);
	execute[0] = '\0';

	while ( (i = findformat(command, '%')) >= 0 ) {

		xstrncat(&execute, command, i, &len); /* copy from start to %-1 */

		/* point to character after '%' */
		command = &command[++i]; /* safe, because findformat returns false if % is the last char */

		switch (*command) {
			case 'c': xstrcat(&execute, color, &len);
				  break;
			case 'i': xstrcat(&execute, image, &len);
				  break;
			case 'w': itoacat(&execute, w, &len);
				  break;
			case 'h': itoacat(&execute, h, &len);
				  break;
			case '%': xstrcat(&execute, "%", &len);
				  break;
			default : fprintf(stderr, "%s: error in format-string!\n", PACKAGE_NAME);
		}
		/* safe, findformat checked it */
		command++;
	}
	
	/* append the rest */
	xstrcat(&execute, command, &len);

	/* execute. maybe fork+exec would be better */
	ret = system(execute);
	freemem(execute, len);

	/* shell + c return-code have the opposite meaning */
	return !ret;
}
