/*
 * Copyright (C) 2004, 2005, 2006, 2007 Moritz Orbach <zufall@apfelboymchen.homeunix.net>
 *
 * Zufall is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zufall is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * $Id: recurse.c,v 1.6 2007/12/27 17:34:24 mori Exp $
 *
 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

#ifdef HAVE_CANONICALIZE_FILE_NAME
#include <unistd.h>
#endif

#include "list.h"
#include "recurse.h"
#include "cfg.h"

#ifdef HAVE_CANONICALIZE_FILE_NAME
t_list *visited = NULL;
#endif

#ifdef FENCE
#include "efence.h"
#endif

#define MAXLINELENGTH 1024

/* can I ask imlib2 for a list of supported filetypes? */
char *supported_filetypes[] = {
	".jpg",
	".jpeg",
	".png",
	".gif",
	".lbm",
	".pnm",
	".bmp",
	".tga",
	".targa",
	".tif",
	".tiff",
	".xpm",
	NULL
};

#ifdef DEBUG
void list_list(t_list *list)
{
	int i;

	for (i = 1; i <= list_count(list); i++)
		printf("%.3d: %s\n", i, (char *)list_get(list, i));

}
#endif

/*
 * returns true if the filetype is supported
 */
int supported(const char *file)
{
	char **s;
	char *ext;

	s = supported_filetypes;

	ext = strrchr(file, '.');
	if (!ext)
		return 0;

	while (*s != NULL) {
		if (!strcasecmp(ext, *s))
			return 1;	/* supported */

		*s++;
	}
	/* nothing found */
	return 0;
}

/*
 * returns
 *    1 if the file is bigger than size
 *    1 if size is 0
 *    0 if the file is smaller than size
 *   -1 on error
 */
int big_enough(const char *file, off_t size)
{
	int sr;	/* stat return */
	struct stat estat;

	if (!size) {
		/* 0 = disabled */
		return 1;
	}
	sr = stat(file, &estat);
	if (sr) {
		perror(file);
		return -1;
	}
	if (estat.st_size <= size) {
		return 0;
	} else {
		return 1;
	}
}


/*
 * returns
 *    1 if path is a directory / a link to a directory
 *    0 if path is a file / a link to a file
 *   -1 on error
 */
int isdir(const char *path)
{
	int sr;	/* stat return */
	struct stat estat;

	sr = stat(path, &estat);
	if (sr) {
		perror(path);
		return -1;
	}

	if (!S_ISDIR(estat.st_mode))
		return 0;
	else return 1;
}


#ifdef HAVE_CANONICALIZE_FILE_NAME
/* symlink-loop-detection under GNU-systems */

int alreadythere(t_list *visited, char *path)
{
	int i;
	char *savedpath = NULL;

	for (i = 1; i <= list_count(visited); i++) {
		savedpath = list_get(visited, i);
		/*
		printf("vergleiche %s und %s\n", path, savedpath);
		*/
		if (!strcmp(path, savedpath))
			return 1;
	}
	return 0;
}

/*
 * returns
 *    0 - no loop
 *    1 - loop detected
 *   -1 - error
 */
int symloop(const char *path)
{
	char *realtarget;
	int loop = 0;
	int dir = 0;

	dir = isdir(path);
	if (dir != 1)
		return dir;

	realtarget = canonicalize_file_name(path);
	if(realtarget == NULL) {
		perror(path);
		return -1;
	}

	if (alreadythere(visited, realtarget))
		loop = 1;
	
	free(realtarget);
	return loop;
}
#endif

/*
 * adds a file / the files in the directory recursivly to the list
 */
void addpath(const char *path)
{
	struct dirent *dp;
	DIR *dirp;

	switch(isdir(path)) {
		case -1: break; /* error */
		case  0: /* path is a file or link to a file - add it to the list */
			 if (supported(path) && big_enough(path, cfg.minsize))
				 list_append(imglist, path, strlen(path) + 1);
			 break;
		case  1: /* path is a directory or a link to a directory */
#ifdef HAVE_CANONICALIZE_FILE_NAME
			 /* remember it */
			 list_append(visited, path, strlen(path) + 1);
#endif

			 dirp = opendir(path);
			 if (dirp == NULL) {
				 /* error or file? */
				 perror(path);
				 return;
			 }

			 /* reading directory entries */
			 while ((dp = readdir (dirp)) != NULL) {

				 /* cat directory + last component of path (2 ISO C90 errors) */
				 char fullpath[strlen(path) + strlen(dp->d_name) + 2]; /* +1 \0, +1 '/'*/
				 strcpy(fullpath, path);
				 /* append slash if there is none */
				 if (fullpath[strlen(fullpath) - 1] != '/')
					 strncat(fullpath, "/", 1);
				 strcat(fullpath, dp->d_name);


				 /* DT_* isn't available with -ansi */
				 if (dp->d_type == DT_LNK) {
				 /* link to a file OR a directory */
#ifdef HAVE_CANONICALIZE_FILE_NAME
					 /* This needs further inspection */
					 char *cpath;
					 cpath = canonicalize_file_name(fullpath);
					 if (cpath == NULL) {
						 fprintf(stderr, "%s: dead symlink: %s\n", PACKAGE, fullpath);
						 break;
					 }
					 switch (symloop(cpath)) {
						 case 0: addpath(cpath);
							 break;
						 case 1: /* hmm, this dir looks familiar */
							 fprintf(stderr, "symlink-loop detected\n");
							 break;
						 default:
							 /* else error */
							 break;
					 }
					 free(cpath);
#else
					 /* no risk, no fun */
					 addpath(fullpath);
#endif
				 } else if (dp->d_type == DT_DIR) {
					 /* directory */
					 if (!strcmp(dp->d_name, ".")
							 || !strcmp(dp->d_name, "..")
							 || !strcmp(dp->d_name, ".xvpics")) /* go away .xvpics! we now have ~/.thumbnails/! */
						 continue;

					 addpath(fullpath);

				 } else if (dp->d_type == DT_REG) {
					 /* file */
					 if (supported(fullpath) && big_enough(fullpath, cfg.minsize))
						 list_append(imglist, fullpath, strlen(fullpath) + 1);

				 }
			 }
			 closedir(dirp);
			 break;
	}
}


/*
 * reads files/directories from file and adds them to imglist
 */
int addlist(char *collection)
{
	FILE *fp;
	char line[MAXLINELENGTH];
	size_t last;
	unsigned int linenum = 0;

	if (strcmp(collection, "-")) {

		fp = fopen(collection, "r");
		if (fp == NULL) {
			perror(collection);
			return 1;
		}
	} else {
		fp = stdin;
	}

	while (fgets (line, MAXLINELENGTH - 1, fp)) {
		linenum++;
		last = strlen(line) - 1;

		/* empty line */
		if (linenum < 1)
			continue;

		if (line[last] != '\n') {
			fprintf(stderr, "%s: line %d too long in %s\n", PACKAGE_NAME, linenum, collection);
			continue;
		}
		/* remove newline */
		line[last] = '\0';
		last--;

		switch(line[0]) {
			case '#': /* comment */
				  continue;
			case '"': if (last > 0) {
					  if (line[last] == '"') {
						  /* gqview-collection */
						  line[last] = '\0';
						  addpath(line + 1);
					  } else
						  fprintf(stderr, "%s: ignoring invalid line %d in %s\n", PACKAGE_NAME, linenum, collection);

				  }
				  break;
			default: /* path? */
				  addpath(line);
				  break;
			/*default:  fprintf(stderr, "%s: ignoring invalid line %d in %s\n", PACKAGE_NAME, linenum, collection);*/
		}
	}

	fclose(fp);
	return 1;
}

void buildlist(int argc, char *argv[], char ***collections, int collectionsc)
{
	int i;
#ifdef HAVE_CANONICALIZE_FILE_NAME
	char *temp;
	char *iwd;	/* initial working directory */

	visited = list_init();
	/* get_current_dir_name() will malloc an array big enough */
	iwd = get_current_dir_name();
	for (i = 0; i < argc; i++) {
		temp = canonicalize_file_name(argv[i]);
		if (temp) {
			addpath(temp);
			free(temp);
		} else
			perror(argv[i]);
		/* cd - */
		chdir(iwd);
	}
	free(iwd);
#else
	/* non-GNU */
	for (i = 0; i < argc; i++)
		addpath(argv[i]);
#endif

	/* both */
	if (collectionsc > 0) {
		for (i = 0; i < collectionsc; i++)
			addlist((char *)collections[i]);
	} else {
		/* lists may have relative paths! */
		/* no coredumps in "/"! */
#ifndef DEBUG
		/* get out of the way */
		chdir("/");
#endif
	}

#ifdef HAVE_CANONICALIZE_FILE_NAME
	list_destroy(visited);
#endif


	return;
}
