/*
 * Copyright (C) 2004, 2005, 2006, 2007 Moritz Orbach <zufall@apfelboymchen.homeunix.net>
 *
 * Zufall is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zufall is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * $Id: zufall.c,v 1.19 2008/01/25 22:37:45 mori Exp $
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>

/*
 * man page:
 * According to earlier standards
 *     #include <sys/time.h>
 *     #include <sys/types.h>
 *     #include <unistd.h>
 */
#include <sys/select.h>

#include <X11/Xutil.h>
#include <Imlib2.h>

#include "config.h"

#include "list.h"
#include "changeimage.h"
#include "cfg.h"
#include "status.h"
#include "recurse.h"
#include "memman.h"

#ifdef FENCE
#include "efence.h"
#endif
#ifdef DEBUG
#include "memwatch.h"
#endif

/* vielleicht
 * - vernnftige hintergrundfarbe:
 *   . zufall ohne grelle farben
 *   . r, g und b max- bzw. minimalwerte festlegen
 * - datei auslesen in denen ausnahmen zu default optionen stehen (pro bild). ugh.
 */

t_list *imglist = NULL;
Display *Xdisplay = NULL;
Screen *scr;
Window Xroot;
int screen;

unsigned int getrandom(unsigned int max)
{
	if (max > 0)
		return(rand() % max + 1);
	else
		return 0;
} 

void exit_func(void)
{
	/*
	 * all cleanups must be done in the atexit()-function, because xlib may 
	 * call exit() at any time.
	 * All this is of course relatively cosmetic, because the kernel would 
	 * clean everything up by itself after the program exits.
	 */
	if (!list_destroy(imglist))
		fprintf(stderr, "error freeing memory..\n");

#ifdef DEBUG
	memwatchstop();
	printf("rest: %d\n", allocated());
	if (imglist == NULL)
		printf("list == NULL\n");
	else
		printf("listcount: %d\n", list_count(imglist));
#endif

}

/*
 * stolen from xcompmgr
 */
static void give_me_a_name(Display *Xdisplay, Window Xroot)
{
	Window w;

	w = XCreateSimpleWindow(Xdisplay, Xroot , 0, 0, 1, 1, 0, None, None);
	Xutf8SetWMProperties(Xdisplay, w, PACKAGE_NAME, PACKAGE_NAME, NULL, 0, NULL, NULL, NULL);
}

int initX(Display **Xdisplay, Screen **scr, Window *Xroot, int *screen)
{
	/* clients without windows don't appear in xlsclients */
	*Xdisplay = XOpenDisplay(NULL);
	if (!*Xdisplay) {
		fprintf(stderr, "%s: Unable to open display\n", PACKAGE_NAME);
		return 0;
	}
	/* XSynchronize(*Xdisplay, True); */
	*screen = DefaultScreen(*Xdisplay);
	*Xroot = RootWindow(*Xdisplay, *screen);
	*scr = ScreenOfDisplay(*Xdisplay, *screen);

	give_me_a_name(*Xdisplay, *Xroot);

	/* receive atom changes from the servers */
	XSelectInput(*Xdisplay, *Xroot, PropertyChangeMask);

	return 1;
}

void init_imlib(void)
{
	imlib_context_set_display(Xdisplay);
	imlib_context_set_visual(DefaultVisual(Xdisplay, DefaultScreen(Xdisplay)));
	imlib_context_set_anti_alias(1);
	imlib_context_set_dither(1);
	imlib_context_set_blend(0);
	/* not sure if this is needed, as no fonts are used */
	imlib_set_font_cache_size(0);
}

/*
 * FINALLY! After searching for over a year and even implementing a funny 
 * solution using a second X connection in a fork, I found the solution very 
 * easily. It's just a matter of the correct search words!
 *
 * http://lists.debian.org/debian-qa/2001/08/msg00039.html
 * http://www.linuxquestions.org/questions/programming-9/xnextevent-select-409355/?s=23201e820d9c750ebc197e352341ca66
 *
 * This function behaves like sleep, but is not only interrupted by signals, 
 * but also by X events.
 *
 * returns
 * - the remaining time to sleep
 * event_pending is
 * - 1 if the sleep was interrupted by a property change
 * - 0 if the time to sleep was over
 */
time_t xsleep(Display *dsp, Window Xroot, time_t t, int *event_pending)
{
	int x11_fd;
	fd_set select_set;
	time_t start;
	struct timeval tv;
	int rc;

	x11_fd = ConnectionNumber(dsp);

	FD_ZERO(&select_set);
	FD_SET(x11_fd, &select_set);
	tv.tv_sec = t;
	tv.tv_usec = 0;

	start = time(NULL);

	/*
	 * needed, X blocks very unkillable otherwise. TODO: understand why. 
	 *  - it's not the XSelectInput
	 */
	XFlush(dsp);

	/* error detection? A signal is "Interrupted system call"... */
	rc = select(x11_fd + 1, &select_set, NULL, NULL, &tv);

	if (rc == 0) {
		/*
		 * set time to zero manually if the time limit expired.
		 * select is not accurate enough to unconditionally calculate 
		 * it. The result could be negative or >0
		 */
		t = 0;
	} else {
		/* remaining time */
		/* printf("sleep vorher : %d\n", t); */
		t = t - (time(NULL) - start);
		/* printf("sleep nachher: %d\n", t); */

		/* t can be < 0 if the program slept longer than it wanted (e.g. busy system) */
		if (t < 0) {
			t = 0;
		}
	}

	/*
	 * Detecting wether it was a PropertyChange or not is not possible with 
	 * FD_ISSET, because in the cases of event *and* signal the fd will not 
	 * be removed from the set.
	 *
	 * There seems to be no other non-blocking "peek" function!
	 * - Another possibility would be XCheckMaskEvent / XPutBackEvent.
	 * - why does XQLength not work? Events in the queue that weren't 
	 *   flushed?
	 */
	if (XPending(dsp)) {
		/* property change */
		*event_pending = 1;
	} else {
		/* sleep finished / signal */
		*event_pending  = 0;
	}

	return t;
}

/* signal handlers */

void infotrap(p)
{
	break_sleep(0);
}

void signal_next(p)
{
	break_sleep(1);
}

void signal_quit(p)
{
	if (running()) {
		halt();
		break_sleep(1);
	}
	/*
	 * some feedback would be nice, but printf can't be called in a signal 
	 * handler
	 * 	printf("%s: i heard you\n", PACKAGE_NAME);
	 */
}

/*
 * signal handling during intialization
 */
void signal_initquit(p)
{
	signal_quit();

	/* this also kills a blocking read(0, ...) */
	signal(p, SIG_DFL);
	raise(p);
}

/* --------- */

int main(int argc, char *argv[])
{
	char *image;
	unsigned int zufall = 0;
	time_t tr;	/* time remaining */
	int error = 0;		/* return code */
	int diroffset = 0;
	char ***collections = NULL;
	int collectionsc = 0;
	int passes;
	int event_pending = 0;

	/* kill / CTRL-C */
	signal(SIGTERM, signal_initquit);
	signal(SIGINT, signal_initquit);

	/*
	 * don't buffer output more than a line, so that the filename can be 
	 * seen immediately with tail -f when the image changes.
	 * "The standard error stream stderr is always unbuffered by default."
	 */
	setvbuf(stdout, (char *)NULL, _IOLBF, 0);

	/* cfg_file(); */
	cfg_cmdln(&argc, argv, &diroffset, &collections, &collectionsc);

	argc = argc - diroffset;
	argv = &argv[diroffset];
	passes = cfg.passes;

	/*
	 * X connection is always needed, even if -n is used. This is because 
	 * the picture size needs to be calculated for use in the %-strings
	 */
	if (!initX(&Xdisplay, &scr, &Xroot, &screen)) {
		exit(EXIT_FAILURE);
	}
	init_imlib();

	if (atexit(exit_func) != 0) {
		fprintf(stderr,"%s: atexit() failed\n", PACKAGE_NAME);
		exit(EXIT_FAILURE);
	}

	imglist = list_init();
	srand(time(0));

	buildlist(argc, argv, collections, collectionsc);
	/* kill / CTRL-C */
	signal(SIGTERM, signal_quit);
	signal(SIGINT, signal_quit);

	/* free memory for pointers to collections */
	freemem(collections, sizeof(*collections));

	if (list_count(imglist) > 0) {
		/* imagechange, info. */
		signal(SIGHUP, signal_next);
		signal(SIGUSR1, infotrap);

		if (cfg.setbg && (list_count(imglist) < 10 && list_count(imglist) > 1)) {
			/* few images. Let imlib2 do caching */
			imlib_set_cache_size(5120 * 1024);
		} else {
			imlib_set_cache_size(0);
		}
	} else {
		error = 1;
		halt();
		fprintf(stderr, "Sorry, no supported images found\n");
	}

	while (running() && !error) {

		if (!cfg.sequential)
			zufall = getrandom(list_count(imglist));
		else {
			zufall++;
			if (zufall > list_count(imglist))
					/* wrap around */
					zufall = 1;
		}
		image = list_get(imglist, zufall);

		printf("so ein zufall!: %s\n", image);

		/* don't sleep if there is only one image left */
		if (list_count(imglist) > 1 && passes != 1)
			tr = cfg.delay;
		else {
			tr = 0;
			printf("no pictures left\n");
			halt();
		}
		if (changeimage(Xdisplay, Xroot, scr, screen, image)) {

			while (tr != 0) {
				/* admire background */
				tr = xsleep(Xdisplay, Xroot, tr, &event_pending);

				/* sleep is over / event / signal. continue sleep or new sleep? */

				if (sleep_broken()) {
					/* imagechange or exit */
					tr = 0;
					break_sleep(0);
				} else if (event_pending) {
					/* event */
					property_changed(Xdisplay, Xroot);
				} else if (tr != 0) {
					/* must be USR1 */

					/*
					 * another Xsync... needed so that in 
					 * case the X server died no message 
					 * will be printed
					 */
					XSync(Xdisplay, False);
					
					unsigned int tmp;
					tmp = list_allocated();
					printf("allocated: %d b (%.2lf kb), %d nodes\n", tmp, (double)tmp / 1024, list_count(imglist));
					printf("%s\n", image);
					printf("%d seconds (%.1d:%.2d) remaining\n", tr, (int)tr / 60, (int)tr % 60);
				}
			}

			if (cfg.passes != -1)
				passes--;
		} else {
			/* changeimage failed */
			fprintf(stderr, "Unable to load image file \"%s\".\n", image);
			if (!list_delete(imglist, zufall)) {
				fprintf(stderr, "error freeing memory..\n");
				error = 1;
			}
		}
	}
	printf("%s: exiting\n", PACKAGE_NAME);

	/*
	 * XSync will directly call exit() in case the connection does 
	 * not exist anymore
	 */
	XSync(Xdisplay, True);
	if (cfg.setbg) {
		permanent_on_exit(Xdisplay, Xroot);
	}
	XCloseDisplay(Xdisplay);

	/* make return-code system-independent. Probably silly. */
	switch (error) {
		case  0: error = EXIT_SUCCESS;
			 break;
		default: error = EXIT_FAILURE;
	}
	exit(error);
}
