//
// C++ Implementation: kpgfunction
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgfunction.h"

// include files for KDE
#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <ktextedit.h>

#include "kpgserver.h"
#include "kpgdatabase.h"
#include "../kpglinklabel.h"
#include "kpgfunctionsfolder.h"
#include "../kpgutil.h"

KPGFunctionArgument::KPGFunctionArgument(unsigned int iSequence, 
    const QString & strName, 
    const QString & strMode, 
    const QString & strTypName)
{
    m_iSequence = iSequence;
    m_strName = strName;
    m_strTypName = strTypName;
        
    if(strMode == "i")
    {
        m_eMode = input;
    }
    else if(strMode == "o")
    {
        m_eMode = output;
    }
    else if(strMode == "b")
    {
        m_eMode = inout;
    }
    else
    {
        m_eMode = input; // Default mode is IN
    }
}

KPGFunctionArgument::KPGFunctionArgument(unsigned int iSequence, 
                                         const QString & strName, 
                                         EMode eMode, 
                                         const QString & strTypName)
{
    m_iSequence = iSequence;
    m_strName = strName;
    m_strTypName = strTypName;
    m_eMode = eMode;
}

KPGFunction::KPGFunction(KPGFunctionsFolder *parent, const QString name, pqxx::oid _oid)
 : KPGObject(parent, name, _oid)
{
	setPixmap(0, *m_pIconFunction);
}

KPGFunction::KPGFunction(KPGFunctionsFolder *parent, KPGFunction *after, const QString name, pqxx::oid _oid)
 : KPGObject(parent, after, name, _oid)
{
	setPixmap(0, *m_pIconFunction);
}


KPGFunction::~KPGFunction()
{
}

void KPGFunction::setProperties(const pqxx::result::tuple &pqxxTuple,
								const KPGConnection *pConnection)
{
	pqxxTuple["proisagg"].to(m_bIsAgg);
	pqxxTuple["prosecdef"].to(m_bIsSecdef);
	pqxxTuple["proisstrict"].to(m_bIsStrict);
	pqxxTuple["proretset"].to(m_bReturnSet);
	pqxxTuple["pronargs"].to(m_uiNArgs);
    pqxxTuple["prorettype"].to(m_oidReturnType);
				
	m_strDescription = pConnection->toUnicode(pqxxTuple["description"].c_str());
	m_strVolatile = pqxxTuple["provolatile"].c_str();
	m_strBin = pqxxTuple["probin"].c_str();
	m_strACL = pqxxTuple["proacl"].c_str();
	m_strReturnTypName = pqxxTuple["typname"].c_str();
	m_strLanName = pqxxTuple["lanname"].c_str();
	m_strFuncOwner = pqxxTuple["funcowner"].c_str();
	
	//--- Set icon for function type
	if(m_strLanName == "c")
		setPixmap(0, *m_pIconFunctionC);
	else
		if(m_strLanName == "sql")
			setPixmap(0, *m_pIconFunctionSql);
		else
			if(m_strLanName == "plpgsql")
				setPixmap(0, *m_pIconFunctionPl);
}

// Set arguments list. All parameters come in form: {aaa, bbb, ccc}
void KPGFunction::setListArguments(const QString &strArgTypes, const QString &strAllArgTypes, const QString &strArgModes, const QString &strArgNames)
{
	m_listArguments.clear();
	
	QStringList strListNames;
	if(strArgNames.length() > 2)
	 	strListNames = QStringList::split (",", strArgNames.mid(1, strArgNames.length() - 2), true);
	
	QStringList strListModes;
	if(strArgModes.length() > 2)
	 	strListModes = QStringList::split (",", strArgModes.mid(1, strArgModes.length() - 2), true);
	
	// Use m_strArgTypes, when strListAllArgTypes is empty
	QStringList strListArgTypesOids;
	if(strAllArgTypes.length() > 2)
       strListArgTypesOids = QStringList::split (",", strAllArgTypes.mid(1, strAllArgTypes.length() - 2), true);
	else
		strListArgTypesOids = QStringList::split (" ", strArgTypes);		
	
	// Number of all arguments - max list size
	unsigned int nArguments = m_uiNArgs;
	if(strListNames.count() > nArguments) nArguments = strListNames.count();
	if(strListModes.count() > nArguments) nArguments = strListModes.count();
	if(strListArgTypesOids.count() > nArguments) nArguments = strListArgTypesOids.count();
	
	if((strListNames.count() > 0) && (nArguments != strListNames.count())) 
	{
		kdDebug() << text(0) << QString(" - Names count %1, num of args %2").arg(strListNames.count()).arg(nArguments) << endl;
		kdDebug() << "strAllArgTypes " << strAllArgTypes << endl;
		kdDebug() << "strArgModes " << strArgModes << endl;
		kdDebug() << "strArgNames " << strArgNames << endl;
	}
	if((strListModes.count() > 0) && (nArguments != strListModes.count())) 
	{
		kdDebug() << text(0) << QString(" - Modes count %1, num of args %2").arg(strListModes.count()).arg(nArguments) << endl;
		kdDebug() << "strAllArgTypes " << strAllArgTypes << endl;
		kdDebug() << "strArgModes " << strArgModes << endl;
		kdDebug() << "strArgNames " << strArgNames << endl;
	}
	if((strListArgTypesOids.count() > 0) && (nArguments != strListArgTypesOids.count())) 
	{
		kdDebug() << text(0) << QString(" - Types count %1, num of args %2").arg(strListArgTypesOids.count()).arg(nArguments) << endl;
		kdDebug() << "strArgTypes " << strArgTypes << endl;
		kdDebug() << "strAllArgTypes " << strAllArgTypes << endl;
		kdDebug() << "strArgModes " << strArgModes << endl;
		kdDebug() << "strArgNames " << strArgNames << endl;
	}
	
	for(unsigned int i = 0; i < nArguments; i++)
	{
        QString strName;
         
        if(strListNames.count() > 0)
        { 
         	strName = strListNames[i]; 
			if(strName == "\"\"") strName.truncate(0);
        }
        
        QString strMode;
        if(strListModes.count() > 0)
        {
        	strMode = strListModes[i];
        }
        
        QString strArgTypeOid;
        QString strArgTypeName;
        if(strListArgTypesOids.count() > 0)
        {
        	strArgTypeOid = strListArgTypesOids[i];
        	QString strQuery("SELECT typname FROM pg_catalog.pg_type typ WHERE oid=" + strArgTypeOid);
        	//QString strQuery("SELECT pg_catalog.format_type(" + strArgTypeOid + ", NULL)");
        	
        	try
        	{
        		pqxx::result pqxxResult = connection()->runQuery(strQuery);
	
				if(pqxxResult.size() != 1)
				{
					kdError() << k_funcinfo "Expect one row in result !" <<  endl;
					continue;
				}
				
				strArgTypeName = pqxxResult[0][0].c_str();
			}
			catch (const std::exception &e)
			{
				kdError() << k_funcinfo << e.what() << endl;
				throw KPGSqlException(e.what(), strQuery);
			} 
        }
        
        
        m_listArguments.append(KPGFunctionArgument(i + 1, strName, strMode, strArgTypeName));
    }
}

// Return name with arguments
const QString KPGFunction::nameWithArguments() const
{
	QString strNameWithArguments = text(0);
	strNameWithArguments.append(arguments());
	return strNameWithArguments;
}

// Return arguments, with names and IN/OUT info. 
const QString KPGFunction::arguments() const
{
	if(m_uiNArgs != m_listArguments.count())
	{
		return argumentTypes();
	}
	
	QString strArguments("(");
	
	bool bFirst = true;
    for(ListFunctionArguments::const_iterator cit = m_listArguments.begin(); cit != m_listArguments.end(); ++cit)
	{ 
		if(bFirst)
			bFirst = false;
		else
			strArguments.append(", ");
		
		if((*cit).mode() == "i") strArguments.append("IN ");
		else if((*cit).mode() == "o") strArguments.append("OUT ");
		else if((*cit).mode() == "b") strArguments.append("INOUT ");
		
		if((*cit).name().length() > 0)
		{
			strArguments.append((*cit).name());
			strArguments.append(" ");
		}
		
		strArguments.append((*cit).typName());
	}
	strArguments.append(")");
	
	return strArguments;
}

// Refresh only function info, without childs objects
void KPGFunction::refreshItem() throw(const KPGSqlException &)
{
	// Get pointer to server for version info
	KPGDatabase *pDatabase = static_cast <KPGDatabase *> (parent()->parent()->parent());
	KPGServer *pServer = static_cast <KPGServer *> (pDatabase->parent());
	
	// obtain function info
	QString strQuery("SELECT pr.oid, pr.proname, description, pr.proisagg, pr.prosecdef, pr.proisstrict, proretset, pr.provolatile, pr.pronargs, pr.probin, pr.proacl, prorettype, \
typ.typname, lanname, pg_catalog.pg_get_userbyid(proowner) AS funcowner, pr.proargtypes, pg_catalog.oidvectortypes(pr.proargtypes) AS arguments, pr.prosrc");
	
	if(((pServer->versionMajor() == 8) && (pServer->versionMiddle() >= 1)) || (pServer->versionMajor() > 8))
    {
  		strQuery.append(", pr.proallargtypes, pr.proargmodes, pr.proargnames ");
  	}
  	else
  		strQuery.append(" ");
	
	strQuery.append("FROM pg_catalog.pg_proc pr ");
	strQuery.append("JOIN pg_catalog.pg_type typ ON typ.oid=prorettype ");
	strQuery.append("JOIN pg_catalog.pg_language LNG ON LNG.oid=prolang ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_description des ON des.objoid=pr.oid ");
	
	strQuery.append("WHERE pr.oid = " + QString("%1").arg(m_oid));
		
	try
	{
		KPGConnection *pConnection = connection();
		pqxx::result pqxxResult = pConnection->runQuery(strQuery);
	
		if(pqxxResult.size() != 1)
		{
			kdError() << k_funcinfo "Expect one row in result !" <<  endl;
			return;
		}
				
		setProperties(pqxxResult[0], pConnection);
		
		setArgTypes(pqxxResult[0]["arguments"].c_str());
		setSourceCode(pConnection->toUnicode(pqxxResult[0]["prosrc"].c_str()));
			
		if(((pServer->versionMajor() == 8) && (pServer->versionMiddle() >= 1)) || (pServer->versionMajor() > 8))
    	{	
			setListArguments(pqxxResult[0]["proargtypes"].c_str(),
			    pqxxResult[0]["proallargtypes"].c_str(),
				pqxxResult[0]["proargmodes"].c_str(),
				pqxxResult[0]["proargnames"].c_str());
		}
		else
		{
			setListArguments(pqxxResult[0]["proargtypes"].c_str(),
			    QString::null,
				QString::null,
				QString::null);
		}
	
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		throw KPGSqlException(e.what(), strQuery);
	} 
}

void KPGFunction::setSourceCode(const QString &strSourceCode) 
{ 
	m_strSourceCode = strSourceCode; 
	
	// Strip the leading blank line from the source
	if(m_strSourceCode.startsWith("\n"))
	{
		m_strSourceCode.replace(0, 1, "");
	}
}
