//
// C++ Implementation: kpgdebugger
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2007
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgdebugger.h"

// include files for Qt
#include <qwidgetstack.h> 
#include <qtextcodec.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qinputdialog.h> 

// include files for KDE
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kcursor.h>
#include <kdebug.h>
#include <kcombobox.h>
#include <ktextedit.h>
#include <kfinddialog.h>
#include <kfind.h>

// include files for KatePart
#include <kate/document.h>  // KatePart document

// application specific includes
#include "../kpgutil.h"
#include "../kpogreview.h"
//#include "../DbObjects/kpgconnection.h"
#include "../DbObjects/kpgdatabase.h"
#include "../DbObjects/kpgfunction.h"
#include "../kpgsqldialog.h"
#include "../DbObjects/kpgtablecolumn.h"
#include "kpgdebuggerchildview.h"

#define NUM_OF_RECORDS_PER_PAGE 500 // How many records are loaded to result table 
#define UPDATE_TIMER_INTERVAL 500 // 0.5 sec. for updating variables, breakpoints, stack ...

// Parameters table columns
#define COL_NAME 0
#define COL_TYPE_NAME 1
#define COL_MODE 2
#define COL_VALUE 3

// Variables table columns
#define COL_VARIABLES_NAME          0
#define COL_VARIABLES_TYPE          1
#define COL_VARIABLES_VALUE         2
#define COL_VARIABLES_LINENUMBER    6

#define TAB_PAGE_PARAMETERS_TABLE   0
#define TAB_PAGE_RESULT_TABLE       1
#define TAB_PAGE_RESULT_TEXT        2
#define TAB_PAGE_VARIABLES_TABLE    3
#define TAB_PAGE_STACK_TABLE        4
#define TAB_PAGE_DEBUGGER_TEXT      5

KPGDebugger::KPGDebugger(KPGDebuggerChildView *pParent, 
	KPoGreView *pPoGreView,
	KParts::Factory *pKatePartFactory,
	KXMLGUIFactory *pXmlGuiFactory,
 	const PGSTD::string& strConnectionOptions,
	const QString & strNamespaceName, 
	KPGFunction *pFunction,
 	const QString& strServerName,
 	pqxx::oid oidDatabase
	)
	:
	KXMLGUIClient(pPoGreView),
    KPGDebuggerBase(pParent, "KPGDebugger"),
    m_pXmlGuiFactory(pXmlGuiFactory),
   m_pFind(0) 
{
	// Setup GUI
    //setFactory(pXmlGuiFactory);
    setXMLFile("kpgdebugger.rc", true);
	m_bIsAddedToGuiFactory = false;
	
	// Initialize members
	m_strConnectionOptions = strConnectionOptions;
	m_strNamespaceName = strNamespaceName;
	m_strFunctionName = pFunction->text(0);
	m_oidFunction = pFunction->oid();
	m_bReturnSet = pFunction->returnSet();
		
	// Store parent DB and connection, to be able to make check,
	// that all debugged functions is from the same Conn/DB
	m_oidDatabase = oidDatabase;
	m_strServerName = strServerName;
	
	m_oidFunctionCurrent = 0;
	
	m_nTotalRows = 0;
	m_nTotalCols = 0;
	m_nFetchedRows = 0;
    
    m_bParametersNotEntered = true;
		
	//-----------------------------------------------------------------
	// Debugger actions
	m_pActRun = new KAction(i18n("Start in debugger"), "dbgrun", Key_F9, this, SLOT(slotExecute()), actionCollection(), "debugger_start");
	
	m_pActStop = new KAction(i18n("Stop"), "stop", 0, this, SLOT(slotStop()), actionCollection(), "debugger_stop");
  	
  	m_pActStepOver = new KAction(i18n("Step &Over"), "dbgnext", Key_F10, this, SLOT(slotStepOver()), actionCollection(), "debugger_step_over");

	m_pActStepInto = new KAction(i18n("Step &Into"), "dbgstep", Key_F11, this, SLOT(slotStepInto()), actionCollection(), "debugger_step_into");
	
	m_pActToggleBreakPoint = new KAction(i18n("Toggle &Breakpoint"), "kpg_togglebp", 0, this, SLOT(slotToggleBreakPoint()), actionCollection(), "debugger_toggle_bpoint");

    m_pActFetchNext = new KAction(i18n("Fetch &next"), "1downarrow", 0, this, SLOT(slotFetchNext()), actionCollection(), "debugger_fetch_next");
    
    m_pActFetchAll = new KAction(i18n("Fetch &all"), "2downarrow", 0, this, SLOT(slotFetchAll()), actionCollection(), "debugger_fetch_all");
    
    m_pActFindInResult = new KAction(i18n("&Find in result..."), "find", 0, this, SLOT(slotFindInResult()), actionCollection(), "edit_find_in_result");
    m_pActFindInResult->setWhatsThis(i18n("Find in result table."));
    
    m_pActFindNextInResult = new KAction(i18n("Find next in result"), "next", 0, this, SLOT(slotFindNextInResult()), actionCollection(), "edit_find_next_in_result");
    m_pActFindNextInResult->setWhatsThis(i18n("Find next in result table."));
    
    //-----------------------------------------------------------------
    // Common clipboard actions - result table
    m_pActResultCopyCell = new KAction(i18n("&Copy cell"), "editcopy", 0, this, SLOT(slotCopyResultCell()), actionCollection(), "result_copy_cell");
    
    m_pActResultCopyRow = new KAction(i18n("Copy &row"), 0, 0, this, SLOT(slotCopyResultRow()), actionCollection(), "result_copy_row");
    
    m_pActResultCopyTableCsv = new KAction(i18n("Copy table to C&SV"), 0, 0, this, SLOT(slotCopyResultTableCsv()), actionCollection(), "result_copy_table_csv");
    
    m_pActResultCopyTableXml = new KAction(i18n("Copy table to &XML"), 0, 0, this, SLOT(slotCopyResultTableXml()), actionCollection(), "result_copy_table_xml");
    
    //-----------------------------------------------------------------
    // Common clipboard actions - variables table
    m_pActVariablesCopyCell = new KAction(i18n("&Copy cell"), "editcopy", 0, this, SLOT(slotCopyVariablesCell()), actionCollection(), "variables_copy_cell");
    
    m_pActVariablesCopyRow = new KAction(i18n("Copy &row"), 0, 0, this, SLOT(slotCopyVariablesRow()), actionCollection(), "variables_copy_row");
    
    m_pActVariablesDepositValue = new KAction(i18n("Deposit value"), "math_brace", 0, this, SLOT(slotVariablesDepositValue()), actionCollection(), "variables_deposit_value");
    
    //-----------------------------------------------------------------
    connect(m_pTableResult, SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(slotContextMenuResultRequested(int, int, const QPoint &)));
    
    connect(m_pTableVariables, SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(slotContextMenuVariablesRequested(int, int, const QPoint &)));
    
    //-----------------------------------------------------------------
    // Initial actions state   
	m_pActRun->setEnabled(true);
    m_pActStop->setEnabled(false);
    m_pActStepOver->setEnabled(false);
	m_pActStepInto->setEnabled(false);
	m_pActToggleBreakPoint->setEnabled(false);
    m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
    
    m_pActFindInResult->setEnabled(false);
    m_pActFindNextInResult->setEnabled(false);
    
    m_pActResultCopyCell->setEnabled(false);
    m_pActResultCopyRow->setEnabled(false);
    m_pActResultCopyTableCsv->setEnabled(false);
    m_pActResultCopyTableXml->setEnabled(false);
    
    m_pActVariablesCopyCell->setEnabled(false);
    m_pActVariablesCopyRow->setEnabled(false);
    m_pActVariablesDepositValue->setEnabled(false);
    
    //-----------------------------------------------------------------
    // Create Kate part editor
	m_pKateView = createKatePart(pKatePartFactory);
    m_pFrameSourceCodeLayout->addWidget(m_pKateView, 0, 0 );
        
    
    // Don't allow user to edit source code  
    m_pKateView->getDoc()->setReadWrite(false);
    KAction *pActToggleWriteLock = m_pKateView->actionCollection()->action("tools_toggle_write_lock");
    if(pActToggleWriteLock)
    {
        pActToggleWriteLock->setEnabled(false);  
    }
			
	// install a working kate part popup dialog thingy
  	if (static_cast<Kate::View*>(m_pKateView->qt_cast("Kate::View")))
  	{
    	static_cast <Kate::View*> (m_pKateView->qt_cast("Kate::View"))->installPopup ((QPopupMenu*)(pXmlGuiFactory->container("ktexteditor_popup", pPoGreView)) );
  	}
    
    m_pKateView->setLineNumbersOn(true);
    m_pKateView->setIconBorder(true);
        
    KTextEditor::HighlightingInterface *pHighlightIface = KTextEditor::highlightingInterface(m_pKateView->getDoc());
        
    for(uint i = 0; i < pHighlightIface->hlModeCount(); i++) 
    {
    	//kdDebug() << "hlmode("<<i<<"): " << pHighlightIface->hlModeName(i) << endl;
        if (pHighlightIface->hlModeName(i).contains("SQL (PostgreSQL)", false))
        {
        	pHighlightIface->setHlMode(i);
        	break;
        }
     }
    
    setEditorText(pFunction->sourceCode());
		
	// Setup breakpoints icons	
	KIconLoader *loader = KGlobal::iconLoader();
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointActive, loader->loadIcon("breakpoint_enabled", KIcon::Small));
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointReached, loader->loadIcon("breakpoint_reached", KIcon::Small));
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointDisabled, loader->loadIcon("breakpoint_disabled", KIcon::Small));
    
    m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::Execution, loader->loadIcon("currentline", KIcon::Small));
    
    pixmapVariableChanged = loader->loadIcon("currentline", KIcon::Small);
				
	// Setup tab icons
	m_pTabWidget->setTabIconSet(pTabParameters, QIconSet(QPixmap(UserIcon("argument"))));
	m_pTabWidget->setTabIconSet(pTabVariables, QIconSet(QPixmap(SmallIcon("math_brace"))));
	m_pTabWidget->setTabIconSet(pTabResult, QIconSet(QPixmap(SmallIcon("view_text"))));
	m_pTabWidget->setTabIconSet(pTabStack, QIconSet(QPixmap(SmallIcon("view_choose"))));
	m_pTabWidget->setTabIconSet(pTabTextResult, QIconSet(QPixmap(SmallIcon("openterm"))));
	m_pTabWidget->setTabIconSet(pTabDebuggerOutput, QIconSet(QPixmap(SmallIcon("kpg_debugger"))));
	
	//-----------------------------------------------------------------
	// Make own copy of arguments info
	m_listArguments = pFunction->listArguments();
	
	// Setup parameters table
    if(m_listArguments.count() > 0)
	{
        m_pTableParameters->setNumRows(m_listArguments.count());
		int nRow = 0;
		
        for(ListFunctionArguments::const_iterator cit = m_listArguments.begin(); cit != m_listArguments.end(); ++cit)
		{
			m_pTableParameters->setText(nRow, COL_NAME, (*cit).name());
			
			if((*cit).typName().length() > 0)
			{
				m_pTableParameters->setText(nRow, COL_TYPE_NAME, (*cit).typName());
				
				QPixmap *pTypePixmap = KPGTableColumn::getPixmapForType((*cit).typName());
				m_pTableParameters->setPixmap(nRow, COL_NAME, *pTypePixmap);
			}
			
			
            m_pTableParameters->setText(nRow, COL_MODE, (*cit).modeAsString());
			nRow++;
		}
	}
	
	m_pTableParameters->setColumnReadOnly(0, true);
	m_pTableParameters->setColumnReadOnly(1, true);
	m_pTableParameters->setColumnReadOnly(2, true);
	
	m_threadDebugger.setParameters(this, strConnectionOptions, pFunction->oid());
	m_connInThreadFunction.setEventReceiver(this, &m_pqxxResultFunction);
}

KPGDebugger::~KPGDebugger()
{
	if(m_pKateView != 0)
    {
        removeFromGuiFactory();

        // remove the view, then the doc
        Kate::Document *pDoc = m_pKateView->getDoc();
        delete m_pKateView;
        delete pDoc;
    }
}

// Add yourself and Kate view to GUI factory
void KPGDebugger::addToGuiFactory()
{
	if(m_pKateView && !m_bIsAddedToGuiFactory)
    {
        //setFactory(m_pXmlGuiFactory);
        m_pXmlGuiFactory->addClient(this);
        m_pXmlGuiFactory->addClient(m_pKateView);
        m_bIsAddedToGuiFactory = true;
		m_pKateView->setFocus();
    }
}

// Remove yourself and Kate view from GUI factory
void KPGDebugger::removeFromGuiFactory()
{
    if(m_pKateView && m_bIsAddedToGuiFactory)
    {
        //setFactory(m_pXmlGuiFactory);
        m_pXmlGuiFactory->removeClient(this);
        m_pXmlGuiFactory->removeClient(m_pKateView);
        m_bIsAddedToGuiFactory = false;
    }
}

// Create Kate part view
Kate::View* KPGDebugger::createKatePart(KParts::Factory* pFactory)
{
    // The library was found, so create the Kate::Document
    KTextEditor::Document *doc = (KTextEditor::Document *) pFactory->createPart(m_pFrameSourceCode, "", this, "", "KTextEditor::Document");
	
    // The document only represents the document, to view
    // the document's content
    // we have to create a view for the document.
	Kate::View *pView = (Kate::View *) doc->createView( m_pFrameSourceCode, 0L );
    
    // all went well, so return the view
	return pView;
}

// Set function source code
void KPGDebugger::setEditorText(const QString &strTest)
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
	m_pKateView->getDoc()->setReadWrite(true);
	pEditIface->setText(strTest);
    m_pKateView->getDoc()->setReadWrite(false);
    
	KTextEditor::UndoInterface *pUndoIface = KTextEditor::undoInterface(m_pKateView->document());
	pUndoIface->clearUndo();
	pUndoIface->clearRedo();
}

// Get function source code
const QString KPGDebugger::editorText() const 
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
    return pEditIface->text();
}

bool KPGDebugger::resultEmpty() const
{ 
    return m_pTableResult->numRows() == 0; 
}


// Enables GUI actions
void KPGDebugger::enableGui()
{
	m_pTableParameters->setEnabled(true);
	
	((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	
	bool bRunning = m_connInThreadFunction.running();
	
	m_pActRun->setEnabled(true);
    m_pActStop->setEnabled(bRunning);
    m_pActStepOver->setEnabled(bRunning); 
    m_pActStepInto->setEnabled(bRunning);
    m_pActToggleBreakPoint->setEnabled(bRunning);
    m_pActVariablesDepositValue->setEnabled(bRunning);
        
    
    m_pActFetchNext->setEnabled(!bRunning && !fetchedAllRows());
    m_pActFetchAll->setEnabled(!bRunning && !fetchedAllRows());
    
    bool bResultNotEmpty = !resultEmpty();
    m_pActFindInResult->setEnabled(!bRunning && bResultNotEmpty);
    
    m_pActResultCopyCell->setEnabled(bResultNotEmpty);
    m_pActResultCopyRow->setEnabled(bResultNotEmpty);
    m_pActResultCopyTableCsv->setEnabled(bResultNotEmpty);
    m_pActResultCopyTableXml->setEnabled(bResultNotEmpty);
    
    bool bVariablesNotEmpty = m_pTableVariables->numRows() > 0;
    m_pActVariablesCopyCell->setEnabled(bVariablesNotEmpty);
    m_pActVariablesCopyRow->setEnabled(bVariablesNotEmpty);
}
	
// Disable GUI actions
void KPGDebugger::disableGui()
{
	m_pTableParameters->setEnabled(false);
	//m_pComboBoxTransaction->setEnabled(false);
	//emit sigSetTerminalIcon(this, 1); // red terminal icon
	((QWidget *) parent())->setCursor(KCursor::waitCursor());
		
	m_pActRun->setEnabled(false);
	m_pActStop->setEnabled(false);
	m_pActStepOver->setEnabled(false); 
	m_pActStepInto->setEnabled(false);
	m_pActToggleBreakPoint->setEnabled(false);
    m_pActVariablesDepositValue->setEnabled(false);
    
    m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
    m_pActFindInResult->setEnabled(false); 
    m_pActFindNextInResult->setEnabled(false);
    
    m_pActResultCopyCell->setEnabled(false);
    m_pActResultCopyRow->setEnabled(false);
    m_pActResultCopyTableCsv->setEnabled(false);
    m_pActResultCopyTableXml->setEnabled(false);
    
    m_pActVariablesCopyCell->setEnabled(false);
    m_pActVariablesCopyRow->setEnabled(false);
}	

// Returns true, if breakpoint is set to given line
bool KPGDebugger::isBreakpointSet(unsigned int uiLineNumber) const
{
	KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
	
	QPtrList<KTextEditor::Mark> listOfMarks = pMarkIface->marks();
	KTextEditor::Mark *pMark = 0;
	for(pMark = listOfMarks.first(); pMark; pMark = listOfMarks.next())
	{
		if(pMark->line != uiLineNumber) continue;
		
        if((pMark->type & KTextEditor::MarkInterface::BreakpointActive) || (pMark->type & KTextEditor::MarkInterface::BreakpointReached))
			return true;
	}
	return false;
}

// Clear all breakpoints marks in editor
void KPGDebugger::clearBreakpointMarks() const
{
	KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
	
	QPtrList<KTextEditor::Mark> listOfMarks = pMarkIface->marks();
	KTextEditor::Mark *pMark = 0;
	for(pMark = listOfMarks.first(); pMark; pMark = listOfMarks.next())
	{
		pMarkIface->removeMark(pMark->line, KTextEditor::MarkInterface::BreakpointActive);
		pMarkIface->removeMark(pMark->line, KTextEditor::MarkInterface::BreakpointReached);
		pMarkIface->removeMark(pMark->line, KTextEditor::MarkInterface::BreakpointDisabled);
        pMarkIface->removeMark(pMark->line, KTextEditor::MarkInterface::Execution);
	}
}

void KPGDebugger::slotExecute()
{
	if(m_connInThreadFunction.running() == false)
	{
        if(m_bParametersNotEntered)
        {
            int nArgs = m_pTableParameters->numRows();
            for(int i = 0; i < nArgs; i++)
            {
                if(m_pTableParameters->text(i, COL_MODE) == "OUT")
                {
                    continue; // skip OUTput parameters 
                }
                
                if(m_pTableParameters->text(i, COL_VALUE).isEmpty()) 
                {
                    if(KMessageBox::questionYesNo(this, i18n("Some parameters not set. Execute anyway ?"), "KPoGre") == KMessageBox::Yes) 
                    {
                        m_bParametersNotEntered = false;
                        break;
                    }
                    else 
                    {
                        m_pTabWidget->setCurrentPage(TAB_PAGE_PARAMETERS_TABLE);
                        m_pTableParameters->setCurrentCell(i, COL_VALUE);
                        return;
                    }
                }
            }
        }
        
        m_pTextEditResult->clear();
		m_pTextEditDebuggerOut->clear();
        
        m_pActRun->setText(i18n("Continue"));
		
		// clear previous result
		m_pTableResult->setNumRows(0);
		m_pTableResult->setNumCols(0);

		m_pqxxResultFunction.clear(); // free memory
        m_nTotalRows = 0;
        m_nTotalCols = 0;
		
		disableGui();
		m_threadDebugger.start();
	}
	else
	{
		m_threadDebugger.setAction(KPGDebuggingThread::actionContinue);
		disableGui();
		m_threadDebugger.unlockMutex();
	}
}

void KPGDebugger::slotStepOver()
{
	m_threadDebugger.setAction(KPGDebuggingThread::actionStepOver);
	disableGui();
	m_threadDebugger.unlockMutex();
}

void KPGDebugger::slotStepInto()
{
    // Check, if is attached other function as root
    bool bAnotherFunctionAttached = false;
    const MapFunctionDebugInfo & mapFunctionDebugInfo = m_threadDebugger.mapFunctionDebugInfo();
    MapFunctionDebugInfo::ConstIterator it;
    for(it = mapFunctionDebugInfo.begin(); it != mapFunctionDebugInfo.end(); ++it) 
    {
        if(it.key() != m_oidFunction)
        {
            bAnotherFunctionAttached = true;
            break;
        }
    }
    
    /*if(!bAnotherFunctionAttached)
    {
        if(KMessageBox::questionYesNo(this, i18n("No another function attached to debugger. Step into has no matter. Continue ?"), "KPoGre") != KMessageBox::Yes) 
        {
            return;
        }
    }*/
    
	m_threadDebugger.setAction(KPGDebuggingThread::actionStepInto);
	disableGui();
	m_threadDebugger.unlockMutex();
}

void KPGDebugger::slotStop()
{
    if(KMessageBox::questionYesNo(this, i18n("Stop debugger ?"), "KPoGre") != KMessageBox::Yes) 
    {
        return;   
    }
    
	m_threadDebugger.setAction(KPGDebuggingThread::actionAbortTarget);
	disableGui();
	m_threadDebugger.unlockMutex();
}

void KPGDebugger::slotFetchNext()
{
    fetchNext();
}

void KPGDebugger::slotFetchAll()
{
    fetchAll();     
}

// Find first occurence of text in result table  
void KPGDebugger::slotFindInResult()
{
    findInResultFirst(m_listOfResultQuerySearchHistory);     
}

// Find next occurence of text in result table  
void KPGDebugger::slotFindNextInResult()
{
    findInResultNext();     
}

void KPGDebugger::slotCopyResultCell()
{
    KPGUtil::copyCell(m_pTableResult);
}

void KPGDebugger::slotCopyResultRow()
{
    KPGUtil::copyRow(m_pTableResult);
}

void KPGDebugger::slotCopyResultTableCsv()
{
    KPGUtil::copyTableCsv(m_pTableResult);
}

void KPGDebugger::slotCopyResultTableXml()
{
    KPGUtil::copyTableXml(m_pTableResult);
}

void KPGDebugger::slotCopyVariablesCell()
{
    KPGUtil::copyCell(m_pTableVariables);
}

void KPGDebugger::slotCopyVariablesRow()
{
    KPGUtil::copyRow(m_pTableVariables);
}

void KPGDebugger::slotToggleBreakPoint()
{
	uint uiLine, uiCol;
	viewCursorInterface(m_pKateView)->cursorPositionReal(&uiLine, &uiCol);
	
	if(isBreakpointSet(uiLine) == false)
	{
		m_threadDebugger.setActionSetBreakpoint(m_oidFunctionCurrent, uiLine);
	}
	else
	{
		m_threadDebugger.setActionDropBreakpoint(m_oidFunctionCurrent, uiLine);
	}
	
	disableGui();
	m_threadDebugger.unlockMutex();
}

void KPGDebugger::slotVariablesDepositValue()
{
    int iRow = m_pTableVariables->currentRow();
    if(iRow < 0) return;
    
    QString strVariableValue(m_pTableVariables->text(iRow, COL_VARIABLES_VALUE));
    QString strVariableName(m_pTableVariables->text(iRow, COL_VARIABLES_NAME));
    
    bool ok;
    QString strNewValue = QInputDialog::getText(i18n("Enter new value"), strVariableName, QLineEdit::Normal, strVariableValue, &ok, this );
    if( ok && !strNewValue.isEmpty()) 
    {
        uint uiVariableLine = m_pTableVariables->text(iRow, COL_VARIABLES_LINENUMBER).toUInt();
        m_threadDebugger.setActionDepositValue(strVariableName,
                                               uiVariableLine,
                                               strNewValue
                                               );
        disableGui();
        m_threadDebugger.unlockMutex();
    } 
}

// Display popup menu for result table
void KPGDebugger::slotContextMenuResultRequested(int, int, const QPoint &pos)
{
    QWidget * pContainer = factory()->container("popupDebuggerResultTable", this);

    if ( ! pContainer )
    {
        kdError() << k_funcinfo << " Couldn't get a container widget for the given menu name popupQueryResultTable" << endl;
        return;
    }

    if ( ! pContainer->inherits("KPopupMenu") )
    {
        kdError() << k_funcinfo << " Wrong container widget" << endl;
        return;
    }

    KPopupMenu * pMenu = static_cast <KPopupMenu*> (pContainer);
    pMenu->popup( pos );
}

// Display popup menu for variables table
void KPGDebugger::slotContextMenuVariablesRequested(int, int, const QPoint &pos)
{
    QWidget * pContainer = factory()->container("popupDebuggerVariablesTable", this);

    if ( ! pContainer )
    {
        kdError() << k_funcinfo << " Couldn't get a container widget for the given menu name popupQueryResultTable" << endl;
        return;
    }

    if ( ! pContainer->inherits("KPopupMenu") )
    {
        kdError() << k_funcinfo << " Wrong container widget" << endl;
        return;
    }

    KPopupMenu * pMenu = static_cast <KPopupMenu*> (pContainer);
    pMenu->popup( pos );
}

// Process notice from connection in thread
void KPGDebugger::slotProcessNoticeDebugger(const QString &strNotice)
{
	qApp->lock();
	m_pTextEditDebuggerOut->append(strNotice);
	qApp->unlock();
}

// Process notice from connection in thread
void KPGDebugger::slotProcessNoticeFunction(const QString &strNotice)
{
	qApp->lock();
	m_pTextEditResult->append(strNotice);
		
	if((m_threadDebugger.state() == KPGDebuggingThread::waitingForDebuggerPort) 
		   && strNotice.startsWith("NOTICE:") 
		   && (strNotice.contains("PLDBGBREAK:")))
	{
		QRegExp regexpNumber("PLDBGBREAK:\\d+", true);
		int iPos = regexpNumber.search(strNotice);
		if(iPos > 0)
		{
			int iLen = regexpNumber.matchedLength();
			QString strDebuggerPort(strNotice.mid(iPos + 11, iLen - 11));
			
			m_pTextEditDebuggerOut->append(i18n("Assigned debugger port: %1 \n").arg(strDebuggerPort));
						
			m_threadDebugger.setDebuggerPort(strDebuggerPort);
			m_threadDebugger.unlockMutex();
		}
	}
	else
	{
        m_pTabWidget->setCurrentPage(TAB_PAGE_RESULT_TEXT);
	}
	
	qApp->unlock();
}

// Attach given function to debugger
void KPGDebugger::attachFunction(pqxx::oid oidFunction)
{
    m_threadDebugger.setActionSetBreakpoint(oidFunction, 1);
    
    disableGui();
    m_threadDebugger.unlockMutex();
}

void KPGDebugger::executeFunction()
{
	//int iTransType = m_pComboBoxTransaction->currentItem();
	KPGConnection::ETransType transType = KPGConnection::eTransNormal;
	/*switch(iTransType)
	{
		case 0: transType = KPGConnection::eTransNormal;
		break;
				
		case 1: transType = KPGConnection::eTransRobust;
		break;
				
		case 2: transType = KPGConnection::eTransNone;
		break;
	}*/
	
    // Construct calling SQL
	QString strSql("SELECT ");
	
	if(m_bReturnSet)
	{
		strSql.append("* FROM ");
	}
	
	strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strFunctionName));
            
	strSql.append("(");
    int i = 1; // prepared arguments counter
    int j = 0; // total arguments counter
    unsigned int nRow = 0; // row in m_pTableParameters
    for(ListFunctionArguments::const_iterator cit = m_listArguments.begin(); cit != m_listArguments.end(); ++cit)
    { 
        if((*cit).mode() == KPGFunctionArgument::output)
        {
            nRow++;
            continue; // skip output arguments
        }
	
		if(j++ > 0) strSql.append(", ");
    	
        if((m_pTableParameters->text(nRow, COL_VALUE) == QString::null) || m_pTableParameters->text(nRow, COL_VALUE).isEmpty())
		{
			strSql.append("NULL");
		}
		else
		{
			strSql.append(QString("$%1").arg(i++));
		}
        nRow++;
	}
    
	strSql.append(")"); 
    
	kdDebug() << strSql << endl;
	
    if(m_listArguments.size() == 0)
	{
		m_connInThreadFunction.setSQL(strSql, transType);
		m_connInThreadFunction.setOperationType(KPGConnectionInThread::eRunSql);
	}
	else
	{
		m_connInThreadFunction.setOperationType(KPGConnectionInThread::eRunPreparedStatement);
		
		QString strStatementName("exec_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strFunctionName));
		
		
		m_connInThreadFunction.setPrepStatementName(strStatementName, transType);
	
		try
		{
			pqxx::prepare::declaration pqxxPrepDecl = m_connInThreadFunction.connection()->prepare(strStatementName, strSql);
		
			// Add declaring parameters to prepared statements
            unsigned int nRow = 0; // row in m_pTableParameters
            for(ListFunctionArguments::const_iterator cit = m_listArguments.begin(); cit != m_listArguments.end(); ++cit)
            { 
                if((*cit).mode() != KPGFunctionArgument::output)
                {
                    if((m_pTableParameters->text(nRow, COL_VALUE) != QString::null) && m_pTableParameters->text(nRow, COL_VALUE).isEmpty() == false)
				    {
                        m_connInThreadFunction.connection()->setPrepareDeclaration(pqxxPrepDecl, (*cit).typName());
				    }
                }
                nRow++;
			}
		}
		catch(const std::exception &e)
		{
			m_pTextEditResult->append(e.what());
            m_pTabWidget->setCurrentPage(TAB_PAGE_RESULT_TABLE);
	
			return;
		} 
		
		//***********************
        ListFunctionArgumentsWithValues listArgumentsWithValues;
				
        unsigned int nRow = 0;
        for(ListFunctionArguments::const_iterator cit = m_listArguments.begin(); cit != m_listArguments.end(); ++cit)
		{ 
            if((*cit).mode() == KPGFunctionArgument::output)
            {
                nRow++;
                continue;
            }
            
            if((m_pTableParameters->text(nRow, COL_VALUE) == QString::null) 
                || m_pTableParameters->text(nRow, COL_VALUE).isEmpty()) 
            {
                nRow++;
                continue;
            }
						
            listArgumentsWithValues.append(KPGFunctionArgumentWithValue((*cit).sequence(), 
                (*cit).name(), 
                (*cit).mode(), 
                (*cit).typName(), 
                m_pTableParameters->text(nRow++, COL_VALUE)));
		}
		
		m_connInThreadFunction.setPrepStatementArgumentValues(listArgumentsWithValues); 
	}
	
	m_connInThreadFunction.setCustomEventType(0);
	m_connInThreadFunction.start();
	return;
}

void KPGDebugger::customEvent(QCustomEvent *pEvent)
{
    if(pEvent->type() == DEBUGGER_EVENT_INIT)
	{
		m_threadDebugger.lockMutex(); // lock mutex to force debugger wait to debugger port
        KPGDebugEventInit *pEvent1 = static_cast <KPGDebugEventInit *> (pEvent);
		
		m_connInThreadFunction.setConnection(pEvent1->connectionFunction());
		
		KPGNoticer * pPgNoticer = m_connInThreadFunction.connection()->kpgNoticer();
        connect(pPgNoticer, SIGNAL(sigNotice(const QString &)), this, SLOT(slotProcessNoticeFunction(const QString &)));
		
		// Executing function cause NOTICE PLDBGBREAK, containing debugger port. Then, mutex will be unlocked
		executeFunction();
		return;
	}
	else if(pEvent->type() == DEBUGGER_EVENT_EXECUTION_PAUSED)
	{
		m_threadDebugger.lockMutex();
		qApp->lock();
        KPGDebugEventExecutionPaused *pEventExecutionPaused = static_cast <KPGDebugEventExecutionPaused *> (pEvent);
		
        bool bHighlightChangedVariables = true;
        if(m_oidFunctionCurrent != pEventExecutionPaused->oidFunctionCurrent())
		{
            /**
              * Current function changed
              */
            KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
            
            // Save marks for old current function
            if(m_oidFunctionCurrent != 0)
            {
                KPGFunctionDebugInfo & functionDebugInfo = m_threadDebugger.getFunctionDebugInfo(m_oidFunctionCurrent);
                functionDebugInfo.clearBookmarks();
                
                QPtrList<KTextEditor::Mark> listOfMarks = pMarkIface->marks();
                KTextEditor::Mark *pMark = 0;
                for(pMark = listOfMarks.first(); pMark; pMark = listOfMarks.next())
                {
                    if(!(pMark->type & KTextEditor::MarkInterface::Bookmark)) continue;
            
                    functionDebugInfo.addBookmark(pMark->line);
                }
            
                // Clear marks in KatePart editor
                pMarkIface->clearMarks();
            }
            
            
            // Set new function OID as current
            pqxx::oid oidFunctionCurrentOld = m_oidFunctionCurrent;
            m_oidFunctionCurrent = pEventExecutionPaused->oidFunctionCurrent();
            setEditorText(pEventExecutionPaused->functionDebugInfo().sourceCode());
            
            if(oidFunctionCurrentOld != 0)
            {
                // Restore bookmarks for new current function
                const KPGFunctionDebugInfo & functionDebugInfoNew = pEventExecutionPaused->functionDebugInfo();
                QValueList<unsigned int>::ConstIterator it;
                for(it = functionDebugInfoNew.listBookmarks().begin(); it != functionDebugInfoNew.listBookmarks().end(); ++it) 
                {
                    unsigned int uiLineNumber = (*it); // 0. based
                    pMarkIface->addMark(uiLineNumber, KTextEditor::MarkInterface::Bookmark);
                }
            }
            
            // When current function changed, don't highlight changes
            bHighlightChangedVariables = false;
            m_listVariables.clear();
		}
		
        m_uiLineCurrent = pEventExecutionPaused->lineCurrent();
        
        bool bAnyVariableChanged = displayVariables(pEventExecutionPaused->variables(), bHighlightChangedVariables);
        displayStack(pEventExecutionPaused->stack());
        displayBreakpoints(pEventExecutionPaused->breakpoints());
				
        viewCursorInterface(m_pKateView)->setCursorPosition(m_uiLineCurrent, 0);
        
        if(bAnyVariableChanged)
        {
            m_pTabWidget->setCurrentPage(TAB_PAGE_VARIABLES_TABLE);
        }
        
		qApp->unlock();
		
		enableGui();
		return;
	}
	else if(pEvent->type() == DEBUGGER_EVENT_MESSAGE)
	{
        KPGDebugEventMessage *pMessageEvent = static_cast <KPGDebugEventMessage *> (pEvent);
		
		qApp->lock();
		m_pTextEditDebuggerOut->append(pMessageEvent->message());
		qApp->unlock();
		return;
	}
	else if(pEvent->type() == DEBUGGER_EVENT_ERROR)
	{
        KPGDebugEventError *pMessageEvent = static_cast <KPGDebugEventError *> (pEvent);
		
		qApp->lock();
		m_pTextEditDebuggerOut->append(pMessageEvent->error() + "\n");
        m_pTabWidget->setCurrentPage(TAB_PAGE_DEBUGGER_TEXT);
		
		KMessageBox::error(this, pMessageEvent->error());
		
		qApp->unlock();
		
		/*if(pMessageEvent->inAction()) // When error occured in action, lock mutex
		{
			m_threadDebugger.lockMutex();
		}*/
        
        if(m_threadDebugger.state() == KPGDebuggingThread::failed)
        {
            m_threadDebugger.wait();
            enableGui();
        }
        
        return;
	}
	else if(pEvent->type() == DEBUGGER_EVENT_BREAKPOINT)
	{
        KPGDebugEventBreakpoint *pBreakpointEvent = static_cast <KPGDebugEventBreakpoint *> (pEvent);
		
		qApp->lock();
		
		KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
		
        bool bBookmarked = pMarkIface->mark(pBreakpointEvent->lineNumber()) & KTextEditor::MarkInterface::Bookmark;
        pMarkIface->clearMark(pBreakpointEvent->lineNumber());
                
		// If breakpoint is set to currently debugged function, show/disable mark
		if(pBreakpointEvent->oidFunction() == m_oidFunctionCurrent)
		{
            switch(pBreakpointEvent->type())
			{
                case KPGDebugEventBreakpoint::settingOk:
                {    
                    if(pBreakpointEvent->lineNumber() == m_uiLineCurrent)
                    {
                        pMarkIface->setMark(pBreakpointEvent->lineNumber(), KTextEditor::MarkInterface::BreakpointReached);
                    }
                    else
                    {
                        pMarkIface->setMark(pBreakpointEvent->lineNumber(), KTextEditor::MarkInterface::BreakpointActive);
                    }
                }    
                break;
                    
			    case KPGDebugEventBreakpoint::droppingOk:
                {     
                    if(pBreakpointEvent->lineNumber() == m_uiLineCurrent)
                    {
                        pMarkIface->setMark(pBreakpointEvent->lineNumber(), KTextEditor::MarkInterface::Execution);
                    }
                }
                break;
                    
                case KPGDebugEventBreakpoint::settingFailed:
                {
                    QString message(i18n("Seting breakpoint failed"));
                
                    m_pTextEditDebuggerOut->append(message);
                    m_pTabWidget->setCurrentPage(TAB_PAGE_DEBUGGER_TEXT);
                
                    KMessageBox::sorry(this, message);
                }
                break;
                    
                case KPGDebugEventBreakpoint::droppingFailed:
                {
                    QString message(i18n("Dropping breakpoint failed"));
                
                    m_pTextEditDebuggerOut->append(message);
                    m_pTabWidget->setCurrentPage(TAB_PAGE_DEBUGGER_TEXT);
                
                    KMessageBox::sorry(this, message);
                }
                break;
            }
            
            if(bBookmarked)
            {
                pMarkIface->addMark(pBreakpointEvent->lineNumber(), KTextEditor::MarkInterface::Bookmark);
            }
		}
		
		qApp->unlock();
		enableGui();
		m_threadDebugger.lockMutex();
		return;
	}
    else if(pEvent->type() == DEBUGGER_EVENT_DEPOSIT_VALUE)
    {
        KPGDebugEventDepositValue *pDepositEvent = static_cast <KPGDebugEventDepositValue *> (pEvent);
                
        if(pDepositEvent->type() == KPGDebugEventDepositValue::depositingFailed)
        {
            qApp->lock();
        
            QString message(i18n("Depositing variable %1 / %2 failed").arg(pDepositEvent->variableName().arg(pDepositEvent->lineNumber())));
            
            m_pTextEditDebuggerOut->append(message);
            m_pTabWidget->setCurrentPage(TAB_PAGE_DEBUGGER_TEXT);
        
            KMessageBox::sorry(this, message);
        
            qApp->unlock();
        }
        else
        {
            qApp->lock();
            displayVariables(pDepositEvent->variables(), true);
            qApp->unlock();
        }
        enableGui();
        m_threadDebugger.lockMutex();
        return;
    }
		
	/**
	 * If we are here, the event was posted from m_connInThreadFunction
     * Debugged function is finished.
	 */
	m_connInThreadFunction.wait();
	
    m_connInThreadFunction.unprepareStatement();
    
	KPGQueryResultEvent *pQueryResultEvent = static_cast <KPGQueryResultEvent *> (pEvent);
	
	if(pQueryResultEvent->error().isEmpty() == false)
	{
		m_pTextEditResult->append(pQueryResultEvent->error());
	}
	
	KPGNoticer * pPgNoticerFunction = m_connInThreadFunction.connection()->kpgNoticer();
    disconnect(pPgNoticerFunction, SIGNAL(sigNotice(const QString &)), this, SLOT(slotProcessNoticeFunction(const QString &)));
			
	// Display result
	if(m_pqxxResultFunction.size() > 0)
	{
		displayResult(); 
        m_pTabWidget->setCurrentPage(TAB_PAGE_RESULT_TABLE);
	}
	else
	{
        m_pTabWidget->setCurrentPage(TAB_PAGE_RESULT_TEXT);
    }
		
    // This cause exception in debugger thread (pqxx::broken_connection), and it's exit
    m_connInThreadFunction.disconnectFromServer();
    
    // Exit debugger
    m_threadDebugger.setAction(KPGDebuggingThread::actionExit);
    if(pQueryResultEvent->error().isEmpty() == false)
    {
        // If function fails, debugger waits for debugger port - exit it
        m_pTextEditResult->append(i18n("\nDebugged function finished with error !"));
        m_threadDebugger.unlockMutex();
    }
    else
    {
        m_pTextEditResult->append(i18n("\nDebugged function finished"));
    }
    m_threadDebugger.wait();
    		
	m_pTableVariables->setNumRows(0);
	m_pTableStack->setNumRows(0);
    displayBreakpoints();
	m_oidFunctionCurrent = 0;
    m_pActRun->setText(i18n("Start in debugger"));
    m_listVariables.clear();
    
	enableGui();
    
    if(m_threadDebugger.mutexLockCount() != 0)
    {
        kdError() << k_funcinfo << endl << "Logic error, mutex lock/unlock inconsistency:" << m_threadDebugger.mutexLockCount() << endl;
    }
}

// Display SQL result
void KPGDebugger::displayResult()
{
	// Fill output area with query result
	m_nTotalRows = m_pqxxResultFunction.size();
	m_nTotalCols = m_pqxxResultFunction.columns();
	m_nFetchedRows = 0;
	bool bAllRowsFetched = true;
		
	m_pTableResult->setNumRows(min(m_nTotalRows, NUM_OF_RECORDS_PER_PAGE));
	m_pTableResult->setNumCols(m_nTotalCols);
	
	// Field names
	QHeader* hHeader = m_pTableResult->horizontalHeader();
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		hHeader->setLabel(nCol, m_pqxxResultFunction.column_name(nCol));
	}
	
	// Data
	for(unsigned int nRow = 0; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_connInThreadFunction.connection()->textCodec()->toUnicode(m_pqxxResultFunction[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		if(nRow + 1 >= NUM_OF_RECORDS_PER_PAGE)
		{
			bAllRowsFetched = false;
			break;
		}
	}
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		m_pTableResult->adjustColumn(nCol);
	}
	
	if(bAllRowsFetched) m_pqxxResultFunction.clear(); // free memory
}

// Fetch next X rows from large result
void KPGDebugger::fetchNext()
{
    unsigned int nRows = m_nTotalRows - m_nFetchedRows;
    nRows = min(nRows, NUM_OF_RECORDS_PER_PAGE);
    m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
    
    // Data
    unsigned int nRow = m_nFetchedRows;
    unsigned int nFetchedRows = 0;
    
    for(; nRow < m_nTotalRows; nRow++)
    {
        for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
        {
            QString strValue(m_connInThreadFunction.connection()->textCodec()->toUnicode(m_pqxxResultFunction[nRow][nCol].c_str()));
            m_pTableResult->setText(nRow, nCol, strValue);
        }
        
        m_nFetchedRows++;
        nFetchedRows++;
        if(nFetchedRows >= NUM_OF_RECORDS_PER_PAGE)
        {
            m_pActFetchNext->setEnabled(true); 
            m_pActFetchAll->setEnabled(true);
            break;
        }
    }
    
    if(nRow == m_nTotalRows)
    {
        m_pActFetchNext->setEnabled(false); 
        m_pActFetchAll->setEnabled(false);
        m_pqxxResultFunction.clear(); // free memory
    }
}
  
// Fetch all rows from large result  
void KPGDebugger::fetchAll()
{
    unsigned int nRows = m_nTotalRows - m_nFetchedRows;
    
    if(nRows > 10000)
    { 
        if( KMessageBox::questionYesNo(this, QString("There are %1 rows in result. Fething them may take long time and much anmount of memory. Continue ?").arg(nRows)) != KMessageBox::Yes )
        {
            return;
        }
    }
    
    ((QWidget *) parent())->setCursor(KCursor::waitCursor());
    m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
    
    // Data
    unsigned int nRow = m_nFetchedRows;
    for(; nRow < m_nTotalRows; nRow++)
    {
        for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
        {
            QString strValue(m_connInThreadFunction.connection()->textCodec()->toUnicode(m_pqxxResultFunction[nRow][nCol].c_str()));
            m_pTableResult->setText(nRow, nCol, strValue);
        }
        
        m_nFetchedRows++;
    }
    
    ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
    
    m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
        
    m_pqxxResultFunction.clear(); // free memory
}  

// Display variables
bool KPGDebugger::displayVariables(const KPGVariableList &listVariables, bool bHighlightChangedVariables)
{
	uint uiRows = listVariables.size();
    m_pTableVariables->setNumRows(0); // clear table, to discard old pixmapVariableChanged
    m_pTableVariables->setNumRows(uiRows);
	bool bAnyVariableChanged = false;
    
	// Data
	int nRow = 0;
	for(KPGVariableList::const_iterator cit = listVariables.begin(); cit != listVariables.end(); ++cit)
	{
        bool bChanged = false;
        if(bHighlightChangedVariables && (listVariables.size() == m_listVariables.size()))
        {
            const KPGVariable & variable = m_listVariables[nRow];
            if(((*cit).name() == variable.name()) && ((*cit).lineNumber() == variable.lineNumber()))
            {
                if((*cit).value() != variable.value())
                {
                    bChanged = true;
                    bAnyVariableChanged = true;
                }  
            }
        }
        
        m_pTableVariables->setText(nRow, 0, (*cit).name());
        
        QPixmap *pTypePixmap = KPGTableColumn::getPixmapForType((*cit).typName());
        m_pTableVariables->setPixmap(nRow, 0, *pTypePixmap);
        
        m_pTableVariables->setText(nRow, 1, (*cit).typName());
        m_pTableVariables->setText(nRow, COL_VARIABLES_VALUE, (*cit).value());
        
        m_pTableVariables->setPixmap(nRow, 3, (*cit).isUnique() ? *KPGUtil::m_pIconTrue : *KPGUtil::m_pIconFalse);
        m_pTableVariables->setPixmap(nRow, 4, (*cit).isConst() ? *KPGUtil::m_pIconTrue : *KPGUtil::m_pIconFalse);
        m_pTableVariables->setPixmap(nRow, 5, (*cit).isNotNull() ? *KPGUtil::m_pIconTrue : *KPGUtil::m_pIconFalse);
        
        m_pTableVariables->setText(nRow, 6, QString("%1").arg((*cit).lineNumber()));
		
        if(bChanged)
        {
            m_pTableVariables->setPixmap(nRow, COL_VARIABLES_VALUE, pixmapVariableChanged);
        }
        
		nRow++;
	}
	
	for(uint nCol = 0; nCol < 5; nCol++)
	{
		m_pTableVariables->adjustColumn(nCol);
	}
    
    // Store variables, to be able detect changed ones next time
    m_listVariables = listVariables;
    return bAnyVariableChanged;
}

// Display stack
void KPGDebugger::displayStack(const KPGStackList &listStack)
{
	uint uiRows = listStack.size();
	m_pTableStack->setNumRows(uiRows);
	
	// Data
	int nRow = 0;
	for(KPGStackList::const_iterator cit = listStack.begin(); cit != listStack.end(); ++cit)
	{
		m_pTableStack->setText(nRow, 0, (*cit).targetName());
		m_pTableStack->setText(nRow, 1, QString("%1").arg((*cit).lineNumber()));
		m_pTableStack->setText(nRow, 2, (*cit).arguments());
        m_pTableStack->setText(nRow, 3, QString("%1").arg((*cit).level()));
		nRow++;
	}
	
	for(uint nCol = 0; nCol < 4; nCol++)
	{
		m_pTableStack->adjustColumn(nCol);
	}
}
	
// Display active/reached/execution breakpoints marks, obtained from debugger frontend
void KPGDebugger::displayBreakpoints(const KPGBreakpointList &listBreakpoints)
{
	clearBreakpointMarks();
	
	KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
	
    bool bMarkReachedIsSet = false;
    
	// Data
	for(KPGBreakpointList::const_iterator cit = listBreakpoints.begin(); cit != listBreakpoints.end(); ++cit)
	{
		uint uiLineNumber = (*cit).lineNumber();
		pqxx::oid oidFunction = (*cit).oidFunction();
		
		if(oidFunction == m_oidFunctionCurrent)
		{
			if(uiLineNumber == m_uiLineCurrent)
			{
				pMarkIface->addMark(uiLineNumber, KTextEditor::MarkInterface::BreakpointReached);
                bMarkReachedIsSet = true;
			}
			else
			{
				pMarkIface->addMark(uiLineNumber, KTextEditor::MarkInterface::BreakpointActive);
			}
		}
	}
    
    if(!bMarkReachedIsSet)
    {
        pMarkIface->addMark(m_uiLineCurrent, KTextEditor::MarkInterface::Execution);
    }
}

// Display disabled breakpoints marks, stored in MapFunctionDebugInfo
void KPGDebugger::displayBreakpoints()
{
    clearBreakpointMarks();
    
    KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());

    const KPGFunctionDebugInfo & functionDebugInfo = m_threadDebugger.mapFunctionDebugInfo()[m_oidFunction];
    
    QValueList<unsigned int>::ConstIterator it;
    for(it = functionDebugInfo.listBreakpoints().begin(); it != functionDebugInfo.listBreakpoints().end(); ++it) 
    {
        unsigned int uiLineNumber = (*it); // 0. based
        pMarkIface->addMark(uiLineNumber, KTextEditor::MarkInterface::BreakpointDisabled);
    }
}

// Find first occurence of text in result table
void KPGDebugger::findInResultFirst(QStringList &listOfSearchHistory)
{
    KFindDialog dlg(this, "", 0, listOfSearchHistory, false);
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
        
    
    if(m_pFind != 0) 
    {
        disconnect( m_pFind, 0, this, 0);
        delete m_pFind;
    }
    m_pFind = new KFind(dlg.pattern(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pFind, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pFind, SIGNAL( findNext() ), this, SLOT( slotFindInResultNext() ) );

    m_iRowToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numRows() - 1 : 0;
    m_iColToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numCols() - 1 : 0;
    m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));

    slotFindInResultNext();
}

// Find next occurence of text in result table
void KPGDebugger::findInResultNext()
{
    m_iRowToSearch = m_pTableResult->currentRow();
    m_iColToSearch = m_pTableResult->currentColumn();
    if(m_iRowToSearch < 0) return;
    if(m_iColToSearch < 0) return;
    
    slotFindInResultNext();
}

// Find next occurence of text
void KPGDebugger::slotFindInResultNext()
{
    if (!m_pFind) // shouldn't be called before find is activated
        return;

    KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
            ((m_pFind->options() & KFindDialog::FindBackwards) ? 
            (m_iRowToSearch >= 0) : (m_iRowToSearch < m_pTableResult->numRows())
            ) 
          ) 
    {
            //kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
            
        Q_ASSERT(m_iRowToSearch >= 0);
        Q_ASSERT(m_iRowToSearch < m_pTableResult->numRows());
        Q_ASSERT(m_iColToSearch >= 0);
        Q_ASSERT(m_iColToSearch < m_pTableResult->numCols());
                
        if(m_pFind->needData()) 
        {
            m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));
        }
        
            // Let KFind inspect the text fragment, and display a dialog if a match is found
        res = m_pFind->find();
        
        if( res == KFind::NoMatch ) 
        {
            if((m_pFind->options() & KFindDialog::FindBackwards))
            {
                if(m_iColToSearch == 0)
                {
                    m_iColToSearch = m_pTableResult->numCols() - 1;
                    m_iRowToSearch--;
                }
                else
                {
                    m_iColToSearch--;
                }
            }
            else
            {
                if(m_iColToSearch == m_pTableResult->numCols() - 1)
                {
                    m_iColToSearch = 0;
                    m_iRowToSearch++;
                }
                else
                {
                    m_iColToSearch++;
                }
            }
        }
            //kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
    }

    if( res == KFind::NoMatch ) 
    { 
        // i.e. at end
        m_pFind->closeFindNextDialog();
        m_pFind->displayFinalDialog();
        m_pFind->resetCounts();
        m_pTableResult->removeSelection(0);
        m_pActFindNextInResult->setEnabled(false);
    }
}

// Highligth found text
void KPGDebugger::slotHighlight(const QString &, int, int)
{
    //kdDebug() << "highlight: " << index << " " << length << endl;
    m_pTableResult->setCurrentCell(m_iRowToSearch, m_iColToSearch);
    m_pActFindNextInResult->setEnabled(true);
}

void KPGDebugger::virtual_hook(int id, void* data)
{
    KXMLGUIClient::virtual_hook(id, data); 
}
