//
// C++ Interface: kpgdebuggingthread
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef KPGDEBUGGINGTHREAD_H
#define KPGDEBUGGINGTHREAD_H

// include files for Qt
#include <qstringlist.h>
#include <qthread.h>
#include <qmutex.h>
#include <qevent.h>

#include "../DbObjects/kpgconnection.h"
#include "kpgfunctiondebuginfo.h"

#define DEBUGGER_EVENT_INIT 		         65435
#define DEBUGGER_EVENT_EXECUTION_PAUSED 	 65436
#define DEBUGGER_EVENT_ERROR 	             65437
#define DEBUGGER_EVENT_MESSAGE               65438
#define DEBUGGER_EVENT_BREAKPOINT            65439
#define DEBUGGER_EVENT_DEPOSIT_VALUE         65440

/**
 * Event, that contain connection for debugged function
 *
 * @author Lumir Vanek
 */
class KPGDebugEventInit : public QCustomEvent
{
public:
    KPGDebugEventInit(KPGConnection *pConnectionFunction)
    : QCustomEvent(DEBUGGER_EVENT_INIT)
	{
		m_pConnectionFunction = pConnectionFunction;
	}
     
	// Return connection for debugged function
	KPGConnection * connectionFunction() const { return m_pConnectionFunction; }		
			
private:
	
	// Connection for debugged function
	KPGConnection *m_pConnectionFunction;
};

/**
 * Variable info for debugger
 *
 *	 @author Lumir Vanek <lvanek@users.sourceforge.net>
 */
class KPGVariable
{
public:
	KPGVariable() {}
				
	KPGVariable(const pqxx::result::tuple &pqxxTuple, const KPGConnection *pConnection)
	{
		m_strName = pConnection->toUnicode(pqxxTuple["name"].c_str());
		m_strVarClass = pqxxTuple["varclass"].c_str();
		m_strValue = pConnection->toUnicode(pqxxTuple["value"].c_str());
        m_strTypName = pConnection->toUnicode(pqxxTuple["typname"].c_str());
        pqxxTuple["vartypid"].to(m_oidType);
        pqxxTuple["isUnique"].to(m_bIsUnique);
        pqxxTuple["isConst"].to(m_bIsConst);
        pqxxTuple["isNotNull"].to(m_bIsNotNull);
        pqxxTuple["linenumber"].to(m_uiLineNumber);
	}
	
	// Copy constructor
	KPGVariable(const KPGVariable &src)
	{
		m_strName = src.name();
		m_strVarClass = src.varClass();
		m_strValue = src.value();
        m_strTypName = src.typName();
        m_oidType = src.oidType();
        m_bIsUnique = src.isUnique();
        m_bIsConst = src.isConst();
        m_bIsNotNull = src.isNotNull();
        m_uiLineNumber = src.lineNumber();
	}
	
	const QString & name() const { return m_strName; }
	const QString & varClass() const { return m_strVarClass; }
	const QString & value() const { return m_strValue; }
    const QString & typName() const { return m_strTypName; }
    pqxx::oid oidType() const { return m_oidType; }
    bool isUnique() const { return m_bIsUnique; }
    bool isConst() const { return m_bIsConst; }
    bool isNotNull() const { return m_bIsNotNull; }
    
    // Returns 1. based line number
    unsigned int lineNumber() const { return m_uiLineNumber; }
    
    	
protected:
	QString m_strName;
	QString m_strVarClass;
	QString m_strValue;
    QString m_strTypName;
    pqxx::oid m_oidType;
    bool m_bIsUnique;
    bool m_bIsConst;
    bool m_bIsNotNull;
    uint m_uiLineNumber;
};
typedef QValueList<KPGVariable> KPGVariableList;

/**
 * Stack info for debugger
 *
 *	 @author Lumir Vanek <lvanek@users.sourceforge.net>
 */
class KPGStackItem
{
public:
	KPGStackItem() {}
				
	KPGStackItem(const pqxx::result::tuple &pqxxTuple, const KPGConnection *pConnection)
	{
        pqxxTuple["func"].to(m_oidFunction);
        pqxxTuple["level"].to(m_iLevel);
		pqxxTuple["linenumber"].to(m_uiLineNumber);
		m_strTargetName = pConnection->toUnicode(pqxxTuple["targetname"].c_str());
		m_strArguments = pqxxTuple["args"].c_str();
	}
    
    // Copy constructor
    KPGStackItem(const KPGStackItem &src)
    {
        m_oidFunction = src.oidFunction();
        m_iLevel = src.lineNumber();
        m_uiLineNumber = src.lineNumber();
        m_strTargetName = src.targetName();
        m_strArguments = src.arguments();
    }
	
    unsigned int level() const { return m_iLevel; }
    pqxx::oid oidFunction() const { return m_oidFunction; }
	
    // Returns 1. based line number
    unsigned int lineNumber() const { return m_uiLineNumber; }
	const QString & targetName() const { return m_strTargetName; }
	const QString & arguments() const { return m_strArguments; }
	
protected:
    pqxx::oid m_oidFunction;
    int m_iLevel;
	uint m_uiLineNumber;
	QString m_strTargetName;
	QString m_strArguments;
};
typedef QValueList<KPGStackItem> KPGStackList;

/**
 * Breakpoint info for debugger
 *
 *	 @author Lumir Vanek <lvanek@users.sourceforge.net>
 */
class KPGBreakpoint
{
public:
	KPGBreakpoint() {}
				
	KPGBreakpoint(const pqxx::result::tuple &pqxxTuple, const KPGConnection *pConnection)
	{
		pqxxTuple["linenumber"].to(m_uiLineNumber);
        m_uiLineNumber--;
		pqxxTuple["func"].to(m_oidFunction);
		m_strTargetName = pConnection->toUnicode(pqxxTuple["targetname"].c_str());
	}
    
    // Copy constructor
    KPGBreakpoint(const KPGBreakpoint &src)
    {
        m_uiLineNumber = src.lineNumber();
        m_oidFunction = src.oidFunction();
        m_strTargetName = src.targetName();
    }
	
    // Returns 0. based line number, where breakpoint is set
	unsigned int lineNumber() const { return m_uiLineNumber; }
	pqxx::oid oidFunction() const { return m_oidFunction; }
	const QString & targetName() const { return m_strTargetName; }
	
protected:
    // 0. based line number, where breakpoint is set
	uint m_uiLineNumber;
	pqxx::oid m_oidFunction;
	QString m_strTargetName;
};
typedef QValueList<KPGBreakpoint> KPGBreakpointList;

/**
 * Event, that contain debugger operation result
 *
 * This event is sent, when breakpoint is reached, or debugger stops 
 * after step into/over operation
 *
 * @author Lumir Vanek
 */
class KPGDebugEventExecutionPaused : public QCustomEvent
{
public:
    KPGDebugEventExecutionPaused(pqxx::oid oidFunctionCurrent, 
        uint uiLineCurrent,
		const KPGFunctionDebugInfo &functionDebugInfo,
  		const KPGVariableList &listVariables,
		const KPGStackList &listStack,
		const KPGBreakpointList &listBreakpoints)
    : QCustomEvent(DEBUGGER_EVENT_EXECUTION_PAUSED)
	{
		m_oidFunctionCurrent = oidFunctionCurrent;
		m_uiLineCurrent = uiLineCurrent;
		m_functionDebugInfo = functionDebugInfo;
		m_listVariables = listVariables;
		m_listStack = listStack;
		m_listBreakpoints = listBreakpoints;
	}
     
	// Return PostgreSQL row identifier of currently debugged function
	pqxx::oid oidFunctionCurrent() const { return m_oidFunctionCurrent; }
	
	// Return line number of current line (0. based)
    uint lineCurrent() const { return m_uiLineCurrent; }
			
	// Informations about function
	const KPGFunctionDebugInfo & functionDebugInfo() const { return m_functionDebugInfo; }
	
	// Returns list of variables
	const KPGVariableList &variables() const { return m_listVariables; }
	
	// Returns list of stack items
	const KPGStackList &stack() const { return m_listStack; }
	
	// List of breakpoints
	const KPGBreakpointList breakpoints() const { return  m_listBreakpoints; }
	
protected:
	
	// PostgreSQL row identifier of currently debugged function
	pqxx::oid m_oidFunctionCurrent;
	
	// Line number of current line (0. based)
    uint m_uiLineCurrent;
	
	// Informations about current function
	KPGFunctionDebugInfo m_functionDebugInfo;
	
	// List of variables
	KPGVariableList m_listVariables;
	
	// List of stack items
	KPGStackList m_listStack;
	
	// List of breakpoints
	KPGBreakpointList m_listBreakpoints;
};

/**
 * Event, that contain debugger operation result
 *
 * @author Lumir Vanek
 */
class KPGDebugEventError : public QCustomEvent
{
public:
    KPGDebugEventError(const QString & strError)
	: QCustomEvent(DEBUGGER_EVENT_ERROR)
	{
		m_strError = strError;
	}
     
	const QString & error() const { return m_strError; }
		
private:
	
	QString m_strError;
};

/**
 * Event, that contain debugger operation result
 *
 * @author Lumir Vanek
 */
class KPGDebugEventMessage : public QCustomEvent
{
public:
    KPGDebugEventMessage(const QString & strMessage)
	: QCustomEvent(DEBUGGER_EVENT_MESSAGE)
		{
			m_strMessage = strMessage;
		}
     
		const QString & message() const { return m_strMessage; }
		
	private:
	
		QString m_strMessage;
};

/**
 * Event, that contain info about set/drop breakpoint
 *
 * @author Lumir Vanek
 */
class KPGDebugEventBreakpoint : public QCustomEvent
{
public:
    
    // Event types
    enum EType 
    {   
        settingOk = 0,
        settingFailed,
        droppingOk,
        droppingFailed
    };
    
    KPGDebugEventBreakpoint(pqxx::oid oidFunction, uint uiLineNumber, EType eType)
	: QCustomEvent(DEBUGGER_EVENT_BREAKPOINT)
	{
		m_oidFunction = oidFunction;
		m_uiLineNumber = uiLineNumber;
        m_eType = eType;
	}
     
    uint lineNumber() const { return m_uiLineNumber; }
	pqxx::oid oidFunction() const { return m_oidFunction; }
    EType type() const { return m_eType; }
		
private:
	
	// PostgreSQL row identifier of function, where breakpoint is set/dropped
	pqxx::oid m_oidFunction;
	
	// Line number (0.-based) 
	uint m_uiLineNumber;
	
	// Event type
    EType m_eType; 
};

/**
 * Event, that contain info about deposit value
 *
 * @author Lumir Vanek
 */
class KPGDebugEventDepositValue : public QCustomEvent
{
public:
    
    // Event types
    enum EType 
    {   
        depositingOk = 0,
        depositingFailed
    };
    
    KPGDebugEventDepositValue(const QString& strVariableName, 
        uint uiLineNumber, 
        EType eType,
        const KPGVariableList &listVariables)
    : QCustomEvent(DEBUGGER_EVENT_DEPOSIT_VALUE)
    {
        m_strVariableName = strVariableName;
        m_uiLineNumber = uiLineNumber;
        m_eType = eType;
        m_listVariables = listVariables;
    }
     
    uint lineNumber() const { return m_uiLineNumber; }
    const QString& variableName() const { return m_strVariableName; }
    EType type() const { return m_eType; }
    
    // Returns list of variables
    const KPGVariableList &variables() const { return m_listVariables; }
        
private:
    
    // Name of the variable
    QString m_strVariableName;
    
    // Line number (0.-based) 
    uint m_uiLineNumber;
    
    // Event type
    EType m_eType; 
    
    // List of variables
    KPGVariableList m_listVariables;
};

/**
  * Debugging thread. It is state machine, that communicate with server side
  * debugger using own connection.
  *
  *	@author Lumir Vanek <lvanek@users.sourceforge.net>
  */
class KPGDebuggingThread : public QThread
{
public:
	
	// Debugger states
	enum EState 
	{ 	
		prepared = 0,
  		waitingForDebuggerPort,
		attached,
		failed
	};
	
	// Debugger actions
	enum EAction 
	{ 	
		actionNone = 0,
  		actionStepOver,
		actionStepInto,
  		actionContinue,
		actionSetBreakpoint,
  		actionDropBreakpoint,
        actionDepositValue,
		actionAbortTarget,
        actionExit
	};
	
    KPGDebuggingThread();
    ~KPGDebuggingThread();
	
	// Set the parameters for degugging
	void setParameters(QObject *pEventReceiver, 
					const PGSTD::string& strConnectionOptions, 
					pqxx::oid oidFunction) 
	{ 
		m_pEventReceiver = pEventReceiver;
		m_strConnectionOptions = strConnectionOptions; 
		m_oidFunction = oidFunction;
	}
	
	// Return debugging connection
	KPGConnection * connectionDebugger() const { return m_pConnectionDebugger; }
	
	// Return debugged connection
	KPGConnection * connectionFunction() const { return m_pConnectionFunction; }
		
	// Lock mutex for synchronizing work with GUI thread
    void lockMutex();
    
    // Unlock mutex for synchronizing work with GUI thread
    void unlockMutex();
    
    // Return state of mutex for synchronizing work with GUI thread
    bool mutexLocked();
    
    // Return state of mutex for synchronizing work with GUI thread
    bool tryLockMutex();
    
    int mutexLockCount() const { return m_iMutexLockCount; }
	
	// Return version of debugger API
	int proxyApiVersion() const { return m_iProxyApiVersion; }
	
	// Set TCP/IP Port number for attach to remote debugger
	void setDebuggerPort(const QString & strDebuggerPort) 
	{ 
		m_strDebuggerPort = strDebuggerPort; 
	}
	
	// Return state of debugger
	EState state() const { return m_state; }
	
	// Set required debugger action
	void setAction(EAction action) { m_action = action; }
	
	// Set required debugger action: Set breakpoint
	void setActionSetBreakpoint(pqxx::oid oidFunction, uint uiLine)
	{
		m_action = actionSetBreakpoint;
		m_oidBreakpointFunction = oidFunction;
		m_uiBreakpointLine = uiLine;
	}

	// Set required debugger action: Drop breakpoint
	void setActionDropBreakpoint(pqxx::oid oidFunction, uint uiLine)
	{
		m_action = actionDropBreakpoint;
		m_oidBreakpointFunction = oidFunction;
		m_uiBreakpointLine = uiLine;
	}
    
    // Set required debugger action: Deposit value
    void setActionDepositValue(const QString& strVariableName, uint uiVariableLine, const QString& strNewValue)
    {
        m_action = actionDepositValue;
        m_strVariableName = strVariableName;
        m_strNewValue = strNewValue;
        m_uiVariableLine = uiVariableLine;        
    }
	
	// Get Map of function sources and another info, for cache it
	const MapFunctionDebugInfo & mapFunctionDebugInfo() const { return m_mapFunctionDebugInfo; }
	
    // Get info about given function
    KPGFunctionDebugInfo & getFunctionDebugInfo(pqxx::oid);
    
protected:
	
	// Do your work
	virtual void run();
		
	// Try connect to PostgreSQL server. Throws exception, if fails
	KPGConnection * connectToServer();
    
    // Close connection
	void disconnectFromServer();
	
    // Clean up resources
    void cleanup();
    
	/*
	* Functions for communicate with server-side debugger
	*/
	
	void getProxyInfo();
	
	void getOidDebug();
	
	void attachToDebugger();
	
	bool waitForBreakpoint();
	bool stepOver();
	bool stepInto();
	bool goContinue();
	void processBreakpoint(const pqxx::result &);
	void abortTarget();
    void setBreakpoint(pqxx::oid, uint, bool) throw(const KPGSqlException &);
    void dropBreakpoint(pqxx::oid, uint, bool) throw(const KPGSqlException &);
    void depositValue() throw(const KPGSqlException &);
	
	void initialize();
	    
    // Get function source code
    const QString getFunctionSourceCode(pqxx::oid);
	
	// List of variables
	const KPGVariableList getVariables();
	
	// Returns list of stack items
	const KPGStackList getStack();
	
	// Returns list of breakpoints
	const KPGBreakpointList getBreakpoints();
	
protected:
	
	// Debugger state
	EState m_state;
	
	// Debugger action
	EAction m_action;
	
	/*
	 * Mutex for synchronizing work with GUI thread. When it's locked by GUI thread,
	 * debugger is in idle state and waits for request
	 */
	QMutex m_mutex;
    
    /*
     * For debugging purposes 
     */
    int m_iMutexLockCount;
	
	// Connection options
	PGSTD::string m_strConnectionOptions;
	
	// Connection for debugging stuff
	KPGConnection *m_pConnectionDebugger;
	
	// Connection for debugged function
	KPGConnection *m_pConnectionFunction;
	
	// PostgreSQL row identifier of debugged function
	pqxx::oid m_oidFunction;
	
	// Event receiver
	QObject *m_pEventReceiver; 
	
	// Version of debugger API
	int m_iProxyApiVersion;
	
	// TCP/IP Port number for attach to remote debugger
	QString m_strDebuggerPort;
	
	// Debugger session handle
	QString m_strSessionHandle;
	
	// Map of function sources and another info, for cache it
	MapFunctionDebugInfo m_mapFunctionDebugInfo;
	
	// OID of function for requested breakpoint
	pqxx::oid m_oidBreakpointFunction;
	
	// Line number for requested breakpoint
	uint m_uiBreakpointLine;
    
    // Variable value to deposit
    QString m_strVariableName;
    
    // New variable value to deposit
    QString m_strNewValue;
    
    // Line number for deposited variable
    uint m_uiVariableLine;
};

#endif
