/***************************************************************************
                          kpgsyntaxhighlighter.cpp  -  description
                             -------------------
    begin                : led 13 2004
    copyright            : (C) 2004 by Lumir Vanek
    email                : lvanek@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kpgsyntaxhighlighter.h"

#include <iostream>

// include files for Qt
#include <qcolor.h>
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

#include <ktextedit.h>

// pl/pgSQL + SQL language keywords list
#define NUM_KEYWORDS 302
const char *arStrKeyWords[NUM_KEYWORDS] = {
"ABORT",
"ABSOLUTE",
"ACCESS",
"ACTION",
"ADD",
"AFTER",
"AGGREGATE",
"ALIAS",
"ALL",
"ALSO",
"ALTER",
"ANALYSE",
"ANALYZE",
"AND",
"ANY",
"ARRAY",
"AS",
"ASC",
"ASSERTION",
"ASSIGNMENT",
"AT",
"AUTHORIZATION",
"BACKWARD",
"BEFORE",
"BEGIN",
"BETWEEN",
"BINARY",
"BOTH",
"BY",
"CACHE",
"CALLED",
"CASCADE",
"CASE",
"CAST",
"CLASS",
"CLOSE",
"CLUSTER",
"COALESCE",
"COLLATE",
"COLUMN",
"COMMENT",
"COMMIT",
"COMMITTED",
"CONSTRAINT",
"CONSTRAINTS",
"CONVERSION",
"CONVERT",
"COPY",
"CREATE",
"CREATEDB",
"CREATEUSER",
"CROSS",
"CSV",
"CURRENT_DATE",
"CURRENT_TIME",
"CURRENT_TIMESTAMP",
"CURRENT_USER",
"CURSOR",
"CYCLE",
"DATABASE",
"DAY",
"DEALLOCATE",
"DEC",
"DECLARE",
"DEFAULT",
"DEFAULTS",
"DEFERRABLE",
"DEFERRED",
"DEFINER",
"DELETE",
"DELIMITER",
"DELIMITERS",
"DESC",
"DISTINCT",
"DO",
"DOMAIN",
"DROP",
"EACH",
"ELSE",
"ENCODING",
"ENCRYPTED",
"END",
"ESCAPE",
"EXCEPT",
"EXCLUDING",
"EXCLUSIVE",
"EXECUTE",
"EXISTS",
"EXPLAIN",
"EXTERNAL",
"EXTRACT",
"FALSE",
"FETCH",
"FIRST",
"FOR",
"FORCE",
"FOREIGN",
"FORWARD",
"FREEZE",
"FROM",
"FULL",
"FUNCTION",
"GLOBAL",
"GRANT",
"GROUP",
"HANDLER",
"HAVING",
"HOLD",
"HOUR",
"CHAIN",
"CHARACTERISTICS",
"CHECK",
"CHECKPOINT",
"ILIKE",
"IMMEDIATE",
"IMMUTABLE",
"IMPLICIT",
"IN",
"INCLUDING",
"INCREMENT",
"INDEX",
"INHERITS",
"INITIALLY",
"INNER",
"INOUT",
"INPUT",
"INSENSITIVE",
"INSERT",
"INSTEAD",
"INTERSECT",
"INTERVAL",
"INTO",
"INVOKER",
"IF",
"IS",
"ISNULL",
"ISOLATION",
"JOIN",
"KEY",
"LANCOMPILER",
"LANGUAGE",
"LARGE",
"LAST",
"LEADING",
"LEFT",
"LEVEL",
"LIKE",
"LIMIT",
"LISTEN",
"LOAD",
"LOCAL",
"LOCALTIME",
"LOCALTIMESTAMP",
"LOCATION",
"LOCK",
"LOOP",
"MATCH",
"MAXVALUE",
"MINUTE",
"MINVALUE",
"MODE",
"MONTH",
"MOVE",
"NAMES",
"NATIONAL",
"NATURAL",
"NEW",
"NEXT",
"NO",
"NOCREATEDB",
"NOCREATEUSER",
"NONE",
"NOT",
"NOTHING",
"NOTIFY",
"NOTNULL",
"NOWAIT",
"NULL",
"NULLIF",
"OBJECT",
"OF",
"OFF",
"OFFSET",
"OIDS",
"OLD",
"ON",
"ONLY",
"OPERATOR",
"OPTION",
"OR",
"ORDER",
"OUT",
"OUTER",
"OVERLAPS",
"OVERLAY",
"OWNER",
"PARTIAL",
"PASSWORD",
"PLACING",
"POSITION",
"PRECISION",
"PREPARE",
"PRESERVE",
"PRIMARY",
"PRIOR",
"PRIVILEGES",
"PROCEDURAL",
"PROCEDURE",
"QUOTE",
"RAISE",
"READ",
"REAL",
"REFERENCES",
"RECHECK",
"REINDEX",
"RELATIVE",
"RELEASE",
"RENAME",
"REPEATABLE",
"REPLACE",
"RESET",
"RESTART",
"RESTRICT",
"RETURN",
"RETURNS",
"REVOKE",
"RIGHT",
"ROLLBACK",
"ROW",
"ROWS",
"RULE",
"SAVEPOINT",
"SCROLL",
"SECOND",
"SECURITY",
"SELECT",
"SEQUENCE",
"SERIALIZABLE",
"SESSION",
"SESSION_USER",
"SET",
"SETOF",
"SHARE",
"SHOW",
"SCHEMA",
"SIMILAR",
"SIMPLE",
"SOME",
"STABLE",
"START",
"STATEMENT",
"STATISTICS",
"STDIN",
"STDOUT",
"STORAGE",
"STRICT",
"SUBSTRING",
"SYSID",
"TABLE",
"TABLESPACE",
"TEMP",
"TEMPLATE",
"TEMPORARY",
"THEN",
"TIME",
"TO",
"TOAST",
"TRAILING",
"TRANSACTION",
"TREAT",
"TRIGGER",
"TRIM",
"TRUE",
"TRUNCATE",
"TRUSTED",
"TYPE",
"UNCOMMITTED",
"UNENCRYPTED",
"UNION",
"UNIQUE",
"UNKNOWN",
"UNLISTEN",
"UNTIL",
"UPDATE",
"USAGE",
"USER",
"USING",
"VACUUM",
"VALID",
"VALIDATOR",
"VALUES",
"VERBOSE",
"VIEW",
"VOLATILE",
"WHEN",
"WHERE",
"WITH",
"WITHOUT",
"WORK",
"WRITE",
"YEAR",
"ZONE"
};

// SQL language datatypes list
#define NUM_DATATYPES 41
const char *arStrDataTypes[NUM_DATATYPES] = {
"boolean",
"bool",
"bit",
"bytea",
"varying",
"character",
"char",
"varchar",
"text",
"smallint",
"int2",
"integer",
"int",
"int4",
"bigint",
"int8",
"real",
"float4",
"double",
"precision",
"float8",
"float",
"numeric",
"decimal",
"money",
"serial",
"date",
"timestamp",
"interval",
"box",
"line",
"lseg",
"circle",
"path",
"point",
"polygon",
"cidr",
"inet",
"macadr",
"oid",
"xid" };

// SQL language operators list
#define NUM_OPERATORS 27
const char *arStrOperators[NUM_OPERATORS] = {
"+",
"-",
"/",
"*",
"%",
"^",
"|/",
"||/",
"!",
"!!",
"@",
"<",
">",
"<=",
">=",
"~",
"!~",
"~*",
"!~*",
"=",
"!=",
"<>",
"&",
"|",
"#",
"<<",
">>"
};

#define STATE_IN_COMMENT 1

KPGSyntaxHighlighter::KPGSyntaxHighlighter(QTextEdit *textEdit)
  : QSyntaxHighlighter(textEdit),
  m_bEnabled(true)
{
  m_clrDefaultText.setRgb(0, 0, 0);
  m_clrKeyWord.setRgb(128, 0, 0);
  m_clrDataType.setRgb(0, 255, 255);
  m_clrOperator.setRgb(0, 255, 0);
  m_clrQuotedString.setRgb(0, 0, 128);
  m_clrNumber.setRgb(255, 255, 0);
  m_clrComment.setRgb(128, 128, 128);
}

KPGSyntaxHighlighter::~KPGSyntaxHighlighter()
{
}


// Return QStringList with SQL keywords
const QStringList KPGSyntaxHighlighter::listOfKeyWords()
{
	QStringList strListOfKeyWords;
	for(int i = 0; i < NUM_KEYWORDS; i++)
	{
		strListOfKeyWords.append(arStrKeyWords[i]);
	}
	
	return strListOfKeyWords;
}

int KPGSyntaxHighlighter::highlightParagraph(const QString& text, int endStateOfLastPara)
{
    if(!m_bEnabled) return 0;
    
    QRegExp patternComment("/\\*[^*]*[^/]*\\*/");
    QRegExp patternCommentEnd("[^*]*[^/]*\\*/"); 
    QRegExp patternQuotedString("\"[^<\"]*\"|'[^<']*'");     	
    QRegExp patternNumber("\\b\\d+\\b", false); 
    
    //first I format the given line to default so any remaining highlighting is removed (Qt does not do it by itself)
    setFormat(0, text.length(), QColor(0, 0, 0));

    int iPos;
	unsigned int i = 0;
	
	if(endStateOfLastPara == STATE_IN_COMMENT)
	{
		for(i = 0; i < text.length(); i++)
		{
			// search end of comment
			iPos = patternCommentEnd.search(text, i);
	
			if(iPos >= 0) // end comment found ?
			{
				int l = patternCommentEnd.matchedLength();
				
				setFormat(0, l, m_clrComment);
				i += l; // skip comment
				break;
			}
			else
			{
				setFormat(0, text.length(), m_clrComment);
				return STATE_IN_COMMENT; 
			}
		}
	}
		
		
    for(; i < text.length(); i++)
    {
		if((text[i] == '-') && (i+1 < text.length()) && (text[i+1] == '-'))
		{
			// simple comment found
			setFormat(i, text.length(), m_clrComment );
			break; // skip rest of line
		}
			
		if((text[i] == '/') && (i+1 < text.length()) && (text[i+1] == '*'))
		{
			iPos=patternComment.search(text, i);

			if((unsigned int) iPos == i) // closed comment found ?
			{
				int l = patternComment.matchedLength();
					
				setFormat(iPos, l, m_clrComment);
				i += l; // skip comment
			}
			else
			{
				setFormat(i, text.length() - i, m_clrComment);
				return STATE_IN_COMMENT; 
			}
		}
      
      	if(text[i] != ' ')
      	{
			// try find keyword
			for(int j = 0; j < NUM_KEYWORDS; j++)
			{
				if(QChar(text[i]).upper() != QChar(arStrKeyWords[j][0]))
					continue; // test first character for speed up  
					
				if(QChar(text[i]).upper() > QChar(arStrKeyWords[j][0]))
					break; // test first character for speed up
				
				// For spped-up first try dumb test    
				// first letter match, test whole keyword
				bool bFound = true;
				int p = 0;
				for(unsigned int o = i; o < text.length(); o++)
				{
					if(((QChar(text[o]).upper() >= 'A') && (QChar(text[o]).upper() <= 'Z')) || (QChar(text[o]).upper() == '_'))
					{
						if(QChar(text[o]).upper() != QChar(arStrKeyWords[j][p++]))
						{
							bFound = false;
							break;
						}
					}
					else
					break;
				}
		
				if(bFound) // If dumb test success, try smart but slow test
				{
					QString strPattern("\\b");
					strPattern.append(arStrKeyWords[j]);
					strPattern.append("\\b");
					QRegExp patternKeyWord(strPattern, false); // search keyword
					iPos=patternKeyWord.search(text, i);
					if((unsigned int) iPos == i)
					{
						int l = patternKeyWord.matchedLength();
						setFormat(i, l, m_clrKeyWord );
						i += l;
						break;
					}
				}
        	}

			// try find datatype
			for(int j = 0; j < NUM_DATATYPES; j++)
			{
				if(QChar(text[i]).lower() != QChar(arStrDataTypes[j][0]).lower())
					continue; // test first character for speed up
	
				QString strPattern("\\b");  
				strPattern.append(arStrDataTypes[j]);
				strPattern.append("\\b");
				QRegExp patternDataType(strPattern, false); // search datatype
				iPos=patternDataType.search(text, i);
				if((unsigned int) iPos == i)
				{
					int l = patternDataType.matchedLength();
					setFormat(i, l, m_clrDataType );
					i += l;
					break;
				}
			}

			// try find operators
			for(int j = 0; j < NUM_OPERATORS; j++)
			{
				if(QChar(text[i]).lower() != QChar(arStrOperators[j][0]).lower())
					continue; // test first character for speed up
	
				QString strPattern("\\b");  
				strPattern.append(arStrOperators[j]);
				strPattern.append("\\b");
				QRegExp patternOperators(strPattern, false); // search datatype
				iPos=patternOperators.search(text, i);
				if((unsigned int) iPos == i)
				{
					int l = patternOperators.matchedLength();
					if(l > 0)
					{
						setFormat(i, l, m_clrOperator );
						i += l;
					}
					break;
				}
			}

			// try find quoted string
			if((text[i] == '\'') || (text[i] == '\"'))
			{
				// search quoted string
				iPos=patternQuotedString.search(text, i);
		
				if((unsigned int) iPos == i) 
				{
					int l = patternQuotedString.matchedLength();
		
					setFormat(i+1, l - 2, m_clrQuotedString);
					
					i += l;
				}
			}

			// try find numbers
			if((text[i] >= '0') && (text[i] <= '9'))
			{
				// search numbers
				iPos=patternNumber.search(text, i);
				if((unsigned int) iPos == i)
				{
					int l = patternNumber.matchedLength();
					setFormat(i, l, m_clrNumber );
					i += l;
				}
			} 
		}
    }

    return 0;
}

