/***************************************************************************
 *   Copyright (C) 2005 by Nick Bargnesi                                                                        *
 *   nbargnesi@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qimage.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdatetime.h>

#include <iostream>

void fail(const char *);
void out(const char *);

struct QRGBImage {
     QString string;
     int width;
     int height;
     bool alpha;
     QString name;
};

class ImageConverter {
    public:
        ImageConverter(const char *);
        ~ImageConverter();

        void convert( const char *, bool, bool);
    private:
        QFile *file;
        QPtrList<QRGBImage> *index;
        QTextStream stream;
};

int main(int argv, char *argc[]) {
	bool b = false, c = false;
	QString *imgFile = NULL, *hFile = NULL;

    for (int i = 0; i < argv; i++) {
		QString arg(argc[i]);
		if (arg == QString("-a") || arg == QString("--attr-only")) {
			b = true;
		} else if (arg == QString("-r") || arg == QString("--replace-underscores")) {
			c = true;
		} else if (arg.contains(".png")) {
			imgFile = new QString(arg);
		} else if (arg.contains(".h")) {
			hFile = new QString(arg);
		}
	}

	if (imgFile == NULL || hFile == NULL)
		fail("Invalid number of arguments");

	QString fileStr(*imgFile);
	QFile file(fileStr);

	if (!file.exists())
		fail("could not read png image");

	ImageConverter imageConverter(*hFile);
	imageConverter.convert(*imgFile, b, c);

	/*
	if (!b)
		qDebug("Generating QRgb arrays...");

	if (argv == 3) {
		// Check arg 1 for png, and arg 2 for h
		if (!QString(argc[1]).contains(".png"))
			fail("First argument must be an existing png image.");
		if (!QString(argc[2]).contains(".h"))
			fail("Second argument must be a filename with a .h extension.");
	    ImageConverter imageConverter(argc[2]);
	    imageConverter.convert(argc[1], b);
	} else if (argv == 4) {
		// Check arg 2 for png, and arg 2 for h
	    ImageConverter imageConverter(argc[3]);
	    imageConverter.convert(argc[2], b);
	}
	*/

    return EXIT_SUCCESS;
}

 ImageConverter::ImageConverter(const char *headerFile) {
    QDateTime date( QDateTime::currentDateTime() );
    QString datestring( date.toString() );

    file = new QFile( headerFile );
}

ImageConverter::~ImageConverter() {
    file->close();
    delete file;
    delete index;
}

void ImageConverter::convert(const char *name, bool indexOnly, bool replace) {
    QFileInfo   fileinfo( name );
    QString     basename( fileinfo.baseName() );
    QString     codename( basename );
    QImage      image( name );

	if (image == NULL) {
		fail("could not read png image");
	}

    file->open( IO_WriteOnly | IO_Truncate );

    stream.setDevice( file );
    index = new QPtrList<QRGBImage>;
    index->setAutoDelete( true );

    codename = codename.replace( QRegExp("[^a-zA-Z0-9]"), "_" );

	if (!indexOnly) {
	    stream << "\tstatic const QRgb " << codename << "_data[] = {" << endl << "\t\t";
	    stream.setf( QTextStream::hex | QTextStream::right );
	    stream.fill( '0' );
	}

    int pixels = image.width() * image.height();
    Q_UINT32 *data = reinterpret_cast<Q_UINT32*>( image.bits() );
    bool hasAlpha = false;

    for ( int i = 0, j = 0; i < pixels; i++ ) {

        if ( qAlpha( *data ) && qAlpha( *data ) != 0xff ) {
             hasAlpha = true;
        }

		if (!indexOnly) {
	        stream << "0x" << qSetW(8) << *(data++);
		}
			
        if ( i != pixels-1 ) {
			if (!indexOnly) {
            	stream << ',';
			}
			
            if ( j++ > 4 ) {
                j = 0;
				if (!indexOnly) {
	                stream << endl << "\t\t";
				}
            } else {
				if (!indexOnly) {
	                stream << ' ';
				}
            }
        }
    }

	if (!indexOnly) {
		stream.reset();
		stream << endl << "\t}; // " << codename << "_data" << endl << endl;
	}

    QRGBImage *imginfo = new QRGBImage;
    imginfo->width = image.width();
    imginfo->height = image.height();
    imginfo->alpha = hasAlpha;
    imginfo->name = codename;
    imginfo->string = basename;
    index->append( imginfo ); 

	if (indexOnly) {
		uint i = 0;
 		for (QRGBImage *image = index->first(); image; image = index->next()) {
			QString imgString(image->string);
			if (replace) {
				imgString = imgString.replace("_", "-");
			}
			stream << "\t\t{ \"" << imgString << "\", "
            		<< image->width << ", " << image->height <<
					", " << (image->alpha ? "true" : "false")
					<< ", " << image->name << "_data }";
			if (i++ < index->count() - 1)
				stream << ',';
				stream << endl;
     	}
	}
}

void fail(const char *reason) {
	qWarning("png2qrgb: %s", reason);
	std::cout << "Usage: png2qrgb [-a] [-r] <image.png> <image_header.h>" << std::endl;
    exit(EXIT_FAILURE);
}

void out(const char *string) {
	qWarning("png2qrgb: %s", string);
}

// vim:ts=4:sw=4
