/*
   FALCON - The Falcon Programming Language.
   FILE: flc_memory.h

   User overridable basic memory allocation
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: mar ago 3 2004

   -------------------------------------------------------------------
   (C) Copyright 2004: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

#ifndef flc_MEMORY_H
#define flc_MEMORY_H

#include <falcon/setup.h>
#include <falcon/globals.h>
#include <stdlib.h>

/** \file
   This files contains the user overridable memory allocation systems.

   Memory is alwas allocated via memAlloc() and released via memFree()
   function pointers. These are initially set at DflMemAlloc() and
   DflMemFree() functions, but they may be overridden with functions
   of the same kind at every point.

   Falcon memory management is twofold: Memory used for internal storage
   (i.e. module loading, symbol tables, file names, configurations etc)
   is not accounted nor controlled, and never garbaged; the only control
   is during allocation and releasing via the basic functions, which
   print an error message and quit in case of allocation errors.

   Items used by the VM (generated by scripts or returned to scripts by
   extensions) are created via the MemPool object; the memory
   allocated by this object is stored in the memory pool and may be garbaged,
   recycled or corrected.

   In either case, Falcon never makes any assumption on the given memory. The
   embedding application may safely add any kind of memory management &
   accounting it likes.
*/

namespace Falcon {

/** Account allocated memory.
   Allocators creating Falcon object should call this function to inform the GC system
   of the memory they are consuming, and so, the memory that may be freed if they are
   reclaimed.
   
   Global functions Falcon::gcAlloc(), Falcon::gcRealloc() and Falcon::gcFree() call automatically this
   account function. Garbageable objects are derived by a base class that automatically calls this
   functions.
   
*/
FALCON_DYN_SYM  void gcMemAccount( size_t memSize );
FALCON_DYN_SYM  void gcMemUnaccount( size_t memSize );
   
/** Return the total memory allocated by the GC system. */
FALCON_DYN_SYM  size_t gcMemAllocated();
   
FALCON_DYN_SYM void * DflMemAlloc( size_t amount );
FALCON_DYN_SYM void DflMemFree( void *mem );
FALCON_DYN_SYM void * DflMemRealloc( void *mem, size_t amount );

FALCON_DYN_SYM void * DflAccountMemAlloc( size_t amount );
FALCON_DYN_SYM void DflAccountMemFree( void *mem );
FALCON_DYN_SYM void * DflAccountMemRealloc( void *mem, size_t amount );

}

#endif

/* end of flc_memory.h */
