/*
Copyright (C) 2009 Parallel Realities

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "../headers.h"

#include "../graphics/animation.h"
#include "../system/properties.h"
#include "../entity.h"
#include "../system/random.h"
#include "../audio/audio.h"
#include "../collisions.h"
#include "../player.h"
#include "../custom_actions.h"

extern Entity *self;

static int bodyParts = 8;

static void wait(void);
static void bodyWait(void);
static void createBody(void);
static void takeDamage(Entity *, int);
static void bodyTakeDamage(Entity *, int);
static void die(void);
static void riseUp(void);
static void sink(void);
static void swimAround(void);
static void alignBodyToHead(void);

Entity *addEyeStalk(int x, int y, char *name)
{
	Entity *e = getFreeEntity();

	if (e == NULL)
	{
		printf("No free slots to add an Eye Stalk\n");

		exit(1);
	}

	loadProperties(name, e);

	e->x = x;
	e->y = y;

	e->action = &createBody;

	e->draw = &drawLoopingAnimationToMap;
	e->touch = &entityTouch;
	e->die = &die;
	e->takeDamage = &takeDamage;

	e->type = ENEMY;

	setEntityAnimation(e, STAND);

	return e;
}

static void riseUp()
{
	if (self->y == self->startY)
	{
		self->startY -= 128;
	}

	if (self->y > self->startY)
	{
		self->y -= 3;
	}

	else
	{
		self->y = self->startY;

		self->action = &wait;
	}

	alignBodyToHead();
}

static void sink()
{
	if (self->y < self->endY)
	{
		self->y += 3;
	}

	else
	{
		self->y = self->endY;

		self->dirX = self->face == RIGHT ? self->speed : -self->speed;

		self->thinkTime = 120 + prand() % 180;

		self->action = &swimAround;
	}

	alignBodyToHead();
}

static void swimAround()
{
	checkToMap(self);

	if (self->dirX == 0)
	{
		self->dirX = self->face == RIGHT ? -self->speed : self->speed;

		self->face = self->dirX < 0 ? LEFT : RIGHT;
	}

	self->thinkTime--;

	if (self->thinkTime <= 0)
	{
		self->action = &riseUp;

		self->thinkTime = 180 + prand() % 120;

		self->startX = 0;
	}

	self->endX = self->x;

	alignBodyToHead();
}

static void wait()
{
	self->startX++;

	if (self->startX >= 360)
	{
		self->startX = 0;
	}

	self->x = self->endX + sin(DEG_TO_RAD(self->startX)) * 10;

	alignBodyToHead();

	if (self->maxThinkTime >= 0)
	{
		self->thinkTime--;

		if (self->thinkTime <= 0)
		{
			self->action = &sink;
		}
	}
}

static void bodyWait()
{
	checkToMap(self);
}

static void createBody()
{
	char bodyName[MAX_VALUE_LENGTH];
	int i;
	Entity **body, *head;

	self->x = self->endX;
	self->y = self->endY;

	body = (Entity **)malloc(bodyParts * sizeof(Entity *));

	if (body == NULL)
	{
		printf("Failed to allocate a whole %d bytes for Eye Stalk body...\n", bodyParts * (int)sizeof(Entity *));

		exit(1);
	}

	snprintf(bodyName, sizeof(bodyName), "%s_body", self->name);

	/* Create in reverse order so that it is draw correctly */

	for (i=bodyParts-1;i>=0;i--)
	{
		body[i] = getFreeEntity();

		if (body[i] == NULL)
		{
			printf("No free slots to add a Eye Stalk body part\n");

			exit(1);
		}

		loadProperties(bodyName, body[i]);

		body[i]->x = self->x;
		body[i]->y = self->y;

		body[i]->action = &bodyWait;

		body[i]->draw = &drawLoopingAnimationToMap;
		body[i]->touch = &entityTouch;
		body[i]->die = &entityDie;
		body[i]->takeDamage = &bodyTakeDamage;

		body[i]->type = ENEMY;

		setEntityAnimation(body[i], STAND);
	}

	/* Recreate the head so that it's on top */

	head = getFreeEntity();

	if (head == NULL)
	{
		printf("No free slots to add an Eye Stalk head\n");
	}

	*head = *self;

	self->inUse = FALSE;

	self = head;

	/* Link the sections */

	for (i=bodyParts-1;i>=0;i--)
	{
		if (i == 0)
		{
			self->target = body[i];
		}

		else
		{
			body[i - 1]->target = body[i];
		}

		body[i]->head = self;
	}

	free(body);

	self->action = self->maxThinkTime < 0 ? &riseUp : &swimAround;
}

static void takeDamage(Entity *other, int damage)
{
	int i;

	if (!(self->flags & INVULNERABLE))
	{
		self->health -= damage;

		if (self->health > 0)
		{
			setCustomAction(self, &flashWhite, 6, 0);
			setCustomAction(self, &invulnerableNoFlash, 20, 0);

			i = prand() % 3;

			switch (i)
			{
				case 0:
					playSoundToMap("sound/common/splat1.ogg", -1, self->x, self->y, 0);
				break;

				case 1:
					playSoundToMap("sound/common/splat2.ogg", -1, self->x, self->y, 0);
				break;

				default:
					playSoundToMap("sound/common/splat3.ogg", -1, self->x, self->y, 0);
				break;
			}
		}

		else
		{
			self->die();
		}
	}
}

static void bodyTakeDamage(Entity *other, int damage)
{
	Entity *temp = self->head;

	self = self->head;

	self->takeDamage(other, damage);

	self = temp;
}

static void die()
{
	Entity *e;

	e = self;

	self = self->target;

	while (self != NULL)
	{
		self->die();

		self->dirX = (prand() % 5) * (prand() % 2 == 0 ? -1 : 1);
		self->dirY = ITEM_JUMP_HEIGHT;

		self = self->target;
	}

	self = e;

	self->die = &entityDie;

	self->die();

	self->dirX = (prand() % 5) * (prand() % 2 == 0 ? -1 : 1);
	self->dirY = ITEM_JUMP_HEIGHT;
}

static void alignBodyToHead()
{
	float x, y, partDistanceX, partDistanceY;
	Entity *e;

	x = self->x;
	y = self->y;

	partDistanceX = self->endX - self->x;
	partDistanceY = fabs(self->endY - self->y);

	partDistanceX /= bodyParts;
	partDistanceY /= bodyParts;

	e = self->target;

	while (e != NULL)
	{
		x += partDistanceX;
		y += partDistanceY;

		e->x = (e->target == NULL ? self->endX : x) + (self->w - e->w) / 2;
		e->y = (e->target == NULL ? self->endY : y);

		e->damage = self->damage;

		e->face = self->face;

		if (self->flags & FLASH)
		{
			e->flags |= FLASH;
		}

		else
		{
			e->flags &= ~FLASH;
		}

		e = e->target;
	}
}
