##---------------------------------------------------------------------------##
##
## Ultrasol -- a Python Solitaire game
##
## Copyright (C) 2000 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
##---------------------------------------------------------------------------##


# imports
import sys

# Ultrasol imports
if sys.modules.has_key("pysoltk"):
    from gamedb import registerGame, GameInfo, GI
    from util import *
    from stack import *
    from game import Game
    from layout import Layout
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint


# /***********************************************************************
# // Tam O'Shanter
# ************************************************************************/

class TamOShanter(Game):
    def createGame(self):
        # create layout
        l, s = Layout(self), self.s

        # set window
        self.setSize(l.XM + 6*l.XS, l.YM + 4*l.YS)

        # create stacks
        x, y, = l.XM, l.YM
        s.talon = DealRowTalonStack(x, y, self, max_rounds=1)
        l.createText(s.talon, "ss")
        for i in range(4):
            x, y = l.XM + (i+2)*l.XS, l.YM
            s.foundations.append(RK_FoundationStack(x, y, self))
        for i in range(4):
            x, y = l.XM + (i+2)*l.XS, l.YM + l.YS
            s.rows.append(BasicRowStack(x, y, self, max_move=1, max_accept=0))
            x = x + l.XS

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def startGame(self):
        self.startDealSample()
        self.s.talon.dealRow()

    def getAutoStacks(self, event=None):
        return ((), (), self.sg.dropstacks)


# /***********************************************************************
# // Auld Lang Syne
# ************************************************************************/

class AuldLangSyne(TamOShanter):
    def _shuffleHook(self, cards):
        # move Aces to top of the Talon (i.e. first cards to be dealt)
        return self._shuffleHookMoveToTop(cards, lambda c: (c.rank == 0, c.suit))

    def startGame(self):
        self.s.talon.dealRow(rows=self.s.foundations, frames=0)
        self.startDealSample()
        self.s.talon.dealRow()


# /***********************************************************************
# // Strategy
# ************************************************************************/

class Strategy_Foundation(SS_FoundationStack):
    def acceptsCards(self, from_stack, cards):
        if not SS_FoundationStack.acceptsCards(self, from_stack, cards):
            return 0
        # we only accept cards if there are no cards in the talon
        return len(self.game.s.talon.cards) == 0


class Strategy_RowStack(BasicRowStack):
    def acceptsCards(self, from_stack, cards):
        if not BasicRowStack.acceptsCards(self, from_stack, cards):
            return 0
        # this stack accepts any one card from the Talon
        return from_stack is self.game.s.talon and len(cards) == 1

    def canMoveCards(self, cards):
        if self.game.s.talon.cards:
            return 0
        return BasicRowStack.canMoveCards(self, cards)

    def clickHandler(self, event):
        if self.game.s.talon.cards:
            self.game.s.talon.playMoveMove(1, self)
            return 1
        return BasicRowStack.clickHandler(self, event)

    def doubleclickHandler(self, event):
        if self.game.s.talon.cards:
            self.game.s.talon.playMoveMove(1, self)
            return 1
        return BasicRowStack.doubleclickHandler(self, event)

    def getBottomImage(self):
        return self.game.app.images.getReserveBottom()


class Strategy(Game):
    def createGame(self):
        # create layout
        l, s = Layout(self), self.s

        # set window
        self.setSize(l.XM + 8*l.XS, l.YM + 4*l.YS)

        # create stacks
        x, y, = l.XM, l.YM
        s.talon = OpenTalonStack(x, y, self)
        l.createText(s.talon, "se")
        for i in range(4):
            x, y = l.XM + (i+2)*l.XS, l.YM
            s.foundations.append(Strategy_Foundation(x, y, self, suit=i, max_move=0))
        for i in range(8):
            x, y = l.XM + i*l.XS, l.YM + l.YS
            s.rows.append(Strategy_RowStack(x, y, self, max_move=1, max_accept=1))
            x = x + l.XS

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def _shuffleHook(self, cards):
        # move Aces to top of the Talon (i.e. first cards to be dealt)
        return self._shuffleHookMoveToTop(cards, lambda c: (c.rank == 0, c.suit))

    def startGame(self):
        self.startDealSample()
        self.s.talon.dealRow(rows=self.s.foundations)
        self.s.talon.fillStack()


# /***********************************************************************
# // Interregnum
# ************************************************************************/

class Interregnum_Foundation(RK_FoundationStack):
    def acceptsCards(self, from_stack, cards):
        if not RK_FoundationStack.acceptsCards(self, from_stack, cards):
            return 0
        if len(self.cards) == 12:
            # the final card must come from the reserve above the foundation
            return from_stack.id == self.id - 8
        else:
            # card must come from rows
            return from_stack in self.game.s.rows


class Interregnum(Game):
    GAME_VERSION = 2

    #
    # game layout
    #

    def createGame(self, rows=8):
        # create layout
        l, s = Layout(self), self.s

        # set window
        self.setSize(l.XM + max(9,rows)*l.XS, l.YM + 5*l.YS)

        # extra settings
        self.base_cards = None

        # create stacks
        for i in range(8):
            x, y, = l.XM + i*l.XS, l.YM
            s.reserves.append(ReserveStack(x, y, self, max_accept=0))
        for i in range(8):
            x, y, = l.XM + i*l.XS, l.YM + l.YS
            s.foundations.append(Interregnum_Foundation(x, y, self, mod=13, max_move=0))
        for i in range(rows):
            x, y, = l.XM + (2*i+8-rows)*l.XS/2, l.YM + 2*l.YS
            s.rows.append(RK_RowStack(x, y, self, max_accept=0, max_move=1))
        s.talon = DealRowTalonStack(self.width-l.XS, self.height-l.YS, self)
        l.createText(s.talon, "nn")

        # define stack-groups
        l.defaultStackGroups()

    #
    # game overrides
    #

    def startGame(self):
        self.startDealSample()
        # deal base_cards to reserves, update foundations cap.base_rank
        self.base_cards = []
        for i in range(8):
            self.base_cards.append(self.s.talon.getCard())
            self.s.foundations[i].cap.base_rank = (self.base_cards[i].rank + 1) % 13
            self.flipMove(self.s.talon)
            self.moveMove(1, self.s.talon, self.s.reserves[i])
        # deal other cards
        self.s.talon.dealRow()

    def getAutoStacks(self, event=None):
        return ((), (), self.sg.dropstacks)

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return ((card1.rank + 1) % 13 == card2.rank or (card2.rank + 1) % 13 == card1.rank)

    def _restoreGameHook(self, game):
        self.base_cards = [None] * 8
        for i in range(8):
            id = game.loadinfo.base_card_ids[i]
            self.base_cards[i] = self.cards[id]
            self.s.foundations[i].cap.base_rank = (self.base_cards[i].rank + 1) % 13

    def _loadGameHook(self, p):
        ids = []
        for i in range(8):
            ids.append(p.load())
        self.loadinfo.addattr(base_card_ids=ids)    # register extra load var.

    def _saveGameHook(self, p):
        for c in self.base_cards:
            p.dump(c.id)


# register the game
registerGame(GameInfo(172, TamOShanter, "Tam O'Shanter",
                      GI.GT_NUMERICA, 1, 0))
registerGame(GameInfo(95, AuldLangSyne, "Auld Lang Syne",
                      GI.GT_NUMERICA, 1, 0))
registerGame(GameInfo(173, Strategy, "Strategy",
                      GI.GT_NUMERICA, 1, 0))
registerGame(GameInfo(123, Interregnum, "Interregnum",
                      GI.GT_NUMERICA, 2, 0))

