##---------------------------------------------------------------------------##
##
## Ultrasol -- a Python Solitaire game
##
## Copyright (C) 2000 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
##---------------------------------------------------------------------------##


# imports
import sys

# Ultrasol imports
if sys.modules.has_key("pysoltk"):
    from gamedb import registerGame, GameInfo, GI
    from util import *
    from stack import *
    from game import Game
    from layout import Layout
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint


# /***********************************************************************
# //
# ************************************************************************/

class Bristol_Hint(CautiousDefaultHint):
    # FIXME: demo is not too clever in this game

    BONUS_CREATE_EMPTY_ROW     = 0           # 0..9000
    BONUS_CAN_DROP_ALL_CARDS   = 0           # 0..4000
    BONUS_CAN_CREATE_EMPTY_ROW = 0           # 0..4000

    # Score for moving a pile from stack r to stack t.
    # Increased score must be in range 0..9999
    def _getMovePileScore(self, score, color, r, t, pile, rpile):
        # prefer reserves
        if not r in self.game.s.reserves:
            score = score - 10000
            # an empty pile doesn't gain anything
            if len(pile) == len(r.cards):
                return -1, color
        return CautiousDefaultHint._getMovePileScore(self, score, color, r, t, pile, rpile)


# /***********************************************************************
# // Bristol
# ************************************************************************/

class Bristol_Talon(TalonStack):
    def dealCards(self, sound=0):
        return self.dealRowAvail(rows=self.game.s.reserves, sound=sound)


class Bristol(Game):
    Layout_Method = Layout.klondikeLayout
    Hint_Class = Bristol_Hint

    #
    # game layout
    #

    def createGame(self, **layout):
        # create layout
        l, s = Layout(self, XOFFSET=10), self.s

        # set window
        self.setSize(l.XM + 10*l.XS, l.YM + 5*l.YS)

        # create stacks
        x, y, = l.XM + 3*l.XS, l.YM
        for i in range(4):
            s.foundations.append(RK_FoundationStack(x, y, self, max_move=0))
            x = x + l.XS
        for i in range(2):
            y = l.YM + (i*2+3)*l.YS/2
            for j in range(4):
                x = l.XM + (j*5)*l.XS/2
                stack = RK_RowStack(x, y, self,  base_rank=NO_RANK, max_move=1)
                stack.CARD_XOFFSET, stack.CARD_YOFFSET = l.XOFFSET, 0
                s.rows.append(stack)
        x, y, = l.XM + 3*l.XS, l.YM + 4*l.YS
        s.talon = Bristol_Talon(x, y, self)
        l.createText(s.talon, "sw")
        for i in range(3):
            x = x + l.XS
            s.reserves.append(ReserveStack(x, y, self, max_accept=0, max_cards=999999))

        # define stack-groups
        self.sg.openstacks = s.foundations + s.rows
        self.sg.talonstacks = [s.talon]
        self.sg.dropstacks = s.rows + s.reserves

    #
    # game overrides
    #

    def _shuffleHook(self, cards):
        # move Kings to bottom of each stack
        i, n = 0, len(self.s.rows)
        kings = []
        for c in cards[:24]:    # search the first 24 cards only
            if c.rank == KING:
                kings.append(i)
            i = i + 1
        for i in kings:
            j = i % n           # j = card index of rowstack bottom
            while j < i:
                if cards[j].rank != KING:
                    cards[j], cards[i] = cards[i], cards[j]
                    break
                j = j + n
        cards.reverse()
        return cards

    def startGame(self):
        r = self.s.rows
        for i in range(2):
            self.s.talon.dealRow(rows=r, frames=0)
        self.startDealSample()
        self.s.talon.dealRow(rows=r)
        self.s.talon.dealCards()          # deal first cards to Reserves


# /***********************************************************************
# // Belvedere
# ************************************************************************/

class Belvedere(Bristol):
    def _shuffleHook(self, cards):
        # remove 1 Ace
        for c in cards:
            if c.rank == 0:
                cards.remove(c)
                break
        # move Kings to bottom
        cards = Bristol._shuffleHook(self, cards)
        # re-insert Ace
        return cards[:-24] + [c] + cards[-24:]

    def startGame(self):
        r = self.s.rows
        for i in range(2):
            self.s.talon.dealRow(rows=r, frames=0)
        self.startDealSample()
        self.s.talon.dealRow(rows=r)
        assert self.s.talon.cards[-1].rank == ACE
        self.s.talon.dealRow(rows=self.s.foundations[:1])
        self.s.talon.dealCards()          # deal first cards to Reserves


# register the game
registerGame(GameInfo(42, Bristol, "Bristol",
                      GI.GT_FAN_TYPE, 1, 0))
registerGame(GameInfo(214, Belvedere, "Belvedere",
                      GI.GT_FAN_TYPE, 1, 0))

