##---------------------------------------------------------------------------##
##
## Ultrasol -- a Python Solitaire game
##
## Copyright (C) 2000 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
##---------------------------------------------------------------------------##


# imports
import sys

# Ultrasol imports
if sys.modules.has_key("pysoltk"):
    from gamedb import registerGame, GameInfo, GI
    from util import *
    from stack import *
    from game import Game
    from layout import Layout
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint
    from pysoltk import MfxCanvasText


# /***********************************************************************
# //
# ************************************************************************/

class Fan_Hint(CautiousDefaultHint):
    # FIXME: demo is not too clever in this game
    pass


# /***********************************************************************
# // Fan
# ************************************************************************/

class Fan(Game):
    Talon_Class = InitialDealTalonStack
    Foundation_Class = SS_FoundationStack
    RowStack_Class = KingSS_RowStack
    Hint_Class = Fan_Hint

    #
    # game layout
    #

    def createGame(self, rows=(5,5,5,3), playcards=9):
        # create layout
        l, s = Layout(self, XOFFSET=10), self.s

        # set window
        # (set size so that at least 9 cards are fully playable)
        w = max(2*l.XS, l.XS+(playcards-1)*l.XOFFSET)
        w = min(3*l.XS, w)
        w = (w + 1) & ~1
        ##print 2*l.XS, w
        self.setSize(l.XM + max(rows)*w, l.YM + (1+len(rows))*l.YS)

        # create stacks
        x, y = l.XM + w, l.YM
        for j in range(self.gameinfo.decks):
            for i in range(4):
                s.foundations.append(self.Foundation_Class(x, y, self, suit=i))
                x = x + w / self.gameinfo.decks
        for i in range(len(rows)):
            x, y = l.XM, y + l.YS
            for j in range(rows[i]):
                stack = self.RowStack_Class(x, y, self, max_move=1, max_accept=1)
                stack.CARD_XOFFSET, stack.CARD_YOFFSET = l.XOFFSET, 0
                s.rows.append(stack)
                x = x + w
        x, y = self.width - l.XS, self.height - l.YS
        s.talon = self.Talon_Class(x, y, self)

        # define stack-groups
        l.defaultStackGroups()
        return l

    #
    # game overrides
    #

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(rows=self.s.rows[:17], frames=0)
        self.startDealSample()
        self.s.talon.dealRow()
        assert len(self.s.talon.cards) == 0

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit == card2.suit and
                (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank))

    def getHighlightPilesStacks(self):
        return ()


# /***********************************************************************
# // Scotch Patience
# ************************************************************************/

class ScotchPatience(Fan):
    Foundation_Class = AC_FoundationStack
    RowStack_Class = StackWrapper(RK_RowStack, base_rank=NO_RANK)


# /***********************************************************************
# // Shamrocks
# ************************************************************************/

class Shamrocks_RowStack(BasicRowStack):
    def acceptsCards(self, from_stack, cards):
        if not BasicRowStack.acceptsCards(self, from_stack, cards):
            return 0
        c1, c2 = self.cards[-1], cards[0]
        return c1.rank == (c2.rank + 1) % self.cap.mod or c2.rank == (c1.rank + 1) % self.cap.mod


class Shamrocks(Fan):
    RowStack_Class = StackWrapper(Shamrocks_RowStack, base_rank=NO_RANK, max_cards=3)

##    def shallHighlightMatch(self, stack1, card1, stack2, card2):
##        return (card1.rank + 1 == card2.rank or card2.rank + 1 == card1.rank)


# /***********************************************************************
# // La Belle Lucie (Midnight Oil)
# ************************************************************************/

class LaBelleLucie_Talon(TalonStack):
    def canDealCards(self):
        return self.round != self.max_rounds and not self.game.isGameWon()

    def dealCards(self, sound=0):
        n = self.redealCards1()
        if n == 0:
            return 0
        self.redealCards2()
        if sound:
            self.game.startDealSample()
        self.redealCards3()
        if sound:
            self.game.stopSamples()
        return n

    # redeal step 1) - collect all cards, move them to the Talon
    def redealCards1(self):
        assert len(self.cards) == 0
        num_cards = 0
        for r in self.game.s.rows:
            if r.cards:
                num_cards = num_cards + len(r.cards)
                self.game.moveMove(len(r.cards), r, self, frames=0)
        assert len(self.cards) == num_cards
        return num_cards

    # redeal step 2) - shuffle
    def redealCards2(self):
        assert self.round != self.max_rounds
        assert self.cards
        self.game.shuffleStackMove(self)
        self.game.nextRoundMove(self)

    # redeal step 3) - redeal cards to stacks
    def redealCards3(self, face_up=1):
        # deal 3 cards to each row, and 1-3 cards to last row
        to_stacks = self.game.s.rows
        n = min(len(self.cards), 3*len(to_stacks))
        for i in range(3):
            j = (n/3, (n+1)/3, (n+2)/3) [i]
            frames = (0, 0, 4) [i]
            for r in to_stacks[:j]:
                if self.cards[-1].face_up != face_up:
                    self.game.flipMove(self)
                self.game.moveMove(1, self, r, frames=frames)


class LaBelleLucie(Fan):
    Talon_Class = StackWrapper(LaBelleLucie_Talon, max_rounds=3)
    RowStack_Class = StackWrapper(SS_RowStack, base_rank=NO_RANK)


# /***********************************************************************
# // Super Flower Garden
# ************************************************************************/

class SuperFlowerGarden(LaBelleLucie):
    RowStack_Class = StackWrapper(RK_RowStack, base_rank=NO_RANK)


# /***********************************************************************
# // Three Shuffles and a Draw
# ************************************************************************/

class ThreeShufflesAndADraw_RowStack(SS_RowStack):
    def moveMove(self, ncards, to_stack, frames=-1, shadow=-1):
        game, r = self.game, self.game.s.reserves[0]
        if to_stack is not r:
            SS_RowStack.moveMove(self, ncards, to_stack, frames=frames, shadow=shadow)
            return
        f = self._canDrawCard()
        assert f and game.draw_done == 0 and ncards == 1
        # 1) top card from self to reserve
        game.updateStackMove(r, 2|16)       # update view for undo
        game.moveMove(1, self, r, frames=frames, shadow=shadow)
        game.updateStackMove(r, 3|64)       # update model
        game.updateStackMove(r, 1|16)       # update view for redo
        # 2) second card from self to foundation/row
        if frames == 0:
            frames = -1
        game.moveMove(1, self, f, frames=frames, shadow=shadow)
        # 3) from reserve back to self
        #    (need S_FILL because the move is normally not valid)
        old_state = game.enterState(game.S_FILL)
        game.moveMove(1, r, self, frames=frames, shadow=shadow)
        game.leaveState(old_state)

    def _canDrawCard(self):
        if len(self.cards) >= 2:
            pile = self.cards[-2:-1]
            for s in self.game.s.foundations + self.game.s.rows:
                if s is not self and s.acceptsCards(self, pile):
                    return s
        return None


class ThreeShufflesAndADraw_ReserveStack(ReserveStack):
    def acceptsCards(self, from_stack, cards):
        if not ReserveStack.acceptsCards(self, from_stack, cards):
            return 0
        if not from_stack in self.game.s.rows:
            return 0
        if self.game.draw_done or not from_stack._canDrawCard():
            return 0
        return 1

    def updateModel(self, undo, flags):
        assert undo == self.game.draw_done
        self.game.draw_done = not self.game.draw_done

    def updateText(self):
        if self.game.preview > 1 or self.texts.misc is None:
            return
        t = ("X", "Draw") [self.game.draw_done == 0]
        self.texts.misc.config(text=t)

    def prepareView(self):
        ReserveStack.prepareView(self)
        if not self.is_visible or self.game.preview > 1:
            return
        images = self.game.app.images
        x, y = self.x + images.CARDW/2, self.y + images.CARDH/2
        self.texts.misc = MfxCanvasText(self.game.canvas, x, y, anchor="center")


class ThreeShufflesAndADraw(LaBelleLucie):
    RowStack_Class = StackWrapper(ThreeShufflesAndADraw_RowStack, base_rank=NO_RANK)

    def createGame(self):
        l = LaBelleLucie.createGame(self)
        s = self.s
        # add a reserve stack
        x, y = s.rows[3].x, s.rows[-1].y
        s.reserves.append(ThreeShufflesAndADraw_ReserveStack(x, y, self))
        # redefine the stack-groups
        l.defaultStackGroups()
        # extra settings
        self.draw_done = 0

    def startGame(self):
        self.draw_done = 0
        self.s.reserves[0].updateText()
        LaBelleLucie.startGame(self)

    def _restoreGameHook(self, game):
        self.draw_done = game.loadinfo.draw_done

    def _loadGameHook(self, p):
        self.loadinfo.addattr(draw_done=p.load())

    def _saveGameHook(self, p):
        p.dump(self.draw_done)


# /***********************************************************************
# // Trefoil
# ************************************************************************/

class Trefoil(LaBelleLucie):
    GAME_VERSION = 2
    Foundation_Class = StackWrapper(SS_FoundationStack, min_cards=1)

    def createGame(self):
        return Fan.createGame(self, rows=(5,5,5,1))

    def _shuffleHook(self, cards):
        # move Aces to bottom of the Talon (i.e. last cards to be dealt)
        return self._shuffleHookMoveToBottom(cards, lambda c: (c.rank == 0, c.suit))

    def startGame(self):
        for i in range(2):
            self.s.talon.dealRow(frames=0)
        self.startDealSample()
        self.s.talon.dealRow()
        self.s.talon.dealRow(rows=self.s.foundations)


# /***********************************************************************
# // Intelligence
# ************************************************************************/

class Intelligence_Talon(LaBelleLucie_Talon):
    # all Aces go to Foundations
    dealToStacks = TalonStack.dealToStacksOrFoundations

    # redeal step 1) - collect all cards, move them to the Talon (face down)
    def redealCards1(self):
        assert len(self.cards) == 0
        r = self.game.s.reserves[0]
        num_cards = len(r.cards)
        self.game.moveMove(len(r.cards), r, self, frames=0)
        for r in self.game.s.rows:
            num_cards = num_cards + len(r.cards)
            while r.cards:
                self.game.moveMove(1, r, self, frames=0)
                self.game.flipMove(self)
        assert len(self.cards) == num_cards
        assert forall(lambda c: not c.face_up, self.cards)
        return num_cards

    # redeal step 3) - redeal cards to stacks
    def redealCards3(self, face_up=1):
        for r in self.game.s.rows:
            while len(r.cards) < 3:
                self.dealToStacks([r], frames=4)
                if not self.cards:
                    return
        # move all remaining cards to the reserve
        self.game.moveMove(len(self.cards), self, self.game.s.reserves[0], frames=0)


# up or down in suit
class Intelligence_RowStack(BasicRowStack):
    def acceptsCards(self, from_stack, cards):
        if not BasicRowStack.acceptsCards(self, from_stack, cards):
            return 0
        assert self.cards
        c1, c2 = self.cards[-1], cards[0]
        if c1.suit != c2.suit:
            return 0
        return c1.rank == (c2.rank + 1) % self.cap.mod or c2.rank == (c1.rank + 1) % self.cap.mod

    def fillStack(self):
        if not self.cards:
            r = self.game.s.reserves[0]
            if r.cards:
                r.dealRow(rows=(self,self,self), sound=0)


class Intelligence_ReserveStack(ReserveStack, DealRow_StackMethods):
    def canFlipCard(self):
        return 0

    # all Aces go to Foundations (used in r.dealRow() above)
    dealToStacks = TalonStack.dealToStacksOrFoundations


class Intelligence(Fan):
    Talon_Class = StackWrapper(Intelligence_Talon, max_rounds=3)
    RowStack_Class = StackWrapper(Intelligence_RowStack, base_rank=NO_RANK)

    def createGame(self):
        l = Fan.createGame(self)
        s = self.s
        # add a reserve stack
        x, y = s.talon.x - l.XS, s.talon.y
        s.reserves.append(Intelligence_ReserveStack(x, y, self, max_move=0, max_accept=0, max_cards=999999))
        l.createText(s.reserves[0], "sw")
        # redefine the stack-groups
        l.defaultStackGroups()

    def startGame(self):
        talon = self.s.talon
        for i in range(2):
            talon.dealRow(frames=0)
        self.startDealSample()
        talon.dealRow()
        # move all remaining cards to the reserve
        self.moveMove(len(talon.cards), talon, self.s.reserves[0], frames=0)


# register the game
registerGame(GameInfo(56, Fan, "Fan",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 0,
                      altnames="Midnight Oil"))
registerGame(GameInfo(87, ScotchPatience, "Scotch Patience",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 0))
registerGame(GameInfo(57, Shamrocks, "Shamrocks",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 0))
registerGame(GameInfo(901, LaBelleLucie, "La Belle Lucie",      # was: 32, 82
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 2))
registerGame(GameInfo(132, SuperFlowerGarden, "Super Flower Garden",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 2))
registerGame(GameInfo(128, ThreeShufflesAndADraw, "Three Shuffles and a Draw",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 2))
registerGame(GameInfo(88, Trefoil, "Trefoil",
                      GI.GT_FAN_TYPE | GI.GT_OPEN, 1, 2))
registerGame(GameInfo(227, Intelligence, "Intelligence",
                      GI.GT_FAN_TYPE, 2, 2))

