##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 2000 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
##---------------------------------------------------------------------------##


# Imports
import sys
from string import letters

# PySol imports
if sys.modules.has_key("pysoltk"):
    from gamedb import registerGame, GameInfo, GI
    from util import *
    from stack import *
    from game import Game
    from layout import Layout
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint
    from pysoltk import MfxCanvasText, getFont


# /***********************************************************************
# //
# ************************************************************************/

class Mahjongg_Hint(AbstractHint):
    # FIXME: no intelligence whatsoever is implemented here
    def computeHints(self):
        game = self.game
        # get free stacks
        stacks = []
        for r in game.s.rows:
            if r.cards and not r.basicIsBlocked():
                stacks.append(r)
        # find matching tiles
        i = 0
        for r in stacks:
            for t in stacks[i+1:]:
                if game.cardsMatch(r.cards[0], t.cards[0]):
                    # simple scoring...
                    score = 10000 + r.id + t.id
                    self.addHint(score, 1, r, t)
            i = i + 1


# /***********************************************************************
# //
# ************************************************************************/

class Mahjongg_Foundation(AbstractFoundationStack):
    def __init__(self, x, y, game, suit=ANY_SUIT, **cap):
        kwdefault(cap, max_move=0, max_accept=0, max_cards=144)
        apply(AbstractFoundationStack.__init__, (self, x, y, game, suit), cap)

    def acceptsCards(self, from_stack, cards):
        # We do not accept any cards - pairs will get
        # delivered by _dropPairMove() below.
        return 0

    def basicIsBlocked(self):
        return 1

    def initBindings(self):
        pass


# /***********************************************************************
# //
# ************************************************************************/

class Mahjongg_RowStack(OpenStack):
    def __init__(self, x, y, game, **cap):
        kwdefault(cap, max_move=1, max_accept=1, max_cards=2,
                  base_rank=NO_RANK)
        apply(OpenStack.__init__, (self, x, y, game), cap)

    def basicIsBlocked(self):
        # any of above blocks
        for stack in self.blockmap.above:
            if stack.cards:
                return 1
        # any of left blocks - but we can try right as well
        for stack in self.blockmap.left:
            if stack.cards:
                break
        else:
            return 0
        # any of right blocks
        for stack in self.blockmap.right:
            if stack.cards:
                return 1
        return 0

    def acceptsCards(self, from_stack, cards):
        if not OpenStack.acceptsCards(self, from_stack, cards):
            return 0
        return self.game.cardsMatch(self.cards[0], cards[-1])

    def canFlipCard(self):
        return 0

    def canDropCards(self, stacks):
        return (None, 0)

    def moveMove(self, ncards, to_stack, frames=-1, shadow=-1):
        self._dropPairMove(ncards, to_stack, frames=-1, shadow=shadow)

    def _dropPairMove(self, n, other_stack, frames=-1, shadow=-1):
        assert n == 1 and self.acceptsCards(other_stack, [other_stack.cards[-1]])
        old_state = self.game.enterState(self.game.S_FILL)
        f = self.game.s.foundations[0]
        self.game.moveMove(n, self, f, frames=frames, shadow=shadow)
        self.game.moveMove(n, other_stack, f, frames=frames, shadow=shadow)
        self.game.leaveState(old_state)
        self.fillStack()
        other_stack.fillStack()


    #
    # Mahjongg special overrides
    #

    # Mahjongg special: we must preserve the relative stacking order
    # to keep our pseudo 3D look.
    def _position(self, card):
        OpenStack._position(self, card)
        #
        rows = filter(lambda s: s.cards, self.game.s.rows[:self.id])
        if rows:
            self.group.tkraise(rows[-1].group)
            return
        rows = filter(lambda s: s.cards, self.game.s.rows[self.id+1:])
        if rows:
            self.group.lower(rows[0].group)
            return

    # In Mahjongg games type there are a lot of stacks, so we optimize
    # and don't create bindings that are not used anyway.
    def initBindings(self):
        group = self.group
        # FIXME: dirty hack to access the Stack's private methods
        bind(group, "<1>", self._Stack__clickEventHandler)
        bind(group, "<3>", self._Stack__controlclickEventHandler)
        bind(group, "<Control-1>", self._Stack__controlclickEventHandler)

    def clickHandler(self, event):
        game = self.game
        drag = game.drag
        # checks
        if not self.cards:
            return 1
        from_stack = drag.stack
        if from_stack is self:
            # remove selection
            self._stopDrag()
            return 1
        if self.basicIsBlocked():
            ### remove selection
            return 1
        # possible move
        if from_stack:
            if self.acceptsCards(from_stack, from_stack.cards):
                self._stopDrag()
                # this code actually moves the tiles
                from_stack.playMoveMove(1, self, frames=0, sound=0)
                return 1
        drag.stack = self
        # move or create the shade image (see stack.py, _updateShade)
        if drag.shade_img:
            img = drag.shade_img
            img.dtag(drag.shade_stack.group)
            img.moveTo(self.x, self.y)
        else:
            img = game.app.images.getShade()
            if img is None:
                return 1
            img = MfxCanvasImage(game.canvas, self.x, self.y, image=img, anchor=ANCHOR_NW)
            drag.shade_img = img
        # raise/lower the shade image to the correct stacking order
        if tkname == "tk":
            img.tkraise(self.cards[-1].item)
            img.addtag(self.group)
        elif tkname == "gnome":
            ## FIXME
            pass
        drag.shade_stack = self
        return 1

    def cancelDrag(self, event=None):
        if event is None:
            self._stopDrag()

    def _findCard(self, event):
        # we need to override this because the shade may be hiding
        # the tile (from Tk's stacking view)
        return len(self.cards) - 1


# /***********************************************************************
# //
# ************************************************************************/

class AbstractMahjonggGame(Game):
    Hint_Class = Mahjongg_Hint

    #
    # game layout
    #

    def createGame(self):
        L = self.L
        assert L[0] == "0"
        assert (len(L) - 1) % 3 == 0

        # decode tile positions
        tiles = []
        max_tl, max_tx, max_ty = -1, -1, -1
	t = string.letters
        for i in range(1, len(L), 3):
            n = string.find(t, L[i])
            level, height = n / 7, n % 7 + 1
            tx = string.find(t, L[i+1])
            ty = string.find(t, L[i+2])
            assert n >= 0 and tx >= 0 and ty >= 0
            max_tl = max(level + height - 1, max_tl)
            max_tx = max(tx, max_tx)
            max_ty = max(ty, max_ty)
            for tl in range(level, level + height):
                tiles.append((tl, tx, ty))
        assert len(tiles) == 144
        tiles.sort()
        tiles = tuple(tiles)

        # start layout
        l, s = Layout(self), self.s
        ##dx, dy = 2, -2
        dx, dy = 3, -3

        # set window size
        dxx, dyy = abs(dx) * max_tl, abs(dy) * max_tl
        w, h = (max_tx + 2) * l.CW / 2, (max_ty + 2) * l.CH / 2
        self.setSize(l.XM + dxx + w + 2*dxx + l.XS + l.XM, l.YM + dyy + h + dyy + l.YM)

        # set game extras
        self.check_dist = l.CW*l.CW + l.CH*l.CH     # see _getClosestStack()

        # create a row stack for each tile and compute the tilemap
        tilemap = {}
        for level, tx, ty in tiles:
            x = l.XM + dxx + (tx * l.CW) / 2 + level * dx
            y = l.YM + dyy + (ty * l.CH) / 2 + level * dy
            stack = Mahjongg_RowStack(x, y, self)
            stack.CARD_XOFFSET = dx
            stack.CARD_YOFFSET = dy
            s.rows.append(stack)
            # tilemap - each tile covers 4 positions
            tilemap[(level, tx, ty)] = stack
            tilemap[(level, tx+1, ty)] = stack
            tilemap[(level, tx, ty+1)] = stack
            tilemap[(level, tx+1, ty+1)] = stack

        # compute blockmap
        for stack in s.rows:
            level, tx, ty = tiles[stack.id]
            above, left, right = {}, {}, {}
            # above blockers
            for tl in range(level+1, level+2):
                above[tilemap.get((tl, tx, ty))] = 1
                above[tilemap.get((tl, tx+1, ty))] = 1
                above[tilemap.get((tl, tx, ty+1))] = 1
                above[tilemap.get((tl, tx+1, ty+1))] = 1
            # left blockers
            left[tilemap.get((level, tx-1, ty))] = 1
            left[tilemap.get((level, tx-1, ty+1))] = 1
            # right blockers
            right[tilemap.get((level, tx+2, ty))] = 1
            right[tilemap.get((level, tx+2, ty+1))] = 1
            # sanity check - assert that there are no overlapping tiles
            assert tilemap.get((level, tx, ty)) is stack
            assert tilemap.get((level, tx+1, ty)) is stack
            assert tilemap.get((level, tx, ty+1)) is stack
            assert tilemap.get((level, tx+1, ty+1)) is stack
            # assemble
            stack.blockmap = Struct(
                above = tuple(filter(None, above.keys())),
                left = tuple(filter(None, left.keys())),
                right = tuple(filter(None, right.keys())),
            )

        # create other stacks
        x = self.width + l.XS
        y = l.YM + dyy
        s.foundations.append(Mahjongg_Foundation(x, y, self))
	self.texts.info = MfxCanvasText(self.canvas, x - l.XM - l.XS, y,
			    anchor = "ne", font = getFont("canvas_card", cardw = l.CW))
        y = self.height - l.YS - dxx
        # the Talon is invisble
        s.talon = InitialDealTalonStack(x, y + l.YS, self)

        font = getFont("canvas_card", cardw=l.CW)

        # Define stack groups
        l.defaultStackGroups()


    #
    # game overrides
    #

    def startGame(self):
        assert len(self.s.talon.cards) == 144
	self.s.talon.dealRow(rows = self.s.rows, frames = 0)
	assert len(self.s.talon.cards) == 0

    def isGameWon(self):
        return len(self.s.foundations[0].cards) == 144

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        if stack1.basicIsBlocked() or stack2.basicIsBlocked():
            return 0
        return self.cardsMatch(card1, card2)

    def getAutoStacks(self, event=None):
        return ((), (), ())

    def updateText(self):
        if self.preview > 1 or self.texts.info is None:
            return
        # find matching tiles
        stacks = []
        for r in self.s.rows:
            if r.cards and not r.basicIsBlocked():
                stacks.append(r)
        f, i = 0, 0
	for r in stacks:
	    for t in stacks[i+1:]:
		if self.app.game.cardsMatch(r.cards[0], t.cards[0]):
		    f = f + 1
	    i = i + 1
	if f == 0:
	    f = "No"
	else:
	    f = str(f)
	f = f + " Free\nMatching\nPair" + "s"[:(f != "1")]
        t = len(self.s.foundations[0].cards)
        t = str(t) + "\nTiles\nRemoved\n\n" + str(144 - t) + "\nTiles\nRemaining\n\n" + f
        self.texts.info.config(text = t)


    #
    # Mahjongg special overrides
    #

    def getHighlightPilesStacks(self):
        # Mahjongg special: highlight all moveable tiles
        return ((self.s.rows, 1),)

    def getCardFaceImage(self, deck, suit, rank):
        if suit == 3:
            cs = self.app.cardset
            if len(cs.ranks) >= 12 and len(cs.suits) >= 4:
                # make Mahjongg type games playable with other cardsets
                if rank >= 8:       # flower
                    suit = 1
                    rank = len(cs.ranks) - 2
                elif rank >= 4:     # season
                    rank = max(10, len(cs.ranks) - 3)
                else:               # wind
                    suit = rank
                    rank = len(cs.ranks) - 1
        return self.app.images.getFace(deck, suit, rank)

    def getCardBackImage(self, deck, suit, rank):
        # We avoid screen updates caused by flipping cards - all
        # cards are face up anyway. The Talon should be invisible
        # or else the top tile of the Talon will be visible during
        # game start.
        return self.getCardFaceImage(deck, suit, rank)

    def _createCard(self, id, deck, suit, rank, x, y):
        if deck >= 1 and suit == 3 and rank >= 4:
            return None
        return Game._createCard(self, id, deck, suit, rank, x, y)

    def _getClosestStack(self, cx, cy, stacks, dragstack):
        closest, cdist = None, 999999999
        # Since we only compare distances,
        # we don't bother to take the square root.
        for stack in stacks:
            dist = (stack.x - cx)**2 + (stack.y - cy)**2
            if dist < cdist:
                # Mahjongg special: if the stack is very close, do
                # not consider blocked stacks
                if dist > self.check_dist or not stack.basicIsBlocked():
                    closest, cdist = stack, dist
        return closest

    #
    # Mahjongg extras
    #

    def cardsMatch(self, card1, card2):
        if card1.suit != card2.suit:
            return 0
        if card1.suit == 3:
            if card1.rank >= 8:
                return card2.rank >= 8
            if card1.rank >= 4:
                return 7 >= card2.rank >= 4
        return card1.rank == card2.rank


# /***********************************************************************
# // register a Mahjongg type game
# ************************************************************************/

def r(id, gameclass, short_name):
    name = "Mahjongg " + short_name
    gi = GameInfo(id, gameclass, name,
                  GI.GT_MAHJONGG, 4, 0,
                  category=GI.GC_MAHJONGG, short_name=short_name,
                  suits=range(3), ranks=range(10), trumps=range(12),
                  si={"decks": 1, "ncards": 144})
    gi.ncards = 144
    gi.rules_filename = "mahjongg.html"
    registerGame(gi)
    return gi


# /***********************************************************************
# // game definitions
# ************************************************************************/

class Mahjongg_Altar(AbstractMahjonggGame):
    L = "0aaaacaaiaakaamaaoaaqaasaauaawaaCaaEaaacaccaicckccmccoccqccsccucawcaCcaEcaieckecmecoecqecsecueaweaigckgcmgcogcqgcsgcugawgaiiakiamiaoiaqiasiauiawiaokaqkaamacmaomaqmaCmaEmaaoacoaooaqoaCoaEohabhcbhCbhEbkpijpkipmhanhcnhCnhEnhpoobboDbobnoDnvlcvncvpcvrcvtcvlevnevpevrevtevlgwngwpgwrgvtgCocCqcCmeCoeCqeCse"

class Mahjongg_Arena(AbstractMahjonggGame):
    L = "0eaadcaceabgaaiaaqabsacuadwaeyadaccccbecagcakcbmcaocascbuccwcdyccaebceaeeameauebwecyebagacgakgbmgaogawgbygcaibciaeiamiauibwicyidakcckbekagkakkbmkaokaskbukcwkdykeamdcmcembgmaimaqmbsmcumdwmeym"

class Mahjongg_Arena2(AbstractMahjonggGame):
    L = "0daadcabeabgaaiaakaamaaoaaqaasabuabwadyadAadaccccbecagcaucbwccycdAcdaecceaeeawecyedAedagccgaegawgcygdAgdaicciaeiawicyidAidakcckbekagkaukbwkcykdAkdamdcmbembgmaimasmbumbwmdymdAm"

class Mahjongg_Arrow(AbstractMahjonggGame):
    L = "0aaaaqbaacaccascaqdaudaaeaceaeeageaieakeameaoeaseaweaqfaufayfaagacgaegaggaigakgamgaogasgawgaAgaCgaqhauhayhaaiaciaeiagiaiiakiamiaoiasiawiaqjaujaakackaskaqlaamhbchrdhbehdehfehhehjehlehnehpehtehrfhvfhbghdghfghhghjghlghnghpghtghxghrhhvhhbihdihfihhihjihlihnihpihtihrjhbkoceoeeogeoieokeomeooeoqeosfocgoegoggoigokgomgoogoqgougoshocioeiogioiiokiomiooioqivfevhevjevlevnevpevfgvhgvjgvlgvngvpgvrgvfivhivjivlivnivpiCkeCmeCoeCkgCmgCogCqgCkiCmiCoi"

class Mahjongg_ArtModerne(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaauaawaaabalcapcatcavcaxcaadaddaleapeaseauebxeaafacfalgangapgargatgavgaxgaahachaliapiasiauibxiaajadjalkapkatkavkaxkaalacmaemagmaimakmammaomaumawmhdahfahhahjahlahnahvahxahuchwchychedhldhpdhaehtehvehdfhlfhpfhaghsghughwghdhhlhhphhaihtihvihejhljhpjhukhwkhykhdmhfmhhmhjmhlmhnmhvmhxmowaoyaovcoxcozcofdokdoueoweoyeoefokfomgotgovgoehokhouiowioyiofjokjovkoxkozkowmoymvgdvjdvffvjfvlgvfhvjhvgjvjjChdCgfCifCkgCghCihChj"

class Mahjongg_Balance(AbstractMahjonggGame):
    L = "0eoaeebbgbbibbkbbmbbqbbsbbubbwbeybeoccedcydcoeccfaefcgfcwfayfcAfcogachaghawhaAhcoiaajacjaejagjaijaujawjayjaAjaCjcokadlaflaxlazlcomagoaioakoamoaooaqoasoauoawohbjhdjhfjhhjhvjhxjhzjhBjjeljylhhoijojloknokpojroitohvoocjoejogjowjoyjoAjvdjvfjvxjvzjCejCyj"

class Mahjongg_Bat(AbstractMahjonggGame):
    L = "0ecaeAaaabalbanbapbarbaCbcccaecayccAcaadandapdaCdcceaeebgeaieauebweayecAeaafanfapfaCfbcgbegaggbigakgasgbugawgbygbAgaahamhbohaqhaChbcibeiagibiiakiasibuiawibyibAiaajamjbojaqjaCjcckaekbgkaikakkaskaukbwkaykcAkaalaolaClccmaemaimakmasmaumaymcAmaanaCnecobkobsoeAohobhodhofhaghCghaihCi"

class Mahjongg_Beatle(AbstractMahjonggGame):
    L = "0aeaagaauaawaaicakcamcaocaqcascaeeageaieakeameaoeaqeaseaueadgafgahgajgalgangapgargatgavgaeiagiaiiakiamiaoiaqiasiauiaikakkamkaokaqkaskaemagmaumawmhhbhtbhjchlchnchpchrchdehfehhehjehlehnehpehrehteiegiggiigikgimgiogiqgisghughdihfihhihjihlihnihpihrihtihjkhlkhnkhpkhrkhhlhtloceogeoieokeomeooeoqeoseociogioiiokiomiooioqiosivbdvhevjevlevnevpevrevfgvhgvjgvlgvngvpgvrgvhivjivlivnivpivrivbjCaaCacCggCigCkgCmgCogCqgCakCam"

class Mahjongg_BigHole(AbstractMahjonggGame):
    L = "0daadcadeadgadiadkadmadoaaaccccdecdgcdicdkccmcaocaaeccedeedkecmeaoeaagccgdegdkgcmgaogaaiccideidgidiidkicmiaoidakdckdekdgkdikdkkdmkdok"

class Mahjongg_Bizarre(AbstractMahjonggGame):
    L = "0aaaaGadkbdmbdobdqbdsbdubdwbdkdcmdcodcqdcsdcuddwddkfcmfbofbqfbsfcufdwfdkhcmhbohaqhbshcuhdwhakjbmjcojdqjcsjbujawjaklbmlcolcqlcslbulawlaknbmnbonbqnbsnbunawnakpampaopaqpaspaupawpaaqaGq"

class Mahjongg_Boat(AbstractMahjonggGame):
    L = "0alaapaataajcblcapcbtcavcahebjecleapectebveaxeafgbhgcjgdlgapgdtgcvgbxgazgadibfichidjieliapietidvicxibziaBiapkaambcmbembgmbimbkmbmmbombqmbsmbumbwmbymbAmbCmaEmadobfobhobjoblobnobpobrobtobvobxobzoaBoaiqbkqbmqboqbqqbsqbuqawq"

class Mahjongg_Bug(AbstractMahjonggGame):
    L = "0bhabnabtaajbapbavbcadaidakdamdaodaqdasdaudawdaceayeagfbifbkfbmfbofbqfbsfbufbwfaAfdegaygbchbghcihckhcmhcohcqhcshcuhbwhaAhdeiayiagjbijbkjbmjbojbqjbsjbujbwjaAjackaykcalailaklamlaolaqlaslaulawlajnapnavnbhobnobtohyhojfolfonfopforfotfovfojjoljonjopjorjotjovjvjhvlhvnhvphvrhvthCkhCmhCohCqhCsh"

class Mahjongg_Butterfly(AbstractMahjonggGame):
    L = "0dmadqaaabaebaybaCbagccocawcaadaedaidaudaydaCdaceageakedoeaseaweaAeaafaefbifamfaqfbufayfaCfacgaggbkgeogbsgawgaAgaahaehbihbmhbqhbuhayhaChaciagibkieoibsiawiaAiaajaejbijamjaqjbujayjaCjackagkakkeokaskawkaAkaalaelailaulaylaClacmagmeomawmaAmaanaenaynaCncoohgdhwdheehyehcfhgfhwfhAfhaghCghaihCihcjhgjhwjhAjhekhykhglhwl"

class Mahjongg_Castle(AbstractMahjonggGame):
    L = "0eaaccaceacgaciackaemacacaccaecagcaicakccmcdaeaceaeeageaieakedmeaoecagacgaegaggaigakgcmgbogaqgdaiaciaeiagiaiiakidmiaoicakackaekagkaikakkcmkeamccmcemcgmcimckmemmhddhfdhhdhjdhdfhffhhfhjfhdhhfhhhhhjhhdjhfjhhjhjjoeeogeoieoegoggoigoeiogioiivffvhfvfhvhhCgg"

class Mahjongg_CatAndMouse(AbstractMahjonggGame):
    L = "0cfabhacjablacnabpacrabtacBacFabdbbvbbbcbxcbBccDcbFcahdajdaldbzdbaecBebDecFeahfajfalfbagahhajhalhbuhbBhbbibsibwibFibqjbBjbckbokbxkbFkcelbglcilbklcmlbsmbwmbunbAocCocEocGohiehkehighkgohdojdoldohfojfolfohhojholhoBkoFloAnvievkevigvkgvBlvFmCjdChfClfCjh"

class Mahjongg_Ceremonial(AbstractMahjonggGame):
    L = "0bcabeaajaalaanaapaaraataavabAabCabdcbfcbzcbBcaadapdaEdbeebgeanearebyebAeaafbifbkfapfbufbwfaEfbmgbsgaahaphaEhbmibsiaajbijbkjapjbujbwjaEjbekbgkankarkbykbAkaalaplaElbdmbfmbzmbBmbcobeoajoaloanoapoaroatoavobAobCohkahmahoahqahsahuahaehoehqehEehagipghEghaiipihEihakhokhqkhEkhkohmohoohqohsohuoonaopaoraopeoahoEhopkonoopoorovph"

class Mahjongg_Checkered(AbstractMahjonggGame):
    L = "0baabCaacbbebagbbibakbbmbaobbqbasbbubawbbybaAbbcdaedbgdaidbkdamdbodaqdbsdaudbwdaydbAdacfbefagfbifakfbmfaofbqfasfbufawfbyfaAfbchaehbghaihbkhamhbohaqhbshauhbwhayhbAhacjbejagjbijakjbmjaojbqjasjbujawjbyjaAjbclaelbglailbklamlbolaqlbslaulbwlaylbAlacnbenagnbinaknbmnaonbqnasnbunawnbynaAnbaobCo"

class Mahjongg_Chip(AbstractMahjonggGame):
    L = "0aeaaiaamaaqaatabecbgcbicbkcbmcbocbqcbscbucbwcaadbcdbydaAdbeecgecieckecmecoecqecsecuebweaagbcgbegdggbigakgamgaogaqgbsgdugbwgbygaAgbeicgiciickicmicoicqicsicuibwiaajbcjbyjaAjbekbgkbikbkkbmkbokbqkbskbukbwkaemaimammaqmaum"

class Mahjongg_Columns(AbstractMahjonggGame):
    L = "0egaaiaakaamaaoaaqaasaauaewaaebaybagcaicaocaucawceadbcdaedaydbAdeCdageekeameaoeaqeeseawebafbCfaggakgasgawgaahamheohaqhaChagiakiasiawibajbCjagkdkkamkaokaqkeskawkealbclaelaylbAleClagmaimaomaumawmaenaynegoaioakoamoaooaqoasoauoewohgfhwfjghjwhhgjhwj"

class Mahjongg_Crown(AbstractMahjonggGame):
    L = "0baabcabeabgabkabmaboabqabsabwabyabAabCabacaccaecbgcbkcamcbocaqcbscbwcaycaAcbCcbaeaeebgebkeameaqebsebweayebCebagaegbggbigbkgamgaqgbsgbugbwgaygbCgbaiaeiagiaiiakiamiaqiasiauiawiayibCibakbCkbamacmaemagmaimakmbomasmaumawmaymaAmbCmbaobcobeobgobiobkoamobooaqobsobuobwobyobAobCo"

class Mahjongg_Cupola(AbstractMahjonggGame):
    L = "0aiaakaamaaoaaqaasaagbaubaecawcacdaydabfeofazfaahajhalhanhapharhathaAhaajeojaAjablazlacnaynaeoawoagpaupaiqakqamqaoqaqqasqhjbhlbhnbhpbhrbhhchtchfdhvdhdehxehcghyghkhhmhhohhqhhshhbihzihckhykhdmhxmhfnhvnhhohtohjphlphnphpphrpokcomcoocoqcoidosdogeoueoefowfodholhonhophorhoxhodjoxjoelowlogmoumoinosnokoomoooooqovldvndvpdvjevrevhfvtfvfgvvgvmhvohvqhveivwivfkvvkvhlvtlvjmvrmvlnvnnvpnCifCsfCggCugCnhCphCfiCviCgkCukCilCsl"

class Mahjongg_DeepWell(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaaccccceccgccicckccmcaocaaecceeeeegeeieekecmeaoeaagccgeegekgcmgaogaaiccieeiekicmiaoiaakcckeekegkeikekkcmkaokaamccmcemcgmcimckmcmmaomacoaeoagoaioakoamo"

class Mahjongg_Dragon(AbstractMahjonggGame):
    L = "0bgaaiaegceicdkccmcbocbqcbscbucawcaycaceaeeageaieakebmeboeaqeaseaueaweayeadgbfgahgajgalgangapgaEgayhaChaaiaciaeiagiaiiakiamiaoiaqiasiauiaAiaEiaCjabkadkafkahkajkalkaEkaamacmaemagmbimakmaaoacobeoagoaiockoamoixchdejhejigkkgjmghEhhbihdikhikjijliiniipihrihtihCihEjhckhgkhkkhbmhfmhboihopneocioEiobn"

class Mahjongg_Dude(AbstractMahjonggGame):
    L = "0bfabtabhbbjbblbbrbaBbatcavcaxcazcaedagdbldbndbpdbrdacebjebueayeaAeaCeaafbhfcmfcofcqfcsfbwfaEfbfgckgcugbygcphbAhbeickicuicpjbBjcjkclkcnkcrkctkcvkcjmclmcnmcrmctmcvmcpncjocvockqcmqcoqcqqcsqcuq"

class Mahjongg_Eagle(AbstractMahjonggGame):
    L = "0cmadoacqaasbbmcbocaedagdaudawdbcebieakebmeboeaqebsebyeaefagfaufawfbcgbigakgbmgbogaqgbsgbygaehaghauhawhaaiacibmiboiayiaAibejbwjaakackbmkbokaykaAkaambkmanmbqmaAmcioclocpocsoheehgehuehweheghgghughwghbihzihbkhzkomdoododeofeoheoteoveoxeomfoofodgofgohgotgovgoxgomhoohobjomjoojozjvndveevgevuevwevnfvegvggvugvwgvnhvnjCfeCveCfgCvg"

class Mahjongg_Enterprise(AbstractMahjonggGame):
    L = "0agaaiaakaamaaoaaqaasaauaawaayaaacbccbecbgcbicbkcbmcbocbqcbscbucbwcbycbAcbCcaEcdqedogdmhaAiaajbcjcejdgjeijekjemjeojcqjayjaCjaAkhhaijailainaipairaitaivahxaiAjodcofcohcojcolconcopcorcotcovcoxcozcoBcvkavmavoavqavsavecvgcvicvkcvmcvocvqcvscvucvwcvycChcCjcClcCncCpcCrcCtcCvc"

class Mahjongg_Eye(AbstractMahjonggGame):
    L = "0amaaoaakbaqbaicamcaocascagdakdaqdaudaeeaieameaoeaseaweacfagfakfaqfaufayfaagaegaigamgaogasgawgaAgachaghakhaqhauhayhaeiaiiamiaoiasiawiagjakjaqjaujaikamkaokaskaklaqlammaomhlbhobhjchqchhdhldhodhsdhfehjehqehuehdfhhfhlfhofhsfhwfhfghjghqghughdhhhhhlhhohhshhwhhfihjihqihuihhjhljhojhsjhjkhqkhllhololcoocojdoqdoheoleooeoseoffojfoqfoufohgolgoogosgowgofhojhoqhouhohioliooiosiojjoqjolkookvldvodvjevqevhfvlfvofvsfvfgvjgvqgvhhvlhvohvshvjivqivljvoj"

class Mahjongg_F15Eagle(AbstractMahjonggGame):
    L = "0aobaqbasbaubbEcbGcandapdardatdalebDebFeajfanfapfarfalgatgavgaxgazgaBgaDgabhadhafhahhajhanhapharhaliatiaviaxiaziaBiaDiajjanjapjarjalkbDkbFkanlaplarlatlbEmbGmaonaqnasnaunhpahrahtahvahochqchschuchmehoehqehsehifhkfhmghoghqghsghughwghyghAghCgiahichjehjghjihjkhhmihoihqihsihuihwihyihAihCihijhkjhmkhokhqkhskhomhqmhsmhumhpohrohtohvoozfoBfoDfomhozjoBjoDjvAfvCfvAjvCjCBfCDfCfhChhCjhCBjCDj"

class Mahjongg_Farandole(AbstractMahjonggGame):
    L = "0beabgabmaboabqabwabyabcbbibbkbbsbbubbAbafcaxcbbdbBdckecmecqecsebbfbgfcifcufbwfbBfbegbygbahbchajhblhcnhcphbrhathbAhbChbeibyibbjbgjcijcujbwjbBjckkcmkcqkcskbblbBlafmaxmbcnbinbknbsnbunbAnbeobgobmoboobqobwobyo"

class Mahjongg_Fish(AbstractMahjonggGame):
    L = "0afaajaasaauaawabhbaobaqbaybaccamcbscbucbwcaAcakdbodbqdaydaCdaceaeeaiebmebsebuebweaEeagfbkfbofbqfayfaCfacgaegaigbmgbsgbugbwgaAgaEgakhbohbqhbyhaChaciamibsibuibwiaAiaojaqjayjahkaskaukawkbjlcemalmbcndgnbCnaaoeioaqoasodAoaEodkpbopbupdypcmqcwqhcdhcfhefhifhchoreoteolfonfopfovforgotgovhoxh"

class Mahjongg_FivePyramids(AbstractMahjonggGame):
    L = "0aaaacaaeaagaayaaAaaCaaEaaacaccaecagcapcaycaAcaCcaEcaaeaceaeeageapeayeaAeaCeaEeaagacgaegaggangapgargaygaAgaCgaEgalhathaniapiariaakackaekagkapkaykaAkaCkaEkaamacmaemagmapmaymaAmaCmaEmaaoacoaeoagoayoaAoaCoaEoaaqacqaeqagqayqaAqaCqaEqhbbhdbhfbhzbhBbhDbhbdhddhfdhpdhzdhBdhDdhbfhdfhffipfhzfhBfhDfhnhhphhrhipjhblhdlhflhplhzlhBlhDlhbnhdnhfnhznhBnhDnhbphdphfphzphBphDpoccoecoAcoCcoceoeeoAeoCeoohoqhocmoemoAmoCmocooeooAooCovddvBdvphvdnvBn"

class Mahjongg_FivePyramids2(AbstractMahjonggGame):
    L = "0aoaaabacbaebagbawbaybaAbaCbbocaadacdaedagdaidakdamdaqdasdaudawdaydaAdaCdcoeaafacfaefagfawfayfaAfaCfamgdogaqgadhazhagibiickidmidoidqicsibuiawiadjazjamkdokaqkaalaclaelaglawlaylaAlaClcomaanacnaenagnainaknamnaqnasnaunawnaynaAnaCnbooaapacpaepagpawpaypaApaCpaoqhbchdchfchxchzchBchbehdehfehxehzehBehbmhdmhfmhxmhzmhBmhbohdohfohxohzohBoocdoedoydoAdocnoenoynoAn"

class Mahjongg_Flowers(AbstractMahjonggGame):
    L = "0baaccaceabgaakabmaboaaqaauabwabyaaAadacdgcckccqccuccAcbaecceceebgeakebmeboeaqeauebwebyeaAeadgangaxgafhahhajhalhapharhathavhadianiaxiaakbckbekagkakkbmkbokaqkaukbwkbykaAkcamcgmckmcqmcumcAmaaobcobeoagoakobmobooaqoauobwobyoaAoonaoxaoneoxeodkonkoxkodoonooxovdavde"

class Mahjongg_FlyingDragon(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaaqaasaauaawaayaagcbicbkcbmcbocbqcbscaucaeeagebieckecmecoecqebseaueaweacgaegaggbigckgdmgdogcqgbsgaugawgaygaahaAhaChaciaeiagibiickidmidoicqibsiauiawiayiaekagkbikckkcmkcokcqkbskaukawkagmbimbkmbmmbombqmbsmaumacoaeoagoaioakoamoaooaqoasoauoawoayoCnh"

class Mahjongg_FortressTowers(AbstractMahjonggGame):
    L = "0faaecadeacgabiabkacmadoaeqafsaeacaccagcaicakcamcaqcescdaeaceageaieakeameaqedsedagacgaggaigakgamgaqgdsgeaiaciagiaiiakiamiaqiesifakeckdekcgkbikbkkcmkdokeqkfskhjchjehjghji"

class Mahjongg_FullVision(AbstractMahjonggGame):
    L = "0aaaaiaamaaoaaqaasaawaaEaacbaebagbaybaAbaCbaacaicamcaocaqcascawcaEcacdaedagdaydaAdaCdaaeaieaweaEeaefamfasfaAfaggaigakgaugawgaygaehamhashaAhagiaiiakiauiawiayiaejamjasjaAjaakaikawkaEkaclaelaglaylaAlaClaamaimaomaqmawmaEmacnaenagnamnasnaynaAnaCnaaoaioaooaqoawoaEohpahbbhhbhnbhrbhxbhDbhdchfchpchzchBchbdhhdhxdhDdhfghlghtghzghhhhjhhvhhxhhfihlihtihzihblhhlhxlhDlhdmhfmhzmhBmhbnhhnhnnhpnhrnhxnhDnooboqboccogcoycoCcoghokhouhoyhocmogmoymoCmvpb"

class Mahjongg_FullVision2(AbstractMahjonggGame):
    L = "0aaaacaafaahaakaamaapaaraauaawaazaaBaaacaccafcahcakcamcapcarcaucawcazcaBcaaeaceafeaheakeameapeareaueaweazeaBeaagacgafgahgakgamgapgargaugawgazgaBgaajacjaejagjakjamjapjarjavjaxjazjaBjaalaclaelaglaklamlaplarlavlaxlazlaBlaeoagoaioakoamoapoaroatoavoaxohbbhgbhlbhqbhvbhAbhadhcdhfdhhdhkdhmdhpdhrdhudhwdhzdhBdhbfhgfhlfhqfhvfhAfhdjhyjhbkhfkhkkhmkhpkhrkhwkhAkhdlhylobcogcolcoqcovcoAcobeogeoleoqeoveoAeockoekolkoqkoxkozkvbdvgdvldvqdvvdvAdvdkvyk"

class Mahjongg_Future(AbstractMahjonggGame):
    L = "0cgaaiaakaamaboaaqaasaauacwaagccicakcamcbocaqcasccucawcaeeageaiebkebmeboebqebseaueaweayeacfaAfaagcegcggdigdkgdmgdogdqgdsgdugcwgcygaCgachaAhaeiagiaiibkibmiboibqibsiauiawiayiagkcikakkamkbokaqkaskcukawkcgmaimakmammbomaqmasmaumcwmhcghAgoneopeoniopiClgCngCpgCrg"

class Mahjongg_Garden(AbstractMahjonggGame):
    L = "0adaafaaoaaqaazaaBaaabaibalbatbawbaEbaccaecagcancapcarcaycaAcaCcaadaidaldatdawdaEdaceaeeageaneapeareayeaAeaCeaafaifalfatfawfaEfachaehaghanhapharhayhaAhaChaajaijaljatjawjaEjackaekagkankapkarkaykaAkaCkaalailallatlawlaElacmaemagmanmapmarmaymaAmaCmaanainalnatnawnaEnadoafoaooaqoazoaBoheahpahAahcdhedhgdhndhpdhrdhydhAdhCdhdhhfhhohhqhhzhhBhhclhelhglhnlhplhrlhylhAlhClheohpohAooddofdoodoqdozdoBdoehophoAhodloflooloqlozloBlvedvpdvAdvelvplvAl"

class Mahjongg_Gayles(AbstractMahjonggGame):
    L = "0dcaceabgaaiaakaamaaoaaqaasabuacwadyaagcbicckccmccoccqcbscaucakebmeboeaqeacgaegaggbigbkgbmgbogbqgbsgaugawgaygaahaAhaciaeiagibiibkibmiboibqibsiauiawiayiakkbmkbokaqkagmbimckmcmmcomcqmbsmaumdcoceobgoaioakoamoaooaqoasobuocwodyoojholhonhophorhvncvmhvohvnmCnh"

class Mahjongg_Glade(AbstractMahjonggGame):
    L = "0aaaacaaCaaEaaacaccaCcaEcahdejdcldcndbpdcrdctdevdaxddhfcjfblfbnfbpfbrfbtfcvfdxfchhbjhblhanharhbthbvhcxhdhjcjjbljbnjbpjbrjbtjcvjdxjahlejlcllcnlbplcrlctlevlaxlaamacmaCmaEmaaoacoaCoaEohbahDahbchDchbmhDmhbohDoobboDbobnoDn"

class Mahjongg_HForHaga(AbstractMahjonggGame):
    L = "0aaaacaaeaagaakaamaaoaaqaaacaccaecagcakcamcaocaqcaaeaceaeeageakeameaoeaqeaifaagacgaegaggakgamgaogaqgaihaaiaciaeiagiakiamiaoiaqiaijaakackaekagkakkamkaokaqkaamacmaemagmakmammaomaqmaaoacoaeoagoakoamoaooaqohbbhdbhfbhlbhnbhpbhbdhddhfdhldhndhpdhbfhdfhffhlfhnfhpfhhghjghbhhdhhfhhlhhnhhphhhihjihbjhdjhfjhljhnjhpjhblhdlhflhllhnlhplhbnhdnhfnhlnhnnhpnoccoecomcoococeoeeomeooeocgoegomgoogoghoihokhocioeiomiooiockoekomkookocmoemommoomvddvndvdlvnl"

class Mahjongg_HForHagaTraditional(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaaqaasaauaawaayabgcbicakcamcaocaqcbscbucaeebgebieakeameaoeaqebsebueaweacgaegbggbigbkgbmgbogbqgbsgbugawgaygaahaAhaciaeibgibiibkibmiboibqibsibuiawiayiaekbgkbikakkamkaokaqkbskbukawkbgmbimakmammaomaqmbsmbumacoaeoagoaioakoamoaooaqoasoauoawoayoklcknckpchdhhxhklmknmkpm"

class Mahjongg_Helios(AbstractMahjonggGame):
    L = "0eaadcaduaewadacbccbucdwcbaeaceaeeaiedkedmeaoeaseauebwebagacgaegaggdigdogaqgasgaugbwgblhbaiaciaeiagidiidoiaqiasiauibwibakackaekaikdkkdmkaokaskaukbwkdambcmbumdwmeaodcoduoewohchhehhghhqhhshhuhCleCihCohClk"

class Mahjongg_HighAndLow(AbstractMahjonggGame):
    L = "0eaadcaceabgaaiabkacmadoaeqadaccccdecagcbicckcbmceocdqccaebceeeebgeciedkeamedoecqebagacgdegcggdigekgbmgcogbqgaaibciceidgieiidkicmiboiaqiaekagkbikakkamkahmajmhim"

class Mahjongg_Hourglass(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaqaasaauaawaayaaacamcaycacdawdaaeaeeameaueayeacfagfasfawfaagaegaigamgaqgaugaygachaghbkhbohashawhaaiaeiaiiamiaqiauiayiacjagjasjawjaakaekamkaukaykaclawlaamammaymaaoacoaeoagoaioakoamoaooaqoasoauoawoayohabhmbhybhadhmdhydhcehwehafhefhufhyfhcghgghsghwghahhehhihhqhhuhhyhhcihgihsihwihajhejhujhyjhckhwkhalhmlhylhanhmnhynoacoycoaeoyeoagoygoaioyioakoykoamoymvadvydvafvyfvahvyhvajvyjvalvylCaeCyeCagCygCaiCyiCakCyk"

class Mahjongg_Inca(AbstractMahjonggGame):
    L = "0aoaaqaaibakbambasbaubawbbocbqcaidbkdbmdbsdbudawdcoecqeaifbkfcmfcsfbufawfaagacgdogdqgaCgaEgahhbjhclhcthbvhaxhaaiacidoidqiaCiaEiahjbjjcljctjbvjaxjaakackdokdqkaCkaEkailbklcmlcslbulawlcomcqmainbknbmnbsnbunawnboobqoaipakpampaspaupawpaoqaqqhbihDiCphCpj"

class Mahjongg_InnerCircle(AbstractMahjonggGame):
    L = "0aaaacaayaaAaaaceccceccgcbicbkcamcaocbqcbsccuccwceycaAcccecyedgfcifbkfbqfcsfdufbcgbygaghbuhbcibyiegjdijckjbmjbojcqjdsjeujcckcykaamecmcemcgmbimbkmbqmbsmcumcwmdymaAmaaoacoakoaqoayoaAo"

class Mahjongg_Joker(AbstractMahjonggGame):
    L = "0aaaaAaadbafbahbajbalbanbapbarbatbavbaxbabdbddbfdbhdbjdbldbndbpdbrdbtdbvdbxdazdcbfbdfaffahfajfalfanfapfarfatfavfbxfczfcbhbdhafhavhbxhczhajiamiapiasicbjbdjafjavjbxjczjcblbdlaflahlajlallanlaplarlatlavlbxlczlabnbdnbfnbhnbjnblnbnnbpnbrnbtnbvnbxnaznadpafpahpajpalpanpapparpatpavpaxpaaqaAqhgghughgkhuk"

class Mahjongg_KForKyodai(AbstractMahjonggGame):
    L = "0caaccaceacmacoacqacacbcccecckcbmccoccaebceceeciebkecmecagbcgcegcggbigckgcaibcibeibgiciicakbckcekcgkbikckkcambcmcemcimbkmcmmcaobcoceockobmocoocaqccqceqcmqcoqcqq"

class Mahjongg_KForKyodaiTraditional(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaaqaasaauaawaayaagcaicakcamcaocaqcascaucaeeageaieakeameaoeaqeaseaueaweacgaegaggaigakgamgaogaqgasgaugawgaygaahaAhaciaeiagiaiiakiamiaoiaqiasiauiawiayiaekagkaikakkamkaokaqkaskaukawkagmaimakmammaomaqmasmaumacoaeoagoaioakoamoaooaqoasoauoawoayokjckrckpdkjehgfknfhufkjghghklhhuhkjihgjknjhujkjkkplkjmkrm"

class Mahjongg_Km(AbstractMahjonggGame):
    L = "0baabcabiaakaboacqacyabAabacaccbgcaicbocaqcdscdwcaycbAcdudbaeacebeeageboeaqeaseaweayebAeeufbagacgbegaggbogaqgaygbAgduhbaiacibgiaiiboiaqiayibAibakbckbikakkbokbqkbykbAkjcfhgfoabooboAboadoodoAdoafoefoofoAfoahoohoAhoajoojoAjvacvocvAcvaevoevAevagvogvAgvaivoivAiCadCodCAdCafCofCAfCahCohCAh"

class Mahjongg_Kujaku(AbstractMahjonggGame):
    L = "0bnabpabrabtabvabxablbczbaBbbhcbjcancapcarcatcavcaxcaddbfdaldazddBdaDdaheajeabfcdfaffaAfdCfaEfbahcchaehakhamhaohashaAhdChaEhabjcdjafjaAjdCjaEjahkajkadlbflallazldBlaDlbhmbjmanmapmarmatmavmaxmblncznaBnbnobpobrobtobvobxohnghpghtghjhhnihpihtioofoqfoufoihoojoqjoujvpevrevvfvhhvvjvpkvrkCwgCCgCghCwiCCi"

class Mahjongg_Labyrinth(AbstractMahjonggGame):
    L = "0caaacaaeaagaaiaakaamaaoaaqaasaauaawaayaaAaaCacEaaacbkcbocbucaEcaaebcebeebgebkeboebsebuebyebAeaEeaagbkgbygaEgaaibeibiibkiboibqibsibuibwibyibAiaEiaakbekbokbwkaEkaambembgmbimbkmbombqmbsmbwmbAmbCmaEmaaobkobwoaEocaqacqbeqdgqdkqbmqaoqaqqasqauqbwqdyqdCqbEq"

class Mahjongg_Lion(AbstractMahjonggGame):
    L = "0bdbbfbcjbclbawbaybbbcbhcaucaAccjdcldasdaCdbaeaqecvfczfaDfbbgapgaEhcbiceichickiaoicxiaFjcckcfkcikclkbokcwkcykbulbAlaElcbmcemchmckmcnmbqmaDnccocfocioclocooaroatoavoaxoazoaBohvahxahzahtbhBbhrchDdhpehEfhoghFhhnihGjhFlhEnhsohCohuphwphyphApwkc"

class Mahjongg_Lost(AbstractMahjonggGame):
    L = "0afaaxaabbadbahbajbblbbnbbpbbrbatbavbazbaBbafcaxcabdaddbkdcodbsdazdaBdbiebmebqebueaafacfaefbgfdofbwfayfaAfaCfaahaehbghcihckhdmhdohdqhcshcuhbwhayhaChaajacjaejbgjdojbwjayjaAjaCjbikbmkbqkbukabladlbklcolbslazlaBlafmaxmabnadnahnajnblnbnnbpnbrnatnavnaznaBnafoaxoombooboqbomnoonoqn"

class Mahjongg_Maya(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaaqaasaauaawaayaaacaccaecagcaicaqcascaucawcaycaaeaceaeeageaieakeameaoeaqeaseaueaweayeaigakgamgaogaqgaiiakiamiaoiaqiaakackaekagkaikakkamkaokaqkaskaukawkaykaamacmaemagmaimaqmasmaumawmaymaaoacoaeoagoaioaqoasoauoawoayohcbhebhgbhsbhubhwbhcdhedhgdhsdhudhwdhkfhmfhofhkhhmhhohhkjhmjhojhclhelhglhslhulhwlhcnhenhgnhsnhunhwnoccoecogcoscoucowcolfonfolhonholjonjocmoemogmosmoumowmvdcvfcvtcvvcvmfvmhvmjvdmvfmvtmvvmCecCucCmgCmiCemCum"

class Mahjongg_Mesh(AbstractMahjonggGame):
    L = "0baabcabeabiabkabmabqabsabuabyabAabCabacbecbicbmcbqcbucbycbCcbaebcebeeagebiebkebmeaoebqebsebueawebyebAebCeaegbigbmgbqgbugaygbaibcibeiagibiibkibmiaoibqibsibuiawibyibAibCibakbekbikbmkbqkbukbykbCkbambcmbembimbkmbmmbqmbsmbumbymbAmbCm"

class Mahjongg_Moth(AbstractMahjonggGame):
    L = "0baaccaceabgaanaapaarabyacAacCabEaaibawbbccagcakccpcaucaycbCcaidamdasdawdadeakeboebqeaueaBeamfasfacgaegahgajgbogbqgavgaxgaAgaCgamhashadiakiboibqiauiaBiaijamjasjawjbckagkakkcpkaukaykbCkailawlccmcembgmbpmbymcAmcCmbanbEnhoahqahichwchmehsehdghighwghBghmihsihikhwkopaoneopeoreppgoniopiorivdavBavoevqevoivqivdmvBmCpeCpi"

class Mahjongg_NForNamida(AbstractMahjonggGame):
    L = "0caaccaceacgacqacsacuacacbccbecbgcbiccqcbsccuccaebcebeebgebiebkecqebsecuecagbcgcegbigbkgbmgcqgbsgcugcaibciceibkibmiboibqibsicuicakbckcekbmkbokbqkbskcukcamccmcemcomcqmcsmcum"

class Mahjongg_NForNamidaTraditional(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaaqaasaauaawaayacgcaicakcbmccoccqcasccucaeecgecieakeameaoeaqeasecueaweacgaegcggcigakgcmgaogaqgasgcugawgaygaahaAhaciaeicgiaiiakicmiaoicqiasicuiawiayiaekcgkaikakkamkaokcqkaskcukawkcgmaimckmcmmbomaqmasmcumacoaeoagoaioakoamoaooaqoasoauoawoayoikfikhiohiojisjisl"

class Mahjongg_NaokiHagaTraditional(AbstractMahjonggGame):
    L = "0acaaeaagaaiaakaamaaoaaqaasaauaawaayadgcaicakcdmcaocaqcascaucaeedgeaiedkedmeaoecqecseaueaweacgaegdggaigakgdmgaogaqgasgaugawgaygaahaAhaciaeiagiaiiakiamidoiaqiasiduiawiayiaekagkcikckkamkdokdqkdskdukawkagmaimakmammdomaqmasmdumacoaeoagoaioakoamoaooaqoasoauoawoayojidvrevjk"

class Mahjongg_NewLayout(AbstractMahjonggGame):
    L = "0aeaagaaiaakabpaauaawaayaaAaaccaCcahdajdavdaxdaaeacealeateaCeaEeanfarfaagacgahgapgaCgaEganharhaaiacialiatiaCiaEiahjajjavjaxjackaCkaemagmaimakmbpmaumawmaymaAmhfahhahjahvahxahzahcdhidhwdhCdhkehuehafhcfhmfhofhqfhsfhCfhEfhahhchhmhhohhqhhshhChhEhhkihuihcjhijhwjhCjhfmhhmhjmhvmhxmhzmogaoiaowaoyaoceojeoveoCeolfotfoagocgongopgorgoCgoEgolhothociojiovioCiogmoimowmoymvhavxavcfvkfvufvCfvmgvogvqgvsgvchvkhvuhvChvhmvxmCcgClgCngCpgCrgCtgCCg"

class Mahjongg_Order(AbstractMahjonggGame):
    L = "0afaahaajaalaanaapaaraataaabaybaicakcamcaocaqcbadacdaedaudawdbydakebmeaoecafbcfaefaufbwfcyfaggaigakgbmgaogaqgasgcahcchbehbuhcwhcyhagiaiiakibmiaoiaqiasicajbcjaejaujbwjcyjakkbmkaokbalaclaelaulawlbylaimakmammaomaqmaanaynafoahoajoaloanoapoaroatohgahiaikaimaioahqahsahlchnchghhihhkhhohhqhhshhlmhnmhgohioikoimoioohqohsoomcpmhomm"

class Mahjongg_Pattern(AbstractMahjonggGame):
    L = "0aaaacaafaahaakaamaapaaraauabwabzaaBaaacaccafcahcakcamcapcarcbuccwcczcbBccafacfaffchfckfcmfapfarfcufawfazfcBfaahcchcfhahhakhamhcphcrhcuhawhazhcBhaakackafkahkakkcmkcpkarkcukcwkczkcBkaamacmafmahmckmammapmcrmaumcwmczmaBmibailaifbihbibciqciqfilhialihl"

class Mahjongg_Phoenix(AbstractMahjonggGame):
    L = "0aaaacaapaaraaEaaGaaebatbaCbaacagcapcarcaAcaGcaidaydakeboebqebseaweaafacfaefamfaufaCfaEfaGfaggbpgbrgaAgaahaihamhauhayhaGhaeiakicpicriawiaCiaajamjaujaGjbpkbrkaclaelaglailamlaulaylaAlaClaElakmbpmbrmawmacnafnamnaunaBnaEnaioaooasoayoacpafpaBpaEpakqawqhbbhFbhdchDchfdhBdhhehzehjfhxfhdghlghvghDghfhhBhhhihnihtihzihjjhxjhdkhlkhvkhDkhflhnlhtlhBlhhmhzmhjnhxnhlohvohnphtponfppfprfotfoplorlvqivqlCqf"

class Mahjongg_Portal(AbstractMahjonggGame):
    L = "0accagcawcaAcaedaydaceageaweaAeamgaqgamiaqiackagkawkaAkaelaylacmagmawmaAmhbbhdbhfbhhbhvbhxbhzbhBbhbdhhdhvdhBdhbfhdfhffhhfhlfhnfhpfhrfhvfhxfhzfhBfhlhhrhhbjhdjhfjhhjhljhnjhpjhrjhvjhxjhzjhBjhblhhlhvlhBlhbnhdnhfnhhnhvnhxnhznhBnoaaocaoeaogaoiaouaowaoyaoAaoCaoacoicoucoCcoaeoieokeomeooeoqeoseoueoCeoagocgoegoggoigokgosgougowgoygoAgoCgoaiocioeiogioiiokiosiouiowioyioAioCioakoikokkomkookoqkoskoukoCkoamoimoumoCmoaoocooeoogooioouoowooyooAooCo"

class Mahjongg_Rocket(AbstractMahjonggGame):
    L = "0amaaoaaqaazaaBaaDaakbaicamcaocaqcascaxcazcaBcaDcagdakdaudaeeaieameaqeaseaweayeacfagfakfaofaufaBfaegasgawgaygaahbchbghbihbkhbmhbohcqhauhaAhaChaeiasiawiayiacjagjakjaojaujaBjaekaikamkaqkaskawkaykaglaklaulaimammaomaqmasmaxmazmaBmaDmaknamoaooaqoazoaBoaDohnahpahlbhBbhjchnchpchhdhsdhfehxehdfhsfhughehhshiwhhyhhuihdjhsjhfkhxkhhlhslhjmhnmhpmhlnhBnhnohpoonbopbosgodhofhohhojholhonhouhosionnopnvobvehvghvihvshvonCfh"

class Mahjongg_Scorpion(AbstractMahjonggGame):
    L = "0avaacbaebagbaibaacaxcazcagdaidakdaoeaseayeaAeaafacfaefagfaifakfcmgaogcqgasgcugawgbygbAgckhciidmiaoicqiasiduiawibyibAickjcmkaokcqkaskcukawkbykaalaclaelaglailaklaomasmawmagnainaknaaoacpaepagpaiphdbhfbhhbhwbhbchychhdhzehbfhdfhffhhfhjfhofhsfhohhshhwhhojhsjhwjhblhdlhflhhlhjlholhslhwlhhnhbohdphfphhpoogosgoyhooiosiowioyjookoskvohvqhvshvojvqjvsj"

class Mahjongg_ScrewUp(AbstractMahjonggGame):
    L = "0ciackacmabgbbobcecbicbkcbmccqcbgdbodcceceeakeamecqecsebgfbofccgcegakgamgcqgcsgbghbohcciceiaiiakicqicsibgjbojcckcekaikakkcqkcskbglbolcembimbkmbmmcqmbgnbonciockocmoilfikhijjvbfvtfvbhvthvbjvtjCafCufCahCuhCajCuj"

class Mahjongg_Seven(AbstractMahjonggGame):
    L = "0aaaacaafaahaakaamaapaaraauaawaazaaBaaEaaGaaacaccafcahcakcamcapcarcaucawcazcaBcaEcaGcaaeaceafeaheakeameapeareaueaweazeaBeaEeaGeaagacgafgahgakgamgapgargaugawgazgaBgaEgaGgaaiaciafiahiakiamiapiariauiawiaziaBiaEiaGiaakackafkahkakkamkapkarkaukawkazkaBkaEkaGkaamacmafmahmakmammapmarmaumawmazmaBmaEmaGmaaoacoafoahoakoamoapoaroauoawoazoaBoaEoaGoaaqacqafqahqakqamqapqarqauqawqazqaBqaEqaGqhqchlehvehggiqghAghbihlihvihFihgkiqkhAkhlmhvmhqo"

class Mahjongg_SevenPyramids(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaoaaqaayaaAaaCaaEaaacaccaecagcaocaqcaycaAcaCcaEcaaeaceaeeageayeaAeaCeaEeaagacgaegaggangapgargaygaAgaCgaEganiapiariaakackaekagkankapkarkaykaAkaCkaEkaamacmaemagmaymaAmaCmaEmaaoacoaeoagoaooaqoayoaAoaCoaEoaaqacqaeqagqaoqaqqayqaAqaCqaEqhbbhdbhfbhpbhzbhBbhDbhbdhddhfdhzdhBdhDdhbfhdfhffhzfhBfhDfhohhqhhojhqjhblhdlhflhzlhBlhDlhbnhdnhfnhznhBnhDnhbphdphfphpphzphBphDpoccoecoAcoCcoceoeeoAeoCeopiocmoemoAmoCmocooeooAooCovddvBdvdnvBn"

class Mahjongg_Shield(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaxaaacaccaecagcaicakcamcaocbxcaaeaceaeeageaieakeameaoecxeabgadgafgahgajgalgangdxgaciaeiagiaiiakiamidxietjeBjaekagkaikakkbvkexkbzkagmaimcxmahodxohcbhebhgbhibhkbhmbhcdhedhgdhidhkdhmdhcfiefigfiifikfhmfhdhifhihhijhhlhhejigjiijhkjhglhilodbofbohbojbolboddofdohdojdoldohlvfcvhcvjcvfevhevjevggvigvhiChdChf"

class Mahjongg_Siam(AbstractMahjonggGame):
    L = "0afaazaadbahbaxbaBbacdaedagdaidandardawdaydaAdaCdaleateabfadfaffahfajfavfaxfazfaBfaDfaahachaehaghaihakhamhaohaqhashauhawhayhaAhaChaEhabjadjafjahjajjavjaxjazjaBjaDjalkatkaclaelaglailanlarlawlaylaAlaCladnahnaxnaBnafoazohddhfdhhdhxdhzdhBdhcfhefhgfhifhwfhyfhAfhCfhbhhdhhfhhhhhjhhlhhnhiphhrhhthhvhhxhhzhhBhhDhhcjhejhgjhijhwjhyjhAjhCjhdlhflhhlhxlhzlhBloedogdoydoAdodfoffohfoxfozfoBfochoehoghoihowhoyhoAhoChodjofjohjoxjozjoBjoelogloyloAl"

class Mahjongg_SpaceShip(AbstractMahjonggGame):
    L = "0afaahaajaalaanaapaaraataavaadbaxbabcancazcaaeafeaheajealeaneapeareateaveaAeadfaxfangadhaxhaniadjaxjankadlaxlanmadnaxnanohgahiahkahmahoahqahsahuahebhwbhcchychadhmdhodhAdhgehiehkehqehsehuehmfhofhdghxghnhhdihxihnjhdkhxkhnlhdmhxmhnnohaojaolaonaopaoraotaofbovbodcoxcobdozdoheojeoleoneopeoreoteqngodhoxhqniodjoxjqnkodloxlqnmviavkavmavoavqavsavgbvubvecvwcvcdvydvievkevmevoevqevsevdivxivdkvxkCnaCjeCleCneCpeCreCdjCxj"

class Mahjongg_Square(AbstractMahjonggGame):
    L = "0daadcadeadgadiadkadacdccdecdgcdicdkcdaedcedeedgediedkedagdcgdegdggdigdkgdaidcideidgidiidkidakdckdekdgkdikdkk"

class Mahjongg_Squares(AbstractMahjonggGame):
    L = "0caabcaceabgaciabkacmaboacqabsacuaaacauccddafdahdajdaldandapdcrdaaeauebdfbrfaagbggcigckgcmgbogaugcdhcrhaaibgiciickicmiboiauibdjbrjaakaukcdlaflahlajlallanlaplcrlaamaumcaobcoceobgociobkocmoboocqobsocuohidikdhmdhiliklhmlvjgvlgvjivli"

class Mahjongg_Squaring(AbstractMahjonggGame):
    L = "0caaacaceaciaakacmacqaasacuacyaaAacCaaacaecaicdkcamcaqcaucaycdAcaCccaeaceceecieakecmecqeasecuecyeaAecCecahachcehcihakhcmhcqhashcuhcyhaAhcChaajdcjaejaijamjaqjdsjaujayjaCjcalaclcelcilaklcmlcqlaslculcylaAlcCl"

class Mahjongg_Stairs(AbstractMahjonggGame):
    L = "0aoaaebaybeacdccagcaicakcbmccocbqcascaucawcdAceCcaedayddaeaieaoeauedCebefbyfaagaigaogaugaCgbchcehbghakhbmhbqhashbwhcyhbAhaaiaiiaoiauiaCibejbyjdakaikaokaukdCkaelayleamdcmagmaimakmbmmcombqmasmaumawmdAmeCmaenaynaoohechychofhahkohhChhojhemhym"

class Mahjongg_StarShip(AbstractMahjonggGame):
    L = "0eoaaabdmbdqbaCbaccckccscaAcaadbidbudaCdbceagecoeawebAeaafaefamfaqfayfaCfecgaggaigbkgdogbsgaugawgeAgaahaehamhaqhayhaChbciagicoiawibAiaajbijbujaCjackckkcskaAkaaldmldqlaCleomhachCchaehCehaghegimgiqghyghCghaihCihakhCkoadoCdoafoCfoahoChoajoCjvaevCevagvCgvaivCiCafCCfCahCCh"

class Mahjongg_StepPyramid(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaqaaacaccaecagcaicakcamcaocaqcaaeaceaoeaqeaagacgaogaqgaaiaciaoiaqiaakackaekagkaikakkamkaokaqkaamacmaemagmaimakmammaomaqmhbbhdbhfbhhbhjbhlbhnbhpbhbdhddhfdhhdhjdhldhndhpdhbfhdfhnfhpfhbhhdhhnhhphhbjhdjhfjhhjhjjhljhnjhpjhblhdlhflhhlhjlhllhnlhplpccoecogcoicokcomcpococepeepgepiepkepmeooeocgpegpmgoogocipeipgipiipkipmiooipckoekogkoikokkomkpokCffChfCjfClfCfhChhCjhClh"

class Mahjongg_Stonehenge(AbstractMahjonggGame):
    L = "0cdachackacoacracvacyacCacaccFcajeaneareavecagcFgddhdhhdlhdphdthdxhdBhcajcFjajkankarkavkcancFncdpchpckpcopcrpcvpcypcCpveavgavlavnavsavuavzavBavadvFdvafvFfvakvFkvamvFmvepvgpvlpvnpvspvupvzpvBpCehCghCihCkhCmhCohCqhCshCuhCwhCyhCAh"

class Mahjongg_SunMoon(AbstractMahjonggGame):
    L = "0dgaciabkaamabyadebbrbbBbdccbvccaddcecheckecnebDecafbtfbAfdcgdjgdlgbxgcahchhcnhdcidjidlibribDicajbvjdckchkckkcnkbAkcalbsldcmbxmdenbBndgociobkoamobuovaevagvaivakCkh"

class Mahjongg_Temple(AbstractMahjonggGame):
    L = "0baaacaaeaalaanaapaaraataaAaaCabEaaacaccalcbncbpcbrcatcaCcaEcajdavdaaeblebnebpebrebteaEeaffahfajfavfaxfazfblgbngbpgbrgbtgadhafhahhajhavhaxhazhaBhblibnibpibribtiafjahjajjavjaxjazjaakblkbnkbpkbrkbtkaEkajlavlaamacmalmbnmbpmbrmatmaCmaEmbaoacoaeoaloanoapoaroatoaAoaCobEohhghjghvghxghhihjihvihxiooeoqeokgomgoogoqgosgougokiomiooioqiosiouiookoqkvpgvpi"

class Mahjongg_Teotihucan(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaqaasaaacascaaeaseaagcggckgcogasgaaicgickicoiasiaakaskaamasmaaoacoaeoagoaioakoamoaooaqoasoajqhbbhdbhfbhhbhjbhlbhnbhpbhrbhbdhrdhbfhrfhbhhrhhbjhrjhblhrlhbnhdnhfnhhnhjnhlnhnnhpnhrnhjpoccoecogcoicokcomcoocoqcoceoqeocgoqgocioqiockoqkocmoemogmoimokmommoomoqmojovddvfdvhdvjdvldvndvpdvdfvffvhfvjfvlfvnfvpfvdhwfhvhhwjhvlhwnhvphvdjvfjvhjvjjvljvnjvpjvdlvflvhlvjlvllvnlvplvjn"

class Mahjongg_TheDoor(AbstractMahjonggGame):
    L = "0amaaoaaqaeicekcemceoceqcesceucagediedueaweaegaggdigdugawgaygaeibgidiiduibwiayiackaekcgkdikakkaskdukcwkaykaAkaamacmbemcgmdimakmasmdumcwmbymaAmaCmaaobcobeocgodioakoasoduocwobyobAoaCo"

class Mahjongg_TheGreatWall(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaqaasaauaawaayaaAaaCaaEaaacaccaecagcaicakcamcaocaqcascaucawcaycaAcaCcaEcaaeaceaeeageaieakeameaoeaqeaseaueaweayeaAeaCeaEeaagacgaegaggaigakgamgaogaqgasgaugawgaygaAgaCgaEgaaiaciaeiagiaiiakiamiaoiaqiasiauiawiayiaAiaCiaEiaakackaekagkaikakkamkaokaqkaskaukawkaykaAkaCkaEkaamacmaemagmaimakmammaomaqmasmaumawmaymaAmaCmaEmaaoacoaeoagoaioakoamoaooaqoasoauoawoayoaAoaCoaEoaaqacqaeqagqaiqakqamqaoqaqqasqauqawqayqaAqaCqaEq"

class Mahjongg_Theater(AbstractMahjonggGame):
    L = "0baaccaceabgaaiaamaaqabsacuacwabyacaccccbecagcakcbmcaocascbuccwccyccaebceaeeaiebkebmeboeaqeauebwecyebagacgaggaigakgbmgaogaqgasgawgbygcaibciaeiaiibkibmiboiaqiauibwicyicakcckbekagkakkbmkaokaskbukcwkcykbamccmcembgmaimammaqmbsmcumcwmbym"

class Mahjongg_TileFighter(AbstractMahjonggGame):
    L = "0bfaahaatabvadccbecakcbmcbocaqcbwcdycbaecceaiebkebmeboebqeasecyebAebagbigckgamgaogcqgbsgbAgcchaehaghauhawhcyhbaibiickiamiaoicqibsibAibakcckaikbkkbmkbokbqkaskcykbAkdcmbemakmbmmbomaqmbwmcymbfoahoatobvohnhonepafpAfpahpAhpajpAjonk"

class Mahjongg_Tilepiles(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaaobaqbasbaubaybaAbaCbaEbahcajcalcacdaedardatdaxdazdaBdakeameaoeaffahfaufawfayfangapgargaihakhaxhazhaqiasiauiajjaljanjaAjaCjatkavkaxkaelaglailaklaolaqlawmaymaAmaCmaEmabnadnafnahnajnhbahdahfahhahpbhrbhtbhzbhBbhDbhichkchddhsdhydhAdhlehnehgfhvfhxfhoghqghjhhyhhrihtihkjhmjhBjhukhwkhflhhlhjlhplhxmhzmhBmhDmhcnhenhgnhinocaoeaogaoqbosboAboCbojcozdomeowfopgosioljovkogloiloymoAmoCmodnofnohnvdavfavrbvBbvhlvzmvBmvenvgnCeaCAmCfn"

class Mahjongg_TimeTunnel(AbstractMahjonggGame):
    L = "0aaabcaceaegaeiaekaemacoabqaasaaacccceeceoccqcascaaecceeeeeoecqeaseaagccgeegeogcqgasgaaiccieeieoicqiasiaakbckcekegkeikekkemkcokbqkaskvcdvqdwcfwqfvchvqh"

class Mahjongg_Tomb(AbstractMahjonggGame):
    L = "0eaabcabeabgabiabkabmaboabqaesabaccccceccgccicckccmccoccqcbscaaedcebeeageaieakeameboedqeasebagccgcegeggaigakgemgcogcqgbsgdaibcibeidgiaiiakidmiboibqidsibgkaikakkbmkaimakmhjevfcvhcvjcvlcvncCgcCicCkcCmc"

class Mahjongg_TowerAndWalls(AbstractMahjonggGame):
    L = "0ekadmaeoadqaesadkccmccoccqcdscdaeecedeeegedieekecmedoecqeesedueewedyeeAedCedkgcmgcogcqgdsgekidmieoidqiesi"

class Mahjongg_TraditionalReviewed(AbstractMahjonggGame):
    L = "0acaaeaaiaakaamaaoaaqaasaawaayaagcaicbkccmccocbqcascaucaeeagebiebkecmecoebqebseaueaweacgaegbggcigckgcmgcogcqgcsgbugawgaygaahaAhaciaeibgiciickicmicoicqicsibuiawiayiaekagkbikbkkcmkcokbqkbskaukawkagmaimbkmcmmcombqmasmaumacoaeoaioakoamoaooaqoasoawoayovnfvlhwnhvphvnj"

class Mahjongg_TreeOfLife(AbstractMahjonggGame):
    L = "0ababdacfadhacjablaanaapabractadvacxabzaaBaaccaAcaadbfdajdaldandapdardatdbxdaCdaceaAeaafaefagfaifbkfbsfaufawfayfaCfacgamgaqgaAgaehaihauhayhaliboiariagjawjblkaokbrkaambcmcembgmaimclmaomcrmaumbwmcymbAmaCmacoagocloaoocroawoaAoaiqakqamqcoqaqqasqauqhoaicdimdiqdiAdhdfiffhhfixfhzfilqirq"

class Mahjongg_TwinTemples(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaaqaasaauaawaayaaAaaacakcaqcaAcamdaodaaeakeaqeaAeagfaifamfaofasfaufaagakgaqgaAgamhaohaaiakiaqiaAiaakackaekagkaikakkaqkaskaukawkaykaAkhbbhdbhfbhhbhjbhrbhtbhvbhxbhzbhbdhjdhldhpdhrdhzdhbfhffhvfhzfhbhhjhhlhhphhrhhzhhbjhdjhfjhhjhjjhrjhtjhvjhxjhzjoccoecogcoicoscoucowcoycokdoqdoceoieoseoyeocgoigosgoygokhoqhocioeiogioiiosiouiowioyivddvfdvhdvjdvrdvtdvvdvxdvdfvhfvjfvrfvtfvxfvdhvfhvhhvjhvrhvthvvhvxhCeeCgeCueCweCegCggCugCwg"

class Mahjongg_Vi(AbstractMahjonggGame):
    L = "0aaaaEaaacaccaCcaEcbaeaceaeeaAeaCebEecagbcgaegaggaygaAgbCgcEgcaibcibeiagiaiiawiayibAibCicEicakcckbekbgkaikakkaukawkbykbAkcCkcEkdamccmcembgmbimakmammasmaumbwmbymcAmcCmdEmeaodcoceocgobiobkoamoaooaqoasobuobwocyocAodCoeEo"

class Mahjongg_VictoryArrow(AbstractMahjonggGame):
    L = "0ataaabbcbbebbgbbibbkbambavbaxcaadamdbvdazdadebfebheajeaBeaafamfaofbvfbxfbzfaDfadgajgaqgaahaghamhaohbshbuhbwhbyhbAhbChbEhadiajiaqiaajamjaojbvjbxjbzjaDjadkbfkbhkajkaBkaalamlbvlazlaxmaanbcnbenbgnbinbknamnavnatohachmchaehmehdfhjfhaghmghoghdhhjhhqhhaihmihoihdjhjjhakhmkhamhmmodbofbohbojboadomdoafomfoahonhophorhothovhoxhozhoBhoajomjoalomlodnofnohnojn"

class Mahjongg_Wavelets(AbstractMahjonggGame):
    L = "0agaaqaaAaagcaqcaAccaeaeeaieaoeaseayeaCecGeaggaqgaAgcaiaeiaiiaoiasiayiaCicGiagkaqkaAkcamaemaimaomasmaymaCmcGmagoaqoaAoagqaqqaAqhgbhqbhAbhdehjehnehtehxehDehghhqhhAhhdihjihnihtihxihDihgjhqjhAjhdmhjmhnmhtmhxmhDmhgphqphApogcoqcoAcoceokeomeoueoweoEeoggoqgoAgociokiomiouiowioEiogkoqkoAkocmokmommoumowmoEmogooqooAovgdvqdvAdvbevlevvevFevgfvqfvAfvghvqhvAhvbivlivvivFivgjvqjvAjvglvqlvAlvbmvlmvvmvFmvgnvqnvAn"

class Mahjongg_Well(AbstractMahjonggGame):
    L = "0aiaakaamaaoaagcaicakcamcaocaqcacebeeegeeieekeemeeoeeqebseaueaafawfacgbegeggaigakgamgaogeqgbsgaugaahawhacibeiegiaiiakiamiaoieqibsiauiaajawjackbekegkeikekkemkeokeqkbskaukbimakmammbomaioakoamoaoohcfhufhchhuhhcjhuj"

class Mahjongg_WhatAPyramid(AbstractMahjonggGame):
    L = "0aaaacaaeaagaaiaakaamaaoaaqaasaauaawaaacaccbecbgcbicbkcbmcbocbqcbscaucawcaceaeebgeciedkedmecoebqeaseaueaegbggdigbkgbmgdogbqgasgaeibgidiibkibmidoibqiasiackaekbgkcikdkkdmkcokbqkaskaukaamacmbembgmbimbkmbmmbombqmbsmaumawmaaoacoaeoagoaioakoamoaooaqoasoauoawo"

class Mahjongg_Yummy(AbstractMahjonggGame):
    L = "0aoaaibakbbmbbqbasbaubaocagdbidbkdbmdbqdbsdbudawdaoeaefbgfcifckfdmfdqfcsfcufbwfayfaogaahachbehcghbihakhashbuhcwhbyhaAhaChaoiaejbgjcijckjdmjdqjcsjcujbwjayjaokaglbilbklbmlbqlbslbulawlaomainaknbmnbqnasnaunaooiobiodkofkohkojiolion"

r(5001, Mahjongg_Altar, "Altar")
r(5002, Mahjongg_Arena, "Arena")
r(5003, Mahjongg_Arena2, "Arena 2")
r(5004, Mahjongg_Arrow, "Arrow")
r(5005, Mahjongg_ArtModerne, "Art Moderne")
r(5006, Mahjongg_Balance, "Balance")
r(5007, Mahjongg_Bat, "Bat")
r(5008, Mahjongg_Beatle, "Beatle")
r(5009, Mahjongg_BigHole, "Big Hole")
r(5010, Mahjongg_Bizarre, "Bizarre")
r(5011, Mahjongg_Boat, "Boat")
r(5012, Mahjongg_Bug, "Bug")
r(5013, Mahjongg_Butterfly, "Butterfly")
r(5014, Mahjongg_Castle, "Castle")
r(5015, Mahjongg_CatAndMouse, "Cat and Mouse")
r(5016, Mahjongg_Ceremonial, "Ceremonial")
r(5017, Mahjongg_Checkered, "Checkered")
r(5018, Mahjongg_Chip, "Chip")
r(5019, Mahjongg_Columns, "Columns")
r(5020, Mahjongg_Crown, "Crown")
r(5021, Mahjongg_Cupola, "Cupola")
r(5022, Mahjongg_DeepWell, "Deep Well")
r(5023, Mahjongg_Dragon, "Dragon")
r(5024, Mahjongg_Dude, "Dude")
r(5025, Mahjongg_Eagle, "Eagle")
r(5026, Mahjongg_Enterprise, "Enterprise")
r(5027, Mahjongg_Eye, "Eye")
r(5028, Mahjongg_F15Eagle, "F-15 Eagle")
r(5029, Mahjongg_Farandole, "Farandole")
r(5030, Mahjongg_Fish, "Fish")
r(5031, Mahjongg_FivePyramids, "Five Pyramids")
r(5032, Mahjongg_FivePyramids2, "Five Pyramids 2")
r(5033, Mahjongg_Flowers, "Flowers")
r(5034, Mahjongg_FlyingDragon, "Flying Dragon")
r(5035, Mahjongg_FortressTowers, "Fortress Towers")
r(5036, Mahjongg_FullVision, "Full Vision")
r(5037, Mahjongg_FullVision2, "Full Vision 2")
r(5038, Mahjongg_Future, "Future")
r(5039, Mahjongg_Garden, "Garden")
r(5040, Mahjongg_Gayles, "Gayle's")
r(5041, Mahjongg_Glade, "Glade")
r(5042, Mahjongg_HForHaga, "H for Haga")
r(5043, Mahjongg_HForHagaTraditional, "H for Haga Traditional")
r(5044, Mahjongg_Helios, "Helios")
r(5045, Mahjongg_HighAndLow, "High and Low")
r(5046, Mahjongg_Hourglass, "Hourglass")
r(5047, Mahjongg_Inca, "Inca")
r(5048, Mahjongg_InnerCircle, "Inner Circle")
r(5049, Mahjongg_Joker, "Joker")
r(5050, Mahjongg_KForKyodai, "K for Kyodai")
r(5051, Mahjongg_KForKyodaiTraditional, "K for Kyodai Traditional")
r(5052, Mahjongg_Km, "Km")
r(5053, Mahjongg_Kujaku, "Kujaku")
r(5054, Mahjongg_Labyrinth, "Labyrinth")
r(5055, Mahjongg_Lion, "Lion")
r(5056, Mahjongg_Lost, "Lost")
r(5057, Mahjongg_Maya, "Maya")
r(5058, Mahjongg_Mesh, "Mesh")
r(5059, Mahjongg_Moth, "Moth")
r(5060, Mahjongg_NForNamida, "N for Namida")
r(5061, Mahjongg_NForNamidaTraditional, "N for Namida Traditional")
r(5062, Mahjongg_NaokiHagaTraditional, "Naoki Haga Traditional")
r(5063, Mahjongg_NewLayout, "New Layout")
r(5064, Mahjongg_Order, "Order")
r(5065, Mahjongg_Pattern, "Pattern")
r(5066, Mahjongg_Phoenix, "Phoenix")
r(5067, Mahjongg_Portal, "Portal")
r(5068, Mahjongg_Rocket, "Rocket")
r(5069, Mahjongg_Scorpion, "Scorpion")
r(5070, Mahjongg_ScrewUp, "Screw Up")
r(5071, Mahjongg_Seven, "Seven")
r(5072, Mahjongg_SevenPyramids, "Seven Pyramids")
r(5073, Mahjongg_Shield, "Shield")
r(5074, Mahjongg_Siam, "Siam")
r(5075, Mahjongg_SpaceShip, "Space Ship")
r(5076, Mahjongg_Square, "Square")
r(5077, Mahjongg_Squares, "Squares")
r(5078, Mahjongg_Squaring, "Squaring")
r(5079, Mahjongg_Stairs, "Stairs")
r(5080, Mahjongg_StarShip, "Star Ship")
r(5081, Mahjongg_StepPyramid, "Step Pyramid")
r(5082, Mahjongg_Stonehenge, "Stonehenge")
r(5083, Mahjongg_SunMoon, "SunMoon")
r(5084, Mahjongg_Temple, "Temple")
r(5085, Mahjongg_Teotihucan, "Teotihucan")
r(5086, Mahjongg_TheDoor, "The Door")
r(5087, Mahjongg_TheGreatWall, "The Great Wall")
r(5088, Mahjongg_Theater, "Theater")
r(5089, Mahjongg_TileFighter, "Tile Fighter")
r(5090, Mahjongg_Tilepiles, "Tilepiles")
r(5091, Mahjongg_TimeTunnel, "Time Tunnel")
r(5092, Mahjongg_Tomb, "Tomb")
r(5093, Mahjongg_TowerAndWalls, "Tower and Walls")
r(5094, Mahjongg_TraditionalReviewed, "Traditional Reviewed")
r(5095, Mahjongg_TreeOfLife, "Tree of Life")
r(5096, Mahjongg_TwinTemples, "Twin Temples")
r(5097, Mahjongg_Vi, "Vi")
r(5098, Mahjongg_VictoryArrow, "Victory Arrow")
r(5099, Mahjongg_Wavelets, "Wavelets")
r(5100, Mahjongg_Well, "Well")
r(5101, Mahjongg_WhatAPyramid, "What a Pyramid")
r(5102, Mahjongg_Yummy, "Yummy")

del r
