#include <gtk/gtk.h>
#include "colorlevels.h"
#include "config.h"


static gint iv_color_levels_apply_calculate_shift_total(
	const gint v,
	const gint min_displace,
	const gint mid_displace,
	const gint max_displace
);
void iv_color_levels_apply(
	const guint8 *src, guint8 *tar,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gint red_min, const gfloat red_mid_coeff, const gint red_max,
	const gint green_min, const gfloat green_mid_coeff, const gint green_max,
	const gint blue_min, const gfloat blue_mid_coeff, const gint blue_max,
	const gint alpha_min, const gfloat alpha_mid_coeff, const gint alpha_max,
	const gboolean interruptable,
	gint (*progress_cb)(const gulong, const gulong, gpointer),     
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Calculates the total shift from the min, mid, and max levels
 *	to the channel value.
 *
 *	The v specifies the channel value, this value must be from
 *	0x00 to 0xff.
 *
 *	The min_displace specifies the min level minus 0x00 value.
 *
 *	The mid_displace specifies the mid level offset value.
 *
 *	The max_displace specifies the 0xff minux max level value.
 */
static gint iv_color_levels_apply_calculate_shift_total(
	const gint v,
	const gint min_displace,
	const gint mid_displace,
	const gint max_displace
)
{
	const gint	v_max = (gint)((guint8)-1),
			v_mid = v_max / 2;

	gint	min_shift = -min_displace * (v_max - v) / v_max,
		max_shift = max_displace * v / v_max,
		mid_shift = (v < v_mid) ?
		    (-mid_displace * v / v_mid) :
		    (-mid_displace * (v_mid - v + v_mid) / v_mid);
	return(min_shift + mid_shift + max_shift);
}

/*
 *	Applies color leveling to the image.
 *
 *	The src specifies the original image data.
 *
 *	The tar specifies the image data that color leveling is to
 *	be applied to.
 *
 *	The width and height specifies the size of the image in pixels.
 *
 *	The bpp specifies the bytes per pixel.
 *
 *	The bpl specifies the bytes per line.
 *
 *	The *_min specifies the minimum color shift value in bytes in the
 *	range of -256 to 255.
 *
 *	The *_mid specifies the midpoint value coefficient.
 *
 *	The *_max specifies the maximum color shift value in bytes in the
 *	range of -256 to 255.
 *
 *	The channel specifies the color channel(s) to operate on.
 *
 *	If interruptable is TRUE then this function will return
 *	when a pending GTK event detected and the color leveling will
 *	not be completed.
 */
void iv_color_levels_apply(
	const guint8 *src, guint8 *tar,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gint red_min, const gfloat red_mid_coeff, const gint red_max,
	const gint green_min, const gfloat green_mid_coeff, const gint green_max,
	const gint blue_min, const gfloat blue_mid_coeff, const gint blue_max,
	const gint alpha_min, const gfloat alpha_mid_coeff, const gint alpha_max,
	const gboolean interruptable,
	gint (*progress_cb)(const gulong, const gulong, gpointer),     
	gpointer progress_data
)
{
	const gint	v_max = (gint)((guint8)-1),
			v_mid = v_max / 2,
			v_min = 0;
	gint		y, shift,
			red_min_displace,
			red_mid_displace,
			red_max_displace,
			green_min_displace,
			green_mid_displace,
			green_max_displace,
			blue_min_displace,
			blue_mid_displace,
			blue_max_displace,
			alpha_min_displace,
			alpha_mid_displace,
			alpha_max_displace,
			grey_min_displace,
			grey_mid_displace,
			grey_max_displace;
	const guint8 *src_ptr, *src_end;
	guint8 *tar_ptr;

	if((src == NULL) || (tar == NULL))
	    return;

	if(progress_cb != NULL)
	{
	    if(progress_cb(0l, (gulong)height, progress_data))
		return;
	}

	/* Calculate the marker value displacements for each
	 * color channel
	 */
	red_min_displace = red_min - v_min;
	red_max_displace = v_max - red_max;
	red_mid_displace = (gint)(v_mid * red_mid_coeff) - v_mid;

	green_min_displace = green_min - v_min;
	green_max_displace = v_max - green_max;
	green_mid_displace = (gint)(v_mid * green_mid_coeff) - v_mid;

	blue_min_displace = blue_min - v_min;
	blue_max_displace = v_max - blue_max;
	blue_mid_displace = (gint)(v_mid * blue_mid_coeff) - v_mid;

	alpha_min_displace = alpha_min - v_min;
	alpha_max_displace = v_max - alpha_max;
	alpha_mid_displace = (gint)(v_mid * alpha_mid_coeff) - v_mid;

	/* Get the level displacements for greyscale as the level
	 * displacements for the red channel
	 */
	grey_min_displace = red_min_displace;
	grey_mid_displace = red_mid_displace;
	grey_max_displace = red_max_displace;

	switch(bpp)
	{
	  case 4:	/* RGBA */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}
		src_ptr = src + (y * bpl);
		src_end = src_ptr + (width * bpp);
		tar_ptr = tar + (y * bpl);
		while(src_ptr < src_end)
		{
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[0],
			red_min_displace,
			red_mid_displace,
			red_max_displace
		    );
		    tar_ptr[0] = (guint8)CLIP(
			((gint)src_ptr[0] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[1],
			green_min_displace,
			green_mid_displace,
			green_max_displace
		    );
		    tar_ptr[1] = (guint8)CLIP(
			((gint)src_ptr[1] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[2],
			blue_min_displace,
			blue_mid_displace,
			blue_max_displace
		    );
		    tar_ptr[2] = (guint8)CLIP(
			((gint)src_ptr[2] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[3],
			alpha_min_displace,
			alpha_mid_displace,
			alpha_max_displace
		    );
		    tar_ptr[3] = (guint8)CLIP(
			((gint)src_ptr[3] + shift),
			v_min, v_max
		    );
		    tar_ptr += bpp;
		    src_ptr += bpp;
		}
		if(interruptable && (gtk_events_pending() > 0))
		    return;
	    }
	    break;

	  case 3:	/* RGB */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}
		src_ptr = src + (y * bpl);
		src_end = src_ptr + (width * bpp);
		tar_ptr = tar + (y * bpl);
		while(src_ptr < src_end)
		{
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[0],
			red_min_displace,
			red_mid_displace,
			red_max_displace
		    );
		    tar_ptr[0] = (guint8)CLIP(
			((gint)src_ptr[0] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[1],
			green_min_displace,
			green_mid_displace,
			green_max_displace
		    );
		    tar_ptr[1] = (guint8)CLIP(
			((gint)src_ptr[1] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[2],
			blue_min_displace,
			blue_mid_displace,
			blue_max_displace
		    );
		    tar_ptr[2] = (guint8)CLIP(
			((gint)src_ptr[2] + shift),
			v_min, v_max
		    );
		    tar_ptr += bpp;
		    src_ptr += bpp;
		}
		if(interruptable && (gtk_events_pending() > 0))
		    return;
	    }
	    break;

	  case 2:	/* Greyscale Alpha */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}
		src_ptr = src + (y * bpl);
		src_end = src_ptr + (width * bpp);
		tar_ptr = tar + (y * bpl);
		while(src_ptr < src_end)
		{
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[0],
			grey_min_displace,
			grey_mid_displace,
			grey_max_displace
		    );
		    tar_ptr[0] = (guint8)CLIP(
			((gint)src_ptr[0] + shift),
			v_min, v_max
		    );
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[1],
			alpha_min_displace,
			alpha_mid_displace,
			alpha_max_displace
		    );
		    tar_ptr[1] = (guint8)CLIP(
			((gint)src_ptr[1] + shift),
			v_min, v_max
		    );
		    tar_ptr += bpp;
		    src_ptr += bpp;
		}
		if(interruptable && (gtk_events_pending() > 0))
		    return;
	    }
	    break;

	  case 1:	/* Greyscale */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}
		src_ptr = src + (y * bpl);
		src_end = src_ptr + (width * bpp);
		tar_ptr = tar + (y * bpl);
		while(src_ptr < src_end)
		{
		    shift = iv_color_levels_apply_calculate_shift_total(
			src_ptr[0],
			grey_min_displace, grey_mid_displace, grey_max_displace
		    );
		    tar_ptr[0] = (guint8)CLIP(
			((gint)src_ptr[0] + shift),
			v_min, v_max
		    );
		    tar_ptr += bpp;
		    src_ptr += bpp;
		}
		if(interruptable && (gtk_events_pending() > 0))
		    return;
	    }
	    break;
	}

	if(progress_cb != NULL)
	{
	    if(progress_cb((gulong)height, (gulong)height, progress_data))
		return;
	}
}
