#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkprivate.h>
#if defined(_WIN32)
# include <gdk/gdkwin32.h>
#else
# include <gdk/gdkx.h>
# define HAVE_X
#endif
#include "guiutils.h"
#include "guirgbimg.h"
#include "guifullscreen.h"
#include "fullscreenimg.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL		0
#endif


typedef struct _FSImgData               FSImgData;


/*
 *	FSImg Data:
 */
struct _FSImgData {

	GtkWidget	*toplevel;	/* GtkWindow (GTK_WINDOW_POPUP) */

	GdkCursor	*invis_cur;
	gboolean	pointer_shown;

	GdkPixmap	**pixmaps_list;
	gulong		*delay_list;
	gint		nframes,
			cur_frame;

	guint		hide_pointer_toid,
			next_frame_toid;

	gint		main_loop_level;

	/* Video modes */
	GList		*vidmodes_list;

	const GdkVideoModeInfo	*prev_vidmode,
				*cur_vidmode;

	gint		prev_viewport_x,
			prev_viewport_y;

	gint		prev_pointer_x,
			prev_pointer_y;

};
#define FSIMG_DATA(p)		((FSImgData *)(p))
#define FSIMG_DATA_KEY		"fsimg_data"


static void FSImgDataDestroyCB(gpointer data);
static void FSImgDestroyCB(GtkObject *object, gpointer data);
static gint FSImgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint FSImgHidePointerTOCB(gpointer data);
static gint FSImgNextFrameTOCB(gpointer data);

static void FSImgWarpPointer(GdkWindow *window, gint x, gint y);
static void FSImgDraw(FSImgData *fsimg);
static guint8 *FSImgCreateWinImage(
	gint win_width, gint win_height,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	const guint8 *src_rgba,
	gint src_width, gint src_height,
	gint src_bpl
);

gint FSImgSet(
	GtkWidget *w,
	guint8 **rgba_list, gulong *delay_list,
	gint nframes,
	gint width, gint height, gint bpl,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint quality			/* 0 to 2 */
);
GtkWidget *FSImgNew(
	guint8 **rgba_list, gulong *delay_list,
	gint nframes,
	gint width, gint height, gint bpl,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint quality,			/* 0 to 2 */
	gboolean block,
	GtkWidget *ref_toplevel
);

static void FSImgUnmapRestore(FSImgData *fsimg);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define FSIMG_HIDE_POINTER_DELAY	3000l


/*
 *	FSImg data destroy callback.
 */
static void FSImgDataDestroyCB(gpointer data)
{
	gint i;
	FSImgData *fsimg = FSIMG_DATA(data);
	if(fsimg == NULL)
	    return;

#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgDataDestroyCB(): fsimg=0x%.8x fsimg->toplevel=0x%.8x\n",
 (guint32)fsimg,
 (guint32)fsimg->toplevel
);
#endif

	/* Remove timeouts */
	GTK_TIMEOUT_REMOVE(fsimg->hide_pointer_toid);
	fsimg->hide_pointer_toid = 0;
	GTK_TIMEOUT_REMOVE(fsimg->next_frame_toid);
	fsimg->next_frame_toid = 0;

	/* Break out of any main loop levels */
	while(fsimg->main_loop_level > 0)
	{
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgDataDestroyCB(): Breaking out of main loop level +%i\n",
 fsimg->main_loop_level
);
#endif
	    gtk_main_quit();
	    fsimg->main_loop_level--;
	}

	/* Ungrabs the pointer as needed, unmaps the FSImg window,
	 * and restores the video mode
	 */
	FSImgUnmapRestore(fsimg);

	/* Destroy widgets */
#if 0
/* Toplevel should already be destroyed when this is called */
	GTK_WIDGET_DESTROY(fsimg->toplevel);
#endif

	for(i = 0; i < fsimg->nframes; i++)
	    GDK_PIXMAP_UNREF(fsimg->pixmaps_list[i]);
	g_free(fsimg->pixmaps_list);
	g_free(fsimg->delay_list);

	GDK_CURSOR_DESTROY(fsimg->invis_cur);

	gdk_video_modes_delete(fsimg->vidmodes_list);
	g_free(fsimg);

#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgDataDestroyCB(): fsimg=0x%.8x deleted\n",
 (guint32)fsimg
);
#endif
}

/*
 *	FSImg toplevel GtkWindow "destroy" signal callback.
 */
static void FSImgDestroyCB(GtkObject *object, gpointer data)
{
	FSImgData *fsimg = FSIMG_DATA(data);
	if(fsimg == NULL)
	    return;

#if (DEBUG_LEVEL >= 1)
g_print("FSImgDestroyCB(): object=0x%.8x\n", (guint32)object);
#endif

	/* Mark widgets being destroyed as NULL so they do destroyed
	 * again later
	 */
	if(object == (GtkObject *)fsimg->toplevel)
	    fsimg->toplevel = NULL;
}

/*
 *      FSImg toplevel GtkWindow event signal callback.
 */
static gint FSImgEventCB( 
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean keypress;
	GdkEventConfigure *configure;
	GdkEventExpose *expose;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GdkEventCrossing *crossing;
	FSImgData *fsimg = FSIMG_DATA(data);
	if((widget == NULL) || (event == NULL) || (fsimg == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    /* Restore video mode and delete the FSImg */
	    GTK_WIDGET_DESTROY(fsimg->toplevel);
	    status = TRUE;
	    break;

	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;   
#if (DEBUG_LEVEL >= 1)                              
g_print(
 "FSImgEventCB(): configure_event\n"
);
#endif
	    status = TRUE;   
	    break;

	  case GDK_EXPOSE:
	    expose = (GdkEventExpose *)event;
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgEventCB(): expose_event\n"
);
#endif
	    FSImgDraw(fsimg);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgEventCB(): %s\n",
 focus->in ? "focus_in_event" : "focus_out_event"
);
#endif
	    if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		FSImgDraw(fsimg);
	    }
	    else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		FSImgDraw(fsimg);
	    }
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    keypress = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgEventCB(): %s keyval=0x%.8x state=0x%.8x\n",
 keypress ? "key_press_event" : "key_release_event",
 (guint32)key->keyval,
 (guint32)key->state
);
#endif
	    switch(key->keyval)
	    {
	      case GDK_Escape:
	      case GDK_space:
	      case GDK_Return:
	      case GDK_KP_Enter:
	      case GDK_q:
		if(keypress)
		{
		    /* Restore video mode and delete the FSImg */
		    GTK_WIDGET_DESTROY(fsimg->toplevel);
		}
		status = TRUE;
		break;

	      case GDK_p:
		/* Play or pause */
		if(keypress)
		{
		    const gint nframes = fsimg->nframes;
		    if(fsimg->next_frame_toid > 0)
		    {
			GTK_TIMEOUT_REMOVE(fsimg->next_frame_toid);
			fsimg->next_frame_toid = 0;
		    }
		    else if(nframes > 1)
		    {
			gulong delay;

			gint cur_frame = fsimg->cur_frame;
			if(cur_frame >= nframes)
			    cur_frame = 0;
			else if(cur_frame < 0)
			    cur_frame = 0;

			fsimg->cur_frame = cur_frame;

			delay = (fsimg->delay_list != NULL) ?
			    fsimg->delay_list[cur_frame] : 0l;

			fsimg->next_frame_toid = gtk_timeout_add(
			    delay,
			    FSImgNextFrameTOCB, fsimg
			);
		    }
		}
		status = TRUE;
		break;

	      case GDK_s:
		/* Stop */
		if(keypress)
		{
		    GTK_TIMEOUT_REMOVE(fsimg->next_frame_toid);
		    fsimg->next_frame_toid = 0;
		}
		status = TRUE;
		break;

	      case '[':
	      case '{':
	      case '<':
	      case ',':
	      case GDK_b:
	      case GDK_Left:
	      case GDK_KP_Left:
		/* Seek previous */
		if(keypress)
		{
		    const gint nframes = fsimg->nframes;
		    if(nframes > 1)
		    {
			fsimg->cur_frame--;
			if(fsimg->cur_frame < 0)
			    fsimg->cur_frame = nframes - 1;
			FSImgDraw(fsimg);
		    }
		}
		status = TRUE;
		break;

	      case ']':
	      case '}':
	      case '>':
	      case '.':
	      case GDK_n:
	      case GDK_Right:
	      case GDK_KP_Right:
		/* Seek next */
		if(keypress)
		{
		    const gint nframes = fsimg->nframes;
		    if(nframes > 1)
		    {  
			fsimg->cur_frame++;
			if(fsimg->cur_frame >= nframes)
			    fsimg->cur_frame = 0;
			FSImgDraw(fsimg);      
		    }     
		}    
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
#if (DEBUG_LEVEL >= 1)   
g_print(
 "FSImgEventCB(): button_press_event button=%i state=0x%.8x\n",
 button->button,
 (guint32)button->state
);
#endif
	    if(!GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    /* Restore video mode and delete the FSImg */
	    GTK_WIDGET_DESTROY(fsimg->toplevel);
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
#if (DEBUG_LEVEL >= 1)   
g_print(
 "FSImgEventCB(): button_release_event button=%i state=0x%.8x\n",
 button->button,
 (guint32)button->state
);
#endif
	    status = TRUE;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(motion->is_hint)
	    {
		GdkModifierType mask;
		gint x, y;
		gdk_window_get_pointer(
		    motion->window, &x, &y, &mask
		);
	    }
	    if(!fsimg->pointer_shown)
	    {
		gdk_window_set_cursor(widget->window, NULL);
		fsimg->pointer_shown = TRUE;
	    }
	    if(TRUE)
	    {
		GTK_TIMEOUT_REMOVE(fsimg->hide_pointer_toid);
		fsimg->hide_pointer_toid = gtk_timeout_add(
		    FSIMG_HIDE_POINTER_DELAY,
		    FSImgHidePointerTOCB, fsimg
		);
	    }
	    status = TRUE;
	    break;

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;

	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
	    crossing = (GdkEventCrossing *)event;

	    status = TRUE;
	    break;

	}

	return(status);
}

/*
 *	Hide pointer timeout callback.
 */
static gint FSImgHidePointerTOCB(gpointer data)
{
	FSImgData *fsimg = FSIMG_DATA(data);
	if(fsimg == NULL)
	    return(FALSE);

	fsimg->hide_pointer_toid = 0;

	if(fsimg->pointer_shown)
	{
	    GtkWidget *w = fsimg->toplevel;
	    if(w != NULL)
		gdk_window_set_cursor(w->window, fsimg->invis_cur);
	    fsimg->pointer_shown = FALSE;
	}

	return(FALSE);
}

/*
 *	Next frame timeout callback.
 */
static gint FSImgNextFrameTOCB(gpointer data)
{
	FSImgData *fsimg = FSIMG_DATA(data);
	if(fsimg == NULL)
	    return(FALSE);

	fsimg->next_frame_toid = 0;

	if((fsimg->nframes <= 1) || (fsimg->delay_list == NULL))
	    return(FALSE);

	/* Go to next frame */
	fsimg->cur_frame++;

	/* If the end is reached then cycle back to the first frame */
	if(fsimg->cur_frame >= fsimg->nframes)
	    fsimg->cur_frame = 0;

	/* Draw the next frame */
	FSImgDraw(fsimg);

	/* Schedual delay to show this frame */
	fsimg->next_frame_toid = gtk_timeout_add(
	    fsimg->delay_list[fsimg->cur_frame],
	    FSImgNextFrameTOCB, fsimg
	);

	return(FALSE);
}


/*
 *	Warps the pointer relative to the desktop.
 */
static void FSImgWarpPointer(GdkWindow *window, gint x, gint y)
{
#if defined(HAVE_X)
	GdkWindowPrivate *private = (GdkWindowPrivate *)window;
	if((private != NULL) ? (private->xdisplay == NULL) : TRUE)
	    return;

	XWarpPointer(
	    private->xdisplay,
	    None,		/* No source window */
	    private->xwindow,	/* Move relative to destination window */
	    0, 0, 0, 0,		/* No source rectangle */
	    (int)x, (int)y
	);
#else

#endif
}

/*
 *	Redraws the FSImg's window image.
 */
static void FSImgDraw(FSImgData *fsimg)
{
	gint frame_num, win_width, win_height;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GtkStyle *style;
	GtkWidget *w = (fsimg != NULL) ? fsimg->toplevel : NULL;
	if(w == NULL)
	    return;

	frame_num = fsimg->cur_frame;
	if((frame_num < 0) || (frame_num >= fsimg->nframes))
	    return;

	window = w->window;
	pixmap = fsimg->pixmaps_list[frame_num];
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	if(pixmap != NULL)
	{
#if (DEBUG_LEVEL >= 3)
g_print(
 "FSImgDraw(): Redrawing window=0x%.8x pixmap=0x%.8x\n",
 (guint32)window,
 (guint32)pixmap
);
#endif
	    gdk_window_get_size(pixmap, &win_width, &win_height);
	    gdk_window_copy_area(
		window, style->black_gc,
		0, 0,
		pixmap,
		0, 0,
		win_width, win_height
	    );
	}
}

/*
 *	Creates an RGBA image data of the specified size containing the
 *	RGBA image specified by src_rgba centered and the
 *	background color specified by bg_color.
 *
 *	Inputs assumed valid.
 */
static guint8 *FSImgCreateWinImage(
	gint win_width, gint win_height,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	const guint8 *src_rgba,
	gint src_width, gint src_height,
	gint src_bpl
)
{
	const gint	bpp = 4,
			tar_bpl = win_width * bpp;
	guint8 *tar_rgba = ((tar_bpl * win_height) > 0) ?
	    (guint8 *)g_malloc(tar_bpl * win_height) : NULL;
	if(tar_rgba == NULL)
	    return(NULL);

	if(src_bpl <= 0)
	    src_bpl = src_width * bpp;

	/* Clear the RGBA image data with the background */
	if(bg_color != NULL)
	{
	    guint32	*ptr = (guint32 *)tar_rgba,
			*end = ptr + (win_width * win_height);
	    while(ptr < end)
		*ptr++ = *(const guint32 *)bg_color;
	}

	/* Combine the specified RGBA image over the target RGBA
	 * image data centered
	 */
	GUIImageBufferCopyArea(
	    bpp,
	    src_rgba,
	    src_width, src_height, src_bpl,
	    tar_rgba,
	    win_width, win_height, tar_bpl,
	    (win_width - src_width) / 2,
	    (win_height - src_height) / 2,
	    TRUE,			/* Blend */
	    NULL, NULL
	);

	return(tar_rgba);
}

/*
 *	Sets the image for the FSImg and switches video modes as needed.
 *
 *	Returns values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value
 *	-3	Systems error
 *	-4	User aborted
 *	-5	No video mode available
 */
gint FSImgSet(
	GtkWidget *w,
	guint8 **rgba_list, gulong *delay_list,
	gint nframes,
	gint width, gint height, gint bpl,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint quality			/* 0 to 2 */
)
{
	gboolean need_change_vidmode = FALSE;
	gint i, win_width, win_height;
	GList *glist;
	GdkVideoModeInfo *m_sel, *m;
	GdkRgbDither dither;
	GdkWindow *window;
	GtkStyle *style;
	FSImgData *fsimg = FSIMG_DATA(
	    GTK_OBJECT_GET_DATA(w, FSIMG_DATA_KEY)
	);
	if((fsimg == NULL) || (rgba_list == NULL) || (nframes <= 0))
	    return(-2);

	/* Remove all timeouts */
	GTK_TIMEOUT_REMOVE(fsimg->next_frame_toid);
	fsimg->next_frame_toid = 0;

	/* Delete previous pixmaps and delay lists */
	for(i = 0; i < fsimg->nframes; i++)
	    GDK_PIXMAP_UNREF(fsimg->pixmaps_list[i]);
	g_free(fsimg->pixmaps_list);
	fsimg->pixmaps_list = NULL;
	g_free(fsimg->delay_list);
	fsimg->delay_list = NULL;

	/* Allocate new pixmaps and delay lists */
	fsimg->nframes = nframes;
	fsimg->cur_frame = 0;
	fsimg->pixmaps_list = (GdkPixmap **)g_malloc0(
	    nframes * sizeof(GdkPixmap *)
	);
	if(delay_list != NULL)
	{
	    fsimg->delay_list = (gulong *)g_malloc(
		nframes * sizeof(gulong)
	    );
	    memcpy(
		fsimg->delay_list, delay_list,
		nframes * sizeof(gulong)
	    );
	}


	switch(quality)
	{
	  case 2:
	    dither = GDK_RGB_DITHER_MAX;
	    break;
	  case 1:
	    dither = GDK_RGB_DITHER_NORMAL;
	    break;
	  default:
	    dither = GDK_RGB_DITHER_NONE;
	    break;
	}

	window = w->window;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return(-1);

	/* Find new video mode with a suitable size for the specified
	 * image
	 */
	m_sel = NULL;
	for(glist = fsimg->vidmodes_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    m = GDK_VIDEO_MODE_INFO(glist->data);
	    if(m == NULL)
		continue;

	    /* Size too small to fit image? */
	    if((m->viewport_width < width) ||
	       (m->viewport_height < height)
	    )
		continue;

	    if(m_sel == NULL)
	    {
		m_sel = m;
	    }
	    else
	    {   
		if((m->viewport_width < m_sel->viewport_width) ||
		   (m->viewport_height < m_sel->viewport_height)
		)
		    m_sel = m;
	    }
	}
	/* If no suitable video mode is found then use the largest
	 * video mode available
	 */
	if(m_sel == NULL)
	{
	    for(glist = fsimg->vidmodes_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = GDK_VIDEO_MODE_INFO(glist->data);
		if(m == NULL)
		    continue;

		if(m_sel == NULL)
		{
		    m_sel = m;
		}
		else
		{
		    if((m->viewport_width > m_sel->viewport_width) ||
		       (m->viewport_height > m_sel->viewport_height)
		    )  
			m_sel = m;
		}
	    }
	}
	/* If there is still no video mode available then just use the
	 * current video mode
	 */
	if(m_sel == NULL)
	    m_sel = gdk_video_mode_get_current(fsimg->vidmodes_list);
	else
	    need_change_vidmode = (m_sel != gdk_video_mode_get_current(fsimg->vidmodes_list)) ?
		TRUE : FALSE;
	if(m_sel == NULL)
	    return(-5);

	/* Get values of the new video mode */
	fsimg->cur_vidmode = m_sel;
	win_width = m_sel->viewport_width;
	win_height = m_sel->viewport_height;
	if((win_width <= 0) || (win_height <= 0))
	    return(-2);

	/* Resize image to fit? */
	if((width > win_width) || (height > win_height))
	{
	    /* Create a resized smaller image to fit within the
	     * viewport
	     */
	    const gint bpp = 4;
	    const gfloat aspect = (gfloat)width / (gfloat)height;
	    gint	rs_width = (gint)(win_height * aspect),
			rs_height = win_height,
			rs_bpl;
	    guint8 *rgba, *rs_rgba, *win_data_rgba;
	    GdkPixmap *pixmap;

	    if((rs_width > win_width) && (aspect > 0.0f))
	    {
		rs_width = win_width;
		rs_height = (gint)(win_width / aspect);
	    }
	    if((rs_width <= 0) || (rs_height <= 0))
		return(-2);

	    /* Size of smaller image that will fit within the viewport
	     * has now been calculated, now allocate it
	     */
	    rs_bpl = rs_width * bpp;
	    rs_rgba = (guint8 *)g_malloc(rs_bpl * rs_height);
	    if(rs_rgba == NULL)
		return(-3);

	    for(i = 0; i < nframes; i++)
	    {
		rgba = rgba_list[i];

		/* Copy/resize the RGBA image data to the smaller
		 * RGBA image
		 */
		GUIImageBufferResize(
		    bpp,
		    rgba, width, height, bpl,
		    rs_rgba, rs_width, rs_height, rs_bpl,
		    NULL, NULL
		);

		/* Combine the smaller RGBA image with the
		 * background
		 */
		win_data_rgba = FSImgCreateWinImage(
		    win_width, win_height,
		    bg_color,
		    rs_rgba, rs_width, rs_height, rs_bpl
		);

		/* Create the GdkPixmap for this frame and put the
		 * combined RGBA image data to it
		 */
		fsimg->pixmaps_list[i] = pixmap = gdk_pixmap_new(
		    window, win_width, win_height, -1
		);
		if((win_data_rgba != NULL) && (pixmap != NULL))
		    gdk_draw_rgb_32_image(
			pixmap, style->black_gc,
			0, 0, win_width, win_height,
			dither,
			(guchar *)win_data_rgba,
			win_width * 4
		    );

		g_free(win_data_rgba);
	    }

	    g_free(rs_rgba);
	}
	else
	{
	    guint8 *rgba, *win_data_rgba;
	    GdkPixmap *pixmap;

	    for(i = 0; i < nframes; i++)
	    {
		rgba = rgba_list[i];

		/* Combine the RGBA image with the background */
		win_data_rgba = FSImgCreateWinImage(
		    win_width, win_height,
		    bg_color,
		    rgba, width, height, bpl
		);

		/* Create the GdkPixmap for this frame and put the
		 * combined RGBA image data to it
		 */
		fsimg->pixmaps_list[i] = pixmap = gdk_pixmap_new(
		    window, win_width, win_height, -1
		);
		if((win_data_rgba != NULL) && (pixmap != NULL))
		    gdk_draw_rgb_32_image(
			pixmap, style->black_gc,
			0, 0, win_width, win_height,
			dither,
			(guchar *)win_data_rgba,
			win_width * 4
		    );

		g_free(win_data_rgba);
	    }
	}

	/* Move and resize the FSImg window to display the image */
	gtk_widget_set_uposition(w, 0, 0);
	gtk_widget_set_usize(w, win_width, win_height);
	gdk_window_move_resize(
	    window,
	    0, 0,
	    win_width, win_height
	);

	/* Map and raise the FSImg window */
	gtk_widget_show_raise(w);

	/* Grab focus and add grab to receive key events */
	gtk_widget_grab_focus(w);
	if(gtk_grab_get_current() != w)
	    gtk_grab_add(w);

	/* Grab the pointer and confine it to the window's area */
	gdk_pointer_grab(
	    window,
	    TRUE,			/* Report events relative to the
					 * window */
	    GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK,
	    window,			/* Confine pointer to the window */
	    GDK_NONE,
	    GDK_CURRENT_TIME
	);

	/* Move pointer to center of window */
	FSImgWarpPointer(
	    window,
	    win_width / 2, win_height / 2
	);

	/* Switch to new video mode as needed */
	if(need_change_vidmode)
	{
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgSet(): Switching to video mode %ix%i\n",
 m_sel->viewport_width, m_sel->viewport_height
);
#endif
	    gdk_video_mode_switch(m_sel);
	}

	/* Move viewport to upper left corner */
	gdk_video_mode_set_viewport(0, 0);

	FSImgDraw(fsimg);

	/* Schedual next frame draw if there are multiple frames */
	if(nframes > 0)
	{
	    gulong delay = (delay_list != NULL) ? delay_list[0] : 0l;
	    fsimg->next_frame_toid = gtk_timeout_add(
		delay,
		FSImgNextFrameTOCB, fsimg
	    );
	}

	return(0);
}


/*
 *	Creates a new FSImg displaying the specified image data and
 *	maps it in fullscreen mode.
 *
 *	If block is TRUE then this function will not return until the
 *	user closes the FSImg window.
 *
 *	The return is always NULL if block is TRUE.
 *
 *	The returned pointer should never be dereferenced since the
 *	widget can be destroyed by the user at any time.
 */
GtkWidget *FSImgNew(
	guint8 **rgba_list, gulong *delay_list,
	gint nframes,
	gint width, gint height, gint bpl,
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint quality,			/* 0 to 2 */
	gboolean block,
	GtkWidget *ref_toplevel
)
{
	GdkWindow *window;
	GdkModifierType mask;
	GtkWidget *w;
	FSImgData *fsimg;

	if(!gdk_video_mode_is_supported() ||
	   (rgba_list == NULL) || (nframes <= 0) ||
	   (width <= 0) || (height <= 0)
	)
	    return(NULL);

	if(bpl <= 0)
	    bpl = width * 4;

	/* Create a new FSImg */
	fsimg = FSIMG_DATA(g_malloc0(sizeof(FSImgData)));
	if(fsimg == NULL)
	    return(NULL);

	fsimg->invis_cur = NULL;
	fsimg->pointer_shown = TRUE;
	fsimg->pixmaps_list = NULL;
	fsimg->delay_list = NULL;
	fsimg->nframes = 0;
	fsimg->cur_frame = 0;
	fsimg->hide_pointer_toid = 0;
	fsimg->next_frame_toid = 0;
	fsimg->main_loop_level = 0;

	/* Get list of video modes */
	fsimg->vidmodes_list = gdk_video_modes_get();

	/* Record original video mode */
	fsimg->prev_vidmode = gdk_video_mode_get_current(
	    fsimg->vidmodes_list
	);

	/* Record original video mode values */
	gdk_video_mode_get_viewport(
	    &fsimg->prev_viewport_x, &fsimg->prev_viewport_y
	);
	gdk_window_get_pointer(
	    GDK_ROOT_PARENT(),
	    &fsimg->prev_pointer_x, &fsimg->prev_pointer_y,
	    &mask
	);


	/* Begin creating widgets */

	fsimg->toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_object_set_data_full(
	    GTK_OBJECT(w), FSIMG_DATA_KEY,
	    fsimg, FSImgDataDestroyCB
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
	    GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "destroy",
	    GTK_SIGNAL_FUNC(FSImgDestroyCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "focus_in_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "focus_out_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(  
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect_after(  
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_widget_realize(w);
	window = w->window;

	/* Create the GdkCursors */
	{
	    GdkColor	c_fg = { 0, 0xffff, 0xffff, 0xffff },
			c_bg = { 0, 0x0000, 0x0000, 0x0000 };
	    const gint width = 1, height = 1;
	    const gchar data[] = { 0x00 };
	    GdkBitmap *color_mask = gdk_bitmap_create_from_data(
		window, data, width, height
	    );   
	    GdkBitmap *trans_mask = gdk_bitmap_create_from_data(
		window, data, width, height
	    );
	    fsimg->invis_cur = gdk_cursor_new_from_pixmap(
		color_mask, trans_mask,
		&c_fg, &c_bg,
		0, 0
	    );
	    GDK_BITMAP_UNREF(color_mask); 
	    GDK_BITMAP_UNREF(trans_mask);
	}

	/* Set image, map window, and switch video modes */
	if(FSImgSet(
		w,
		rgba_list, delay_list,
		nframes,
		width, height,
		bpl,
		bg_color,
		quality
	))
	{
	    /* Failed to set image, map window, or switch video mode,
	     * delete the FSImg and give up
	     */
	    GTK_WIDGET_DESTROY(w);
	    return(NULL);
	}

	/* Schedual hide pointer */
	fsimg->hide_pointer_toid = gtk_timeout_add(
	    FSIMG_HIDE_POINTER_DELAY,
	    FSImgHidePointerTOCB, fsimg
	);

	/* Block input until user closes the FSImg? */
	if(block)
	{
	    fsimg->main_loop_level++;
#if (DEBUG_LEVEL >= 1)
g_print(
 "FSImgNew(): Entering main loop level +%i\n",
 fsimg->main_loop_level
);
#endif
	    gtk_main();

	    /* The FSImg should be deleted once it breaks out of the
	     * main loop level, so do not reference it again
	     */

	    /* Always return NULL since the FSImg and its widgets
	     * should be is deleted now
	     */
	    return(NULL);
	}
	else
	{
	    return(w);
	}
}

/*
 *	Ungrabs the pointer as needed, unmaps the FSImg window, and
 *	restores the original video mode.
 */
static void FSImgUnmapRestore(FSImgData *fsimg)
{
	const GdkVideoModeInfo *m;
	GtkWidget *w;
	if(fsimg == NULL)
	    return;

	/* Ungrab pointer */
	if(gdk_pointer_is_grabbed())
	    gdk_pointer_ungrab(GDK_CURRENT_TIME);

	/* Unmap the FSImg toplevel window */
	w = fsimg->toplevel;
	if(w != NULL)
	    gtk_widget_hide(w);

	/* Switch back to the original video mode */
	m = fsimg->prev_vidmode;
#if (DEBUG_LEVEL >= 1)
if(m != NULL)
 g_print(
  "FSImgUnmapRestore(): Restoring video mode %ix%i\n",
  m->viewport_width, m->viewport_height
 );
#endif
	if(gdk_video_mode_switch(m))
	{
	    /* Restore pointer position */
	    FSImgWarpPointer(
		GDK_ROOT_PARENT(),
		fsimg->prev_pointer_x, fsimg->prev_pointer_y
	    );

	    /* Restore viewport position */
	    gdk_video_mode_set_viewport(
		fsimg->prev_viewport_x, fsimg->prev_viewport_y
	    );
	}
	else
	{
	    /* Failed to switch video modes, try to switch to the
	     * default video mode
	     */
	    GList *glist = fsimg->vidmodes_list;
	    while(glist != NULL)
	    {
		m = GDK_VIDEO_MODE_INFO(glist->data);
		if((m != NULL) ? (m->flags & GDK_VIDEO_MODE_DEFAULT) : FALSE)
		{
		    gdk_video_mode_switch(m);
		    break;
		}
		glist = g_list_next(glist);
	    }
	}
}
