#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "guiutils.h"
#include "guirgbimg.h"
#include "headerdlg.h"
#include "config.h"

#include "images/icon_edit_32x32.xpm"
#include "images/icon_edit_48x48.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


typedef struct _Dialog			Dialog;
#define DIALOG(p)			((Dialog *)(p))


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void changed_cb(GtkWidget *widget, gpointer data);
static void set_cb(GtkWidget *widget, gpointer data);
static void cancel_cb(GtkWidget *widget, gpointer data);
static void spin_changed_cb(GtkWidget *widget, gpointer data);
static void to_fit_cb(GtkWidget *widget, gpointer data);
static void center_cb(GtkWidget *widget, gpointer data);
static gint preview_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data);
static void preview_resize(Dialog *d);
static void preview_draw(Dialog *d);

static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
);

gboolean iv_edit_header(
	GtkWidget *ref_toplevel,
	const gchar *filename,
	const guint8 *rgba,
	gint width, gint height, gint bpl,
	gint nframes,
	guint8 *bg_color,	/* 4 bytes in RGBA format */
	gchar **title, gchar **author, gchar **comments,
	gint *x, gint *y,
	gint *base_width, gint *base_height 
);


/*
 *	Dialog:
 */
struct _Dialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gboolean	has_changes,
			got_response;

	GdkCursor	*text_cur,
			*translate_cur;

	GtkWidget	*title_entry,
			*author_entry,
			*x_spin,
			*y_spin,
			*base_width_spin,
			*base_height_spin,
			*to_fit_btn,
			*center_btn,
			*preview_da,
			*comments_text,
			*ok_btn,
			*cancel_btn,
			*close_btn;

	GdkPixmap	*preview_pm;

	gint		preview_button,
			preview_last_x,
			preview_last_y;

	gfloat		preview_dtow_coeff;

	/* Original image (shared) */
	const guint8	*rgba;
	gint		width, height, bpl,
			nframes;

	guint8		bg_color[4];	/* 4 bytes RGBA */

	/* Thumb image for base size preview */
	guint8		*thumb_rgba;
	gint		thumb_width, thumb_height;
};


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	d->got_response = FALSE;
	gtk_main_quit();

	return(TRUE);
}

/*
 *	GtkText event signal callback.
 */
static gint text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Scroll up */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment
		    );
		    gfloat v = adj->value - inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;

	      case GDK_BUTTON5:
		/* Scroll down */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment
		    );
		    gfloat v = adj->value + inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(                   
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	      case GDK_BUTTON5:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(                   
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	    }
	    break;
	}

	return(status);
}

/*
 *	Changed callback.
 */
static void changed_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	{
	    gtk_widget_set_sensitive(d->ok_btn, TRUE);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	    d->has_changes = TRUE;
	}

	d->freeze_count--;
}

/*
 *	Set callback.
 */
static void set_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Cancel callback.
 */
static void cancel_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}



/*
 *	GtkSpinButton "changed" signal callback.
 */
static void spin_changed_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *w;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	w = d->preview_da;
	if(w != NULL)
	{
	    preview_resize(d);
	    gtk_widget_queue_draw(w);
	}

	d->freeze_count--;
}


/*
 *	To fit callback.
 */
static void to_fit_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->x_spin), 0.0f
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->y_spin), 0.0f 
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->base_width_spin), (gfloat)d->width
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->base_height_spin), (gfloat)d->height
	);
}

/*
 *	Center callback.
 */
static void center_cb(GtkWidget *widget, gpointer data)
{
	gint base_width, base_height;
	Dialog *d = DIALOG(data);
	if(d == NULL)                      
	    return;

	if(d->freeze_count > 0)
	    return;

	base_width = gtk_spin_button_get_value_as_int(
	    GTK_SPIN_BUTTON(d->base_width_spin)
	);
	base_height = gtk_spin_button_get_value_as_int(
	    GTK_SPIN_BUTTON(d->base_height_spin)
	);

	if(base_width > 0)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->x_spin),
		(gfloat)((base_width - d->width) / 2)
	    );
	if(base_height > 0)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->y_spin),
		(gfloat)((base_height - d->height) / 2)
	    );
}


/*
 *	Preview GtkDrawingArea event signal callback.
 */
static gint preview_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint status = FALSE;
	gboolean key_press;
	guint keyval;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    GDK_PIXMAP_UNREF(d->preview_pm)
	    d->preview_pm = gdk_pixmap_new(
		widget->window, configure->width, configure->height, -1
	    );
	    preview_resize(d);
	    gtk_widget_queue_draw(widget);
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    preview_draw(d);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget)) 
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
#define STOP_EMIT(_w_)				\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  key_press ?					\
   "key_press_event" : "key_release_event"	\
 );
	    switch(keyval)
	    {
	      case GDK_Up:
	      case GDK_KP_Up:
		if(key_press)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->y_spin)
		    );
		    if(adj != NULL)
			gtk_adjustment_set_value(
			    adj,
			    adj->value - adj->step_increment
			);
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_Down:
	      case GDK_KP_Down:
		if(key_press)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->y_spin)
		    );
		    if(adj != NULL)
			gtk_adjustment_set_value(
			    adj,
			    adj->value + adj->step_increment
			);
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_Left:
	      case GDK_KP_Left:
		if(key_press)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->x_spin)
		    );
		    if(adj != NULL)
			gtk_adjustment_set_value(
			    adj,
			    adj->value - adj->step_increment
			);
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_Right:
	      case GDK_KP_Right:
		if(key_press)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->x_spin)
		    );
		    if(adj != NULL)
			gtk_adjustment_set_value(
			    adj,
			    adj->value + adj->step_increment
			);
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;
	    }
#undef STOP_EMIT
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(GTK_WIDGET_CAN_FOCUS(widget) && !GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    d->preview_button = button->button;
	    d->preview_last_x = (gint)button->x;
	    d->preview_last_y = (gint)button->y;
	    /* Grab pointer */
	    if(!gdk_pointer_is_grabbed())
		gdk_pointer_grab(
		    widget->window,
		    FALSE,
		    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK,
		    NULL,
		    d->translate_cur,
		    button->time
		);
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    d->preview_button = 0;
	    /* Ungrab pointer */
	    if(gdk_pointer_is_grabbed())
		gdk_pointer_ungrab(button->time);
	    status = TRUE;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(d->preview_button == 1)
	    {
		const gfloat dtow_coeff = d->preview_dtow_coeff;
		gint x, y, dx, dy;
		GdkModifierType mask;
		GdkWindow *window = motion->window;

		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
			window, &x, &y, &mask
		    );
		}
		else
		{
		    x = (gint)motion->x;
		    y = (gint)motion->y;
		}
		dx = x - d->preview_last_x;
		dy = y - d->preview_last_y;

		d->preview_last_x = x;
		d->preview_last_y = y;

		if(dtow_coeff > 0.0f)
		{
		    if(dx != 0)
		    {
			const gint dwx = (dx > 0) ?
			    MAX((gint)(dx / dtow_coeff), 1) :
			    MIN((gint)(dx / dtow_coeff), -1);
			x = GTK_ENTRY_GET_VALUEI(d->x_spin) + dwx;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(d->x_spin), (gfloat)x
			);
		    }
		    if(dy != 0)
		    {
			const gint dwy = (dy > 0) ?
			    (gint)MAX(dy / dtow_coeff, 1) :
			    (gint)MIN(dy / dtow_coeff, -1);
			y = GTK_ENTRY_GET_VALUEI(d->y_spin) + dwy;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(d->y_spin), (gfloat)y
			);
		    }
		}
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}


/*
 *	Updates the geometry of the Preview GtkDrawingArea and the
 *	preview_dtow_coeff.
 */
static void preview_resize(Dialog *d)
{
	gint width, height, bwidth, bheight;
	GtkWidget *w = (d != NULL) ? d->preview_da : NULL;
	if(w == NULL)
	    return;

	/* Get size of the preview widget */
	width = w->allocation.width;
	height = w->allocation.height;

	/* Get the base size */
	bwidth = GTK_ENTRY_GET_VALUEI(d->base_width_spin);
	bheight = GTK_ENTRY_GET_VALUEI(d->base_height_spin);

	/* If base size is not set then use the image data size */
	if(bwidth <= 0)
	    bwidth = d->width;
	if(bheight <= 0)
	    bheight = d->height;

	/* Reduce window size to accomidate for the frame borders */
	width -= 4;
	height -= 4;

	/* Calculate the RGBA image data to window coordinates
	 * coeffcicient
	 */
	if(bwidth > bheight)
	    d->preview_dtow_coeff = (bwidth > 0) ?
		((gfloat)width / (gfloat)bwidth) : 0.0f;
	else
	    d->preview_dtow_coeff = (bheight > 0) ?
		((gfloat)height / (gfloat)bheight) : 0.0f;
}

/*
 *	Redraws the Preview GtkDrawingArea.
 */
static void preview_draw(Dialog *d)
{
	gfloat baspect, dtow_coeff;
	gint x, y, width, height, bwidth_orig;
	GdkRectangle base;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w;

	if(d == NULL)
	    return;

	w = d->preview_da;
	if(w == NULL)
	    return;

	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	pixmap = d->preview_pm;
	drawable = (GdkDrawable *)((pixmap != NULL) ? pixmap : window);
	gdk_window_get_size(drawable, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	/* Draw background */
	gdk_draw_rectangle(
	    drawable, style->bg_gc[state], TRUE,
	    0, 0, width, height
	);

	/* Get base geometry */
	bwidth_orig = base.width = GTK_ENTRY_GET_VALUEI(d->base_width_spin);
	base.height = GTK_ENTRY_GET_VALUEI(d->base_height_spin);
	if(base.width <= 0)
	    bwidth_orig = base.width = d->width;
	if(base.height <= 0)
	    base.height = d->height;

	/* Calculate the base's aspect */
	baspect = (base.height > 0) ?
	    ((gfloat)base.width / (gfloat)base.height) : 0.0f;

	/* Calculate the base geometry in window coordinates and
	 * include the frame borders
	 */
	if(base.width > base.height)
	{
	    base.width = width - 4;
	    base.height = (baspect > 0.0f) ?
		(gint)(base.width / baspect) : base.width;
	    base.x = 2;
	    base.y = (height - base.height) / 2;
	}
	else
	{
	    base.height = height - 4;
	    base.width = (baspect > 0.0f) ?
		(gint)(base.height * baspect) : base.height;
	    base.x = (width - base.width) / 2;
	    base.y = 2;
	}
	if((base.width <= 5) || (base.height <= 5))
	    return;

	/* Draw base background */
	gdk_draw_rectangle(
	    drawable, style->base_gc[state], TRUE,
	    base.x, base.y, base.width, base.height
	);

	/* Get the RGBA image data to window coordinates coefficient */
	dtow_coeff = d->preview_dtow_coeff;

	/* Calculate thumb position in window coordinates */
	x = (gint)(GTK_ENTRY_GET_VALUEI(d->x_spin) * dtow_coeff) + base.x;
	y = (gint)(GTK_ENTRY_GET_VALUEI(d->y_spin) * dtow_coeff) + base.y;

	/* Draw thumb */
	if((d->thumb_rgba != NULL) && (d->rgba != NULL))
	{
	    const gint	src_width = d->width,
			src_height = d->height,
			src_bpl = d->bpl,
			thumb_width = MIN(
		(gint)(src_width * dtow_coeff), d->thumb_width
	    ),
			thumb_height = MIN(
		(gint)(src_height * dtow_coeff), d->thumb_height
	    );
	    guint8 *thumb_rgba = d->thumb_rgba;
	    const guint8 *src_rgba = d->rgba;
	    guint8 bg[4];
	    GdkColor *bg_c = &style->base[state];
	    GdkGC *gc = style->fg_gc[state];

	    bg[0] = (guint8)(bg_c->red >> 8);
	    bg[1] = (guint8)(bg_c->green >> 8);
	    bg[2] = (guint8)(bg_c->blue >> 8);
	    bg[3] = 0xff;

	    /* Copy/resize the source image data to the thumb image data */
	    GUIImageBufferResize(
		4,
		src_rgba, src_width, src_height, src_bpl,
		thumb_rgba, thumb_width, thumb_height, -1,
		NULL, NULL
	    );
	    /* Apply background to the thumb image data */
	    GUIImageBufferFlattenWithBG(
		4,
		thumb_rgba, thumb_width, thumb_height, -1,
		bg,
		NULL, NULL
	    );
	    /* Draw thumb image data */
	    gdk_gc_set_clip_rectangle(gc, &base);
	    gdk_draw_rgb_32_image(
		drawable, gc,
		x, y,
		thumb_width, thumb_height,
		GDK_RGB_DITHER_NORMAL,
		(guchar *)thumb_rgba,
		thumb_width * 4
	    );
	    gdk_gc_set_clip_rectangle(gc, NULL);
	}

	/* Draw the frame */
	gtk_draw_shadow(
	    style, drawable, state, GTK_SHADOW_IN,
	    base.x - 2, base.y - 2,
	    base.width + 4, base.height + 4
	);

	/* Draw the focus */
	if(GTK_WIDGET_HAS_FOCUS(w))
#if defined(WIN32)
	    gdk_draw_rectangle(
		drawable, style->fg_gc[state], FALSE,
		base.x - 2, base.y - 2,
		base.width + 4 - 1, base.height + 4 - 1
	    );
#else
	    gtk_draw_focus(
		style, drawable,
		base.x - 2, base.y - 2,
		base.width + 4 - 1, base.height + 4 - 1
	    );
#endif

	/* Send drawable to window if drawable is not the window */
	if(drawable != window)
	    gdk_draw_pixmap(
		window, style->fg_gc[state], drawable,
		0, 0, 0, 0, width, height
	    );
}

/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;

	/* GtkAccelGroup */
	d->accelgrp = accelgrp = gtk_accel_group_new();

	/* Toplevel GtkWindow */
	d->toplevel = toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_edit_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Set Button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Conjunto",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Serie",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Satz",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Serie",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Het Stel",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Jogo",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Sett",
#else
	    "Set",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(set_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);

	/* Cancel Button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close Button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
	    "Close",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);
}


/*
 *	Queries user to edit header.
 *
 *	Returns TRUE on Set or FALSE on Cancel.
 */
gboolean iv_edit_header(
	GtkWidget *ref_toplevel,
	const gchar *filename,
	const guint8 *rgba,
	gint width, gint height, gint bpl,
	gint nframes,
	guint8 *bg_color,	/* 4 bytes in RGBA format */
	gchar **title, gchar **author, gchar **comments,
	gint *x, gint *y,
	gint *base_width, gint *base_height
)
{
	const gint	border_major = 5,
			border_minor = 2;
	const guint32	def_bg_color = 0xffffffff;
	gboolean got_response;
	gchar *s;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	Dialog *d;

	/* Create dialog base widgets */
	d = DIALOG(g_malloc0(sizeof(Dialog)));
	s = g_strdup_printf(
	    "Header: %s",
	    STRISEMPTY(filename) ? "Untitled" : filename
	);
	build_std_dialog(
	    d,
	    400, -1,
	    s,					/* Title */
	    ref_toplevel,
	    &toplevel, &client_vbox
	);
	g_free(s);
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	d->preview_button = 0;
	d->preview_last_x = 0;
	d->preview_last_y = 0;
	d->preview_dtow_coeff = 0.0f;

	d->rgba = rgba;
	d->width = width;
	d->height = height;
	d->bpl = bpl;
	d->nframes = nframes;

	memcpy(
	    d->bg_color,
	    (bg_color != NULL) ?
		bg_color : (const guint8 *)&def_bg_color,
	    4 * sizeof(guint8)
	);

	d->freeze_count++;

	/* Create Header Dialog widgets */
	if(client_vbox != NULL)
	{
	    gint	label_width, value_width,
			preview_width = 160,
			preview_height = 160;
	    GdkWindow *window = toplevel->window;
	    GtkAdjustment *adj;
	    GtkRcStyle *rcstyle;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget	*w, *sb,
			*parent = client_vbox,
			*parent2, *parent3, *parent4, *parent5,
			*parent6, *parent7;
	    GtkEditable *editable;
	    GtkText *text;

	    d->text_cur = gdk_cursor_new(GDK_XTERM);
	    d->translate_cur = gdk_cursor_new(GDK_FLEUR);

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent = w;


	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		window,
		(style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		(guint8 **)icon_edit_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    label_width = 60;

	    /* Title */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Title:");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_container_add(GTK_CONTAINER(parent4), w);
	    gtk_widget_show(w);
	    /* GtkEntry */
	    d->title_entry = w = gtk_entry_new();
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Enter the title of this image"
	    );
	    gtk_widget_show(w);

	    /* Author */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Author:");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_container_add(GTK_CONTAINER(parent4), w);
	    gtk_widget_show(w);
	    /* GtkEntry */
	    d->author_entry = w = gtk_entry_new();
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Enter the name of the author of this image"
	    );
	    gtk_widget_show(w);

	    w = gtk_hseparator_new();
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* GtkBox for the Position & Base Size and the Layout
	     * Preview
	     */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    label_width = 85;
	    value_width = 60;

	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;

	    /* X Position */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent7 = w;
	    w = gtk_label_new("X:");
	    gtk_container_add(GTK_CONTAINER(parent7), w);
	    gtk_widget_show(w);
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		(gfloat)-((guint16)-1),
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f  
	    );
	    d->x_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Set the x coordinate offset of this image on its base (in pixels)"
	    );
	    gtk_widget_show(w);

	    /* Y Position */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent7 = w;
	    w = gtk_label_new("Y:");
	    gtk_container_add(GTK_CONTAINER(parent7), w);
	    gtk_widget_show(w); 
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		(gfloat)-((guint16)-1),
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f  
	    );
	    d->y_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Set the y coordinate offset of this image on its base (in pixels)"
	    );
	    gtk_widget_show(w);

	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;

	    /* Base Width */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent7 = w;
	    w = gtk_label_new("Base Width:");
	    gtk_container_add(GTK_CONTAINER(parent7), w);
	    gtk_widget_show(w);
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		0.0f,
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f
	    );
	    d->base_width_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Set the width of the base (in pixels)"
	    );
	    gtk_widget_show(w);

	    /* Base Height */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkAlignment & GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent7 = w;
	    w = gtk_label_new("Base Height:");
	    gtk_container_add(GTK_CONTAINER(parent7), w);
	    gtk_widget_show(w);
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		0.0f,
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f
	    );
	    d->base_height_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(
		w,
"Set the height of the base (in pixels)"
	    );
	    gtk_widget_show(w);


	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;


	    /* To Fit */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkButton */
	    d->to_fit_btn = w = gtk_button_new_with_label("To Fit");
	    gtk_box_pack_end(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 60, -1);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(to_fit_cb), d
	    );
	    GUISetWidgetTip(w,
"Click this to automatically set the position and base size to fit\
 the current image's size"
	    );
	    gtk_widget_show(w);

	    /* Center */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent6 = w;
	    /* GtkButton */
	    d->center_btn = w = gtk_button_new_with_label("Center");
	    gtk_box_pack_end(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 60, -1);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(center_cb), d
	    );
	    GUISetWidgetTip(w,
"Click this to automatically set the position of the image so that it\
 is centered within the specified base size"
	    );
	    gtk_widget_show(w);


	    /* Layout Frame */
	    w = gtk_frame_new("Layout");
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_hbox_new(TRUE, 0);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_container_add(GTK_CONTAINER(parent4), w);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Preview GtkDrawingArea */
	    d->preview_da = w = gtk_drawing_area_new();
	    GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	    gtk_widget_set_usize(w, preview_width, preview_height);
	    gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(preview_event_cb), d
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_realize(w);
	    gtk_widget_show(w);
		     
	    /* Modify the preview drawing area style to match the
	     * background color
	     */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->color_flags[GTK_STATE_NORMAL] = GTK_RC_BASE;
	    GDK_COLOR_SET_BYTE(
		&rcstyle->base[GTK_STATE_NORMAL],
		d->bg_color[0], d->bg_color[1], d->bg_color[2]
	    );
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);

	    /* Preview thumb data
	     *
	     * Allocate it to two times the size of the preview
	     * GtkDrawingArea for reasonable (but not perfect) 2 times
	     * the max zoom coverage
	     */
	    d->thumb_width = preview_width * 2;
	    d->thumb_height = preview_height * 2;
	    d->thumb_rgba = (guint8 *)g_malloc(
		d->thumb_width * d->thumb_height * 4 * sizeof(guint8)
	    );


	    /* Comments GtkFrame */
	    w = gtk_frame_new("Comments");
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w); 
	    parent3 = w;
	    w = gtk_table_new(1, 2, FALSE);
	    gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	    gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Text GtkRcStyle */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	    );
	    /* GtkText */
	    d->comments_text = w = gtk_text_new(NULL, NULL);
	    editable = GTK_EDITABLE(w);
	    text = GTK_TEXT(w);
	    gtk_widget_set_usize(w, -1, 80);
	    text->default_tab_width = 8;
	    gtk_text_set_editable(text, TRUE);
	    gtk_text_set_word_wrap(text, TRUE);
	    gtk_table_attach(
		GTK_TABLE(parent3), w,
		0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(text_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(text_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_widget_realize(w);
	    gtk_widget_modify_style(w, rcstyle);
	    gdk_window_set_cursor(w->window, d->text_cur);
	    GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_UNDO);
	    gtk_widget_show(w);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    /* Vertical GtkScrollBar */
	    sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	    gtk_table_attach(
		GTK_TABLE(parent3), sb,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	    );
	    gtk_widget_show(sb);


#if 0
	    /* Limitations notice */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    w = gtk_pixmap_new_from_xpm_d(
		window,
		(style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		(guint8 **)icon_info_20x20_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    w = gtk_label_new(
"Not all values are supported by all image formats, verify\n\
that the image format supports the value(s) in question\n\
before editing and saving."
	    );
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
#endif


	    /* Set initial values */

	    /* Title */
	    if((title != NULL) ? !STRISEMPTY(*title) : FALSE)
		gtk_entry_set_text(
		    GTK_ENTRY(d->title_entry), *title
		);
	    gtk_widget_grab_focus(d->title_entry);	/* Grab focus */

	    /* Author */
	    if((author != NULL) ? !STRISEMPTY(*author) : FALSE)
		gtk_entry_set_text(
		    GTK_ENTRY(d->author_entry), *author
		);

	    /* X */
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->x_spin),
		(x != NULL) ? (gfloat)*x : 0.0f
	    );

	    /* Y */
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->y_spin),
		(y != NULL) ? (gfloat)*y : 0.0f
	    );

	    /* Base Width */
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->base_width_spin),
		(base_width != NULL) ? (gfloat)*base_width : width
	    );
	    /* Base Height */
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->base_height_spin),
		(base_height != NULL) ? (gfloat)*base_height : height
	    );

	    /* Comments */
	    if((comments != NULL) ? !STRISEMPTY(*comments) : FALSE)
	    {
		text = GTK_TEXT(d->comments_text);
		gtk_text_freeze(text);
		gtk_text_insert(
		    text, NULL, NULL, NULL,
		    *comments, -1
		);
		gtk_text_thaw(text);
	    }
	}

	d->freeze_count--;

	/* Map the dialog */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	got_response = d->got_response;
	if(got_response)
	{
	    if(title != NULL)
	    {
		g_free(*title);
		*title = STRDUP(gtk_entry_get_text(
		    GTK_ENTRY(d->title_entry)
		));
	    }
	    if(author != NULL)
	    {
		g_free(*author);
		*author = STRDUP(gtk_entry_get_text(
		    GTK_ENTRY(d->author_entry)
		));
	    }
	    if(x != NULL)
		*x = GTK_ENTRY_GET_VALUEI(d->x_spin);
	    if(y != NULL)
		*y = GTK_ENTRY_GET_VALUEI(d->y_spin);
	    if(base_width != NULL)
		*base_width = GTK_ENTRY_GET_VALUEI(d->base_width_spin);
	    if(base_height != NULL)
		*base_height = GTK_ENTRY_GET_VALUEI(d->base_height_spin);
	    if(comments != NULL)
	    {
		g_free(*comments);
		*comments = gtk_editable_get_chars(
		    GTK_EDITABLE(d->comments_text), 0, -1
		);
	    }
	}


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(d->title_entry);
	GTK_WIDGET_DESTROY(d->author_entry);
	GTK_WIDGET_DESTROY(d->x_spin);
	GTK_WIDGET_DESTROY(d->y_spin);
	GTK_WIDGET_DESTROY(d->base_width_spin);
	GTK_WIDGET_DESTROY(d->base_height_spin);
	GTK_WIDGET_DESTROY(d->to_fit_btn);
	GTK_WIDGET_DESTROY(d->center_btn);
	GTK_WIDGET_DESTROY(d->preview_da);
	GTK_WIDGET_DESTROY(d->comments_text);
	GTK_WIDGET_DESTROY(d->toplevel);
	GDK_PIXMAP_UNREF(d->preview_pm);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	GDK_CURSOR_DESTROY(d->text_cur);
	GDK_CURSOR_DESTROY(d->translate_cur);
	g_free(d->thumb_rgba);
	g_free(d);

	return(got_response);
}
