#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "guiutils.h"
#include "guirgbimg.h"
#include "imginfodlg.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_info_32x32.xpm"


/*
 *	Display style:
 *
 *	0	Thumb on the left and Image and File labels on the
 *		right
 *	1	Thumb on the left and Image, File, and Display labels
 *		on the right
 *	2	Info icon on the left and Image, File, and Display
 *		labels on the right (thumb on right)
 */
#define DISPLAY_STYLE		1


static gchar *ImgInfoGetDeliminatedSizeString(gulong i);

static gint ImgInfoDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void ImgInfoDlgOKCB(GtkWidget *widget, gpointer data);

imginfodlg_struct *ImgInfoDlgNew(GtkWidget *ref_toplevel);
void ImgInfoDlgSetValues(
	imginfodlg_struct *d,
	const guint8 *rgba,
	gint width, gint height,
	gint depth,			/* Bytes Per Pixel */
	gulong line,			/* Bytes Per Line */
	gint nframes,
	gulong size,			/* Total Bytes */
	gint alpha_channel,		/* 0 = None
					 * 1 = Allocated But Uniform
					 * 2 = Allocated & Defined */
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint x, gint y,
	gint base_width, gint base_height,
	gboolean has_changes,

	const gchar *file_name,
	const gchar *file_format,
	gulong file_size,		/* Bytes */
	const gchar *title,
	const gchar *author,
	const gchar *creator,
	const gchar *comments,
	gulong last_modified,

	const gchar *display,		/* X Display Address */
	const gchar *visual_type,	/* X Visual Type */
	gint display_depth		/* X Depth In Bits Per Pixel */
);
void ImgInfoDlgMap(imginfodlg_struct *d);
void ImgInfoDlgUnmap(imginfodlg_struct *d);
void ImgInfoDlgDelete(imginfodlg_struct *d);


#define IMG_INFO_DLG_DEF_TITLE		"Information"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a statically allocated string describing the value i
 *	with ',' deliminators at each thousands place.
 */
static gchar *ImgInfoGetDeliminatedSizeString(gulong i)
{
	gint comma_countdown, slen;
	gchar ss[256], *ss_ptr;
	static gchar ts[256], *ts_ptr;


	g_snprintf(ss, sizeof(ss), "%ld", i);
	slen = STRLEN(ss);

	/* 3 digits or less? (no commas needed) */
	if(slen <= 3)
	{
	    strcpy(ts, ss);
	    return(ts);
	}

	ts_ptr = ts;
	ss_ptr = ss;

	/* Initialize comma counter */
	comma_countdown = slen % 3;
	if(comma_countdown <= 0)
	    comma_countdown = 3;

	/* Iterate through size string until end is reached */
	while(*ss_ptr != '\0')
	{
	    /* Reset comma counter and put in a comma? */
	    if(comma_countdown <= 0)
	    {
		*ts_ptr++ = ',';
		comma_countdown = 3;
	    }

	    *ts_ptr++ = *ss_ptr++;
	    comma_countdown--;
	}
	 
	/* Null terminate return string */
	*ts_ptr = '\0';

	return(ts);
}


/*
 *	Image Info Dialog toplevel GtkWindow "delete_event" signal
 *	callback.
 */
static gint ImgInfoDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	imginfodlg_struct *d = IMGINFODLG(data);
	if(d == NULL)
	    return(FALSE);

	ImgInfoDlgOKCB(d->ok_btn, d);
	return(TRUE);
}

/*
 *	Image Info Dialog OK callback.
 */
static void ImgInfoDlgOKCB(GtkWidget *widget, gpointer data)
{
	imginfodlg_struct *d = IMGINFODLG(data);
	if(d == NULL)
	    return;

	ImgInfoDlgUnmap(d);
}


/*
 *	Creates a new Image Info Dialog.
 */
imginfodlg_struct *ImgInfoDlgNew(GtkWidget *ref_toplevel)
{
	const gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
			bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
			border_major = 5;
	const gchar *banner_font_name =
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1";
	GdkColor fg, bg;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4,
			*parent5, *parent6;
	GtkWidget *main_vbox;
	imginfodlg_struct *d = IMGINFODLG(g_malloc0(
	    sizeof(imginfodlg_struct)
	));

	/* Reset values */

	/* Create keyboard accelerator group */
	d->accelgrp = accelgrp = gtk_accel_group_new();

	/* Create toplevel */
	d->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_info_32x32_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(ImgInfoDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if(ref_toplevel != NULL)
	    gtk_window_set_transient_for(
		GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	    );
	style = gtk_widget_get_style(w);
	parent = w;

	d->main_vbox = main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Hbox for multiple columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Left side GtkVBox for the icon */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icon */
#if (DISPLAY_STYLE == 2)
	w = gtk_pixmap_new_from_xpm_d(
	    window, &style->bg[GTK_STATE_NORMAL],
	    (guint8 **)icon_info_32x32_xpm
	);
#else
        d->thumb_pm = w = gtk_pixmap_new_from_xpm_d(
            window, &style->bg[GTK_STATE_NORMAL],
	    (guint8 **)icon_info_32x32_xpm
        );
#endif
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Right side GtkVBox for labels */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkHBox for the labels */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkVBox for the Image information */
	d->img_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Image Banner */
        memcpy(&fg, &style->white, sizeof(GdkColor));
/*	GDK_COLOR_SET_COEFF(&fg, 1.0f, 1.0f, 1.0f); */
	memcpy(&bg, &style->black, sizeof(GdkColor));
/*	GDK_COLOR_SET_COEFF(&bg, 0.0f, 0.0f, 0.0f); */
	w = GUIBannerCreate(
	    "Image",
	    banner_font_name,
	    fg, bg,
	    GTK_JUSTIFY_CENTER,
	    FALSE
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Hbox for Image labels */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Parameters Label */
 	d->img_parm_label = w = gtk_label_new("\
Width:\n\
Height:\n\
Bytes Per Pixel:\n\
Bytes Per Line:\n\
Frames:\n\
Total Bytes:\n\
Alpha Channel:\n\
X:\n\
Y:\n\
Base Width:\n\
Base Height:\n\
Has Changes:"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Values Label */
	d->img_val_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* GtkVBox for the File information */
	d->file_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* File Banner */
        memcpy(&fg, &style->white, sizeof(GdkColor));
/*      GDK_COLOR_SET_COEFF(&fg, 1.0f, 1.0f, 1.0f); */
        memcpy(&bg, &style->black, sizeof(GdkColor));
/*      GDK_COLOR_SET_COEFF(&bg, 0.0f, 0.0f, 0.0f); */
	w = GUIBannerCreate(
	    "File", 
	    banner_font_name,
	    fg, bg,
	    GTK_JUSTIFY_CENTER,
	    FALSE
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* GtkHBox for File labels */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Parameters Label */
	d->file_parm_label = w = gtk_label_new("\
File:\n\
Format:\n\
Size:\n\
Title:\n\
Author:\n\
Creator:\n\
Last Modified:"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Values Label */
	d->file_val_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

#if (DISPLAY_STYLE == 1) || (DISPLAY_STYLE == 2)
	/* GtkVBox for the Display information */
	d->dpy_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Display Banner */
        memcpy(&fg, &style->white, sizeof(GdkColor));
/*      GDK_COLOR_SET_COEFF(&fg, 1.0f, 1.0f, 1.0f); */
        memcpy(&bg, &style->black, sizeof(GdkColor));
/*      GDK_COLOR_SET_COEFF(&bg, 0.0f, 0.0f, 0.0f); */
	w = GUIBannerCreate(
	    "Display",
	    banner_font_name,
	    fg, bg,
	    GTK_JUSTIFY_CENTER,
	    FALSE
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
#if (DISPLAY_STYLE == 2)
	/* GtkHBox for the Thumb */
	w = gtk_hbox_new(TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Thumb Pixmap */
	d->thumb_pm = w = gtk_pixmap_new_from_xpm_d(
	    window, NULL, (guint8 **)icon_info_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
#endif
	/* Hbox for Display labels */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Parameters Label */
	d->dpy_parm_label = w = gtk_label_new("\
Display:\n\
Visual:\n\
Depth:"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Values Label */
	d->dpy_val_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
#endif

	/* GtkFrame for comments */
	d->comments_frame = w = gtk_frame_new("Comments");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Comments GtkLabel */
	d->comments_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);



	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for the buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK Button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(ImgInfoDlgOKCB), d
	);
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_o, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);



	return(d);
}

/*
 *	Sets the Image Info Dialog's values.
 */
void ImgInfoDlgSetValues(
	imginfodlg_struct *d,
	const guint8 *rgba,
	gint width, gint height,
	gint depth,			/* Bytes Per Pixel */
	gulong line,			/* Bytes Per Line */
	gint nframes,
	gulong size,			/* Total Bytes */
	gint alpha_channel,		/* 0 = None
					 * 1 = Allocated But Uniform
					 * 2 = Allocated & Defined */
	const guint8 *bg_color,		/* 4 bytes RGBA */
	gint x, gint y,
	gint base_width, gint base_height,
	gboolean has_changes,

	const gchar *file_name,
	const gchar *file_format,
	gulong file_size,		/* Bytes */
	const gchar *title,
	const gchar *author,
	const gchar *creator, 
	const gchar *comments,
	gulong last_modified,

	const gchar *display,		/* X Display Address */
	const gchar *visual_type,	/* X Visual Type */
	gint display_depth		/* X Depth In Bits Per Pixel */
)
{
	const gchar	*no_val_s = "-",
			*alpha_channel_s,
			*last_modified_s;
	gchar *s, *format_desc = NULL;
	GtkWidget *w, *toplevel;

	if(d == NULL)
	    return;

	toplevel = d->toplevel;

	/* Get Alpha Channel String */
	switch(alpha_channel)
	{
	  case 0:
	    alpha_channel_s = "No";
	    break;
	  case 1:
	    alpha_channel_s = "Yes (Uniform)";
	    break;
	  default:
	    alpha_channel_s = "Yes";
	    break;
	}

	/* Get Last Modified String */
	if(last_modified > 0)
	{
	    time_t t = (time_t)last_modified;
	    last_modified_s = ctime(&t);
	    s = strchr(last_modified_s, '\n');
	    if(s != NULL)
		*s = '\0';
	}
	else 
	{
	    last_modified_s = no_val_s;
	}

	/* Get format string */
	if(!STRISEMPTY(file_format))
	{
	    const gchar *img_list[] =
#ifdef IV_IMAGE_LIST
		IV_IMAGE_LIST;
#else
		{ NULL, NULL };
#endif
	    gint i;
	    for(i = 0; img_list[i] != NULL; i += 2)
	    {
		if(!g_strcasecmp(file_format, img_list[i]))
		{
		    g_free(format_desc);
		    format_desc = STRDUP(img_list[i + 1]);
		    break;
		}
	    }
	}

	/* Title */
	w = d->toplevel;
	s = g_strdup_printf(
	    IMG_INFO_DLG_DEF_TITLE ": %s",
	    STRISEMPTY(file_name) ? "Untitled" : file_name
	);
	gtk_window_set_title(GTK_WINDOW(w), s);
	g_free(s);


	/* Thumb */
	w = d->thumb_pm;
	if((w != NULL) && (rgba != NULL) && (width > 0) && (height > 0))
	{
	    const gint	thumb_bpp = 4,
			thumb_width_max = 100,
			thumb_height_max = 100;
	    gfloat aspect = (gfloat)width / (gfloat)height;
	    gint	thumb_width = width,
			thumb_height = height,
			thumb_bpl;
	    guint8	*thumb_rgba;
	    GdkBitmap	*mask;
	    GdkPixmap	*pixmap;
	    GdkWindow	*window = w->window;
	    GtkStyle	*style = gtk_widget_get_style(w);

	    gtk_widget_show(w);

	    if(aspect <= 0.0f)
		aspect = 1.0f;

	    if(thumb_width > thumb_width_max)
	    {
		thumb_width = thumb_width_max;
		thumb_height = (gint)(thumb_width / aspect);
	    }
	    if(thumb_height > thumb_height_max)
	    {
		thumb_height = thumb_height_max;
		thumb_width = (gint)(thumb_height * aspect);
	    }
	    thumb_bpl = thumb_width * thumb_bpp;

	    /* Allocate thumb image data */
	    thumb_rgba = (guint8 *)g_malloc(thumb_bpl * thumb_height);
	    GUIImageBufferResize(
		thumb_bpp,
		rgba,
		width, height, (gint)line,
		thumb_rgba,
		thumb_width, thumb_height, thumb_bpl,
		NULL, NULL
	    );

	    /* Create the pixmap and mask from the thumb image data*/
	    pixmap = gdk_pixmap_new(
		window, thumb_width, thumb_height, -1
	    );
	    if((pixmap != NULL) && (thumb_rgba != NULL))
		gdk_draw_rgb_32_image(
		    pixmap, style->black_gc,
		    0, 0, thumb_width, thumb_height,
		    GDK_RGB_DITHER_NORMAL,
		    (guchar *)thumb_rgba,
		    thumb_bpl
		);
	    mask = GUICreateBitmapFromDataRGBA(
		thumb_width, thumb_height, -1,
		thumb_rgba,
		0x80,			/* Alpha byte threshold */
		window
	    );

	    /* Delete the thumb image data */
	    g_free(thumb_rgba);

	    /* Set new thumb pixmap */
	    if(pixmap != NULL)
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);

	    GDK_PIXMAP_UNREF(pixmap);
	    GDK_BITMAP_UNREF(mask);
	}
	if((w != NULL) && (rgba == NULL))
	    gtk_widget_hide(w);

	/* Image */
	w = d->img_vbox;
	if(w != NULL)
	{
	    if((width > 0) && (height > 0))
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	w = d->img_val_label;
	if((w != NULL) && (width > 0) && (height > 0))
	{
	    gchar	*width_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(width)
	    ),
			*height_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(height)
	    ),
			*depth_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(depth)
	    ),
			*line_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(line)
	    ),
			*nframes_s = STRDUP(
                ImgInfoGetDeliminatedSizeString(nframes)
            ),
			*size_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(size)
	    ),
			*base_width_s = (base_width > 0) ?
		STRDUP(ImgInfoGetDeliminatedSizeString(base_width)) :
		STRDUP(no_val_s),
			*base_height_s = (base_height > 0) ?
		STRDUP(ImgInfoGetDeliminatedSizeString(base_height)) :
		STRDUP(no_val_s);
	    s = g_strdup_printf("\
%s\n\
%s\n\
%s\n\
%s\n\
%s\n\
%s\n\
%s\n\
%i\n\
%i\n\
%s\n\
%s\n\
%s",
		width_s,	/* Pixels */
		height_s,	/* Pixels */
		depth_s,	/* Bytes Per Pixel */
		line_s,		/* Bytes Per Line */
		nframes_s,	/* Number Of Frames */
		size_s,		/* Total Bytes */
		alpha_channel_s,
		x, y,
		base_width_s, base_height_s,
		(has_changes) ? "Yes" : "No"
	    );
	    gtk_label_set_text(GTK_LABEL(w), s);
	    g_free(s);

	    g_free(width_s);
	    g_free(height_s);
	    g_free(depth_s);
	    g_free(line_s);
	    g_free(nframes_s);
	    g_free(size_s);
            g_free(base_width_s);
	    g_free(base_height_s);
	}


	/* File */
	w = d->file_vbox;
	if(w != NULL)
	{
	    if(!STRISEMPTY(file_name))
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	w = d->file_val_label;
	if((w != NULL) && !STRISEMPTY(file_name))
	{
	    gchar	*file_size_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(file_size)
	    );

	    s = g_strdup_printf("\
%s\n\
%s\n\
%s byte%s\n\
%s\n\
%s\n\
%s\n\
%s",
		file_name,
		STRISEMPTY(format_desc) ?
		    (STRISEMPTY(file_format) ? no_val_s : file_format) :
		    format_desc,
		file_size_s, (file_size == 1) ? "" : "s",
		STRISEMPTY(title) ? no_val_s : title,
		STRISEMPTY(author) ? no_val_s : author,
		STRISEMPTY(creator) ? no_val_s : creator,
		last_modified_s
	    );
	    gtk_label_set_text(GTK_LABEL(w), s);
	    g_free(s);

	    g_free(file_size_s);
	}

	/* Display */
	w = d->dpy_vbox;
	if(w != NULL)
	{
	    gtk_widget_show(w);
	}

	w = d->dpy_val_label;
	if(w != NULL)
	{
	    gchar       *display_depth_s = STRDUP(
		ImgInfoGetDeliminatedSizeString(display_depth)
	    );

	    s = g_strdup_printf("\
%s\n\
%s\n\
%s bit%s per pixel",
		STRISEMPTY(display) ? no_val_s : display,
		STRISEMPTY(visual_type) ? no_val_s : visual_type,
		display_depth_s, (display_depth == 1) ? "" : "s"
	    );
	    gtk_label_set_text(GTK_LABEL(w), s);
	    g_free(s);

	    g_free(display_depth_s);
	}


	/* Comments */
	w = d->comments_frame;
	if(w != NULL)
	{
	    if(!STRISEMPTY(comments))
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	w = d->comments_label;
	if((w != NULL) && !STRISEMPTY(comments))
	{
	    gtk_label_set_text(GTK_LABEL(w), comments);
	}


	g_free(format_desc);
}

/*
 *	Checks if the Image Info Dialog is mapped.
 */
gboolean ImgInfoDlgIsMapped(imginfodlg_struct *d)
{
	GtkWidget *w = (d != NULL) ? d->toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Maps the Image Info Dialog.
 */
void ImgInfoDlgMap(imginfodlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	w = d->toplevel;
	if(w != NULL)
	    gtk_widget_show_raise(w);

	w = d->ok_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_default(w);
	    gtk_widget_grab_focus(w);
	}
}

/*
 *	Unmaps the Image Info Dialog.
 */
void ImgInfoDlgUnmap(imginfodlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	w = d->toplevel;
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Deletes the Image Info Dialog.
 */
void ImgInfoDlgDelete(imginfodlg_struct *d)
{
	if(d == NULL)
	    return;

	GTK_WIDGET_DESTROY(d->thumb_pm);
	GTK_WIDGET_DESTROY(d->img_parm_label);
	GTK_WIDGET_DESTROY(d->img_val_label);
	GTK_WIDGET_DESTROY(d->img_vbox);
	GTK_WIDGET_DESTROY(d->file_parm_label);
	GTK_WIDGET_DESTROY(d->file_val_label);
	GTK_WIDGET_DESTROY(d->file_vbox);
	GTK_WIDGET_DESTROY(d->dpy_parm_label);
	GTK_WIDGET_DESTROY(d->dpy_val_label);
	GTK_WIDGET_DESTROY(d->dpy_vbox);
	GTK_WIDGET_DESTROY(d->comments_label);
	GTK_WIDGET_DESTROY(d->comments_frame);
	GTK_WIDGET_DESTROY(d->ok_btn);
	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	g_free(d);
}
