#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(HAVE_IMLIB)
# include <Imlib.h>
#elif defined(HAVE_IMLIB2)
# include <X11/Xlib.h>
# include <Imlib2.h>
#endif

#if defined(_WIN32)
# include "../include/string.h"
#endif

#include "libps.h"
#include "imgio.h"
#include "config.h"


#ifdef HAVE_IMLIB
extern void	*imlib_handle;
#endif  /* HAVE_IMLIB */


/* Last error message pointer */
const char	*imgio_last_load_error = NULL;
const char	*imgio_last_write_error = NULL;


/*
 *	IO Types:
 *
 *	Determines which library to use to open or save the image.
 */
typedef enum {
	IO_TYPE_ANY,		/* Let us decide */
	IO_TYPE_IMLIB,		/* Imlib or Imlib2 */
	IO_TYPE_GIF,
	IO_TYPE_JPEG,
	IO_TYPE_PNG,		/* PNG or MNG */
	IO_TYPE_PS,
	IO_TYPE_TGA,
	IO_TYPE_XPM
} imgio_io_type;


#ifdef HAVE_IMLIB2
static int (*img_imlib2_progress_cb)(
	void *,		/* Client data */
	int, int,	/* Progress current & end values */
	int, int,	/* Width & height */
	int, int,	/* Bytes per line & bytes per pixel */
	u_int8_t *	/* RGBA data */
);
static void *img_imlib2_progress_data;
static int ImgImlib2ProgressCB(
	Imlib_Image im, char percent,
	int update_x, int update_y,
	int update_w, int update_h
);
#endif

#ifdef HAVE_IMLIB
u_int8_t *ConvertImlibDataToRGBA(
	const u_int8_t *rgb, const u_int8_t *alpha,
	int width, int height,
	int rgb_bpl, int alpha_bpl,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
#endif


const char *ImgLoadGetError(void);
const char *ImgWriteGetError(void);

int ImgLoadFileRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t ***rgba_list_rtn, unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,	/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn, 
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
);
int ImgWriteImlibFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	float quality,		/* 0.0 to 1.0 */
	int save_as_color,	/* True means to save as color */
	u_int8_t def_alpha_value,
	int imlib_fallback,	/* 0 = none or 1 = ImageMajick & NetPBM */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)


#ifdef HAVE_IMLIB2
/*
 *	Imlib2 Load Progress Callback.
 */
static int ImgImlib2ProgressCB(
	Imlib_Image im, char percent,
	int update_x, int update_y,
	int update_w, int update_h
)
{
	if(img_imlib2_progress_cb != NULL)
	    return(img_imlib2_progress_cb(
		img_imlib2_progress_data,
		percent,	/* [0,100], so report 0% to 50% */
		200,		/* Limit to 50% */
		0, 0,
		0, 0,
		NULL
	    ));
	else
	    return(0);
}
#endif


#ifdef HAVE_IMLIB
/*
 *	Converts the given Imlib RGB and Alpha image data to RGBA.
 */
u_int8_t *ConvertImlibDataToRGBA(
	const u_int8_t *rgb, const u_int8_t *alpha,
	int width, int height,
	int rgb_bpl, int alpha_bpl,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
) 
{
	int rgba_bpp = 4, rgba_bpl;
	u_int8_t *rgba;

	if((rgb == NULL) || (width <= 0) || (height <= 0) || (*user_aborted))
	    return(NULL);

	/* Calculate bytes per line for all buffers */
	if(rgb_bpl <= 0)
	    rgb_bpl = width * 3;
	if(alpha_bpl <= 0)
	    alpha_bpl = width * 1;
	rgba_bpl = width * rgba_bpp;

	/* Allocate RGBA data */
	rgba = (u_int8_t *)malloc(rgba_bpl * height);
	if(rgba == NULL)
	    return(NULL);

	/* Alpha channel available? */
	if(alpha != NULL)
	{
	    u_int8_t *rgba_line, *rgba_line_end, *rgba_ptr, *rgba_end;
	    const u_int8_t *rgb_line, *rgb_ptr, *alpha_line, *alpha_ptr;

	    /* Iterate through each line */
	    for(rgba_line       = rgba,
		rgba_line_end   = rgba_line + (rgba_bpl * height),
		rgb_line        = rgb,
		alpha_line      = alpha;
		rgba_line < rgba_line_end;
		rgba_line       += rgba_bpl,
		rgb_line        += rgb_bpl,
		alpha_line      += alpha_bpl
	    )
	    {
		rgba_ptr        = rgba_line;
		rgba_end        = rgba_ptr + (width * 4);
		rgb_ptr         = rgb_line;
		alpha_ptr       = alpha_line;

		/* Call progress callback */
		if(progress_cb != NULL)
		{
		    const int i = (int)(rgba_line - rgba);
		    if((i % 5) == 0)
		    {
			if(!progress_cb(
			    client_data,
			    i, (int)(rgba_line_end - rgba),
			    width, height,
			    rgba_bpl, rgba_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }
		}

		/* Iterate through current line */
		while(rgba_ptr < rgba_end)
		{
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *alpha_ptr++;
		}
	    }
	}
	else
	{
	    u_int8_t *rgba_line, *rgba_line_end, *rgba_ptr, *rgba_end;
	    const u_int8_t *rgb_line, *rgb_ptr;

	    /* Iterate through each line */
	    for(rgba_line       = rgba,
		rgba_line_end   = rgba_line + (rgba_bpl * height),
		rgb_line        = rgb;
		rgba_line < rgba_line_end;
		rgba_line       += rgba_bpl,
		rgb_line        += rgb_bpl
	    )
	    {
		rgba_ptr        = rgba_line;
		rgba_end        = rgba_ptr + (width * 4);
		rgb_ptr         = rgb_line;

		/* Call progress callback */
		if(progress_cb != NULL)
		{
		    const int i = (int)(rgba_line - rgba);
		    if((i % 5) == 0)
		    {
		        if(!progress_cb(
			    client_data,
			    i, (int)(rgba_line_end - rgba),
			    width, height,
			    rgba_bpl, rgba_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }
		}

		/* Iterate through current line */
		while(rgba_ptr < rgba_end)
		{
		    *rgba_ptr++ = rgb_ptr[0];
		    *rgba_ptr++ = rgb_ptr[1];
		    *rgba_ptr++ = rgb_ptr[2];
#ifdef USE_IMLIB_TRANSPIXEL_FIX
		    *rgba_ptr++ = (
			(rgb_ptr[0] == 0xff) &&
			(rgb_ptr[1] == 0x00) &&
			(rgb_ptr[2] == 0xff)
		    ) ?
			0x00 : 0xff;
#else
		    *rgba_ptr++ = 0xff;
#endif
		    rgb_ptr += 3;
		}
	    }
	}

	return(rgba);
}
#endif	/* HAVE_IMLIB */


/*
 *	Returns the pointer to a error message string describing
 *	the last error encountered for a prior call to ImgLoad*().
 *
 *	Can return NULL if there was no error.
 */
const char *ImgLoadGetError(void)
{
	return(imgio_last_load_error);
}
 
/*
 *	Returns the pointer to a error message string describing
 *	the last error encountered for a prior call to ImgWrite*().
 *
 *	Can return NULL if there was no error.
 */
const char *ImgWriteGetError(void)
{
	return(imgio_last_write_error);
}


/*
 *	Loads the image from file.
 *
 *	All return values must be valid!
 *
 *	If progress_cb is not NULL then it will be called during loading,
 *	if it returns 0 then loading will be aborted.
 *
 *	Calling function must delete the returned rgba_rtn and
 *	comments_rtn pointers (the ones that are not returned NULL).
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value or unable to determine file format
 *	-3	Systems error
 *	-4	User abort
 */
int ImgLoadFileRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t ***rgba_list_rtn, unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn, 
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int status;
	int user_aborted = 0;
	imgio_io_type io_type = IO_TYPE_ANY;
	const char *ext;
	int		width = 0, height = 0,
			bpp = 0, bpl = 0,
			x = 0, y = 0,
			base_width = 0, base_height = 0;
	u_int8_t *rgba = NULL;
	char		*creator = NULL,
			*title = NULL,
			*author = NULL,
			*comments = NULL;

#define APPEND_FRAME_RGBA(_rgba_)	{	\
 if((_rgba_) != NULL) {				\
  const int frame_num = *nframes_rtn;		\
  *nframes_rtn = frame_num + 1;			\
  *rgba_list_rtn = (u_int8_t **)realloc(	\
   *rgba_list_rtn,				\
   (*nframes_rtn) * sizeof(u_int8_t *)		\
  );						\
  if(*rgba_list_rtn != NULL) {			\
   (*rgba_list_rtn)[frame_num] = (_rgba_);	\
  } else {					\
   free(_rgba_);				\
   *nframes_rtn = 0;				\
}}}

	/* Reset globals */
	imgio_last_load_error = NULL;

	/* All return values must be valid */
	if((width_rtn == NULL) || (height_rtn == NULL) ||
	   (bpl_rtn == NULL) || (bpp_rtn == NULL) ||
	   (nframes_rtn == NULL) || (rgba_list_rtn == NULL) ||
	   (delay_list_rtn == NULL)
	)
	{
	    imgio_last_load_error =
		"Addresses for image return values not available";
 	    return(-2);
	}

	/* Reset returns */
	*width_rtn = width;
	*height_rtn = height;
	*bpp_rtn = bpp;
	*bpl_rtn = bpl;
	*rgba_list_rtn = NULL;
	*delay_list_rtn = NULL;
	*nframes_rtn = 0;
	if(x_rtn != NULL)
	    *x_rtn = x;
	if(y_rtn != NULL)
	    *y_rtn = y;
	if(base_width_rtn != NULL)
	    *base_width_rtn = base_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = base_height;
	if(creator_rtn != NULL)
	    *creator_rtn = creator;
	if(title_rtn != NULL)
	    *title_rtn = title;
	if(author_rtn != NULL)
	    *author_rtn = author;
	if(comments_rtn != NULL)
	    *comments_rtn = comments;

	if(STRISEMPTY(filename))
	{
	    imgio_last_load_error = "The file name was not specified";
	    return(-2);
	}

	/* Get the file name's extension */
	ext = strrchr(filename, '.');
	if(ext == NULL)
	    ext = "";

	/* Determine the image IO type from the extension */
	if(io_type == IO_TYPE_ANY)
	{
	    if(STRISEMPTY(ext))
	    {
		/* No extension found on the file's name, so use
		 * Imlib to attempt to open it
		 */
		io_type = IO_TYPE_IMLIB;
	    }
	    else
	    {
		/* Determine image IO type from extension */
		if(!strcasecmp(ext, FTYPE_EXT_GIF))
		    io_type =
#if defined(HAVE_LIBGIF)
			IO_TYPE_GIF
#else
			IO_TYPE_IMLIB
#endif
		    ;
		else if(!strcasecmp(ext, FTYPE_EXT_JPG) ||
			!strcasecmp(ext, FTYPE_EXT_JPEG)
		)
		    io_type =
#if defined(HAVE_LIBJPEG)
			IO_TYPE_JPEG
#else
			IO_TYPE_IMLIB
#endif
		    ;
		else if(!strcasecmp(ext, FTYPE_EXT_PNG) ||
			!strcasecmp(ext, FTYPE_EXT_MNG)
		)
		    io_type =
#if defined(HAVE_LIBPNG)
			IO_TYPE_PNG
#else
			IO_TYPE_IMLIB
#endif
		    ;
		else if(!strcasecmp(ext, FTYPE_EXT_PS))
		    io_type = IO_TYPE_PS;
		else if(!strcasecmp(ext, FTYPE_EXT_TGA))
		    io_type = IO_TYPE_TGA;
		else if(!strcasecmp(ext, FTYPE_EXT_XPM))
		    io_type =
#if defined(HAVE_LIBXPM)
			IO_TYPE_XPM
#else
			IO_TYPE_IMLIB
#endif
		    ;
		else
		    io_type = IO_TYPE_IMLIB;
	    }
	}


	/* Reset status to -1 and load by IO type */
	status = -1;

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		NULL
	    ))
		user_aborted = 1;
	}


	/* Begin loading the image by the selected image IO type */
	switch(io_type)
	{
	  case IO_TYPE_ANY:
	  case IO_TYPE_IMLIB:	/* Imlib or Imlib2 */
#if defined(HAVE_IMLIB)
	    if(imlib_handle != NULL)
	    {
		/* Load image */
		char *dpath = STRDUP(filename);
		ImlibImage *imlib_image = Imlib_load_image(
		    imlib_handle, dpath
		);
		free(dpath);

		/* Image loaded successfully? */
		if(imlib_image != NULL)
		{
		    u_int8_t *rgba;

		    status = 0;		/* Mark that load was successful */

		    /* Need to realize changes */
		    Imlib_changed_image(imlib_handle, imlib_image);

		    /* Update image information */
		    width = imlib_image->rgb_width;
		    height = imlib_image->rgb_height;
		    bpp = 4;
		    bpl = width * bpp;

		    /* Convert Imlib data to RGBA data */
		    rgba = ConvertImlibDataToRGBA(
			(const u_int8_t *)imlib_image->rgb_data,
			(const u_int8_t *)imlib_image->alpha_data,
			width, height,
			-1, -1,
			client_data, progress_cb,
			&user_aborted
		    );

		    /* Unref Imlib image, it is no longer needed */
		    Imlib_destroy_image(imlib_handle, imlib_image);

		    APPEND_FRAME_RGBA(rgba);
		}	/* Image loaded successfully? */
		else
		{
		    imgio_last_load_error =
			"Imlib was unable to open the image";
		}
	    }
	    else
	    {
		imgio_last_load_error = "Imlib was not initialized";
	    }
#elif defined(HAVE_IMLIB2)
	    if(1)
	    {
		Imlib_Image *imlib_image;

		/* Set load progress callback */
		img_imlib2_progress_cb = progress_cb;
		img_imlib2_progress_data = client_data;
		imlib_context_set_progress_function(ImgImlib2ProgressCB);

		/* Load Imlib image
		 *
		 * This places the image in cache if it was not already
		 * loaded or just increases the refcount for an image
		 * already loaded in the cache
		 */
		imlib_image = imlib_load_image(filename);
		if(imlib_image != NULL)
		{
		    const u_int8_t *src_data_rgba;
		    u_int8_t *rgba = (u_int8_t *)malloc(bpl * height);

		    status = 0;		/* Mark that load was successful */

		    /* Set loaded image into Imlib2's context */
		    imlib_context_set_image(imlib_image);

		    /* Get size of image */
		    width = imlib_image_get_width();
		    height = imlib_image_get_height();
		    bpp = 4;
		    bpl = width * bpp;

		    /* Get pointer to Imlib image's RGBA data */
		    src_data_rgba = (const u_int8_t *)imlib_image_get_data_for_reading_only();

		    /* Source & target values valid? */
		    if((src_data_rgba != NULL) && (rgba != NULL) &&
		       (width > 0) && (height > 0)
		    )
		    {
			int tar_len = bpl * height;
			u_int8_t	*tar = rgba,
					*tar_end = tar + tar_len;
			const u_int8_t	*src = src_data_rgba;

			/* Copy/convert source image data to the target
			 * image data
			 */
			while(tar < tar_end)
			{
			    /* Convert source BGRA pixel to target RGBA
			     * pixel
			     */
			    *tar++ = src[2];
			    *tar++ = src[1];
			    *tar++ = src[0];
			    *tar++ = src[3];
			    src += bpp;

			    /* Report progress */
			    if(progress_cb != NULL)
			    {
				const int i = (int)((tar - rgba) + tar_len);
				if((i % 5) == 0)
				{
				    if(!progress_cb(
				        client_data,
					i, 2 * tar_len,
					width, height,
					bpl, bpp,
					rgba
				    ))
				    {
					user_aborted = 1;
					break;
				    }
				}
			    }
			}
		    }

		    /* Always delete Imlib2 images after rendering, this
		     * is efficient because the image* is cached and
		     * managed by Imlib2
		     *
		     * This will reduce the refcount of the image,
		     * when it is no longer needed then Imlib2 will
		     * actually delete it
		     */
		    imlib_free_image();

		    APPEND_FRAME_RGBA(rgba);
		}
	    }
#endif
	    break;

	  case IO_TYPE_GIF:
#if defined(HAVE_LIBGIF)
	    status = ImgGIFReadRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		rgba_list_rtn, delay_list_rtn, nframes_rtn,
		bg_color,	/* 4 bytes RGBA (will be modified) */
		&x, &y, &base_width, &base_height,
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted
	    );
#endif
	    break;

	  case IO_TYPE_JPEG:
#if defined(HAVE_LIBJPEG)
	    status = ImgJPEGReadRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		&rgba,
		bg_color,	/* 4 bytes RGBA (will be modified) */
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted 
	    );
	    APPEND_FRAME_RGBA(rgba);
#endif
	    break;

	  case IO_TYPE_PNG:
#if defined(HAVE_LIBPNG)
	    status = ImgPNGReadRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		&rgba,
		bg_color,	/* 4 byte RGBA (will be modified) */
		&x, &y, &base_width, &base_height,
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted
	    );
	    APPEND_FRAME_RGBA(rgba);
#endif
	    break;

	  case IO_TYPE_PS:
	    status = PSReadFileRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		&rgba,
		bg_color,	/* 4 byte RGBA (will be modified) */
		&x, &y, &base_width, &base_height,
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted
	    );
	    APPEND_FRAME_RGBA(rgba);
	    imgio_last_load_error = PSLastError();
	    switch(status)
	    {
	      case PSSuccess:
		status = 0;
		break;
	      case PSError:
		status = -1;
		break;
	      case PSBadValue:
		status = -2;
		break;
	      case PSErrorSystem:
		status = -3;
		break;
	      case PSAbort:
		status = -4;
		break;
	      default:
		status = -1;
		break;
	    }
	    break;

	  case IO_TYPE_TGA:
	    status = ImgTGAReadRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		&rgba,
		bg_color,	/* 4 bytes RGBA (will be modified) */
		&x, &y, &base_width, &base_height,
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted
	    );
	    APPEND_FRAME_RGBA(rgba);
	    break;

	  case IO_TYPE_XPM:
#if defined(HAVE_LIBXPM)
	   status = ImgXPMReadRGBA(
		filename,
		&width, &height, &bpp, &bpl,
		&rgba,
		bg_color,	/* 4 bytes RGBA (will be modified) */
		&x, &y, &base_width, &base_height,
		&creator, &title, &author, &comments,
		def_alpha_value,
		client_data,
		progress_cb,
		&user_aborted
	    );
	    APPEND_FRAME_RGBA(rgba);
#endif
	    break;
	}


	/* Update returns */
	*width_rtn = width;
	*height_rtn = height;
	*bpp_rtn = bpp;
	*bpl_rtn = bpl;

	if(x_rtn != NULL)
	    *x_rtn = x;
	if(y_rtn != NULL)
	    *y_rtn = y;
	if(base_width_rtn != NULL)
	    *base_width_rtn = base_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = base_height;

	if(creator_rtn != NULL)
	    *creator_rtn = creator;
	else
	    free(creator);
	if(title_rtn != NULL)
	    *title_rtn = title;
	else
	    free(title);
	if(author_rtn != NULL)
	    *author_rtn = author;
	else
	    free(author);
	if(comments_rtn != NULL)
	    *comments_rtn = comments;
	else
	    free(comments);

	/* Report final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
	    const int nframes = *nframes_rtn;
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		(nframes > 0) ?
		    (*rgba_list_rtn)[nframes - 1] : NULL
	    ))
		user_aborted = 1;
	}

	if(user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);

#undef APPEND_FRAME_RGBA
}

/*
 *	Writes the image to file using Imlib.
 *
 *	If progress_cb is not NULL then it will be called during loading,
 *	if it returns 0 then loading will be aborted.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Unable to determine file format
 *	-3	Systems error
 *	-4	User abort
 */
int ImgWriteImlibFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	const char *comments,
	float quality,		/* 0.0 to 1.0 */
	int save_as_color,	/* True means to save as color */
	u_int8_t def_alpha_value,
	int imlib_fallback,	/* 0 = none or 1 = ImageMajick & NetPBM */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int status;
	int user_aborted = 0;
	imgio_io_type io_type = IO_TYPE_ANY;
	const char *ext;


	/* Reset global last write error message pointer */
	imgio_last_write_error = NULL;

	if((rgba == NULL) || (width < 1) || (height < 1) ||
	   (bpp != 4)
	)
	{
	    imgio_last_write_error = "Invalid image description";
	    return(-2);
	}
	if(STRISEMPTY(filename))
	{
	    imgio_last_write_error = "The file name was not specified";
	    return(-2);
	}

	/* Automatically calculate bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;

	/* Get the file name's extension */
	ext = strrchr(filename, '.');
	if(ext == NULL)
	    ext = "";

	/* Must have extension to determine format */
	if(STRISEMPTY(ext))
	{
	    imgio_last_write_error = "Unable to determine format due to missing extension";
	    return(-2);
	}

	/* Explicitly set io type to use Imlib or Imlib2 */
	io_type = IO_TYPE_IMLIB;


	/* Reset status to -1 and load by IO type */
	status = -1;

	/* Call progress callback for the first time as needed */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, 1,		/* Progress starting */
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	/* Use Imlib or Imlib2? */
	if((io_type == IO_TYPE_IMLIB) && !user_aborted)
	{
#if defined(HAVE_IMLIB)
	    if(imlib_handle != NULL)
	    {
		int x, y, bc, min_rgb_bpp;
		ImlibImage *imlib_image = NULL;
		ImlibSaveInfo *si;

		int tbpl, tbpp;
		u_int8_t *tar_ptr, *tar_data_rgb, *tar_data_alpha;

		int sbpl, sbpp;
		const u_int8_t *src_ptr, *src_data;


		/* Get source image data */
		sbpp = bpp;
		sbpl = bpl;
		src_data = (const u_int8_t *)rgba;

		/* Allocate target rgb and alpha buffers */
		tbpp = 3;
		tbpl = width * tbpp;
		tar_data_rgb = (u_int8_t *)malloc(tbpl * height);
		tar_data_alpha = (u_int8_t *)malloc(width * height);
		if((tar_data_rgb != NULL) && (tar_data_alpha != NULL))
		{
		    /* Calculate smaller of rgb bytes per pixel */
		    min_rgb_bpp = MIN(sbpp, tbpp);

		    /* Copy source image to our target buffers */
		    for(y = 0; y < height; y++)
		    {
			if(progress_cb != NULL)
			{
			    if(!progress_cb(
				client_data,
				y, height,
				width, height,
				bpl, bpp,
				rgba
			    ))
			    {
				user_aborted = 1;
				break;
			    }
			}

			for(x = 0; x < width; x++)
			{
			    src_ptr = &src_data[
				(y * sbpl) + (x * sbpp)
			    ];

			    /* Copy rgb */
			    tar_ptr = &tar_data_rgb[
				(y * tbpl) + (x * tbpp)
			    ];
			    for(bc = 0; bc < min_rgb_bpp; bc++)
				*tar_ptr++ = *src_ptr++;

			    /* Copy alpha */
			    tar_ptr = &tar_data_alpha[
				(y * width) + (x * 1)
			    ];
			    /* More bytes left on source pixel pointer? */
			    if(bc < sbpp)
			    {
				*tar_ptr = *src_ptr++;
				bc++;
			    }
			    else
			    {
				*tar_ptr = def_alpha_value;
			    }
			}
		    }	/* Copy source image to our target buffers */


		    /* Create Imlib image from loaded data only if
		     * user did not abort
		     */
		    if(!user_aborted)
			imlib_image = Imlib_create_image_from_data(
			    imlib_handle,
			    (unsigned char *)tar_data_rgb,
			    (unsigned char *)tar_data_alpha,
			    width, height
			);
		    if(imlib_image != NULL)
		    {
			/* Need to realize changes */
			Imlib_changed_image(imlib_handle, imlib_image);
		    }
		}
		/* Delete target buffers used to create the Imlib
		 * image
		 */
		free(tar_data_rgb);
		tar_data_rgb = NULL;
		free(tar_data_alpha);
		tar_data_alpha = NULL;


		/* Allocate and set up save info */
		si = (ImlibSaveInfo *)calloc(1, sizeof(ImlibSaveInfo));
		if(si != NULL)
		{
		    /* Quality for lossy formats (ie jpg), valid values
		     * from 0 = 0% to 256 = 100%.
		     */
		    si->quality = CLIP(quality, 0.0f, 1.0f) * 256;

		    /* Scaling for paper relative formats, valid values
		     * from 0 = 0% to 1024 = 100%.
		     */
		    si->scaling = 1024;

		    /* Bottom-left relative paper justification, valid
		     * values from 0 to 1024.
		     */
		    si->xjustification = 0;
		    si->yjustification = 1024;

		    si->page_size = PAGE_SIZE_LETTER;

		    /* Save as greyscale (value of 0) or color (value of 1) */
		    si->color = ((save_as_color) ? 1 : 0);
		}

		/* Set Imlib fallback:
		 *	0 = No fallback
		 *	1 = Use ImageMagick and NetPBM.
		 */
		Imlib_set_fallback(imlib_handle, imlib_fallback);

		/* Was the Imlib image created successfully? */
		if(imlib_image != NULL)
		{
		    char *dpath = STRDUP(filename);
		    /* Save Imlib image */
		    status = Imlib_save_image(
			imlib_handle, imlib_image, dpath, si
		    );
		    free(dpath);

		    /* Save successful? */
		    if(status == 1)
		    {
			/* Update status to our interpritation of success */
			status = 0;
		    }
		    else
		    {
			/* Update status to our interpritation of error */
			status = -1;
		    }

		    /* Destroy Imlib image, it is no longer needed in
		     * the cache.
		     */
		    Imlib_kill_image(imlib_handle, imlib_image);
		    imlib_image = NULL;
		}
		/* At this point imlib_image has been deleted */

		/* Delete save info and any allocated members */
		free(si);
		si = NULL;
	    }
#elif defined(HAVE_IMLIB2)






#endif
	}

	/* If not user aborted, then call progress callback one last time */
	if((progress_cb != NULL) && !user_aborted)
	{
	    if(!progress_cb(
		client_data,
		1, 1,		/* Progress complete */
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}


	if(user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}
