#ifdef HAVE_LIBXPM
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <X11/xpm.h>
#include "../include/string.h"
#include "rgba_to_cidx.h"
#include "imgio.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0
#endif


/*
 *	RGBA Color:
 */
typedef struct _rgba_color_struct		rgba_color_struct;
struct _rgba_color_struct {
	u_int8_t	r, g, b, a;
};
#define RGBA_COLOR(p)	((rgba_color_struct *)(p))

/*
 *	Symbolic RGBA Color:
 */
typedef struct _sym_rgba_color_struct	sym_rgba_color_struct;
struct _sym_rgba_color_struct {
	char		*name;
	u_int8_t	r, g, b, a;
};
#define SYM_RGBA_COLOR(p)	((sym_rgba_color_struct *)(p))

/* Last error message pointer */
extern const char *imgio_last_load_error;
extern const char *imgio_last_write_error;


/*
 *	RGBAToCIdx Callback Data:
 */
typedef struct {
	int		(*progress_cb)(
		void *,		/* Data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	);
	void		*data;
	u_int8_t	*rgba;
	int		width, height,
			bpp, bpl;
	int		*user_aborted;
} RGBAToCIdxData;


/* Color Utilities */
static int ImgXPMGetSymbolicColor(
	ColormapColor *c, const char *name
);
static int ImgXPMColormapColorExists(
	rgba_color_struct **colormap, unsigned int ncolors,
	const u_int8_t *rgba		/* 4 bytes RGBA */
);
static unsigned int ImgXPMColormapMatchColorFromRGBA(
	rgba_color_struct **colormap, unsigned int ncolors,
	const u_int8_t *rgba,		/* 4 bytes RGBA */
	int closeness			/* In bytes, 0 for exact */
);
static rgba_color_struct *ImgXPMColormapColorAppend(
	rgba_color_struct ***colormap, unsigned int *ncolors,
	const u_int8_t *rgba
);


/* XPM Library Version */
void ImgXPMVersion(int *major, int *minor, int *release);


/* XPM Reading */
static char *ImgXPMReadCID(const char *filename);
int ImgXPMReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);


/* XPM Writing */
static int ImgXpmRGBAToCIdxCB(int i, int m, void *data);
static int ImgXPMWriteFileXPMImage(
	const char *filename, const char *c_id,
	XpmImage *img, XpmInfo *info
);
static int ImgXPMWriteFileRGBAToColor(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
static int ImgXPMWriteFileRGBAToGreyscale(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */ 
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
static int ImgXPMWriteFileRGBAToBW(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments, 
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(       
		void *,		/* Client data */
		int, int,	/* Progress current & end values */ 
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
int ImgXPMWriteFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int format,		/* 0 = Black & White
				 * 1 = Greyscale    
				 * 2 = Color */ 
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)( 
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
);


/*
 *	XPM Pixel ASCII Characters:
 *
 *	These are ASCII character values that represent a range of
 *	ASCII characters that may be used as characters to represent
 *	a pixel value in an XPM file.
 *
 *	The range here must not contain the '"' or '\' characters since
 *	these characters will be parsed by C interpreters when the
 *	XPM file is used in C source code.
 */
#define XPM_ASCII_CHAR_MIN		35	/* '#' */
#define XPM_ASCII_CHAR_MAX		90	/* 'Z' */
#define XPM_ASCII_CHAR_RANGE		(XPM_ASCII_CHAR_MAX - XPM_ASCII_CHAR_MIN + 1)


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)

/*
 *	Symbolic Colors List:
 */
#define IMG_XPM_SYM_RGBA_COLOR_LIST	{		\
{	"Black",	0x00, 0x00, 0x00, 0xff },	\
{	"White",	0xff, 0xff, 0xff, 0xff },	\
{	"None",		0x00, 0x00, 0x00, 0x00 },	\
{	"Transparent",	0x00, 0x00, 0x00, 0x00 },	\
{	"Background",	0x00, 0x00, 0x00, 0x00 },	\
{	"LightGrey",	0xc0, 0xc0, 0xc0, 0xff },	\
{	"Grey",		0x80, 0x80, 0x80, 0xff },	\
{	"DarkGrey",	0x40, 0x40, 0x40, 0xff },	\
{	"Red",		0xff, 0x00, 0x00, 0xff },	\
{	"Green",	0x00, 0xff, 0x00, 0xff },	\
{	"Blue",		0x00, 0x00, 0xff, 0xff }	\
}
static sym_rgba_color_struct sym_rgba_color_list[] = IMG_XPM_SYM_RGBA_COLOR_LIST;


/*
 *	Sets the color to the symbolic color specified by name.
 *
 *	Returns 1 if set or 0 if not.
 */
static int ImgXPMGetSymbolicColor(
	ColormapColor *c, const char *name
)
{
	int i;
	const sym_rgba_color_struct *sym_c;
	const int m = sizeof(sym_rgba_color_list) / sizeof(sym_rgba_color_struct);

	for(i = 0; i < m; i++)
	{
	    sym_c = &sym_rgba_color_list[i];
	    if(!strcasecmp(name, sym_c->name))
	    {
		c->r = sym_c->r;
		c->g = sym_c->g;
		c->b = sym_c->b;
		return(1);
	    }
	}

	return(0);
}

/*
 *	Checks if the RGBA color exists in the RGBA colormap.
 */
static int ImgXPMColormapColorExists(
	rgba_color_struct **colormap, unsigned int ncolors,
	const u_int8_t *rgba		/* 4 bytes RGBA */
)
{
	return((ImgXPMColormapMatchColorFromRGBA(
	    colormap, ncolors, rgba, 0) == (unsigned int)-1) ? 0 : 1
	);
}

/*
 *	Returns a color index of a color on the RGBA colormap that
 *	matches the specified RGBA color with the specified closeness.
 *
 *	Inputs assumed valid.
 *
 *	Can return (unsigned int)-1 on failed match.
 */
static unsigned int ImgXPMColormapMatchColorFromRGBA(
	rgba_color_struct **colormap, unsigned int ncolors,
	const u_int8_t *rgba,		/* 4 bytes RGBA */
	int closeness			/* In bytes, 0 for exact */
)
{
	unsigned int i;
	const rgba_color_struct *c;

	if(closeness == 0)
	{
	    for(i = 0; i < ncolors; i++)
	    {
		c = colormap[i];
		if((c->r == rgba[0]) &&
		   (c->g == rgba[1]) &&
		   (c->b == rgba[2]) &&
		   (c->a == rgba[3])
		)
		    return(i);
	    }
	}
	else
	{
	    for(i = 0; i < ncolors; i++)
	    {
		c = colormap[i];
		if((((int)c->r + closeness) >= rgba[0]) &&
		   (((int)c->r - closeness) <= rgba[0]) &&
		   (((int)c->g + closeness) >= rgba[1]) &&
		   (((int)c->g - closeness) <= rgba[1]) &&
		   (((int)c->b + closeness) >= rgba[2]) &&
		   (((int)c->b - closeness) <= rgba[2]) &&
		   (((int)c->a + closeness) >= rgba[3]) &&
		   (((int)c->a - closeness) <= rgba[3])
		)
		    return(i);
	    }

	}
	return((unsigned int)-1);
}

/*
 *	Appends a new color to the RGBA colormap.
 */
static rgba_color_struct *ImgXPMColormapColorAppend(
	rgba_color_struct ***colormap, unsigned int *ncolors,
	const u_int8_t *rgba
)
{
	rgba_color_struct *c;
	unsigned int i = MAX(*ncolors, 0);
	*ncolors = i + 1;
	*colormap = (rgba_color_struct **)realloc(
	    *colormap,
	    (*ncolors) * sizeof(rgba_color_struct *)
	);
	if(*colormap == NULL)
	{
	    *ncolors = 0;
	    return(NULL);
	}

	(*colormap)[i] = c = RGBA_COLOR(
	    malloc(sizeof(rgba_color_struct))
	);
	if(c == NULL)
	{
	    *ncolors = (*ncolors) - 1;
	    return(NULL);
	}

	c->r = rgba[0];
	c->g = rgba[1];
	c->b = rgba[2];
	c->a = rgba[3];

#if (DEBUG_LEVEL >= 3)
printf(
 "ImgXPMColormapColorAppend(): Appended color %i = 0x%.2x%.2x%.2x%.2x\n",
 i, c->r, c->g, c->b, c->a
);
#endif

	return(c);
}


/*
 *	Gets the C ID from the XPM file.
 */
static char *ImgXPMReadCID(const char *filename)
{
	const int buf_len = 4 * 80;	/* First 4 lines */
	char *buf, *s, *c_id = NULL;
	int bytes_read;
	FILE *fp = fopen(filename, "rb");
	if(fp == NULL)
	    return(NULL);

	buf = (char *)malloc((buf_len + 1) * sizeof(char));
	if(buf == NULL)
	{
	    fclose(fp);
	    return(NULL);
	}

	/* Read a few characters at the start of the file, hoping to
	 * find the C ID
	 */
	bytes_read = fread(buf, sizeof(char), buf_len, fp);

	fclose(fp);

	if((bytes_read > 0) && (bytes_read <= buf_len))
	{
	    buf[bytes_read] = '\0';

	    /* Seek to type declaration of C ID (if any) */
	    s = strstr(buf, "char");
	    if(s != NULL)
		s = strchr(s, '*');
	    if(s != NULL)
	    {
		/* Found declaration of C ID, now see s to start of the
		 * C ID
		 */
		char *s_end;

		while(*s == '*')
		    s++;
		while(ISSPACE(*s))
		    s++;

		/* Seek s_end to end of C ID */
		s_end = s;
		while((*s_end != '\0') && (*s_end != '[') &&
		      (*s_end != '=') && !ISSPACE(*s_end)
		)
		    s_end++;

		/* Copy C ID to return string */
		if(s_end > s)
		{
		    int c_id_len = s_end - s;
		    c_id = (char *)malloc((c_id_len + 1) * sizeof(char));
		    if(c_id != NULL)
		    {
			memcpy(c_id, s, c_id_len);
			c_id[c_id_len] = '\0';
		    }
		}
	    }
	}

	free(buf);

	return(c_id);
}


/*
 *	Gets the XPM library's version.
 */
void ImgXPMVersion(int *major, int *minor, int *release)
{
	if(major != NULL)
	    *major = XpmFormat;
	if(minor != NULL)
	    *minor = XpmVersion;
	if(release != NULL)
	    *release = XpmRevision;
}


/*
 *	Reads the XPM file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value (invalid format/not a xpm file)
 *	-3	Systems error
 *	-4	User abort
 */
int ImgXPMReadRGBA(       
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	const int bpp = 4;
	int width, height, bpl, ncolors = 0, trans_color_num = -1;
	unsigned long info_flags;
	ColormapColor *cmap = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	u_int8_t *rgba = NULL;

#define FREE_ALL	{	\
 if(img != NULL) {		\
  XpmFreeXpmImage(img);		\
  free(img);			\
  img = NULL;			\
 }				\
 if(info != NULL) {		\
  XpmFreeXpmInfo(info);		\
  free(info);			\
  info = NULL;			\
 }				\
				\
 free(rgba);			\
 rgba = NULL;			\
				\
 free(cmap);			\
 cmap = NULL;			\
 ncolors = 0;			\
 trans_color_num = -1;		\
}

	/* Reset last load error */
	imgio_last_load_error = NULL;

	/* Skip if user aborted */
	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}

	/* Allocate the XPM image and XPM info */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));	

	/* Read the XPM file and get the XpmImage and XpmInfo */
	switch(XpmReadFileToXpmImage((char *)filename, img, info))
	{
	  case XpmColorError:
	    FREE_ALL
	    imgio_last_load_error = "XPM color error"; 
	    return(-1);
	    break;
	  case XpmSuccess:
	    break;
	  case XpmOpenFailed:
	    FREE_ALL 
	    imgio_last_load_error = "Unable to open the XPM file for reading"; 
	    return(-1);
	    break;
	  case XpmFileInvalid:
	    FREE_ALL
	    imgio_last_load_error = "Invalid XPM file";
	    return(-2);
	    break;
	  case XpmNoMemory:
	    FREE_ALL
	    imgio_last_load_error = "Memory allocation error"; 
	    return(-3);
	    break;
	  case XpmColorFailed:
	    FREE_ALL
	    imgio_last_load_error = "XPM color failed";
	    return(-1);
	    break;
	  default:
	    FREE_ALL
	    imgio_last_load_error = "Error reading XPM file";
	    return(-1);
	    break;
	}

	/* Get XPM values */
	info_flags = info->valuemask;
	width = img->width;
	height = img->height;
	bpl = width * bpp;

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgXPMReadRGBA(): size: %ix%i  hotspot: %i %i  colors: %i  chars/pixel: %i\n",
 width, height,
 (info_flags & XpmHotspot) ? (int)info->x_hotspot : 0,
 (info_flags & XpmHotspot) ? (int)info->y_hotspot : 0,
 img->ncolors, img->cpp
);
if(info_flags & XpmComments) {
 if(info->hints_cmt != NULL)
  printf(
   "ImgXPMReadRGBA(): Hints Comment: \"%s\"\n",
   info->hints_cmt
  );
 if(info->colors_cmt != NULL)
  printf(
   "ImgXPMReadRGBA(): Colors Comment: \"%s\"\n",
   info->colors_cmt
  );
 if(info->pixels_cmt != NULL)
  printf(
   "ImgXPMReadRGBA(): Image Comment: \"%s\"\n",
   info->pixels_cmt
  );
}
#endif

	if((width <= 0) || (height <= 0))
	{
	    FREE_ALL
	    imgio_last_load_error = "Invalid image size";
	    return(-2);
	}

	if(img->data == NULL)
	{
	    FREE_ALL
	    imgio_last_load_error = "No image data";
	    return(-2);
	}

	/* Create the colormap from the XPM colormap */
	ncolors = img->ncolors;
	if(ncolors > 0)
	{
	    const int	po = 0,
			pm = 2 * ncolors;
	    int i;
	    const char *s;
	    ColormapColor *c;
	    const XpmColor *xpm_color = img->colorTable, *xpm_c;

	    cmap = (ColormapColor *)realloc(
		cmap,
		ncolors * sizeof(ColormapColor)
	    );
	    if(cmap == NULL)
	    {
		ncolors = 0;
		FREE_ALL  
		imgio_last_load_error = "Memory allocation error";
		return(-3);
	    }

	    /* Get each color from the XPM colormap */
	    for(i = 0; i < ncolors; i++)
	    {
		xpm_c = &xpm_color[i];
		c = &cmap[i];

		/* Report progress */
		if((progress_cb != NULL) && ((i % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + i, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}

#define IS_COLOR_STRING_TRANSPARENT(_s_)	\
 (!strcasecmp((_s_), "None") || !strcasecmp((_s_), "Transparent"))

		/* Symbolic color? */
		s = xpm_c->symbolic;
		if(s != NULL)
		{
		    if(IS_COLOR_STRING_TRANSPARENT(s))
		    {
			if(trans_color_num < 0)
			    trans_color_num = i;
		    }
		    if(!ImgXPMGetSymbolicColor(c, s))
		    {
			c->r = 0xff;
			c->g = 0xff;
			c->b = 0xff;
		    }
		    continue;
		}

		/* Get color string and parse it */
		s = xpm_c->c_color;
		if(s != NULL)
		{
		    /* Symbolic color? */
		    if(*s != '#')
		    {
			if(IS_COLOR_STRING_TRANSPARENT(s))
			{
			    if(trans_color_num < 0)
				trans_color_num = i;
			}
			if(!ImgXPMGetSymbolicColor(c, s))
			{
			    c->r = 0xff;
			    c->g = 0xff;
			    c->b = 0xff;
			}
			continue;
		    }

		    /* Seek past blanks and the '#' prefix */
		    while((*s == ' ') || (*s == '\t') || (*s == '#'))
			s++;
		    if(STRLEN(s) >= 6)
		    {
#define HEX_CTOI(_c_)	(u_int8_t)(((_c_) >= 'a') ?	\
 ((_c_) - 'a' + 10) : ((_c_) - '0')			\
)
			c->r = (HEX_CTOI(tolower(s[0])) * 16) +
			    HEX_CTOI(tolower(s[1]));
			c->g = (HEX_CTOI(tolower(s[2])) * 16) +
			    HEX_CTOI(tolower(s[3]));
			c->b = (HEX_CTOI(tolower(s[4])) * 16) +
			    HEX_CTOI(tolower(s[5]));
#undef HEX_CTOI
		    }
		    continue;
		}
#undef IS_COLOR_STRING_TRANSPARENT
	    }
	}
	if(*user_aborted)
	{
	    FREE_ALL
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}

	/* Allocate the RGBA image data */
	rgba = (u_int8_t *)realloc(rgba, bpl * height);
	if(rgba == NULL)
	{
	    FREE_ALL
	    imgio_last_load_error = "Memory allocation error";
	    return(-3);
	}

	/* Copy/convert the XPM color index image data to the RGBA
	 * image data
	 */
	if(!(*user_aborted))
	{
	    const int	po = height,
			pm = 2 * height;
	    int y;
	    unsigned int ci;
	    const unsigned int *cidx_ptr, *cidx_end;
	    u_int8_t *rgba_ptr;
	    const ColormapColor *c;

	    /* Iterate through the XPM color index image */
	    for(y = 0; y < height; y++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((y % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + y, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}

		cidx_ptr = img->data + (y * width);
		cidx_end = cidx_ptr + width;
		rgba_ptr = rgba + (y * bpl);
		while(cidx_ptr < cidx_end)
		{
		    /* Get the colormap color index for this pixel */
		    ci = *cidx_ptr++;
		    c = (ci < ncolors) ? &cmap[ci] : NULL;

		    /* Transparent? */
		    if((int)ci == trans_color_num)
		    {
			*rgba_ptr++ = c->r;
			*rgba_ptr++ = c->g;
			*rgba_ptr++ = c->b;
			*rgba_ptr++ = 0x00;
		    }
		    else if(c != NULL)
		    {
			*rgba_ptr++ = c->r;
			*rgba_ptr++ = c->g;
			*rgba_ptr++ = c->b;
			*rgba_ptr++ = def_alpha_value;
		    }
		    else
		    {
			rgba_ptr += bpp;
		    }
		}
	    }
	}
	if(*user_aborted)
	{
	    FREE_ALL
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}

	/* Handle extensions */
	if(info_flags & XpmExtensions)
	{
	    unsigned int i;
	    XpmExtension *ext;

	    for(i = 0; i < info->nextensions; i++)
	    {
		ext = &info->extensions[i];
#if (DEBUG_LEVEL >= 1)
printf(
 "ImgXPMReadRGBA(): Extension \"%s\" encountered.\n",
 ext->name
);
#endif

	    }
	}


	/* Update returns */
	if(width_rtn != NULL)
	    *width_rtn = width;
	if(height_rtn != NULL)
	    *height_rtn = height;
	if(bpp_rtn != NULL)
	    *bpp_rtn = bpp;
	if(bpl_rtn != NULL)
	    *bpl_rtn = bpl;
	if(rgba_rtn != NULL)
	{
	    free(*rgba_rtn);
	    *rgba_rtn = rgba;
	    rgba = NULL;
	}
	if(x_rtn != NULL)
	    *x_rtn = (info_flags & XpmHotspot) ? (int)info->x_hotspot : 0;
	if(y_rtn != NULL)
	    *y_rtn = (info_flags & XpmHotspot) ? (int)info->y_hotspot : 0;
	if(base_width_rtn != NULL)
	    *base_width_rtn = width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = height;

	/* Get C ID (if any) and return it as the title */
	if(title_rtn != NULL)
	{
	    free(*title_rtn);
	    *title_rtn = ImgXPMReadCID(filename);
	}

	/* Report final progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		*user_aborted = 1;
	}

	FREE_ALL

	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(0);

#undef FREE_ALL
}


/*
 *	RGBAToCIdx progress callback.
 */
static int ImgXpmRGBAToCIdxCB(int i, int m, void *data)
{
	RGBAToCIdxData *d = (RGBAToCIdxData *)data;
	if(d == NULL)
	    return(1);

	if((d->progress_cb != NULL) && ((i % 5) == 0))
	{
	    if(!d->progress_cb(
		d->data,
		i / 3, m,
		d->width, d->height,
		d->bpl, d->bpp,
		d->rgba
	    ))
		*d->user_aborted = 1;
	}

	return(*d->user_aborted);
}
 
/*
 *	Writes the XpmImage and XpmInfo to file.
 */
static int ImgXPMWriteFileXPMImage(
	const char *filename, const char *c_id,
	XpmImage *img, XpmInfo *info
)
{
	char *xpm_buf, *s;
	FILE *fp;

	/* Generate XPM file data from the XpmImage and XpmInfo */
	switch(XpmCreateBufferFromXpmImage(&xpm_buf, img, info))
	{
	  case XpmColorError:
	    imgio_last_write_error = "XPM color error";
	    return(-1);
	    break;
	  case XpmSuccess:
	    break;
	  case XpmOpenFailed:
	    imgio_last_write_error = "Unable to open the XPM file for writing";
	    return(-1);
	    break;
	  case XpmFileInvalid:
	    imgio_last_write_error = "Invalid XPM file";
	    return(-2);
	    break;
	  case XpmNoMemory:
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	    break;
	  case XpmColorFailed:
	    imgio_last_write_error = "XPM color failed";
	    return(-1);
	    break;
	  default:
	    imgio_last_write_error = "Error writing XPM file";
	    return(-1);
	    break;
	}
	if(xpm_buf == NULL)
	{
	    imgio_last_write_error =
"Unable to generate the XPM file data";
	    return(-1);
	}

	/* Seek to C ID */
	s = strstr(xpm_buf, "char");
	if(s != NULL)
	    s = strchr(s, '*');
	if(s != NULL)
	{
	    char *s_end;

	    while(*s == '*')
		s++;
	    while(ISSPACE(*s))
		s++;

	    /* Seek s_end to end of C ID */
	    s_end = s;
	    while((*s_end != '\0') && (*s_end != '[') &&
	          (*s_end != '=') && !ISSPACE(*s_end)
	    )
		s_end++;
	    if(s_end != NULL)
	    {
		/* Delete existing C ID in the XPM buffer and insert
		 * our C ID
		 */
		xpm_buf = strdelchrs(
		    xpm_buf, s - xpm_buf, s_end - s
		);
		xpm_buf = strinsstr(
		    xpm_buf, s - xpm_buf, c_id
		);
	    }
	}
	if(xpm_buf == NULL)
	{
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Open the XPM file for writing */
	fp = fopen(filename, "wb");
	if(fp == NULL)
	{
	    XpmFree(xpm_buf);
	    imgio_last_write_error = "Unable to open the file for writing";
	    return(-1);
	}

	/* Write the XPM buffer to the XPM file */
	if(fwrite(xpm_buf, sizeof(char), STRLEN(xpm_buf), fp) <= 0)
	{
	    XpmFree(xpm_buf);
	    fclose(fp);
	    imgio_last_write_error =
		"Unable to write the XPM buffer to the file";
	    return(-1);
	}

	/* Close the XPM file and delete the buffer */
	fclose(fp);
	XpmFree(xpm_buf);

	return(0);
}

/*
 *	Writes the specified image data to an XPM file in color.
 */
static int ImgXPMWriteFileRGBAToColor(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	int trans_color_num = -1;
	int ncolors = 0;
	u_int8_t *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;
	RGBAToCIdxData *d;

#define FREE_ALL	{		\
 if(img != NULL) {			\
  XpmFreeXpmImage(img);			\
  free(img);				\
 }					\
 if(info != NULL) {			\
  XpmFreeXpmInfo(info);			\
  free(info);				\
 }					\
 free(cmap);				\
 free(cidx);				\
}
	/* If specifying no limit on colors, then we still need to
	 * limit it to 255 since RGBAToCIdx() can only reduce to 256
	 */
	if(max_colors <= 0)
	    max_colors = 255;

	/* Limit the maximum number of colors to 255, so we have room
	 * to add the transparent color which would make it 256
	 * colors afterwards
	 */
	if(max_colors > 255)
	    max_colors = 255;

	/* Allocate the RGBAToCIdx() callback data */
	d = (RGBAToCIdxData *)calloc(1, sizeof(RGBAToCIdxData));
	d->progress_cb = progress_cb;
	d->data = client_data;
	d->width = width;
	d->height = height;
	d->bpp = bpp;
	d->bpl = bpl;
	d->rgba = rgba;
	d->user_aborted = user_aborted;

	/* Generate the color index image data and the corresponding
	 * colormap
	 */
	RGBAToCIdx(
	    rgba, width, height, bpl,
	    max_colors,
	    &cidx,
	    &cmap,
	    &ncolors,
	    ImgXpmRGBAToCIdxCB,
	    d
	);
	free(d);		/* Delete the callback data */

	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Apply transparency to the color index image data */
	if(1)
	{
	    int i = ncolors;

	    /* Insert the transparent color at the beginning of the
	     * colormap
	     */
	    ncolors = i + 1;
	    cmap = (ColormapColor *)realloc(
		cmap,
		ncolors * sizeof(ColormapColor)
	    );
	    if(cmap != NULL)
	    {
		u_int8_t *rgba_ptr, *cidx_ptr, *cidx_end;

		trans_color_num = 0;

		/* Shift colors on the colormap */
		for(i = ncolors - 1; i > 0; i--)
		    memcpy(
			&cmap[i],
			&cmap[i - 1],
			sizeof(ColormapColor)
		    );

		/* Increment each color value on the color index image
		 * data and set all transparent colors
		 */
		for(y = 0; y < height; y++)
		{
		    rgba_ptr = rgba + (y * bpl);
		    cidx_ptr = cidx + (y * width);
		    cidx_end = cidx_ptr + width;
		    while(cidx_ptr < cidx_end)
		    {
			if(rgba_ptr[3] < threshold)
			{
			    *cidx_ptr++ = (u_int8_t)trans_color_num;
			}
			else
			{
			    *cidx_ptr = (*cidx_ptr) + 1;
			    cidx_ptr++;
			}
			rgba_ptr += bpp;
		    }
		}
	    }
	    else
	    {
		ncolors = 0;
		FREE_ALL
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}
	info->valuemask = XpmComments | XpmHotspot;
	info->hints_cmt = NULL;
	info->colors_cmt = NULL;
	info->pixels_cmt = STRDUP(comments);
	info->x_hotspot = x;
	info->y_hotspot = y;

	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error"; 
	    return(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
	    unsigned int m = XPM_ASCII_CHAR_RANGE;
	    for(img->cpp = 1;
		m < ncolors;
		img->cpp++, m *= XPM_ASCII_CHAR_RANGE
	    );
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
	    const int	po = ncolors,
			pm = 3 * ncolors;
	    const unsigned int cpp = img->cpp;
	    int i;
	    char *s, *c_string = (char *)malloc((cpp + 1) * sizeof(char));
	    const ColormapColor *src_c;
	    XpmColor *tar_c;

	    /* Reset the color string */
	    memset(c_string, XPM_ASCII_CHAR_MIN, cpp);
	    c_string[cpp] = '\0';

	    /* Allocate the XpmColors on the XPM colormap */
	    img->ncolors = ncolors;
	    img->colorTable = (XpmColor *)calloc(
		img->ncolors, sizeof(XpmColor)
	    );
	    if(img->colorTable == NULL)
	    {
		free(c_string);
		img->ncolors = 0;
		FREE_ALL 
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }

	    /* Iterate through the colormap and create the XpmColors */
	    for(i = 0; i < ncolors; i++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((i % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + i, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    { 
			*user_aborted = 1;
			break;
		    }
		}

		/* Set this colormap color to the XPM colormap */
		src_c = &cmap[i];
		tar_c = &img->colorTable[i];

		/* Set the XpmColor's ASCII characters string that
		 * represent this color
		 */
		tar_c->string = STRDUP(c_string);

		/* Set the XpmColor's color value string
		 *
		 * Check for symbolic colors
		 */
		if(i == trans_color_num)
		{
		    tar_c->c_color = STRDUP("None");
		}
		else
		{
		    tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
		    sprintf(
			s, "#%.2X%.2X%.2X",
			src_c->r, src_c->g, src_c->b
		    );
		}

		/* Increment characters in the color string */
		for(s = c_string; *s != '\0'; s++)
		{
		    if(*s < XPM_ASCII_CHAR_MAX)
		    {
			*s = (*s) + 1;
			break;
		    }
		    else
		    {
			*s = XPM_ASCII_CHAR_MIN;
		    }
		}
	    }

	    free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Allocate the XPM image data */
	img->data = (unsigned int *)malloc(
	    width * height * sizeof(unsigned int)
	);
	if(img->data == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Copy/convert the color index image data to the XPM color
	 * index image data
	 */
	if((img->data != NULL) && (cidx != NULL))
	{
	    const int	po = 2 * height,
			pm = 3 * height;
	    int y;
	    unsigned int *tar_ptr;
	    const u_int8_t *src_ptr, *src_end;

	    for(y = 0; y < height; y++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((y % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + y, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    { 
			*user_aborted = 1;
			break;
		    }
		}

		/* Copy/convert this line */
		src_ptr = cidx + (y * width);
		src_end = src_ptr + width;
		tar_ptr = img->data + (y * width);
		while(src_ptr < src_end)
		    *tar_ptr++ = (unsigned int)*src_ptr++;
	    }
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteFileXPMImage(filename, c_id, img, info))
	{
	    FREE_ALL
	    return(-1);
	}
	else
	{
	    FREE_ALL
	    return(0);
	}
#undef FREE_ALL
}

/*
 *	Writes the specified image data to an XPM file in greyscale.
 */
static int ImgXPMWriteFileRGBAToGreyscale(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(       
		void *,		/* Client data */
		int, int,	/* Progress current & end values */ 
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	int ncolors = 0, trans_color_num = -1;
	unsigned int *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;

#define FREE_ALL	{		\
 if(img != NULL) {			\
  XpmFreeXpmImage(img);			\
  free(img);				\
 }					\
 if(info != NULL) {			\
  XpmFreeXpmInfo(info);			\
  free(info);				\
 }					\
 free(cidx);				\
 free(cmap);				\
}
 
	/* Allocate the color index image data */
	cidx = (unsigned int *)realloc(
	    cidx, width * height * sizeof(unsigned int)
	);
	if(cidx == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Create the colormap and the color index image data from the
	 * RGBA image data
	 *
	 * Note that we are ignoring max_colors, as many as 257 colors
	 * may be generated (256 + transparent color).
	 */
	if(cidx != NULL)
	{
	    const int	po = 0,
			pm = 2 * height;
	    int i, y;
	    u_int32_t gamma;
	    unsigned int *cidx_ptr;
	    const u_int8_t *rgba_ptr, *rgba_end;
	    ColormapColor tc, *c;

	    /* Append the transparent color */
	    i = ncolors;
	    ncolors = i + 1;
	    cmap = (ColormapColor *)realloc(
		cmap, ncolors * sizeof(ColormapColor)
	    );
	    if(cmap == NULL)
	    {
		ncolors = 0;
		FREE_ALL
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }
	    c = &cmap[i];
	    c->r = 0x00;
	    c->g = 0x00;
	    c->b = 0x00;
	    trans_color_num = i;

	    /* Iterate through each line */
	    for(y = 0; y < height; y++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((y % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + y, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}

		/* Iterate through this line */
		cidx_ptr = cidx + (y * width);
		rgba_ptr = rgba + (y * bpl);
		rgba_end = rgba_ptr + (width * bpp);
		while(rgba_ptr < rgba_end)
		{
		    /* Skip transparent pixels */
		    if(rgba_ptr[3] < threshold)
		    {
			*cidx_ptr++ = (unsigned int)trans_color_num;
			rgba_ptr += bpp;
			continue;
		    }

		    /* Get the grey value of the current RGBA pixel */
		    gamma = ((u_int32_t)rgba_ptr[0] +
			 (u_int32_t)rgba_ptr[1] +
			 (u_int32_t)rgba_ptr[2]
			) / 3;
		    tc.r = tc.g = tc.b = (u_int8_t)gamma;

		    /* Check if the color exists in the colormap */
		    for(i = 0; i < ncolors; i++)
		    {
			/* Skip transparent color */
			if(i == trans_color_num)
			    continue;

			c = &cmap[i];
			if((c->r == tc.r) &&
			   (c->g == tc.g) &&
			   (c->b == tc.b)
			)
			    break;
		    }
		    if(i >= ncolors)
		    {
			/* Need to append this color */
			const int n = ncolors;
			ncolors = n + 1;
			cmap = (ColormapColor *)realloc(
			    cmap,
			    ncolors * sizeof(ColormapColor)
			);
			if(cmap != NULL)
			{
			    c = &cmap[n];
			    c->r = tc.r;
			    c->g = tc.g;
			    c->b = tc.b;
			}
			else
			{
			    ncolors = 0;
			    FREE_ALL
			    imgio_last_write_error =
				"Memory allocation error";
			    return(-3);
			}
			*cidx_ptr++ = (unsigned int)n;
			rgba_ptr += bpp;
		    }
		    else
		    {
			*cidx_ptr++ = (unsigned int)i;
			rgba_ptr += bpp;
		    }
		}
	    }
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}
	info->valuemask = XpmComments | XpmHotspot;
	info->hints_cmt = NULL;
	info->colors_cmt = NULL; 
	info->pixels_cmt = STRDUP(comments);
	info->x_hotspot = x;
	info->y_hotspot = y;

	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
	    unsigned int m = XPM_ASCII_CHAR_RANGE;
	    for(img->cpp = 1;
		m < ncolors;
		img->cpp++, m *= XPM_ASCII_CHAR_RANGE
	    );
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
	    const int	po = ncolors,
			pm = 2 * ncolors;
	    const unsigned int cpp = img->cpp;
	    int i;
	    char *s, *c_string = (char *)malloc((cpp + 1) * sizeof(char));
	    const ColormapColor *src_c;
	    XpmColor *tar_c;

	    /* Reset color string */
	    memset(c_string, XPM_ASCII_CHAR_MIN, cpp);
	    c_string[cpp] = '\0';

	    /* Allocate XpmColors on the XPM colormap */
	    img->ncolors = ncolors;
	    img->colorTable = (XpmColor *)calloc(
		img->ncolors, sizeof(XpmColor)
	    );
	    if(img->colorTable == NULL)
	    {
		free(c_string);
		img->ncolors = 0;
		FREE_ALL
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }

	    /* Iterate through the colormap and create the XpmColors */
	    for(i = 0; i < ncolors; i++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((i % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + i, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    { 
			*user_aborted = 1;
			break;
		    }
		}

		/* Set this colormap color to the XPM colormap */
		src_c = &cmap[i];
		tar_c = &img->colorTable[i];

		/* Set the XpmColor's ASCII characters string that
		 * represent this color
		 */
		tar_c->string = STRDUP(c_string);

		/* Set the XpmColor's color value string
		 *
		 * Check for symbolic colors
		 */
		if(i == trans_color_num)
		{
		    tar_c->c_color = STRDUP("None");
		}
		else
		{
		    tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
		    sprintf(
			s, "#%.2X%.2X%.2X",
			src_c->r, src_c->g, src_c->b
		    );
		}

		/* Increment characters in the color string */
		for(s = c_string; *s != '\0'; s++)
		{
		    if(*s < XPM_ASCII_CHAR_MAX)
		    {
			*s = (*s) + 1;
			break;
		    }
		    else
		    {
			*s = XPM_ASCII_CHAR_MIN;
		    }
		}
	    }

	    free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Set the color index image data as the XPM image data */
	img->data = cidx;
	cidx = NULL;

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteFileXPMImage(filename, c_id, img, info))
	{
	    FREE_ALL
	    return(-1);
	}
	else
	{
	    FREE_ALL
	    return(0);
	}
#undef FREE_ALL
}

/*
 *	Writes the specified image data to an XPM file in black & white.
 */
static int ImgXPMWriteFileRGBAToBW(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	int ncolors = 0, trans_color_num = -1;
	unsigned int *cidx = NULL;
	XpmInfo *info = NULL;
	XpmImage *img = NULL;
	ColormapColor *cmap = NULL;

#define FREE_ALL	{		\
 if(img != NULL) {			\
  XpmFreeXpmImage(img);			\
  free(img);				\
 }					\
 if(info != NULL) {			\
  XpmFreeXpmInfo(info);			\
  free(info);				\
 }					\
 free(cidx);				\
 free(cmap);				\
}
 
	/* Allocate the color index image data */
	cidx = (unsigned int *)realloc(
	    cidx, width * height * sizeof(unsigned int)
	);
	if(cidx == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Create the colormap and the color index image data from the
	 * RGBA image data
	 *
	 * Note that we are ignoring max_colors, as many as 257 colors
	 * may be generated (256 + transparent color).
	 */
	if(cidx != NULL)
	{
	    const int	po = 0,
			pm = 2 * height;
	    int i, y;
	    u_int32_t gamma;
	    unsigned int *cidx_ptr;
	    const u_int8_t *rgba_ptr, *rgba_end;
	    ColormapColor tc, *c;

	    /* Append the transparent color */
	    i = ncolors;
	    ncolors = i + 1;
	    cmap = (ColormapColor *)realloc(
		cmap, ncolors * sizeof(ColormapColor)
	    );
	    if(cmap == NULL)
	    {
		ncolors = 0;
		FREE_ALL
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }
	    c = &cmap[i];
	    c->r = 0x00;
	    c->g = 0x00;
	    c->b = 0x00;
	    trans_color_num = i;

	    /* Iterate through each line */
	    for(y = 0; y < height; y++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((y % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + y, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}

		/* Iterate through this line */
		cidx_ptr = cidx + (y * width);
		rgba_ptr = rgba + (y * bpl);
		rgba_end = rgba_ptr + (width * bpp);
		while(rgba_ptr < rgba_end)
		{
		    /* Skip transparent pixels */
		    if(rgba_ptr[3] < threshold)
		    {
			*cidx_ptr++ = (unsigned int)trans_color_num;
			rgba_ptr += bpp;
			continue;
		    }

		    /* Get the grey value of the current RGBA pixel */
		    gamma = ((u_int32_t)rgba_ptr[0] +
			 (u_int32_t)rgba_ptr[1] +
			 (u_int32_t)rgba_ptr[2]
			) / 3;
		    tc.r = tc.g = tc.b = ((u_int8_t)gamma < 0x80) ?
			0x00 : 0xff;

		    /* Check if the color exists in the colormap */
		    for(i = 0; i < ncolors; i++)
		    {
			/* Skip transparent color */
			if(i == trans_color_num)
			    continue;

			c = &cmap[i];
			if((c->r == tc.r) &&
			   (c->g == tc.g) &&
			   (c->b == tc.b)
			)
			    break;
		    }
		    if(i >= ncolors)
		    {
			/* Need to append this color */
			const int n = ncolors;
			ncolors = n + 1;
			cmap = (ColormapColor *)realloc(
			    cmap,
			    ncolors * sizeof(ColormapColor)
			);
			if(cmap != NULL)
			{
			    c = &cmap[n];
			    c->r = tc.r;
			    c->g = tc.g;
			    c->b = tc.b;
			}
			else
			{
			    ncolors = 0;
			    FREE_ALL
			    imgio_last_write_error =
				"Memory allocation error";
			    return(-3);
			}
			*cidx_ptr++ = (unsigned int)n;
			rgba_ptr += bpp;
		    }
		    else
		    {
			*cidx_ptr++ = (unsigned int)i;
			rgba_ptr += bpp;
		    }
		}
	    }
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Create the XpmInfo */
	info = (XpmInfo *)calloc(1, sizeof(XpmInfo));
	if(info == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}
	info->valuemask = XpmComments | XpmHotspot;
	info->hints_cmt = NULL;
	info->colors_cmt = NULL; 
	info->pixels_cmt = STRDUP(comments);
	info->x_hotspot = x;
	info->y_hotspot = y;

	/* Create the XpmImage */
	img = (XpmImage *)calloc(1, sizeof(XpmImage));
	if(img == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}
	img->width = width;
	img->height = height;
	if(ncolors > 0)
	{
	    unsigned int m = XPM_ASCII_CHAR_RANGE;
	    for(img->cpp = 1;
		m < ncolors;
		img->cpp++, m *= XPM_ASCII_CHAR_RANGE
	    );
	}

	/* Create the XPM colormap on the XpmImage */
	if((img->cpp > 0) && (ncolors > 0))
	{
	    const int	po = ncolors,
			pm = 2 * ncolors;
	    const unsigned int cpp = img->cpp;
	    int i;
	    char *s, *c_string = (char *)malloc((cpp + 1) * sizeof(char));
	    const ColormapColor *src_c;
	    XpmColor *tar_c;

	    /* Reset color string */
	    memset(c_string, XPM_ASCII_CHAR_MIN, cpp);
	    c_string[cpp] = '\0';

	    /* Allocate XpmColors on the XPM colormap */
	    img->ncolors = ncolors;
	    img->colorTable = (XpmColor *)calloc(
		img->ncolors, sizeof(XpmColor)
	    );
	    if(img->colorTable == NULL)
	    {
		free(c_string);
		img->ncolors = 0;
		FREE_ALL
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }

	    /* Iterate through the colormap and create the XpmColors */
	    for(i = 0; i < ncolors; i++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && ((i % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			po + i, pm,
			width, height,
			bpl, bpp,
			rgba
		    ))
		    { 
			*user_aborted = 1;
			break;
		    }
		}

		/* Set this colormap color to the XPM colormap */
		src_c = &cmap[i];
		tar_c = &img->colorTable[i];

		/* Set the XpmColor's ASCII characters string that
		 * represent this color
		 */
		tar_c->string = STRDUP(c_string);

		/* Set the XpmColor's color value string
		 *
		 * Check for symbolic colors
		 */
		if(i == trans_color_num)
		{
		    tar_c->c_color = STRDUP("None");
		}
		else
		{
		    tar_c->c_color = s = (char *)malloc((7 + 1) * sizeof(char));
		    sprintf(
			s, "#%.2X%.2X%.2X",
			src_c->r, src_c->g, src_c->b
		    );
		}

		/* Increment characters in the color string */
		for(s = c_string; *s != '\0'; s++)
		{
		    if(*s < XPM_ASCII_CHAR_MAX)
		    {
			*s = (*s) + 1;
			break;
		    }
		    else
		    {
			*s = XPM_ASCII_CHAR_MIN;
		    }
		}
	    }

	    free(c_string);
	}
	/* User aborted? */
	if(*user_aborted)
	{
	    FREE_ALL
	    return(-4);
	}

	/* Set the color index image data as the XPM image data */
	img->data = cidx;
	cidx = NULL;

	/* Write XpmImage and XpmInfo to the XPM file */
	if(ImgXPMWriteFileXPMImage(filename, c_id, img, info))
	{
	    FREE_ALL
	    return(-1);
	}
	else
	{
	    FREE_ALL
	    return(0);
	}
#undef FREE_ALL
}

/*
 *	Writes the specified image data to an XPM file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value
 *	-3	Systems error
 *	-4	User abort
 */
int ImgXPMWriteFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes RGBA */
	int x, int y, int base_width, int base_height,
	const char *c_id, const char *comments,
	int format,		/* 0 = Black & White
				 * 1 = Greyscale
				 * 2 = Color */
	int max_colors,		/* -1 for no color limit */
	u_int8_t threshold,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int status, user_aborted = 0;
	char *s, *new_c_id;

	/* Reset global last write error message pointer */
	imgio_last_write_error = NULL;

	if((filename == NULL) || (rgba == NULL) ||
	   (width <= 0) || (height <= 0) ||
	   (bpp != 4)
	)
	{
	    imgio_last_write_error = "Invalid value used to describe image";
	    return(-1); 
	}

	/* Automatically calculate bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgXPMWriteFileRGBA(): Writing XPM file \"%s\"...\n",
 filename
);
#endif

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	/* Generate C ID as needed */
	if(STRISEMPTY(c_id))
	{
	    const char *s = strrchr(filename, '/');
	    new_c_id = (s != NULL) ? STRDUP(s + 1) : STRDUP(filename);
	}
	else
	{
	    new_c_id = STRDUP(c_id);
	}
	/* Replace unacceptable characters in the C ID */
	for(s = new_c_id; *s != '\0'; s++)
	{
	    if(!isalnum(*s))
		*s = '_';
	}


#if (DEBUG_LEVEL >= 1)
printf(
 "ImgXPMWriteFileRGBA(): C ID=\"%s\"\n",
 new_c_id
);
#endif

	/* Save by format */
	status = -2;
	switch(format)
	{
	  case 0:
	    status = ImgXPMWriteFileRGBAToBW(
		filename,
		width, height, bpl, bpp,
		rgba,
		bg_color,
		x, y, base_width, base_height,
		new_c_id, comments,
		max_colors, threshold,
		client_data, progress_cb, &user_aborted
	    );
	    break;

	  case 1:
	    status = ImgXPMWriteFileRGBAToGreyscale(
		filename,
		width, height, bpl, bpp,
		rgba,
		bg_color,
		x, y, base_width, base_height,
		new_c_id, comments,
		max_colors, threshold,
		client_data, progress_cb, &user_aborted
	    );
	    break;

	  case 2:
	    status = ImgXPMWriteFileRGBAToColor(
		filename,
		width, height, bpl, bpp,
		rgba,
		bg_color,
		x, y, base_width, base_height,
		new_c_id, comments,
		max_colors, threshold,
		client_data, progress_cb, &user_aborted
	    );
	    break;

	  default:
	    imgio_last_write_error = "Unsupported XPM color format";
	    status = -2;
	    break;
	}

	/* Report final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	free(new_c_id);

#if (DEBUG_LEVEL >= 1)
printf(
 "ImgXPMWriteFileRGBA(): Done.\n"
);
#endif

	if(user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

#endif	/* HAVE_LIBXPM */
