/*
				  ImgView
 */

#ifndef IMGVIEW_H
#define IMGVIEW_H

#include <gtk/gtk.h>


typedef struct _imgview_struct		imgview_struct;
#define IMGVIEW(p)			((imgview_struct *)(p))
typedef struct _imgview_frame_struct    imgview_frame_struct;
#define IMGVIEW_FRAME(p)		((imgview_frame_struct *)(p))
typedef struct _imgview_image_struct	imgview_image_struct;
#define IMGVIEW_IMAGE(p)		((imgview_image_struct *)(p))
typedef struct _imgview_cropdlg_struct	imgview_cropdlg_struct;
#define IMGVIEW_CROPDLG(p)		((imgview_cropdlg_struct *)(p))


/*
 *	Flags:
 */
typedef enum {

	/* Toplevel widget is a GtkWindow, otherwise it is a
	 * GtkContainer */
	IMGVIEW_FLAG_TOPLEVEL_IS_WINDOW	= (1 << 0),

	/* Specifies if geometry, zoom, zoom rectangular, etc values
	 * should be shown */
	IMGVIEW_FLAG_VALUES_SHOWABLE	= (1 << 1),
	IMGVIEW_FLAG_SHOW_VALUES	= (1 << 2),

	/* Specifies to display the contents of the ImgView on the
	 * WM Icon? */
	IMGVIEW_FLAG_SHOW_IMAGE_ON_WM_ICON = (1 << 4),

	IMGVIEW_FLAG_TOOL_BAR_SHOWABLE	= (1 << 8),
	IMGVIEW_FLAG_TOOL_BAR_MAPPED	= (1 << 9),
	IMGVIEW_FLAG_MENU_BUTTON_SHOWABLE = (1 << 10),
	IMGVIEW_FLAG_MENU_BUTTON_MAPPED	= (1 << 11),
	IMGVIEW_FLAG_STATUS_BAR_SHOWABLE = (1 << 12),
	IMGVIEW_FLAG_STATUS_BAR_MAPPED	= (1 << 13)

} imgview_flags;


/*
 *	Drag Modes:
 */
typedef enum {
	IMGVIEW_DRAG_MODE_NONE,
	IMGVIEW_DRAG_MODE_TRANSLATE,
	IMGVIEW_DRAG_MODE_ZOOM,
	IMGVIEW_DRAG_MODE_ZOOM_RECTANGLE,
	IMGVIEW_DRAG_MODE_CROP_RECTANGLE,
	IMGVIEW_DRAG_MODE_COLOR_PROBE
} imgview_drag_mode;

/*
 *	Image Data Formats:
 */
typedef enum {
	IMGVIEW_FORMAT_GREYSCALE8	= 10,	/* 1 bpp */
	IMGVIEW_FORMAT_GREYSCALE16	= 11,	/* 2 bpp */
	IMGVIEW_FORMAT_GREYSCALE32	= 12,	/* 4 bpp */
	IMGVIEW_FORMAT_GREYSCALEA32	= 13,	/* 4 bpp */
	IMGVIEW_FORMAT_RGB		= 20,	/* 3 bpp */
	IMGVIEW_FORMAT_RGBA		= 21	/* 4 bpp */
} imgview_format;

/*
 *	Alpha Channel Flags:
 */
typedef enum {

	/* Alpha channel is defined and is not uniform */
	IMGVIEW_ALPHA_DEFINED		= (1 << 0),

	/* Use thresholding instead of blending the alpha value */
	IMGVIEW_ALPHA_THRESHOLD		= (1 << 1),

	/* Alpha values are inverted (0xff = transparent and 0x00
	 * means solid) */
	IMGVIEW_ALPHA_INVERTED		= (1 << 2)

} imgview_alpha_flags;        

/*
 *	Crop Flags:
 */
typedef enum {

	/* Crop is allowed */
	IMGVIEW_CROP_ALLOWED		= (1 << 0),

	/* Crop rectangle defined (or is being defined) by user in the
	 * middle of a drag crop rectangle */
	IMGVIEW_CROP_DEFINED		= (1 << 1),

	/* Crop rectangle defined and finalized */
	IMGVIEW_CROP_FINALIZED		= (1 << 2)

} imgview_crop_flags;


/*
 *	ImgView Image Frame:
 */
struct _imgview_frame_struct {

	guint8		*buf;		/* Image data */
	gulong		delay;		/* Show this frame for this many ms */

};

/*
 *	ImgView Image:
 */
struct _imgview_image_struct {

	gint		width,
			height,
			bpp,		/* Bytes per pixel */
			bpl;		/* Bytes per line */
	GList		*frames_list;	/* List of imgview_frame_struct * */
	gint		nframes;

};

/*
 *	ImgView Crop Dialog:
 */
struct _imgview_cropdlg_struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	imgview_struct	*imgview;

	GtkWidget	*val_label,
			*val2_label,
			*crop_btn,
			*cancel_btn;

	GdkRectangle	rect;		/* Crop Geometry */

};

/*
 *	ImgView:
 */
struct _imgview_struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gboolean	map_state;
	gint		freeze_count,
			busy_count;

	imgview_flags	flags;

	/* Quality, from 0 to 2
	 *
	 *	0	poor/fastest
	 *	1	standard
	 *	2	best/slowest
	 */
	gint		quality;

	/* Alpha channel flags */
	imgview_alpha_flags	alpha_flags;

	/* Alpha threshold (used IMGVIEW_ALPHA_THRESHOLD is set) */
	guint8		alpha_threshold;

	/* Current pointer drag mode */
	imgview_drag_mode	drag_mode;

	/* Last drag positions (in window coordinates) */
	gint		drag_last_x,
			drag_last_y;

	/* Zoom Rectangular (in window coordinates) */
	gint		drag_zoom_rectangle_start_x,
			drag_zoom_rectangle_start_y,
			drag_zoom_rectangle_cur_x,
			drag_zoom_rectangle_cur_y;

	/* Crop Rectangle (in image data coordinates) */
	imgview_crop_flags	crop_flags;
	gint		crop_rectangle_start_x,
			crop_rectangle_start_y,
			crop_rectangle_cur_x,
			crop_rectangle_cur_y;
	imgview_cropdlg_struct	*crop_dialog;

	/* Color Probe */
	gint		color_probe_x,
			color_probe_y;

	/* Graphic context for drawing selection graphics on the view */
	GdkGC		*view_selection_gc;
	/* Bitmaps for drawing selection icons on the view */
	GdkBitmap	*scissor_bm;

	GdkCursor	*busy_cur,
			*translate_cur,
			*zoom_cur,
			*zoom_rectangle_cur,
			*crop_cur,
			*color_probe_cur;

	/* Main vbox parent (this is NULL if toplevel_is_window is FALSE) */
	GtkWidget	*main_vbox;

	/* Window manager icon pixmap/mask pair */
	GdkPixmap	*wm_icon_pixmap;
	GdkBitmap	*wm_icon_mask;

	/* Tool Bar */
	GtkWidget	*tool_bar_toplevel,	/* GtkContainer */
			*menu_button,
			*info_label,		/* GtkDrawingArea */
			*zoom_in_btn,
			*zoom_out_btn,
			*zoom_tofit_btn,
			*zoom_onetoone_btn;

	/* Scrollbars */
	GtkWidget	*hscrollbar,
			*vscrollbar;

	/* Status Bar */
	GtkWidget	*status_bar_toplevel,	/* GtkContainer */
			*progress_bar;

	/* Right-click menu */
	GtkWidget	*menu,
			*zoom_in_mi,
			*zoom_out_mi,
			*zoom_tofit_mi,
			*zoom_onetoone_mi,
			*rotate_transform_submenu_mi,
			*show_submenu_mi,
			*quality_submenu_mi;

	/* Rotate/Transform submenu */
	GtkWidget	*rotate_transform_menu,
			*rotate_cw_90_mi,
			*rotate_ccw_90_mi,
			*rotate_cw_180_mi,
			*mirror_horizontal_mi,
			*mirror_vertical_mi;

	/* Show submenut */
 	GtkWidget	*view_menu,
			*view_play_mi,
			*view_pause_mi,
			*view_prev_mi,
			*view_next_mi,
			*view_tool_bar_micheck,
			*view_values_micheck,
			*view_status_bar_micheck;
 
	/* Quality submenu */
	GtkWidget	*quality_menu,
			*quality_poor_mi,
			*quality_optimal_mi,
			*quality_best_mi;

	/* View translate and zoom values
	 *
	 * Translation is from the upper left corner in window
	 * coordinates */
	GtkAdjustment	*view_x_adj,
			*view_y_adj;

	gfloat		last_view_x,
			last_view_y;

	/* Zoom, original image geometry units are to be divided by
	 * member view_zoom to get the window units
	 *
	 * Example: If member view_zoom is 2.0, then orig_x / 2.0 */
	gfloat		view_zoom,
			view_last_zoom;

	guint		view_zoom_idle,		/* Button zoom in/out idle */
			view_zoom_toid,		/* Button zoom in/out timeout */
			view_draw_idle_id;	/* View draw idle */

	/* View drawing area */
	GtkWidget	*view_da;

	/* View image, used as the view_da's image buffer */
	imgview_image_struct	*view_img;

	/* Original image, used to hold the original image data */
	imgview_image_struct	*orig_img;

	/* Current frame being displayed on the original image */
	gint			cur_frame_num;

	/* Draw next frame Timeout */
	guint			next_frame_toid;

	/* Changed callback */
	void	(*changed_cb)(
		imgview_struct *,		/* ImgView */
		imgview_image_struct *,		/* Changed Image */
		gpointer			/* Data */
	);
	gpointer	changed_data;

};

#define IMGVIEW_IS_TOPLEVEL_WINDOW(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_TOPLEVEL_IS_WINDOW)

#define IMGVIEW_VALUES_SHOWABLE(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_VALUES_SHOWABLE)
#define IMGVIEW_SHOW_VALUES(_imgview_)		\
 ((_imgview_)->flags & IMGVIEW_FLAG_SHOW_VALUES)

#define IMGVIEW_SHOW_IMAGE_ON_WM_ICON(_imgview_) \
 ((_imgview_)->flags & IMGVIEW_FLAG_SHOW_IMAGE_ON_WM_ICON)

#define IMGVIEW_TOOL_BAR_SHOWABLE(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_TOOL_BAR_SHOWABLE)
#define IMGVIEW_TOOL_BAR_MAPPED(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_TOOL_BAR_MAPPED)

#define IMGVIEW_MENU_BUTTON_SHOWABLE(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_MENU_BUTTON_SHOWABLE)
#define IMGVIEW_MENU_BUTTON_MAPPED(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_MENU_BUTTON_MAPPED)

#define IMGVIEW_STATUS_BAR_SHOWABLE(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_STATUS_BAR_SHOWABLE)
#define IMGVIEW_STATUS_BAR_MAPPED(_imgview_)	\
 ((_imgview_)->flags & IMGVIEW_FLAG_STATUS_BAR_MAPPED)


/* ImgView Frame */
extern imgview_frame_struct *ImgViewFrameNew(void);
extern imgview_frame_struct *ImgViewFrameCopy(
	imgview_image_struct *img,
	imgview_frame_struct *frame
);
extern void ImgViewFrameDelete(imgview_frame_struct *frame);

/* ImgView Image */
extern imgview_image_struct *ImgViewImageNew(
	const gint width, const gint height,
	const gint bpp, const gint bpl
);
extern imgview_image_struct *ImgViewImageCopy(
	imgview_image_struct *img
);
extern gint ImgViewImageInsertFrame(
	imgview_image_struct *img, const gint frame_num,
	guint8 *buf,			/* Transfered */
	const gulong delay
);
extern gint ImgViewImageAppendFrame(
	imgview_image_struct *img,
	guint8 *buf,			/* Transfered */
	const gulong delay
);
extern void ImgViewImageSwapFrames(
        imgview_image_struct *img,
        const gint frame_num1,
        const gint frame_num2
);
extern gint ImgViewImageCopyFrame(
	imgview_image_struct *img, const gint frame_num,
	const gint src_frame_num
);
extern gint ImgViewImageAppendFrameNew(imgview_image_struct *img);
extern void ImgViewImageDeleteFrame(
	imgview_image_struct *img, const gint frame_num
);
extern imgview_frame_struct *ImgViewImageGetFrame(
	imgview_image_struct *img, const gint frame_num
);
extern void ImgViewImageClear(
	imgview_image_struct *img, const gint frame_num,
	const guint32 v
);
extern void ImgViewImageSendRectangle(
	imgview_image_struct *image, const gint frame_num,
	GdkDrawable *d, GdkGC *gc,
	const gint quality,
	const GdkRectangle *rect
);
extern void ImgViewImageSend(
	imgview_image_struct *image, const gint frame_num,
	GdkDrawable *d, GdkGC *gc,
	const gint quality
);
extern void ImgViewImageDelete(imgview_image_struct *image);

/* ImgView utilities */
extern gint ImgViewConvertUnitViewToOrigX(imgview_struct *iv, const gint x);
extern gint ImgViewConvertUnitViewToOrigY(imgview_struct *iv, const gint y);
extern gint ImgViewConvertUnitOrigToViewX(imgview_struct *iv, const gint x);
extern gint ImgViewConvertUnitOrigToViewY(imgview_struct *iv, const gint y);
extern GtkAccelGroup *ImgViewGetAccelGroup(imgview_struct *iv);
extern gboolean ImgViewIsToplevelWindow(imgview_struct *iv);
extern GtkWidget *ImgViewGetToplevelWidget(imgview_struct *iv);
extern GtkDrawingArea *ImgViewGetViewWidget(imgview_struct *iv);
extern GtkMenu *ImgViewGetMenuWidget(imgview_struct *iv);
extern gboolean ImgViewIsLoaded(imgview_struct *iv);
extern imgview_image_struct *ImgViewGetImage(imgview_struct *iv);
extern guint8 *ImgViewGetImageData(
	imgview_struct *iv, gint frame_num,
	gint *width, gint *height, gint *bpp, gint *bpl,
	imgview_format *format
);
extern gint ImgViewGetCurrentFrame(imgview_struct *iv);
extern gint ImgViewGetTotalFrames(imgview_struct *iv);

/* ImgView front ends */
extern void ImgViewClear(imgview_struct *iv);
extern gint ImgViewSetImage(
	imgview_struct *iv,
	imgview_image_struct *img	/* Transfered */
);
extern gint ImgViewSetImageToFit(
	imgview_struct *iv,
	imgview_image_struct *img	/* Transfered */
);
extern gint ImgViewLoad(
	imgview_struct *iv,
	gint width, gint height,
	gint bytes_per_line,		/* Set 0 to auto calculate */
	imgview_format format,
	const guint8 *data		/* RGBA image data */
);
extern gint ImgViewLoadToFit(
	imgview_struct *iv,
	gint width, gint height,
	gint bytes_per_line,		/* Set 0 to auto calculate */
	imgview_format format,
	const guint8 *data		/* RGBA image data */
);

extern void ImgViewPlay(imgview_struct *iv);
extern void ImgViewPause(imgview_struct *iv);
extern void ImgViewSeek(imgview_struct *iv, const gint frame_num);
extern gboolean ImgViewIsPlaying(imgview_struct *iv);

extern imgview_struct *ImgViewNew(
	const gboolean show_toolbar,
	const gboolean show_values,
	const gboolean show_statusbar,
	const gboolean show_image_on_wm_icon,
	const gint quality,
	const gboolean toplevel_is_window,
	GtkWidget **toplevel_rtn
);
extern void ImgViewSetChangedCB(
	imgview_struct *iv,
	void (*changed_cb)(
		imgview_struct *,
		imgview_image_struct *,
		gpointer
	),
	gpointer data
);
extern void ImgViewReset(imgview_struct *iv, gboolean need_unmap);
extern void ImgViewQueueDrawView(imgview_struct *iv);
extern void ImgViewQueueSendView(imgview_struct *iv);
extern void ImgViewDraw(imgview_struct *iv);
extern void ImgViewUpdateMenus(imgview_struct *iv);

extern void ImgViewSetValuesShowable(imgview_struct *iv, const gboolean b);
extern void ImgViewSetToolBarShowable(imgview_struct *iv, const gboolean b);
extern void ImgViewShowToolBar(imgview_struct *iv, const gboolean show);
extern void ImgViewSetToolBarMenuShowable(imgview_struct *iv, const gboolean b);
extern void ImgViewShowToolBarMenu(imgview_struct *iv, const gboolean show);
extern void ImgViewSetStatusBarShowable(imgview_struct *iv, const gboolean b);
extern void ImgViewShowStatusBar(imgview_struct *iv, const gboolean show);

extern void ImgViewSetViewBG(
	imgview_struct *iv,
	const GdkColor *c		/* 5 colors */
);

extern void ImgViewZoomIn(imgview_struct *iv);
extern void ImgViewZoomOut(imgview_struct *iv);
extern void ImgViewZoomOneToOne(imgview_struct *iv);
extern void ImgViewZoomToFit(imgview_struct *iv);
extern void ImgViewZoomPrev(imgview_struct *iv);

extern void ImgViewAllowCrop(imgview_struct *iv, const gboolean allow_crop);
extern void ImgViewSetBusy(imgview_struct *iv, const gboolean busy);
extern void ImgViewProgressUpdate(
	imgview_struct *iv,
	const gfloat v,
	const gboolean allow_gtk_iteration
);
extern void ImgViewMap(imgview_struct *iv);
extern void ImgViewUnmap(imgview_struct *iv);
extern void ImgViewDelete(imgview_struct *iv);


#endif	/* IMGVOEW_H */
