#include <string.h>
#if defined(HAVE_IMLIB)
# include <Imlib.h>
#elif defined(HAVE_IMLIB2)
# include <X11/Xlib.h>
# include <Imlib2.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>

#include "imgio.h"
#include "guiutils.h"
#include "guirgbimg.h"
#include "imgview.h"
#include "ivtitle.h"
#include "config.h"

/* #undef INCLUDE_IV_TITLE_IMAGE */

#ifdef INCLUDE_IV_TITLE_IMAGE
/* Uncomment only one, each xpm image defines title_320x240_xpm */
#include "titles/blue_320x240.xpm"
/* #include "titles/rainbow_320x240.xpm" */
/* #include "titles/classic_320x240.xpm" */
#endif


#ifdef HAVE_IMLIB
extern void *imlib_handle;
#endif  /* HAVE_IMLIB */

static GdkPixmap *make_simple_title(
	GdkWindow *window, gint width, gint height
);
void iv_load_title(
	imgview_struct *iv,
	gboolean resize_on_load,
	gboolean simple_title
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a simple title on a GdkPixmap.
 */
static GdkPixmap *make_simple_title(
	GdkWindow *window, gint width, gint height
)
{
	const gint border_major = 5;
	gint x = 0, y = 0;
	GdkFont *font;
	GdkColor fg, bg;
	GdkColormap *colormap = gdk_window_get_colormap(window);
	GdkPixmap *pixmap = gdk_pixmap_new(window, width, height, -1);
	GdkDrawable *drawable = pixmap;
	GdkGC *gc = gdk_gc_new(window);
	if((colormap == NULL) || (pixmap == NULL) || (gc == NULL))
	    return(NULL);

	/* Draw the background */ 
	GDK_COLOR_SET_COEFF(&bg, 1.0f, 1.0f, 1.0f);
	GDK_COLORMAP_ALLOC_COLOR(colormap, &bg);
	gdk_gc_set_foreground(gc, &bg);
	gdk_draw_rectangle(   
	    drawable, gc, TRUE,
	    0, 0, width, height
	);
	GDK_COLORMAP_FREE_COLOR(colormap, &bg);

	/* Draw the title banner & text */
	font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-34-*-*-*-p-*-iso8859-1"
	);
	if(font == NULL)
	    font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-20-*-*-*-p-*-iso8859-1"
	    );
	if(font == NULL)
	    font = gdk_font_load("8x16");
	if(font != NULL)
	{
#ifdef PROG_NAME_FULL
	    const gfloat y_coeff = 0.30f;
	    gint tar_width, tar_height;
	    GdkTextBounds b;
	    const gchar *s = PROG_NAME_FULL;

	    GDK_COLOR_SET_COEFF(&fg, 1.0f, 1.0f, 1.0f);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &fg);
	    GDK_COLOR_SET_COEFF(&bg, 0.25f, 0.25f, 1.0f);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &bg);

	    gdk_string_bounds(font, s, &b);

	    tar_width = b.width + (4 * border_major);
	    tar_height = font->ascent + font->descent + (4 * border_major);
	    x = (width / 2) - (tar_width / 2);
	    y = (gint)(height * y_coeff) - (tar_height / 2);
	    gdk_gc_set_foreground(gc, &bg);
	    gdk_draw_rectangle(
		drawable, gc, TRUE,
		x, y, tar_width, tar_height
	    );

	    tar_width = b.width + (2 * border_major);
	    tar_height = font->ascent + font->descent + (2 * border_major);
	    x = (width / 2) - (tar_width / 2);
	    y = (gint)(height * y_coeff) - (tar_height / 2);
	    gdk_gc_set_foreground(gc, &fg);
	    gdk_draw_rectangle(
		drawable, gc, FALSE,
		x, y, tar_width - 1, tar_height - 1
	    );

	    tar_width = b.width;
	    tar_height = font->ascent + font->descent;
	    x = (width / 2) - (tar_width / 2);
	    y = (gint)(height * y_coeff) - (tar_height / 2);
	    gdk_gc_set_foreground(gc, &fg);
	    gdk_gc_set_font(gc, font);
	    gdk_draw_string(
		drawable, font, gc,
		x + b.lbearing,
		y + font->ascent,
		s
	    );

	    GDK_COLORMAP_FREE_COLOR(colormap, &fg);
	    GDK_COLORMAP_FREE_COLOR(colormap, &bg);

	    y += tar_height + (4 * border_major);
#endif
	}
	GDK_FONT_UNREF(font);

	/* Draw the version text */
	font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-20-*-*-*-p-*-iso8859-1"
	);
	if(font == NULL)
	    font = gdk_font_load("8x16");
	if(font != NULL)
	{
#ifdef PROG_VERSION
	    gint tar_width, tar_height;
	    GdkTextBounds b;
	    const gchar *s = "Version " PROG_VERSION;

	    GDK_COLOR_SET_COEFF(&fg, 0.0f, 0.0f, 0.0f);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &fg);

	    gdk_string_bounds(font, s, &b);
	    tar_width = b.width;
	    tar_height = font->ascent + font->descent;
	    x = (width / 2) - (tar_width / 2);

	    gdk_gc_set_foreground(gc, &fg);
	    gdk_gc_set_font(gc, font);
	    gdk_draw_string(
		drawable, font, gc,
		x + b.lbearing,
		y + font->ascent,
		s
	    );

	    GDK_COLORMAP_FREE_COLOR(colormap, &fg);
	    y += tar_height + border_major;
#endif
	}
	GDK_FONT_UNREF(font)

	/* Draw the home page URL text */
	font = gdk_font_load(
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-p-*-iso8859-1"
	);
	if(font == NULL)
	    font = gdk_font_load("7x14");
	if(font != NULL)
	{
#ifdef PROG_URL
	    gint tar_width, tar_height;
	    GdkTextBounds b;
	    const gchar *s = PROG_URL;

	    GDK_COLOR_SET_COEFF(&fg, 0.0f, 0.0f, 0.0f);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &fg);

	    gdk_string_bounds(font, s, &b);
	    tar_width = b.width;
	    tar_height = font->ascent + font->descent;
	    x = (width / 2) - (tar_width / 2);

	    gdk_gc_set_foreground(gc, &fg);
	    gdk_gc_set_font(gc, font);
	    gdk_draw_string(
		drawable, font, gc,
		x + b.lbearing,
		y + font->ascent,
		s
	    );

	    GDK_COLORMAP_FREE_COLOR(colormap, &fg);
#endif
	}
	GDK_FONT_UNREF(font);

	/* Draw the "(Press Button3 for menu)" text */
	font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-20-*-*-*-p-*-iso8859-1"
	);
	if(font == NULL)
	    font = gdk_font_load("8x16");
	if(font != NULL)
	{
	    gint tar_width, tar_height;
	    GdkTextBounds b;
	    const gchar *s = "(Press Button3 for menu)";

	    gdk_string_bounds(font, s, &b);
	    tar_width = b.width;
	    tar_height = font->ascent + font->descent;
	    x = (width / 2) - (tar_width / 2);
	    y = (gint)(height * 0.80f) - (tar_height / 2);

	    gdk_gc_set_foreground(gc, &fg);
	    gdk_gc_set_font(gc, font);
	    gdk_draw_string(
		drawable, font, gc,
		x + b.lbearing,
		y + font->ascent,
		s
	    );

	    GDK_COLORMAP_FREE_COLOR(colormap, &fg);
	}
	GDK_FONT_UNREF(font);

	/* Draw the copyright text */
	font = gdk_font_load(
"-adobe-helvetica-medium-r-normal-*-10-*-*-*-p-*-iso8859-1"
	);
	if(font == NULL)
	    font = gdk_font_load("6x10");
	if(font != NULL)
	{
#ifdef PROG_COPYRIGHT
	    gint tar_width, tar_height;
	    GdkTextBounds b;
	    gchar *s = STRDUP(PROG_COPYRIGHT), *s2;

	    s2 = strchr(s, '\n');
	    if(s2 != NULL)
		*s2 = '\0';
	    s2 = strchr(s, '.');
	    if(s2 != NULL)
		*s2 = '\0';

	    GDK_COLOR_SET_COEFF(&fg, 0.0f, 0.0f, 0.0f);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &fg);

	    gdk_string_bounds(font, s, &b);
	    tar_width = b.width;
	    tar_height = font->ascent + font->descent;
	    x = width - tar_width - border_major;
	    y = height - tar_height - border_major;

	    gdk_gc_set_foreground(gc, &fg);
	    gdk_gc_set_font(gc, font);
	    gdk_draw_string(
		drawable, font, gc,
		x + b.lbearing,
		y + font->ascent,
		s
	    );

	    GDK_COLORMAP_FREE_COLOR(colormap, &fg);
	    g_free(s);
#endif
	}
	GDK_FONT_UNREF(font);

	GDK_GC_UNREF(gc);

	return(pixmap);
}


/*
 *	Loads the title image to the ImgView.
 *
 *	If INCLUDE_IV_TITLE_IMAGE is defined then it means the image
 *	data is compiled in and it will be the one loaded.
 *
 *	If INCLUDE_IV_TITLE_IMAGE is not defined then a genertic
 *	title image will be generated.
 *
 *	If simple_title is TRUE then a simpler (less graphical) title
 *	image will be used.
 */
void iv_load_title(
	imgview_struct *iv,
	gboolean resize_on_load,
	gboolean simple_title
)
{
#ifdef INCLUDE_IV_TITLE_IMAGE
	GdkWindow *window;
	GtkWidget *toplevel;
#if defined(HAVE_IMLIB)
	ImlibImage *imlib_image;
#elif defined(HAVE_IMLIB2)
	Imlib_Image imlib_image;
#endif

	if(iv == NULL)
	    return;

	/* Get image viewer toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	window = (toplevel != NULL) ? toplevel->window : NULL;
	if(window == NULL)
	    return;

	/* Load simple title instead? */
	if(simple_title)
	{
	    /* Create simple title on a GdkPixmap */
	    GdkPixmap *pixmap = make_simple_title(window, 320, 240);
	    if(pixmap != NULL)
	    {
		gint width, height, rgba_bpl;
		GtkWidget *w = (GtkWidget *)ImgViewGetViewWidget(iv);
		guint8 *rgba = gdk_get_rgba_image(
		    pixmap, NULL,
		    &width, &height,
		    &rgba_bpl
		);
		imgview_image_struct *img = ImgViewImageNew(
		    width, height, 4, rgba_bpl
		);
		ImgViewImageAppendFrame(img, rgba, 0l);

		/* Resize ImgView? */
		if((w != NULL) && resize_on_load)
		{
		    /* Resize the ImgView's view and toplevel widgets */
		    gtk_widget_set_usize(w, width, height);
		    gtk_widget_queue_resize(w);
		    if(toplevel != NULL)
			gtk_widget_queue_resize(toplevel);
		}
	     
		/* Set new image to the ImgView
		 *
		 * Note that the image passed to this function should
		 * not be referenced again after this call
		 */
		ImgViewSetImage(iv, img);
	    }
	    GDK_PIXMAP_UNREF(pixmap);
	    return;
	}

#if defined(HAVE_IMLIB)
	if(imlib_handle != NULL)
	{
	    /* Load title image */
	    imlib_image = Imlib_create_image_from_xpm_data(
		imlib_handle,
		(char **)title_320x240_xpm
	    );
	    if(imlib_image != NULL)
	    {
		const guint bpp = 4;
		gint width, height, user_aborted = 0;
		GtkWidget *w = (GtkWidget *)ImgViewGetViewWidget(iv);
		imgview_image_struct *img;

		/* Need to realize changes */
		Imlib_changed_image(imlib_handle, imlib_image);
		width = imlib_image->rgb_width;
		height = imlib_image->rgb_height;

		/* Create an ImgView image from the Imlib image */
		img = ImgViewImageNew(
		    width, height, bpp, width * bpp
		);
		if(img != NULL)
		{
		    guint8 *rgba = ConvertImlibDataToRGBA(
			imlib_image->rgb_data, imlib_image->alpha_data,
			img->width, img->height,
			-1, -1,
			NULL, NULL,
			&user_aborted
		    );
		    ImgViewImageAppendFrame(img, rgba, 0l);
		}

		/* Unref Imlib image, it is no longer needed */
		Imlib_destroy_image(imlib_handle, imlib_image);

		/* Resize ImgView? */
		if((w != NULL) && resize_on_load)
		{
		    /* Resize the ImgView's view and toplevel
		     * widgets
		     */
		    gtk_widget_set_usize(w, img->width, img->height);
		    gtk_widget_queue_resize(w);
		    if(toplevel != NULL)
			gtk_widget_queue_resize(toplevel);
		}

		/* Set new image to the ImgView
		 *
		 * Note that the image passed to this function should
		 * not be referenced again after this call
		 */
		ImgViewSetImage(iv, img);
	    }
	}
#elif defined(HAVE_IMLIB2)

/* TODO */




#endif
#else	/* INCLUDE_IV_TITLE_IMAGE not defined */
	GdkPixmap *pixmap;
	GdkWindow *window;
	GtkWidget *toplevel;

	if(iv == NULL)
	    return;

	/* Get image viewer toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	window = (toplevel != NULL) ? toplevel->window : NULL;
	if(window == NULL)
	    return;

	/* Create simple title on a GdkPixmap */
	pixmap = make_simple_title(window, 320, 240);
	if(pixmap != NULL)
	{
	    gint width, height, rgba_bpl;
	    GtkWidget *w = (GtkWidget *)ImgViewGetViewWidget(iv);
	    guint8 *rgba = gdk_get_rgba_image(
		pixmap,
		NULL,			/* Get entire area */
		&width, &height,
		&rgba_bpl
	    );
	    imgview_image_struct *img = ImgViewImageNew(
		width, height, 4, rgba_bpl
	    );
	    ImgViewImageAppendFrame(img, rgba, 0l);

	    /* Resize ImgView? */
	    if((w != NULL) && resize_on_load)
	    {
		/* Resize the ImgView's view and toplevel widgets */
		gtk_widget_set_usize(w, width, height);
		gtk_widget_queue_resize(w);
		if(toplevel != NULL)
		    gtk_widget_queue_resize(toplevel);
	    }

	    /* Set new image to the ImgView
	     *
	     * Note that the image passed to this function should
	     * not be referenced again after this call
	     */
	    ImgViewSetImage(iv, img);
	}

	GDK_PIXMAP_UNREF(pixmap);

#endif	/* INCLUDE_IV_TITLE_IMAGE */
}
