#if !defined(_WIN32) && defined(HAVE_XPRINT)
#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifdef HAVE_LIBENDEAVOUR2
# include <endeavour2.h>
#endif

#include "../include/string.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "pulist.h"
#include "imgview.h"
#include "cdialog.h"
#include "print_values.h"
#include "print.h"
#include "print_dlg.h"
#include "config.h"

#include "images/icon_print2_48x48.xpm"
#include "images/icon_print2_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"

#include "images/icon_minus_16x16.xpm"
#include "images/icon_plus_16x16.xpm"
#include "images/icon_brightness_16x16.xpm"
#include "images/icon_contrast_16x16.xpm"

#include "images/icon_file_20x20.xpm"
#include "images/icon_printer_tray_20x20.xpm"


typedef struct _PrintDlg		PrintDlg;
#define PRINT_DLG(p)			((PrintDlg *)(p))

typedef enum {
	PRINT_DLG_DRAG_NONE		= 0,
	PRINT_DLG_DRAG_TRANSLATE	= (1 << 0),
	PRINT_DLG_DRAG_RESIZE		= (1 << 1)
} PrintDlgDragFlags;


/* Callbacks */
static gint PrintDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void PrintDlgAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
);
static void PrintDlgToggledCB(GtkWidget *widget, gpointer data);
static void PrintDlgXPrintServerConnectCB(
	GtkWidget *widget, gpointer data
);
static void PrintDlgPrinterChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void PrintDlgTrayChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void PrintDlgMediumSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
static void PrintDlgOrientationChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void PrintDlgResolutionChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void PrintDlgPrintCB(GtkWidget *widget, gpointer data);
static void PrintDlgCancelCB(GtkWidget *widget, gpointer data);
static void PrintDlgOutputChangedCB(GtkWidget *widget, gpointer data);
static gint PrintDlgPreviewEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

/* Utilities */
static print_visual PrintDlgGetVisual(PrintDlg *d);
static gboolean PrintDlgGetMediumSize(
	PrintDlg *d,
	gint *width_rtn, gint *height_rtn
);
static gfloat PrintDlgGetSourceImageAspect(PrintDlg *d);
static void PrintDlgOutputSetWidgets(PrintDlg *d);
static gboolean PrintDlgOutputGetOffsetW(
	PrintDlg *d, gint *x, gint *y
);
static void PrintDlgRevertImages(
	imgview_image_struct *src_img, imgview_image_struct *tar_img
);
static void PrintDlgRevertPreview(PrintDlg *d);
static void PrintDlgPreviewUpdateVisual(PrintDlg *d);
static void PrintDlgPreviewRecreate(PrintDlg *d);
static void PrintDlgPreviewResize(PrintDlg *d);
static void PrintDlgPreviewDraw(PrintDlg *d);
static void PrintDlgPreviewQueueDraw(PrintDlg *d);
static void PrintDlgConnectXPrintServer(
	PrintDlg *d,
	const gchar *xprint_server_address,
	const gboolean verbose
);
static void PrintDlgPrintersListUpdate(
	PrintDlg *d,
	const gchar *xprint_server_address,
	const gboolean verbose
);
static void PrintDlgSetPrinterValues(
	PrintDlg *d,
	const gchar *printer_name,
	const gboolean verbose
);
static void PrintDlgMediumsListUpdate(
	PrintDlg *d,
	const gchar *printer_name,
	const gboolean verbose
);

static PrintDlg *PrintDlgNew(
	const gchar *filename,
	GtkWidget *ref_toplevel
);
static void PrintDlgSetBusy(PrintDlg *d, const gboolean busy);
static void PrintDlgUpdate(PrintDlg *d);
static void PrintDlgMap(PrintDlg *d);
static void PrintDlgUnmap(PrintDlg *d);
static void PrintDlgDelete(PrintDlg *d);

/* Front ends */

#ifdef HAVE_LIBENDEAVOUR2
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
);
#else
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
);
#endif


/*
 *	Print Dialog:
 */
struct _PrintDlg {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	GtkWidget	*ref_toplevel;
	gboolean	map_state,
			got_response;
	gint		freeze_count,
			busy_count,
			main_level;

	/* Cursors */
	GdkCursor	*busy_cur,
			*translate_cur,
			*resize_cur;

	/* Source Image */
	imgview_image_struct	*src_img;
	gint			src_frame_num;

	/* Address */
	GtkWidget	*xprint_server_address_entry;

	/* Printer */
	pulistbox_struct	*printer_pulistbox;

	/* Visual */
	GtkWidget	*visual_bw_radio,
			*visual_greyscale_radio,
			*visual_color_radio;
	GtkWidget	*brightness_scale;
	GtkAdjustment	*brightness_adj;
	GtkWidget	*contrast_scale;
	GtkAdjustment	*contrast_adj;

	/* Medium */
	GtkWidget	*medium_ctree;
	GdkPixmap	*tray_icon_pm;
	GdkBitmap	*tray_icon_mask;
	GdkPixmap	*medium_icon_pm;
	GdkBitmap	*medium_icon_mask;
	GList		*medium_toplevel_nodes_list;

	/* Orientation */
	pulistbox_struct	*orientation_pulistbox;

	/* Resolution */
	pulistbox_struct	*resolution_pulistbox;

	/* Output */
	GtkWidget	*output_x_spin,
			*output_y_spin,
			*output_width_spin,
			*output_height_spin,
			*output_frame_spin,
			*output_maintain_aspect_check,
			*output_ncoppies_spin;

	/* Preview */
	GtkWidget	*preview_da;
	GdkGC		*preview_invert_gc;
	GdkPixmap	*preview_pm;

	/* Preview ImgView image
	 *
	 * This image is post processed with the modified window
	 * coordinate size, color, brightness, rotations, etc which
	 * makes it suitable and ready to be drawn on the preview_da
	 */
	imgview_image_struct	*preview_img;

	/* Original Preview ImgView image
	 *
	 * It matches the size of the preview_img but does not have
	 * brightness or contrast modifications made to it, this is so
	 * that it can be quickly copied to the preview_img with
	 * modifications to the contrast and brightness
	 */
	imgview_image_struct	*orig_preview_img;

	/* Preview popup menu */
	GtkWidget	*menu;

	/* Medium Size */
	GdkRectangle	medium_size;	/* x and y ignored */

	/* Output Image Geometry */
	GdkRectangle	output_img_geometry;
	gboolean	output_maintain_aspect;

	guint8		bg_color[4];	/* 4 bytes RGBA */

	/* Buttons */
	GtkWidget	*print_btn,
			*cancel_btn;

	/* Drag */
	PrintDlgDragFlags	drag_modifiers,
				drag_operation;
	gint		drag_last_x, drag_last_y;

	/* Rectangular drag outline in window coordinates, for drawing
	 * a rectangular rubber band on the preview widget while dragging
	 */
	gboolean	only_draw_outline;	/* Set by PrintDlgPreviewEventCB() and
						 * cleared by PrintDlgPreviewDraw()
						 */
	gint		drag_outline_x[2],
			drag_outline_y[2];
	gboolean	have_prev_outline;
	gint		prev_outline_x[2],
			prev_outline_y[2];

};


#define PRINT_DLG_TITLE			"Print"

#define PRINT_DLG_WIDTH			640
#define PRINT_DLG_HEIGHT		480

#define PRINT_DLG_DEF_MEDIUM_WIDTH	240
#define PRINT_DLG_DEF_MEDIUM_HEIGHT	320

/* Nominal size of the preview GtkDrawingArea, this size must be
 * square in shape
 */
#define PRINT_DLG_PREVIEW_WIDTH		200
#define PRINT_DLG_PREVIEW_HEIGHT	200


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0) 
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f) 
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint PrintDlgDeleteEventCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return(FALSE);

	PrintDlgCancelCB(d->cancel_btn, d);

	return(TRUE);
}

/*
 *	GtkAdjustment "value_changed" signal callback.
 */
static void PrintDlgAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
)
{
	PrintDlg *d = PRINT_DLG(data);
	if((adj == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	PrintDlgPreviewUpdateVisual(d);
	PrintDlgPreviewQueueDraw(d);

	d->freeze_count--;
}

/*
 *	GtkToggleButton "toggled" signal callback.
 */
static void PrintDlgToggledCB(GtkWidget *widget, gpointer data)
{
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	/* Only handle toggled to TRUE signals */
	if(GTK_IS_TOGGLE_BUTTON(widget) ?
	    !GTK_TOGGLE_BUTTON_GET_ACTIVE(widget) : TRUE
	)
	    return;

	d->freeze_count++;

	/* Visual Black & White */
	if(widget == d->visual_bw_radio)
	{
	    PrintDlgPreviewUpdateVisual(d);
	    PrintDlgPreviewQueueDraw(d);
	}
	/* Visual Greyscale? */
	else if(widget == d->visual_greyscale_radio)
	{
	    PrintDlgPreviewUpdateVisual(d);
	    PrintDlgPreviewQueueDraw(d);
	}
	/* Visual Color? */
	else if(widget == d->visual_color_radio)
	{
	    PrintDlgPreviewUpdateVisual(d);
	    PrintDlgPreviewQueueDraw(d);
	}

	d->freeze_count--;
}

/*
 *	X Print server connect callback.
 */
static void PrintDlgXPrintServerConnectCB(
	GtkWidget *widget, gpointer data
)
{
	PrintDlg *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	PrintDlgConnectXPrintServer(
	    d,
	    gtk_entry_get_text(
		GTK_ENTRY(d->xprint_server_address_entry)
	    ),
	    TRUE			/* Verbose */
	);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Printer popup list box "changed" signal callback.
 */
static void PrintDlgPrinterChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gint medium_width, medium_height;
	GdkRectangle *geo;
	pulist_struct *pulist;
	iv_print_xprint_printer_struct *p;
	PrintDlg *d = PRINT_DLG(data);
	if((pulistbox == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	pulist = PUListBoxGetPUList(pulistbox);
	p = IV_PRINT_XPRINT_PRINTER(
	    PUListGetItemData(pulist, i)
	);
	if(p != NULL)
	    PrintDlgSetPrinterValues(
		d,
		p->name,
		TRUE
	    );

	/* Update the medium size */
	if(PrintDlgGetMediumSize(
	    d,
	    &medium_width, &medium_height
	))
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = medium_width;
	    medium->height = medium_height;
	}
	else
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
	    medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;
	}

	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgPreviewResize(d);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Tray Popup List Box "changed" signal callback.
 */
static void PrintDlgTrayChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gint medium_width, medium_height;
	GdkRectangle *geo;
	pulist_struct *pulist;
	iv_print_xprint_printer_struct *p;
	PrintDlg *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	/* Update the medium list */
	pulist = PUListBoxGetPUList(d->printer_pulistbox);
	p = IV_PRINT_XPRINT_PRINTER(
	    PUListGetItemData(
		pulist,
		PUListGetSelectedLast(pulist)
	    )
	);
	if(p != NULL)
	    PrintDlgMediumsListUpdate(d, p->name, TRUE);

	/* Update the medium size */
	if(PrintDlgGetMediumSize(
	    d,
	    &medium_width, &medium_height
	))
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = medium_width;
	    medium->height = medium_height;
	}
	else
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
	    medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;
	}

	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgPreviewResize(d);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Medium GtkCTreet "tree_select_row" signal callback.
 */
static void PrintDlgMediumSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	gint medium_width, medium_height;
	GdkRectangle *geo;
	PrintDlg *d = PRINT_DLG(data);
	if((ctree == NULL) || (node == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	if(gtk_ctree_node_is_visible(ctree, node)
	    != GTK_VISIBILITY_FULL
	)
	    gtk_ctree_node_moveto(
		ctree,
		node, -1,
		0.5f, 0.0f              /* Row align, column align */
	    );

	/* Update the medium size */
	if(PrintDlgGetMediumSize(
	    d,
	    &medium_width, &medium_height
	))
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = medium_width;
	    medium->height = medium_height;
	}
	else
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
	    medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;

	    if(TRUE)
	    {
		iv_print_xprint_medium_struct *m = IV_PRINT_XPRINT_MEDIUM(
		    gtk_ctree_node_get_row_data(ctree, node)
		);
		const gchar *medium_name = (m != NULL) ? m->name : NULL;
		gchar *msg = g_strdup_printf(
"Unable to get the size of the medium \"%s\".",
		    medium_name
		);
		CDialogSetTransientFor(d->toplevel);
		CDialogGetResponse(
		    "Get Medium Size Failed",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);
	    }
	}

	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgPreviewResize(d);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Orientation popup list box "changed" signal callback.
 */
static void PrintDlgOrientationChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gint medium_width, medium_height;
	GdkRectangle *geo;
	PrintDlg *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	/* Update the medium size */
	if(PrintDlgGetMediumSize(
	    d,
	    &medium_width, &medium_height
	))
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = medium_width;
	    medium->height = medium_height;
	}
	else
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
	    medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;
	}

	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgPreviewResize(d);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Resolution popup list box "changed" signal callback.
 */
static void PrintDlgResolutionChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gint medium_width, medium_height;
	GdkRectangle *geo;
	PrintDlg *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;
	PrintDlgSetBusy(d, TRUE);

	/* Update the medium size */
	if(PrintDlgGetMediumSize(
	    d,
	    &medium_width, &medium_height
	))
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = medium_width;
	    medium->height = medium_height;
	}
	else
	{
	    GdkRectangle *medium = &d->medium_size;
	    medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
	    medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;
	}

	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgPreviewResize(d);

	PrintDlgSetBusy(d, FALSE);
	d->freeze_count--;
}

/*
 *	Output GtkWidget "changed" or "toggled" signal callback.
 */
static void PrintDlgOutputChangedCB(GtkWidget *widget, gpointer data)
{
	GdkRectangle *output;
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	output = &d->output_img_geometry;

	if(widget == d->output_x_spin)
	{
	    output->x = GTK_ENTRY_GET_VALUEI(widget);
	    PrintDlgPreviewQueueDraw(d);
	}
	else if(widget == d->output_y_spin)
	{
	    output->y = GTK_ENTRY_GET_VALUEI(widget);
	    PrintDlgPreviewQueueDraw(d);
	}
	else if(widget == d->output_width_spin)
	{
	    output->width = GTK_ENTRY_GET_VALUEI(widget);
	    if(output->width > 0)
	    {
		/* Maintain aspect by updating the height? */
		if(d->output_maintain_aspect)
		{ 
		    const gfloat aspect = PrintDlgGetSourceImageAspect(d);
		    if(aspect > 0.0f)
		    {
			output->height = output->width / aspect;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(d->output_height_spin),
			    (gfloat)output->height
			);
		    }
		}
		PrintDlgPreviewRecreate(d);
		PrintDlgPreviewQueueDraw(d);
	    }
	}
	else if(widget == d->output_height_spin)
	{
	    output->height = GTK_ENTRY_GET_VALUEI(widget);
	    if(output->height > 0)
	    {
		/* Maintain aspect by updating the width? */
		if(d->output_maintain_aspect)
		{
		    const gfloat aspect = PrintDlgGetSourceImageAspect(d);
		    if(aspect > 0.0f)
		    {
			output->width = output->height * aspect;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(d->output_width_spin),
			    (gfloat)output->width
			);
		    }
		}
		PrintDlgPreviewRecreate(d);
		PrintDlgPreviewQueueDraw(d);
	    }
	}
	else if(widget == d->output_frame_spin)
	{
	    d->src_frame_num = GTK_ENTRY_GET_VALUEI(widget) - 1;
	    PrintDlgPreviewRecreate(d);
	    PrintDlgPreviewQueueDraw(d);
	}
	else if(widget == d->output_maintain_aspect_check)
	{
	    d->output_maintain_aspect = GTK_TOGGLE_BUTTON_GET_ACTIVE(widget);
	    if(d->output_maintain_aspect)
	    {
		const gfloat aspect = PrintDlgGetSourceImageAspect(d);
		if(aspect > 0.0f)
		{
		    /* Update the width and height values */
		    output->height = output->width / aspect;
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_width_spin),
			(gfloat)output->width
		    );
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_height_spin),
			(gfloat)output->height
		    );
		    PrintDlgPreviewRecreate(d);
		    PrintDlgPreviewQueueDraw(d);
		}
	    }
	}

	d->freeze_count--;
}

/*
 *	Print callback.
 */
static void PrintDlgPrintCB(GtkWidget *widget, gpointer data)
{
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(!GTK_WIDGET_SENSITIVE(widget))
	    return;

	/* Update response code */
	d->got_response = TRUE;

	/* Break out of highest GTK+ main loop */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}

/*
 *	Cancel callback.
 */
static void PrintDlgCancelCB(GtkWidget *widget, gpointer data)
{
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(!GTK_WIDGET_SENSITIVE(widget))
	    return;

	/* Update response code */
	d->got_response = FALSE;

	/* Break out of highest GTK+ main loop */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}

/*
 *	Preview GtkDrawingArea event signal callback.
 */
static gint PrintDlgPreviewEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean need_grab_pointer = FALSE;
	GdkCursor *cur = NULL;
	GdkWindow *window;
	GdkEventConfigure *configure;
	GdkEventCrossing *crossing;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	gboolean keystate;
	guint keyval;
	imgview_image_struct *preview_img;
	PrintDlg *d = PRINT_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	window = widget->window;
	preview_img = d->preview_img;

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    if((configure->width > 0) && (configure->height > 0))
	    {
		GDK_PIXMAP_UNREF(d->preview_pm);
		d->preview_pm = gdk_pixmap_new(
		    window, configure->width, configure->height, -1
		);
	    }
	    PrintDlgPreviewRecreate(d);
	    PrintDlgPreviewQueueDraw(d);
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    PrintDlgPreviewDraw(d);
	    status = TRUE;
	    break;

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    if(!crossing->send_event)
	    {
#if 0
		/* Need to reset all modifiers */
		if(d->drag_modifiers != 0x00000000)
		{
		    d->drag_modifiers = 0x00000000;
		    if(window != NULL)
			gdk_window_set_cursor(window, NULL);
		}
#endif
		/* Set up widget to grab focus so "key_press_event" and
		 * and "key_release_event" events will be sent to it
		 */
		gtk_widget_grab_focus(widget);

		/* Force tooltip to show, this will send a synthetic
		 * GDK_ENTER_NOTIFY event which we will ignore
		 */
		GUIShowTipsNow(widget);
		status = TRUE;
	    }
	    break;

	  case GDK_LEAVE_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    keyval = key->keyval;
	    keystate = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    /* Handle by keyval */
	    /* ALT? */
	    if((keyval == GDK_Alt_L) || (keyval == GDK_Alt_R))
	    {
		if(keystate)
		    d->drag_modifiers |= PRINT_DLG_DRAG_RESIZE;
		else
		    d->drag_modifiers &= ~PRINT_DLG_DRAG_RESIZE;
		status = TRUE;
	    }
	    /* CTRL? */
	    else if((keyval == GDK_Control_L) || (keyval == GDK_Control_R))
	    {
		if(keystate)
		    d->drag_modifiers |= PRINT_DLG_DRAG_TRANSLATE;
		else
		    d->drag_modifiers &= ~PRINT_DLG_DRAG_TRANSLATE;
		status = TRUE;
	    }
	    /* SHIFT? */
	    else if((keyval == GDK_Shift_L) || (keyval == GDK_Shift_R))
	    {
		if(keystate)
		    d->drag_modifiers |= PRINT_DLG_DRAG_RESIZE;
		else
		    d->drag_modifiers &= ~PRINT_DLG_DRAG_RESIZE;
		status = TRUE;
	    }
	    if(status)
	    {
		/* Is there currently no drag operation in progress? */
		if(d->drag_operation == PRINT_DLG_DRAG_NONE)
		{
		    if(d->drag_modifiers & PRINT_DLG_DRAG_TRANSLATE)
			cur = d->translate_cur;
		    if(d->drag_modifiers & PRINT_DLG_DRAG_RESIZE)
			cur = d->resize_cur;

		    /* Update the cursor */
		    if(window != NULL)
			gdk_window_set_cursor(window, cur);
		}
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:		/* Translate */
		if(d->drag_modifiers & PRINT_DLG_DRAG_RESIZE)
		{
		    /* Resize */
		    cur = d->resize_cur;
		    need_grab_pointer = TRUE;

		    d->drag_operation |= PRINT_DLG_DRAG_RESIZE;
		    d->drag_last_x = (gint)button->x;
		    d->drag_last_y = (gint)button->y;

		    if(!PrintDlgOutputGetOffsetW(
			d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		    ))
		    {
			d->drag_outline_x[0] = 0;
			d->drag_outline_y[0] = 0;
		    }
		    if(preview_img != NULL)
		    {
			d->drag_outline_x[1] = d->drag_outline_x[0] +
			    preview_img->width;
			d->drag_outline_y[1] = d->drag_outline_y[0] +
			    preview_img->height;
		    }
		    d->have_prev_outline = FALSE;
		    d->only_draw_outline = TRUE;;
		    PrintDlgPreviewQueueDraw(d);
		}
		else
		{
		    /* Translate */
		    cur = d->translate_cur;
		    need_grab_pointer = TRUE;
		    d->drag_operation |= PRINT_DLG_DRAG_TRANSLATE;
		    d->drag_last_x = (gint)button->x;
		    d->drag_last_y = (gint)button->y;

		    if(!PrintDlgOutputGetOffsetW(
		        d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		    ))
		    {
		        d->drag_outline_x[0] = 0;
		        d->drag_outline_y[0] = 0;
		    }
		    if(preview_img != NULL)
		    {
		        d->drag_outline_x[1] = d->drag_outline_x[0] +
			    preview_img->width;
			d->drag_outline_y[1] = d->drag_outline_y[0] +
			    preview_img->height;
		    }
		    d->have_prev_outline = FALSE;
		    d->only_draw_outline = TRUE;
		    PrintDlgPreviewQueueDraw(d);
		}
		status = TRUE;
		break;

	      case GDK_BUTTON2:		/* Resize */
		cur = d->resize_cur;
		need_grab_pointer = TRUE;
		d->drag_operation |= PRINT_DLG_DRAG_RESIZE;
		d->drag_last_x = (gint)button->x;
		d->drag_last_y = (gint)button->y;

		if(!PrintDlgOutputGetOffsetW(
		    d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		))
		{
		    d->drag_outline_x[0] = 0;
		    d->drag_outline_y[0] = 0;
		}
		if(preview_img != NULL)
		{
		    d->drag_outline_x[1] = d->drag_outline_x[0] +
			preview_img->width;
		    d->drag_outline_y[1] = d->drag_outline_y[0] +
			preview_img->height;
		}
		/* If mapped, then draw only the outline */
		d->have_prev_outline = FALSE;
		d->only_draw_outline = TRUE;
		PrintDlgPreviewQueueDraw(d);
		status = TRUE;
		break;

	      case GDK_BUTTON4:		/* Shrink */
		if(d->drag_modifiers == 0x00000000)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->output_width_spin)
		    );
		    GdkRectangle *geo = &d->output_img_geometry;
		    if(adj != NULL)
		    {
			const gfloat aspect = (geo->height > 0) ?
			    ((gfloat)geo->width / (gfloat)geo->height) : 1.0f;
			if(aspect > 0.0f)
			{
			    geo->width -= MAX(
				adj->page_increment,
				adj->step_increment
			    );
			    if(geo->width < 1)
				geo->width = 1;

			    geo->height = MAX(geo->width / aspect, 1);

			    PrintDlgOutputSetWidgets(d);
			    PrintDlgPreviewRecreate(d);
			    PrintDlgPreviewQueueDraw(d);
			}
		    }
		}
		status = TRUE;
		break;

	      case GDK_BUTTON5:		/* Increase size  */
		if(d->drag_modifiers == 0x00000000)
		{
		    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			GTK_SPIN_BUTTON(d->output_width_spin)
		    );
		    GdkRectangle *geo = &d->output_img_geometry;
		    if(adj != NULL)
		    {
			const gfloat aspect = (geo->height > 0) ?
			    ((gfloat)geo->width / (gfloat)geo->height) : 1.0f;
			if(aspect > 0.0f)
			{
			    geo->width += MAX(
				adj->page_increment,
				adj->step_increment
			    );
			    if(geo->width < 1)
				geo->width = 1;

			    geo->height = MAX(geo->width / aspect, 1);

			    PrintDlgOutputSetWidgets(d);
			    PrintDlgPreviewRecreate(d);
			    PrintDlgPreviewQueueDraw(d);
			}
		    }
		}
		status = TRUE;
		break;
	    }
	    /* Set window cursor if specified */
	    if((window != NULL) && (cur != NULL))
		gdk_window_set_cursor(window, cur);

	    /* Grab pointer? */
	    if(need_grab_pointer && (window != NULL))
	    {
		gdk_pointer_grab(
		    window,
		    FALSE,
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK,
		    NULL,
		    NULL,
		    button->time
		);
		need_grab_pointer = FALSE;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;

	    /* Ungrab the pointer */
	    if(gdk_pointer_is_grabbed())
		gdk_pointer_ungrab(button->time);

	    /* Translate */
	    if(d->drag_operation & PRINT_DLG_DRAG_TRANSLATE)
	    {
		const gint	ox = MIN(
		    d->drag_outline_x[0], d->drag_outline_x[1]
		),
				oy = MIN(
		    d->drag_outline_y[0], d->drag_outline_y[1]
		);
		gint ww, wh;
		gfloat wtoo_coeff;
		GdkRectangle	*medium = &d->medium_size,
				*output = &d->output_img_geometry;

		ww = widget->allocation.width;
		wh = widget->allocation.height;
		if(ww > 0)
		    wtoo_coeff = (gfloat)medium->width / (gfloat)ww;
		else
		    wtoo_coeff = 0.0f;
		output->x = ox * wtoo_coeff;

		if(wh > 0)
		    wtoo_coeff = (gfloat)medium->height / (gfloat)wh;
		else
		    wtoo_coeff = 0.0f;
		output->y = oy * wtoo_coeff;

		d->drag_operation &= ~PRINT_DLG_DRAG_TRANSLATE;
		PrintDlgOutputSetWidgets(d);
                d->have_prev_outline = FALSE;
                d->only_draw_outline = FALSE;
		PrintDlgPreviewQueueDraw(d);
		status = TRUE;
	    }
	    /* Resize */
	    if(d->drag_operation & PRINT_DLG_DRAG_RESIZE)
	    {
		const gint	ow = ABSOLUTE(
		    d->drag_outline_x[1] - d->drag_outline_x[0]
		),
				oh = ABSOLUTE(
		    d->drag_outline_y[1] - d->drag_outline_y[0]
		);
		const gint	ww = widget->allocation.width,
				wh = widget->allocation.height;
		gfloat wtoo_coeff;
		GdkRectangle	*medium = &d->medium_size,
				*geo = &d->output_img_geometry;

		/* Calculate the output size */
		wtoo_coeff = (ww > 0) ?
		    ((gfloat)medium->width / (gfloat)ww) : 0.0f;
		if((ow > 0) && (wtoo_coeff > 0.0f))
		    geo->width = ow * wtoo_coeff;

		wtoo_coeff = (wh > 0) ?
		    ((gfloat)medium->height / (gfloat)wh) : 0.0f;
		if((oh > 0) && (wtoo_coeff > 0.0f))
		    geo->height = oh * wtoo_coeff;

		d->drag_operation &= ~PRINT_DLG_DRAG_RESIZE;
		PrintDlgOutputSetWidgets(d);
		PrintDlgPreviewRecreate(d);
                d->have_prev_outline = FALSE;
                d->only_draw_outline = FALSE;
		PrintDlgPreviewQueueDraw(d);
		status = TRUE;
	    }
	    /* If there are no drag modifiers then reset the cursor */
	    if(d->drag_modifiers == PRINT_DLG_DRAG_NONE)
	    {
		if(window != NULL)
		    gdk_window_set_cursor(window, NULL);
	    }
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(d->drag_operation != PRINT_DLG_DRAG_NONE)
	    {
		gint x, y, dx, dy;
		GdkModifierType mask;

		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
			motion->window, &x, &y, &mask
		    );
		}
		else
		{
		    x = (gint)motion->x;
		    y = (gint)motion->y;
		    mask = motion->state;
		}

		/* Translate */
		if(d->drag_operation & PRINT_DLG_DRAG_TRANSLATE)
		{
		    dx = x - d->drag_last_x;
		    dy = y - d->drag_last_y;

		    d->drag_outline_x[0] += dx;
		    d->drag_outline_y[0] += dy;
		    d->drag_outline_x[1] += dx;
		    d->drag_outline_y[1] += dy;

		    /* Draw only the outline if mapped */
		    if(d->map_state)
		    {
		        d->only_draw_outline = TRUE;
			PrintDlgPreviewQueueDraw(d);
		    }

		    d->drag_last_x = x;
		    d->drag_last_y = y;

		    status = TRUE;
		}
		/* Resize */
		if(d->drag_operation & PRINT_DLG_DRAG_RESIZE)
		{
		    dx = x - d->drag_last_x;
		    dy = y - d->drag_last_y;

		    /* Apply the delta values to resize drag outline
		     * based on if maintaining aspect or not
		     */
		    if(d->output_maintain_aspect)
		    {
			/* Maintaining aspect, so get aspect and resize
			 * drag output while maintaining aspect
			 */
			gint len;
			gfloat aspect = PrintDlgGetSourceImageAspect(d);

			if(aspect <= 0.0f)
			    aspect = 1.0f;

			/* Use greater of the two compoent movements */
			if(ABSOLUTE(dx) > ABSOLUTE(dy))
			{
			    d->drag_outline_x[1] += dx;
			    len = MAX(
				(d->drag_outline_x[1] - 
				    d->drag_outline_x[0]),
				0
			    );
			    d->drag_outline_y[1] = d->drag_outline_y[0] +
				(len / aspect);
			}
			else
			{
			    d->drag_outline_y[1] += dy;
			    len = MAX(
				(d->drag_outline_y[1] - 
				    d->drag_outline_y[0]),
				0 
			    );
			    d->drag_outline_x[1] = d->drag_outline_x[0] +
				(len * aspect);
			}
		    }
		    else
		    {
			d->drag_outline_x[1] += dx;
			d->drag_outline_y[1] += dy;
		    }
		    /* Make sure drag outline does not get smaller
		     * than its origin
		     */
		    if(d->drag_outline_x[1] < d->drag_outline_x[0])
			d->drag_outline_x[1] = d->drag_outline_x[0];
		    if(d->drag_outline_y[1] < d->drag_outline_y[0])
			d->drag_outline_y[1] = d->drag_outline_y[0];

		    /* Draw only the outline */
		    d->only_draw_outline = TRUE;
		    PrintDlgPreviewQueueDraw(d);

		    d->drag_last_x = x;
		    d->drag_last_y = y;

		    status = TRUE;
		}
	    }
	    break;
	}

	return(status);
}


/*
 *	Gets the visual.
 */
static print_visual PrintDlgGetVisual(PrintDlg *d)
{
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_bw_radio
	))
	    return(PRINT_VISUAL_BW);
	else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_greyscale_radio
	))
	    return(PRINT_VISUAL_GREYSCALE);
	else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_color_radio
	))
	    return(PRINT_VISUAL_RGB);
	else
	    return(PRINT_VISUAL_BW);
}

/*
 *	Gets the paper size based on the currently selected printer,
 *	orientation, and resolution.
 *
 *	The width_rtn and height_rtn specifies the return values for
 *	the paper size in pixels.
 *
 *	Returns TRUE if the paper size was obtained or FALSE on error.
 */
static gboolean PrintDlgGetMediumSize(
	PrintDlg *d,
	gint *width_rtn, gint *height_rtn
)
{
	gint resolution;
	const gchar	*xprint_server_address = gtk_entry_get_text(
	    GTK_ENTRY(d->xprint_server_address_entry)
			),
			*printer_name,
			*tray_name,
			*medium_name,
			*orientation_name;
	GList *glist;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	pulist_struct *pulist;
	iv_print_xprint_printer_struct *p;
	iv_print_xprint_medium_struct *m;

	/* Get the selected printer */
	pulist = PUListBoxGetPUList(d->printer_pulistbox);
	p = IV_PRINT_XPRINT_PRINTER(
	    PUListGetItemData(
		pulist,
		PUListGetSelectedLast(pulist)
	    )
	);
	if(p != NULL)
	    printer_name = p->name;
	else
	    printer_name = NULL;

	/* Get the selected medium */
	clist = GTK_CLIST(d->medium_ctree);
	ctree = GTK_CTREE(clist);
	glist = clist->selection_end;
	node = (glist != NULL) ?
	    (GtkCTreeNode *)glist->data : NULL;
	m = IV_PRINT_XPRINT_MEDIUM(
	    gtk_ctree_node_get_row_data(ctree, node)
	);
	if(m != NULL)
	{
	    tray_name = m->tray;
	    medium_name = m->name;
	}
	else
	{
	    tray_name = NULL;
	    medium_name = NULL;
	}

	/* Get the selected orientation */
	pulist = PUListBoxGetPUList(d->orientation_pulistbox);
	orientation_name = (const gchar *)PUListGetItemData(
	    pulist,
	    PUListGetSelectedLast(pulist)
	);

	/* Get the selected resolution */
	pulist = PUListBoxGetPUList(d->resolution_pulistbox);
	resolution = (gint)PUListGetItemData(
	    pulist,
	    PUListGetSelectedLast(pulist)
	);

	/* Get the medium size from the X Print server based on the
	 * currently selected printer, tray, medium, orientation, and
	 * resolution
	 */
	if(iv_print_xprint_get_medium_size(
	    xprint_server_address,
	    printer_name,
	    tray_name,
	    medium_name,
	    orientation_name,
	    resolution,
	    width_rtn, height_rtn
	))
	    return(FALSE);
	else
	    return(TRUE);
}

/*
 *	Returns the Print Window's source image aspect.
 */
static gfloat PrintDlgGetSourceImageAspect(PrintDlg *d)
{
	const imgview_image_struct *img = d->src_img;
	if(img == NULL)
	    return(0.0f);

	if(img->height <= 0)
	    return(0.0f);
	else
	    return((gfloat)img->width / (gfloat)img->height);
}

/*
 *	Sets the output widget values to match the current output
 *	values.
 *
 *	The preview will not be updated or drawn.
 */
static void PrintDlgOutputSetWidgets(PrintDlg *d)
{
	GdkRectangle *output = &d->output_img_geometry;

	d->freeze_count++;

	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->output_x_spin),
	    (gfloat)output->x
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->output_y_spin),
	    (gfloat)output->y
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->output_width_spin),
	    (gfloat)output->width
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->output_height_spin),
	    (gfloat)output->height
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->output_frame_spin),
	    (gfloat)(d->src_frame_num + 1)
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(d->output_maintain_aspect_check),
	    d->output_maintain_aspect
	);

	d->freeze_count--;
}

/*
 *	Gets the output image's upper-left corner offset from
 *	the window's upper-left corner.
 *
 *	Returns TRUE if the output x and y offsets were obtained
 *	or FALSE on error.
 */
static gboolean PrintDlgOutputGetOffsetW(
	PrintDlg *d, gint *x, gint *y
)
{
	gfloat otow_coeff;
	GdkRectangle	*medium = &d->medium_size,
			*output = &d->output_img_geometry;
	GtkWidget *w = d->preview_da;

	if((medium->width <= 0) || (medium->height <= 0))
	{
	    if(x != NULL)
		*x = 0;
	    if(y != NULL)
		*y = 0;
	    return(TRUE);
	}

	/* Calculate offsets */
	otow_coeff = (gfloat)w->allocation.width /
	    (gfloat)medium->width;
	if(x != NULL)
	    *x = output->x * otow_coeff;
	otow_coeff = (gfloat)w->allocation.height /
	    (gfloat)medium->height;
	if(y != NULL)
	    *y = output->y * otow_coeff;

	return(TRUE);
}


/*
 *	Copies the image data from src_img to tar_img.
 *
 *	Both images must have the same size and depth.
 */
static void PrintDlgRevertImages(
	imgview_image_struct *src_img, imgview_image_struct *tar_img
)
{
	gint bpp, src_bpl, tar_bpl, min_bpl, height;
	const guint8 *src_data, *src_line, *src_line_end;
	guint8 *tar_data, *tar_line;
	imgview_frame_struct *src_frame, *tar_frame;

	if((src_img == NULL) || (tar_img == NULL))
	    return;

	/* Source image geometry and bpp must match the target image */
	if((src_img->bpp != tar_img->bpp) ||
	   (src_img->width != tar_img->width) ||
	   (src_img->height != tar_img->height)
	)
	{
	    g_printerr(
"PrintDlgRevertImages(): BPP or geometry mismatch, cannot copy.\n"
	    );
	    return;
	}

	bpp = src_img->bpp;
	height = src_img->height;

	src_bpl = src_img->bpl;
	tar_bpl = tar_img->bpl;
	min_bpl = src_img->width * bpp;

	src_frame = ImgViewImageGetFrame(src_img, 0);
	tar_frame = ImgViewImageGetFrame(tar_img, 0);

	/* Get pointers to both image's buffers, both must be allocated */
	src_data = src_frame->buf;
	tar_data = tar_frame->buf;
	if((src_data == NULL) || (tar_data == NULL))
	    return;

	for(src_line = src_data,
	    src_line_end = src_line + (src_bpl * height),
	    tar_line = tar_data;
	    src_line < src_line_end;
	    src_line += src_bpl,
	    tar_line += tar_bpl
	)
	    memcpy(tar_line, src_line, min_bpl);
}


/*
 *	Resets the image data in preview_img with the data from 
 *	orig_preview_img.
 */
static void PrintDlgRevertPreview(PrintDlg *d)
{
	if(d == NULL)
	    return;

	/* Copy data from the original preview image to the preview
	 * image that will be ready for drawing on preview_da
	 */
	PrintDlgRevertImages(
	    d->orig_preview_img,	/* Source */
	    d->preview_img		/* Target */
	);
}


/*
 *	Updates the Print Window's preview_img to reflect the current
 *	visual values.
 *
 *	PrintDlgRevertPreview() will be called to revert the data.
 */
static void PrintDlgPreviewUpdateVisual(PrintDlg *d)
{
	print_values_struct *pv = PrintValuesNew();

	/* Copy the data from the orig_preview_img to the preview_img */
	PrintDlgRevertPreview(d);

	/* Set up the print values to match the current print values */
	pv->flags = PRINT_VALUE_BRIGHTNESS | PRINT_VALUE_CONTRAST |
	    PRINT_VALUE_VISUAL;
	pv->brightness = GTK_ADJUSTMENT_GET_VALUE(d->brightness_adj);
	pv->contrast = GTK_ADJUSTMENT_GET_VALUE(d->contrast_adj);
	pv->visual = PrintDlgGetVisual(d);

	/* Modify the preview's color, brightness, and contrast */
	PrintAdjustImageColor(d->preview_img, pv, NULL, NULL);

	PrintValuesDelete(pv);
}

/*
 *	Recreates the original preview image orig_preview_img and
 *	the preview image preview_img.
 *
 *	The new preview_img will contain the same data as 
 *	orig_preview_img.
 *
 *	This function will call PrintDlgPreviewUpdateVisual() to make
 *	sure that the new images are updated and ready to be drawn.
 */
static void PrintDlgPreviewRecreate(PrintDlg *d)
{
	gint i, new_width, new_height;
	gfloat x_coeff, y_coeff;
	GdkRectangle	*medium = &d->medium_size,
			*geo = &d->output_img_geometry;
	GtkWidget *w = d->preview_da;
	imgview_frame_struct	*src_frame,
				*tar_frame;
	imgview_image_struct	*tar_img,
				*src_img;

	/* Delete the existing preview images */
	ImgViewImageDelete(d->preview_img);
	d->preview_img = NULL;
	ImgViewImageDelete(d->orig_preview_img);
	d->orig_preview_img = NULL;

	/* Get the source image */
	src_img = d->src_img;
	src_frame = ImgViewImageGetFrame(src_img, d->src_frame_num);
	if((src_frame != NULL) ? (src_frame->buf == NULL) : TRUE)
	    return;

	if((medium->width <= 0) || (medium->height <= 0) ||
	   (geo->width <= 0) || (geo->height <= 0)
	)
	    return;

	/* Calculate the coefficients that represent the amount of
	 * the output image visible on the paper
	 */
	x_coeff = (gfloat)geo->width / (gfloat)medium->width;
	y_coeff = (gfloat)geo->height / (gfloat)medium->height;

	/* Calculate the size of the new preview image */
	new_width = (gint)(w->allocation.width * x_coeff);
	new_height = (gint)(w->allocation.height * y_coeff);

	if((new_width <= 0) || (new_height <= 0))
	    return;


	/* Create the new preview images */
	d->orig_preview_img = tar_img = ImgViewImageNew(
	    new_width, new_height, src_img->bpp, 0
	);
	i = ImgViewImageAppendFrameNew(tar_img);
	tar_frame = ImgViewImageGetFrame(tar_img, i);

	d->preview_img = ImgViewImageNew(
	    new_width, new_height, src_img->bpp, 0
	);
	i = ImgViewImageAppendFrameNew(d->preview_img);

	if((tar_frame != NULL) ? (tar_frame->buf != NULL) : FALSE)
	{
	    const gint bpp = src_img->bpp;
	    if(bpp != tar_img->bpp)
	    {
		g_printerr(
"PrintDlgPreviewRecreate(): Error:\
Source and target images do not have the same bytes per pixel.\n"
		);
		return;
	    }

#if 0
	    /* Copy/resize by orientation */
	    if(orientation == PRINT_ORIENTATION_LANDSCAPE)
	    {
		/* Create a tempory image rotated 90 degrees of the
		 * target image's size
		 */
		imgview_frame_struct *tmp_frame;
		imgview_image_struct *tmp_img = ImgViewImageNew(
		    tar_img->height, tar_img->width,
		    tar_img->bpp, 0
		);
		i = ImgViewImageAppendFrameNew(tmp_img);
		tmp_frame = ImgViewImageGetFrame(tmp_img, i);
		if(tmp_frame != NULL)
		{
		    /* Copy/resize source image to tempory image */
		    GUIImageBufferResize(
			bpp,
			src_frame->buf,
			src_img->width, src_img->height, src_img->bpl,
			tmp_frame->buf,
			tmp_img->width, tmp_img->height, tmp_img->bpl,
			NULL, NULL
		    );

		    /* Copy/rotate tempory image to target image */
		    GUIImageBufferRotateCCW90(
			bpp,
			tmp_frame->buf,
			tmp_img->width, tmp_img->height, tmp_img->bpl,
			tar_frame->buf,
			tar_img->width, tar_img->height, tar_img->bpl
		    );
		}

		/* Delete the tempory image */
		ImgViewImageDelete(tmp_img);
	    }
	    else
#endif
	    {
		/* Copy/resize the source image to the target image */
		GUIImageBufferResize(
		    bpp,
		    src_frame->buf,
		    src_img->width, src_img->height, src_img->bpl,
		    tar_frame->buf,
		    tar_img->width, tar_img->height, tar_img->bpl,
		    NULL, NULL
		);
	    }

	    /* Apply background to the target image */
	    GUIImageBufferFlattenWithBG(
		bpp,
		tar_frame->buf,
		tar_img->width, tar_img->height, tar_img->bpl,
		d->bg_color,
		NULL, NULL
	    );
	}

	/* This will call PrintDlgRevertPreview() to first copy the data
	 * from orig_preview_img to preview_img and modify preview_img to
	 * reflect the current visual settings
	 */
	PrintDlgPreviewUpdateVisual(d);
}

/*
 *	Resizes the preview GtkDrawingArea based on the current
 *	size of the medium.
 *
 *	The preview GtkDrawingArea will be queued to resize and will
 *	result in the preview images being recreated
 */
static void PrintDlgPreviewResize(PrintDlg *d)
{
	gint	width, height,		/* Nominal size of preview_da */
		new_width, new_height;
	gfloat aspect;
	GdkRectangle *medium = &d->medium_size;

	/* Set nominal display size of preview drawing area */
	width = PRINT_DLG_PREVIEW_WIDTH;
	height = PRINT_DLG_PREVIEW_HEIGHT;

	if((medium->width <= 0) || (medium->height <= 0))
	    return;

	/* Calculate the aspect ratio of the medium */
	aspect = (gfloat)medium->width / (gfloat)medium->height;
	if(aspect <= 0.0f)
	    return;

	/* Calculate new size of the preview GtkDrawingArea */
	if(aspect >= 1.0f)
	{
	    new_width = width;
	    new_height = height / aspect;
	}
	else
	{
	    new_width = width * aspect;
	    new_height = height;
	}

	/* Set the new size of preview GtkDrawingArea */
	gtk_widget_set_usize(
	    d->preview_da,
	    new_width, new_height
	);

	/* Queue the toplevel GtkWindow to resize */
	gtk_widget_queue_resize(d->toplevel);

	/* We do not need to recreate the preview images since a
	 * "configure_event" will be generated and the signal handler
	 * will see it and recreate the preview images
	 */
}

/*
 *	Draws the preview.
 *
 *	The d->only_draw_outline will be cleared by this function.
 */
static void PrintDlgPreviewDraw(PrintDlg *d)
{
	gint ww, wh;
	GdkGC *gc;
	gboolean only_draw_outline;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStateType state;
	GtkWidget *w;
	GtkStyle *style;

	/* Get marker for if we only want to draw the drag outlines */
	only_draw_outline = d->only_draw_outline;
	d->only_draw_outline = FALSE;	/* Always reset it in this function */

	w = d->preview_da;
	ww = w->allocation.width;
	wh = w->allocation.height;
	window = w->window;
	state = GTK_WIDGET_STATE(w);
	gc = d->preview_invert_gc;
	style = gtk_widget_get_style(w);
	if((ww <= 0) || (wh <= 0) || (window == NULL))
	    return;

	drawable = (GdkDrawable *)((d->preview_pm != NULL) ?
	    d->preview_pm : window
	);

	/* Draw the background only if not drawing outline */
	if(!only_draw_outline)
	    gdk_draw_rectangle(
		drawable, style->base_gc[state], TRUE,
		0, 0, ww, wh
	    );

	/* Draw the preview image? */
	if(GTK_WIDGET_SENSITIVE(w) && !only_draw_outline)
	{
	    imgview_image_struct *img = d->preview_img;
	    imgview_frame_struct *frame = ImgViewImageGetFrame(img, 0);
	    if((frame != NULL) ? (frame->buf != NULL) : FALSE)
	    {
		gint x, y;
		if(!PrintDlgOutputGetOffsetW(d, &x, &y))
		{
		    x = 0;
		    y = 0;
		}
		switch(img->bpp)
		{
		  case 4:
		    gdk_draw_rgb_32_image(
			drawable, style->white_gc,
			x, y,
			img->width, img->height,
			GDK_RGB_DITHER_NONE,
			(guchar *)frame->buf,
			img->bpl
		    );
		    break;
		  case 3:
		    gdk_draw_rgb_image(
			drawable, style->white_gc,
			x, y,
			img->width, img->height,
			GDK_RGB_DITHER_NONE,
			(guchar *)frame->buf,
			img->bpl
		    );
		    break;
		  case 1:
		    gdk_draw_gray_image(
			drawable, style->white_gc,
			x, y,
			img->width, img->height,
			GDK_RGB_DITHER_NONE,
			(guchar *)frame->buf,
			img->bpl
		    );
		    break;
		}
	    }
	    else if(style->font != NULL)
	    {
		const gchar *s = "No Image Data";
		GdkFont *font = style->font;
		const gint font_height = font->ascent + font->descent;
		GdkTextBounds b;
		gdk_string_bounds(font, s, &b);
		gdk_draw_string(
		    drawable, font, style->text_gc[state],
		    ((ww - b.width) / 2) - b.lbearing,
		    ((wh - font_height) / 2) + font->ascent,
		    s
		);
	    }
	}


	/* Clear the previous drag marks if defined and if only
	 * drawing for drag outlines
	 */
	if(d->have_prev_outline && only_draw_outline)
	{
	     gint        x = MIN(d->prev_outline_x[0], d->prev_outline_x[1]),
			 y = MIN(d->prev_outline_y[0], d->prev_outline_y[1]);
	     gint        width = ABSOLUTE(
		d->prev_outline_x[1] - d->prev_outline_x[0]
			 ),
			 height = ABSOLUTE(
		d->prev_outline_y[1] - d->prev_outline_y[0]
			 );

	    if((width > 0) && (height > 0))
		gdk_draw_rectangle(
		    drawable, gc, FALSE, x, y, width, height
		);

	    d->have_prev_outline = FALSE;
	}

	/* Draw new drag marks? */
	if((d->drag_operation != PRINT_DLG_DRAG_NONE) &&
	   (gc != NULL)
	)
	{
	    gint	x = MIN(d->drag_outline_x[0], d->drag_outline_x[1]),
			y = MIN(d->drag_outline_y[0], d->drag_outline_y[1]);
	    gint	width = ABSOLUTE(
		d->drag_outline_x[1] - d->drag_outline_x[0]
			),
			height = ABSOLUTE(
		d->drag_outline_y[1] - d->drag_outline_y[0]
			);

	    if((width > 0) && (height > 0))
	    {
		gdk_draw_rectangle(
		    drawable, gc, FALSE, x, y, width, height
		);

		/* Record newly drawn drag outline marks */
		d->have_prev_outline = TRUE;
		d->prev_outline_x[0] = d->drag_outline_x[0];
		d->prev_outline_y[0] = d->drag_outline_y[0];
		d->prev_outline_x[1] = d->drag_outline_x[1];
		d->prev_outline_y[1] = d->drag_outline_y[1];
	    }
	}

        /* Send drawable to window if drawable is not the window */
	if(drawable != (GdkDrawable *)window)
            gdk_draw_pixmap(
                window, style->fg_gc[state], drawable,
                0, 0, 0, 0, ww, wh
            );
}

/*
 *	Queues the preview to draw.
 */
static void PrintDlgPreviewQueueDraw(PrintDlg *d)
{
	gtk_widget_queue_draw(d->preview_da);
}

/*
 *	Connects to the X Print server and gets the listing of
 *	printers from it. The first printer will be selected and
 *	subsequently the values for the first printer. The preview
 *	will be updated.
 *
 *	The xprint_server_address specifies the address to the X
 *	Print server. If xprint_server_address is NULL then the default
 *	address will be used instead.
 */
static void PrintDlgConnectXPrintServer(
	PrintDlg *d,
	const gchar *xprint_server_address,
	const gboolean verbose
)
{
	gchar *xprint_server_address_d = STRDUP(xprint_server_address);
	const gchar *printer_name;
	GdkRectangle *geo;
	pulistbox_struct *pulistbox;
	pulist_struct *pulist;
	iv_print_xprint_printer_struct *p;

	/* Update the X Print server address entry */
	if(xprint_server_address_d != NULL)
	    gtk_entry_set_text(
		GTK_ENTRY(d->xprint_server_address_entry),
		xprint_server_address_d
	    );

	/* Get the list of printers */
	PrintDlgPrintersListUpdate(d, xprint_server_address_d, TRUE);

	/* Get values for the selected printer, updating the
	 * mediums list, orientations list, and resolutions list
	 */
	pulistbox = d->printer_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	p = IV_PRINT_XPRINT_PRINTER(
	    PUListGetItemData(
		pulist,
		PUListGetSelectedLast(pulist)
	    )
	);
	if(p != NULL)
	    printer_name = p->name;
	else
	    printer_name = NULL;
	PrintDlgSetPrinterValues(d, printer_name, FALSE);

	/* Update the output values and the preview */
	geo = &d->output_img_geometry;
	geo->x = 0;
	geo->y = 0;
	PrintDlgOutputSetWidgets(d);
	PrintDlgUpdate(d);
	PrintDlgPreviewResize(d);

	g_free(xprint_server_address_d);
}

/*
 *	Gets the list of printers.
 *
 *	The xprint_server_address specifies the address to the X
 *	Print server. If xprint_server_address is NULL then the default
 *	address will be used instead.
 */
static void PrintDlgPrintersListUpdate(
	PrintDlg *d,
	const gchar *xprint_server_address,
	const gboolean verbose
)
{
	GList *printers_list;
	pulistbox_struct *pulistbox;
	pulist_struct *pulist;

	/* Get the printers list */
	pulistbox = d->printer_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	PUListClear(pulist);
	PUListBoxSetLinesVisible(pulistbox, 1);
	printers_list = iv_print_xprint_get_printers_list(
	    xprint_server_address
	);
	if(printers_list != NULL)
	{
	    gint i = 0;
	    gchar *s;
	    GList *glist;
	    iv_print_xprint_printer_struct *p, *p2;

	    for(glist = printers_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		p = IV_PRINT_XPRINT_PRINTER(glist->data);
		if(p == NULL)
		    continue;

		if(STRISEMPTY(p->name))
		    continue;

		s = g_strconcat(
		    p->name,
		    " - ",
		    p->description,
		    NULL
		);
		i = PUListAddItem(pulist, s);
		g_free(s);
		p2 = iv_print_xprint_printer_new();
		if(p2 != NULL)
		{
		    p2->name = STRDUP(p->name);
		    p2->description = STRDUP(p->description);
		    PUListSetItemDataFull(
			pulist, i,
			p2, (GtkDestroyNotify)iv_print_xprint_printer_delete
		    );
		}
	    }

	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	    PUListBoxSelect(pulistbox, 0);

	    iv_print_xprint_printers_list_delete(printers_list);
	}
	else if(verbose)
	{
	    gchar *msg = g_strdup_printf(
"No printers found or unable to connect to the X Print server at:\n\
\n\
    %s",
		xprint_server_address
	    );
	    CDialogSetTransientFor(d->toplevel);
	    CDialogGetResponse(
		"No Printers Found",
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	}
}

/*
 *	Gets the printer's tray, medium, orientation, and resolution
 *	values.
 *
 *	The tray, medium, orientation, and resolution lists will be
 *	updated and the first value in each will be selected.
 *
 *	The printer_name specifies the printer. If printer_name is
 *	NULL then the lists will be cleared.
 *
 *	If verbose is TRUE then any errors or problems that are
 *	encountered will be displayed.
 */
static void PrintDlgSetPrinterValues(
	PrintDlg *d,
	const gchar *printer_name,
	const gboolean verbose
)
{
	const gchar *xprint_server_address = gtk_entry_get_text(
	    GTK_ENTRY(d->xprint_server_address_entry)
	);
	GList *glist;
	pulistbox_struct *pulistbox;
	pulist_struct *pulist;

	/* Mediums */
	PrintDlgMediumsListUpdate(
	    d,
	    printer_name,
	    verbose
	);

	/* Orientations */
	pulistbox = d->orientation_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	PUListClear(pulist);
	PUListBoxSetLinesVisible(pulistbox, 1);
	glist = iv_print_xprint_get_printer_orientations(
	    xprint_server_address,
	    printer_name
	);
	if(glist != NULL)
	{
	    gint i = 0;
	    GList *glist2 = glist;
	    while(glist2 != NULL)
	    {
		i = PUListAddItem(pulist, (const gchar *)glist2->data);
		PUListSetItemDataFull(
		    pulist, i,
		    STRDUP(glist2->data),
		    (GtkDestroyNotify)g_free
		);
		glist2 = g_list_next(glist2);
	    }
	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	    PUListBoxSelect(pulistbox, 0);
	    g_list_foreach(glist, (GFunc)g_free, NULL);
	    g_list_free(glist);
	}

	/* Resolutions */
	pulistbox = d->resolution_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	PUListClear(pulist);
	PUListBoxSetLinesVisible(pulistbox, 1);
	glist = iv_print_xprint_get_printer_resolutions(
	    xprint_server_address,
	    printer_name
	);
	if(glist != NULL)
	{
	    gint i = 0;
	    gchar *s;
	    GList *glist2 = glist;
	    while(glist2 != NULL)
	    {
		s = g_strdup_printf("%i DPI", (gint)glist2->data);
		i = PUListAddItem(pulist, s);
		g_free(s);
		PUListSetItemData(
		    pulist, i, glist2->data
		);
		glist2 = g_list_next(glist2);
	    }
	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	    PUListBoxSelect(pulistbox, 0);
	    g_list_free(glist);
	}
}

/*
 *	Updates the mediums list with the mediums from the tray.
 *
 *	The printer_name specifies the printer.
 *
 *	The tray_name specifies the tray. If tray_name is NULL then
 *	the mediums from all the trays will be added to the list.
 *	the default tray will be used.
 *
 *	If verbose is TRUE then any errors or problems that are
 *	encountered will be displayed.
 */
static void PrintDlgMediumsListUpdate(
	PrintDlg *d,
	const gchar *printer_name,
	const gboolean verbose
)
{
	const gchar *xprint_server_address = gtk_entry_get_text(
	    GTK_ENTRY(d->xprint_server_address_entry)
	);
	GList *mediums_list;
	GtkCList *clist = GTK_CLIST(d->medium_ctree);
	GtkCTree *ctree = GTK_CTREE(clist);

	gtk_clist_freeze(clist);

	/* Delete all the existing nodes */
	g_list_free(d->medium_toplevel_nodes_list);
	d->medium_toplevel_nodes_list = NULL;
	gtk_clist_clear(clist);
	gtk_clist_columns_autosize(clist);

	/* Get the new mediums list */
	mediums_list = iv_print_xprint_get_printer_mediums(
	    xprint_server_address,
	    printer_name
	);
	if(mediums_list != NULL)
	{
	    gint i;
	    gchar **strv = (gchar **)g_malloc(clist->columns * sizeof(gchar *));
	    GList *glist;
	    GtkCTreeNode *parent = NULL, *first_medium_node = NULL;
	    iv_print_xprint_medium_struct *m;

	    for(i = 0; i < clist->columns; i++)
		strv[i] = "";

	    /* Append the mediums to the mediums GtkCTree */
	    i = 0;
	    for(glist = mediums_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = IV_PRINT_XPRINT_MEDIUM(glist->data);
		if(m != NULL)
		{
		    GtkCTreeNode *node;
		    iv_print_xprint_medium_struct *m2;

		    /* If this medium has a specified tray then
		     * look for the parent node who's tray matches
		     * this medium's tray
		     */
		    if(!STRISEMPTY(m->tray))
		    {
			/* Search through the medium toplevel nodes
			 * list for one who's tray matches this
			 * medium's tray
			 */
			GList *glist;
			GtkCTreeNode *node;
			iv_print_xprint_medium_struct *m_tray;

			parent = NULL;	/* Need to clear the last parent node
					 * before searching */

			for(glist = d->medium_toplevel_nodes_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    node = (GtkCTreeNode *)glist->data;
			    if(node == NULL)
				continue;

			    m_tray = IV_PRINT_XPRINT_MEDIUM(
				gtk_ctree_node_get_row_data(ctree, node)
			    );
			    if(m_tray == NULL)
				continue;

			    if(STRISEMPTY(m_tray->tray))
				continue;

			    if(!g_strcasecmp(m->tray, m_tray->tray))
			    {
				parent = node;
				break;
			    }
			}
		    }

		    /* If no toplevel node has been created for
		     * this tray then create a new one and append
		     * it to the medium toplevel nodes list
		     */
		    if(parent == NULL)
		    {
			if(!STRISEMPTY(m->tray))
			    strv[0] = STRDUP(m->tray);
			else if(d->medium_toplevel_nodes_list != NULL)
			    strv[0] = g_strdup_printf(
				"Tray #%i",
				g_list_length(d->medium_toplevel_nodes_list) + 1
			    );
			else
			    strv[0] = STRDUP("Default Tray");
			parent = node = gtk_ctree_insert_node(
			    ctree,
			    NULL,		/* Parent */
			    NULL,		/* Sibling */
			    strv,
			    2,			/* Spacing */
			    d->tray_icon_pm,
			    d->tray_icon_mask,
			    d->tray_icon_pm,
			    d->tray_icon_mask,
			    FALSE,		/* Not leaf */
			    TRUE		/* Expanded */
			);
			g_free(strv[0]);
			strv[0] = "";
			gtk_ctree_node_set_selectable(ctree, node, FALSE);
			m2 = iv_print_xprint_medium_new();  
			if(m2 != NULL)
			{
			    if(!STRISEMPTY(m->tray))
				m2->tray = STRDUP(m->tray);
			    gtk_ctree_node_set_row_data_full(
				ctree, node,
				m2,
				(GtkDestroyNotify)iv_print_xprint_medium_delete
			    );
			}
			d->medium_toplevel_nodes_list = g_list_append(
			    d->medium_toplevel_nodes_list, node
			);
		    }

		    /* Append a node to the current parent tray for
		     * thie medium
		     */
		    if(m->name != NULL)
			strv[0] = g_strdup_printf(
			    "%s (%.0fmm x %.0fmm)",
			    m->name,
			    m->width_mm + m->x_mm,
			    m->height_mm + m->y_mm
			);
		    else
			strv[0] = g_strdup_printf(
			    "%.0fmm x %.0fmm",
			    m->width_mm + m->x_mm,
			    m->height_mm + m->y_mm
			);
		    node = gtk_ctree_insert_node(
			ctree,
			parent,
			NULL,		/* Sibling */
			strv,
			2,		/* Spacing */
			d->medium_icon_pm,
			d->medium_icon_mask,
			d->medium_icon_pm,
			d->medium_icon_mask,
			TRUE,		/* Leaf */
			FALSE		/* Not expanded */
		    );
		    g_free(strv[0]);
		    strv[0] = "";

		    m2 = iv_print_xprint_medium_new();
		    if(m2 != NULL)
		    {
			m2->name = STRDUP(m->name);
			m2->tray = STRDUP(m->tray);
			m2->x_mm = m->x_mm;
			m2->y_mm = m->y_mm;
			m2->width_mm = m->width_mm;
			m2->height_mm = m->height_mm;
			gtk_ctree_node_set_row_data_full(
			    ctree, node,
			    m2,
			    (GtkDestroyNotify)iv_print_xprint_medium_delete
			);
		    }

		    /* Record the first medium node */
		    if(first_medium_node == NULL)
			first_medium_node = node;
		}
	    }
	    gtk_clist_columns_autosize(clist);
	    if(first_medium_node != NULL)
		gtk_ctree_select(ctree, first_medium_node);
	    iv_print_xprint_mediums_list_delete(mediums_list);
	    g_free(strv);
	}
	else if(verbose)
	{
	    gchar *msg = g_strdup_printf(
"No mediums found on printer \"%s\".",
		printer_name
	    );
	    CDialogSetTransientFor(d->toplevel);
	    CDialogGetResponse(
		"No Printer Mediums Found",
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	}
	gtk_clist_thaw(clist);
}


/*
 *	Creates a new print dialog.
 */
static PrintDlg *PrintDlgNew(
	const gchar *filename,
	GtkWidget *ref_toplevel
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5,
		border_minor = 2;
	gchar *s;
	GSList *gslist;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkAdjustment *adj;
	GtkWidget	*w, *parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *main_vbox, *scroll_parent;
	GtkCList *clist;
	GtkCTree *ctree;
	pulistbox_struct *pulistbox;
	PrintDlg *d = PRINT_DLG(
	    g_malloc0(sizeof(PrintDlg))
	);
	if(d == NULL)
	    return(NULL);

	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->ref_toplevel = ref_toplevel;
	d->map_state = FALSE;
	d->got_response = FALSE;
	d->freeze_count = 0;
	d->busy_count = 0;
	d->main_level = 0;
	d->src_img = NULL;
	d->preview_img = NULL;
	d->orig_preview_img = NULL;
	d->drag_modifiers = PRINT_DLG_DRAG_NONE;
	d->drag_operation = PRINT_DLG_DRAG_NONE;

	d->busy_cur = gdk_cursor_new(GDK_WATCH);
	d->translate_cur = gdk_cursor_new(GDK_FLEUR);
	d->resize_cur = gdk_cursor_new(GDK_SIZING);

	d->tray_icon_pm = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &d->tray_icon_mask,
	    (guint8 **)icon_printer_tray_20x20_xpm
	);
	d->medium_icon_pm = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &d->medium_icon_mask,
	    (guint8 **)icon_file_20x20_xpm
	);

	/* Create the toplevel GtkWindow */
	gtk_widget_push_visual(gdk_rgb_get_visual());
	gtk_widget_push_colormap(gdk_rgb_get_cmap());
	d->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
	gtk_window_set_policy(
	    GTK_WINDOW(w), TRUE, TRUE, TRUE
	);
	gtk_widget_set_usize(w, PRINT_DLG_WIDTH, PRINT_DLG_HEIGHT);
	s = g_strdup_printf(
	    PRINT_DLG_TITLE ": %s",
	    STRISEMPTY(filename) ? "Untitled" : filename
	);
	gtk_window_set_title(GTK_WINDOW(w), s);
	g_free(s);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_print2_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(PrintDlgDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	parent = w;


	/* Main vbox */
	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Vbox for child widgets */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent = w;


	/* Print To GtkFrame */
	w = gtk_frame_new("Print To");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* X Print Server Address */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("X Print Server Address:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->xprint_server_address_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(PrintDlgXPrintServerConnectCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
"Enter the address to the X Print Server in the format\
 \"hostname:displaynumber\" or just \":displaynumber\" and\
 then press ENTER"
	);
	gtk_widget_show(w);

	/* Printer */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Printer:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->printer_pulistbox = pulistbox = PUListBoxNew(
	    parent3,
	    -1, -1
	);
	PUListBoxSetChangedCB(
	    pulistbox,
	    PrintDlgPrinterChangedCB, d
	);
	PUListBoxMap(pulistbox);


	/* GtkHBox to divide into two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Visual GtkFrame */
	w = gtk_frame_new("Visual");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Visual GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Black & White, Greyscale, and Color GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Black & White GtkRadioButton */
	gslist = NULL;
	d->visual_bw_radio = w =
	    gtk_radio_button_new_with_label(gslist, "B&W");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Greyscale GtkRadioButton */
	d->visual_greyscale_radio = w =
	    gtk_radio_button_new_with_label(gslist, "GreyScale");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Color GtkRadioButton */
	d->visual_color_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Color");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);

	/* Brightness GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Hbox for ticks */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_minus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_brightness_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_plus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Brightness scale */
	d->brightness_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Initial value */
	    -1.0f,		/* Minimum */
	    1.0f,		/* Maximum */
	    0.1f,		/* Step inc */
	    0.1f,		/* Page inc */
	    0.0f		/* Page size */
	);
	/* Scale */
	gtk_object_ref(GTK_OBJECT(adj));	/* Keep adj around */
	d->brightness_scale = w = gtk_hscale_new(adj);
	gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	gtk_scale_set_value_pos(GTK_SCALE(w), GTK_POS_RIGHT);
	gtk_scale_set_digits(GTK_SCALE(w), 2);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(PrintDlgAdjustmentValueChangedCB), d
	);
	GUISetWidgetTip(w, "Brightness");
	gtk_widget_show(w);

	/* Contrast GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Hbox for ticks */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_minus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_contrast_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_plus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Contrast scale */
	d->contrast_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Initial value */
	    -1.0f,		/* Minimum */
	    1.0f,		/* Maximum */
	    0.1f,		/* Step inc */
	    0.1f,		/* Page inc */
	    0.0f		/* Page size */
	);
	/* Scale */
	gtk_object_ref(GTK_OBJECT(adj));	/* Keep adj around */
	d->contrast_scale = w = gtk_hscale_new(adj);
	gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	gtk_scale_set_value_pos(GTK_SCALE(w), GTK_POS_RIGHT);
	gtk_scale_set_digits(GTK_SCALE(w), 2);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(PrintDlgAdjustmentValueChangedCB), d
	);
	GUISetWidgetTip(w, "Contrast");
	gtk_widget_show(w);


	/* Medium GtkFrame */
	w = gtk_frame_new("Medium");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;


	/* Medium GtkScrolledWindow */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	scroll_parent = w;

	/* Medium GtkCTree */
	d->medium_ctree = w = gtk_ctree_new(1, 0);
	clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);
	gtk_widget_set_usize(w, 250, -1);
	gtk_container_add(GTK_CONTAINER(scroll_parent), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_hide(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_BROWSE);
#ifdef CLIST_DEF_ROW_SPACING
	gtk_clist_set_row_height(clist, CLIST_DEF_ROW_SPACING);
#else
	gtk_clist_set_row_height(clist, 20);
#endif
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_select_row",
	    GTK_SIGNAL_FUNC(PrintDlgMediumSelectRowCB), d
	);
	gtk_widget_show(w);


	/* Orientation GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new("Orientation:");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->orientation_pulistbox = pulistbox = PUListBoxNew(
	    parent5,
	    -1, -1
	);
	PUListBoxSetChangedCB(
	    pulistbox,
	    PrintDlgOrientationChangedCB, d
	);
	PUListBoxMap(pulistbox);


	/* Resolution GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new("Resolution:");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->resolution_pulistbox = pulistbox = PUListBoxNew(
	    parent5,
	    -1, -1
	);
	PUListBoxSetChangedCB(
	    pulistbox,
	    PrintDlgResolutionChangedCB, d
	);
	PUListBoxMap(pulistbox);



	/* Right column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Output geometry */
	w = gtk_frame_new("Output");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Output X */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("X:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    -(gfloat)((guint16)-1 / 2),
	    (gfloat)((guint16)-1 / 2),
	    1.0f,
	    10.0f,
	    0.0f
	);
	d->output_x_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Output Y */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("Y:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    -(gfloat)((guint16)-1 / 2),
	    (gfloat)((guint16)-1 / 2),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_y_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Output Width */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("W:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    0.0f,
	    (gfloat)((guint16)-1),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_width_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Output Height */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("H:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    0.0f,
	    (gfloat)((guint16)-1),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_height_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	/* Hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Output Frame */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("Frame:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    1.0f,
	    1.0f,
	    (gfloat)((guint32)-1),
	    1.0f,
	    5.0f,
	    0.0f  
	);
	d->output_frame_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 50, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	/* Maintain Aspect */
	d->output_maintain_aspect_check = w = gtk_check_button_new_with_label(
	    "Maintain Aspect"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);

	/* Output Number Of Coppies */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("Coppies:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    1.0f,
	    1.0f,
	    (gfloat)((guint32)-1),
	    1.0f,
	    5.0f,
	    0.0f  
	);
	d->output_ncoppies_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 50, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);


	/* Preview */
	w = gtk_frame_new("Preview");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(TRUE, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Hbox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Frame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Nother vbox */
	w = gtk_vbox_new(TRUE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	parent5 = w;
	/* Drawing area */
	d->preview_da = w = gtk_drawing_area_new();
	/* Needs to accept focus for keyboard modifier events */
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK | GDK_KEY_PRESS_MASK |
	    GDK_KEY_RELEASE_MASK | GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
	    GDK_POINTER_MOTION_HINT_MASK | GDK_STRUCTURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
	    w,
"Press and hold Button1 to adjust the position,\
 press and hold SHIFT + Button1 (or Button2) to\
 adjust the size"
	);
	gtk_widget_show(w);

	/* Create the invert GC */
	if(window != NULL)
	{
	    GdkGCValues gcv;
	    GdkGCValuesMask gcv_mask = GDK_GC_FUNCTION |
		GDK_GC_LINE_WIDTH | GDK_GC_LINE_STYLE |
		GDK_GC_CAP_STYLE | GDK_GC_JOIN_STYLE;
	    gcv.function = GDK_INVERT;
	    gcv.line_width = 1;
	    gcv.line_style = GDK_LINE_SOLID;
	    gcv.cap_style = GDK_CAP_NOT_LAST;
	    gcv.join_style = GDK_JOIN_MITER;
	    d->preview_invert_gc = gdk_gc_new_with_values(
		window, &gcv, gcv_mask
	    );
	}

	d->preview_pm = NULL;


	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Print button */
	d->print_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_print2_20x20_xpm, "Print", NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintDlgPrintCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_p);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele", 
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"                             
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	return(d);
}

/*
 *	Sets the print dialog busy or ready.
 */
static void PrintDlgSetBusy(PrintDlg *d, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w = d->toplevel;
        if(busy)
        {
            d->busy_count++;
            if(d->busy_count > 1)
                return;
            cur = d->busy_cur;
        }
        else
        {   
            d->busy_count--;
            if(d->busy_count < 0)
                d->busy_count = 0;
            if(d->busy_count > 0)
                return;
            cur = NULL;			/* Use default cursor */
        }
        if(w->window != NULL)
        {
            gdk_window_set_cursor(w->window, cur);
            gdk_flush();
        }
}

/*
 *	Updates the print dialog's widgets to reflect current values.
 */
static void PrintDlgUpdate(PrintDlg *d)
{
	gboolean sensitive;
	pulistbox_struct *pulistbox;
	pulist_struct *pulist;

	pulistbox = d->printer_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	sensitive = (PUListGetTotalItems(pulist) > 0) ? TRUE : FALSE;

	gtk_widget_set_sensitive(
	    PUListBoxGetToplevel(d->printer_pulistbox), sensitive
	);
	gtk_widget_set_sensitive(d->visual_bw_radio, sensitive);
	gtk_widget_set_sensitive(d->visual_greyscale_radio, sensitive);
	gtk_widget_set_sensitive(d->visual_color_radio, sensitive);
	gtk_widget_set_sensitive(d->brightness_scale, sensitive);
	gtk_widget_set_sensitive(d->contrast_scale, sensitive);
	gtk_widget_set_sensitive(d->medium_ctree, sensitive);
	gtk_widget_set_sensitive(
	    PUListBoxGetToplevel(d->orientation_pulistbox), sensitive
	);
	gtk_widget_set_sensitive(
	    PUListBoxGetToplevel(d->resolution_pulistbox), sensitive
	);
	gtk_widget_set_sensitive(d->output_x_spin, sensitive);
	gtk_widget_set_sensitive(d->output_y_spin, sensitive);
	gtk_widget_set_sensitive(d->output_width_spin, sensitive);
	gtk_widget_set_sensitive(d->output_height_spin, sensitive);
	gtk_widget_set_sensitive(d->output_frame_spin, sensitive);
	gtk_widget_set_sensitive(d->output_maintain_aspect_check, sensitive);
	gtk_widget_set_sensitive(d->output_ncoppies_spin, sensitive);
	gtk_widget_set_sensitive(d->preview_da, sensitive);
	gtk_widget_set_sensitive(d->print_btn, sensitive);
}

/*
 *	Maps the print dialog.
 */
static void PrintDlgMap(PrintDlg *d)
{
	GtkWidget *w = d->print_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);

	gtk_widget_show(d->toplevel);
	d->map_state = TRUE;
}

/*
 *	Unmaps the print dialog.
 */
static void PrintDlgUnmap(PrintDlg *d)
{
	gtk_widget_hide(d->toplevel);
	d->map_state = FALSE;
}

/*
 *	Destroys the print dialog.
 */
static void PrintDlgDelete(PrintDlg *d)
{
	if(d == NULL)
	    return;

	d->src_img = NULL;

	/* Delete the preview images */
	gdk_flush();
	ImgViewImageDelete(d->preview_img);
	d->preview_img = NULL;
	ImgViewImageDelete(d->orig_preview_img);
	d->orig_preview_img = NULL;

	/* Delete the existing medium nodes */
	g_list_free(d->medium_toplevel_nodes_list);
	d->medium_toplevel_nodes_list = NULL;
	gtk_clist_clear(GTK_CLIST(d->medium_ctree));

	d->freeze_count++;

	/* Begin destroying widgets */
	GTK_WIDGET_DESTROY(d->menu);

	GTK_WIDGET_DESTROY(d->xprint_server_address_entry);
	PUListBoxDelete(d->printer_pulistbox);

	GTK_WIDGET_DESTROY(d->visual_bw_radio);
	GTK_WIDGET_DESTROY(d->visual_greyscale_radio);
	GTK_WIDGET_DESTROY(d->visual_color_radio);
	GTK_WIDGET_DESTROY(d->brightness_scale);
	GTK_WIDGET_DESTROY(d->contrast_scale);

	GTK_WIDGET_DESTROY(d->medium_ctree);
	PUListBoxDelete(d->orientation_pulistbox);
	PUListBoxDelete(d->resolution_pulistbox);

	GTK_WIDGET_DESTROY(d->output_x_spin);
	GTK_WIDGET_DESTROY(d->output_y_spin);
	GTK_WIDGET_DESTROY(d->output_width_spin);
	GTK_WIDGET_DESTROY(d->output_height_spin);
	GTK_WIDGET_DESTROY(d->output_frame_spin);
	GTK_WIDGET_DESTROY(d->output_maintain_aspect_check);
	GTK_WIDGET_DESTROY(d->output_ncoppies_spin);

	GTK_WIDGET_DESTROY(d->preview_da);
	GDK_PIXMAP_UNREF(d->preview_pm);

	GTK_WIDGET_DESTROY(d->print_btn);
	GTK_WIDGET_DESTROY(d->cancel_btn);

	GTK_WIDGET_DESTROY(d->toplevel);

	GTK_ACCEL_GROUP_UNREF(d->accelgrp);


	GTK_OBJECT_UNREF(d->brightness_adj);
	GTK_OBJECT_UNREF(d->contrast_adj);

	GDK_CURSOR_DESTROY(d->busy_cur);
	GDK_CURSOR_DESTROY(d->translate_cur);
	GDK_CURSOR_DESTROY(d->resize_cur);

	GDK_GC_UNREF(d->preview_invert_gc);

	GDK_PIXMAP_UNREF(d->tray_icon_pm);
	GDK_BITMAP_UNREF(d->tray_icon_mask);
	GDK_PIXMAP_UNREF(d->medium_icon_pm);
	GDK_BITMAP_UNREF(d->medium_icon_mask);

	d->freeze_count--;

	g_free(d);
}


/*
 *	Queries user to print the specified ImgView image.
 *
 *	Creates a new print dialog and blocks output until the user
 *	responds.
 *
 *	The specified values *v will be used to initialize the print
 *	values and *v will be modified if TRUE is returned, otherwise
 *	*v should be considered garbage after this call.
 */
#ifdef HAVE_LIBENDEAVOUR2
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
)
#else
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
)
#endif
{
	gboolean got_response;
	gchar *xprint_server_address = NULL;
	GdkRectangle *output;
	GtkWidget *w;
	print_value_flags vflags;
	PrintDlg *d;

	if((v == NULL) || (img == NULL))
	    return(FALSE);

	/* Create a new print dialog */
	d = PrintDlgNew(filename, toplevel);
	if(d == NULL)
	    return(FALSE);

	d->freeze_count++;

	/* Begin setting values */
	vflags = v->flags;

	/* Source ImgView image that user wants to print */
	d->src_img = img;
	if(vflags & PRINT_VALUE_FRAME)
	{
	    GtkAdjustment *adj = gtk_spin_button_get_adjustment(
		GTK_SPIN_BUTTON(d->output_frame_spin)
	    );
	    d->src_frame_num = v->frame;
	    adj->lower = 1.0f;
	    adj->upper = (gfloat)img->nframes;
	    adj->value = (gfloat)(v->frame + 1);
	    gtk_adjustment_changed(adj);
	    gtk_adjustment_value_changed(adj);
	}
	else
	{
	    d->src_frame_num = 0;
	}

	/* Background color */
	if(bg_color != NULL)
	    memcpy(&d->bg_color, bg_color, 4);
	else
	    memset(&d->bg_color, 0xffffffff, 4);

	/* X Print Server Address */
	if(vflags & PRINT_VALUE_XPRINT_SERVER_ADDRESS)
	{
	    g_free(xprint_server_address);
	    xprint_server_address = STRDUP(v->xprint_server_address);

	    if(xprint_server_address != NULL)
		gtk_entry_set_text(
		    GTK_ENTRY(d->xprint_server_address_entry),
		    xprint_server_address
		);		
	}

	/* Get the printers list */
	if(TRUE)
	{
	    PrintDlgPrintersListUpdate(
		d, xprint_server_address, FALSE
	    );
	}

	/* Printer */
	if(vflags & PRINT_VALUE_PRINTER)
	{
	    pulistbox_struct *pulistbox = d->printer_pulistbox;
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);
	    const gint m = PUListGetTotalItems(pulist);
	    iv_print_xprint_printer_struct *p;

	    if(v->printer != NULL)
	    {
		const gchar *printer_name = v->printer;
		gint i;
		for(i = 0; i < m; i++)
		{
		    p = IV_PRINT_XPRINT_PRINTER(
			PUListGetItemData(pulist, i)
		    );
		    if(p == NULL)
			continue;

		    if(STRISEMPTY(p->name))
			continue;

		    if(!g_strcasecmp(printer_name, p->name))
		    {
			PUListBoxSelect(pulistbox, i);
			break;
		    }
		}
	    }
	    else
	    {
		PUListBoxSelect(pulistbox, 0);
	    }

	    /* Get the selected printer values */
	    p = IV_PRINT_XPRINT_PRINTER(
		PUListGetItemData(
		    pulist,
		    PUListGetSelectedLast(pulist)
		)
	    );
	    if(p != NULL)
		PrintDlgSetPrinterValues(
		    d,
		    p->name,
		    FALSE
		);
	}

	/* Medium */
	if((vflags & PRINT_VALUE_INPUT_TRAY) &&
	   (vflags & PRINT_VALUE_MEDIUM)
	)
	{
	    gboolean selected = FALSE;
	    GtkCList *clist = GTK_CLIST(d->medium_ctree);
	    GtkCTree *ctree = GTK_CTREE(clist);

	    gtk_clist_freeze(clist);

	    /* Tray specified? */
	    if(!STRISEMPTY(v->input_tray) && !STRISEMPTY(v->medium))
	    {
		const gchar	*tray_name = v->input_tray,
				*medium_name = v->medium;
		GList *glist;
		GtkCTreeNode *parent, *node;
		GtkCTreeRow *row;
		iv_print_xprint_medium_struct *m;

		/* Select the first medium who's tray and medium
		 * matches the specified tray and medium
		 */
		for(glist = d->medium_toplevel_nodes_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    parent = (GtkCTreeNode *)glist->data;
		    if(parent == NULL)
			continue;

		    row = GTK_CTREE_ROW(parent);
		    node = (row != NULL) ? row->children : NULL;
		    while(node != NULL)
		    {
			m = IV_PRINT_XPRINT_MEDIUM(
			    gtk_ctree_node_get_row_data(ctree, node)
			);
			if((m != NULL) ?
	    ((m->tray != NULL) && (m->name != NULL)) : FALSE
			)
			{
			    if(!g_strcasecmp(m->tray, tray_name) &&
			       !g_strcasecmp(m->name, medium_name)
			    )
			    {
				gtk_clist_unselect_all(clist);
#if 0
				gtk_ctree_node_moveto(
				    ctree, node, -1,
				    0.5f, 0.0f
				);
#endif
				gtk_ctree_select(ctree, node);
				selected = TRUE;
				break;
			    }
			}
			row = GTK_CTREE_ROW(node);
			node = (row != NULL) ? row->sibling : NULL;
		    }
		}
	    }
	    else if(!STRISEMPTY(v->medium))
	    {
		const gchar *medium_name = v->medium;
		GList *glist;
		GtkCTreeNode *parent, *node;
		GtkCTreeRow *row;
		iv_print_xprint_medium_struct *m;

		/* No tray specified, select the first medium that
		 * matches the specified medium
		 */
		for(glist = d->medium_toplevel_nodes_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    parent = (GtkCTreeNode *)glist->data;
		    if(parent == NULL)
			continue;

		    row = GTK_CTREE_ROW(parent);
		    node = (row != NULL) ? row->children : NULL;
		    while(node != NULL)
		    {
			m = IV_PRINT_XPRINT_MEDIUM(
			    gtk_ctree_node_get_row_data(ctree, node)
			);
			if((m != NULL) ? (m->name != NULL) : FALSE)
			{
			    if(!g_strcasecmp(m->name, medium_name))
			    {
				gtk_clist_unselect_all(clist);
#if 0
				gtk_ctree_node_moveto(
				    ctree, node, -1,
				    0.5f, 0.0f
				);
#endif
				gtk_ctree_select(ctree, node);
				selected = TRUE;
				break;
			    }
			}
			row = GTK_CTREE_ROW(node);
			node = (row != NULL) ? row->sibling : NULL;
		    }
		}
	    }

	    gtk_clist_thaw(clist);
	}

	/* Orientation */
	if(vflags & PRINT_VALUE_ORIENTATION)
	{
	    pulistbox_struct *pulistbox = d->orientation_pulistbox;
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);
	    const gint m = PUListGetTotalItems(pulist);

	    if(v->orientation != NULL)
	    {
		const gchar *orientation_name = v->orientation;
		gint i;
		const gchar *s;
		for(i = 0; i < m; i++)
		{
		    s = (const gchar *)PUListGetItemData(pulist, i);
		    if(s == NULL)
			continue;

		    if(!g_strcasecmp(orientation_name, s))
		    {
			PUListBoxSelect(pulistbox, i);
			break;
		    }
		}
	    }
	    else
	    {
		PUListBoxSelect(pulistbox, 0);
	    }
	}

	/* Resolution */
	if(vflags & PRINT_VALUE_RESOLUTION)
	{
	    pulistbox_struct *pulistbox = d->resolution_pulistbox;
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);
	    const gint m = PUListGetTotalItems(pulist);

	    if(v->resolution > 0)
	    {
		gint i, dpi;
		for(i = 0; i < m; i++)
		{
		    dpi = (gint)PUListGetItemData(pulist, i);
		    if(dpi == 0)
			continue;

		    if(v->resolution == dpi)
		    {
			PUListBoxSelect(pulistbox, i);
			break;
		    }
		}
	    }
	    else
	    {
		PUListBoxSelect(pulistbox, 0);
	    }
	}

	/* Brightness */
	if(vflags & PRINT_VALUE_BRIGHTNESS)
	{
	    GtkAdjustment *adj = d->brightness_adj;
	    if(adj != NULL)
	    {
		adj->value = v->brightness;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		if(adj->value > adj->upper)
		    adj->value = adj->upper;
		gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    }
	}

	/* Contrast */
	if(vflags & PRINT_VALUE_CONTRAST)
	{
	    GtkAdjustment *adj = d->contrast_adj;
	    if(adj != NULL)
	    {
		adj->value = v->contrast;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		if(adj->value > adj->upper)
		    adj->value = adj->upper;
		gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    }
	}

	/* Visual */
	if(vflags & PRINT_VALUE_VISUAL)
	{
	    w = d->visual_bw_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = d->visual_greyscale_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = d->visual_color_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;

	    switch(v->visual)
	    {
	      case PRINT_VISUAL_BW:
		w = d->visual_bw_radio;
		break;
	      case PRINT_VISUAL_GREYSCALE:
		w = d->visual_greyscale_radio;
		break;
	      case PRINT_VISUAL_RGB:
		w = d->visual_color_radio;
		break;
	    }
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
	}

	/* Number Of Coppies */
	if(vflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->output_ncoppies_spin),
		(gfloat)v->output_ncoppies
	    );

	/* Output Image Geometry */
	output = &d->output_img_geometry;
	if(vflags & PRINT_VALUE_OUTPUT_X)
	    output->x = v->output_x;
	if(vflags & PRINT_VALUE_OUTPUT_Y)
	    output->y = v->output_y;
	if(vflags & PRINT_VALUE_OUTPUT_WIDTH)
	    output->width = v->output_width;
	if(vflags & PRINT_VALUE_OUTPUT_HEIGHT)
	    output->height = v->output_height;
	if(vflags & PRINT_VALUE_OUTPUT_MAINTAIN_ASPECT)
	    d->output_maintain_aspect = v->output_maintain_aspect;
	PrintDlgOutputSetWidgets(d);

	PrintDlgUpdate(d);

	/* Get the initial medium size */
	if(TRUE)
	{
	    gint medium_width, medium_height;
	    if(PrintDlgGetMediumSize(
		d,
		&medium_width, &medium_height
	    ))
	    {
		GdkRectangle *medium = &d->medium_size;
		medium->width = medium_width;
		medium->height = medium_height;
	    }
	    else
	    {
		GdkRectangle *medium = &d->medium_size;
		medium->width = PRINT_DLG_DEF_MEDIUM_WIDTH;
		medium->height = PRINT_DLG_DEF_MEDIUM_HEIGHT;
	    }
	    PrintDlgPreviewResize(d);
	}


	/* Set the reference toplevel? */
	if((toplevel != NULL) && (d->toplevel != NULL))
	{
	    if(GTK_IS_WINDOW(toplevel))
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(d->toplevel), TRUE
	        );
	        gtk_window_set_transient_for(
		    GTK_WINDOW(d->toplevel), GTK_WINDOW(toplevel)
	        );
	    }
	}

	g_free(xprint_server_address);
	xprint_server_address = NULL;

	d->freeze_count--;

	/* Map the print dialog */
	PrintDlgMap(d);

	/* Enter main loop and wait for user response */
	d->main_level++;
	gtk_main();

	/* Break out of any additional main loop levels */
	while(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}


	/* Got response? */
	got_response = d->got_response;
	if(got_response)
	{
	    /* X Print Server Address */
	    if(vflags & PRINT_VALUE_XPRINT_SERVER_ADDRESS)
	    {
		g_free(v->xprint_server_address);
		v->xprint_server_address = STRDUP(gtk_entry_get_text(
		    GTK_ENTRY(d->xprint_server_address_entry)
		));
	    }
	    /* Printer */
	    if(vflags & PRINT_VALUE_PRINTER)
	    {
		pulist_struct *pulist = PUListBoxGetPUList(d->printer_pulistbox);
		iv_print_xprint_printer_struct *p = IV_PRINT_XPRINT_PRINTER(
		    PUListGetItemData(
			pulist,
			PUListGetSelectedLast(pulist)
		    )
		);
		if(p != NULL)
		{
		    g_free(v->printer);
		    v->printer = STRDUP(p->name);
		}
	    }
	    /* Input Tray & Medium */
	    if((vflags & PRINT_VALUE_INPUT_TRAY) &&
	       (vflags & PRINT_VALUE_MEDIUM)
	    )
	    {
		GtkCList *clist = GTK_CLIST(d->medium_ctree);
		GList *glist = clist->selection_end;
		GtkCTreeNode *node = (glist != NULL) ?
		    (GtkCTreeNode *)glist->data : NULL;
		iv_print_xprint_medium_struct *m = IV_PRINT_XPRINT_MEDIUM(
		    gtk_ctree_node_get_row_data(GTK_CTREE(clist), node)
		);
		if(m != NULL)
		{
		    g_free(v->input_tray);
		    v->input_tray = STRDUP(m->tray);
		    g_free(v->medium);
		    v->medium = STRDUP(m->name);
		}
	    }
	    /* Orientation */
	    if(vflags & PRINT_VALUE_ORIENTATION)
	    {
		pulist_struct *pulist = PUListBoxGetPUList(d->orientation_pulistbox);
		g_free(v->orientation);
		v->orientation = STRDUP((const gchar *)PUListGetItemData(
		    pulist,
		    PUListGetSelectedLast(pulist)
		));
	    }
	    /* Resolution */
	    if(vflags & PRINT_VALUE_RESOLUTION)
	    {
		pulist_struct *pulist = PUListBoxGetPUList(d->resolution_pulistbox);
		v->resolution = (gint)PUListGetItemData(
		    pulist,
		    PUListGetSelectedLast(pulist)
		);
	    }
	    /* Brightness */
	    if(vflags & PRINT_VALUE_BRIGHTNESS)
	    {
		GtkAdjustment *adj = d->brightness_adj;
		if(adj != NULL)
		{
		    v->brightness = adj->value;
		}
	    }
	    /* Contrast */
	    if(vflags & PRINT_VALUE_CONTRAST)
	    {
		GtkAdjustment *adj = d->contrast_adj;
		if(adj != NULL)
		{
		    v->contrast = adj->value;
		}
	    }
	    /* Visual */
	    if(vflags & PRINT_VALUE_VISUAL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->visual_bw_radio
		))
		    v->visual = PRINT_VISUAL_BW;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(   
		    d->visual_greyscale_radio 
		)) 
		    v->visual = PRINT_VISUAL_GREYSCALE; 
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->visual_color_radio 
		)) 
		    v->visual = PRINT_VISUAL_RGB;
	    }

	    /* Frame */
	    if(vflags & PRINT_VALUE_FRAME)
		v->frame = d->src_frame_num;

	    /* Output image geometry */
	    output = &d->output_img_geometry;
	    if(vflags & PRINT_VALUE_OUTPUT_X)
		v->output_x = output->x;
	    if(vflags & PRINT_VALUE_OUTPUT_Y)
		v->output_y = output->y;
	    if(vflags & PRINT_VALUE_OUTPUT_WIDTH)
		v->output_width = output->width;
	    if(vflags & PRINT_VALUE_OUTPUT_HEIGHT)
		v->output_height = output->height;

	    if(vflags & PRINT_VALUE_OUTPUT_MAINTAIN_ASPECT)
		v->output_maintain_aspect = d->output_maintain_aspect;

	    if(vflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	    {
		w = d->output_ncoppies_spin;
		if((w != NULL) ? GTK_IS_SPIN_BUTTON(w) : FALSE)
		    v->output_ncoppies = gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
	    }
	}


	/* Unset the reference toplevel */
	if(d->toplevel != NULL)
	{
	    gtk_widget_hide(d->toplevel);
	    gtk_window_set_modal(
		GTK_WINDOW(d->toplevel), FALSE
	    );
	    gtk_window_set_transient_for(
		GTK_WINDOW(d->toplevel), NULL
	    );
	}

	/* Delete the print dialog */
	PrintDlgDelete(d);

	return(got_response);
#undef PRINT_COMMAND_MAX
}

#endif
