#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "guiutils.h"
#include "guirgbimg.h"
#include "progressdialog.h"
#include "imgview.h"
#include "resizedlg.h"
#include "config.h"

#include "images/icon_resize_20x20.xpm"
#include "images/icon_resize_32x32.xpm"
#include "images/icon_resize_48x48.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


typedef struct _Dialog			Dialog;
#define DIALOG(p)			((Dialog *)(p))
typedef struct _ResizeProgressData	ResizeProgressData;
#define RESIZE_PROGRESS_DATA(p)		((ResizeProgressData *)(p))


static gint resize_progress_cb(
	const gulong i, const gulong m, gpointer data
);
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void changed_cb(GtkWidget *widget, gpointer data);
static void resize_cb(GtkWidget *widget, gpointer data);
static void cancel_cb(GtkWidget *widget, gpointer data);
static void coeff_changed_cb(GtkWidget *widget, gpointer data);
static void size_changed_cb(GtkWidget *widget, gpointer data);

static void resize_image(
	Dialog *d,
	const gint width, const gint height,
	const gboolean verbose
);

static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
);

gboolean iv_resize(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gint *width,
	gint *height,
	gboolean *maintain_aspect
);


/*
 *	Resize Dialog:
 */
struct _Dialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gboolean	has_changes,
			got_response;
	imgview_struct	*iv;

	GtkWidget	*ok_btn,
			*cancel_btn,
			*close_btn,
			*maintain_aspect_check,
			*width_spin,
			*height_spin,
			*width_coeff_spin,
			*height_coeff_spin;

	GtkAdjustment	*width_adj,
			*height_adj,
			*width_coeff_adj,
			*height_coeff_adj;

	gfloat		orig_width,
			orig_height;
	gfloat		orig_aspect;		/* Width / height */
};


/*
 *	Progress Callback Data:
 */
struct _ResizeProgressData {
	gint            cur_frame,
			nframes;
	gfloat          unit_frame_coeff;
	gboolean        aborted;
};


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Resize progress callback.
 */
static gint resize_progress_cb(
	const gulong i, const gulong m, gpointer data
)
{
	gfloat v;
	ResizeProgressData *d = RESIZE_PROGRESS_DATA(data);
	if(d == NULL)
	    return(-2);

	v = (d->cur_frame * d->unit_frame_coeff) +
	    ((m > 0) ? (d->unit_frame_coeff * (gfloat)i / (gfloat)m) : 0.0f);

	if(ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		v,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		d->aborted = TRUE;
		return(-4);
	    }
	}    

	return(0);
}  

/*
 *	Resize Dialog toplevel GtkWindow "delete_event" signal
 *	callback.
 */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	d->got_response = FALSE;
	gtk_main_quit();

	return(TRUE);
}

/*
 *	Changed callback.
 */
static void changed_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	{
	    gtk_widget_set_sensitive(d->ok_btn, TRUE);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	    d->has_changes = TRUE;
	}

	d->freeze_count--;
}

/*
 *	Resize callback.
 */
static void resize_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Cancel callback.
 */
static void cancel_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}

/*
 *	Coefficient GtkSpinButton "changed" signal callback.
 */
static void coeff_changed_cb(GtkWidget *widget, gpointer data)
{
	gboolean maintain_aspect;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(!GTK_IS_SPIN_BUTTON(widget))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	maintain_aspect = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->maintain_aspect_check
	);

	if(widget == d->width_coeff_spin)
	{
	    const gfloat	v = GTK_ENTRY_GET_VALUEF(widget),
				new_width = v * d->orig_width;
	    gtk_adjustment_set_value(d->width_adj, new_width);
	    if(maintain_aspect)
	    {
		const gfloat new_height = v * d->orig_height;
		gtk_adjustment_set_value(d->height_adj, new_height);
		gtk_adjustment_set_value(d->height_coeff_adj, v);
	    }
	}
	else if(widget == d->height_coeff_spin)
	{
	    const gfloat	v = GTK_ENTRY_GET_VALUEF(widget),
				new_height = v * d->orig_height;
	    gtk_adjustment_set_value(d->height_adj, new_height);
	    if(maintain_aspect)
	    {
		const gfloat new_width = v * d->orig_width;
		gtk_adjustment_set_value(d->width_adj, new_width);
		gtk_adjustment_set_value(d->width_coeff_adj, v);
	    }
	}

	d->freeze_count--;
}

/*
 *	Size GtkSpinButton "changed" signal callback.
 */
static void size_changed_cb(GtkWidget *widget, gpointer data)
{
	gboolean maintain_aspect;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(!GTK_IS_SPIN_BUTTON(widget))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	maintain_aspect = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->maintain_aspect_check
	);

	/* Maintain aspect? */
	if(maintain_aspect)
	{
	    /* Original aspect known? */
	    const gfloat aspect = d->orig_aspect;
	    if(aspect > 0.0f)
	    {
		/* Set the other adjustment's value in order to maintain
		 * the aspect
		 */
		if(widget == d->width_spin)
		{
		    const gfloat new_height = GTK_ENTRY_GET_VALUEF(widget) /
			aspect;
		    gtk_adjustment_set_value(d->height_adj, new_height);
		}
		else if(widget == d->height_spin)
		{
		    const gfloat new_width = GTK_ENTRY_GET_VALUEF(widget) *
			aspect;
		    gtk_adjustment_set_value(d->width_adj, new_width);
		}
	    }
	}

	/* Update the coefficient size adjustments after updating the
	 * size adjustments
	 */
	if(maintain_aspect)
	{
	    const gfloat new_coeff = (d->orig_width > 0.0f) ?
		(d->width_adj->value / d->orig_width) : 0.0f;
	    gtk_adjustment_set_value(d->width_coeff_adj, new_coeff);
	    gtk_adjustment_set_value(d->height_coeff_adj, new_coeff);
	}
	else
	{
	    gtk_adjustment_set_value(
	        d->width_coeff_adj,
	        (d->orig_width > 0.0f) ?
		    (d->width_adj->value / d->orig_width) : 0.0f
	    );
	    gtk_adjustment_set_value(
	        d->height_coeff_adj,
	        (d->orig_height > 0.0f) ?
		    (d->height_adj->value / d->orig_height) : 0.0f
	    );
	}

	d->freeze_count--;
}


/*
 *	Resizes the image.
 *
 *	The width and height specifies the new size.
 *
 *	If verbose is TRUE then the progress dialog will be used to
 *	display the progress.
 */
static void resize_image(
	Dialog *d,
	const gint width, const gint height,
	const gboolean verbose
)
{
	gint i, bpp, tar_width = width, tar_height = height;
	guint8 *tar_buf;
	GList *glist;
	imgview_frame_struct *frame;
	imgview_struct *iv = d->iv;
	const gint m = ImgViewGetTotalFrames(iv);
	imgview_image_struct *tar_img, *src_img = ImgViewGetImage(iv);
	gint (*progress_cb)(const gulong, const gulong, gpointer);
	ResizeProgressData *progress_data;

	if((tar_width <= 0) || (tar_height <= 0) || (src_img == NULL))
	    return;

	bpp = src_img->bpp;

	/* Create the resized image at the new size */
	tar_img = ImgViewImageNew(
	    tar_width, tar_height, bpp, 0
	);
	if(tar_img == NULL)
	    return;

	/* Allocate the progress callback data if verbose */
	if(verbose)
	{
	    progress_cb = resize_progress_cb;
	    progress_data = RESIZE_PROGRESS_DATA(g_malloc0(
		sizeof(ResizeProgressData)
	    ));

	    /* Map the progress dialog and set up the progress callback
	     * data
	     */    
	    if((progress_data != NULL) && (m > 0))
	    {
		gchar *msg = (m > 1) ? g_strdup_printf(
"Resizing frame %i of %i...",
		    0 + 1, m
		) : STRDUP(
"Resizing..."
		);
		GtkWidget *toplevel = d->toplevel;
		if(ProgressDialogIsQuery())
		    ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(toplevel);   
		ProgressDialogMap(
		    "Resizing",
		    msg,
		    (const guint8 **)icon_resize_32x32_xpm,
		    "Stop"
		);
		g_free(msg);
		gdk_flush();

		progress_data->nframes = m;
		progress_data->unit_frame_coeff = 1.0f / (gfloat)m;
	    }
	}
	else
	{
	    progress_cb = NULL;
	    progress_data = NULL;
	}

	/* Iterate through each frame on the source image */
	for(glist = src_img->frames_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    frame = IMGVIEW_FRAME(glist->data);
	    if(frame == NULL)
		continue;

	    if(progress_data != NULL)
	    {
		gchar *msg;

		if(progress_data->aborted)
		    break;
		progress_data->cur_frame = i;

		msg = (m > 1) ? g_strdup_printf(
"Resizing frame %i of %i...",
		    i + 1, m
		) : STRDUP(
"Resizing..."
		);
		ProgressDialogUpdate(
		    NULL, msg, NULL, NULL,
		    i * progress_data->unit_frame_coeff,
#ifdef PROGRESS_BAR_DEF_TICKS
		    PROGRESS_BAR_DEF_TICKS,
#else
		    25,
#endif
		    TRUE
		);
		g_free(msg);
	    }

	    /* Allocate a new image data of the new size */
	    tar_buf = (guint8 *)g_malloc(
		tar_img->bpl * tar_img->height
	    );
	    GUIImageBufferResize(
		bpp,
		frame->buf,
		src_img->width, src_img->height, src_img->bpl,
		tar_buf,
		tar_img->width, tar_img->height, tar_img->bpl,
		progress_cb, progress_data
	    );
	    ImgViewImageAppendFrame(
		tar_img, tar_buf, frame->delay
	    );
	}

	/* User aborted? */
	if(progress_data->aborted)
	{
	    /* Delete the target image */
	    ImgViewImageDelete(tar_img);
	}
	else
	{
	    /* Set the newly resized image to the ImgView */
	    ImgViewSetImage(iv, tar_img);
	}

	ProgressDialogBreakQuery(TRUE);    
	ProgressDialogSetTransientFor(NULL);

	g_free(progress_data);
}


/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;


	/* GtkAccelGroup */
	d->accelgrp = accelgrp = gtk_accel_group_new();

	/* Toplevel GtkWindow */
	d->toplevel = toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_resize_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Resize Button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_resize_20x20_xpm,
	    "Resize",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(resize_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_r);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);

	/* Cancel Button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele", 
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close Button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
	    "Close",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);
}


/*
 *	Queries user for Resize options.
 *
 *	Returns TRUE on Resize or FALSE on Cancel.
 */
gboolean iv_resize(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gint *width,
	gint *height,
	gboolean *maintain_aspect
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gboolean got_response;
	gchar *s;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	gfloat aspect = 0.0f;
	Dialog *d;

	/* Create dialog base widgets */
	d = DIALOG(g_malloc0(sizeof(Dialog)));
	s = g_strdup_printf(
	    "Resize: %s",
	    STRISEMPTY(filename) ? "Untitled" : filename
	);
	build_std_dialog(
	    d,
	    -1, -1,
	    s,				/* Title */
	    ref_toplevel,
	    &toplevel, &client_vbox
	);
	g_free(s);
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	if((width != NULL) && (height != NULL))
	{
	    if(*height > 0)
		aspect = (gfloat)(*width) / (gfloat)(*height);
	}
	d->iv = iv;
	d->orig_aspect = aspect;
	d->orig_width = (width != NULL) ? *width : 1.0f;
	d->orig_height = (height != NULL) ? *height : 1.0f;

	d->freeze_count++;

	/* Create the Resize Dialog widgets */
	if(client_vbox != NULL)
	{
	    gchar *s;
	    GdkWindow *window = toplevel->window;
	    GtkAdjustment *adj;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget	*w,
			*parent = client_vbox,
			*parent2, *parent3, *parent4, *parent5;


	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent = w;


	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		window,
		(style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		(guint8 **)icon_resize_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);   


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Hbox for Original Size */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Original Size Label */
	    s = g_strdup_printf(
		"Original Size: %ix%i",
		(gint)d->orig_width, (gint)d->orig_height
	    );
	    w = gtk_label_new(s);
	    g_free(s);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Hbox for Maintain Aspect */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Maintain Aspect Check Button */
	    s = g_strdup_printf(
		"Maintain Original Aspect: %.4f",
		d->orig_aspect
	    );
	    d->maintain_aspect_check = w = gtk_check_button_new_with_label(s);
	    g_free(s);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Hbox for Width & Width Coeff */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for Width */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Alignment & Label */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, 50, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    w = gtk_label_new("Width:");
	    gtk_container_add(GTK_CONTAINER(parent5), w);
	    gtk_widget_show(w);
	    /* Spin Button */
	    d->width_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f,
		1.0f,
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f 
	    );
	    d->width_spin = w = gtk_spin_button_new(
		adj,
		1.0,		/* Climb rate (0.0 to 1.0) */
		0		/* Digits */
	    );
	    gtk_widget_set_usize(w, 80, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_grab_focus(w);		/* Grab focus */
	    gtk_widget_show(w);

	    /* Hbox for Width Coeff */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Alignment & Label */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, 40, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    w = gtk_label_new("Coeff:");
	    gtk_container_add(GTK_CONTAINER(parent5), w);
	    gtk_widget_show(w);
	    /* Spin button */
	    d->width_coeff_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f,
		0.0f,
		(gfloat)((guint16)-1),
		0.01f,
		0.1f,
		0.0f
	    );
	    d->width_coeff_spin = w = gtk_spin_button_new(
		adj,
		1.0,		/* Climb rate (0.0 to 1.0) */
		4		/* Digits */
	    );
	    gtk_widget_set_usize(w, 80, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);


	    /* Hbox for Height & Height Coeff */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for Height */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Alignment & Label */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, 50, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    w = gtk_label_new("Height:");
	    gtk_container_add(GTK_CONTAINER(parent5), w);
	    gtk_widget_show(w);
	    /* Spin Button */
	    d->height_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f,
		1.0f,
		(gfloat)((guint16)-1),
		1.0f,
		10.0f,
		0.0f  
	    );
	    d->height_spin = w = gtk_spin_button_new(
		adj,
		1.0,		/* Climb rate (0.0 to 1.0) */
		0		/* Digits */
	    );
	    gtk_widget_set_usize(w, 80, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);

	    /* Hbox for Height Coeff */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Alignment & Label */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, 40, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    w = gtk_label_new("Coeff:");
	    gtk_container_add(GTK_CONTAINER(parent5), w);
	    gtk_widget_show(w);
	    /* Spin button */
	    d->height_coeff_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f,
		0.0f,
		(gfloat)((guint16)-1),
		0.01f,
		0.1f, 
		0.0f  
	    );
	    d->height_coeff_spin = w = gtk_spin_button_new(
		adj,
		1.0,		/* Climb rate (0.0 to 1.0) */
		4		/* Digits */
	    );
	    gtk_widget_set_usize(w, 80, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);

	    /* Connect signals for size adjustments */
	    gtk_signal_connect(
		GTK_OBJECT(d->width_spin), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->width_spin), "changed",
		GTK_SIGNAL_FUNC(size_changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->height_spin), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->height_spin), "changed",
		GTK_SIGNAL_FUNC(size_changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->width_coeff_spin), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->width_coeff_spin), "changed",
		GTK_SIGNAL_FUNC(coeff_changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->height_coeff_spin), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(d->height_coeff_spin), "changed",
		GTK_SIGNAL_FUNC(coeff_changed_cb), d
	    );


	    /* Set initial values */
	    w = d->maintain_aspect_check;
	    if((maintain_aspect != NULL) ? *maintain_aspect : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;

	    adj = d->width_adj;
	    if(width != NULL)
		gtk_adjustment_set_value(adj, (gfloat)(*width));

	    adj = d->height_adj;
	    if(height != NULL)
		gtk_adjustment_set_value(adj, (gfloat)(*height));
	}

	d->freeze_count--;

	/* Map the dialog */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	got_response = d->got_response;
	if(got_response)
	{
	    GtkAdjustment *adj;
	    GtkWidget *w = d->maintain_aspect_check;
	    if((maintain_aspect != NULL) && (w != NULL))
		*maintain_aspect = GTK_TOGGLE_BUTTON(w)->active;

	    adj = d->width_adj;
	    if((width != NULL) && (adj != NULL))
		*width = (gint)GTK_ADJUSTMENT_GET_VALUE(adj);

	    adj = d->height_adj;
	    if((height != NULL) && (adj != NULL))  
		*height = (gint)GTK_ADJUSTMENT_GET_VALUE(adj);

	    /* Resize the image */
	    if((width != NULL) && (height != NULL))
		resize_image(
		    d,
		    *width, *height,
		    TRUE			/* Verbose */
		);
	}


	/* Destroy the dialog */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	g_free(d);

	return(got_response);
}
