#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#if defined(_WIN32)
# include <gdk/gdkwin32.h>
#else
# include <gdk/gdkx.h>
# define HAVE_X
#endif

#include "guiutils.h"
#include "guirgbimg.h"
#include "screenshot.h"


/*
 *	Grab Types:
 */
typedef enum {
	GRAB_UNDETERMINED,		/* Not yet determined */
	GRAB_WINDOW,			/* Grab a window */
	GRAB_RECTANGULAR,		/* Grab a rectangular area */
	GRAB_ABORT			/* User aborted grab */
} grab_type;


/* Pointer to last grab error message */
static const gchar	*last_screenshot_error = NULL;


typedef struct _grab_data_struct	grab_data_struct;


/*
 *	Grab Data:
 */
struct _grab_data_struct {

	grab_type	type;
	GdkWindow	*root_win;
	gboolean	active;		/* Indicates when a grab becomes
					 * active, while this is FALSE no grab
					 * stuff should be processed on the
					 * signal callbacks */
	gint		gtk_main_level;

	gint		grab_button_number;	/* Indicates which button is
						 * currently pressed, can be
						 * -1 for no button pressed */

	/* Coordinates for grab, x[0] and y[0] represent starting
	 * coordinates for grab_type GRAB_RECTANGULAR or GRAB_WINDOW
	 *
	 * The x[1] and y[1] are only defined if grab_type is
	 * GRAB_RECTANGULAR
	 *
	 * All these values are defined if grab_type is GRAB_UNDETERMINED
	 */
	gint		x[2], y[2];

#if defined(HAVE_X)
	Display		*xdisplay;
	int		xscreen_num;
	Window		xrootwin;
	GC		gc_inverse;
	/* Last drawn rectangle position and size */
	Bool		have_last_rect;
	int		last_rect_x,
			last_rect_y;
	unsigned int	last_rect_width,
			last_rect_height;
#endif

};
#define GRAB_DATA(p)	((grab_data_struct *)(p))


#if defined(HAVE_X)
static void draw_X_grab_rectangle(grab_data_struct *grab_data, Bool draw_new);
static gint grab_X_window(
	imgview_struct *iv, grab_data_struct *grab_data,
	gint *width, gint *height,
	gboolean include_decorations, gboolean resize_on_load
);
static gint grab_X_rectangular(
	imgview_struct *iv, grab_data_struct *grab_data,
	gint *width, gint *height,
	gboolean include_decorations, gboolean resize_on_load
);
#endif

static gint grab_window_destroy_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint grab_delay_tocb(gpointer data);
static gint grab_button_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint grab_motion_cb(
	GtkWidget *widget, GdkEventMotion *motion, gpointer data
);

const gchar *iv_screenshot_get_error(void);
gint iv_screenshot_start(
	imgview_struct *iv,
	gint *width_rtn, gint *height_rtn,
	const gulong delay,			/* In milliseconds */
	const gboolean include_decorations,
	const gboolean resize_on_load,
	GtkWidget *grab_window,
	GdkCursor *grab_cursor
);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE_VALUE(x)	(((x) < 0) ? ((x) * -1) : (x))


#if defined(HAVE_X)
/*
 *	Draws an X grab rectangle.
 *
 *	This will use the inverse GC on the given grab_data structure to;
 *	first reset the bits drawn by a previous call to this function
 *	(if there was one, check grab_data->have_last_rect == True) by
 *	drawing the rectangle bits using the previous geometry values on
 *	the grab_data structure, then draw the new rectangle bits if
 *	draw_new is True (using the same inverse GC).
 *
 *	Note that if draw_new is False then grab_data->have_last_rect will
 *	be set to False, indicating for the next call to this function
 *	there would have been no previous rectangle drawn.
 */
static void draw_X_grab_rectangle(grab_data_struct *grab_data, Bool draw_new)
{
	Display *dpy;
	Window xrootwin;
	Drawable d;
	int x, y;
	int width, height;
	GC gc;

	if(grab_data == NULL)
	    return;

	dpy = grab_data->xdisplay;
	xrootwin = grab_data->xrootwin;
	gc = grab_data->gc_inverse;
	if((dpy == NULL) || (xrootwin == None) || (gc == None))
	    return;

	/* Get the root window as the drawable to draw to */
	d = (Drawable)xrootwin;

	/* Get new rectangle bits geometry values */
	x = MIN(grab_data->x[0], grab_data->x[1]);
	y = MIN(grab_data->y[0], grab_data->y[1]);
	width = ABSOLUTE_VALUE(grab_data->x[1] - grab_data->x[0]);
	height = ABSOLUTE_VALUE(grab_data->y[1] - grab_data->y[0]);

	/* Draw over last rectangle?
	 *
	 * This is to effectivly reset the bits drawn by a prior call
	 * to this function
	 */
	if(grab_data->have_last_rect)
	    XDrawRectangle(
		dpy, d, gc,
		grab_data->last_rect_x,
		grab_data->last_rect_y,
		grab_data->last_rect_width,
		grab_data->last_rect_height
	    );

	/* Draw the new rectangle bits and the new rectangle has
	 * valid size?
	 */
	if(draw_new && (width > 0) && (height > 0))
	{
	    XDrawRectangle(dpy, d, gc, x, y, width, height);

	    /* Since the new rectangle bits have been drawn, its geometry
	     * values should be recorded so that the next call to this
	     * function resets the bits just drawn
	     */
	    grab_data->have_last_rect = True;
	    grab_data->last_rect_x = x;
	    grab_data->last_rect_y = y;
	    grab_data->last_rect_width = width;
	    grab_data->last_rect_height = height;
	}
	else
	{
	    /* Do not draw new rectangle bits */

	    /* Need to mark that there would be no prior rectangle
	     * drawn for the next call to this function
	     */
	    grab_data->have_last_rect = False;
	}
}

/*
 *	Processes a window grab procedure, the pointer will be warped
 *	(moved) to the coordinates specified in grab_data structure and
 *	then the pointer will be queried to see which Window the
 *	pointer is over. That Window (or next child if include_decorations
 *	is FALSE) will have its geometry fetched and a rectangular area
 *	will be grabbed from the root window that matches the geometry
 *	of the Window in question.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint grab_X_window(
	imgview_struct *iv, grab_data_struct *grab_data,
	gint *width, gint *height,
	gboolean include_decorations,
	gboolean resize_on_load
)
{
	int x, y;
	Display *dpy;
	Window xrootwin, xwin;
	Window xrootrtn;		/* Dummy root for fetches */
	int x_return, y_return, xr2, yr2;
	unsigned int	width_return, height_return,
			border_width_return,
			depth_return;
	unsigned int mask;
	gint status;
	gint cx, cy, cwidth, cheight, rwidth, rheight;

	if((iv == NULL) || (grab_data == NULL))
	    return(-1);

	dpy = grab_data->xdisplay;
	xrootwin = grab_data->xrootwin;
	if((dpy == NULL) || (xrootwin == None))
	    return(-1);

	/* Make sure output has been flushed before grabbing, this
	 * is to ensure we don't have any messy marks that we drew
	 * in the image that we want to grab
	 */
	XFlush(dpy);

	/* Grab X server so things don't change while getting 
	 * window geometry
	 */
	XGrabServer(dpy);


	/* Get grab window coordinates relative to root window */
	x = (int)grab_data->x[0];
	y = (int)grab_data->y[0];

	/* Find child of root window that the given coordinates lie
	 * under
	 */
	XWarpPointer(dpy, None, xrootwin, 0, 0, 0, 0, x, y);
	if(!XQueryPointer(
	    dpy, xrootwin,
	    &xrootrtn, &xwin, &x_return, &y_return, &xr2, &yr2, &mask
	))
	{
	    last_screenshot_error =
"XQueryPointer() failed to find the clicked on Window";
	    XUngrabServer(dpy);
	    return(-1);
	}

	/* If coordinates do not lie within a child window then set
	 * the window to grab as the root window itself
	 */
	if(xwin == None)
	    xwin = xrootwin;


	/* At this point we have the X id of the root window and the
	 * immediate child or the root window that the pointer was over
	 *
	 * We need to handle two possible cases, if the root and child
	 * window are the same, then we grab the root window. Otherwise
	 * we grab the portion of the root window that the child window is
	 * displayed at
	 *
	 * In either case the values for variables x, y, rwidth, rheight,
	 * cwidth, and cheight will be set
	 */
	if(xwin == xrootwin)
	{
	    /* Grabbing the root window, get size of root window */
	    XGetGeometry(
		dpy, xwin, &xrootrtn,
		&x_return, &y_return, &width_return, &height_return,
		&border_width_return, &depth_return
	    );
	    cx = (gint)x_return;
	    cy = (gint)y_return;
	    rwidth = cwidth = (gint)width_return;
	    rheight = cheight = (gint)height_return;
	}
	else
	{
	    /* Grabbing a child window, there are two posibilites now.
	     * The first is that we just grab the child window without
	     * the WM decorations. The second is that we grab the
	     * entire child window and its WM decorations if any
	     */

	    /* First get size of root window */
	    XGetGeometry(
		dpy, xrootwin, &xrootrtn,
		&x_return, &y_return, &width_return, &height_return,
		&border_width_return, &depth_return
	    );
	    rwidth = (gint)width_return;
	    rheight = (gint)height_return;

	    /* Now work on getting the clicked on child window */
	    if(include_decorations)
	    {
		/* Grab child window with its WM decorations or the
		 * child window has no WM decorations
		 */
		XGetGeometry(
		    dpy, xwin, &xrootrtn,
		    &x_return, &y_return, &width_return, &height_return,
		    &border_width_return, &depth_return
		);
		cx = (gint)x_return;
		cy = (gint)y_return;
		cwidth = (gint)width_return;
		cheight = (gint)height_return;
	    }
	    else
	    {
		/* Do not grab WM decorations and this window is
		 * suspected to have decorations
		 */
		Window tw;

		/* Get position of child window */
		XGetGeometry(
		    dpy, xwin, &xrootrtn,
		    &x_return, &y_return, &width_return, &height_return,
		    &border_width_return, &depth_return
		);
		cx = (gint)x_return;
		cy = (gint)y_return;
		cwidth = (gint)width_return;
		cheight = (gint)height_return;

		/* Now get the next child if any */
		if(!XQueryPointer(
		    dpy, xwin,
		    &xrootrtn, &tw, &x_return, &y_return, &xr2, &yr2, &mask
		))
		{
		    last_screenshot_error =
"XQueryPointer() failed to find undecorated clicked on Window";
		    XUngrabServer(dpy);
		    return(-1);
		}
		if(tw != None)
		{
		    /* Got next child, so update xwin and its geometry */
		    xwin = tw;
		    XGetGeometry(
			dpy, xwin, &xrootrtn,
			&x_return, &y_return, &width_return, &height_return,
			&border_width_return, &depth_return
		    );
		    cx += (gint)x_return;
		    cy += (gint)y_return;
		    cwidth = (gint)width_return;
		    cheight = (gint)height_return;
		}
	    }

	}
	/* At this point we now have the size of the root window in
	 * rwidth and rheight plus the child window geometry relative
	 * to the root window at x, y, cwidth, and cheight
	 */
	/* Sanitize values of child window geometry */
	if((cx + cwidth) > rwidth)
	    cwidth = rwidth - cx;
	if(cx < 0)
	{
	    cwidth += cx;
	    cx = 0;
	}
	if((cy + cheight) > rheight)
	    cheight = rheight - cy;
	if(cy < 0)
	{
	    cheight += cy;
	    cy = 0;
	}
	/* Check if child window still has area */
	if((cwidth < 1) || (cheight < 1))
	{
	    last_screenshot_error =
		"Window has no area (width and/or height < 1)";
	    XUngrabServer(dpy);
	    return(-1);
	}

	/* We now have the rectangular area we want to grab, specified in
	 * cx, cy, cwidth, and cheight.
	 */

	/* Reset status to -1, indicating error. If we were able to get
	 * the image then we'll set it to 0.
	 */
	status = -1;

	/* Grab window and get image data */
	if(TRUE)
	{
	    /* Create a GdkWindow to reference the xroot_win */
	    gint bpl;
	    guint8 *data_rgba;
	    GdkRectangle rect;
	    GdkWindowPrivate *private = (GdkWindowPrivate *)g_malloc0(
		sizeof(GdkWindowPrivate)
	    );
	    GdkWindow *window = (GdkWindow *)private;
	    private->parent = NULL;
	    private->xwindow = xrootwin;
	    private->xdisplay = dpy;
	    private->x = 0;
	    private->y = 0;
	    private->width = rwidth;
	    private->height = rheight;
	    private->mapped = 1;

	    /* Set grab area */
	    rect.x = cx;
	    rect.y = cy;
	    rect.width = cwidth;
	    rect.height = cheight;

	    /* Grab and get RGBA data */
	    data_rgba = gdk_get_rgba_image(
		window, &rect,
		&cwidth, &cheight, &bpl
	    );
	    if(data_rgba != NULL)
	    {
		GtkWidget *toplevel, *w;
		imgview_image_struct *img = ImgViewImageNew(
		    cwidth, cheight, 4, bpl
		);
		ImgViewImageAppendFrame(img, data_rgba, 0l);
		data_rgba = NULL;

		/* Get the ImgView's toplevel widget if it is a GtkWindow */
		if(ImgViewIsToplevelWindow(iv))
		    toplevel = ImgViewGetToplevelWidget(iv);
		else
		    toplevel = NULL;

		/* Get ImgView's view widget */
		w = (GtkWidget *)ImgViewGetViewWidget(iv);

		/* Resize the ImgView? */
		if((w != NULL) && resize_on_load)
		{
		    gtk_widget_set_usize(w, cwidth, cheight);
		    gtk_widget_queue_resize(w);
		    if(toplevel != NULL)
			gtk_widget_queue_resize(toplevel);
		}

		/* Set new image to the ImgView
		 *
		 * Note that the image passed to this function should not
		 * be referenced again after this call
		 */
		ImgViewSetImage(iv, img);

		/* Mark success in grabbing image */
		status = 0;
	    }
	    else
	    {
		last_screenshot_error = "Unable to grab an X Drawable";
	    }
	}

	/* Ungrab X server */
	XUngrabServer(dpy);

	/* Update size return */
	if(width != NULL)
	    *width = cwidth;
	if(height != NULL)
	    *height = cheight;

	return(status);
}

/*
 *	Processes a rectangular grab procedure, the geometry of the
 *	rectangle will be obtained from the given grab_data structure.
 *	This geometry will then be used to get a rectangular image from
 *	the root window.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint grab_X_rectangular(
	imgview_struct *iv, grab_data_struct *grab_data,
	gint *width, gint *height,
	gboolean include_decorations,
	gboolean resize_on_load
)
{
	Display *dpy;
	Window	xrootwin,
		xrootrtn;		/* Dummy root for fetches */
	int x_return, y_return;
	unsigned int    width_return, height_return,
			border_width_return,
			depth_return;
	gint status;
	gint cx, cy, cwidth, cheight, rwidth, rheight;

	if((iv == NULL) || (grab_data == NULL))
	    return(-1);

	dpy = grab_data->xdisplay;
	xrootwin = grab_data->xrootwin;
	if((dpy == NULL) || (xrootwin == None))
	    return(-1);

	/* Make sure output has been flushed before grabbing, this
	 * is to ensure we don't have any messy marks that we drew
	 * in the image that we want to grab.
	 */
	XFlush(dpy);

	/* Grab X server so things don't change while getting
	 * window geometry.
	 */
	XGrabServer(dpy);


	/* Get size of root window */
	XGetGeometry(
	    dpy, xrootwin, &xrootrtn,
	    &x_return, &y_return, &width_return, &height_return,
	    &border_width_return, &depth_return
	);
	rwidth = (gint)width_return;
	rheight = (gint)height_return;

	/* Get size of child area from grab_data */
	cx = MIN(grab_data->x[0], grab_data->x[1]);
	cy = MIN(grab_data->y[0], grab_data->y[1]);
	cwidth = ABSOLUTE_VALUE(grab_data->x[1] - grab_data->x[0]);
	cheight = ABSOLUTE_VALUE(grab_data->y[1] - grab_data->y[0]);

	/* At this point we now have the size of the root window in
	 * rwidth and rheight plus the child window geometry relative
	 * to the root window at x, y, cwidth, and cheight.
	 */
	/* Sanitize values of child window geometry */
	if((cx + cwidth) > rwidth)
	    cwidth = rwidth - cx;
	if(cx < 0)
	{
	    cwidth += cx;
	    cx = 0;
	}
	if((cy + cheight) > rheight)
	    cheight = rheight - cy;
	if(cy < 0)
	{
	    cheight += cy;
	    cy = 0;
	}
	/* Check if child window still has area */
	if((cwidth < 1) || (cheight < 1))
	{
	    last_screenshot_error =
		"Grab rectangle has no area (width and/or height < 1)";
	    XUngrabServer(dpy);
	    return(-1);
	}

	/* We now have the rectangular area we want to grab, specified in
	 * cx, cy, cwidth, and cheight
	 */

	/* Reset status to -1, indicating error. If we were able to get
	 * the image then we'll set it to 0
	 */
	status = -1;

	/* Grab window and get image data */
	if(TRUE)
	{
	    /* Create a GdkWindow to reference the xroot_win */
	    gint bpl;
	    guint8 *data_rgba;
	    GdkRectangle rect;
	    GdkWindowPrivate *private = (GdkWindowPrivate *)g_malloc0(
		sizeof(GdkWindowPrivate)
	    );
	    GdkWindow *window = (GdkWindow *)private;
	    private->parent = NULL;
	    private->xwindow = xrootwin;
	    private->xdisplay = dpy;
	    private->x = 0;
	    private->y = 0;
	    private->width = rwidth;
	    private->height = rheight;
	    private->mapped = 1;

	    /* Set grab area */
	    rect.x = cx;
	    rect.y = cy;
	    rect.width = cwidth;
	    rect.height = cheight;

	    /* Grab and get RGBA data */
	    data_rgba = gdk_get_rgba_image(
		window, &rect,
		&cwidth, &cheight, &bpl
	    );
	    if(data_rgba != NULL)
	    {
		GtkWidget *toplevel, *w;
		imgview_image_struct *img = ImgViewImageNew(
		    cwidth, cheight, 4, bpl
		);
		ImgViewImageAppendFrame(img, data_rgba, 0l);

		/* Get the ImgView's toplevel widget if it is a GtkWindow */
		if(ImgViewIsToplevelWindow(iv))
		    toplevel = ImgViewGetToplevelWidget(iv);
		else
		    toplevel = NULL;

		/* Get ImgView's view widget */
		w = (GtkWidget *)ImgViewGetViewWidget(iv);

		/* Resize the ImgView? */
		if((w != NULL) && resize_on_load)
		{
		    gtk_widget_set_usize(w, cwidth, cheight);
		    gtk_widget_queue_resize(w);
		    if(toplevel != NULL)
			gtk_widget_queue_resize(toplevel);
		}

		/* Set new image to the ImgView
		 *
		 * Note that the image passed to this function should not
		 * be referenced again after this call
		 */
		ImgViewSetImage(iv, img);

		/* Mark success in grabbing image */
		status = 0;
	    }
	    else
	    {
		last_screenshot_error = "Unable to grab an X Drawable";
	    }
	}

	/* Ungrab X server */
	XUngrabServer(dpy);

	/* Update size return */
	if(width != NULL)
	    *width = cwidth;
	if(height != NULL)
	    *height = cheight;

	return(status);
}
#endif	/* HAVE_X */


/*
 *	Grab window destroyed notify.
 *
 *	User to abort the grab when the grab window has been
 *	destroyed.
 */
static gint grab_window_destroy_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	grab_data_struct *grab_data = GRAB_DATA(data);
	if((widget == NULL) || (grab_data == NULL))
	    return(FALSE);

	/* Was a button pressed? */
	if(grab_data->grab_button_number > -1)
	{
	    grab_data->grab_button_number = -1;


	    /* If grabbing a rectangular area then update
	     * the rectangle's end coordinates and redraw the
	     * rectangle outline
	     */
	    if(grab_data->type == GRAB_RECTANGULAR)
	    {
		grab_data->x[1] = grab_data->x[0];
		grab_data->y[1] = grab_data->y[0];
#if defined(HAVE_X)
		draw_X_grab_rectangle(grab_data, False);
#endif
	    }
	}

	/* Mark that the grab was aborted due to the grab window
	 * being destroyed
	 */
	grab_data->type = GRAB_ABORT;

	/* Break out of the GTK main loop */
	if(grab_data->gtk_main_level > 0)
	{
	    grab_data->gtk_main_level--;
	    gtk_main_quit();
	}

	return(FALSE);
}

/*
 *	Grab delay timeout callback.
 *
 *	This is called when the delay time expires in
 *	iv_screenshot_start() and to break out of the GTK+ main
 *	loop set just after the setting of this callback.
 */
static gint grab_delay_tocb(gpointer data)
{
	gint *gtk_main_level = (gint *)data;

	if(*gtk_main_level > 0)
	{
	    *gtk_main_level = (*gtk_main_level) - 1;
	    gtk_main_quit();
	}

	return(FALSE);		/* Don't call this timeout again */
}

/*
 *	Grab "button_press_event" signal callback.
 */
static gint grab_button_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	grab_data_struct *grab_data = GRAB_DATA(data);
	if((button == NULL) || (grab_data == NULL))
	    return(FALSE);

	/* Do not process event if grabbing is not schedualed to
	 * be active yet
	 */
	if(!grab_data->active)
	    return(TRUE);

	/* Button press? */
	if(button->type == GDK_BUTTON_PRESS)
	{
	    /* Get number of the button that was pressed */
	    const gint button_num = button->button;

	    /* Set grab_type by which button number was pressed */
	    switch(button_num)
	    {
	      case GDK_BUTTON1:
		grab_data->grab_button_number = button_num;
		grab_data->type = GRAB_WINDOW;
		grab_data->x[0] = (gint)button->x;
		grab_data->y[0] = (gint)button->y;
		break;

	      case GDK_BUTTON2:
		grab_data->grab_button_number = button_num;
		grab_data->type = GRAB_RECTANGULAR;
		grab_data->x[0] = (gint)button->x;
		grab_data->y[0] = (gint)button->y;
		grab_data->x[1] = grab_data->x[0];
		grab_data->y[1] = grab_data->y[0];
		break;

	      case GDK_BUTTON3:
		grab_data->grab_button_number = button_num;
		grab_data->type = GRAB_ABORT;
		break;

	      case GDK_BUTTON4:
	      case GDK_BUTTON5:
		break;

	      default:	/* All other button numbers, assume abort */
		grab_data->grab_button_number = button_num;
		grab_data->type = GRAB_ABORT;
		break;
	    }
	}
	/* Assume GDK_BUTTON_RELEASE */
	else
	{
	    /* Was a valid grab button pressed? */
	    if(grab_data->grab_button_number > -1)
	    {
		grab_data->grab_button_number = -1;

		/* If grabbing a rectangular area then update
		 * the rectangle's end coordinates and redraw the
		 * rectangle outline
		 */
		if(grab_data->type == GRAB_RECTANGULAR)
		{
		    grab_data->x[1] = (gint)button->x;
		    grab_data->y[1] = (gint)button->y;
#if defined(HAVE_X)
		    draw_X_grab_rectangle(grab_data, False);
#endif
		}

		/* Break out of the GTK main loop */
		if(grab_data->gtk_main_level > 0)
		{
		    grab_data->gtk_main_level--;
		    gtk_main_quit();
		}
	    }
	}

	return(TRUE);
}

/*
 *	Grab "motion_notify_event" signal callback.
 */
static gint grab_motion_cb(
	GtkWidget *widget, GdkEventMotion *motion, gpointer data
)
{
	grab_data_struct *grab_data = GRAB_DATA(data);
	if((motion == NULL) || (grab_data == NULL))
	    return(FALSE);

	/* Do not process event if grabbing is not schedualed to
	 * be active yet
	 */
	if(!grab_data->active)
	    return(TRUE);

	/* Button pressed? */
	if(grab_data->grab_button_number > -1)
	{
	    switch(grab_data->type)
	    {
	      case GRAB_UNDETERMINED:
		/* Pending */
		break;

	      case GRAB_WINDOW:
		/* Do nothing */
		break;

	      case GRAB_RECTANGULAR:
		/* Update the grab coordinates */
		grab_data->x[1] = (gint)motion->x;
		grab_data->y[1] = (gint)motion->y;
#if defined(HAVE_X)
		draw_X_grab_rectangle(grab_data, True);
#endif
		break;

	      case GRAB_ABORT:
		/* Do nothing */
		break;
	    }
	}

	return(TRUE);
}


/*
 *	Returns the pointer to a string describing the last error or
 *	NULL if there was no error.
 */
const gchar *iv_screenshot_get_error(void)
{
	return(last_screenshot_error);
}

/*
 *	Start a grab procedure.
 *
 *	The iv specifies the ImgView that started this grab.
 *
 *	The pointer will be grabbed to query which window or rectangular
 *	area to grab. On success, the given ImgView will have its
 *	image data cleared and then replaced with the grabbed image.
 *
 *	The width_rtn and height_rtn specifies the return values for the
 *	grabbed region.
 *
 *	The delay specifies the delay in milliseconds to wait before
 *	starting the grab.
 *
 *	If include_decorations is TRUE then WM decorations will be
 *	included in the user clicked on a toplevel window.
 *
 *	If resize_on_load is TRUE then the iv will be resized after
 *	the grab.
 *
 *	The grab_window specifies the GtkWindow to refer grab values
 *	to, it must be a valid and mapped GtkWindow, and it will be
 *	used to grab the pointer and report relative events to.
 *
 *	The grab_cursor specifies the cursor to display during the
 *	grab.
 *
 *	Return values:
 *
 *		0	Success.
 *		-1	General error (see iv_screenshot_get_error()).
 *		-2	User did not select grab window/area properly.
 *		-3	System error.
 *		-4	User aborted.
 *
 *	When 0 is returned, the exact error message can be retrieved by
 *	calling iv_screenshot_get_error().
 */
gint iv_screenshot_start(
	imgview_struct *iv,
	gint *width_rtn, gint *height_rtn,
	const gulong delay,
	const gboolean include_decorations,
	const gboolean resize_on_load,
	GtkWidget *grab_window,
	GdkCursor *grab_cursor
)
{
	gint status, gtk_main_level;
	guint grab_signal_id[4];
	GdkWindow	*window,
			*root_win;
	GtkWidget *w;
	grab_data_struct *grab_data;
#if defined(HAVE_X)
	int scr_num;
	Display *dpy;
	Window xrootwin;
	XGCValues gcv;
	GC gc_inverse;
#endif	/* HAVE_X */


	/* Reset the last grab error string */
	last_screenshot_error = NULL;

	/* Reset the return size */
	if(width_rtn != NULL)
	    *width_rtn = 0;
	if(height_rtn != NULL)
	    *height_rtn = 0;

	if(iv == NULL)
	{
	    last_screenshot_error = "ImgView is NULL";
	    return(-1);
	}
	if((grab_window != NULL) ?
	    GTK_WIDGET_NO_WINDOW(grab_window) : TRUE
	)
	{
	    last_screenshot_error = "Grab window is NULL";
	    return(-1);
	}


	/* Wait for the specified delay */
	gtk_main_level = 0;
	if(delay > 0l)
	{
	    gtk_main_level++;
	    gtk_timeout_add(
		delay,			/* Interval In Milliseconds */
		grab_delay_tocb,	/* Timeout Callback */
		&gtk_main_level		/* Data */
	    );
	    gtk_main();
	}
	while(gtk_main_level > 0)
	{
	    gtk_main_level--;
	    gtk_main_quit();
	}


	/* Get the root GdkWindow */
	root_win = GDK_ROOT_PARENT();
	if(root_win == NULL)
	{
	    last_screenshot_error =
"GDK_ROOT_PARENT(): Unable to obtain root rindow";
	    return(-1);
	}

	/* Get the grab_window's GdkWindow */
	window = grab_window->window;
	if(window == NULL)
	{
	    last_screenshot_error = "Grab window's GdkWindow is NULL";
	    return(-1);
	}

	/* Get image viewer toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    w = ImgViewGetToplevelWidget(iv);
	else
	    w = NULL;

#if defined(HAVE_X)
	/* Get X resources */
	dpy = GDK_WINDOW_XDISPLAY(root_win);
	scr_num = DefaultScreen(dpy);
	xrootwin = GDK_WINDOW_XWINDOW(root_win);
	if((dpy == NULL) || (xrootwin == None))
	{
	    last_screenshot_error = "Unable obtain X resources";
	    return(-3);
	}

	/* Sync X to ensure most recent characteristics are being
	 * displayed just before grabbing
	 */
	XSync(dpy, False);
#endif	/* HAVE_X */

	/* Allocate grab callback data */
	grab_data = GRAB_DATA(g_malloc0(sizeof(grab_data_struct)));
	if(grab_data == NULL)
	{
	    last_screenshot_error = "Memory allocation error";
	    return(-3);
	}
	/* Reset grab data structure values */
	grab_data->type = GRAB_UNDETERMINED;
	grab_data->active = FALSE;
	grab_data->root_win = root_win;
	grab_data->gtk_main_level = 0;
	grab_data->grab_button_number = -1;
#if defined(HAVE_X)
	grab_data->xdisplay = dpy;
	grab_data->xscreen_num = scr_num;
	grab_data->xrootwin = xrootwin;
	/* Create GCs for grab related drawing */
	gcv.function = GXinvert;
	gcv.plane_mask = AllPlanes;
	gcv.foreground = WhitePixel(dpy, scr_num);
	gcv.background = BlackPixel(dpy, scr_num);
	gcv.line_width = 1;
	gcv.line_style = LineSolid;
	gcv.cap_style = CapNotLast;
	gcv.join_style = JoinMiter;
	gcv.fill_style = FillSolid;
	gcv.fill_rule = EvenOddRule;
	gcv.subwindow_mode = IncludeInferiors;
	grab_data->gc_inverse = gc_inverse = XCreateGC(
	    dpy, xrootwin,
	    GCFunction | GCForeground | GCBackground | GCPlaneMask |
	    GCLineWidth | GCLineStyle | GCCapStyle | GCJoinStyle |
	    GCFillStyle | GCFillRule | GCSubwindowMode,
	    &gcv
	);
	grab_data->have_last_rect = False;
#endif	/* HAVE_X */

	/* Install signal handlers for the grab GtkWindow */
	grab_signal_id[0] = gtk_signal_connect(
	    GTK_OBJECT(grab_window), "destroy_event",
	    GTK_SIGNAL_FUNC(grab_window_destroy_cb), grab_data
	);
	grab_signal_id[1] = gtk_signal_connect(
	    GTK_OBJECT(grab_window), "button_press_event",
	    GTK_SIGNAL_FUNC(grab_button_cb), grab_data
	);
	grab_signal_id[2] = gtk_signal_connect(
	    GTK_OBJECT(grab_window), "button_release_event",
	    GTK_SIGNAL_FUNC(grab_button_cb), grab_data
	);
	grab_signal_id[3] = gtk_signal_connect(
	    GTK_OBJECT(grab_window), "motion_notify_event",
	    GTK_SIGNAL_FUNC(grab_motion_cb), grab_data
	);


#define DO_UNGRAB_POINTER	{	\
 if(gdk_pointer_is_grabbed()) {		\
  gdk_flush();				\
  gdk_pointer_ungrab(GDK_CURRENT_TIME);	\
  gdk_flush();				\
 }					\
}

#if defined(HAVE_X)
#define DO_FREE_GRAB_CBDATA	{	\
 if(grab_data != NULL) {		\
  /* Delete the X resources */		\
  if(grab_data->gc_inverse != None) {	\
   XFreeGC(dpy, grab_data->gc_inverse);	\
   grab_data->gc_inverse = None;	\
  }					\
					\
  g_free(grab_data);			\
  grab_data = NULL;			\
 }					\
}
#else
#define DO_FREE_GRAB_CBDATA	{	\
 if(grab_data != NULL) {		\
					\
  g_free(grab_data);			\
  grab_data = NULL;			\
 }					\
}
#endif

/* Ungrab pointer, disconnect signal handlers, and delete the grab
 * callback data
 */
#define GRAB_CLEANUP	{			\
 gint i, m;					\
						\
 /* Ungrab pointer as needed */			\
 DO_UNGRAB_POINTER				\
						\
 /* Disconnect our tempory signal handlers */	\
 m = sizeof(grab_signal_id) / sizeof(guint);	\
 for(i = 0; i < m; i++) {			\
  gtk_signal_disconnect(			\
   GTK_OBJECT(grab_window), grab_signal_id[i]	\
  );						\
  grab_signal_id[i] = (guint)-1;		\
 }						\
						\
 /* Break out of any remaining main loops */	\
 while(grab_data->gtk_main_level > 0) {		\
  grab_data->gtk_main_level--;			\
  gtk_main_quit();				\
 }						\
						\
						\
 /* Delete the grab callback data */		\
 DO_FREE_GRAB_CBDATA				\
}

	/* Now mark grab as active, this will allow grab related events 
	 * to be processed in the signal callbacks we set above
	 */
	grab_data->active = TRUE;

	/* Begin grabbing the pointer so grab related events get reported
	 * to and with respect to the given grab window
	 */
	gdk_flush();	/* Must flush GDK output before grabbing pointer */
	status = gdk_pointer_grab(
	    window,
	    FALSE,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK,
	    /* root_win */ NULL,
	    grab_cursor,
	    GDK_CURRENT_TIME
	);
	gdk_flush();	/* Must flush GDK output after grabbing pointer */
	/* Was there an error while attempting to grab the pointer? */
	switch(status)
	{
#ifdef GrabInvalidTime
	  case GrabInvalidTime:
	    last_screenshot_error =
		"Invalid time specified for grabbing pointer";
	    GRAB_CLEANUP
	    return(-3);
	    break;
#endif
#ifdef GrabNotViewable
	  case GrabNotViewable:
	    last_screenshot_error =
		"Grab window is not viewable to grab pointer";
	    GRAB_CLEANUP
	    return(-3);
	    break;
#endif
#ifdef GrabFrozen
	  case GrabFrozen:
	    last_screenshot_error =
		"Grab is frozen and unable to grab pointer";
	    GRAB_CLEANUP
	    return(-3);
	    break;
#endif
#ifdef AlreadyGrabbed
	  case AlreadyGrabbed:
	    last_screenshot_error =
"Another window is currently grabbing the pointer, please\n\
wait and try again later";
	    GRAB_CLEANUP
	    return(-3);
	    break;
#endif
#ifdef GrabSuccess
	  case GrabSuccess:
	    break;
#endif
	}

	/* Enter our GTK+ main loop */
	grab_data->gtk_main_level++;
	gtk_main();

	/* Mark grab as expired (no longer active) so no more grab
	 * related events will be processed on our signal callbacks
	 */
	grab_data->active = FALSE;

	/* Ungrab the pointer, we do not need it anymore */
	DO_UNGRAB_POINTER


	/* Handle by grab type, assume error first */
	last_screenshot_error = "Specified grab operation no supported";
	status = -1;
	switch(grab_data->type)
	{
	  case GRAB_UNDETERMINED:
	    break;

	  case GRAB_WINDOW:
#if defined(HAVE_X)
	    status = grab_X_window(
		iv, grab_data, width_rtn, height_rtn,
		include_decorations, resize_on_load
	    );
#endif
	    break;

	  case GRAB_RECTANGULAR:
#if defined(HAVE_X)
	    status = grab_X_rectangular(
		iv, grab_data, width_rtn, height_rtn,
		include_decorations, resize_on_load
	    );
#endif
	    break;

	  case GRAB_ABORT:
	    last_screenshot_error = "User aborted";
	    GRAB_CLEANUP
	    return(-4);
	    break;
	}

	/* Ungrab pointer and disconnect signal handlers */
	GRAB_CLEANUP

	return(0);

#undef GRAB_CLEANUP
#undef DO_UNGRAB_POINTER   
#undef DO_FREE_GRAB_CBDATA
}
