#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "fsd.h"
#include "csd.h"
#include "imgview.h"

#include "textdlg.h"
#include "config.h"

#include "images/icon_li_16x16.xpm"
#include "images/icon_fonts_20x20.xpm"
#include "images/icon_fonts_32x32.xpm"
#include "images/icon_fonts_48x48.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


typedef struct _Dialog			Dialog;
#define DIALOG(p)			((Dialog *)(p))


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void add_cb(GtkWidget *widget, gpointer data);
static void cancel_cb(GtkWidget *widget, gpointer data);
static void select_font_cb(GtkWidget *widget, gpointer data);
static void select_color_cb(GtkWidget *widget, gpointer data);
static void outline_toggled_cb(GtkWidget *widget, gpointer data);
static void select_outline_color_cb(GtkWidget *widget, gpointer data);
static void opacity_scale_cb(GtkAdjustment *adj, gpointer data);
static void opacity_entry_cb(GtkWidget *widget, gpointer data);
static gint blink_text_cur_tocb(gpointer data);

static gint imgview_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

static void draw_text_cursor_on_imgview(Dialog *d);
static void draw_text_on_imgview(Dialog *d);
static void queue_draw_text_on_imgview(Dialog *d);

static void set_text_window_position(
	Dialog *d,
	const gint x, const gint y
);

static void render_text_on_imgview_iterate(
	const guint8 *src,		/* RGB image data */
	guint8 *tar,			/* RGBA image data */
	const gint width, const gint height,
	const gint src_bpp, const gint src_bpl,
	const gint tar_bpp, const gint tar_bpl,
	const gfloat src_opacity, const gfloat tar_opacity,
	const guint8 r, const guint8 g, const guint8 b
);
static void render_text_on_imgview(
	imgview_struct *iv,
	const gchar *text,
	const gint x, const gint y,
	GdkFont *font,
	GdkColor *color,
	const gboolean outline,
	GdkColor *outline_color,
	const gfloat opacity,
	const gboolean all_frames
);

static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
);

gboolean iv_add_text(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gchar **font_name,
	GdkColor *color,
	gboolean *outline,
	GdkColor *outline_color,
	gfloat *opacity,
	gboolean *all_frames
);


/*
 *	Text Dialog:
 */
struct _Dialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
        gboolean        has_changes, 
                        got_response;
	imgview_struct	*iv;
	GtkWidget	*iv_drawing_area;

	GdkColormap	*colormap;
	GdkGC		*gc;
	GdkCursor	*text_cur;

	GtkWidget	*font_prompt,
			*color_btn,
			*outline_check,
			*outline_color_btn,
			*opacity_scale,
			*opacity_entry,
			*all_frames_check,
			*add_btn,
			*cancel_btn,
			*close_btn;

	guint		iv_expose_sigid,
			iv_key_press_sigid,
			iv_key_release_sigid,
			iv_button_press_sigid,
			iv_button_release_sigid,
			iv_enter_notify_sigid;

	/* Position of our text cursor on the ImgView */
	gint		text_start_wx,	/* In window coordinates */
			text_start_wy,
			text_dx,	/* In image coordinates */
			text_dy;

	gchar		*text;		/* Text value */
	gint		text_cur_pos;	/* Cursor position within the
					 * text */
	gint		text_cur_wx,	/* Cursor position on the window */
			text_cur_wy;
	guint		draw_text_cur_toid;	/* Text cursor draw timeout
						 * callback id for blinking */
	gboolean	draw_text_cur_blink_state;

	gchar		*font_name;
	GdkFont		*font;		/* Current selected font */
	GdkColor	color,
			outline_color;
	gfloat		opacity;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Resize Dialog toplevel GtkWindow "delete_event" signal
 *	callback.
 */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	d->got_response = FALSE;
	gtk_main_quit();
	return(TRUE);
}

/*
 *	Add callback.
 */
static void add_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Cancel callback.
 */
static void cancel_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}

/*
 *	Select font callback.
 */
static void select_font_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gchar *new_font_name;
	Dialog *d = DIALOG(data);
	if((d == NULL) || FSDIsQuery())
	    return;

	/* Query user for new font */
	FSDSetTransientFor(d->toplevel);
	response = FSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El Tipo De Letra",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Jeu De Caractres Privilgi",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Schriftart Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere Font",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer Lettertype",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Fonte",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Font",
	    "Velg", "Kanseler",
#else
	    "Select Font",
	    "Select", "Cancel",
#endif
	    d->font_name,
	    &new_font_name
	);
	FSDSetTransientFor(NULL);

	if(response)
	{
	    /* Set new font */
	    g_free(d->font_name);
	    d->font_name = STRDUP(new_font_name);

	    GDK_FONT_UNREF(d->font);
	    d->font = gdk_font_load(d->font_name);

	    FSDPromptSetFontName(
		d->font_prompt, d->font_name
	    );

	    queue_draw_text_on_imgview(d);
	}
}

/*
 *	Select color callback.
 */
static void select_color_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	csd_color_struct c_cur, *c_new;
	Dialog *d = DIALOG(data);
	if((d == NULL) || CSDIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	CSDColorButtonGetColor(d->color_btn, &c_cur);

	/* Query user for new color */
	CSDSetTransientFor(d->toplevel);
	response = CSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El Color",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Couleur Privilgie",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Farbe Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere Il Colore",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer De Kleur",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Cor",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Farge",
	    "Velg", "Kanseler",
#else
	    "Select Color",
	    "Select", "Cancel",
#endif
	    &c_cur,
	    &c_new,
	    NULL, NULL
	);
	CSDSetTransientFor(NULL);

	if(response)
	{
	    /* Set the new color */
	    if(c_new != NULL)
	    {
		GdkColor *c = &d->color;
		GDK_COLORMAP_FREE_COLOR(d->colormap, c);
		GDK_COLOR_SET_COEFF(c, c_new->r, c_new->g, c_new->b);
		GDK_COLORMAP_ALLOC_COLOR(d->colormap, c);

		CSDColorButtonSetColor(d->color_btn, c_new);

		queue_draw_text_on_imgview(d);
	    }
	}

	d->freeze_count--;
}

/*
 *	Outline toggled signal callback.
 */
static void outline_toggled_cb(GtkWidget *widget, gpointer data)
{
	gboolean toggled;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	toggled = GTK_TOGGLE_BUTTON_GET_ACTIVE(widget);
	gtk_widget_set_sensitive(
	    d->outline_color_btn, toggled
	);

	queue_draw_text_on_imgview(d);

	d->freeze_count--;
}

/*
 *	Select outline color callback.
 */
static void select_outline_color_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	csd_color_struct c_cur, *c_new;
	Dialog *d = DIALOG(data);
	if((d == NULL) || CSDIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	CSDColorButtonGetColor(d->outline_color_btn, &c_cur);

	/* Query user for new color */
	CSDSetTransientFor(d->toplevel);
	response = CSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El Color",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Couleur Privilgie",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Farbe Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere Il Colore",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer De Kleur",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Cor",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Farge",
	    "Velg", "Kanseler",
#else
	    "Select Outline Color",
	    "Select", "Cancel",
#endif
	    &c_cur,
	    &c_new,
	    NULL, NULL
	);
	CSDSetTransientFor(NULL);

	if(response)
	{
	    /* Set the new color */
	    if(c_new != NULL)
	    {
		GdkColor *c = &d->outline_color;
		GDK_COLORMAP_FREE_COLOR(d->colormap, c);
		GDK_COLOR_SET_COEFF(c, c_new->r, c_new->g, c_new->b);
		GDK_COLORMAP_ALLOC_COLOR(d->colormap, c);

		CSDColorButtonSetColor(d->outline_color_btn, c_new);

		queue_draw_text_on_imgview(d);
	    }
	}

	d->freeze_count--;
}

/*
 *	Opacity GtkAdjustment "value_changed" signal callback.
 */
static void opacity_scale_cb(GtkAdjustment *adj, gpointer data)
{
	gchar *s;
	Dialog *d = DIALOG(data);
	if((adj == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	d->opacity = adj->value;
	s = g_strdup_printf("%i", (gint)(d->opacity * 100.0f));
	gtk_entry_set_text(GTK_ENTRY(d->opacity_entry), s);
	g_free(s);

	queue_draw_text_on_imgview(d);

	d->freeze_count--;
}

/*
 *	Opacity GtkEntry "changed" signal callback.
 */
static void opacity_entry_cb(GtkWidget *widget, gpointer data)
{
	GtkAdjustment *adj;
	GtkEntry *entry = (GtkEntry *)widget;
	Dialog *d = DIALOG(data);
	if((entry == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	d->opacity = (gfloat)ATOF(gtk_entry_get_text(entry)) / 100.0f;
	adj = gtk_range_get_adjustment(GTK_RANGE(d->opacity_scale));
	if(adj != NULL)
	    gtk_adjustment_set_value(adj, d->opacity);
	    
	queue_draw_text_on_imgview(d);

	d->freeze_count--;
}

/*
 *	Blink/draw text cursor timeout callback.
 */
static gint blink_text_cur_tocb(gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	draw_text_cursor_on_imgview(d);

	return(TRUE);
}


/*
 *	ImgView event signal callback.
 */
static gint imgview_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data  
)
{
	gint status = FALSE;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventCrossing *crossing;
	GtkWidget *w;
	imgview_struct *iv;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	iv = d->iv;
	w = d->iv_drawing_area;
	if((iv == NULL) || (w == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_EXPOSE:
	    draw_text_on_imgview(d);
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	    key = (GdkEventKey *)event;
	    if(d->text_cur_pos > -1)
	    {
		guint keyval = key->keyval;
		/* Control? */
		if(keyval & 0xffffff00)
		{
		    if(keyval == GDK_BackSpace)
		    {
			d->text = strdelchr(
			    d->text, d->text_cur_pos - 1
			);
			if(d->text_cur_pos > 0)
			     d->text_cur_pos--;
		    }
		    else if(keyval == GDK_Delete)
		    {
			d->text = strdelchr(
			    d->text, d->text_cur_pos
			);
		    }
		    else if((keyval == GDK_Return) ||
			    (keyval == GDK_KP_Enter)
		    )
		    {
			d->text = strinschr(
			    d->text, d->text_cur_pos, '\n'
			);
			d->text_cur_pos++;
		    }
		    else if((keyval == GDK_Left) ||
			    (keyval == GDK_KP_Left)
		    )
		    {
			d->text_cur_pos--;
			if(d->text_cur_pos < 0)
			    d->text_cur_pos = 0;
		    }
		    else if((keyval == GDK_Right) ||
			    (keyval == GDK_KP_Right)
		    )
		    {
			const gint len = STRLEN(d->text);
			d->text_cur_pos++;
			if(d->text_cur_pos > len)
			    d->text_cur_pos = len;
		    }
		    else if((keyval == GDK_Up) ||
			    (keyval == GDK_KP_Up)
		    )
		    {
			if(d->text != NULL)
			{
			    const gchar	*s = d->text, *s_cur,
					*line_cur = s,
					*line_prev = s;
			    const gint len = STRLEN(s);
			    gint	i = CLIP(d->text_cur_pos, 0, len),
					n;

			    s_cur = (const gchar *)(s + i);
			    for(; *s != '\0'; s++)
			    {
				if(s == s_cur)
				    break;

				if(*s == '\n')
				{
				    line_prev = line_cur;
				    line_cur = s + 1;
				}
			    }
			    s = d->text;

			    n = (gint)(s_cur - line_cur);
			    s_cur = line_prev;
			    for(i = 0; i < n; i++)
			    {
				if((*s_cur == '\n') || (*s_cur == '\0'))
				    break;
				s_cur++;
			    }

			    d->text_cur_pos = (line_cur == line_prev) ?
				0 : (gint)(s_cur - s);
			}
		    }
		    else if((keyval == GDK_Down) ||
			    (keyval == GDK_KP_Down)
		    )
		    {
			if(d->text != NULL)
			{
			    const gchar *s = d->text, *s_cur,
					*line_cur = s,
					*line_next;
			    const gint len = STRLEN(s);
			    gint        i = CLIP(d->text_cur_pos, 0, len),
					n;

			    s_cur = (const gchar *)(s + i);
			    for(; *s != '\0'; s++)
			    {
				if(s == s_cur)
				    break;

				if(*s == '\n')
				    line_cur = s + 1;
			    }
			    line_next = line_cur;
			    for(s = s_cur; *s != '\0'; s++)
			    {
				if(*s == '\n')
				{
				    line_next = s + 1;
				    break;
				}
			    }
			    s = d->text;

			    n = (gint)(s_cur - line_cur);
			    s_cur = line_next;
			    for(i = 0; i < n; i++)
			    {
				if((*s_cur == '\n') || (*s_cur == '\0'))
				    break;
				s_cur++;
			    }

			    d->text_cur_pos = (line_next == line_cur) ?
				len : (gint)(s_cur - s);
			}
		    }
		    else if(keyval == GDK_Home)
		    {
			d->text_cur_pos = 0;
		    }
		    else if(keyval == GDK_End)
		    {
			d->text_cur_pos = STRLEN(d->text);
		    }
		}
		else
		{
		    d->text = strinschr(
		        d->text, d->text_cur_pos, (char)keyval
		    );
		    d->text_cur_pos++;
	        }
	    }

	    /* Set the Add button to sensitive only after text has
	     * been typed in
	     */
	    if((d->text != NULL) ? (*d->text != '\0') : FALSE)
	    {
		if(!d->has_changes)
		{
		    gtk_widget_set_sensitive(d->add_btn, TRUE);
		    gtk_widget_show(d->cancel_btn);
		    gtk_widget_hide(d->close_btn);
		    d->has_changes = TRUE;
		}
	    }

	    queue_draw_text_on_imgview(d);
	    if(w->window != NULL)
		gdk_window_set_cursor(w->window, d->text_cur);

	    gtk_signal_emit_stop_by_name(
		GTK_OBJECT(widget), "key_press_event"
	    );
	    status = TRUE;
	    break;

	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    gtk_signal_emit_stop_by_name(
		GTK_OBJECT(widget), "key_release_event"
	    );
	    status = TRUE;
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;

	    if(GTK_WIDGET_CAN_FOCUS(widget) && !GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);

	    /* Set the text coordinates */
	    set_text_window_position(
		d,
		(gint)button->x, (gint)button->y
	    );

	    if(w->window != NULL)
		gdk_window_set_cursor(w->window, d->text_cur);
	    gtk_signal_emit_stop_by_name(  
		GTK_OBJECT(widget), "button_press_event"
	    );
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    if(w->window != NULL)
		gdk_window_set_cursor(w->window, d->text_cur);
	    gtk_signal_emit_stop_by_name(  
		GTK_OBJECT(widget), "button_release_event"
	    );
	    status = TRUE;
	    break;

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    if(w->window != NULL)
		gdk_window_set_cursor(w->window, d->text_cur);
	    gtk_signal_emit_stop_by_name(  
		GTK_OBJECT(widget), "enter_notify_event"
	    );
	    status = TRUE;
	    break;

	}

	return(status);
}


/*
 *	Draws the text cursor on the ImgView.
 *
 *	The d->text_cur_wx and d->text_cur_wy will be used as
 *	the position of the text cursor.
 *
 *	The d->draw_text_cur_blink_state will be updated by this
 *	call to reflect its last blink state.
 */
static void draw_text_cursor_on_imgview(Dialog *d)
{
	const gint	cur_x = d->text_cur_wx,
			cur_y = d->text_cur_wy;
	GtkWidget *w = d->iv_drawing_area;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	GdkFont *font = d->font;
	GdkGC *gc = d->gc;
	imgview_struct *iv = d->iv;

	if(iv == NULL)
	    return;

	/* Draw the text cursor if its coordinates are set */
	if((d->text_cur_pos > -1) && (font != NULL))
	{
	    const gint	font_height = font->ascent + font->descent,
			x = cur_x,
			y = cur_y - (font_height / 2),
			y2 = y + font_height;
	    gdk_gc_set_function(gc, GDK_INVERT);
	    gdk_draw_line(
	        drawable, gc,
		x, y,
		x, y2
	    );
	}

	/* Update the blink state */
	d->draw_text_cur_blink_state = !d->draw_text_cur_blink_state;
}

/*
 *	Draws the text on the ImgView.
 *
 *	The text cursor's position will be calculated and set in
 *	d->text_cur_wx and d->text_cur_wy by this call.
 *
 *	The text cursor will always be drawn on this call and
 *	d->draw_text_cur_blink_state will always be set to TRUE
 *	by this call.
 */
static void draw_text_on_imgview(Dialog *d)
{
	gint	cur_x = d->text_start_wx,
		cur_y = d->text_start_wy;
	GtkWidget *w = d->iv_drawing_area;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	GdkFont *font = d->font;
	GdkGC *gc = d->gc;
	imgview_struct *iv = d->iv;

	if(iv == NULL)
	    return;

	/* Need to flush any prior drawing commands issued by the
	 * imgview segment of drawing
	 */
	gdk_flush();

	/* Draw text */
	if((font != NULL) && (d->text != NULL))
	{
	    const gboolean outline = GTK_TOGGLE_BUTTON_GET_ACTIVE(d->outline_check);
	    const gint font_height = font->ascent + font->descent;
	    const gchar	*s = d->text,
			*s_end,
			*s_cur = s + d->text_cur_pos;
	    gint	x = d->text_start_wx,
			y = d->text_start_wy,
			fc_x, fc_y, len;
	    GdkColor	*color = &d->color,
			*outline_color = &d->outline_color;
	    GdkTextBounds b;

	    gdk_gc_set_function(gc, GDK_COPY);

	    /* Iterate through text and draw each line and the cursor */
	    while(s != NULL)
	    {
		/* Seek s_end to the next newline character or set it
		 * to NULL if this is the last line, then calculate
		 * the lenth of this line
		 */
		s_end = strchr(s, '\n');
		len = (s_end != NULL) ? (gint)(s_end - s) : STRLEN(s);
		if(len > 0)
		{
		    gdk_text_bounds(font, s, len, &b);
		    fc_x = x - b.lbearing;
		    fc_y = y - (font_height / 2) + font->ascent;
		    if(outline)
		    {
			gdk_gc_set_foreground(gc, outline_color);
			gdk_draw_text(
			    drawable, font, gc,
			    fc_x, fc_y - 1,
			    s, len
			);
			gdk_draw_text(
			    drawable, font, gc,
			    fc_x + 1, fc_y,
			    s, len
			);
			gdk_draw_text(
			    drawable, font, gc,
			    fc_x, fc_y + 1,
			    s, len
			);
			gdk_draw_text(
			    drawable, font, gc,
			    fc_x - 1, fc_y,
			    s, len
			);
		    }
		    gdk_gc_set_foreground(gc, color);
		    gdk_draw_text(
			drawable, font, gc,
			fc_x, fc_y,
			s, len
		    );
		}

		/* Calculate the cursor coordinates if it is on this
		 * line
		 */
		if((s_cur >= s) && (s_cur <= (s + len)))
		{
		    const gint len2 = (gint)(s_cur - s);
		    if(len2 > 0)
			gdk_text_bounds(font, s, len2, &b);
		    else
			b.width = 0;
		    cur_x = x + b.width;
		    cur_y = y;
		}

		y += font_height;
		s = (s_end != NULL) ? (s_end + 1) : NULL;
	    }
	}

	/* Update the text cursor window coordinate position */
	d->text_cur_wx = cur_x;
	d->text_cur_wy = cur_y;

	/* Force the blink state to FALSE to ensure that it gets
	 * drawn
	 */
	d->draw_text_cur_blink_state = FALSE;

	draw_text_cursor_on_imgview(d);
}

/*
 *	Queues a draw text on the ImgView.
 */
static void queue_draw_text_on_imgview(Dialog *d)
{
	if(d == NULL)
	    return;

	gtk_widget_queue_draw(d->iv_drawing_area);
}


/*
 *	Sets the text starting position in window coordinates.
 *
 *	The d specifies the Dialog.
 *
 *	The x and y specifies the text starting position in window
 *	coordinates.
 */
static void set_text_window_position(
	Dialog *d,
	const gint x, const gint y
)
{
	imgview_struct *iv = d->iv;

	d->text_start_wx = x;
	d->text_start_wy = y;

	d->text_dx = ImgViewConvertUnitViewToOrigX(iv, x);
	d->text_dy = ImgViewConvertUnitViewToOrigY(iv, y);

	/* Set the initial cursor position in the text as needed */
	if(d->text_cur_pos < 0)
	    d->text_cur_pos = 0;

	/* Remove the cursor blink draw timeout */
	GTK_TIMEOUT_REMOVE(d->draw_text_cur_toid);
	d->draw_text_cur_toid = 0;

	/* Clear the existing cursor? */
	if(d->draw_text_cur_blink_state)
	    draw_text_cursor_on_imgview(d);

	queue_draw_text_on_imgview(d);

	/* Set a new cursor blink draw timeout */
	d->draw_text_cur_toid = gtk_timeout_add(
	    CURSOR_BLINK_INT,
	    blink_text_cur_tocb, d
	);
}


/*
 *	Renders the RGB text image data specified by src to the RGBA
 *	image data specified by tar.
 *
 *	The RGB text image data contains black pixels that indicate
 *	transparent, white pixels that indicate solid, and grey
 *	pixels that indicate background.
 *
 *	The r, g, and b represent the color of the text.
 */
static void render_text_on_imgview_iterate(
	const guint8 *src,		/* RGB image data */
	guint8 *tar,			/* RGBA image data */
	const gint width, const gint height,
	const gint src_bpp, const gint src_bpl,
	const gint tar_bpp, const gint tar_bpl,
	const gfloat src_opacity, const gfloat tar_opacity,
	const guint8 r, const guint8 g, const guint8 b
)
{
	const guint8 black_rgb[] = { 0x00, 0x00, 0x00 };
	gint y;
	const guint8 *src_ptr, *src_end;
	guint8 *tar_ptr;

	for(y = 0; y < height; y++)
	{
	    src_ptr = src + (y * src_bpl);
	    src_end = src_ptr + (width * src_bpp);
	    tar_ptr = tar + (y * tar_bpl);
	    while(src_ptr < src_end)
	    {
		if(memcmp(src_ptr, black_rgb, src_bpp))
		{
		    *tar_ptr++ = (guint8)(
			(*tar_ptr * tar_opacity) +
			(r * src_opacity)
		    );
		    *tar_ptr++ = (guint8)(
			(*tar_ptr * tar_opacity) +
			(g * src_opacity)
		    );
		    *tar_ptr++ = (guint8)(
			(*tar_ptr * tar_opacity) +
			(b * src_opacity)
		    );
		    *tar_ptr++ = 0xff;
		}
		else
		{
		    tar_ptr += tar_bpp;
		}
		src_ptr += src_bpp;
	    }
	}
}

/*
 *	Renders the text to the ImgView.
 *
 *	The iv specifies the ImgView.
 *
 *	The text specifies the text to render to the ImgView.
 *
 *	The x and y specifies the window coordinates to start
 *	rendering the text at.
 *
 *	The font specifies the font.
 *
 *	The color specifies the color of the text.
 *
 *	If outline is TRUE then an outline will be drawn around the
 *	text.
 *
 *	The outline_color specifies the color of the outline (used
 *	only if outline is TRUE).
 *
 *	The opacity specifies the opacity of the text from 0.0 to 1.0.
 *
 *	If all_frames is TRUE then the text will be rendered to all
 *	the frames.
 */
static void render_text_on_imgview(
	imgview_struct *iv,
	const gchar *text,
	const gint x, const gint y,
	GdkFont *font,
	GdkColor *color,
	const gboolean outline,
	GdkColor *outline_color,
	const gfloat opacity,
	const gboolean all_frames
)
{
	gint width, height, width2, height2, text_rgb_bpl;
	guint8 *text_rgb;
	GdkColor fg_mark_color, transparent_mark_color;
	GdkColormap *colormap;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkGC *gc;
	GtkWidget *w;
	imgview_image_struct *img;

	if((iv == NULL) || (text == NULL) || (font == NULL) ||
	   (color == NULL)
	)
	    return;

	/* The text is drawn to the ImgView image as follows:
	 *
	 * A tempory GdkPixmap, the size of the current ImgView image,
	 * will be created and the text will be drawn to it using
	 * white color
	 *
	 * Then the tempory GdkPixmap is grabbed to a RGBA buffer and
	 * then unref'ed
	 *
	 * The RGBA buffer, containing the drawn text in white, will
	 * then be overlayed to the current ImgView frame taking into
	 * account the specified color and opacity
	 */

	/* Get the ImgView's image that we will draw the new text to */
	img = ImgViewGetImage(iv);
	w = ImgViewGetToplevelWidget(iv);
	if((img == NULL) || (w == NULL))
	    return;

	window = w->window;
	width = img->width;
	height = img->height;

	/* Create the tempory GdkPixmap */
	pixmap = gdk_pixmap_new(window, width, height, -1);

	/* Get the colormap and allocate the GdkColors */
	colormap = gdk_window_get_colormap(window);
	GDK_COLORMAP_REF(colormap);
	GDK_COLOR_SET_COEFF(&fg_mark_color, 1.0f, 1.0f, 1.0f);
	GDK_COLORMAP_ALLOC_COLOR(colormap, &fg_mark_color);
	GDK_COLOR_SET_COEFF(&transparent_mark_color, 0.0f, 0.0f, 0.0f);
	GDK_COLORMAP_ALLOC_COLOR(colormap, &transparent_mark_color);


	/* Create the Graphics Context and set it up to draw the
	 * text to our new GdkPixmap
	 */
	gc = gdk_gc_new(window);

	/* Begin drawing the outline */

	/* Clear the GdkPixmap */
	gdk_gc_set_foreground(gc, &transparent_mark_color);
	gdk_draw_rectangle(
	    pixmap, gc, TRUE,
	    0, 0, width, height
	);

	/* Draw the text to the tempory GdkPixmap, black pixels will
	 * be considered transparent and white pixels will be
	 * considered solid when this GdkPixmap is read to an RGB
	 * image data
	 */
	if((text != NULL) && outline)
	{
	    const gint font_height = font->ascent + font->descent;
	    const gchar *s = text, *s_end;
	    gint	cur_x = x,
			cur_y = y,
			fc_x, fc_y, len;
	    GdkTextBounds b;

	    while(s != NULL)
	    {
		s_end = strchr(s, '\n');
		len = (s_end != NULL) ? (gint)(s_end - s) : STRLEN(s);
		if(len > 0)
		{
		    gdk_text_bounds(font, s, len, &b);
		    fc_x = cur_x - b.lbearing;
		    fc_y = cur_y - (font_height / 2) + font->ascent;
		    gdk_gc_set_foreground(gc, &fg_mark_color);
		    gdk_draw_text(
			pixmap, font, gc,
			fc_x, fc_y - 1,
			s, len
		    );
		    gdk_draw_text(
			pixmap, font, gc,
			fc_x + 1, fc_y,
			s, len
		    );
		    gdk_draw_text(
			pixmap, font, gc,
			fc_x, fc_y + 1,
			s, len
		    );
		    gdk_draw_text(
			pixmap, font, gc,
			fc_x - 1, fc_y,
			s, len
		    );
		    gdk_gc_set_foreground(gc, &transparent_mark_color);
		    gdk_draw_text(
                        pixmap, font, gc,
                        fc_x, fc_y,
                        s, len
                    );
		}
		cur_y += font_height;
		s = (s_end != NULL) ? (s_end + 1) : NULL;
	    }

	    /* Get the GdkPixmap's contents to an RGB image and then
	     * unref the GdkPixmap
	     */
	    text_rgb = gdk_get_rgb_image(
		pixmap, NULL,
		&width2, &height2, &text_rgb_bpl
	    );
	    if((text_rgb != NULL) && (width == width2) && (height == height2))
	    {
		const guint8	r = (guint8)(outline_color->red >> 8),
				g = (guint8)(outline_color->green >> 8),
				b = (guint8)(outline_color->blue >> 8);
		const gfloat	src_opacity = CLIP(opacity, 0.0f, 1.0f),
				tar_opacity = 1.0f - src_opacity;
		const gint	src_bpp = 3,
				src_bpl = text_rgb_bpl,
				tar_bpp = img->bpp,
				tar_bpl = img->bpl;
		imgview_frame_struct *frame;

		if(all_frames)
		{
		    GList *glist = img->frames_list;
		    while(glist != NULL)
		    {
			frame = IMGVIEW_FRAME(glist->data);
			if((frame != NULL) ? (frame->buf != NULL) : FALSE)
			    render_text_on_imgview_iterate(
				text_rgb,
				frame->buf,
				width, height,
				src_bpp, src_bpl, tar_bpp, tar_bpl,
				src_opacity, tar_opacity,
				r, g, b
			    );
			glist = g_list_next(glist);
		    }
		}
		else
		{
		    frame = ImgViewImageGetFrame(
			img, ImgViewGetCurrentFrame(iv)
		    );
		    if((frame != NULL) ? (frame->buf != NULL) : FALSE)
			render_text_on_imgview_iterate(
			    text_rgb,
			    frame->buf,
			    width, height,
			    src_bpp, src_bpl, tar_bpp, tar_bpl,
			    src_opacity, tar_opacity,
			    r, g, b
			);
		}
	    }
	    g_free(text_rgb);
	}


	/* Begin drawing the text */

	/* Clear the GdkPixmap */
	gdk_gc_set_foreground(gc, &transparent_mark_color);
	gdk_draw_rectangle(
	    pixmap, gc, TRUE,
	    0, 0, width, height
	);

	/* Draw the text to the tempory GdkPixmap, black pixels will
	 * be considered transparent and white pixels will be
	 * considered solid when this GdkPixmap is read to an RGB
	 * image data
	 */
	if(text != NULL)
	{
	    const gint font_height = font->ascent + font->descent;
	    const gchar *s = text, *s_end;
	    gint	cur_x = x,
			cur_y = y,
			fc_x, fc_y, len;
	    GdkTextBounds b;

	    gdk_gc_set_foreground(gc, &fg_mark_color);

	    while(s != NULL)
	    {
		s_end = strchr(s, '\n');
		len = (s_end != NULL) ? (gint)(s_end - s) : STRLEN(s);
		if(len > 0)
		{
		    gdk_text_bounds(font, s, len, &b);
		    fc_x = cur_x - b.lbearing;
		    fc_y = cur_y - (font_height / 2) + font->ascent;
		    gdk_draw_text(
			pixmap, font, gc,
			fc_x, fc_y,
			s, len
		    );
		}
		cur_y += font_height;
		s = (s_end != NULL) ? (s_end + 1) : NULL;
	    }
	}

	/* Get the GdkPixmap's contents to an RGB image and then unref
	 * the GdkPixmap
	 */
	text_rgb = gdk_get_rgb_image(
	    pixmap, NULL,
	    &width2, &height2, &text_rgb_bpl
	);
	if((text_rgb != NULL) && (width == width2) && (height == height2))
	{
	    const guint8	r = (guint8)(color->red >> 8),
				g = (guint8)(color->green >> 8),
				b = (guint8)(color->blue >> 8);
	    const gfloat	src_opacity = CLIP(opacity, 0.0f, 1.0f),
				tar_opacity = 1.0f - src_opacity;
	    const gint	src_bpp = 3,
			src_bpl = text_rgb_bpl,
			tar_bpp = img->bpp,
			tar_bpl = img->bpl;
	    imgview_frame_struct *frame;

	    if(all_frames)
	    {
		GList *glist = img->frames_list;
		while(glist != NULL)
		{
		    frame = IMGVIEW_FRAME(glist->data);
		    if((frame != NULL) ? (frame->buf != NULL) : FALSE)
			render_text_on_imgview_iterate(
			    text_rgb,
			    frame->buf,
			    width, height,
			    src_bpp, src_bpl, tar_bpp, tar_bpl,
			    src_opacity, tar_opacity,
			    r, g, b
			);
		    glist = g_list_next(glist);
		}
	    }
	    else
	    {
		frame = ImgViewImageGetFrame(
		    img, ImgViewGetCurrentFrame(iv)
		);
		if((frame != NULL) ? (frame->buf != NULL) : FALSE)
		    render_text_on_imgview_iterate(
			text_rgb,
			frame->buf,
			width, height,
			src_bpp, src_bpl, tar_bpp, tar_bpl,
			src_opacity, tar_opacity,
			r, g, b
		    );
	    }
	}
	else
	{
	    if(text_rgb == NULL)
		g_printerr(
"render_text_on_imgview(): Unable to get image data from text pixmap.\n"
		);
	    else if((width != width2) || (height != height2))
		g_printerr(
"render_text_on_imgview(): Unable to get exact image data size from text pixmap.\n"
		);
	}
	g_free(text_rgb);

	GDK_PIXMAP_UNREF(pixmap);

	GDK_GC_UNREF(gc);
	GDK_COLORMAP_FREE_COLOR(colormap, &transparent_mark_color);
	GDK_COLORMAP_FREE_COLOR(colormap, &fg_mark_color);
	GDK_COLORMAP_UNREF(colormap);
}


/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;

	/* GtkAccelGroup */
	d->accelgrp = accelgrp = gtk_accel_group_new();

	/* Toplevel GtkWindow */
	d->toplevel = toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_fonts_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
/*	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE); */
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Add Button */
	d->add_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_fonts_20x20_xpm,
	    "Add",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(add_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);

	/* Cancel Button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close Button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
	    "Close",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);
}


/*
 *	Queries the user to add text.
 *
 *	Returns TRUE if new text was added or FALSE if the user
 *	canceled.
 */
gboolean iv_add_text(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gchar **font_name,
	GdkColor *color,
	gboolean *outline,
	GdkColor *outline_color,
	gfloat *opacity,
	gboolean *all_frames
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gboolean got_response;
	gchar *s;
	GdkColormap *colormap;
	GdkWindow *window;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	Dialog *d;

	if(iv == NULL)
	    return(FALSE);

	/* Create dialog base widgets */
	d = DIALOG(g_malloc0(sizeof(Dialog)));
	if(ImgViewGetTotalFrames(iv) > 1)
	    s = g_strdup_printf(
		"Add Text: %s: Frame %i",
		STRISEMPTY(filename) ? "Untitled" : filename,
		ImgViewGetCurrentFrame(iv) + 1
	    );
	else
	    s = g_strconcat(
		"Add Text: ",
		STRISEMPTY(filename) ? "Untitled" : filename,
		NULL
	    );
	build_std_dialog(
	    d,
	    400, -1,
	    s,					/* Title */
	    ref_toplevel,
	    &toplevel, &client_vbox
	);
	g_free(s);
	window = (toplevel != NULL) ? toplevel->window : NULL;
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	d->colormap = colormap = gdk_window_get_colormap(window);
	GDK_COLORMAP_REF(colormap);
	d->gc = gdk_gc_new(window);
	d->text_cur = gdk_cursor_new(GDK_XTERM);
	d->iv = iv;
	d->iv_drawing_area = (GtkWidget *)ImgViewGetViewWidget(iv);
	d->iv_expose_sigid = gtk_signal_connect_after(
	    GTK_OBJECT(d->iv_drawing_area), "expose_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	/* Connect key events before ImgView event handlers */
	d->iv_key_press_sigid = gtk_signal_connect(
	    GTK_OBJECT(d->iv_drawing_area), "key_press_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	d->iv_key_release_sigid = gtk_signal_connect(
	    GTK_OBJECT(d->iv_drawing_area), "key_release_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	d->iv_button_press_sigid = gtk_signal_connect(
	    GTK_OBJECT(d->iv_drawing_area), "button_press_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	d->iv_button_release_sigid = gtk_signal_connect(
	    GTK_OBJECT(d->iv_drawing_area), "button_release_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	d->iv_enter_notify_sigid = gtk_signal_connect(
	    GTK_OBJECT(d->iv_drawing_area), "enter_notify_event",
	    GTK_SIGNAL_FUNC(imgview_event_cb), d
	);
	d->text_start_wx = 0;
	d->text_start_wy = 0;
	d->text_dx = 0;
	d->text_dy = 0;
	d->text = NULL;
	d->text_cur_pos = -1;			/* Not set */
	d->draw_text_cur_toid = 0;
	d->draw_text_cur_blink_state = FALSE;
	d->font_name = (font_name != NULL) ?
	    ((*font_name != NULL) ?
		g_strdup(*font_name) : g_strdup(ADD_TEXT_DEF_FONT_NAME)
	    ) : g_strdup(ADD_TEXT_DEF_FONT_NAME);
	d->font = gdk_font_load(d->font_name);
	if(color != NULL)
	    memcpy(&d->color, color, sizeof(GdkColor));
	else
	    GDK_COLOR_SET_COEFF(&d->color, 1.0f, 1.0f, 1.0f);
	GDK_COLORMAP_ALLOC_COLOR(d->colormap, &d->color);
	if(outline_color != NULL)
	    memcpy(&d->outline_color, outline_color, sizeof(GdkColor));
	else
	    GDK_COLOR_SET_COEFF(&d->outline_color, 0.0f, 0.0f, 0.0f);
	GDK_COLORMAP_ALLOC_COLOR(d->colormap, &d->outline_color);
	d->opacity = CLIP(((opacity != NULL) ?
	    *opacity : ADD_TEXT_DEF_OPACITY), 0.0f, 1.0f
	);

	/* Change the ImgView's cursor to the text cursor */
	if(d->iv_drawing_area != NULL)
	{
	    GdkWindow *window = d->iv_drawing_area->window;
	    gdk_window_set_cursor(window, d->text_cur);
	}

	d->freeze_count++;

	/* Create the widgets */
	if(client_vbox != NULL)
	{
	    GtkAdjustment *adj;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget	*w,
			*parent = client_vbox,
			*parent2, *parent3, *parent4, *parent5;


	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent = w;


	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		window,
		(style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		(guint8 **)icon_fonts_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Vbox for Instructions */
	    w = gtk_vbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

#define ADD_LINE(_p_,_d_,_s_)	{	\
 GtkWidget *p2, *p3;			\
					\
 /* Hbox */				\
 w = gtk_hbox_new(FALSE, 0);		\
 gtk_box_pack_start(			\
  GTK_BOX(_p_), w, FALSE, FALSE, 0	\
 );					\
 gtk_widget_show(w);			\
 p2 = w;				\
					\
 /* Icon */				\
 if((_d_) != NULL) {			\
  w = gtk_vbox_new(FALSE, 0);		\
  gtk_box_pack_start(			\
   GTK_BOX(p2), w, FALSE, FALSE, 0	\
  );					\
  gtk_widget_show(w);			\
  p3 = w;				\
  w = gtk_pixmap_new_from_xpm_d(	\
   window,				\
   (style != NULL) ?			\
    &style->bg[GTK_STATE_NORMAL] : NULL,\
   (guint8 **)(_d_)			\
  );					\
  gtk_box_pack_start(			\
   GTK_BOX(p3), w, FALSE, FALSE, 0	\
  );					\
  gtk_widget_show(w);			\
 }					\
					\
 /* Label */				\
 w = gtk_label_new(_s_);		\
 gtk_label_set_justify(			\
  GTK_LABEL(w), GTK_JUSTIFY_LEFT	\
 );					\
 gtk_box_pack_start(			\
  GTK_BOX(p2), w, FALSE, FALSE, 0	\
 );					\
 gtk_widget_show(w);			\
}

#if defined(PROG_LANGUAGE_SPANISH)
	    ADD_LINE(
		parent3, NULL,
"Para agregar texto:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Escoja el tipo de letra y el color deseados."  
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Ponga la posicio'n deseada de texto haciendo clic en la\nimagen."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Entre el texto deseado y entonces clic en \"Add\"."
	    );
	    ADD_LINE(
		parent3, NULL,
"Ponga el va zumbando a 1:1 para la presentacio'n de\ntexto ma's exacta."
	    );
#elif defined(PROG_LANGUAGE_FRENCH)
	    ADD_LINE(
		parent3, NULL,
"Pour ajouter le texte:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Choisir le jeu de caracteres et la couleur desirees."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Regler la position desiree de texte par clique sur\nl'image."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Entrer le texte desire et cliquer sur alors \"Add\"."
	    );
	    ADD_LINE(
		parent3, NULL,
"Regler le faire un zoom a 1:1 pour l'avant-premiere du\ntexte le plus precis."
	    );
#elif defined(PROG_LANGUAGE_GERMAN)
	    ADD_LINE(
		parent3, NULL,
"Um text hinzuzufugen:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Wahlen sie die gewunschte schriftart und die farbe aus."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Setzen sie die gewunschte textposition durch klicken auf\ndem bildnis."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Tragen sie den gewunschten text und das dann klicken auf\n\"Add\" ein."
	    );
	    ADD_LINE(
		parent3, NULL,
"Setzen aie das zoom zu 1:1 fur die genauste textvorschau."
	    );
#elif defined(PROG_LANGUAGE_ITALIAN)
	    ADD_LINE(
		parent3, NULL,
"Per aggiungere il testo:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Scegliere il font desiderati ed il colore."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Regolare la posizione di testo desiderata scattando\nsull'immagine."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Entrare il testo desiderato ed il poi scatto su \"Add\"."
	    );
	    ADD_LINE(
		parent3, NULL,
"Regolare lo zoom a 1:1 per il testo piu esatto vede in\nanteprima."
	    );

#elif defined(PROG_LANGUAGE_DUTCH)
	    ADD_LINE(
		parent3, NULL,
"Om tekst toe te voegen:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Selecteer het gewenste lettertype en kleur."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Zet de gewenste tekstpositie door het klikken op het beeld."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Ga de gewenste tekst en dan klik op \"Add\" binnen."
	    );
	    ADD_LINE(
		parent3, NULL,
"Zet de zoomlens tot 1:1 voor de nauwkeurigste\nteksttrailer."
	    );
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    ADD_LINE(
		parent3, NULL,
"Adicionar texto:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Selecione a fonte desejada e cor."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Ponha a posicao desejada de texto por clicar na imagem."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Entre o texto desejado e entao estalido em \"Add\"."
	    );
	    ADD_LINE(
		parent3, NULL,
"Ponha o ronco a 1:1 para a apresentacao bem exata de\ntexto."
	    );
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    ADD_LINE(
		parent3, NULL,
"Tilfoye tekst:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Velg ut den ?nskede fonten og den farge."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Sett den onskede tekstposisjon ved a klikke pa\navbildet."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Ga inn i den onskede teksten og den da klikk pa\n\"Add\"."
	    );
	    ADD_LINE(
		parent3, NULL,
"Sett zoomen til 1:1 for den meste noyaktige\nteksten preview."
	    );
#else
	    ADD_LINE(
		parent3, NULL,
"To add text:"
	    );
	    parent4 = w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(
		GTK_BOX(parent3), w, FALSE, FALSE, 12
	    );
	    gtk_widget_show(w);
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Select the desired font and color."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Set the desired text position by clicking on the image."
	    );
	    ADD_LINE(
		parent4, icon_li_16x16_xpm,
"Enter the desired text and then click the \"Add\" button."
	    );
	    ADD_LINE(
		parent3, NULL,
"Set the zoom to 1:1 for the most accurate text preview."
	    );
#endif

#undef ADD_LINE

	    w = gtk_hseparator_new();
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Vbox for text values */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;


	    /* Hbox for font prompt */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Font prompt */
	    d->font_prompt = w = FSDPromptNew(
		"Font:", -1, -1,
		d, select_font_cb
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	    FSDPromptSetFontName(
		d->font_prompt, d->font_name
	    );
	    gtk_widget_show(w);


	    /* Hbox for color and outline color */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Color */
	    d->color_btn = w = CSDColorButtonNew(
		"Color:", -1,
		d, select_color_cb
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    if(w != NULL)
	    {
		const GdkColor *sc = &d->color;
		csd_color_struct c;
		c.r = (gfloat)sc->red / (gfloat)((gushort)-1);
		c.g = (gfloat)sc->green / (gfloat)((gushort)-1);
		c.b = (gfloat)sc->blue / (gfloat)((gushort)-1);
		CSDColorButtonSetColor(w, &c);
	    }
	    gtk_widget_show(w);

	    /* Outline */
	    w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent5 = w;

	    /* Outline check */
	    d->outline_check = w = gtk_check_button_new_with_label(
		"Outline"
	    );
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
            gtk_signal_connect(  
                GTK_OBJECT(w), "toggled",
                GTK_SIGNAL_FUNC(outline_toggled_cb), d
            );
	    gtk_widget_show(w);
	    if(outline != NULL)
		GTK_TOGGLE_BUTTON_SET_ACTIVE(w, *outline);

	    /* Outline Color */
	    d->outline_color_btn = w = CSDColorButtonNew(
		NULL, -1,
		d, select_outline_color_cb
	    );
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    if(w != NULL)
	    {
		const GdkColor *sc = &d->outline_color;
		csd_color_struct c;
		c.r = (gfloat)sc->red / (gfloat)((gushort)-1);
		c.g = (gfloat)sc->green / (gfloat)((gushort)-1);
		c.b = (gfloat)sc->blue / (gfloat)((gushort)-1);
		CSDColorButtonSetColor(w, &c);
	    }
	    gtk_widget_show(w);


	    /* Hbox for color and outline color */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Opacity */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    /* GtkLabel */
	    w = gtk_label_new("Opacity:");
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* GtkHScale */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		d->opacity,
		0.0f, 1.0f,
		0.1f, 0.1f, 0.0f
	    );
	    gtk_signal_connect(
		GTK_OBJECT(adj), "value_changed",
		GTK_SIGNAL_FUNC(opacity_scale_cb), d
	    );
	    d->opacity_scale = w = gtk_hscale_new(adj);
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 100, -1);
	    gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	    gtk_widget_show(w);
	    /* GtkEntry */
	    d->opacity_entry = w = gtk_entry_new();
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 40, -1);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(opacity_entry_cb), d
	    );
	    gtk_widget_show(w);
	    /* GtkLabel */
	    w = gtk_label_new("%");
	    gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    d->freeze_count--;
	    opacity_scale_cb(adj, d);
	    d->freeze_count++;

	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    d->all_frames_check = w = gtk_check_button_new_with_label(
		"All Frames"
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_set_sensitive(
		w,
		(ImgViewGetTotalFrames(iv) > 1) ? TRUE : FALSE
	    );
	    GUISetWidgetTip(
		w,
		"Check this to add text to all the\
 frames instead of just the current frame"
	    );
	    gtk_widget_show(w);

	    if(all_frames != NULL)
		GTK_TOGGLE_BUTTON_SET_ACTIVE(w, *all_frames);
	}

	d->freeze_count--;

	/* Map toplevel */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Restore the ImgView's cursor */
	if(d->iv_drawing_area != NULL)
	{
	    GdkWindow *window = d->iv_drawing_area->window;
	    gdk_window_set_cursor(window, NULL);
	}

	/* User clicked on "Add"? */
	got_response = d->got_response;
	if(got_response)
	{
	    /* Render text to the ImgView's image */
	    render_text_on_imgview(
		d->iv, d->text,
		d->text_dx,		/* Text position in image */
		d->text_dy,
		d->font,
		&d->color,
		GTK_TOGGLE_BUTTON_GET_ACTIVE(d->outline_check),
		&d->outline_color,
		d->opacity,		/* 0.0 to 1.0 */
		GTK_TOGGLE_BUTTON_GET_ACTIVE(d->all_frames_check)
	    );

	    /* Record values */
	    if(font_name != NULL)
	    {
		g_free(*font_name);
		*font_name = STRDUP(d->font_name);
	    }
	    if(color != NULL)
		memcpy(color, &d->color, sizeof(GdkColor));
	    if(outline_color != NULL)
		memcpy(outline_color, &d->outline_color, sizeof(GdkColor));
	    if(opacity != NULL)
		*opacity = d->opacity;
	    if(all_frames != NULL)
		*all_frames = GTK_TOGGLE_BUTTON_GET_ACTIVE(d->all_frames_check);
	}

	/* Redraw the ImgView to clean up text markings and to show
	 * the newly added text (if any)
	 */
	ImgViewQueueDrawView(iv);


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
/*	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE); */
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_TIMEOUT_REMOVE(d->draw_text_cur_toid);

	/* Need to disconnect the signals from the ImgView */
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_expose_sigid
	);
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_key_press_sigid
	);
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_key_release_sigid
	);
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_button_press_sigid
	);
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_button_release_sigid
	);
	gtk_signal_disconnect(
	    GTK_OBJECT(d->iv_drawing_area), d->iv_enter_notify_sigid
	);

	GTK_WIDGET_DESTROY(d->all_frames_check);
	GTK_WIDGET_DESTROY(d->opacity_entry);
	GTK_WIDGET_DESTROY(d->opacity_scale);
	GTK_WIDGET_DESTROY(d->color_btn);
	GTK_WIDGET_DESTROY(d->outline_color_btn);
	GTK_WIDGET_DESTROY(d->outline_check);
	GTK_WIDGET_DESTROY(d->font_prompt);
	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	GDK_GC_UNREF(d->gc);
	GDK_CURSOR_DESTROY(d->text_cur);
	GDK_FONT_UNREF(d->font);
	GDK_COLORMAP_FREE_COLOR(d->colormap, &d->outline_color);
	GDK_COLORMAP_FREE_COLOR(d->colormap, &d->color);
	GDK_COLORMAP_UNREF(d->colormap);
	g_free(d->font_name);
	g_free(d->text);
	g_free(d);

	return(got_response);
}
