//LabPlot : FitListDialog.cc

#include <kconfig.h>
#ifdef HAVE_GSL
#include <gsl/gsl_rng.h>
#include <gsl/gsl_blas.h>
#endif
#include "FitListDialog.h"
#include "fit.h"
#include "parser_extern.h"
#include "defs.h"

using namespace std;

struct data {
	int n;
	double *x;
	double *y;
	double *sigma;
	int np;
	FModel model;
	double base;
	QString fun;
	MainWin *mw;
};

FitListDialog::FitListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	setCaption(i18n("Fit Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Fit" );

	Plot *plot = 0;
	if(p != 0) plot = p->getPlot(p->API());

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	QLabel *tmp = new QLabel(i18n("Model : "),hb);
	tmp->setMaximumWidth(50);
	modelcb = new KComboBox(hb);
	modelcb->insertStrList(fit_modelnames);

	config->setGroup("Fit Functions" );
	// read functions
	for(int i=0;i<10;i++) {
		QString fun = config->readEntry(QString("function%1").arg(i+1));
		if (!fun.isEmpty())
			modelcb->insertItem(fun);
		else
			break;
	}
	config->setGroup( "Fit" );

	modelcb->setCurrentItem(config->readNumEntry("Model",0));
	QObject::connect(modelcb,SIGNAL(activated (int)),SLOT(updateModel(int)));
	// custom fit function
	hb = new QHBox(tab1);
	new QLabel(i18n("Fit Function :"),hb);
	new QLabel(i18n(" y = "),hb);
	Graph *g=0;
	if(plot) {
		GraphList *gl = plot->getGraphList();
		g = gl->getGraph(0);
	}
	funle = new KLineEdit(g==0?config->readEntry("Function","a*x+b"):g->FitFunction(),hb);
	funle->setReadOnly(true);

	hb = new QHBox(tab1);
	new QLabel(i18n("Nr. of Parameter : "),hb);
	parni = new KIntNumInput(config->readNumEntry("Parameter",2),hb);
	parni->setEnabled(false);
	parni->setRange(1,NR_PARS);
	QObject::connect(parni,SIGNAL(valueChanged (int)),SLOT(updateParameter()));

	new QLabel(i18n("Initial Values : "),tab1);
	hb = new QHBox(tab1);
	new QLabel(i18n(" a = "),hb);
	parNle[0] = new KLineEdit(config->readEntry("Parameter0","1.0"),hb);
	new QLabel(i18n(" b = "),hb);
	parNle[1] = new KLineEdit(config->readEntry("Parameter1","1.0"),hb);
	new QLabel(i18n(" c = "),hb);
	parNle[2] = new KLineEdit(config->readEntry("Parameter2","1.0"),hb);
	hb = new QHBox(tab1);
	new QLabel(i18n(" d = "),hb);
	parNle[3] = new KLineEdit(config->readEntry("Parameter3","1.0"),hb);
	new QLabel(i18n(" e = "),hb);
	parNle[4] = new KLineEdit(config->readEntry("Parameter4","1.0"),hb);
	new QLabel(i18n(" f = "),hb);
	parNle[5] = new KLineEdit(config->readEntry("Parameter5","1.0"),hb);
	hb = new QHBox(tab1);
	new QLabel(i18n(" g = "),hb);
	parNle[6] = new KLineEdit(config->readEntry("Parameter6","1.0"),hb);
	new QLabel(i18n(" h = "),hb);
	parNle[7] = new KLineEdit(config->readEntry("Parameter7","1.0"),hb);
	new QLabel(i18n(" i = "),hb);
	parNle[8] = new KLineEdit(config->readEntry("Parameter8","1.0"),hb);

	for (int i=0;i<NR_PARS;i++) {
		parNle[i]->setValidator(new QDoubleValidator(parNle[i]));
		if (i>1) parNle[i]->setEnabled(false);
	}

	hb = new QHBox(tab1);
	resultcb = new QCheckBox(i18n("add result"),hb);
	if(plot == 0)
		resultcb->setChecked(false);
	else {
		resultcb->setChecked(config->readBoolEntry("AddResult",true));
		labelcb = new QCheckBox(i18n("add label"),hb);
		labelcb->setChecked(config->readBoolEntry("AddLabel",true));
	}

	QVBox *tab2 = new QVBox(tw);
	hb = new QHBox(tab2);
	new QLabel(i18n("Maximum Steps : "),hb);
	stepsni = new KIntNumInput(config->readNumEntry("Steps",100),hb);
	stepsni->setRange(1,INF,1,false);
	new QLabel(i18n(" Tolerance : "),hb);
	tolle = new KLineEdit(config->readEntry("Tolerance","0.001"),hb);
	tolle->setValidator(new QDoubleValidator(0,1,20,tolle));

	hb = new QHBox(tab2);
	new QLabel(i18n("Weight : "),hb);
	weightcb = new KComboBox(hb);
	int i=0;
	while(weightitems[i] != 0) weightcb->insertItem(i18n(weightitems[i++]));
	weightcb->setCurrentItem(config->readNumEntry("Weight",0));
	QObject::connect(weightcb,SIGNAL(activated(int)),SLOT(weightChanged()));
	hb = new QHBox(tab2);
	new QLabel(i18n("Weight Function : "),hb);
	weightle = new KLineEdit(config->readEntry("WeightFunction",i18n("equal")),hb);
	weightle->setReadOnly(true);

	hb = new QHBox(tab2);
	regioncb = new QCheckBox(i18n("use Region "),hb);
	if(plot != 0 && plot->RegionMin() != plot->RegionMax() )
		regioncb->setChecked(config->readBoolEntry("Region",true));
	else
		regioncb->setChecked(false);
	new QLabel(i18n("( From "),hb);
	regionminle = new KLineEdit(QString::number(plot !=0 ? plot->RegionMin():0),hb);
	regionminle->setValidator(new QDoubleValidator(regionminle));
	new QLabel(i18n(" To "),hb);
	regionmaxle = new KLineEdit(QString::number(plot !=0 ? plot->RegionMax():0),hb);
	regionmaxle->setValidator(new QDoubleValidator(regionmaxle));
	new QLabel(i18n(" )"),hb);
	hb = new QHBox(tab2);
	negate_regioncb = new QCheckBox(i18n("negate Region"),hb);
	negate_regioncb->setChecked(config->readBoolEntry("NegateRegion",false));

	hb = new QHBox(tab2);
	baselinecb = new QCheckBox(i18n("Use Baseline @ y = "),hb);
	baselinecb->setChecked(config->readBoolEntry("Baseline",false));
	baselinele = new KLineEdit(QString::number(plot != 0 ? plot->Baseline():0),hb);
	baselinele->setValidator(new QDoubleValidator(baselinele));

	hb = new QHBox(tab2);
	new QLabel(i18n("Number of Points for fit function : "),hb);
	int number = config->readNumEntry("Number",100);
	if(plot != 0) {
		GraphList *gl = plot->getGraphList();
		GRAPHType st = gl->getType(0);
		if (st == GRAPH2D) {
			Graph2D *g = gl->getGraph2D(0);
			number = g->Number();
		}
	}
	numberni = new KIntNumInput(config->readNumEntry("Number",number),hb);
	numberni->setRange(1,INF,1,false);

	hb = new QHBox(tab2);
	new QLabel(i18n("Range of fit function : "),hb);
	LRange *range = 0;
	if(plot !=0) range = plot->Ranges();
	minle = new KLineEdit(QString::number(plot ? range[0].rMin():0),hb);
	minle->setValidator(new QDoubleValidator(minle));
	new QLabel(i18n(" .. "),hb);
	maxle = new KLineEdit(QString::number(plot ? range[0].rMax():0),hb);
	maxle->setValidator(new QDoubleValidator(maxle));

	hb = new QHBox(tab2);
	rescb = new QCheckBox(i18n("Show Residuals"),hb);
	rescb->setChecked(config->readBoolEntry("Residuals",false));

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab=0;
	if(p!=0) {
		if(p->getPlot(p->API())->Type() == PSURFACE)
			styletab = surfaceStyle(tw,true);
		else
			styletab = simpleStyle(tw, style, symbol);
	}
	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(tab2,i18n("Advanced"));
	if(p!=0)
		tw->addTab(styletab,i18n("Style"));

	infote = new QTextEdit(vbox);

#if QT_VERSION > 0x030005
	infote->setTextFormat( Qt::LogText );
#else
	infote->setTextFormat( Qt::PlainText );
#endif

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void FitListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Fit" );

	config->writeEntry("Model",modelcb->currentItem());
	config->writeEntry("Function",funle->text());
	config->writeEntry("Parameter",parni->value());
	for(int i=0;i<NR_PARS;i++)
		config->writeEntry(QString("Parameter%1").arg(i),parNle[i]->text());
	config->writeEntry("AddResult",resultcb->isChecked());
	config->writeEntry("AddLabel",labelcb->isChecked());
	config->writeEntry("Steps", stepsni->value());
	config->writeEntry("Tolerance", tolle->text());
	config->writeEntry("Weight", weightcb->currentItem());
	config->writeEntry("WeightFunction", weightle->text());
	config->writeEntry("Region", regioncb->isChecked());
	config->writeEntry("NegateRegion", negate_regioncb->isChecked());
	config->writeEntry("Baseline", baselinecb->isChecked());
	config->writeEntry("Number", numberni->value());
	config->writeEntry("Residuals", rescb->isChecked());
}

void FitListDialog::weightChanged() {
	kdDebug()<<"FitListDialog::weightChanged()"<<endl;
	int item = weightcb->currentItem();

	weightle->setText(i18n(weightitems[item]));
	if(item==WUSER)
		weightle->setReadOnly(false);
	else
		weightle->setReadOnly(true);
}

//! called when the number of parameter is changed in user fit model
void FitListDialog::updateParameter() {
	int np = parni->value();

	for (int i=0;i<NR_PARS;i++) {
		parNle[i]->setEnabled(false);
		if(np>i) parNle[i]->setEnabled(true);
	}
}

//! called when a model is selected
void FitListDialog::updateModel(int model) {
	// TODO : set inital values using selected graph values
	parni->setEnabled(false);
	if((FModel)model<=MUSER)
		funle->setText(QString(fit_modelitems[model]));
	funle->setReadOnly(true);

	if ((FModel)model == MEXP || (FModel)model == MLORENTZ || (FModel)model == MGAUSSIAN) {
		parni->setValue(3);
		parNle[2]->setEnabled(true);
	}
	else if ((FModel)model == MMULTIEXP2) {
		parni->setValue(4);
		parNle[2]->setEnabled(true);
		parNle[3]->setEnabled(true);
	}
	else if ((FModel)model == MMULTIEXP3) {
		parni->setValue(6);
		parNle[2]->setEnabled(true);
		parNle[3]->setEnabled(true);
		parNle[4]->setEnabled(true);
		parNle[5]->setEnabled(true);
	}
	else if ((FModel)model == MUSER) {
		parni->setValue(2);
		parni->setEnabled(true);
		// set from selected graph
		int item = 0;
		if(lv != 0 && lv->currentItem() != 0)
			item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());

		Graph *g = 0;
		if(p != 0) {
			Plot *plot = p->getPlot(p->API());
			if(plot != 0) {
				GraphList *gl = plot->getGraphList();
				if(gl != 0)
					g = gl->getGraph(item);
			}
		}
		funle->setText(g==0?QString("a*x+b"):g->FitFunction());
		funle->setReadOnly(false);
	}
	else if ((FModel)model > MUSER) {
		QString string = modelcb->currentText();
		QStringList list = QStringList::split(QRegExp(";"),string);
		parni->setValue(list[1].toInt());
		parni->setEnabled(true);
		funle->setText(list[0]);
		funle->setReadOnly(false);
		for(int i=0;i<list[1].toInt();i++)
			parNle[i]->setEnabled(true);
	}
	else  {
		parni->setValue(2);
		parNle[2]->setEnabled(false);
		parNle[3]->setEnabled(false);
		parNle[4]->setEnabled(false);
		parNle[5]->setEnabled(false);
	}
}

void FitListDialog::setFunction(QString fun) {
	modelcb->setCurrentItem(MUSER);
	funle->setText(fun);
}

void FitListDialog::setNrParameter(int par) {
	if(par<0) par=2;
	if(par>NR_PARS) {
		KMessageBox::error(this,i18n("Not more than %1 parameters allowed!").arg(par));
		par=NR_PARS;
	}
	parni->setValue(par);
	for(int i=0;i<par;i++)
		parNle[i]->setEnabled(true);
}

void FitListDialog::setInitialValue(int par, double v) {
	if(par>NR_PARS || par<0) {
		KMessageBox::error(this,i18n("Not more than %1 parameters allowed!").arg(par));
		return;
	}

	parNle[par]->setText(QString::number(v));
}

double FitListDialog::initialValue(int par) {
	if(par>NR_PARS || par<0) {
		KMessageBox::error(this,i18n("Not more than %1 parameters allowed!").arg(par));
		return 0;
	}

	return parNle[par]->text().toDouble();
}

void FitListDialog::setWeightFunction(QString w) {
	weightcb->setCurrentItem(WUSER);
	weightle->setText(w);
}

void FitListDialog::setRegion(double a,double b) {
	regioncb->setChecked(true);
	regionminle->setText(QString::number(a));
	regionmaxle->setText(QString::number(b));
}

void FitListDialog::setBaseline(double b) {
	baselinecb->setChecked(true);
	baselinele->setText(QString::number(b));
}

void FitListDialog::setRange(double a,double b) {
	minle->setText(QString::number(a));
	maxle->setText(QString::number(b));
}

#ifdef HAVE_GSL
int fun_f(const gsl_vector *v, void *params, gsl_vector *f) {
//	kdDebug()<<"fun_f() "<<endl;
	int n = ((struct data *)params)->n;
	int np = ((struct data *)params)->np;
	double *x = ((struct data *)params)->x;
	double *y = ((struct data *)params)->y;
	double *sigma = ((struct data *)params)->sigma;
	FModel model = ((struct data *)params)->model;
	double base = ((struct data *)params)->base;
	QString fun = ((struct data *)params)->fun;
	MainWin *mw =  ((struct data *)params)->mw;

	char var[2];
	var[1]=0;
	init_table();
	for (int j=0;j<np;j++) {
		var[0]=97+j;	// "a","b",...
		assign_variable(var,gsl_vector_get(v,j));
	}

	for (int i = 0; i < n; i++) {
//		kdDebug()<<"x["<<i<<"]="<<x[i]<<endl;
		double Yi=0;
		if((model != MLN || x[i] > 0) && (model != MPLANCK || x[i] != 0) ) {
			// kdDebug()<<"	ASSIGN "<<t<<" to x"<<endl;
			char var[]="x";
			assign_variable(var,x[i]);
			Yi = parse((char *) fun.latin1());

			if(parse_errors()>0) {
				KMessageBox::error(mw, i18n("Parse Error!\n Please check the given function or parameter."));
				delete_table();
				return GSL_EINVAL;
			}
		}

		Yi += base;
//		kdDebug()<<"Yi = "<<Yi<<endl;

		gsl_vector_set (f, i, (Yi - y[i])/sigma[i]);
	}
	delete_table();

	return GSL_SUCCESS;
}

int fun_df(const gsl_vector *v, void *params, gsl_matrix *J) {
//	kdDebug()<<"fun_df() "<<endl;
	int n = ((struct data *)params)->n;
	int np = ((struct data *)params)->np;
	double *x = ((struct data *)params)->x;
	double *sigma = ((struct data *) params)->sigma;
	FModel model = ((struct data *) params)->model;
	QString fun = ((struct data *)params)->fun;

	double* p = new double[np];
	for (int i=0;i<np;i++) {
		p[i]=gsl_vector_get(v,i);
//		kdDebug()<<"fun_df Parameter "<<i<<" = "<<p[i]<<endl;
	}

	init_table();
	for (int i = 0; i < n; i++) {
		/* Jacobian matrix J(i,j) = dfi / dxj, */
		/* where fi = (Yi - yi)/sigma[i],      */
		/*	Yi = model  */
		/* and the xj are the parameters */
		double t = x[i];
//		kdDebug()<<"x["<<i<<"]="<<t<<endl;
		double s = sigma[i];
//		kdDebug()<<"fun_df : sigma["<<i<<"]="<<s<<endl;

		// use general case for all models ? : NO. this is faster.
		switch(model) {
		case MFLINEAR:
			gsl_matrix_set (J, i, 0, t/s);
			gsl_matrix_set (J, i, 1, 1.0/s);
			break;
		case MEXP: {
			double e = exp(-p[1] * t);
			gsl_matrix_set (J, i, 0, e/s);
			gsl_matrix_set (J, i, 1, - t * p[0] * e/s);
			gsl_matrix_set (J, i, 2, 1.0/s);
			}; break;
		case MPOT: {
			double tlog;
			if (t<=0)
				tlog = 0;
			else
				tlog=log(t);
			gsl_matrix_set (J, i, 0, pow(t,p[1])/s);
			gsl_matrix_set (J, i, 1, p[0] * pow(t,p[1]) * tlog/s);
			}; break;
		case MLN: {
			double plog;
			if (p[1]==0)
				plog = 0;
			else if (p[1]<0)
				plog = log(-p[1]);
			else
				plog=log(p[1]);
			gsl_matrix_set (J, i, 0, 1/s);
			gsl_matrix_set (J, i, 1, plog/s);
			}; break;
		case M1L: {
			double tmp = s*(p[0]+p[1]*t)*(p[0]+p[1]*t);
			gsl_matrix_set (J, i, 0, -1/tmp);
			gsl_matrix_set (J, i, 1, - t/tmp);
			}; break;
		case MEXP2:
			gsl_matrix_set (J, i, 0, t*exp(-p[1]*t)/s);
			gsl_matrix_set (J, i, 1, -p[0]*exp(-p[1]*t)*t*t/s);
			break;
		case MGAUSSIAN: {
			// TODO : parameter c
			double e = exp(-(t-p[1])*(t-p[1])/(2*p[2]*p[2]));
			double p2 = p[2]*p[2];
			gsl_matrix_set (J, i, 0, e/(p[2]*sqrt(2*M_PI))/s);
			gsl_matrix_set (J, i, 1, p[0]*e*(t-p[1])/(p[2]*p2*sqrt(2*M_PI))/s);
			gsl_matrix_set (J, i, 2, p[0]*e*(p[1]*p[1]-p[2]*p[2]-2*p[1]*t+t*t)/(sqrt(2*M_PI)*p2*p2)/s);
			}; break;
		case MMAXWELL: {
			double e = exp(-p[1]*t*t);
			gsl_matrix_set (J, i, 0, t*t*e/s);
			gsl_matrix_set (J, i, 1, -p[0]*e*t*t*t*t/s);
			}; break;
		case MPLANCK: {
			if (t==0) {
				gsl_matrix_set (J, i, 0, 0);
			 	gsl_matrix_set (J, i, 1, 0);
			}
			else {
				double e = exp(p[1]*t);
				gsl_matrix_set (J, i, 0, t*t*t/(e-1)/s);
			 	gsl_matrix_set (J, i, 1, -p[0]*e*t*t*t*t/((e-1)*(e-1))/s);
			}
			}; break;
		case MLORENTZ: {
			double tmp = p[2]*p[2]/4+(t-p[1])*(t-p[1]);
			gsl_matrix_set (J, i, 0, 1/tmp/s);
			gsl_matrix_set (J, i, 1, 2*p[0]*(t-p[1])/(tmp*tmp)/s);
			gsl_matrix_set (J, i, 2, -p[0]*p[2]/(2*tmp*tmp)/s);
			}; break;
		case MMULTIEXP2: {
			double e1 = exp(p[1] * t);
			double e2 = exp(p[3] * t);
			gsl_matrix_set (J, i, 0, e1/s);
			gsl_matrix_set (J, i, 1, t * p[0] * e1/s);
			gsl_matrix_set (J, i, 2, e2/s);
			gsl_matrix_set (J, i, 3, t * p[2] * e2/s);
			}; break;
		case MMULTIEXP3: {
			double e1 = exp(p[1] * t);
			double e2 = exp(p[3] * t);
			double e3 = exp(p[5] * t);
			gsl_matrix_set (J, i, 0, e1/s);
			gsl_matrix_set (J, i, 1, t * p[0] * e1/s);
			gsl_matrix_set (J, i, 2, e2/s);
			gsl_matrix_set (J, i, 3, t * p[2] * e2/s);
			gsl_matrix_set (J, i, 4, e3/s);
			gsl_matrix_set (J, i, 5, t * p[4] * e3/s);
			}; break;
		default: {	// user defined
			char var[]="x";
			assign_variable(var,t);

			for(int j=0;j<np;j++) {	//parameter
				double dp=1.0e-5;	// variation of parameter
				for(int k=0;k<np;k++) {	// set other parameter
					if(k!=j) {
						var[0]=97+k;
						assign_variable(var,p[k]);
					}
				}
				var[0]=97+j;

				assign_variable(var,p[j]);
				double f_p = parse((char *) fun.latin1());
				assign_variable(var,p[j]+dp*p[j]);
				double f_pdp = parse((char *) fun.latin1());

				gsl_matrix_set(J,i,j,1.0/s*(f_pdp-f_p)/(dp*p[j]));
			}
			}; break;
		}

	}
	delete_table();

	return GSL_SUCCESS;
}

int fun_fdf(const gsl_vector *x, void *params, gsl_vector *f,gsl_matrix *J) {
	fun_f (x, params, f);
	fun_df (x, params, J);

	return GSL_SUCCESS;
}

void FitListDialog::print_state(int iter, gsl_multifit_fdfsolver * s) {
	int np = parni->value();

	QString text;
	text+= "iter : "+QString::number(iter)+"| x = ";
	for (int i=0;i<np;i++)
		text+=QString::number(gsl_vector_get (s->x, i))+" ";
	text+="|f(x)| = "+QString::number(gsl_blas_dnrm2 (s->f));

	infote->append(text);
}
#endif

//! save user defined function
void FitListDialog::saveFunction() {
	KConfig *config = mw->Config();
	config->setGroup("Fit Functions" );
	// rotate if necessary
	int nr_functions=10;
	if(!config->readEntry(QString("function%1").arg(nr_functions)).isEmpty()) {
		for(int i=0;i<nr_functions-1;i++) {
			config->writeEntry(QString("function%1").arg(i+1),config->readEntry(QString("function%1").arg(i+2)));
		}
		config->writeEntry(QString("function%1").arg(nr_functions),"");
	}

	//save values
	for(int i=0;i<nr_functions;i++) {
		QString fun = config->readEntry(QString("function%1").arg(i+1));
		if(fun.isEmpty()) {
			QString newfun = funle->text()+QString(";%1").arg(parni->value());
			bool isnew=true;
			for (int j=1;j<=i;j++) {
				if(newfun == config->readEntry(QString("function%1").arg(j)))
					isnew=false;
			}
			if(isnew)
				config->writeEntry(QString("function%1").arg(i+1),newfun);
			break;
		}
	}
}

int FitListDialog::apply_clicked() {
	kdDebug()<<"FitListDialog::apply_clicked()"<<endl;
#ifdef HAVE_GSL
	FModel model = (FModel)  modelcb->currentItem();
	kdDebug()<<" MODEL = "<<model<<endl;

	if(model >= MUSER) saveFunction();

	double base=0;
	if (baselinecb->isChecked())
		base = baselinele->text().toDouble();

	// start values
	int np = parni->value();	 //  number of parameter
	kdDebug()<<" NP = "<<np<<endl;
	if (np<=0) {
		KMessageBox::error(this,i18n("Number of parameter must be > 0."));
		return -1;
	}

	double* x_init = new double[np];
	for (int i=0;i<np;i++)
		x_init[i] = parNle[i]->text().toDouble();

	gsl_vector_view v = gsl_vector_view_array (x_init, np);
	gsl_rng_env_setup();

	int N=0;
	double *xdata, *ydata, *sigma;
	MainWin *mw=0;
	Plot *plot=0;
	QString funlabel;

	if(s) {	// from spreadsheet
		kdDebug()<<"Fit in Spreadsheet"<<endl;
		mw = s->getMainWin();
		QTable *table = s->Table();
		int col = table->currentColumn(), xcol=-1;

		if(s->columnType(col) == "[X]") {
			KMessageBox::error(this,i18n("Please select the Y column to fit."));
			return -1;
		}

		// find x column
		for(int i=0;i<col;i++) {	// search before y col
			if(s->columnType(i) == "[X]")
				xcol=i;
		}
		if(xcol == -1) {
			for(int i=col+1;i<table->numCols();i++) {	// search after y col
				if(s->columnType(i) == "[X]")
					xcol=i;
			}
		}
		if(xcol == -1) {
			KMessageBox::error(this,i18n("No X column found! Giving up."));
			return -1;
		}
		kdDebug()<<"		Using y column "<<col<<endl;
		kdDebug()<<"		Using x column "<<xcol<<endl;

		// fill data
		QTableSelection ts = table->selection(table->currentSelection());
#if QT_VERSION > 0x030102
		N = ts.numRows();
#else
		N = ts.bottomRow()-ts.topRow();
#endif
		if(N <= 0) {
			KMessageBox::error(this,i18n("Sorry. Something went wrong. Giving up!"));
			return -1;
		}

		xdata = new double[N];
		ydata = new double[N];
		sigma = new double[N];
		for(int i=0; i<N; i++) {
			double x = table->text(ts.topRow()+i,xcol).toDouble();
			double y = table->text(ts.topRow()+i,col).toDouble();

			// create weight
			double weight=1.0;		// not nice for errors !
			switch(weightcb->currentItem()){
			case WY:		weight=y; break;
			case WYY:	weight=y*y; break;
			case W1Y:	if (y != 0) weight=1.0/y; break;
			case W1YY:	if (y != 0) weight=1.0/(y*y); break;
			case WX:		weight = x; break;
			case WXX:	weight=x*x; break;
			case W1X:	if (x != 0) weight=1.0/x; break;
			case W1XX:	if (x != 0) weight=1.0/(x*x); break;
			case WUSER: {
					char xvar[]="x", yvar[]="y";
					assign_variable(xvar,x);
					assign_variable(yvar,y);

					double value = parse((char *) weightle->text().latin1());
					if(parse_errors()>0) {
						KMessageBox::error(mw, i18n("Parse Error!\n Please check the given weight function."));
						delete_table();
						return -3;
					}

					if(!finite(value))
						value=0;
					weight=value;
					}; break;
			}

			// TODO : normalize? : No; scaling problem
			//sum += sigma[N];

			// catch sigma[i]==0 problems
			if(weight<1.0e-15) weight=1.0e-15;

			sigma[i]=1.0/sqrt(weight);
			xdata[i] = x;
			ydata[i] = y;
		}

		funlabel = s->columnTitle(col);
		kdDebug()<<"	Spreadsheet DONE"<<endl;
	}
	else {	// from plot
		plot = p->getPlot(p->API());
		GraphList *gl = plot->getGraphList();
		if(gl->Number()==0) {
			KMessageBox::error(this,i18n("No graph found!"));
			return -2;
		}
		int item = 0;
		if(lv != 0 && lv->currentItem() != 0)
			item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
		GRAPHType st = gl->getType(item);

		if(st != GRAPH2D && st != GRAPH3D && st != GRAPH4D) {
			KMessageBox::error(this,i18n("Sorry. This function is not yet implemented!"));
			return -1;
		}

		if (baselinecb->isChecked())
			plot->setBaseline(base);

		// 2d : x-y, 3d : x-y-dy, 4d : x-y-dx-dy+x-y-dy1-dy2
		Point *a2d=0;
		Point3D *a3d=0;
		Point4D *a4d=0;
		bool g4type=0;
		Graph *g=gl->getGraph(item);
		int nx = g->Number();
		LRange range;
		kdDebug()<<"NX="<<nx<<endl;
		if(nx <= 0) {
			KMessageBox::error(this,i18n("Sorry. Something went wrong. Giving up!"));
			return -1;
		}

		funlabel = g->getLabel()->simpleTitle();
		if(st == GRAPH2D) {
			Graph2D *g2d = gl->getGraph2D(item);
			a2d = g2d->Data();
			range = g2d->Range(1);
		}
		else if(st == GRAPH3D) {
			Graph3D *g3d = gl->getGraph3D(item);
			a3d = g3d->Data();
			range = g3d->Range(1);
		}
		else if(st == GRAPH4D) {
			Graph4D *g4d = gl->getGraph4D(item);
			a4d = g4d->Data();
			g4type = g4d->Type();
			range = g4d->Range(1);
		}

		xdata = new double[nx];
		ydata = new double[nx];
		sigma = new double[nx];

		//create weight
		init_table();
		double regionmin = regionminle->text().toDouble(), regionmax = regionmaxle->text().toDouble();
		for (int i = 0; i < nx; i++) {
			double xx=0,yy=0;
			if(st == GRAPH2D) {
				xx=a2d[i].X();
				yy=a2d[i].Y();
				if(a2d[i].Masked())
					continue;
			}
			else if(st == GRAPH3D) {
				xx=a3d[i].X();
				yy=a3d[i].Y();
				if(a3d[i].Masked())
					continue;
			}
			else if(st == GRAPH4D) {
				xx=a4d[i].X();
				yy=a4d[i].Y();
				if(a4d[i].Masked())
					continue;
			}

			// outside region
			if(regioncb->isChecked() && !negate_regioncb->isChecked()  && ( xx < regionmin || xx > regionmax ) )
				continue;
			// inside neg. region
			if(regioncb->isChecked() && negate_regioncb->isChecked()  && ( xx > regionmin && xx < regionmax ) )
				continue;

			xdata[N] = xx;
			ydata[N] = yy;

			// create weight
			// TODO : why not a3d, a4d ?
			double weight=1.0;
			switch(weightcb->currentItem()){
			case WY:		weight=a2d[i].Y(); break;
			case WYY:	weight=a2d[i].Y()*a2d[i].Y(); break;
			case W1Y:	if (a2d[i].Y() != 0) weight=1.0/a2d[i].Y(); break;
			case W1YY:	if (a2d[i].Y() != 0) weight=1.0/(a2d[i].Y()*a2d[i].Y()); break;
			case WX:		weight=a2d[i].X(); break;
			case WXX:	weight=a2d[i].X()*a2d[i].X(); break;
			case W1X:	if (a2d[i].X() != 0) weight=1.0/a2d[i].X(); break;
			case W1XX:	if (a2d[i].X() != 0) weight=1.0/(a2d[i].X()*a2d[i].X()); break;
			case WERROR: {
				double e=1.0;
				if(st == GRAPH3D)
					e = a3d[i].Z();
				else if (st == GRAPH4D) {
					if(g4type)
						e = a4d[i].Z()+a4d[i].T();	// x-y-dy1-dy2
					else
						e = a4d[i].T();				// x-y-dx-dy
				}
				if(e!=0)
					weight=1.0/(e*e);
				}; break;
			case WUSER: {
				char xvar[]="x", yvar[]="y";
				if(st == GRAPH2D) {
					assign_variable(xvar,a2d[i].X());
					assign_variable(yvar,a2d[i].Y());
				}
				else if(st == GRAPH3D) {
					assign_variable(xvar,a3d[i].X());
					assign_variable(yvar,a3d[i].Y());
				}
				else if(st == GRAPH4D) {
					assign_variable(xvar,a4d[i].X());
					assign_variable(yvar,a4d[i].Y());
				}

				double value = parse((char *) weightle->text().latin1());
				if(parse_errors()>0) {
					KMessageBox::error(mw, i18n("Parse Error!\n Please check the given weight function."));
					delete_table();
					return -3;
				}

				if(!finite(value))
					value=0;
				weight=value;
				}; break;
			}

			// TODO : normalize? : No; scaling problem
			//sum += sigma[N];

			// catch sigma[i]==0 problems
			if(weight<1.0e-15) weight=1.0e-15;

			sigma[N]=1.0/sqrt(weight);

			N++;
		}
		delete_table();

		mw = p->getMainWin();
	}

	kdDebug()<<"Creating Fit Structure"<<endl;
	struct data d = { N, xdata, ydata, sigma, np, model, base, funle->text(), mw};
	gsl_multifit_function_fdf f;
	f.f = &fun_f;
	f.df = &fun_df;
	f.fdf = &fun_fdf;
	f.n = N;
	f.p = np;
	f.params = &d;

	const gsl_multifit_fdfsolver_type *T = gsl_multifit_fdfsolver_lmsder;
	gsl_multifit_fdfsolver *s = gsl_multifit_fdfsolver_alloc (T, N, np);
	gsl_multifit_fdfsolver_set (s, &f, &v.vector);
	int status,iter = 0;
	int maxsteps = stepsni->value();

	print_state (iter, s);

	kdDebug()<<"Do Fitting"<<endl;
	do {
		iter++;
		status = gsl_multifit_fdfsolver_iterate (s);

		print_state (iter, s);

		if (status) break;

		double tolerance = tolle->text().toDouble();
		status = gsl_multifit_test_delta (s->dx, s->x,tolerance, tolerance);
	} while (status == GSL_CONTINUE && iter < maxsteps);

	kdDebug()<<"Getting Fit Result"<<endl;
	gsl_matrix *covar = gsl_matrix_alloc (np, np);
	gsl_multifit_covar (s->J, 0.0, covar);

#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))

	// info
	QString text;
	double dof = N-np;
	double chi = gsl_blas_dnrm2(s->f) / sqrt(dof);
	double c = GSL_MIN_DBL(1, chi);	// limit error for poor fit

	for (int i=0;i<np;i++) {
		if (i>0) text += "\n";
		text += QChar(97+i)+ QString(" = ") + QString::number(FIT(i))+" +/- "+QString::number(c*ERR(i));
	}
	text += "\nstatus = "+i18n(gsl_strerror(status));
	text += "\nchi^2 = "+QString::number(chi*chi);
	infote->append(text);
	infote->scrollToBottom();

	// update parameter start values
	for (int i=0;i<np;i++)
		parNle[i]->setText(QString::number(FIT(i)));

	// create fit function
	kdDebug()<<"Creating Fit Function"<<endl;
	int numberx = numberni->value();
	if(plot == 0) numberx=N;
	Point *ptr = new Point[numberx];
	double rangemin=minle->text().toDouble();
	double rangemax=maxle->text().toDouble();
	double xmin=0,xmax=0,ymin=0, ymax=1;

	// reset values for residuals
	if(plot && rescb->isChecked()) {
		LRange *range = plot->Ranges();
		numberx=N;
		rangemin=range[0].rMin();
		rangemax=range[0].rMax();
	}

	init_table();
	for (int i = 0;i<numberx;i++) {
		double y=0,x=0;

		if(plot != 0) {
			// use scale of x-axis of plot
			int scale = plot->getAxis(0)->Scale();
			switch(scale) {
			case LINEAR:
			case SQRT:
			case SX2: x = rangemin+i*(rangemax-rangemin)/(double)(numberx-1); break;
			case LOG10: x = pow(10,log10(rangemin)+i*(log10(rangemax/rangemin))/(numberx-1)); break;
			case LOG2: x = pow(2,log2(rangemin)+i*(log2(rangemax/rangemin))/(numberx-1)); break;
			case LN: x = pow(M_E,log(rangemin)+i*(log(rangemax/rangemin))/(numberx-1)); break;
			}
		}
		else
			x = xdata[i];

		if( (model != MLN || x > 0) && (model != MPLANCK || x != 0) ) {
			char var[]="x";
			assign_variable(var,x);
			for(int j=0;j<np;j++) {
				var[0] = 97+j;
				assign_variable(var,FIT(j));
			}
			y = parse((char *) funle->text().latin1());

			if(parse_errors()>0) {
				KMessageBox::error(mw, i18n("Parse Error!\n Please check the given function."));
				delete_table();
				return -3;
			}
		}

		y += base;

		if(rescb->isChecked())
			y = ydata[i]-y;

		ptr[i].setPoint(x,y);
	}
	delete_table();

	mw->calculateRanges2D(ptr,numberx,&xmin,&xmax,&ymin,&ymax);

	gsl_multifit_fdfsolver_free (s);

	LRange range[2];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);

	// add result label to active worksheet
	// label for fit result in plot
	if (plot !=0 && labelcb->isChecked()) {
		Label *label = new Label(i18n("fit result : (f(x) = "+
			funle->text()+")<br>"+text.replace(QChar('\n'),QString("<br>"))),QFont("Times",10));
		// TODO : dont overwrite other label ?
		label->setPosition(0.13,0.1);
		label->setBoxed(true);
		p->setLabel(0,label);
	}

	// TODO : use fit function? (might be long)
	QString fun = QString(i18n("fit of")+QString(" ")+ funlabel);
	if(rescb->isChecked())
		fun.prepend(i18n("residuals of "));

	Style *style=0;
	Symbol *symbol=0;
	if(p!=0) {
		style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	kdDebug()<<"Creating new GRAPH"<<endl;
	Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,numberx);
	ng->setFitFunction(funle->text());

	int item=-2;
	if(plot != 0)
		item = sheetcb->currentItem();
	if(resultcb->isChecked())
		mw->addGraph2D(ng,item);

	if(plot != 0)
		updateList();
#else
	KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
#endif

	return 0;
}
