#include <QtGui>
#include "convertthread.h"
#include "defines.h"

ConvertThread::ConvertThread(QObject *parent):QThread(parent) {
    m_width = 0;
    m_height = 0;
    m_hasWidth = false;
    m_hasHeight = false;
    m_rotate = false;
    m_angle = 0.0;
    m_quality = 100;
}


void ConvertThread::setDesiredSize(int width, int height, bool hasWidth, bool hasHeight, bool maintainAspect) {
	
    QMutexLocker locker(&m_mutex);
    m_width = width;
    m_height = height;
    m_hasWidth = hasWidth;
    m_hasHeight = hasHeight;
    m_maintainAspect = maintainAspect;
}


void ConvertThread::setDesiredFormat(const QString& format) {
    m_format = format;
}

void ConvertThread::setDesiredRotation(bool rotate, double angle) {
    QMutexLocker locker(&m_mutex);
    m_rotate = rotate;
    m_angle = angle;
}

void ConvertThread::setQuality(int quality) {
    QMutexLocker locker(&m_mutex);
    m_quality = quality;
}

void ConvertThread::setDestPrefix(const QString& destPrefix) {
    QMutexLocker locker(&m_mutex);
    m_destPrefix = destPrefix;
}

void ConvertThread::setDestFolder(const QDir& destFolder) {
    QMutexLocker locker(&m_mutex);
    m_destFolder = destFolder;
}

void ConvertThread::setOverwriteAll(bool overwriteAll) {
    QMutexLocker locker(&m_mutex);
    m_overwriteAll = overwriteAll;
}

void ConvertThread::enqueueImage(const QString& name, const QString& extension, const QString& path) {
    m_mutex.lock();
    m_imageQueue.enqueue(QStringList() << name << extension << path);
    m_mutex.unlock();

    if(!isRunning()) {
        start();
    }
}

void ConvertThread::confirmOverwrite(int result) {
    QMutexLocker locker(&m_overwriteMutex);
    m_overwriteResult = result;
    m_overwriteCondition.wakeOne();
}

void ConvertThread::run() {
    m_mutex.lock();
    int width = m_width;
    int height = m_height;
    bool hasWidth = m_hasWidth;
    bool hasHeight = m_hasHeight;
    bool maintainAspect = m_maintainAspect;
    QString format = m_format;
    bool rotate = m_rotate;
    double angle = m_angle;
    int quality = m_quality;	
    QString destPrefix = m_destPrefix;
    QDir destFolder = m_destFolder;
    bool overwriteAll = m_overwriteAll;
    m_mutex.unlock();

    while(!m_imageQueue.isEmpty()) {
        m_mutex.lock();
        QStringList imageData = m_imageQueue.dequeue();
        m_mutex.unlock();

        QString targetFile;
        if (destPrefix.isEmpty()) {
                targetFile = destFolder.absolutePath() + QDir::separator() + imageData.at(0) + "." + format;
        }
        else {
            targetFile = destFolder.absolutePath() + QDir::separator() + destPrefix + "_" + imageData.at(0) + "." + format;
        }

        QString imagePath = imageData.at(2) + QDir::separator() + imageData.at(0) + "." + imageData.at(1);

        QImage *image = new QImage(imagePath);

        if (rotate && angle != 0.0) {
                QMatrix m;
                m.rotate( angle );
                *image = image->transformed( m,Qt::SmoothTransformation );
        }

        QImage destImg;
        
        if (hasWidth && hasHeight && !maintainAspect)
            destImg = image->scaled(width, height, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
        else if (hasWidth && hasHeight && maintainAspect)
            destImg =  image->scaled(width, height, Qt::KeepAspectRatio, Qt::SmoothTransformation);
        else if (hasWidth && !hasHeight)
            destImg = image->scaledToWidth(width, Qt::SmoothTransformation);
        else if (!hasWidth && hasHeight)
            destImg = image->scaledToHeight(height, Qt::SmoothTransformation);
        else if (!hasWidth && !hasHeight)
            destImg = image->scaled(image->width(), image->height(), Qt::KeepAspectRatio, Qt::SmoothTransformation);

        if ( QFile::exists( targetFile ) && !overwriteAll) {
            m_overwriteMutex.lock();
            m_overwriteResult = 1;
            emit overwriteQuestion(targetFile);
            m_overwriteCondition.wait(&m_overwriteMutex);
            int overwriteResult = m_overwriteResult;
            m_overwriteMutex.unlock();

            if(overwriteResult == 2) {
                m_mutex.lock();
                m_overwriteAll = true;
                m_mutex.unlock();
                overwriteAll = true;
            }

            if(overwriteResult == 0 || overwriteResult == 2) {
                if (destImg.save(targetFile, 0, quality))
                    emit imageStatus(imageData, tr("Converted"), CONVERTED);
                else
                    emit imageStatus(imageData, tr("Failed to Convert"), FAILED);
            } 
            else
                emit imageStatus(imageData, tr("Skipped"), SKIPPED);
        } 
        else {
            if (destImg.save(targetFile, 0, quality)) {
                emit imageStatus(imageData, tr("Converted"),CONVERTED);
            }
            else {
                emit imageStatus(imageData, tr("Failed to Convert"), FAILED);
            }
        }
        delete image;
    }
}
