/*
* This file is part of Sir, an open-source cross-platform Image tool
* 2007  Rafael Sachetto
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Rafael Sachetto <rsachetto@gmail.com>
* Program URL   : http://rsachetto.googlepages.com
*
*/

#include <QSettings>
#include <QDir>
#include <QImageWriter>
#include <QCompleter>
#include <QDirModel>
#include <QLocale>
#include <QMap>
#include <QIcon>

#include "optionsdialog.h"
#include "languageutils.h"


OptionsDialog::OptionsDialog( QWidget * parent, Qt::WFlags f) : QDialog(parent, f) {
    
    setupUi(this);

    QList<QByteArray> imageFormats = QImageWriter::supportedImageFormats();
    QStringList list;

    foreach(QByteArray format, imageFormats) {
        list.append(*new QString(format));
    }

    languages = new LanguageUtils();

    fileToNiceName = new QMap<QString, QString>();

    targetFormatComboBox->insertItems(0,list);
    connect(okButton,SIGNAL(clicked()), this, SLOT(writeSettings()));

    QCompleter *completer = new QCompleter(this);
    QDirModel *dir = new QDirModel(completer);
    dir->setFilter(QDir::AllDirs);
    completer->setModel(dir);
    targetFolderLineEdit->setCompleter(completer);

    QStringList wordList;
    wordList << "web" << "thumb" << "thumbnail" << "resized" << "new";
    QCompleter *completer2 = new QCompleter(wordList, this);
    completer2->setCaseSensitivity(Qt::CaseInsensitive);
    targetPrefixLineEdit->setCompleter(completer2);

    createLanguageMenu();
    readSettings();
}

void OptionsDialog::writeSettings() {
    QSettings settings("SIR");

    settings.beginGroup("MainWindow");
    settings.setValue("targetFolder", targetFolderLineEdit->text());
    settings.setValue("targetFormat", targetFormatComboBox->currentIndex());
    settings.setValue("width",widthLineEdit->text());
    settings.setValue("height",heightLineEdit->text());
    settings.setValue("targetPrefix",targetPrefixLineEdit->text());
    settings.setValue("quality", qualitySpinBox->value());
    settings.setValue("languageNiceName", languagesComboBox->currentText());
    settings.setValue("languageFileName", fileToNiceName->value(languagesComboBox->currentText()));
    settings.endGroup();
    emit ok();
    this->close();

}

void OptionsDialog::readSettings() {

    QSettings settings("SIR");

    QString locale = QLocale::system().name();

    QString defaultLanguage = languages->getLanguageInfo("sir_"+locale+".qm").niceName;

    settings.beginGroup("MainWindow");
    targetFolderLineEdit->setText(settings.value("targetFolder", QDir::homePath()).toString());
    targetFormatComboBox->setCurrentIndex(settings.value("targetFormat", 0).toInt());
    widthLineEdit->setText(settings.value("width", "800").toString());
    heightLineEdit->setText(settings.value("height", "600").toString());
    targetPrefixLineEdit->setText(settings.value("targetPrefix", "web").toString());
    qualitySpinBox->setValue(settings.value("quality", 100).toInt());
    languagesComboBox->setCurrentIndex(languagesComboBox->findText(settings.value("languageNiceName",defaultLanguage).toString(),Qt::MatchExactly));
    settings.endGroup();
}

void OptionsDialog::createLanguageMenu() {
    
    QDir dir(":translations/");
    QStringList filter;
    filter << "sir_*.qm";
    QStringList fileNames = dir.entryList(filter);

    QString languageName;
    QIcon * countryFlag;
    LanguageInfo info;
    info = languages->getLanguageInfo("sir_en_US.qm");

    languageName = info.niceName;
    QString trlangName = tr(languageName.toStdString().c_str());

    countryFlag = new QIcon(":images/" + info.flagFile);

    languagesComboBox->insertItem(0, *(countryFlag), trlangName);

    fileToNiceName->insert(trlangName, "sir_en_US.qm");

    for (int i = 0; i < (int)fileNames.size(); ++i) {
        info = languages->getLanguageInfo(fileNames[i]);
        languageName = info.niceName;

        trlangName = tr(languageName.toStdString().c_str());
        countryFlag = new QIcon(":images/" + info.flagFile);

        languagesComboBox->insertItem(i+1, *(countryFlag), trlangName);

        fileToNiceName->insert(trlangName, fileNames[i]);
    }

}
