/***************************************************************************
 *   Copyright (C) 2005 Meni Livne <livne@kde.org>                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef __PHISH_SETTINGS_H
#define __PHISH_SETTINGS_H


/** Settings the library uses */
typedef struct
{
  /** Running mode */
  phish_mode_t mode;

  /** URL used for site queries on server */
  char *site_query_url;

  /** URL used for country queries on server */
  char *country_query_url;

  /** URL used for reporting sites on server */
  char *report_site_url;

  /** URL of XML file to download on server */
  char *remote_xml_url;

  /** ETag of remote XML file */
  char *remote_xml_etag;

  /** Filename of local XML file */
  char *local_xml_file;

  /** Filename of safe list file */
  char *safe_list_file;
} phish_settings_t;


/** Initialises the settings to their defaults.
 *  @param settings settings structure to initialise
 */
void phish_settings_init(phish_settings_t *settings);

/** Frees memory associated with settings structure.
 *  @param settings settings strcuture to free
 */
void phish_settings_free(phish_settings_t *settings);

/** Load settings from a file.
 *  @param settings settings structure to load settings into
 *  @param settings_file_name name of setting file to load
 */
phish_result_t phish_settings_load(phish_settings_t *settings,
                                   const char *settings_file_name);

/** Saves settings to a file.
 *  @param settings settings structure to save
 *  @param settings_file_name name of setting file to save settings into
 */
phish_result_t phish_settings_write(phish_settings_t *settings,
                                    const char *settings_file_name);

/** Returns the running mode.
 *  @param settings settings structure to return setting from
 */
phish_mode_t phish_settings_runningMode(phish_settings_t *settings);

/** Returns the URL used for site queries on the server.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_siteQueryURL(phish_settings_t *settings);

/** Returns the URL used for country queries on the server.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_countryQueryURL(phish_settings_t *settings);

/** Returns the URL used for reporting sites on the server.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_reportSiteURL(phish_settings_t *settings);

/** Returns the URL to the XML file to download on the server
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_remoteXMLURL(phish_settings_t *settings);

/** Returns the name of the local XML file.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_localXMLFile(phish_settings_t *settings);

/** Returns the name of the safe list file.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_safeListFile(phish_settings_t *settings);

/** Returns the ETag of the XML file on the server.
 *  @param settings settings structure to return setting from
 */
const char *phish_settings_remoteXMLETag(phish_settings_t *settings);

/** Sets the running mode.
 *  @param settings settings structure to set setting in
 *  @param mode new mode to use
 */
void phish_settings_setRunningMode(phish_settings_t *settings,
                                   phish_mode_t mode);

/** Sets the site queries URL.
 *  @param settings settings structure to set setting in
 *  @param url new URL to use
 */
void phish_settings_setSiteQueryURL(phish_settings_t *settings,
                                    const char *url);

/** Sets the country queries URL.
 *  @param settings settings structure to set setting in
 *  @param url new URL to use
 */
void phish_settings_setCountryQueryURL(phish_settings_t *settings,
                                       const char *url);

/** Sets the site reporting URL.
 *  @param settings settings structure to set setting in
 *  @param url new URL to use
 */
void phish_settings_setReportSiteURL(phish_settings_t *settings,
                                     const char *url);

/** Sets the remote XML file URL.
 *  @param settings settings structure to set setting in
 *  @param url new URL to use
 */
void phish_settings_setRemoteXMLURL(phish_settings_t *settings,
                                    const char *url);

/** Sets the local XML file name.
 *  @param settings settings structure to set setting in
 *  @param path new file name to use
 */
void phish_settings_setLocalXMLFile(phish_settings_t *settings,
                                    const char *path);

/** Sets the safe list file name.
 *  @param settings settings structure to set setting in
 *  @param path new file name to use
 */
void phish_settings_setSafeListFile(phish_settings_t *settings,
                                    const char *path);

/** Sets the ETag of the XML file on the server.
 *  @param settings settings structure to set setting in
 *  @param etag new etag
 */
void phish_settings_setRemoteXMLETag(phish_settings_t *settings,
                                     const char *etag);


#endif /* __PHISH_SETTINGS_H */

