/***************************************************************************
 *   Copyright (C) 2005 by Luc Willems   *
 *   willems.luc@pandora.be   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kdebug.h>
#include "kmidiplayer.h"

//tse3 stuff
#include "tse3/plt/Factory.h"
#include "tse3/app/Application.h"
#include "tse3/util/NoteNumber.h"
#include "tse3/Metronome.h"
#include "tse3/MidiFile.h"
#include "tse3/TempoTrack.h"
#include "tse3/Mutex.h"
#include "tse3/Track.h"

//QT &KDE stuff
#include <kprogress.h>
#include <kapplication.h>

//my stuff
#include "midistatusevent.h"


using namespace TSE3;
using namespace TSE3::App;


kmidiPlayer::kmidiPlayer()
{
   m_tse3Appl=0;
   m_prevstate=0;
   m_song=0;
   m_shutdown=0;
   m_progressdialog=0;
   m_midiport=0;
   t_tempo=50;
   t_lastbeat=0;
   m_verbose=false;
   m_playstatus=stop;
}

kmidiPlayer::~kmidiPlayer()
{
   m_tse3Appl->transport()->detachCallback(this);

   delete m_tse3Appl;
   delete m_song;
   m_tse3Appl=0;
   m_song=0;
}

int kmidiPlayer::initTSE3()
{

   kdDebug() << "init TSE3\n";
  /* create default alsa midi scheduler */ 
  { 
    using namespace Plt::UnixMidiSchedulerFactory;
    setPreferredPlatform(UnixPlatform_Alsa);
  }

  m_tse3Appl=new Application::Application("kmidplay","1.0",0,"");
  m_tse3Appl->metronome()->setBeatNote(44);
  kdDebug() << "tse3 scheduler : " << m_tse3Appl->scheduler()->implementationName() << "\n";
  kdDebug() << "tse3 scheduler int port : " << m_tse3Appl->scheduler()->portName(m_tse3Appl->scheduler()->defaultInternalPort()) << "\n";
  kdDebug() << "tse3 scheduler ext port : " << m_tse3Appl->scheduler()->portName(m_tse3Appl->scheduler()->defaultExternalPort()) << "\n";

  m_tse3Appl->transport()->midiEcho()->filter()->setStatus(true);
  m_tse3Appl->transport()->attachCallback(this);

  //start internal thread
  start();

  //we are ready :-)
  emit MidiReady();
  return 0;
}


void kmidiPlayer::Shutdown()
{
  {
    //this part must be protected by the mutex
    Impl::CritSec cs;
    kdDebug() << "TSE3 shutdown\n";
    //stop player if running
    if (m_playstatus != stop)
        _dostop();
    m_shutdown=true;
  }
  //
  wait();
}


int kmidiPlayer::loadsong(QString filename)
{
   TSE3::Impl::CritSec cs;
   kdDebug() << "TSE3 load song " << filename << "\n";

   m_progressdialog = new KProgressDialog::KProgressDialog(0,"","Load file",filename,false);
   //stop any playing song
   _dostop();
   kapp->processEvents(100);
   //delete previous song
   delete m_song;
   m_song=0;
   int result=1;

   kdDebug() << "tse3 loadsong " << filename << "\n";
   // load MIDI song
   MidiFileImport *mfi = new MidiFileImport(filename,1);
   try {
     m_progressdialog->show();
     kapp->processEvents(100);
     m_song = mfi->load(this);
   } catch (const MidiFileImportError &mf) {
     m_song=0;
     kdDebug() << "Midi File Import was unsuccessful\n";
     result=0;
   }
   delete m_progressdialog;
   m_progressdialog=0;

   return result;
}

int kmidiPlayer::LastBeat()
{
    Impl::CritSec cs;

    int lastClock = 0;
    for (size_t i=0;i < m_song->size();i++)
    {
        TSE3::Track* track=(*m_song)[i];
        if ( track->lastClock().beat() > lastClock )
            lastClock=track->lastClock().beat();
        ++i;
    }
    return lastClock;
}

int kmidiPlayer::CurrentBeat()
{
    Impl::CritSec cs;
    return  m_tse3Appl->scheduler()->clock().beat();
}

void kmidiPlayer::Play(bool metronome)
{
   TSE3::Impl::CritSec cs;
   kdDebug() << "TSE3 start play\n";
   _setmetronome(metronome);
   m_tse3Appl->transport()->filter()->setPort(m_midiport);
   m_song->tempoTrack()->setStatus(false);
   //set lead in when selecting metronome
   if (metronome)
       m_tse3Appl->transport()->setPlayLeadIn(Clock::Clock(Clock::PPQN*6));
   m_tse3Appl->transport()->play(m_song,0);

   kdDebug() << " tse3 title " << m_song->title()<< "\n";
   kdDebug() << " tse3 Author " << m_song->author()<< "\n";
   kdDebug() << " tse3 CopyrightmidiPoll " << m_song->copyright()<< "\n";
   kdDebug() << " tse3 Date " << m_song->date()<< "\n";

   m_prevstate=m_tse3Appl->transport()->status();
   m_stopped=false;
   m_eventtime=0;
   t_lastbeat=0;
   m_playstatus=play;
}

void kmidiPlayer::Stop()
{
   TSE3::Impl::CritSec cs;
   kdDebug() << "TSE3 stop\n";
   m_stopped=true;
   _dostop();
}

void kmidiPlayer::Pause()
{
   TSE3::Impl::CritSec cs;
   _dopause();
}

void kmidiPlayer::Forward()
{
  TSE3::Impl::CritSec cs;
  kdDebug() << "TSE3 Forward\n";
  m_tse3Appl->transport()->ff(true);
}

void kmidiPlayer::Back()
{
  TSE3::Impl::CritSec cs;
  kdDebug() << "TSE3 Back\n";
  m_tse3Appl->transport()->rew(true);
}


void kmidiPlayer::panic()
{
  TSE3::Impl::CritSec cs;
  kdDebug() << "TSE3 panic\n";
  _setPanic(m_tse3Appl->transport()->startPanic());
  _setPanic(m_tse3Appl->transport()->endPanic());
}

void kmidiPlayer::setMetronome(bool b)
{
  TSE3::Impl::CritSec cs;
  kdDebug() << "TSE3 setMetronome :" << b << "\n";
  _setmetronome(b);
}

int kmidiPlayer::getTempo()
{
 TSE3::Impl::CritSec cs;
 kdDebug() << "TSE3 getTempo\n";
 return _gettempo();
}

void kmidiPlayer::setPosition(int pos)
{
 //TODO : do something with pos
 pos++;
}

void kmidiPlayer::setTempo( int bpm )
{
 TSE3::Impl::CritSec cs;
 bool wasrunning=false;

 kdDebug() << "TSE3 setTempo " << bpm << "\n";
 if (m_song != 0)
 {
  //if we are playing , pause play and get current clock
  wasrunning= m_tse3Appl->scheduler()->running();
  if (wasrunning)
  {
      kdDebug() << "TSE3 was running ,pause scheduler\n";
      m_tse3Appl->scheduler()->stop(m_tse3Appl->scheduler()->clock());
  }
  TempoTrack* tempo = m_song->tempoTrack();
  kdDebug() << " tempo track size : " << tempo->size()<< "\n";
  //clear al tempo changes and add 1
  while ( tempo->size() > 0)
    tempo->erase(0);
  tempo->insert(Event<Tempo>(Tempo(bpm),0));
  m_tse3Appl->scheduler()->setTempo(bpm,0);
  //restart schedular if needed
  if (wasrunning)
      m_tse3Appl->scheduler()->start();
 }
 //set heartbeat tempo 
 t_tempo=(60000/bpm/2);
 kdDebug() << " t_tempo " << t_tempo<< "\n";
}




void kmidiPlayer::progressRange(int min ,int max )
{
  TSE3::Impl::CritSec cs;
   kdDebug() << "TSE3 progressRange " << min << " " << max<< "\n";
  if (m_progressdialog != 0)
  {
      m_progressdialog->progressBar()->setTotalSteps(max);
      kapp->processEvents(100);
  }
}


void kmidiPlayer::progress(int current)
{
  TSE3::Impl::CritSec cs;
  kdDebug() << "TSE3 progress " << current<< "\n";
  if (m_progressdialog != 0)
  {
    m_progressdialog->progressBar()->setValue(current);
    kapp->processEvents(10);
  }
}

int kmidiPlayer::Status()
{
  TSE3::Impl::CritSec cs;
  return m_tse3Appl->transport()->status();
}

TSE3::Song* kmidiPlayer::Song()
{
  TSE3::Impl::CritSec cs;
  return m_song;
}

QStringList* kmidiPlayer::getMidiPorts()
{
  TSE3::Impl::CritSec cs;
  QStringList* ports=new QStringList();
  if ( m_tse3Appl !=0) 
  {
    std::vector<int> portNums;
    m_tse3Appl->scheduler()->portNumbers(portNums);
    for (size_t port = 0; port < m_tse3Appl->scheduler()->numPorts(); port++)
    {
      int i=portNums[port];
      kdDebug() << "port :"  << i << "\n";
      ports->append(m_tse3Appl->scheduler()->portName(i));
    }
  }
  return ports;
}

QStringList* kmidiPlayer::MidiPorts()
{
  TSE3::Impl::CritSec cs;
  return getMidiPorts();
}

int kmidiPlayer::getMidiPort()
{
  TSE3::Impl::CritSec cs;
  return m_midiport;
}

bool kmidiPlayer::setMidiPort( int port ,bool isIndex)
{
  TSE3::Impl::CritSec cs;
  //if port=index , convert to port number 
  if (isIndex)
     port=m_tse3Appl->scheduler()->portNumber(port);

  //check if valid
  if ( ! m_tse3Appl->scheduler()->validPort(port))
      return false;
  kdDebug() << "Set port : "<< port<< "\n";
  m_midiport=port;
  //set TSE port in transport filter
  m_tse3Appl->transport()->filter()->setPort(port);
  return true;
}


QString* kmidiPlayer::getMidiPortName( int port )
{
  TSE3::Impl::CritSec cs;
  if (port == -1)
      port=m_midiport;
  if ( ! m_tse3Appl->scheduler()->validPort(port))
      return new QString("invalid");
  return new QString( m_tse3Appl->scheduler()->portName(port));
}

int kmidiPlayer::getEventTime()
{
  TSE3::Impl::CritSec cs;
  if ( m_tse3Appl->transport()->status() != 0)
     return m_eventtime;
  return 0;
}


////////////////////////////////////////////////////////////
// Protected internal code                                //
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Convert Custom Events to signals                       //
////////////////////////////////////////////////////////////
void  kmidiPlayer::_setPanic(TSE3::Panic* panic)
{
    kdDebug() << "tse3 _setpanic " << panic<< "\n";
    panic->setMidiReset(true);
    panic->setGmReset(true);
    panic->setGsReset(true);
    panic->setXgReset(true);
}

int kmidiPlayer::_gettempo()
{
 kdDebug() << "tse3 _gettempo\n";
 int bpm=120;
 TempoTrack * tempo;
 tempo = m_song->tempoTrack();
 kdDebug() << " tse3 tempo track " << tempo << "\n";
 kdDebug() << " tse3 tempo size " << tempo->size() << "\n";
 if (tempo->status()) 
 {
    bpm=(*tempo)[0].data.tempo;
    int time=(*tempo)[0].time;
    kdDebug() << " tse3 tempo time " << time << "\n";
 }
 else
   kdDebug() << " tse3 not tempo found , using default " << bpm << "\n";
 return bpm;
}

void kmidiPlayer::_dostop()
{
  kdDebug() << "tse3 _dostop" << "\n";
  m_tse3Appl->transport()->stop();
  emit SongStopped();
  m_playstatus=stop;
}

void kmidiPlayer::_dopause()
{
 if (m_playstatus == play) 
   {
     kdDebug() << "TSE3 pause play" << "\n";
     m_playstatus=pause;
     m_tse3Appl->scheduler()->stop(m_tse3Appl->scheduler()->clock());
   }
 else 
   {
     kdDebug() << "TSE3 play after pause\n";
     m_tse3Appl->scheduler()->start();
     m_playstatus=play;
   }
}

void kmidiPlayer::_setmetronome(bool b)
{
  kdDebug() << "tse3 _setmetronome\n";
  if (b)
  {
        m_tse3Appl->metronome()->setStatus(Transport::Playing, true);
        m_tse3Appl->metronome()->setStatus(Transport::SynchroPlaying, true);
  } 
  else
  {
        m_tse3Appl->metronome()->setStatus(Transport::Playing,false);
        m_tse3Appl->metronome()->setStatus(Transport::SynchroPlaying, true);
  }
}

bool kmidiPlayer::event( QEvent* e )
{
  if (e->type() == midiIOEventid )
  {
       MidiIOEvent* ce = (MidiIOEvent*)e;
       emit MidiEvent(*ce->getCommand());
       return true;
  }
  if (e->type() == midistatusEventid )
  {
    MidiStatusEvent* ce = (MidiStatusEvent*)e;
    switch (ce->getStatus())
    {
      case Midi_Stopped :   emit SongStopped();break;
      case Midi_Finished :  emit SongFinished();break;
      case Midi_beat :      emit MidiBeat();break;
    }
    return true;
  }
  return false;
}


int kmidiPlayer::playStatus()
{
  return m_playstatus;
}


////////////////////////////////////////////////////////////
// Following code is run in the MIDI thread context       //
////////////////////////////////////////////////////////////
void kmidiPlayer::run()
{
 kdDebug() << "TSE3 thread has started\n";
 //m_tse3Appl->transport()->setAdaptiveLookAhead(true);
 while (! m_shutdown) 
 {
    {
      TSE3::Impl::CritSec cs;
      
      //curent timestamp
      int now=m_tse3Appl->scheduler()->msecs();

      // do something
      m_tse3Appl->transport()->poll();

      // check status changes
      int state=m_tse3Appl->transport()->status();
      /* send signal when song  is finished */
      if ( ( state == 0 ) and ( m_playstatus == play))
      {
          if ( ! m_stopped )
          {
            m_eventtime=now;
            kdDebug() << "T-tse3 song finished\n";
            MidiStatusEvent* event=new MidiStatusEvent(Midi_Finished);
            kapp->postEvent(this,event);
            _dostop();
            //emit SongFinished();
          }
      }
      /* save state info */
      m_prevstate=state;

      // check for breakups
      int br=m_tse3Appl->transport()->breakUps();
      if (br != m_prevbr)
          kdDebug() << "T-BREAKUPS... "<<br<< "\n";
      m_prevbr=br;
      //send beat events to update clocks
      if ( m_tse3Appl->scheduler()->clock().beat() != t_lastbeat)
      {
         //send heartbeat puls
         m_eventtime=now;
         t_lastbeat=m_tse3Appl->scheduler()->clock().beat();
         MidiStatusEvent* event=new MidiStatusEvent(Midi_beat);
         kapp->postEvent(this,event);
      }
    }
    usleep(100);
 }
 kdDebug() << "TSE3 thread has finished.\n";
}


void kmidiPlayer::Transport_MidiIn(TSE3::MidiCommand m)
{
  sendMidiEvent(m,true);
}

void kmidiPlayer::Transport_MidiOut(TSE3::MidiCommand m)
{
  sendMidiEvent(m,false);
}

void kmidiPlayer::sendMidiEvent(TSE3::MidiCommand m, bool input)
{
  if ( qApp )
  {
      int now=m_tse3Appl->scheduler()->msecs();
      m_eventtime=now;
      MidiIOEvent* event=new MidiIOEvent(now,m,input);
      //std::cerr << "Midi event : " << now << "\n";
      kapp->postEvent(this,event);
  }
}





#include "kmidiplayer.moc"
