/***************************************************************************
 *   Copyright (C) 2005 by Luc Willems   *
 *   willems.luc@pandora.be   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kdebug.h>
#include <kmainwindow.h>
#include <klocale.h>
#include <kurl.h>
#include <kio/netaccess.h>
#include <kaction.h>
#include <kstdaction.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kfiledialog.h>
#include <kinputdialog.h>
#include <qiconset.h>
#include <ksimpleconfig.h>
#include <ktoolbar.h>
#include <knuminput.h>
#include <qapplication.h>

#include "tse3/Midi.h"
#include "tse3/Song.h"

#include "kmidplay.h"
#include "kmidplaywidget.h"
#include "kmidiiowindow.h"
#include "kmidiplayer.h"
#include "ktrackframe.h"
#include "kmidtrackview.h"

kmidplay::kmidplay()
    : KMainWindow( 0, "kmidplay",0),
    m_autourl(0)
{
  //play widget
  playWidget=new kmidplayWidget(this);
  setCentralWidget( playWidget );

  //create midi layer
  midiplayer= new kmidiPlayer();

  //MIDI io window
  m_midiwindow= new KmidiIOWindow(0,0);

  //setup action stuff
  m_config=new KSimpleConfig("kmidplay");

  //set menu actions & toolbar
  setupActions();

  //delay init of widgets
  QTimer::singleShot( 0, this, SLOT(initObjects()) );
}


kmidplay::~kmidplay()
{
  kdDebug() << "Close window";
  midiplayer->Shutdown();
  delete m_midiwindow;
  delete m_config;
  delete midiplayer;
}



void kmidplay::initObjects()
{

  /* Main external signals */
  connect (midiplayer,SIGNAL(SongFinished()),
            SLOT(midi_songFinished()));
  connect (midiplayer,SIGNAL(SongStopped()),
            SLOT(midi_songStopped()));
  connect (midiplayer,SIGNAL(MidiReady()),
            SLOT(midi_ready()));
  connect (midiplayer,SIGNAL(MidiBeat()),
            SLOT(midi_beat()));
  connect (midiplayer,SIGNAL(MidiEvent(TSE3::MidiCommand)),
            m_midiwindow->trackFrame(),SLOT(midi_event(TSE3::MidiCommand)));
  connect (midiplayer,SIGNAL(MidiEvent(TSE3::MidiCommand)),
            this, SLOT(midi_event(TSE3::MidiCommand)));

  connect (playWidget,SIGNAL( tempoChange(int)),
             SLOT(setBPM(int)));
  connect (playWidget,SIGNAL( positionChange(int)),
             SLOT(setPosition(int)));

  connect (m_midiwindow,SIGNAL(hideIOWindow()),
             SLOT(hideIOWindow()));
  //init widgets
  playWidget->initObjects();
  //init midi
  midiplayer->initTSE3();

  //setup program
  setup();
  kdDebug() << this->width () << " " << this->height() << "\n";
  setMinimumSize(380,200);
  setMaximumSize(380,200);

  if ( m_autourl )
  {
    //start autoplay
    kdDebug() << "autoplay :" << m_autourl->prettyURL()<<"\n";;
    LoadUrl(*m_autourl);
    QTimer::singleShot(0, this, SLOT(midi_play()));
  }

}


void kmidplay::setup()
{
  KToggleAction* metronome;
  KToggleAction* loop;

  kdDebug() << "setup\n";;
  KRecentFilesAction *recent;
  recent = (KRecentFilesAction*)actionCollection()->action(KStdAction::stdName(KStdAction::OpenRecent));
  recent->loadEntries(m_config,"midi_files");
  recent->setEnabled(true);

  //get general options
  metronome=(KToggleAction*)actionCollection()->action("midi_metronome");
  loop=(KToggleAction*)actionCollection()->action("midi_loop");

  m_config->setGroup( "General" );
  metronome->setChecked(m_config->readBoolEntry("metronome",false));
  loop->setChecked(m_config->readBoolEntry("loop",false));

  m_config->setGroup( "General" );
  metronome->setChecked(m_config->readBoolEntry("metronome",false));
  loop->setChecked(m_config->readBoolEntry("loop",false));
 
  midiplayer->setMidiPort(m_config->readNumEntry("midiport",0));
  kdDebug() << "Midiport : " << midiplayer->getMidiPortName()->latin1() << "\n";
  if ( midiplayer->getMidiPort() == 0)
      QTimer::singleShot(0, this, SLOT(midi_setup()));
  statusBar()->changeItem((*midiplayer->getMidiPortName()),1);

}

void kmidplay::savesetup()
{
  KToggleAction* metronome;
  KToggleAction* loop;

  kdDebug() << "save setup\n";;

  metronome=(KToggleAction*)actionCollection()->action("midi_metronome");
  loop=(KToggleAction*)actionCollection()->action("midi_loop");

  m_config->setGroup( "General" );
  m_config->writeEntry("metronome",metronome->isChecked());
  m_config->writeEntry("loop",loop->isChecked());
  m_config->writeEntry("midiport",midiplayer->getMidiPort());

  //Save recent file menu 
  KRecentFilesAction *recent;
  recent = (KRecentFilesAction*)actionCollection()->action(KStdAction::stdName(KStdAction::OpenRecent));
  recent->saveEntries(m_config,"midi_files");
  
  m_config->sync();

}

void kmidplay::setBPM(int tempo)
{
 midiplayer->setTempo(tempo);
 kdDebug() << "set song tempo : " << tempo << "\n";;
}

void kmidplay::setPosition(int pos)
{
// midiplayer->setTempo(tempo);
 kdDebug() << "set song position: " << pos << "\n";;
 midiplayer->setPosition(pos);
}

bool kmidplay::queryClose()
{
  kdDebug() << "query close\n";;
  //prepare for shutdown
  midiplayer->Shutdown();
  savesetup();
  return true;
}


void kmidplay::setupActions()
{
  //setup standard actions 
  KStdAction::open(this, SLOT(file_open()),actionCollection());
  KStdAction::openRecent(this,SLOT(file_openRecent(const KURL&)),actionCollection());
  KStdAction::quit(this,SLOT(close()),actionCollection());

  KStdAction::saveOptions(this, SLOT(save_options()), actionCollection());
  KStdAction::preferences(this, SLOT(midi_setup()), actionCollection());
  //midi menu
  new KAction(i18n("Play"),            0, this, SLOT(midi_play())     ,actionCollection(), "midi_play");
  new KAction(i18n("Pause"),           0, this, SLOT(midi_pause())    ,actionCollection(), "midi_pause");
  new KAction(i18n("Forward"),         0, this, SLOT(midi_forward())  ,actionCollection(), "midi_forward");
  new KAction(i18n("Back"),            0, this, SLOT(midi_back())    ,actionCollection(), "midi_back");
  new KAction(i18n("Stop"),            0, this, SLOT(midi_stop())     ,actionCollection(), "midi_stop");
  new KToggleAction(i18n("Loop"),      0, this, SLOT(midi_loop())     ,actionCollection(), "midi_loop");
  new KToggleAction(i18n("Metronome"), 0, this, SLOT(midi_metronome()),actionCollection(), "midi_metronome");
  new KToggleAction(i18n("Midi IO"),   0, this, SLOT(midi_iowindow()) ,actionCollection(), "midi_iowindow");
  new KAction(i18n("&Panic"),          0, this, SLOT(midi_panic())    ,actionCollection(), "midi_panic");


  statusBar()->insertItem("",0,false);  
  statusBar()->insertItem("",1,false);  
  statusBar()->insertItem("",2,true);  

  //linki buttons with actions
  actionCollection()->action("midi_play")->plug(toolBar());
  actionCollection()->action("midi_pause")->plug(toolBar());
  actionCollection()->action("midi_forward")->plug(toolBar());
  actionCollection()->action("midi_back")->plug(toolBar());
  actionCollection()->action("midi_stop")->plug(toolBar());
  actionCollection()->action("midi_loop")->plug(toolBar());
  actionCollection()->action("midi_metronome")->plug(toolBar());
  actionCollection()->action("midi_iowindow")->plug(toolBar());
  actionCollection()->action("midi_panic")->plug(toolBar());

  //save settings
  setupGUI();
  setAutoSaveSettings();
  //set basic state
  stateChanged("disabled");
}

//actions slots
void kmidplay::file_open()
{
  QString filename;
  KURL url;

  url=KFileDialog::getOpenURL(QDir::homeDirPath(),"*.mid *.midi", this,"Open MIDI file");
  if (! url.isEmpty()) 
  {
    file_addRecent(url);
    LoadUrl(url);
  }
}

void kmidplay::file_openRecent(const KURL &url)
{
  kdDebug() << "Open recent " << url.url() << "\n";
  LoadUrl(url);

  KRecentFilesAction *recent;
  recent = ( KRecentFilesAction*)actionCollection()->action(KStdAction::stdName(KStdAction::OpenRecent));
  recent->setCurrentItem(-1);

}


void kmidplay::autoPlay(const KURL &url)
{
  if (m_autourl == 0)
  {
    m_autourl=new KURL(url);
  }
}

void kmidplay::file_addRecent(const KURL &url)
{
  kdDebug() << "Add recent " << url.url() << "\n";
  KRecentFilesAction *recent;
  recent = ( KRecentFilesAction*)actionCollection()->action(KStdAction::stdName(KStdAction::OpenRecent));
  recent->addURL(url);
}


void kmidplay::toggleToolBar()
{
  if(toolBar()->isVisible())
    toolBar()->hide();
  else
    toolBar()->show();
}


void kmidplay::setStatus(const QString s)
{
  statusBar()->changeItem(s,0);
}

void kmidplay::toggleStatusBar()
{
 if(statusBar()->isVisible())
    statusBar()->hide();
 else
    statusBar()->show();
}

void kmidplay::save_options()
{
   kdDebug() << "TODO :Save options\n";
}

void kmidplay::LoadUrl(const KURL &url)
{
  QString filename;
//  if ( midiplayer->Status() != 0 )
      //TODO : show error dialog
//      return;

  //download if needed and read the file
  if ( KIO::NetAccess::download(url, filename,0)) {
     //load song and add to playlist
     kdDebug() << "Open : " << filename << "\n";
     midiplayer->loadsong(filename);
     KIO::NetAccess::removeTempFile(filename);
     stateChanged("standby-play");
     statusBar()->changeItem(url.fileName(),2);
     playWidget->setOriginalTempo(midiplayer->getTempo());
     playWidget->setSliderMax(midiplayer->LastBeat());
     playWidget->setSliderPosition(0);
  }
}


void kmidplay::midi_ready()
{
  kdDebug() << "midi ready\n";
  stateChanged("standby");
  setStatus("Ready");
}


void kmidplay::midi_play()
{
  KToggleAction* metronome;
  kdDebug() << "midi play\n";
  if ( midiplayer->Status() != 0 )
      //TODO : show error
      return;

  //start playing
  metronome=(KToggleAction*)actionCollection()->action("midi_metronome");
  midiplayer->setTempo(playWidget->getTempo());
  midiplayer->Play(metronome->isChecked());
  stateChanged("playing");
}

void kmidplay::midi_pause()
{
  midiplayer->Pause();
  if (midiplayer->playStatus() == kmidiPlayer::pause)
      stateChanged("plause");
  else
      stateChanged("play");
}


void kmidplay::midi_stop()
{
  kdDebug() << "midi stop\n";
  midiplayer->Stop();
  m_midiwindow->trackFrame()->midi_stop();
  playWidget->setSliderPosition(0);
}

void kmidplay::midi_forward()
{
  kdDebug() << "midi forward\n";
  midiplayer->Forward();
}

void kmidplay::midi_back()
{
  kdDebug() << "midi back\n";
  midiplayer->Back();
}

void kmidplay::midi_loop()
{
  kdDebug() << "midi loop\n";

}

void kmidplay::midi_iowindow()
{
  KToggleAction* midi_io;
  midi_io=(KToggleAction*)actionCollection()->action("midi_iowindow");

  kdDebug() << "midi iowindow\n";
  if (midi_io->isChecked())
  {
      kdDebug() << m_midiwindow->x() << " " << m_midiwindow->y() << "\n";
      if ((m_midiwindow->x() == 0) & (m_midiwindow->y() == 0))
      {
         kdDebug() << "set position\n";
      }
      m_midiwindow->show();
  }
  else
      m_midiwindow->hide();
}

void kmidplay::midi_songStopped()
{
  kdDebug() << "midi song stopped\n";
  setStatus("Ready");
  stateChanged("standby-play");
  playWidget->setLcdTime(0);
  playWidget->setSliderPosition(0);
}

void kmidplay::midi_songFinished()
{
  KToggleAction* loop;
  kdDebug() << "Finished \n";
  loop=(KToggleAction*)actionCollection()->action("midi_loop");
  setStatus("Ready");
  stateChanged("standby-play");
  playWidget->setLcdTime(0);
  playWidget->setSliderPosition(0);
  if (loop->isChecked())
      QTimer::singleShot(2000,this, SLOT(midi_play()));
}

void kmidplay::setToggleAction(const QString name)
{
 KToggleAction* a;
 a=(KToggleAction*)actionCollection()->action(name);
 if ( a != 0 ) 
   {
     kdDebug() << "enable " << name << "\n";
     a->setChecked(true);
   }
}

void kmidplay::midi_metronome()
{ 
  KToggleAction* metronome;
  kdDebug() << "midi metronome\n";
  metronome=(KToggleAction*)actionCollection()->action("midi_metronome");
  midiplayer->setMetronome(metronome->isChecked());
}

void kmidplay::midi_panic()
{
  kdDebug() << "midi panic\n";
  midiplayer->panic();
}


#include "kmidplay.moc"


/*!
    \fn kmidplay::midi_setup()
 */
void kmidplay::midi_setup()
{
  kdDebug() << "TODO :midi_setup\n";
  QString     port;
  QStringList* ports;
  ports=midiplayer->MidiPorts();
  port = KInputDialog::getItem("Select Midi port", 
                               "Port",
			       (*ports)
                               );
 kdDebug() << "port : "<< port <<"\n";
 int index=ports->findIndex(port);
 kdDebug() << "index : "<< index <<"\n";
 midiplayer->setMidiPort(index,true);
 statusBar()->changeItem(port,1);
}

void kmidplay::midi_event(TSE3::MidiCommand m)
{
 if (m.channel >=0)
     playWidget->setLcdTime(midiplayer->getEventTime());
}

void kmidplay::midi_beat()
{
  playWidget->setLcdTime(midiplayer->getEventTime());
  playWidget->setSliderPosition(midiplayer->CurrentBeat());
  playWidget->beat();
}

void kmidplay::hideIOWindow()
{
  kdDebug() << "midi hide iowindow\n";
  KToggleAction* midi_io;
  midi_io=(KToggleAction*)actionCollection()->action("midi_iowindow");
  m_midiwindow->hide();
  midi_io->setChecked(false);
}
