/***************************************************************************
 *   Copyright (C) 2005 by Luc Willems   *
 *   willems.luc@pandora.be   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <kdebug.h>

#include <qlayout.h>
#include <qframe.h>
#include <qlabel.h>
#include <kled.h>

#include "tse3/Midi.h"
#include "tse3/util/NoteNumber.h"

#include "vumeter.h"
#include "kmidtrackview.h"

#include "sys/time.h"

kmidTrackview::kmidTrackview( QWidget * parent , int id )
      : QFrame(parent, 0,0),
      m_id(id)
{

    int m_trackheight=22;

    m_channeltimer= new QTimer(this);
    connect (m_channeltimer,SIGNAL(timeout()),
              this, SLOT(channeltimer()));


    QLabel* tracknr=0;

    // Create a horizontal box for each track
    //
    QHBoxLayout *hblayout = new QHBoxLayout(this,2,2);
    setFixedHeight(m_trackheight);

    //tracknumber
    tracknr = new QLabel(this);
    tracknr->setText(QString("%1").arg(id+1,2));
    tracknr->setFixedSize(m_trackheight,m_trackheight);
    hblayout->addWidget(tracknr);

    m_syschannel= new KLed(Qt::red,this);
    m_syschannel->setFixedSize(m_trackheight,m_trackheight);
    hblayout->addWidget(m_syschannel);
    m_syschannel->off();

    m_channel= new KLed(this);
    m_channel->setFixedSize(m_trackheight,m_trackheight);
    hblayout->addWidget(m_channel);
    PulsOn();


    // Create a VU meter
    m_vumeter = new VUMeter(this,
                       128,m_trackheight,
                       VUMeter::Horizontal);
    hblayout->addWidget(m_vumeter);
    m_vumeter->setLevel(0,false);

    hblayout->addSpacing(1);

    m_trackvoice = new QLabel(this);
    m_trackvoice->setText("");
    m_trackvoice->setFixedHeight(m_trackheight);
    m_trackvoice->setFixedWidth(16);
    m_trackvoice->setPaletteBackgroundColor(Qt::black);
    m_trackvoice->setPaletteForegroundColor(0xC0D8FF);
    m_trackvoice->setAlignment(Qt::AlignVCenter || Qt::AlignHCenter);
    hblayout->addWidget(m_trackvoice);
    hblayout->addSpacing(1);

    m_tracknote = new QLabel(this);
    m_tracknote->setText("");
    m_tracknote->setFixedHeight(m_trackheight);
    m_tracknote->setFixedWidth(150);
    m_tracknote->setPaletteBackgroundColor(Qt::black);
    m_tracknote->setPaletteForegroundColor(0xC0D8FF);
    m_tracknote->setAlignment(Qt::AlignVCenter);
    hblayout->addWidget(m_tracknote);
    hblayout->addSpacing(1);

    m_trackinstrument = new QLabel(this);
    m_trackinstrument->setText("");
    m_trackinstrument->setFixedHeight(m_trackheight);
    m_trackinstrument->setAlignment(Qt::AlignVCenter);
    hblayout->addWidget(m_trackinstrument);

    //fillup
    hblayout->addStretch(0);
   
    /* Main external signals */
    connect (m_vumeter,SIGNAL (levelzero()),
              SLOT (levelzero()));
 
    m_voices=0;
    m_level=0;
    
    //create string arrays
    createdrumnames();
    createGM1();
}

kmidTrackview::~kmidTrackview()
{
}


void kmidTrackview::midi_event(TSE3::MidiCommand m)
{
  long int starttime=time(NULL);

  switch (m.status)
  {
   case TSE3::MidiCommand_NoteOn: 
         PulsOn();
         if (m_voices == 0)
             m_level=int(m.data2);
         else
             m_level=((m_level + int(m.data2)) / 2);
         m_vumeter->setLevel(m_level,false);
         noteOn(m.data1);
/*
         if (isDrum())
             m_trackinstrument->setText(QString("%1").arg(noteName(m.data1)));
         else
             m_tracknote->setText(QString("%1").arg(noteName(m.data1)));
*/
         //for drumchannel do always timer (fast)
         if (m.channel == 9)
             m_vumeter->startFallTimer(20);
         break;
   case TSE3::MidiCommand_NoteOff: 
         PulsOn();
         noteOff(m.data1);
         break;
   case TSE3::MidiCommand_ProgramChange :
         sysPulsOn();
         if (! isDrum()) 
             m_trackinstrument->setText(InstrumentGM1Name(m.data1));
         break;
   case TSE3::MidiCommand_KeyPressure :
         PulsOn();
         kdDebug() << "Key Pressure....\n";
         break;
   case TSE3::MidiCommand_ChannelPressure :
         PulsOn();
         kdDebug() << "Channel Pressure\n"; 
         break;
   case TSE3::MidiCommand_PitchBend :
         PulsOn();
         //kdDebug() << "Pitch Bend......\n"; 
         break;
   case TSE3::MidiCommand_System :
         sysPulsOn();
         if (m.channel ==  TSE3::MidiSystem_SystemReset)
             midi_stop(); 
         break;
  }
  m_time=time(NULL)-starttime;
}

void kmidTrackview::midi_stop()
{
  m_vumeter->startFallTimer(1);
  m_tracknote->setText("");
  m_trackvoice->setText("");
  m_trackinstrument->setText("");
  m_level=0;
  voices.clear();
  sysPulsOn();
}

void kmidTrackview::levelzero()
{
   m_voices=0; 
   m_level=0;
   m_tracknote->setText("");
   m_trackvoice->setText("");
}
void kmidTrackview::PulsOn()
{
  m_channel->on();
  if (m_channeltimer->isActive())
     m_channeltimer->stop();
  m_channeltimer->start(200);
}

void kmidTrackview::sysPulsOn()
{
  m_syschannel->on();
  if (m_channeltimer->isActive())
     m_channeltimer->stop();
  m_channeltimer->start(200);
}

void kmidTrackview::channeltimer()
{
  m_channel->off();
  m_syschannel->off();
  m_channeltimer->stop();
}

void kmidTrackview::createdrumnames()
{
   m_drumnames.push_back(QString("Acoustic Bass Drum"));
   m_drumnames.push_back(QString("Bass Drum 1"));
   m_drumnames.push_back(QString("Side Stick"));
   m_drumnames.push_back(QString("Acoustic Snare"));
   m_drumnames.push_back(QString("Hand Clap"));
   m_drumnames.push_back(QString("Electric Snare"));
   m_drumnames.push_back(QString("Low Floor Tom"));
   m_drumnames.push_back(QString("Closed Hi Hat"));
   m_drumnames.push_back(QString("High Floor Tom"));
   m_drumnames.push_back(QString("Pedal Hi-Hat"));
   m_drumnames.push_back(QString("Low Tom"));
   m_drumnames.push_back(QString("Open Hi-Hat"));
   m_drumnames.push_back(QString("Low-Mid Tom"));
   m_drumnames.push_back(QString("Hi-Mid Tom"));
   m_drumnames.push_back(QString("Crash Cymbal"));
   m_drumnames.push_back(QString("High Tom"));
   m_drumnames.push_back(QString("Ride Cymbal"));
   m_drumnames.push_back(QString("Chinese Cymbal"));
   m_drumnames.push_back(QString("Ride Bell"));
   m_drumnames.push_back(QString("Tambourine"));
   m_drumnames.push_back(QString("Splash Cymbal"));
   m_drumnames.push_back(QString("Cowbell"));
   m_drumnames.push_back(QString("Crash Cymbal"));
   m_drumnames.push_back(QString("Vibraslap"));
   m_drumnames.push_back(QString("Ride Cymbal "));
   m_drumnames.push_back(QString("Hi Bongo"));
   m_drumnames.push_back(QString("Low Bongo"));
   m_drumnames.push_back(QString("Mute Hi Conga"));
   m_drumnames.push_back(QString("Open Hi Conga"));
   m_drumnames.push_back(QString("Low Conga"));
   m_drumnames.push_back(QString("High Timbale"));
   m_drumnames.push_back(QString("Low Timbale"));
   m_drumnames.push_back(QString("High Agogo"));
   m_drumnames.push_back(QString("Low Agogo"));
   m_drumnames.push_back(QString("Cabasa"));
   m_drumnames.push_back(QString("Maracas"));
   m_drumnames.push_back(QString("Short Whistle"));
   m_drumnames.push_back(QString("Long Whistle"));
   m_drumnames.push_back(QString("Short Guiro"));
   m_drumnames.push_back(QString("Long Guiro"));
   m_drumnames.push_back(QString("Claves"));
   m_drumnames.push_back(QString("Hi Wood Block"));
   m_drumnames.push_back(QString("Low Wood Block"));
   m_drumnames.push_back(QString("Mute Cuica"));
   m_drumnames.push_back(QString("Open Cuica"));
   m_drumnames.push_back(QString("Mute Triangle"));
   m_drumnames.push_back(QString("Open Triangle"));
}

/*!
    \fn kmidTrackview::isDrum()
 */
bool kmidTrackview::isDrum()
{
  if (m_id == 9)
      return true;
  else
      return false;
}


/*!
    \fn kmidTrackview::noteName(int note)
 */
QString kmidTrackview::noteName(int note)
{
  QString name;
  if (isDrum())
     if ((note >=35) and (note <= (int)m_drumnames.size()+35))
         name=m_drumnames[note-35];
     else
         name=QString("%1").arg(note);
  else
     name=QString(TSE3::Util::numberToNote(note));
  return name;
}

QString kmidTrackview::InstrumentGM1Name(int instrument)
{
  kdDebug() << "select instrument " << instrument << "\n";
  return m_gm1[instrument];
}


/*!
    \fn kmidTrackview::createGM1
 */
void kmidTrackview::createGM1()
{
  m_gm1.push_back(QString("Acoustic Grand Piano    "));
  m_gm1.push_back(QString("Bright Acoustic Piano    "));
  m_gm1.push_back(QString("Electric Grand Piano    "));
  m_gm1.push_back(QString("Honky-tonk Piano     "));
  m_gm1.push_back(QString("Electric Piano 1    "));
  m_gm1.push_back(QString("Electric Piano 2    "));
  m_gm1.push_back(QString("Harpsichord      "));
  m_gm1.push_back(QString("Bassoon      "));
  m_gm1.push_back(QString("Clavi      "));
  m_gm1.push_back(QString("Celesta      "));
  m_gm1.push_back(QString("Glockenspiel      "));
  m_gm1.push_back(QString("Music Box     "));
  m_gm1.push_back(QString("Vibraphone      "));
  m_gm1.push_back(QString("Marimba      "));
  m_gm1.push_back(QString("Xylophone      "));
  m_gm1.push_back(QString("Tubular Bells     "));
  m_gm1.push_back(QString("Dulcimer      "));
  m_gm1.push_back(QString("Drawbar Organ     "));
  m_gm1.push_back(QString("Percussive Organ     "));
  m_gm1.push_back(QString("Rock Organ     "));
  m_gm1.push_back(QString("Church Organ     "));
  m_gm1.push_back(QString("Reed Organ     "));
  m_gm1.push_back(QString("Accordion      "));
  m_gm1.push_back(QString("Harmonica      "));
  m_gm1.push_back(QString("Tango Accordion     "));
  m_gm1.push_back(QString("Acoustic Guitar (nylon)    "));
  m_gm1.push_back(QString("Acoustic Guitar (steel)    "));
  m_gm1.push_back(QString("Electric Guitar (jazz)    "));
  m_gm1.push_back(QString("Electric Guitar (clean)    "));
  m_gm1.push_back(QString("Electric Guitar (muted)    "));
  m_gm1.push_back(QString("Overdriven Guitar     "));
  m_gm1.push_back(QString("Distortion Guitar     "));
  m_gm1.push_back(QString("Guitar harmonics     "));
  m_gm1.push_back(QString("Acoustic Bass     "));
  m_gm1.push_back(QString("Electric Bass (finger)    "));
  m_gm1.push_back(QString("Electric Bass (pick)    "));
  m_gm1.push_back(QString("Fretless Bass     "));
  m_gm1.push_back(QString("Slap Bass 1    "));
  m_gm1.push_back(QString("Slap Bass 2    "));
  m_gm1.push_back(QString("Synth Bass 1    "));
  m_gm1.push_back(QString("Synth Bass 2    "));
  m_gm1.push_back(QString("Violin      "));
  m_gm1.push_back(QString("Viola      "));
  m_gm1.push_back(QString("Cello      "));
  m_gm1.push_back(QString("Contrabass      "));
  m_gm1.push_back(QString("Tremolo Strings     "));
  m_gm1.push_back(QString("Pizzicato Strings     "));
  m_gm1.push_back(QString("Orchestral Harp     "));
  m_gm1.push_back(QString("Timpani      "));
  m_gm1.push_back(QString("String Ensemble 1    "));
  m_gm1.push_back(QString("String Ensemble 2    "));
  m_gm1.push_back(QString("SynthStrings 1     "));
  m_gm1.push_back(QString("SynthStrings 2     "));
  m_gm1.push_back(QString("Choir Aahs     "));
  m_gm1.push_back(QString("Voice Oohs     "));
  m_gm1.push_back(QString("Synth Voice     "));
  m_gm1.push_back(QString("Orchestra Hit     "));
  m_gm1.push_back(QString("Trumpet      "));
  m_gm1.push_back(QString("Trombone      "));
  m_gm1.push_back(QString("Tuba      "));
  m_gm1.push_back(QString("Muted Trumpet     "));
  m_gm1.push_back(QString("French Horn     "));
  m_gm1.push_back(QString("Brass Section     "));
  m_gm1.push_back(QString("SynthBrass 1     "));
  m_gm1.push_back(QString("SynthBrass 2     "));

  m_gm1.push_back(QString("Soprano Sax     "));
  m_gm1.push_back(QString("Alto Sax     "));
  m_gm1.push_back(QString("Tenor Sax     "));
  m_gm1.push_back(QString("Baritone Sax     "));
  m_gm1.push_back(QString("Oboe      "));
  m_gm1.push_back(QString("English Horn     "));
  m_gm1.push_back(QString("Clarinet      "));
  m_gm1.push_back(QString("Piccolo      "));
  m_gm1.push_back(QString("Flute      "));
  m_gm1.push_back(QString("Recorder      "));
  m_gm1.push_back(QString("Pan Flute     "));
  m_gm1.push_back(QString("Blown Bottle     "));
  m_gm1.push_back(QString("Shakuhachi      "));
  m_gm1.push_back(QString("Whistle      "));
  m_gm1.push_back(QString("Ocarina      "));
  m_gm1.push_back(QString("Lead 1 (square)    "));
  m_gm1.push_back(QString("Lead 2 (sawtooth)    "));
  m_gm1.push_back(QString("Lead 3 (calliope)    "));
  m_gm1.push_back(QString("Lead 4 (chiff)    "));
  m_gm1.push_back(QString("Lead 5 (charang)    "));
  m_gm1.push_back(QString("Lead 6 (voice)    "));
  m_gm1.push_back(QString("Lead 7 (fifths)    "));
  m_gm1.push_back(QString("Lead 8 (bass + lead)  "));
  m_gm1.push_back(QString("Pad 1 (new age)   "));
  m_gm1.push_back(QString("Pad 2 (warm)    "));
  m_gm1.push_back(QString("Pad 2 (warm)    "));
  m_gm1.push_back(QString("Pad 4 (choir)    "));
  m_gm1.push_back(QString("Pad 5 (bowed)    "));
  m_gm1.push_back(QString("Pad 6 (metallic)    "));
  m_gm1.push_back(QString("Pad 7 (halo)    "));
  m_gm1.push_back(QString("Pad 8 (sweep)    "));
  m_gm1.push_back(QString("FX 1 (rain)    "));
  m_gm1.push_back(QString("FX 2 (soundtrack)    "));
  m_gm1.push_back(QString("FX 3 (crystal)    "));
  m_gm1.push_back(QString("FX 4 (atmosphere)    "));
  m_gm1.push_back(QString("FX 5 (brightness)    "));
  m_gm1.push_back(QString("FX 6 (goblins)    "));
  m_gm1.push_back(QString("FX 6 (goblins)    "));
  m_gm1.push_back(QString("FX 8 (sci-fi)    "));
  m_gm1.push_back(QString("Sitar      "));
  m_gm1.push_back(QString("Banjo      "));
  m_gm1.push_back(QString("Shamisen      "));
  m_gm1.push_back(QString("Koto      "));
  m_gm1.push_back(QString("Kalimba      "));
  m_gm1.push_back(QString("Bag pipe     "));
  m_gm1.push_back(QString("Fiddle      "));
  m_gm1.push_back(QString("Shanai      "));
  m_gm1.push_back(QString("Guitar Fret Noise    "));
  m_gm1.push_back(QString("Tinkle Bell     "));
  m_gm1.push_back(QString("Agogo      "));
  m_gm1.push_back(QString("Steel Drums     "));
  m_gm1.push_back(QString("Woodblock      "));
  m_gm1.push_back(QString("Taiko Drum     "));
  m_gm1.push_back(QString("Melodic Tom     "));
  m_gm1.push_back(QString("Synth Drum     "));
  m_gm1.push_back(QString("Reverse Cymbal     "));
  m_gm1.push_back(QString("Breath Noise     "));
  m_gm1.push_back(QString("Seashore      "));
  m_gm1.push_back(QString("Bird Tweet     "));
  m_gm1.push_back(QString("Telephone Ring     "));
  m_gm1.push_back(QString("Helicopter      "));
  m_gm1.push_back(QString("Applause      "));
  m_gm1.push_back(QString("Gunshot      "));
}
void kmidTrackview::show()
{
 QFrame::show();
 //kdDebug() << "track " << m_id << " show\n";
 setNoteNames();
}

void kmidTrackview::noteOn(int note)
{
  voices.push_back(note);
  setNoteNames();
}

void kmidTrackview::noteOff(int note)
{
  std::vector<int>::iterator i; 
  i = std::find(voices.begin(), voices.end(),note); // Search the list.
  if (i != voices.end())
      voices.erase(i);
  m_vumeter->startFallTimer(20);
  setNoteNames();
}

void kmidTrackview::setNoteNames() 
{
 QString notes;
 std::vector<int>::iterator it;
 int size= voices.size();
 //if not notes sounding
 if (size == 0)
     return;
 if (! isVisible ())
    return;

 std::sort(voices.begin(), voices.end());
 m_trackvoice->setText(QString("%1").arg(size));
 for( it = voices.begin(); it != voices.end(); it++ ) {
   if (isDrum())
       notes.append(QString("%1 ").arg(noteName(*it)));
   else
       notes.append(QString("%1 ").arg(noteName(*it)));
 } 
 m_tracknote->setText(notes);
 // kdDebug() << notes << "\n";
}

