// -*- c-basic-offset: 4 -*-

/*
    Rosegarden-4
    A sequencer and musical notation editor.

    This program is Copyright 2000-2005
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <bownie@bownie.com>

    The moral right of the authors to claim authorship of this work
    has been asserted.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include <iostream>
#include "vumeter.h"


VUMeter::VUMeter(QWidget *parent,
                 int width,
                 int height,
                 VUAlignment alignment,
                 const char *name):
    QFrame(parent, name),
    m_originalHeight(height),
    m_alignment(alignment),
    m_level(0),
    m_peakLevel(0),
    m_levelStep(m_baseLevelStep),
    m_fallTimer(0),
    m_peakTimer(0),
    m_showPeakLevel(true),
    m_baseLevelStep(3)
{

    // Always init the left fall timer
    //
    m_fallTimer = new QTimer();

    connect(m_fallTimer, SIGNAL(timeout()),
              this,       SLOT(slotReduceLevelLeft()));

    if (m_showPeakLevel)
    {
        m_peakTimer = new QTimer();
        connect(m_peakTimer, SIGNAL(timeout()),
                  this,           SLOT(slotStopShowingPeakLeft()));
    }

    setMinimumSize(width, m_originalHeight);
    setMaximumSize(width, m_originalHeight);

    if (m_alignment == Vertical) m_maxLevel = height;
    else m_maxLevel = width;

    int max = m_maxLevel;
    int red, orange, green;
    red    = max * 92 / 100;
    orange = max * 60 / 100;
    green  = max * 10 / 100;
    m_background = Qt::black;

    //use default QT colors
    m_velocityColour =  new VelocityColour(Qt::red,Qt::yellow,Qt::green,max, red, orange, green);
}

VUMeter::~VUMeter()
{
    delete m_velocityColour;
    delete m_peakTimer;
    delete m_fallTimer;
}


void VUMeter::setLevel(int Level , bool startfall)
{

    m_level=Level;
    if (m_level < 0) 
        m_level = 0;
    if (m_level > m_maxLevel) 
        m_level = m_maxLevel;

    m_levelStep = m_baseLevelStep;

    if (m_showPeakLevel) 
    {
      if (m_level >= m_peakLevel)
	  {
	    m_peakLevel = m_level;
	    if (m_peakTimer->isActive())
		m_peakTimer->stop();
	    m_peakTimer->start(1000); // milliseconds of peak hold
	  }
    }

    QPainter paint(this);
    drawMeterLevel(&paint);
    if (startfall)
        startFallTimer();
}

void VUMeter::paintEvent(QPaintEvent*)
{
    QPainter paint(this);

    //background
    paint.setPen(m_background);
    paint.setBrush(m_background);
    paint.drawRect(0, 0, width(), height());

    //meter
    drawMeterLevel(&paint);

    //Peak
    paint.setPen(colorGroup().background());
    paint.drawPoint(0, 0);
    paint.drawPoint(width()-1, 0);
    paint.drawPoint(0, height()-1);
    paint.drawPoint(width()-1, height()-1);
    
}

void VUMeter::drawColouredBar(QPainter *paint, int x, int y, int w, int h)
{
  Qt::BrushStyle style = Qt::SolidPattern;

  int medium = m_velocityColour->getMediumKnee(),
  loud = m_velocityColour->getLoudKnee();

  if (m_alignment == Vertical) {
    if (h > loud) {
	paint->setPen(m_velocityColour->getLoudColour());
	paint->setBrush(QBrush(m_velocityColour->getLoudColour(),style));
	paint->drawRect(x, y, w, h - loud);
    }
  } else {
    if (w > loud) {
	paint->setPen(m_velocityColour->getLoudColour());
	paint->setBrush(QBrush(m_velocityColour->getLoudColour(),style));
	paint->drawRect(x + loud, y, w - loud, h);
    }
  }

  if (m_alignment == Vertical) {
    if (h > medium) {
	paint->setPen(m_velocityColour->getMediumColour());
	paint->setBrush(QBrush(m_velocityColour->getMediumColour(),style));
	paint->drawRect(x, y + (h > loud ? (h - loud) : 0),w, std::min(h - medium, loud - medium));
    }
  } else {
    if (w > medium) {
	paint->setPen(m_velocityColour->getMediumColour());
	paint->setBrush(QBrush(m_velocityColour->getMediumColour(),style));
	paint->drawRect(x + medium, y,std::min(w - medium, loud - medium), h);
    }
  }

  if (m_alignment == Vertical) {
        paint->setPen(m_velocityColour->getQuietColour());
	paint->setBrush(QBrush(m_velocityColour->getQuietColour(),style));
        paint->drawRect(x, y + (h > medium ? (h - medium) : 0),w, std::min(h, medium));
  } else {
        paint->setPen(m_velocityColour->getQuietColour());
	paint->setBrush(QBrush(m_velocityColour->getQuietColour(),style));
        paint->drawRect(x, y, std::min(w, medium), h);
  }
	
}

void VUMeter::drawMeterLevel(QPainter* paint)
{
        // Paint a vertical meter according to type
        //
        if (m_alignment == VUMeter::Vertical)
        {
            int y = height() - (m_level * height()) / m_maxLevel;
	    drawColouredBar(paint,  0, y, width(), height());

	    paint->setPen(m_background);
	    paint->setBrush(m_background);
	    paint->drawRect(0, 0, width(), y);

            if (m_showPeakLevel)
            {
                paint->setPen(Qt::white);
                paint->setBrush(Qt::white);

                y = height() - (m_peakLevel * height()) / m_maxLevel;

                paint->drawLine(0, y, width(), y);
            }
        }
        else
        {
            int x = (m_level * width()) / m_maxLevel;
	    if (x > 0) drawColouredBar(paint, 0, 0, x, height());

	    paint->setPen(m_background);
	    paint->setBrush(m_background);
	    paint->drawRect(x, 0, width() - x, height());

            if (m_showPeakLevel)
            {
                paint->setPen(Qt::white);
                paint->setBrush(Qt::white);

                // show peak level
                x = (m_peakLevel * width()) / m_maxLevel;
                if (x < (width() - 1))
                    x++;
                else
                    x = width() - 1;

                paint->drawLine(x, 0, x, height());
            }
        }
}


void VUMeter::slotReduceLevelLeft()
{
    m_levelStep = int(m_level) * m_baseLevelStep / 100 + 1;
    if (m_levelStep < 1) m_levelStep = 1;


    if (m_level > 0) m_level -= m_levelStep;

    if (m_level <= 0) {
        m_level = 0;
        m_peakLevel = 0;
    }


    if (m_level == 0 ) {
        // Always stop the timer when we don't need it
        if (m_fallTimer) m_fallTimer->stop();
        emit levelzero();
    }

    QPainter paint(this);
    drawMeterLevel(&paint);
}


void VUMeter::slotStopShowingPeakLeft()
{
    m_peakLevel = 0;
}

void VUMeter::startFallTimer(int time)
{
  // Only start the timer when we need it
  if ( m_fallTimer != 0) 
  {
   if ( (m_level > 0) and  (m_fallTimer->isActive() == false) )
       m_fallTimer->start(time); // 40 ms per level fall iteration
  }
}


#include "vumeter.moc"
